#include "hal.h"

#include "stdio.h"

#ifdef SDB_UBOOT //added by Ted for UBoot
#include <asm/arch/hardware.h>
#define halUARTPrintf 	printf
#define halDelay		mdelay
#endif

#if defined(HAL_INC_I2S)
INLINE bool_T halDAIIsBusy(void)
{
	register unsigned int r;

	r = HAL_GETREG(HAL_REG_I2S_init);

	return (r & 0x06);
}

INLINE void halDAIResetFIFO(void)
{
	register unsigned int r;

	r = HAL_GETREG(HAL_REG_I2S_init);

	r |= 0x01;
	HAL_SETREG(HAL_REG_I2S_init, r);

	r &= 0x06;
	HAL_SETREG(HAL_REG_I2S_init, r);
}

INLINE void halDAIEnable(halDAIId_E nId)
{
	register unsigned int r;

	r = HAL_GETREG(HAL_REG_I2S_init);

	if (nId == halcDAITX)
		r |= 4;
	else if (nId == halcDAIRX)
		r |= 2;
	else if (nId == halcDAITXRX)
		r |= 6;

	HAL_SETREG(HAL_REG_I2S_init, r);
}

INLINE void halDAIDisable(halDAIId_E nId)
{
	register unsigned int r;

	r = HAL_GETREG(HAL_REG_I2S_init);

	if (nId == halcDAITX)
		r &= 0x03;
	else if (nId == halcDAIRX)
		r &= 0x05;
	else if (nId == halcDAITXRX)
		r &= 0x01;

	HAL_SETREG(HAL_REG_I2S_init, r);
}

bool_T halDAIConfig_master(halDAISetting_S *p, unsigned int Fin, unsigned int *pCodecCLK)
{
	/*
	 * Current stage:
	 *
	 * We allow I2S mode only.
	 * Precision for one channel-sample * is 16 bits only.
	 * TX always use "unison".
	 *
	 */

	register unsigned int r;

	halSysPLLIsMaster(TRUE);

	r = HAL_GETREG(HAL_REG_I2S_init);
	if (r & 6)
	{
		/* TX or RX is working now; not allow to program the protocol */
		return (FALSE);
	}

	/* data protocol between DAI and audio AD-DA codec */
	switch (p->nDataProtocol)
	{
		case halcDAII2SMode:
			HAL_SETREG8(HAL_REG_I2S_mode, 4);
			break;

		case halcDAILJustMode:
		case halcDAIRJustMode:
		case halcDAIMSBExtMode:
		case halcDAIDSPMode:
			/* driver doesn't supported yet */
			return (FALSE);
	}

	/* FIFO thresholds */
	HAL_SETREG8(HAL_REG_I2S_tx_fifo_lth, p->nTXFIFOThreshold);
	HAL_SETREG8(HAL_REG_I2S_rx_fifo_gth, p->nRXFIFOThreshold);

	/* Precision for one channel-sample * is 16 bits only. */
	HAL_SETREG8(HAL_REG_I2S_wlen, 16-1);
	HAL_SETREG8(HAL_REG_I2S_wpos, 0);

	/* RX control register */
	r = 2;	/* receiver is in master mode, default in driver */
	if (!p->bStereo)
		r += (1 << 4);
	HAL_SETREG8(HAL_REG_I2S_rx_ctrl, r);

	/* TX control register */
	r = 0;
	r += (1 << 5);	/* transmitter is in master mode, default in driver */
	r += (1 << 3);	/* transmiter unison */
	if (!p->bStereo)
		r += 1;
	HAL_SETREG8(HAL_REG_I2S_tx_ctrl, r);

	/*
	 *	Codec clock control register (0x4c),
	 *	Codec clock divider register (0x50),
	 */
	*pCodecCLK = halSysGetDeviceClockSrc(Fin, HAL_MFP_I2S, NULL)/2;
	HAL_SETREG8(HAL_REG_I2S_CODEC_clk_div, 1);
	HAL_SETREG8(HAL_REG_I2S_CODEC_clk_ctrl, 3);
	halUARTPrintf("I2S Main-clock %d\n\r", *pCodecCLK);

	/*
	 * clock control register setting (0x20) 
	 *
	 * sck_o_frequency = sysclk_frequency / (2*(clock_divider+1))
	 *
	 *	==> clock_divider = sysclk_frequency/(sck_o_frequency*2) - 1,
	 *	where sck_o_frequency = sampling_rate*32 or sampling_rate*64
	 *  In our setting, sck_o_frequency = sampling_rate*32 = sampling_rate*2*16
	 */
	/* for the new DAI with new function : dividing clock and mono */

	/*
	 * bit 12: ws signal output enable (1: enable; 0: disable)
	 * bit 11: sck output enable (1: enable; 0: disable)
	 * bit 10: sck source source selection (1: generate sck_o to codec; 0: accept external sck_i)
	 * bit 9 : transmitter sck polarity
	 * bit 8 : receiver sck polarity
	 * bit7:0: clock_divider
	 */

	r = (1 << 12) + (1 << 11) + (1 << 10) + (1 << 9) + (0 << 8);
	r += ((halSysGetDeviceClockSrc(Fin, HAL_MFP_I2S, NULL)/(p->nSamplingRate*32*2) - 1) & 0xFF);

	HAL_SETREG(HAL_REG_I2S_clock, r);

	halUARTPrintf("I2S serial clock %d\n\r", halSysGetDeviceClockSrc(Fin, HAL_MFP_I2S, NULL) / 2 / ((r & 0xff) +1));

	return (TRUE);

}

bool_T halDAIConfig_slave(halDAISetting_S *p, unsigned int Fin, unsigned int *pCodecCLK)
{
	/*
	 * We allow I2S mode only.
	 * Precision for one channel-sample * is 16 bits only.
	 * TX always use "unison".
	 */
	register unsigned int r;

	halSysPLLIsMaster(TRUE);

	r = HAL_GETREG(HAL_REG_I2S_init);
	if (r & 6)
	{
		return (FALSE);	/* TX or RX is working now; not allow to program the protocol */
	}

	/* data protocol between DAI and audio AD-DA codec */
	switch (p->nDataProtocol)
	{
		case halcDAII2SMode:
			HAL_SETREG8(HAL_REG_I2S_mode, 4);
			break;

		case halcDAILJustMode:
		case halcDAIRJustMode:
		case halcDAIMSBExtMode:
		case halcDAIDSPMode:
			/* driver doesn't supported yet */
			return (FALSE);
	}

	/* FIFO thresholds */
	HAL_SETREG8(HAL_REG_I2S_tx_fifo_lth, p->nTXFIFOThreshold);
	HAL_SETREG8(HAL_REG_I2S_rx_fifo_gth, p->nRXFIFOThreshold);

	/* Precision for one channel-sample * is 16 bits only. */
	HAL_SETREG8(HAL_REG_I2S_wlen, 16-1);
	HAL_SETREG8(HAL_REG_I2S_wpos, 0);

	// from betta's linux driver
	/* codec clk & frame master */
	unsigned int clk_ctl;

	if(p->bStereo == TRUE)
		HAL_SETREG(HAL_REG_I2S_tx_ctrl,  (1<<3) | HAL_GETREG(HAL_REG_I2S_tx_ctrl));
	else
		HAL_SETREG(HAL_REG_I2S_tx_ctrl,  (1<<3) | HAL_GETREG(HAL_REG_I2S_tx_ctrl) | 0x02);

	if(p->bStereo == TRUE)
		HAL_SETREG(HAL_REG_I2S_rx_ctrl, ~0x02 & HAL_GETREG(HAL_REG_I2S_rx_ctrl));
	else
		HAL_SETREG(HAL_REG_I2S_rx_ctrl, (~0x02 & HAL_GETREG(HAL_REG_I2S_rx_ctrl)) | 0x20);

	if(p->bMaster == TRUE)
	{
		*pCodecCLK = halSysGetDeviceClockSrc(Fin, HAL_MFP_I2S, NULL)/2;
		HAL_SETREG8(HAL_REG_I2S_CODEC_clk_div, 1);
		HAL_SETREG8(HAL_REG_I2S_CODEC_clk_ctrl, 3);
		halUARTPrintf("I2S Main-clock %d\n\r", *pCodecCLK);

		r = (1 << 12) + (1 << 11) + (1 << 10) + (1 << 9) + (0 << 8);
		r += ((halSysGetDeviceClockSrc(Fin, HAL_MFP_I2S, NULL)/(p->nSamplingRate*32*2) - 1) & 0xFF);
	}
	else
	{
		clk_ctl = HAL_GETREG(HAL_REG_I2S_clock);
		clk_ctl |= (0 << 9) | (1 << 10);
		clk_ctl &= ~((1 << 11) | (1 << 12)) ;
	}
	HAL_SETREG(HAL_REG_I2S_clock, clk_ctl);

	return (TRUE);

}

bool_T halDAISetup(halDAISetting_S *p, unsigned int Fin, unsigned int *pCodecCLK)
{
	/*
	 * We allow I2S mode only.
	 * Precision for one channel-sample * is 16 bits only.
	 * TX always use "unison".
	 */
	register unsigned int r;

	halSysPLLIsMaster(TRUE);

	r = HAL_GETREG(HAL_REG_I2S_init);
	if (r & 6)
		return (FALSE);	/* TX or RX is working now; not allow to program the protocol */

	/* data protocol between DAI and audio AD-DA codec */
	switch (p->nDataProtocol)
	{
		case halcDAII2SMode:
			HAL_SETREG8(HAL_REG_I2S_mode, 4);
			break;
		case halcDAILJustMode:
		case halcDAIRJustMode:
		case halcDAIMSBExtMode:
		case halcDAIDSPMode:
		/* driver doesn't supported yet */
			return (FALSE);
	}

	/* FIFO thresholds */
	HAL_SETREG8(HAL_REG_I2S_tx_fifo_lth, p->nTXFIFOThreshold);
	HAL_SETREG8(HAL_REG_I2S_rx_fifo_gth, p->nRXFIFOThreshold);

	/* Precision for one channel-sample * is 16 bits only. */
	HAL_SETREG8(HAL_REG_I2S_wlen, 16-1);
	HAL_SETREG8(HAL_REG_I2S_wpos, 0);

	unsigned int clk, divider;

	/* RX control register */
	r = (p->bMaster == TRUE) ? (1 << 1) : (0 << 1);
	r |= (p->bStereo == TRUE) ? (0x00 << 4) : (0x01 << 4);	// normal or left channel only
	HAL_SETREG8(HAL_REG_I2S_rx_ctrl, r);

	/* TX control register */
	r = (1 << 3);	/* transmiter unison */
	r |= (p->bMaster == TRUE) ? (1 << 5) : (0 << 5);
	r |= (p->bStereo == TRUE) ? (0x00 << 0) : (0x01 << 0);	// stereo or same data from sd_0
	HAL_SETREG8(HAL_REG_I2S_tx_ctrl, r);

	//
	// set I2S clock source is 12.288MHz
	divider = PLL_GetFout(1, Fin) / (12.288 * 1000 * 1000);
	clk = halSysSetDeviceClockDivider(Fin, HAL_MFP_I2S, divider);
	//sdbvPrintf_g("\r\nSet I2S clock source to %d (by divider=%d)\r\n", clk, divider);

	*pCodecCLK = halSysGetDeviceClockSrc(Fin, HAL_MFP_I2S, NULL);
	HAL_SETREG8(HAL_REG_I2S_CODEC_clk_div, 1);
	HAL_SETREG8(HAL_REG_I2S_CODEC_clk_ctrl, 0);
	//halUARTPrintf("I2S Main-clock %d\n\r", *pCodecCLK);

	if(p->bMaster == TRUE)
	{
		r = (1 << 12) + (1 << 11) + (1 << 10) + (1 << 9) + (0 << 8);
		r += ((halSysGetDeviceClockSrc(Fin, HAL_MFP_I2S, NULL)/(p->nSamplingRate*32*2) - 1) & 0xFF);
		HAL_SETREG(HAL_REG_I2S_clock, r);
	}
	else
	{
		r = (0 << 12) + (0 << 11) + (0 << 10) + (0 << 9) + (0 << 8);
		HAL_SETREG(HAL_REG_I2S_clock, r | (1 << 10));	//ted: should set bit 10 to 1. why???
		halDelay(HAL_TICKS_PER_SEC/10);					//ted: after set bit 10 to 1 should delay a while. why???
		HAL_SETREG(HAL_REG_I2S_clock, r);
	}

	return (TRUE);

}
#endif

