/*******************************************************************************
 (c) Copyright 2010, ACTi Corporation, Inc. ALL RIGHTS RESERVED

 All software are Copyright 2010 by ACTi Corporation. ALL RIGHTS RESERVED.
 Redistribution and use in source and binary forms, with or
 without modification, are strictly prohibited.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY EXPRESS
 OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

#include <common.h>
#include <command.h>

#include <lwip/netif.h>
#include <lwip/init.h>
//#include <lwip/dhcp.h>
#include <lwip/tcp.h>
#include <lwip/tcp_impl.h>
#include <lwip/ip_frag.h>
#include <netif/etharp.h>


//extern void  dma_memorypool_reset(void);
extern void* memory_set(void *p, int c, unsigned int n);

extern void timer2_start(unsigned int interval/* in ms */);
extern void timer2_stop(void);
extern unsigned int timer2_now(void);

extern err_t umvp2500_if_init(struct netif *netif);
extern void umvp2500_if_poll(struct netif *netif);


enum {
	TIMER_INTERVAL             = 10,
	ETHARP_TIMER_INTERVAL      = ARP_TMR_INTERVAL / TIMER_INTERVAL,
	TCP_TIMER_INTERVAL         = TCP_TMR_INTERVAL / TIMER_INTERVAL,
	IPREASS_TIMER_INTERVAL     = IP_TMR_INTERVAL / TIMER_INTERVAL,
	//DHCP_COARSE_TIMER_INTERVAL = DHCP_COARSE_TIMER_SECS * 1000 / TIMER_INTERVAL,
	//DHCP_FINE_TIMER_INTERVAL   = DHCP_FINE_TIMER_MSECS / TIMER_INTERVAL,
	ECHO_SERVER_IP_ADDRESS     = (172 << 24) | (16 << 16) | (14 << 8) | 65,  /* 172.16.14.65 */
	ECHO_SERVER_PORT           = 8888,
	ECHO_CLIENT_IP_ADDRESS     = (172 << 24) | (16 << 16) | (14 << 8) | 87,  /* 172.16.14.87 */
	ECHO_CLIENT_NETMASK        = (255 << 24) | (255 << 16) | (255 << 8) | 0, /* 255.255.255.0 */
	ECHO_CLIENT_GATEWAY        = (172 << 24) | (16 << 16) | (14 << 8) | 253, /* 172.16.14.253 */
	ECHO_CLIENT_INITIALIZED    = 0,
	ECHO_CLIENT_CONNECTING     = 1,
	ECHO_CLIENT_CONNECTED      = 2,
	ECHO_CLIENT_SENDING        = 3,
	ECHO_CLIENT_RECEIVING      = 4,
	ECHO_CLIENT_RECEIVED       = 5,
};

struct echo_client
{
	unsigned int     state;
	unsigned int     sleep;
	unsigned int     limit;
	unsigned int     count;

	struct tcp_pcb*  tcp;
	int (*execute)(struct echo_client* client);
};


static char* ECHO_PHARSE = "Hello ";


static int do_client_connected(struct echo_client* client);
static int do_client_received(struct echo_client* client);

static err_t echo_connected(void *arg, struct tcp_pcb *tpcb, err_t err)
{
	struct echo_client* client = (struct echo_client*)arg;

	client->execute = do_client_connected;

	printf("Connects to server\n");

	return ERR_OK;
}

static err_t echo_sent(void *arg, struct tcp_pcb *tpcb, u16_t len)
{
	struct echo_client* client = (struct echo_client*)arg;

	printf("Echo pharse sent\n");

	client->execute = NULL;

	return ERR_OK;
}

static err_t echo_received(void *arg, struct tcp_pcb *tpcb,
						   struct pbuf *p, err_t err)
{
	struct echo_client* client = (struct echo_client*)arg;

	if (p == NULL) return ERR_OK;

	client->execute = do_client_received;

	printf("Response received(%u): %s\n", p->len, (char*)p->payload);

	/* Do Something */
	//printf("Response(%d): %s\n", p->len, (char*)p->payload);

	tcp_recved(client->tcp, p->len);

	return ERR_OK;
}

static int do_client_prepare(struct echo_client* client)
{
	struct ip_addr server;

	IP4_ADDR(&server, 172, 16, 14, 65);
	tcp_connect(client->tcp, &server, ECHO_SERVER_PORT,
				echo_connected);

	printf("Connecting to server\n");

	client->execute = NULL;

	return 0;
}

static int do_client_connected(struct echo_client* client)
{
	if (client->count >= client->limit) return -1;

	if (tcp_sndbuf(client->tcp) < 6) {
		printf("TCP sand buffer too small\n");
		return -1;
	}

	printf("Sending echo pharse\n");

	tcp_write(client->tcp, ECHO_PHARSE, 6, TCP_WRITE_FLAG_COPY);
	tcp_output(client->tcp);

	client->execute = NULL;

	++client->count;

	return 0;
}

static int do_client_received(struct echo_client* client)
{
	unsigned int now;

	now = timer2_now();
	printf("Sleeping\n");
	while ((timer2_now() - now) < client->sleep) { }

	client->execute = do_client_connected;

	return 0;
}

static int echo_client_initialize(struct echo_client* client)
{
	memory_set(client, 0, sizeof(*client));

	client->state = ECHO_CLIENT_INITIALIZED;
	client->sleep = 1000 / TIMER_INTERVAL;
	client->limit = 1;
	client->count = 0;

	client->tcp = tcp_new();
	if (client->tcp == NULL) return -1;

	tcp_arg(client->tcp, client);
	tcp_sent(client->tcp, echo_sent);
	tcp_recv(client->tcp, echo_received);

	client->execute = do_client_prepare;

	return 0;
}

static void echo_client_finalize(struct echo_client* client)
{
	if (client->tcp != NULL) {
		tcp_close(client->tcp);
	}
}

int do_tcp_echo(cmd_tbl_t *cmdtp, int flag, int argc, char** argv)
{
	static char* _progress = "|/-\\";

	unsigned int now;
	unsigned int ts_etharp;
    unsigned int ts_tcp;
    unsigned int ts_ipreass;
	//unsigned int ts_dhcp_coarse;
	//unsigned int ts_dhcp_fine;

	struct netif net_dev;
	struct ip_addr ip, gw, mask/*, server*/;

	//unsigned int count;
	struct echo_client echo;

	//dma_memorypool_reset();

	timer2_start(TIMER_INTERVAL /* ms */);

	// Initialize LWIP
	lwip_init();

	//count = 0;
	if (echo_client_initialize(&echo) != 0) return -1;

	// Add our netif to LWIP (netif_add calls our driver initialization function)
	IP4_ADDR(&ip,   172, 16, 14, 87);
	IP4_ADDR(&gw,   172, 16, 14, 253);
	IP4_ADDR(&mask, 255, 255, 255, 0);
	if (netif_add(&net_dev, &ip, &mask, &gw, NULL,
				  umvp2500_if_init, ethernet_input) == NULL)
	{
        printf("netif_add() failed\n");
        return -1;
    }

	netif_set_default(&net_dev);
	netif_set_up(&net_dev);
	//dhcp_start(&net_dev);

	ts_etharp = ts_tcp = ts_ipreass =
		/* ts_dhcp_coarse = ts_dhcp_fine = */ timer2_now();

	while (1) {
		// Call network interface to process incoming packets and do housekeeping
		umvp2500_if_poll(&net_dev);

		// Processes lwip network-related timers.
		now = timer2_now();
		if ((now - ts_etharp) >= ETHARP_TIMER_INTERVAL) {
			etharp_tmr();
			ts_etharp = now;
		}
/*
		if ((now - ts_dhcp_coarse) >= DHCP_COARSE_TIMER_INTERVAL) {
			dhcp_coarse_tmr();
			ts_dhcp_coarse = now;
		}
		if ((now - ts_dhcp_fine) >= DHCP_FINE_TIMER_INTERVAL) {
			dhcp_fine_tmr();
			ts_dhcp_fine = now;
		}
*/
		if ((now - ts_tcp) >= TCP_TIMER_INTERVAL) {
			tcp_tmr();
			ts_tcp = now;
		}
		if ((now - ts_ipreass) >= IPREASS_TIMER_INTERVAL) {
			ip_reass_tmr();
			ts_ipreass = now;
		}

		//if (net_dev.dhcp->offered_ip_addr.addr == 0) continue;

		if (echo.execute == NULL) {
			printf("\rtcp-echo: %c", _progress[now & 0x3]);
			continue;
		}
		if (echo.execute(&echo) != 0) break;
	}

	echo_client_finalize(&echo);
	timer2_stop();

	return 0;
}


U_BOOT_CMD(
	tcp_echo,	CONFIG_SYS_MAXARGS,	0,	do_tcp_echo,
	"Obtains IP adress from DHCP server",
	""
);
