/*******************************************************************************
 (c) Copyright 2010, ACTi Corporation, Inc. ALL RIGHTS RESERVED

 All software are Copyright 2010 by ACTi Corporation. ALL RIGHTS RESERVED.
 Redistribution and use in source and binary forms, with or
 without modification, are strictly prohibited.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY EXPRESS
 OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

#include <common.h>
#include <command.h>

#include <lwip/netif.h>
#include <lwip/init.h>
#include <lwip/dhcp.h>
#include <lwip/tcp.h>
#include <lwip/tcp_impl.h>
#include <lwip/ip_frag.h>
#include <netif/etharp.h>


extern void  dma_memorypool_reset(void);
extern void* memory_set(void *p, int c, unsigned int n);

extern void timer2_start(unsigned int interval/* in ms */);
extern void timer2_stop(void);
extern unsigned int timer2_now(void);

extern err_t umvp2500_if_init(struct netif *netif);
extern void umvp2500_if_poll(struct netif *netif);


enum {
	TIMER_INTERVAL             = 10,
	ETHARP_TIMER_INTERVAL      = ARP_TMR_INTERVAL / TIMER_INTERVAL,
    TCP_TIMER_INTERVAL         = TCP_TMR_INTERVAL / TIMER_INTERVAL,
    IPREASS_TIMER_INTERVAL     = IP_TMR_INTERVAL / TIMER_INTERVAL,
	DHCP_COARSE_TIMER_INTERVAL = DHCP_COARSE_TIMER_SECS * 1000 / TIMER_INTERVAL,
    DHCP_FINE_TIMER_INTERVAL   = DHCP_FINE_TIMER_MSECS / TIMER_INTERVAL,
};


int do_dhcpc(cmd_tbl_t *cmdtp, int flag, int argc, char** argv)
{
	unsigned int now;
	unsigned int ts_etharp;
    unsigned int ts_tcp;
    unsigned int ts_ipreass;
	unsigned int ts_dhcp_coarse;
	unsigned int ts_dhcp_fine;

	struct netif net_dev;
	struct ip_addr ip, gw, mask;

	dma_memorypool_reset();

	timer2_start(10 /* ms */);

	// Initialize LWIP
	lwip_init();

	memory_set(&ip,   0, sizeof(ip));
	memory_set(&gw,   0, sizeof(gw));
	memory_set(&mask, 0, sizeof(mask));

    // Add our netif to LWIP (netif_add calls our driver initialization function)
    if (netif_add(&net_dev, &ip, &mask, &gw, NULL,
				  umvp2500_if_init, ethernet_input) == NULL)
	{
        printf("netif_add() failed\n");
        return -1;
    }

	netif_set_default(&net_dev);
	dhcp_start(&net_dev);

	ts_etharp = ts_tcp = ts_ipreass =
		ts_dhcp_coarse = ts_dhcp_fine = timer2_now();

	while (1) {
		// Call network interface to process incoming packets and do housekeeping
		umvp2500_if_poll(&net_dev);

		// Processes lwip network-related timers.
		now = timer2_now();
		if ((now - ts_etharp) >= ETHARP_TIMER_INTERVAL) {
			etharp_tmr();
			ts_etharp = now;
		}
		if ((now - ts_dhcp_coarse) >= DHCP_COARSE_TIMER_INTERVAL) {
			dhcp_coarse_tmr();
			ts_dhcp_coarse = now;
		}
		if ((now - ts_dhcp_fine) >= DHCP_FINE_TIMER_INTERVAL) {
			dhcp_fine_tmr();
			ts_dhcp_fine = now;
		}
		if ((now - ts_tcp) >= TCP_TIMER_INTERVAL) {
			tcp_tmr();
			ts_tcp = now;
		}
		if ((now - ts_ipreass) >= IPREASS_TIMER_INTERVAL) {
			ip_reass_tmr();
			ts_ipreass = now;
		}
	}

	timer2_stop();

	return 0;
}


U_BOOT_CMD(
	dhcpc,	CONFIG_SYS_MAXARGS,	0,	do_dhcpc,
	"dhcpc - Obtains IP adress from DHCP server\n",
	"dhcpc\n"
);
