/*
 * Universal Host Controller Interface driver for USB.
 *
 * Maintainer: Alan Stern <stern@rowland.harvard.edu>
 *
 * (C) Copyright 1999 Linus Torvalds
 * (C) Copyright 1999-2002 Johannes Erdfelt, johannes@erdfelt.com
 * (C) Copyright 1999 Randy Dunlap
 * (C) Copyright 1999 Georg Acher, acher@in.tum.de
 * (C) Copyright 1999 Deti Fliegl, deti@fliegl.de
 * (C) Copyright 1999 Thomas Sailer, sailer@ife.ee.ethz.ch
 * (C) Copyright 1999 Roman Weissgaerber, weissg@vienna.at
 * (C) Copyright 2000 Yggdrasil Computing, Inc. (port of new PCI interface
 *               support from usb-ohci.c by Adam Richter, adam@yggdrasil.com).
 * (C) Copyright 1999 Gregory P. Smith (from usb-ohci.c)
 * (C) Copyright 2004-2007 Alan Stern, stern@rowland.harvard.edu
 *
 * Intel documents this fairly well, and as far as I know there
 * are no royalties or anything like that, but even so there are
 * people who decided that they want to do the same thing in a
 * completely different way.
 *
 */

#include <linux/module.h>
#include <linux/pci.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/ioport.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/unistd.h>
#include <linux/interrupt.h>
#include <linux/spinlock.h>
#include <linux/debugfs.h>
#include <linux/pm.h>
#include <linux/dmapool.h>
#include <linux/dma-mapping.h>
#include <linux/usb.h>
#include <linux/bitops.h>
#include <linux/dmi.h>

#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/system.h>

//#include "../core/hcd.h"
#include "uhci-hcd.h"
//#include "pci-quirks.h"

/* for flakey hardware, ignore overcurrent indicators */
static int ignore_oc;

/*
 * debug = 0, no debugging messages
 * debug = 1, dump failed URBs except for stalls
 * debug = 2, dump all failed URBs (including stalls)
 *            show all queues in /debug/uhci/[pci_addr]
 * debug = 3, show all TDs in URBs when dumping
 */
#ifdef DEBUG
#define DEBUG_CONFIGURED	1
static int debug = 1;

#else
#define DEBUG_CONFIGURED	0
#define debug			0
#endif

static char *errbuf;
#define ERRBUF_LEN    (32 * 1024)

static struct kmem_cache *uhci_up_cachep;	/* urb_priv */

void suspend_rh(struct uhci_hcd *uhci, enum uhci_rh_state new_state);
void wakeup_rh(struct uhci_hcd *uhci);
void uhci_get_current_frame_number(struct uhci_hcd *uhci);

// Shirley add from 2.6.17, please check them before porting finished
void uhci_reset_hc(void);
int uhci_check_and_reset_hc(void);
int uhci_suspend(struct usb_hcd *hcd, pm_message_t message);

//////////////////////////////////

/*
 * Calculate the link pointer DMA value for the first Skeleton QH in a frame.
 */
__le32 uhci_frame_skel_link(struct uhci_hcd *uhci, int frame)
{
	int skelnum;

	/*
	 * The interrupt queues will be interleaved as evenly as possible.
	 * There's not much to be done about period-1 interrupts; they have
	 * to occur in every frame.  But we can schedule period-2 interrupts
	 * in odd-numbered frames, period-4 interrupts in frames congruent
	 * to 2 (mod 4), and so on.  This way each frame only has two
	 * interrupt QHs, which will help spread out bandwidth utilization.
	 *
	 * ffs (Find First bit Set) does exactly what we need:
	 * 1,3,5,...  => ffs = 0 => use period-2 QH = skelqh[8],
	 * 2,6,10,... => ffs = 1 => use period-4 QH = skelqh[7], etc.
	 * ffs >= 7 => not on any high-period queue, so use
	 *	period-1 QH = skelqh[9].
	 * Add in UHCI_NUMFRAMES to insure at least one bit is set.
	 */
	skelnum = 8 - (int) __ffs(frame | UHCI_NUMFRAMES);
	if (skelnum <= 1)
		skelnum = 9;
	return LINK_TO_QH(uhci->skelqh[skelnum]);
}

#include "uhci-debug.c"
#include "uhci-q.c"
#include "uhci-hub.c"

/*
 * Finish up a host controller reset and update the recorded state.
 */
void finish_reset(struct uhci_hcd *uhci)
{
	int port;

	/* HCRESET doesn't affect the Suspend, Reset, and Resume Detect
	 * bits in the port status and control registers.
	 * We have to clear them by hand.
	 */
	for (port = 0; port < uhci->rh_numports; ++port)
		writel(0, uhci->regbase+ USBPORTSC1 + (port * 2));

	uhci->port_c_suspend = uhci->resuming_ports = 0;
	uhci->rh_state = UHCI_RH_RESET;
	uhci->is_stopped = UHCI_IS_STOPPED;
	uhci_to_hcd(uhci)->state = HC_STATE_HALT;
	uhci_to_hcd(uhci)->poll_rh = 0;

	uhci->dead = 0;		/* Full reset resurrects the controller */
}

/*
 * Last rites for a defunct/nonfunctional controller
 * or one we don't want to use any more.
 */
void uhci_hc_died(struct uhci_hcd *uhci)
{
	uhci_get_current_frame_number(uhci);
	// uhci_reset_hc(to_pci_dev(uhci_dev(uhci)), uhci->io_addr);
	finish_reset(uhci);
	uhci->dead = 1;

	/* The current frame may already be partway finished */
	++uhci->frame_number;
}

void uhci_reset_hc(void)
{
	/* Reset the HC - this will force us to get a
	 * new notification of any already connected
	 * ports due to the virtual disconnect that it
	 * implies.
	 */

    writel(0,(uhci->regbase + USBINTR));



    writel(USBCMD_GRESET,(uhci->regbase + USBCMD));
    mdelay(50);
    
    writel(0,(uhci->regbase + USBCMD));
    mdelay(10);

}

int uhci_check_and_reset_hc(void)
{ 
    unsigned int cmd, intr;

    printk("uhci_check_and_reset_hc() enter\n");
    cmd = readl((uhci->regbase + USBCMD));
	if ((cmd & USBCMD_RS) || !(cmd & USBCMD_CF) ||
			!(cmd & USBCMD_EGSM)) {
		printk("%s: cmd = 0x%04x\n", __FUNCTION__, cmd);
		goto reset_needed;
	}

	//intr = inw(base + UHCI_USBINTR);
    intr = readl((uhci->regbase + USBINTR));
	if (intr & (~USBINTR_RESUME)) {
		printk("%s: intr = 0x%04x\n", __FUNCTION__, intr);
		goto reset_needed;
	}
	return 0;

reset_needed:
	printk("Performing full reset\n");
	uhci_reset_hc();
	return 1;
}

/*
 * Initialize a controller that was newly discovered or has lost power
 * or otherwise been reset while it was suspended.  In none of these cases
 * can we be sure of its previous state.
 */
void check_and_reset_hc(struct uhci_hcd *uhci)
{
	//if (uhci_check_and_reset_hc(to_pci_dev(uhci_dev(uhci)), uhci->io_addr))
	  if (uhci_check_and_reset_hc())
		finish_reset(uhci);
}

/*
 * Store the basic register settings needed by the controller.
 */
void configure_hc(struct uhci_hcd *uhci)
{
	//printk("configure_hc: 1 regbase = %08x, uhci %p\n", uhci->regbase, uhci);
	/* Set the frame length to the default: 1 ms exactly */
	writel(USBSOF_DEFAULT, uhci->regbase+ USBSOF);
	//printk("configure_hc: 2\n");

	/* Store the frame list base address */
	writel(uhci->frame_dma_handle, uhci->regbase+ USBFLBASEADD);
	//printk("configure_hc: 3\n");
	/* Set the current frame number */
	writel(uhci->frame_number & UHCI_MAX_SOF_NUMBER,
			uhci->regbase+ USBFRNUM);
	//printk("configure_hc: 4\n");

	/* Mark controller as not halted before we enable interrupts */
	uhci_to_hcd(uhci)->state = HC_STATE_SUSPENDED;
	//printk("configure_hc: 5\n");
	mb();

	/* Enable PIRQ */
//	pci_write_config_word(to_pci_dev(uhci_dev(uhci)), USBLEGSUP,
//			USBLEGSUP_DEFAULT);
}

int global_suspend_mode_is_broken(struct uhci_hcd *uhci)
{
	int port;
	const char *sys_info;
	static char bad_Asus_board[] = "A7V8X";

	/* One of Asus's motherboards has a bug which causes it to
	 * wake up immediately from suspend-to-RAM if any of the ports
	 * are connected.  In such cases we will not set EGSM.
	 */
	sys_info = dmi_get_system_info(DMI_BOARD_NAME);
	if (sys_info && !strcmp(sys_info, bad_Asus_board)) {
		for (port = 0; port < uhci->rh_numports; ++port) {
			if (readl(uhci->regbase + USBPORTSC1 + port * 2) &
					USBPORTSC_CCS)
				return 1;
		}
	}

	return 0;
}

void suspend_rh(struct uhci_hcd *uhci, enum uhci_rh_state new_state)
__releases(uhci->lock)
__acquires(uhci->lock)
{
	int auto_stop;
	int int_enable, egsm_enable, wakeup_enable;
	struct usb_device *rhdev = uhci_to_hcd(uhci)->self.root_hub;

	auto_stop = (new_state == UHCI_RH_AUTO_STOPPED);
	dev_dbg(&rhdev->dev, "%s%s\n", __func__,
			(auto_stop ? " (auto-stop)" : ""));

	/* Start off by assuming Resume-Detect interrupts and EGSM work
	 * and that remote wakeups should be enabled.
	 */
	egsm_enable = USBCMD_EGSM;
	uhci->RD_enable = 1;
	int_enable = USBINTR_RESUME;
	wakeup_enable = 1;

	/* In auto-stop mode wakeups must always be detected, but
	 * Resume-Detect interrupts may be prohibited.  (In the absence
	 * of CONFIG_PM, they are always disallowed.)
	 */
	if (auto_stop) {
		//if (!device_may_wakeup(&rhdev->dev))
		//	int_enable = 0;

	/* In bus-suspend mode wakeups may be disabled, but if they are
	 * allowed then so are Resume-Detect interrupts.
	 */
	} else {
#ifdef CONFIG_PM
		if (!rhdev->do_remote_wakeup)
			wakeup_enable = 0;
#endif
	}

	/* EGSM causes the root hub to echo a 'K' signal (resume) out any
	 * port which requests a remote wakeup.  According to the USB spec,
	 * every hub is supposed to do this.  But if we are ignoring
	 * remote-wakeup requests anyway then there's no point to it.
	 * We also shouldn't enable EGSM if it's broken.
	 */
	//if (!wakeup_enable || global_suspend_mode_is_broken(uhci))
	//	egsm_enable = 0;

	/* If we're ignoring wakeup events then there's no reason to
	 * enable Resume-Detect interrupts.  We also shouldn't enable
	 * them if they are broken or disallowed.
	 *
	 * This logic may lead us to enabling RD but not EGSM.  The UHCI
	 * spec foolishly says that RD works only when EGSM is on, but
	 * there's no harm in enabling it anyway -- perhaps some chips
	 * will implement it!
	 */
	//if (!wakeup_enable || resume_detect_interrupts_are_broken(uhci) ||
		//	!int_enable)
		//uhci->RD_enable = int_enable = 0;

	writel(int_enable, uhci->regbase + USBINTR);
	writel(USBCMD_EGSM | USBCMD_CF, uhci->regbase + USBCMD);
	mb();
	//udelay(5);

	/* If we're auto-stopping then no devices have been attached
	 * for a while, so there shouldn't be any active URBs and the
	 * controller should stop after a few microseconds.  Otherwise
	 * we will give the controller one frame to stop.
	 */
	if (!auto_stop && !(readl(uhci->regbase + USBSTS) & USBSTS_HCH)) {
		uhci->rh_state = UHCI_RH_SUSPENDING;
		spin_unlock_irq(&uhci->lock);
		//msleep(1);
		mdelay(1);
		spin_lock_irq(&uhci->lock);
		if (uhci->dead)
			return;
	}
	//if (!(readl(uhci->regbase + USBSTS) & USBSTS_HCH))
	//	dev_warn(uhci_dev(uhci), "Controller not stopped yet!\n");

	uhci_get_current_frame_number(uhci);

	uhci->rh_state = new_state;
	uhci->is_stopped = UHCI_IS_STOPPED;

	/* If interrupts don't work and remote wakeup is enabled then
	 * the suspended root hub needs to be polled.
	 */
	uhci_to_hcd(uhci)->poll_rh = (!int_enable && wakeup_enable);

	uhci_scan_schedule(uhci);
	uhci_fsbr_off(uhci);
}

void stop_uhci(struct uhci_hcd *uhci)
{
    
    suspend_rh(uhci, UHCI_RH_SUSPENDED);
	
}

void start_rh(struct uhci_hcd *uhci)
{
	unsigned int status;

	uhci_to_hcd(uhci)->state = HC_STATE_RUNNING;
	uhci->is_stopped = 0;

	/* Mark it configured and running with a 64-byte max packet.
	 * All interrupts are enabled, even though RESUME won't do anything.
	 */
//	writel(USBCMD_RS | USBCMD_CF | USBCMD_MAXP, uhci->regbase + USBCMD);
	writel(USBINTR_TIMEOUT | USBINTR_RESUME | USBINTR_IOC | USBINTR_SP,
			uhci->regbase + USBINTR);
	writel(0,(uhci->regbase + USBCMD));
	
	status =  readl(uhci->regbase + USBCMD);

	while (status & USBCMD_FGR)
        status = readl((uhci->regbase + USBCMD));
        
    
    writel((USBCMD_RS | USBCMD_CF | USBCMD_MAXP),(uhci->regbase + USBCMD));
	
	mb();
	uhci->rh_state = UHCI_RH_RUNNING;
	uhci_to_hcd(uhci)->poll_rh = 1;
}

void wakeup_rh(struct uhci_hcd *uhci)
__releases(uhci->lock)
__acquires(uhci->lock)
{
	dev_dbg(&uhci_to_hcd(uhci)->self.root_hub->dev,
			"%s%s\n", __func__,
			uhci->rh_state == UHCI_RH_AUTO_STOPPED ?
				" (auto-start)" : "");

	/* If we are auto-stopped then no devices are attached so there's
	 * no need for wakeup signals.  Otherwise we send Global Resume
	 * for 20 ms.
	 */
	if (uhci->rh_state == UHCI_RH_SUSPENDED) {		
		uhci->rh_state = UHCI_RH_RESUMING;
		
	}

	start_rh(uhci);

	/* Restart root hub polling */
	mod_timer(&uhci_to_hcd(uhci)->rh_timer, jiffies);
}

void wakeup_hc(struct uhci_hcd *uhci) {
	
	printk("wake_uhci\n");
	wakeup_rh(uhci);
	
}

irqreturn_t uhci_irq(struct uhci_hcd *hcd,struct pt_regs *regs)
{
	//struct uhci_hcd *uhci = hcd_to_uhci(hcd);
	unsigned short status;

	/*
	 * Read the interrupt status, and write it back to clear the
	 * interrupt cause.  Contrary to the UHCI specification, the
	 * "HC Halted" status bit is persistent: it is RO, not R/WC.
	 */
	status = readl(uhci->regbase + USBSTS);
	if (!(status & ~USBSTS_HCH))	/* shared interrupt, not mine */
		return IRQ_NONE;
	writel(status, uhci->regbase + USBSTS);		/* Clear it */

	if (status & ~(USBSTS_USBINT | USBSTS_ERROR | USBSTS_RD)) {
		if (status & USBSTS_HSE)
			dev_err(uhci_dev(uhci), "host system error \n ");
		if (status & USBSTS_HCPE)
			dev_err(uhci_dev(uhci), "host controller process "
					"error, something bad happened!\n");
		if (status & USBSTS_HCH) {
			spin_lock(&uhci->lock);
			if (uhci->rh_state >= UHCI_RH_RUNNING) {
				dev_err(uhci_dev(uhci),
					"host controller halted, "
					"very bad!\n");
				if (debug > 1 && errbuf) {
					/* Print the schedule for debugging */
					uhci_sprint_schedule(uhci,
							errbuf, ERRBUF_LEN);
					lprintk(errbuf);
				}
				uhci_hc_died(uhci);

				/* Force a callback in case there are
				 * pending unlinks */
				mod_timer(&uhci_to_hcd(uhci)->rh_timer, jiffies);
			}
			spin_unlock(&uhci->lock);
		}
	}

	if (status & USBSTS_RD)
		usb_hcd_poll_rh_status(uhci_to_hcd(hcd));
	else {
		spin_lock(&uhci->lock);
		uhci_scan_schedule(uhci);
		spin_unlock(&uhci->lock);
	}

	return IRQ_HANDLED;
}

/*
 * Store the current frame number in uhci->frame_number if the controller
 * is runnning.  Expand from 11 bits (of which we use only 10) to a
 * full-sized integer.
 *
 * Like many other parts of the driver, this code relies on being polled
 * more than once per second as long as the controller is running.
 */
void uhci_get_current_frame_number(struct uhci_hcd *uhci)
{
	if (!uhci->is_stopped) {
		unsigned delta;

		delta = (readl(uhci->regbase + USBFRNUM) - uhci->frame_number) &
				(UHCI_NUMFRAMES - 1);
		uhci->frame_number += delta;
	}
}

/*
 * De-allocate all resources
 */
void release_uhci(struct uhci_hcd *uhci)
{
	int i;

	if (DEBUG_CONFIGURED) {
		spin_lock_irq(&uhci->lock);
		uhci->is_initialized = 0;
		spin_unlock_irq(&uhci->lock);

		debugfs_remove(uhci->dentry);
	}

	for (i = 0; i < UHCI_NUM_SKELQH; i++)
		uhci_free_qh(uhci, uhci->skelqh[i]);

	uhci_free_td(uhci, uhci->term_td);

	dma_pool_destroy(uhci->qh_pool);

	dma_pool_destroy(uhci->td_pool);

	kfree(uhci->frame_cpu);

	dma_free_coherent(uhci_dev(uhci),
			UHCI_NUMFRAMES * sizeof(*uhci->frame),
			uhci->frame, uhci->frame_dma_handle);
}

int uhci_init(struct usb_hcd *hcd)
{
	
	uhci->rh_numports = 1;
	check_and_reset_hc(uhci);
	
	return 0;
}


/*
 * Allocate a frame list, and then setup the skeleton
 *
 * The hardware doesn't really know any difference
 * in the queues, but the order does matter for the
 * protocols higher up.  The order in which the queues
 * are encountered by the hardware is:
 *
 *  - All isochronous events are handled before any
 *    of the queues. We don't do that here, because
 *    we'll create the actual TD entries on demand.
 *  - The first queue is the high-period interrupt queue.
 *  - The second queue is the period-1 interrupt and async
 *    (low-speed control, full-speed control, then bulk) queue.
 *  - The third queue is the terminating bandwidth reclamation queue,
 *    which contains no members, loops back to itself, and is present
 *    only when FSBR is on and there are no full-speed control or bulk QHs.
 */
int uhci_start(struct usb_hcd *hcd)
{
	//struct uhci_hcd *uhci = hcd_to_uhci(hcd);
	int retval = -EBUSY;
	int i;
	struct dentry *dentry;

	hcd->uses_new_polling = 1;

	spin_lock_init(&uhci->lock);
	setup_timer(&uhci->fsbr_timer, uhci_fsbr_timeout,
			(unsigned long) uhci);
	INIT_LIST_HEAD(&uhci->idle_qh_list);
	init_waitqueue_head(&uhci->waitqh);
	//printk("uhci_start: 1 uhci %p\n", uhci);
	if (DEBUG_CONFIGURED) {
		dentry = debugfs_create_file(hcd->self.bus_name,
				S_IFREG|S_IRUGO|S_IWUSR, uhci_debugfs_root,
				uhci, &uhci_debug_operations);
		if (!dentry) {
			dev_err(uhci_dev(uhci), "couldn't create uhci "
					"debugfs entry\n");
			retval = -ENOMEM;
			goto err_create_debug_entry;
		}
		uhci->dentry = dentry;
	}
	
	uhci->frame = dma_alloc_coherent(uhci_dev(uhci),
			UHCI_NUMFRAMES * sizeof(*uhci->frame),
			&uhci->frame_dma_handle, 0);
	//printk("uhci_start: 2\n");
	if (!uhci->frame) {
		dev_err(uhci_dev(uhci), "unable to allocate "
				"consistent memory for frame list\n");
		goto err_alloc_frame;
	}
	memset(uhci->frame, 0, UHCI_NUMFRAMES * sizeof(*uhci->frame));

	uhci->frame_cpu = kcalloc(UHCI_NUMFRAMES, sizeof(*uhci->frame_cpu),
			GFP_KERNEL);
	//printk("uhci_start: 3\n");
	if (!uhci->frame_cpu) {
		dev_err(uhci_dev(uhci), "unable to allocate "
				"memory for frame pointers\n");
		goto err_alloc_frame_cpu;
	}

	uhci->td_pool = dma_pool_create("uhci_td", uhci_dev(uhci),
			sizeof(struct uhci_td), 16, 0);
	if (!uhci->td_pool) {
		dev_err(uhci_dev(uhci), "unable to create td dma_pool\n");
		goto err_create_td_pool;
	}

	uhci->qh_pool = dma_pool_create("uhci_qh", uhci_dev(uhci),
			sizeof(struct uhci_qh), 16, 0);
	if (!uhci->qh_pool) {
		dev_err(uhci_dev(uhci), "unable to create qh dma_pool\n");
		goto err_create_qh_pool;
	}
	//printk("uhci_start: 4\n");
	uhci->term_td = uhci_alloc_td(uhci);
	if (!uhci->term_td) {
		dev_err(uhci_dev(uhci), "unable to allocate terminating TD\n");
		goto err_alloc_term_td;
	}

	for (i = 0; i < UHCI_NUM_SKELQH; i++) {
		uhci->skelqh[i] = uhci_alloc_qh(uhci, NULL, NULL);
		if (!uhci->skelqh[i]) {
			dev_err(uhci_dev(uhci), "unable to allocate QH\n");
			goto err_alloc_skelqh;
		}
	}
	//printk("uhci_start: 5\n");
	/*
	 * 8 Interrupt queues; link all higher int queues to int1 = async
	 */
	for (i = SKEL_ISO + 1; i < SKEL_ASYNC; ++i)
		uhci->skelqh[i]->link = LINK_TO_QH(uhci->skel_async_qh);
	uhci->skel_async_qh->link = UHCI_PTR_TERM;
	uhci->skel_term_qh->link = LINK_TO_QH(uhci->skel_term_qh);

	/* This dummy TD is to work around a bug in Intel PIIX controllers */
	uhci_fill_td(uhci->term_td, 0, uhci_explen(0) |
			(0x7f << TD_TOKEN_DEVADDR_SHIFT) | USB_PID_IN, 0);
	uhci->term_td->link = UHCI_PTR_TERM;
	uhci->skel_async_qh->element = uhci->skel_term_qh->element =
			LINK_TO_TD(uhci->term_td);
	//printk("uhci_start: 6\n");
	/*
	 * Fill the frame list: make all entries point to the proper
	 * interrupt queue.
	 */
	for (i = 0; i < UHCI_NUMFRAMES; i++) {

		/* Only place we don't use the frame list routines */
		uhci->frame[i] = uhci_frame_skel_link(uhci, i);
	}

	/*
	 * Some architectures require a full mb() to enforce completion of
	 * the memory writes above before the I/O transfers in configure_hc().
	 */
	mb();
	//printk("uhci_start: 7\n");

	configure_hc(uhci);
	//printk("uhci_start: 8\n");
	uhci->is_initialized = 1;
	start_rh(uhci);
	//printk("uhci_start: 9\n");
	return 0;

/*
 * error exits:
 */
err_alloc_skelqh:
	for (i = 0; i < UHCI_NUM_SKELQH; i++) {
		if (uhci->skelqh[i])
			uhci_free_qh(uhci, uhci->skelqh[i]);
	}

	uhci_free_td(uhci, uhci->term_td);

err_alloc_term_td:
	dma_pool_destroy(uhci->qh_pool);

err_create_qh_pool:
	dma_pool_destroy(uhci->td_pool);

err_create_td_pool:
	kfree(uhci->frame_cpu);

err_alloc_frame_cpu:
	dma_free_coherent(uhci_dev(uhci),
			UHCI_NUMFRAMES * sizeof(*uhci->frame),
			uhci->frame, uhci->frame_dma_handle);

err_alloc_frame:
	debugfs_remove(uhci->dentry);

err_create_debug_entry:
	return retval;
}

void uhci_stop(struct usb_hcd *hcd)
{
	//struct uhci_hcd *uhci = hcd_to_uhci(hcd);

	spin_lock_irq(&uhci->lock);
	if (test_bit(HCD_FLAG_HW_ACCESSIBLE, &hcd->flags) && !uhci->dead)
		uhci_hc_died(uhci);
	uhci_scan_schedule(uhci);
	spin_unlock_irq(&uhci->lock);

	del_timer_sync(&uhci->fsbr_timer);
	release_uhci(uhci);
}

#if 0
//alex
static void reset_hc(struct uhci *uhci)
{
//	unsigned int io_addr = uhci->io_addr;

	//printk("reset_hc\n");
	/* Global reset for 50ms */
	writel((__u32)USBCMD_GRESET,(uhci->regbase + (__u32)USBCMD));
	wait_ms(50);
	
	writel(0,(uhci->regbase + USBCMD));
	wait_ms(10);
}


void suspend_hc(struct uhci *uhci)
{
	//printk("suspend_hc\n");
	writel(USBCMD_MAXP,(uhci->regbase + USBCMD)); // disable hc
	udelay(100);
	writel(USBCMD_EGSM,(uhci->regbase + USBCMD));	
	uhci->is_suspended = 1;

}
#else


#if 0
//alex
static void reset_hc(struct uhci_hcd *uhci)
{
//	unsigned int io_addr = uhci->io_addr;

	printk("reset_hc\n");
	/* Global reset for 50ms */
	writel((__u32)USBCMD_GRESET,(uhci->regbase + (__u32)USBCMD));
	//wait_ms(50);
	mdelay(50);
//	interruptible_sleep_on_timeout(&timer_queue, HZ/20);
	
	writel(0,(uhci->regbase + USBCMD));
       //wait_ms(10);
       mdelay(10);
//       interruptible_sleep_on_timeout(&timer_queue, HZ/100);
}
#endif

void suspend_hc(struct uhci_hcd *uhci)
{
	//printk("suspend_hc\n");
	writel(USBCMD_MAXP,(uhci->regbase + USBCMD)); // disable hc
	udelay(100);
	writel(USBCMD_EGSM,(uhci->regbase + USBCMD));	
	uhci->is_suspended = 1;

}

void stop_hc(struct uhci_hcd *uhci)
{
    // Disable all interrupts
    printk("stop_hc()in uhci-hcd.c\n");
    writel(0, (uhci->regbase + USBINTR));
    writel(USBCMD_MAXP,(uhci->regbase + USBCMD)); // disable hc
                
}

#if 0

void fast_wakeup_hc(struct uhci *uhci)
{
	unsigned int status;
	
        writel(USBPORTSC_PR, uhci->regbase + USBPORTSC1);
        mdelay(55); // At least 50 mSec
        writel(0, uhci->regbase + USBPORTSC1);
        writel((USBCMD_RS | USBCMD_CF | USBCMD_MAXP),(uhci->regbase + USBCMD));		
	//printk("fast_wakeup_hc %x %x\n", readl(uhci->regbase + USBCMD), readl(uhci->regbase + USBSTS));
	
	//writel(0,(uhci->regbase + USBCMD));
	
	uhci->is_suspended = 0;

	/* Run and mark it configured with a 64-byte max packet */	
	//writel((USBCMD_RS | USBCMD_CF | USBCMD_MAXP),(uhci->regbase + USBCMD));
	
	//printk("fast_wakeup_hc: %x\n", readl(otg_base+OTG_BCSR1));	
	
	// Re-enable all interrupts, by Macleod 2006.01.23
	writel((USBINTR_TIMEOUT | USBINTR_RESUME | USBINTR_IOC | USBINTR_SP),(uhci->regbase + USBINTR));

	// disable then enable port 1
	writel(0, uhci->regbase + USBPORTSC1);
	writel(USBPORTSC_PE, uhci->regbase + USBPORTSC1); 
	
	//printk("USBPORTSC1: %x\n", readl(uhci->regbase + USBPORTSC1));
	
}



#else
void fast_wakeup_hc(struct uhci_hcd *uhci)
{  
 //   printk("fast_wakeup_hc()\n");  // don't print message while do logo test
    udelay(400);
    writel(USBPORTSC_PR, uhci->regbase + USBPORTSC1);
    mdelay(55);    
    writel(0, uhci->regbase + USBPORTSC1);
    //printk("fast_wakeup_hc do port reset\n");
    writel((USBCMD_RS | USBCMD_CF | USBCMD_MAXP),(uhci->regbase + USBCMD));        
    //printk("fast_wakeup_hc %x %x\n", readl(uhci->regbase + USBCMD), readl(uhci->regbase + USBSTS));
    
//    writel(0,(uhci->regbase + USBCMD));
    
    uhci->is_suspended = 0;

    /* Run and mark it configured with a 64-byte max packet */    
    //writel((USBCMD_RS | USBCMD_CF | USBCMD_MAXP),(uhci->regbase + USBCMD));
    
    //printk("fast_wakeup_hc: %x\n", readl(otg_base+OTG_BCSR1));    
    
    // Re-enable all interrupts, by Macleod 2006.01.23
    writel((USBINTR_TIMEOUT | USBINTR_RESUME | USBINTR_IOC | USBINTR_SP),(uhci->regbase + USBINTR));

    // disable then enable port 1
    writel(0, uhci->regbase + USBPORTSC1);
    writel(USBPORTSC_PE, uhci->regbase + USBPORTSC1); 
    
    //printk("USBPORTSC1: %x\n", readl(uhci->regbase + USBPORTSC1));
    
}

#endif

#endif

#ifdef CONFIG_PM
int uhci_rh_suspend(struct usb_hcd *hcd)
{
	//struct uhci_hcd *uhci = hcd_to_uhci(hcd);
	int rc = 0;

	spin_lock_irq(&uhci->lock);
	if (!test_bit(HCD_FLAG_HW_ACCESSIBLE, &hcd->flags))
		rc = -ESHUTDOWN;
	else if (!uhci->dead)
		suspend_rh(uhci, UHCI_RH_SUSPENDED);
	spin_unlock_irq(&uhci->lock);
	return rc;
}

int uhci_rh_resume(struct usb_hcd *hcd)
{
	//struct uhci_hcd *uhci = hcd_to_uhci(hcd);
	int rc = 0;

	spin_lock_irq(&uhci->lock);
	if (!test_bit(HCD_FLAG_HW_ACCESSIBLE, &hcd->flags))
		rc = -ESHUTDOWN;
	else if (!uhci->dead)
		wakeup_rh(uhci);
	spin_unlock_irq(&uhci->lock);
	return rc;
}
#if 0
int uhci_suspend(struct usb_hcd *hcd, pm_message_t message)
{
	//struct uhci_hcd *uhci = hcd_to_uhci(hcd);
	int rc = 0;

	dev_dbg(uhci_dev(uhci), "%s\n", __FUNCTION__);

	spin_lock_irq(&uhci->lock);
	if (uhci->hc_inaccessible)	/* Dead or already suspended */
		goto done;

	if (uhci->rh_state > UHCI_RH_SUSPENDED) {
		dev_warn(uhci_dev(uhci), "Root hub isn't suspended!\n");
		rc = -EBUSY;
		goto done;
	};

	/* All PCI host controllers are required to disable IRQ generation
	 * at the source, so we must turn off PIRQ.
	 */
// eric	pci_write_config_word(to_pci_dev(uhci_dev(uhci)), USBLEGSUP, 0);
	mb();
	clear_bit(HCD_FLAG_HW_ACCESSIBLE, &hcd->flags);
	uhci->hc_inaccessible = 1;
	hcd->poll_rh = 0;

	/* FIXME: Enable non-PME# remote wakeup? */

done:
	spin_unlock_irq(&uhci->lock);
	return rc;
}

int uhci_resume(struct usb_hcd *hcd)
{
	//struct uhci_hcd *uhci = hcd_to_uhci(hcd);

	dev_dbg(uhci_dev(uhci), "%s\n", __FUNCTION__);

	/* Since we aren't in D3 any more, it's safe to set this flag
	 * even if the controller was dead.  It might not even be dead
	 * any more, if the firmware or quirks code has reset it.
	 */
	set_bit(HCD_FLAG_HW_ACCESSIBLE, &hcd->flags);
	mb();

	if (uhci->rh_state == UHCI_RH_RESET)	/* Dead */
		return 0;
	spin_lock_irq(&uhci->lock);

	/* FIXME: Disable non-PME# remote wakeup? */

	uhci->hc_inaccessible = 0;

	/* The BIOS may have changed the controller settings during a
	 * system wakeup.  Check it and reconfigure to avoid problems.
	 */
	check_and_reset_hc(uhci);
	configure_hc(uhci);

	if (uhci->rh_state == UHCI_RH_RESET) {

		/* The controller had to be reset */
		usb_root_hub_lost_power(hcd->self.root_hub);
		suspend_rh(uhci, UHCI_RH_SUSPENDED);
	}

	spin_unlock_irq(&uhci->lock);

	if (!uhci->working_RD) {
		/* Suspended root hub needs to be polled */
		hcd->poll_rh = 1;
		usb_hcd_poll_rh_status(hcd);
	}
	return 0;
}
#endif
#endif

/* Wait until a particular device/endpoint's QH is idle, and free it */
void uhci_hcd_endpoint_disable(struct usb_hcd *hcd,
		struct usb_host_endpoint *hep)
{
	//struct uhci_hcd *uhci = hcd_to_uhci(hcd);
	struct uhci_qh *qh;

	spin_lock_irq(&uhci->lock);
	qh = (struct uhci_qh *) hep->hcpriv;
	if (qh == NULL)
		goto done;

	while (qh->state != UHCI_QH_STATE_IDLE) {
		++uhci->num_waiting;
		spin_unlock_irq(&uhci->lock);
		wait_event_interruptible(uhci->waitqh,
				qh->state == UHCI_QH_STATE_IDLE);
		spin_lock_irq(&uhci->lock);
		--uhci->num_waiting;
	}

	uhci_free_qh(uhci, qh);
done:
	spin_unlock_irq(&uhci->lock);
}

int uhci_hcd_get_frame_number(struct usb_hcd *hcd)
{
//	struct uhci_hcd *uhci = hcd_to_uhci(hcd);
	unsigned frame_number;
	unsigned delta;

	/* Minimize latency by avoiding the spinlock */
	frame_number = uhci->frame_number;
	barrier();
	delta = (readl(uhci->regbase + USBFRNUM) - frame_number) &
			(UHCI_NUMFRAMES - 1);
	return frame_number + delta;
}

//const char hcd_name[] = "uhci_hcd";

static const struct hc_driver _uhci_driver = {
	.description =		uhci_name,
	.product_desc =		"UHCI Host Controller",
	.hcd_priv_size =	sizeof(struct uhci_hcd),

	/* Generic hardware linkage */
	//.irq =			uhci_irq,
	.irq = 		common_interrupt_handler,
	.flags =		HCD_USB11,

	/* Basic lifecycle operations */
	.reset =		uhci_init,
	.start =		uhci_start,
#ifdef CONFIG_PM	
	//.suspend =			uhci_suspend,
	//.resume =			uhci_resume,
	.bus_suspend =		uhci_rh_suspend,
	.bus_resume =		uhci_rh_resume,
#endif
	.stop =			uhci_stop,

	.urb_enqueue =		uhci_urb_enqueue,
	.urb_dequeue =		uhci_urb_dequeue,

	.endpoint_disable =	uhci_hcd_endpoint_disable,
	.get_frame_number =	uhci_hcd_get_frame_number,

	.hub_status_data =	uhci_hub_status_data,
	.hub_control =		uhci_hub_control,
};

struct uhci_hcd	uhciHcd;

static struct device uhci_dev;

int alloc_uhci(void)
{
	unsigned int *base;
	int retval;
	//struct kobject *kobj;
	
	struct hc_driver	*uhci_driver;
	struct usb_hcd		*_uhci_hcd;	
   
//	printk("alloc_uhci() enter\n");
	retval = -EBUSY;

	uhci_driver = (struct hc_driver *)&_uhci_driver;
	if (!(uhci_driver))
	{
	    printk("alloc_uhci:driver pointer is null\n");
	    return -EINVAL;
	}
    
	if (usb_disabled())
		return -ENODEV;

	device_initialize(&uhci_dev);
	kobject_set_name(&uhci_dev.kobj, "guc_uhci");
	uhci_dev.dma_mask = (u64 *)kzalloc(sizeof(u64), GFP_KERNEL);
	//kobject_add(kobj, &uhci_dev.kobj, "%s", "guc_uhci");
	kobject_add(&uhci_dev.kobj, uhci_dev.kobj.parent, "guc_uhci");
	kobject_uevent(&uhci_dev.kobj, KOBJ_ADD);
 
 	
 	
	_uhci_hcd = usb_create_hcd (uhci_driver, &uhci_dev, uhci_name);
	
//	printk("alloc_uhci(2): uhci_dev:%x, _uhci_hcd.state:%x\n", &uhci_dev, _uhci_hcd->state);
	
	if (!_uhci_hcd) {
		retval = -ENOMEM;
		return retval;
	}

    *(_uhci_hcd->self.controller->dma_mask) = (u64)0xffffffffUL;
    _uhci_hcd->self.controller->coherent_dma_mask = 0xffffffffUL;

    _uhci_hcd->driver = uhci_driver;
    otg->otg_uhci = &uhciHcd; //_uhci_hcd;
    uhci = &uhciHcd;
    uhci->ptr_usb_hcd = (unsigned int *) _uhci_hcd;
    base = ioremap_nocache (UHC_BASE_ADDR, 4096);
    uhciHcd.regbase = (unsigned int *)base;
    uhciHcd.io_addr = (unsigned int)base;

    printk("UHCI Base address is %x, uhci %p\n",(unsigned int)base, _uhci_hcd);


    retval = usb_add_hcd (_uhci_hcd, otg->irq, IRQF_DISABLED | IRQF_SHARED);
    
//    printk("alloc_uhci(3): uhci_dev:%x, _uhci_hcd.state:%x\n", &uhci_dev, _uhci_hcd->state);
    
    if (retval != 0)
		printk("add UHCI to USB host controller list failed!\n");
    return retval;

}

int uhci_ahb_probe(void)
{
    return alloc_uhci();
}
 
int uhci_hcd_init(void)
{
	int retval = -ENOMEM;

	if (usb_disabled())
		return -ENODEV;

	printk(KERN_INFO "uhci_hcd: " DRIVER_DESC "%s\n",
			ignore_oc ? ", overcurrent ignored" : "");
	set_bit(USB_UHCI_LOADED, &usb_hcds_loaded);

	if (DEBUG_CONFIGURED) {
		errbuf = kmalloc(ERRBUF_LEN, GFP_KERNEL);
		if (!errbuf)
			goto errbuf_failed;
		uhci_debugfs_root = debugfs_create_dir("uhci", NULL);
		if (!uhci_debugfs_root)
			goto debug_failed;
	}

	uhci_up_cachep = kmem_cache_create("uhci_urb_priv",
		sizeof(struct urb_priv), 0, 0, NULL);
	if (!uhci_up_cachep)
		goto up_failed;

  /*
	retval = pci_register_driver(&uhci_pci_driver);
	if (retval)
		goto init_failed;
  */
  
	return 0;

//init_failed:
//	kmem_cache_destroy(uhci_up_cachep);

up_failed:
	debugfs_remove(uhci_debugfs_root);

debug_failed:
	kfree(errbuf);

errbuf_failed:

	clear_bit(USB_UHCI_LOADED, &usb_hcds_loaded);
	return retval;
}

void uhci_hcd_cleanup(void) 
{
	//pci_unregister_driver(&uhci_pci_driver);
	kmem_cache_destroy(uhci_up_cachep);
	debugfs_remove(uhci_debugfs_root);
	kfree(errbuf);
	clear_bit(USB_UHCI_LOADED, &usb_hcds_loaded);
}
