/*
 * Author	: Leepipi@GUC
 * Date		: 2009/05/25
 * Description	: LCM Device Driver for UMVP3000
 */

#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/cdev.h>
#include <linux/stat.h>
#include <linux/moduleparam.h>
#include <mach/hardware.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <mach/lcm.h>

#if 0

#define LCM_BASE        IO_ADDRESS(UMVP_LCM_BASE)
#define LCM_CTRL_START  (LCM_BASE+0x00)
#define LCM_DATA_START  (LCM_BASE+0x04)
#define LCM_CTRL1       (LCM_BASE+0x08)
#define LCM_CTRL2       (LCM_BASE+0x0C)
#define LCM_DATA1       (LCM_BASE+0x20)
#define LCM_DATA2       (LCM_BASE+0x24)
#define LCM_DATA3       (LCM_BASE+0x28)
#define LCM_DATA4       (LCM_BASE+0x2C)
#define LCM_DATA5       (LCM_BASE+0x30)
#define LCM_DATA6       (LCM_BASE+0x34)
#define LCM_DATA7       (LCM_BASE+0x38)
#define LCM_DATA8       (LCM_BASE+0x3C)

#define UMVP_LCM_READ_REG(reg)	(*((volatile unsigned int *)(reg)))
#define UMVP_LCM_WRITE_REG(reg, value)	(*((volatile unsigned int *)(reg))) = ((unsigned int)(value))

#endif

#define UMVP_LCM_MAJOR	10
#define UMVP_LCM_MINOR	241
static int lcm_major = UMVP_LCM_MAJOR;
static int lcm_minor = UMVP_LCM_MINOR;

struct lcm_dev_t {
	struct cdev	*cdev;
	unsigned char 	row;
	unsigned char	col; 	
	char		buf[16];
};

struct lcm_dev_t *plcm_dev;

char lcm_read_one_data(unsigned char nRow, unsigned char nCol)
{
	unsigned char 	nRegOffset, nDataOffset;
	unsigned int 	nData;

	nRow &= 1;
	nCol &= 0xf;

	nRegOffset = nCol/4;
	nDataOffset= nCol%4;

	if(nRow)
		nData = UMVP_LCM_READ_REG(LCM_DATA1+nRegOffset*4 + 0x10);
	else
		nData = UMVP_LCM_READ_REG(LCM_DATA1+nRegOffset*4);

	switch(nDataOffset)
	{
		case 0:
			nData = ((nData>>24) & 0xff);
			break;
		case 1:
			nData = ((nData>>16) & 0xff);
			break;
		case 2:
			nData = ((nData>>8) & 0xff);
			break;
		case 3:
			nData = ((nData>>0) & 0xff);
			break;
	}

	return nData;
}

ssize_t lcm_dev_read(struct file *filp, 
				char __user *buf, 
				size_t count, 
				loff_t *offset ) 
{
	unsigned int 	i;
	unsigned char	nCol = 0; 

	while((count + nCol) > 16) {
		if(--count == 0)
			break;
	}
	//printk(KERN_INFO "LCM_READ_DATA... \n");
	for(i=0; i<count; i++) {
		plcm_dev->buf[i] = lcm_read_one_data(plcm_dev->row, nCol++);
	}
	copy_to_user(buf, plcm_dev->buf, count);

	return 0;
}

void lcm_write_one_data(unsigned char nRow, unsigned char nCol, char cData)
{
	unsigned char 	nRegOffset,nDataOffset;
	unsigned int 	nData;

	nRow &= 1;
	nCol &= 0xf;
	
	nRegOffset = nCol/4;
	nDataOffset= nCol%4;
			
	if(nRow)
		nData = UMVP_LCM_READ_REG(LCM_DATA1+nRegOffset*4 + 0x10);
	else
		nData = UMVP_LCM_READ_REG(LCM_DATA1+nRegOffset*4);

	switch(nDataOffset)
	{
		case 0:
			nData &= 0x00ffffff;
			nData |= (cData<<24);
			break;
		case 1:
			nData &= 0xff00ffff;
			nData |= (cData<<16);
			break;
		case 2:
			nData &= 0xffff00ff;
			nData |= (cData<<8);
			break;
		case 3:
			nData &= 0xffffff00;
			nData |= cData;
			break;
	}
	if(nRow)
		UMVP_LCM_WRITE_REG(LCM_DATA1+nRegOffset*4 + 0x10, nData);
	else
		UMVP_LCM_WRITE_REG(LCM_DATA1+nRegOffset*4, nData);
}

ssize_t lcm_dev_write(struct file *filp, 
				const char __user *pbuf, 
				size_t count,
				loff_t *offset )
{
	unsigned int 	i;
	unsigned char	nCol = plcm_dev->col; 
	while((count + nCol) > 16) {
		if(--count == 0)
			break;
	}
//	printk(KERN_INFO "LCM_WRITE_DATA... \n");
	copy_from_user(plcm_dev->buf, pbuf, count);
	for(i=0; i<count; i++) {
		lcm_write_one_data(plcm_dev->row, nCol++, plcm_dev->buf[i]);
	}
	UMVP_LCM_WRITE_REG(LCM_DATA_START, 1);

	return 0;
}

void lcm_cls_one_row(unsigned char nRow)
{
	unsigned int i;
	char whitespace[] = "                ";

	for(i=0; i<sizeof(whitespace)-1; i++) {
		lcm_write_one_data(nRow, i, whitespace[i]);
	}
	UMVP_LCM_WRITE_REG(LCM_DATA_START, 1);
}

int lcm_dev_ioctl(struct inode *inode,
				struct file *filp, 
				unsigned int cmd,
				unsigned long arg )
{
	unsigned char i;
	switch(cmd)
	{
		case LCM_INIT_CMD:
			printk(KERN_INFO "LCM initializing... \n");
			UMVP_LCM_WRITE_REG(LCM_CTRL1, 0x05014015);
			UMVP_LCM_WRITE_REG(LCM_CTRL2, 0x0C006000);
			UMVP_LCM_WRITE_REG(LCM_CTRL_START, 0x00000000);
			for(i=0; i<8; i++) {
				UMVP_LCM_WRITE_REG(LCM_DATA1+i*4, 0x20202020);
			}
			UMVP_LCM_WRITE_REG(LCM_DATA_START, 1);
			break;
		case LCM_ROW_CMD:
			plcm_dev->row = ((unsigned char)arg & 1);
			//printk(KERN_INFO "Set LCM row=%d \n", plcm_dev->row);
			break;
		case LCM_COL_CMD:
			plcm_dev->col = (unsigned char)arg;
			break;
		case LCM_CLEAR_CMD:
			lcm_cls_one_row(plcm_dev->row);
			//printk(KERN_INFO "Clear LCM Row-%d \n", plcm_dev->row);
			break;
		default:
			return -EINVAL;	
	}
	return 0;
}
#if 1
int lcm_dev_open(struct inode *inode, struct file *filp) 
{
	filp->private_data = plcm_dev; 
	printk(KERN_INFO "LCM Device Open! \n");
	return 0;
}
#endif
static const struct file_operations lcm_fops = {
	.owner		= THIS_MODULE,
	.llseek		= NULL,
	.read		= lcm_dev_read,
	.write		= lcm_dev_write,
	.ioctl		= lcm_dev_ioctl,
	.open		= lcm_dev_open,
};


void lcm_cdev_setup(struct lcm_dev_t *dev, int index)
{
	int err, devno;
	devno = MKDEV(lcm_major, index);

	dev->cdev = cdev_alloc();
	kobject_set_name(&dev->cdev->kobj, DEV_NAME);
	cdev_init(dev->cdev, &lcm_fops);
	dev->cdev->owner = THIS_MODULE;
	dev->cdev->ops	= &lcm_fops;
	dev->cdev->dev	= devno;
	err = cdev_add(dev->cdev, devno, 1);
	if(err) {
		printk(KERN_NOTICE "Error %d adding LCM %d in Kernel! \n",
				err, index);
	}	
	printk(KERN_INFO "LCM is added to Kernel:\n major: %d, minor:%d\n ",
				 MAJOR(devno), MINOR(devno));
}

static int __init lcm_dev_init(void)
{
	int ret;
	dev_t devno = MKDEV(lcm_major, lcm_minor);
	
	printk("UMVP LCM Driver Init \n");
	ret = register_chrdev_region(devno, 1, DEV_NAME);
	if(ret < 0)
		printk("Failed to register LCM! \n");
	plcm_dev = kmalloc(sizeof(struct lcm_dev_t), GFP_KERNEL);
	if(!plcm_dev) {
		ret = -ENOMEM;
		goto fail;
	}
	memset(plcm_dev, 0, sizeof(struct lcm_dev_t));
	lcm_cdev_setup(plcm_dev, lcm_minor);

	return 0;
fail:
	printk("Failed to allocate MEMORY for LCM Driver! \n");
	unregister_chrdev_region(devno, 1);
	return ret;
}

static void __exit lcm_dev_exit(void)
{
	cdev_del(plcm_dev->cdev);
	kfree(plcm_dev);
	unregister_chrdev_region(MKDEV(lcm_major, 0), 1);
}
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Leepipi <hugo.lee@globalunichip.com>");

module_param(lcm_major, int, S_IRUGO);

module_init(lcm_dev_init);
module_exit(lcm_dev_exit);
