/*
 *  linux/arch/arm/arch-umvp/umvp3000_timer.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/device.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/irq.h>

#include <asm/system.h>
#include <asm/io.h>
#include <mach/hardware.h>
#include <mach/platform2500.h>
#include <mach/irqs.h>
#include <asm/mach/time.h>

static unsigned long A1_timer_reload;
static unsigned long A1_ticks_per_usec;

#define TICKS2USECS(x)		(((x) / A1_ticks_per_usec) << 1)

#define TIMER0_ENABLE   	0x0001

#define VIC_VA_BASE			(IO_ADDRESS(UMVP_VIC_BASE))
#define TIMER0_VA_BASE		(IO_ADDRESS(UMVP_TIMER_BASE) + UMVP_TIMER0_OFFSET)
#define TIMERC_VA_BASE   	(IO_ADDRESS(UMVP_TIMER_BASE) + UMVP_TIMER_CTRL_OFFSET)

typedef struct TimerStruct {
    unsigned long TimerValue;
    unsigned long TimerLoad;
    unsigned long TimerMatch1;
    unsigned long TimerMatch2;
} TimerStruct_t;

extern unsigned int PLL_GetAPBClock(unsigned int Fin);
extern void vic_clear_intr(int irqnr);
/*
 * Returns number of ms since last clock interrupt.  Note that interrupts
 * will have been disabled by do_gettimeoffset()
 */
static unsigned long umvp_gettimeoffset(void)
{
	volatile TimerStruct_t *timer0 = (TimerStruct_t *)TIMER0_VA_BASE;
	unsigned long ticks1, ticks2, status;

	/*
	 * Get the current number of ticks.  Note that there is a race
	 * condition between us reading the timer and checking for
	 * an interrupt.  We get around this by ensuring that the
	 * counter has not reloaded between our two reads.
	 */
	ticks2 = timer0->TimerValue;
	do {
		ticks1 = ticks2;
		status = __raw_readl(VIC_VA_BASE + UMVP_VIC_RAW_STATUS_OFFSET);
		ticks2 = timer0->TimerValue;
	} while (ticks2 > ticks1);

	/*
	 * Number of ticks since last interrupt.
	 */
	ticks1 = A1_timer_reload - ticks2;

	/*
	 * Interrupt pending?  If so, we've reloaded once already.
	 */
	if (status & IRQMASK_TIMERINT0)
		ticks1 += A1_timer_reload;

	/*
	 * Convert the ticks to usecs
	 */
	return TICKS2USECS(ticks1);

}

/*
 * IRQ handler for the timer
 */
static irqreturn_t
umvp_timer_interrupt(int irq, void *dev_id)
{
	/*
	 * clear the interrupt
	 */
	vic_clear_intr(IRQ_TIMERINT0);

	timer_tick();

	return IRQ_HANDLED;
}

static struct irqaction umvp_timer_irq = {
	.name		= "UMVP Timer Tick",
	.flags		= IRQF_DISABLED | IRQF_TIMER | IRQF_IRQPOLL,
	.handler	= umvp_timer_interrupt,
};


/*
 * Set up timer interrupt, and return the current time in seconds.
 */
static void umvp_setup_timer(void)
{
	volatile TimerStruct_t *timer0 = (volatile TimerStruct_t *)TIMER0_VA_BASE;
	volatile __u32         *timerc = (volatile __u32*)TIMERC_VA_BASE;

	/*
	 * Initialise to a known state (all timers off)
	 */
	*timerc = 0;

	A1_timer_reload = PLL_GetAPBClock(A1_SYS_CLOCK_SRC)/HZ;

	/* 
	 * TICKS per uSEC by default is 225/2 * 10^6 / 10^6 = 112.5
	 * We make it left shift by 1 bit and shift it back, later
	 */
	A1_ticks_per_usec = (A1_timer_reload << 1) * HZ / 1000000;
	timer0->TimerLoad    = A1_timer_reload - 1;
	timer0->TimerValue   = A1_timer_reload - 1;
	*timerc              = TIMER0_ENABLE;

	/*
	 * Make irqs happen for the system timer
	 */
	setup_irq(IRQ_TIMERINT0, &umvp_timer_irq);
}

struct sys_timer umvp2500_timer = {
	.init		= umvp_setup_timer,
	.offset		= umvp_gettimeoffset,
};

