/*
 *  linux/arch/arm/mach-umvp/umvp3000_irq.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/module.h>
#include <linux/kernel.h>

#include <linux/init.h>
#include <linux/stddef.h>
#include <linux/list.h>
#include <linux/timer.h>

#include <mach/hardware.h>
#include <mach/platform2500.h>
#include <asm/irq.h>
#include <asm/io.h>

#include <asm/arch-umvp/a1_sys.h>

#include <asm/mach/irq.h>

#ifdef DBG
#define	KDUMP(x...)	do{printk(x);} while(0)
#else
#define	KDUMP(x...)	do { } while (0)
#endif

#define HAL_REG_SYS_BASE             IO_ADDRESS(UMVP_SYSTEM_BASE)
#define HAL_REG_POM_BASE             IO_ADDRESS(UMVP_POM_BASE)

//system registers
#define HAL_REG_SYS_OSCCtrl				(HAL_REG_SYS_BASE+0x00)
#define HAL_REG_SYS_CPUMode				(HAL_REG_SYS_BASE+0x04)
#define HAL_REG_SYS_PMUSts				(HAL_REG_SYS_BASE+0x08)

#define HAL_REG_SYS_AHBGateCtrl			(HAL_REG_SYS_BASE+0x24)
#define HAL_REG_SYS_APBGateCtrl			(HAL_REG_SYS_BASE+0x28)
#define HAL_REG_SYS_CLKDivCnt0			(HAL_REG_SYS_BASE+0x34)
#define HAL_REG_SYS_CLKDivCnt1			(HAL_REG_SYS_BASE+0x38)
#define HAL_REG_SYS_CLKDivCnt2			(HAL_REG_SYS_BASE+0x3c)
#define HAL_REG_SYS_CLKDivCnt3			(HAL_REG_SYS_BASE+0x40)
#define HAL_REG_SYS_CLKDivCnt4			(HAL_REG_SYS_BASE+0x44)
#define HAL_REG_SYS_CLKDivCnt5			(HAL_REG_SYS_BASE+0x48)

//POM registers
#define HAL_REG_PLL0_CTRL				(0x00 + HAL_REG_POM_BASE)
#define HAL_REG_PLL1_CTRL				(0x04 + HAL_REG_POM_BASE)

#define UMVP_READ_REG(r)        (*((volatile unsigned int *) (r)))
#define UMVP_WRITE_REG(r,v)     (*((volatile unsigned int *) (r)) = ((unsigned int)   (v)))
#define HAL_GETREG32	UMVP_READ_REG
#define HAL_SETREG32	UMVP_WRITE_REG

typedef struct
{
	unsigned int nand_cntx:5;
	unsigned int reserved0:3;
	unsigned int usb_cntx:5;
	unsigned int reserved1:3;
	unsigned int sdc_cntx:5;
	unsigned int reserved2:11;
}sys_div_cnt0_t;
typedef struct
{
	unsigned int vcap_cntx:5;
	unsigned int reserved0:11;
	unsigned int isp_cntx:5;
	unsigned int reserved1:11;
}sys_div_cnt1_t;
typedef struct
{
	unsigned int h8290_cntx:5;
	unsigned int reserved0:11;
	unsigned int h9170_cntx:5;
	unsigned int reserved1:11;
}sys_div_cnt2_t;
typedef struct
{
	unsigned int lcd_cntx:8;
	unsigned int sensor_cntx_div:8;
	unsigned int uart4_cntx:8;
	unsigned int sensor_cntx:8;
}sys_div_cnt3_t;
typedef struct
{
	unsigned int uar0_cntx:8;
	unsigned int uar1_cntx:8;
	unsigned int uar2_cntx:8;
	unsigned int uar3_cntx:8;
}sys_div_cnt4_t;
typedef struct
{
	unsigned int i2s_cntx:8;
	unsigned int ssi_cntx:8;
	unsigned int tmr_cntx:8;
	unsigned int wdt_cntx:8;
}sys_div_cnt5_t;

typedef struct pll_ctrl_s
{
	unsigned int by_pass:1;
	unsigned int pd:1;
	unsigned int op_div:2;
	unsigned int bs:2;
	unsigned int lock_det:1;
	unsigned int reserved0:1;
	unsigned int nr:5;
	unsigned int reserved1:11;
	unsigned int nf:7;
	unsigned int update:1;
}pll_ctrl_t;

/* clock source table */
#define hal_wdt_clk_src			(PLL_GetAHBClock(Fin)/2) //PLL_GetFout(1,Fin ) if using external clk
#define hal_tmr_clk_src			(PLL_GetAHBClock(Fin)/2) //PLL_GetFout(1, Fin) if using external clk
#define hal_ssi_clk_src			(PLL_GetAHBClock(Fin)/2)
#define hal_i2s_clk_src			PLL_GetFout(1, Fin)
#define hal_lcd_clk_src			PLL_GetFout(1, Fin)
#define hal_h8290_clk_src			PLL_GetFout(1, Fin)
#define hal_h9170_clk_src			PLL_GetFout(1, Fin)
#define hal_vcap_clk_src			PLL_GetAHBClock(Fin)
#define hal_isp_clk_src			PLL_GetFout(1, Fin)
#define hal_sensor_clk_src		PLL_GetFout(1, Fin)
#define hal_sensor_clk_div_src	halSysGetDeviceClockSrc(Fin, HAL_MFP_ISP, NULL)
#define hal_nand_clk_src			PLL_GetAHBClock(Fin)
#define hal_usb_clk_src			(480*1000*1000)
#define hal_sdc_clk_src			PLL_GetAHBClock(Fin)
#define hal_uart_clk_src			(PLL_GetAHBClock(Fin)/2)
#define hal_smc_clk_src			PLL_GetAHBClock(Fin)
#define hal_i2c_clk_src			(PLL_GetAHBClock(Fin)/2)

void halSysInternalISPEnable(int enable)
{
	unsigned int reg;

	reg = HAL_GETREG32(HAL_REG_SYS_OSCCtrl);
	if(enable)
		reg |= 0x1 << 24;
	else
		reg &= ~(0x1 << 24);
	HAL_SETREG32(HAL_REG_SYS_OSCCtrl, reg);
}
EXPORT_SYMBOL_GPL (halSysInternalISPEnable);

void halSysSetMultiFunctionalPin(int Device)
{
	unsigned int mfs0, mfs1, mfs2, mfs3, mfs4, mfs5;

	mfs0 = HAL_GETREG32(HAL_REG_SYS_MSF0);
	mfs1 = HAL_GETREG32(HAL_REG_SYS_MSF1);
	mfs2 = HAL_GETREG32(HAL_REG_SYS_MSF2);
	mfs3 = HAL_GETREG32(HAL_REG_SYS_MSF3);
	mfs4 = HAL_GETREG32(HAL_REG_SYS_MSF4);
	mfs5 = HAL_GETREG32(HAL_REG_SYS_MSF5);

	switch(Device)
	{
		case HAL_MFP_I2C0:
			mfs4 = (mfs4 & 0xFFFFCFFF) | 0x00000000;
			break;
		case HAL_MFP_I2C1:
			mfs4 = (mfs4 & 0xFFFF3FFF) | 0x00000000;
		    mfs5 = (mfs5 & 0xFFFFFCFF) | 0x00000000;
			break;
		case HAL_MFP_I2C2:
			mfs4 = (mfs4 & 0xF3FFFFFF) | 0x00000000;
		//  mfs5 = (mfs5 & 0xFFFFF3FF) | 0x00000000;
			mfs5 = (mfs5 & 0xFFFFFDFF) | 0x00000000;  //bit9 = 0 select CPU's I2C-2, bit9 = 1 select ISP's I2C-2
			break;
		case HAL_MFP_SSI:	// normal A1 SSI pin mux
			mfs4 = (mfs4 & ~0x01E00000) | 0x00000000;
			break;
		case HAL_MFP_SSI_ISP:	// used for ISP purpose
			mfs4 = (mfs4 & ~0x01E00000) | 0x00200000;	// set SSI CS pin to GPIO, requested by ISP team
			break;
		case HAL_MFP_NAND:
			mfs1 = (mfs1 & 0x000000FF) | 0x00000000;
			mfs2 = 0x00000000;
			mfs3 = (mfs3 & 0xFFFFF000) | 0x00000000;
			mfs4 = (mfs4 & 0xFDFFFFFF) | 0x00000000;
			break;
		case HAL_MFP_SDC:
			mfs3 = (mfs3 & 0xFFE00FFF) | 0x00000000;
			break;
		case HAL_MFP_SMC:
			mfs1 = (mfs1 & 0x000000FF) | 0x55555500;
			mfs2 = 0x55555555;
			mfs3 = (mfs3 & 0xFFFFF000) | 0x00000555;
			mfs4 |= 0x02000000;
			break;
		case HAL_MFP_LCD:
			mfs0 = (mfs0 & ~0x1FFFFFFF) | 0x0AAABFFF;
			mfs1 = (mfs1 & ~0x000000FF) | 0x000000FF;
			mfs5 = (mfs5 & ~0x000000FF) | 0x00000000;
			break;
		case HAL_MFP_WDT:
			mfs4 |= 0x10000000;
			break;
		case HAL_MFP_SENSOR:
			mfs5 |= 0x80000000;
			break;
		case HAL_MFP_ISP:
			halSysInternalISPEnable(1);
			// CMOS interface
			mfs0 = (mfs0 & ~0x001FFFE7) | 0x00000000;
			mfs0 = (mfs0 & ~0x1FE00000) | 0x15400000;
			mfs1 = (mfs1 & ~0x000000FF) | 0x00000000;
			// I2C
			mfs4 = (mfs4 & ~0x0C00C000) | 0x00000000;
			mfs5 = (mfs5 & ~0x00000300) | 0x00000300;
			//SPI
			mfs4 = (mfs4 & ~0xE0000000) | 0x00000000;
			mfs5 = (mfs5 & ~0x00000400) | 0x00000400;
			//GPIO
			mfs5 = (mfs5 & (~0x007f8000)) | 0x7f807fff;
			
			mfs1 = (mfs1 & ~0x000000FF) | 0x00000000;
			mfs3 = (mfs3 & ~0x00000550) | 0x00000550;
			break;
		case HAL_MFP_ISP_DEBUG:
			halSysInternalISPEnable(1);
			// CMOS interface
			mfs0 = (mfs0 & ~0x001FFFE7) | 0x00000000;
			mfs0 = (mfs0 & ~0x1FE00000) | 0x15400000;
			mfs1 = (mfs1 & ~0x000000FF) | 0x00000000;
			// I2C
			mfs4 = (mfs4 & ~0x0C00C000) | 0x00000000;
			mfs5 = (mfs5 & ~0x00000F00) | 0x00000300;
			//SPI
			mfs4 = (mfs4 & ~0xE0000000) | 0x00000000;
			mfs5 = (mfs5 & ~0x00000400) | 0x00000400;
			//GPIO
			mfs5 = (mfs5 & (~0x007f8000)) | 0x7f807fff;
			mfs1 = (mfs1 & ~0x000000FF) | 0x00000000;
			mfs3 = (mfs3 & ~0x00000550) | 0x00000550;
			// for YUV output, for debug
			mfs1 = (mfs1 & ~0x00FC0000) | 0x00FC0000;
			mfs2 = (mfs2 & ~0xFFFFFFF0) | 0xFFFFFFF0;
			mfs3 = (mfs3 & ~0x0000000F) | 0x0000000F;
			break;
		case HAL_MFP_ISP_DEBUG_OFF:
			halSysInternalISPEnable(1);
			// CMOS interface
			mfs0 = (mfs0 & ~0x001FFFE7) | 0x00000000;
			mfs0 = (mfs0 & ~0x1FE00000) | 0x15400000;
			mfs1 = (mfs1 & ~0x000000FF) | 0x00000000;
			// I2C
			mfs4 = (mfs4 & ~0x0C00C000) | 0x00000000;
			mfs5 = (mfs5 & ~0x00000F00) | 0x00000300;
			//SPI
			mfs4 = (mfs4 & ~0xE0000000) | 0x00000000;
			mfs5 = (mfs5 & ~0x00000400) | 0x00000400;
			//GPIO
			mfs5 = (mfs5 & (~0x007f8000)) | 0x7f807fff;	
			mfs1 = (mfs1 & ~0x000000FF) | 0x00000000;
			mfs3 = (mfs3 & ~0x00000550) | 0x00000550;
			// for YUV output, for debug
			mfs1 = (mfs1 & ~0x00FC0000);
			mfs2 = (mfs2 & ~0xFFFFFFF0);
			mfs3 = (mfs3 & ~0x0000000F);
			break;
		case HAL_MFP_EXTERNAL_ISP:
			halSysInternalISPEnable(0);
//			mfs0 = (mfs0 & ~0x01FFFFE7) | 0x00000000;
			mfs0 = (mfs0 & ~0x1FFFFFE7) | 0x00000000;  //R:20130111
			mfs1 = (mfs1 & ~0x000000FF) | 0x00000000;
			mfs5 = (mfs5 & ~0x000000FF) | 0x00000000;
			/* set isp sensor reset pin to gpio4 bit0 */
			mfs5 = (mfs5 & ~0x00800000) | 0x00000000;
			//This is a software workaround for pin mux function failed at X_CMOS_DATA[13]
			//by setting the direction of GPIO0 bit 29 to input mode.
			{
			#define GPIO0_BASE_ADDR		IO_ADDRESS(UMVP_GPIO0_BASE)
			unsigned int gpio0_dir = HAL_GETREG32(GPIO0_BASE_ADDR + 4);
			gpio0_dir = (gpio0_dir & ~0x20000000) | 0x00000000;
			HAL_SETREG32(GPIO0_BASE_ADDR + 4, gpio0_dir);
			}
			break;
			
		case HAL_MFP_GPIO_AP:
		    printk("-->%s: HAL_MFP_GPIO_AP: MFS3(0x%8x)\n", __func__, mfs3);
			//GPIO0:0,1,2,3 and 4,5,6,7
			mfs5 = (mfs5 & ~0x007F8000) | 0x00000000; 	// GPIO0:0,1,2,3,4,5,6,7
			//			
			//GPIO2: 30, 31
			mfs0 = (mfs0 & ~0x00000018) | 0x00000000;
			//GPIO3: 25,26,27,28,30
			mfs3 = (mfs3 & ~0x80000000) | 0x80000000;	// GPIO3:25
//			mfs4 = (mfs4 & ~0x00000017) | 0x00000017;	// GPIO3:26,27,28,30
			mfs4 = (mfs4 & ~0x0000001F) | 0x0000001F;	// GPIO3:26,27,28,29,30  29 is I2C gpio Expander (XRA 1201)
			break;
		case HAL_MFP_GPIO_AP_POWER:
			mfs4 = (mfs4 & ~0x00000004) | 0x00000004;	// GPIO3:28, for Power LED only
			break;
		default:
			break;
	}
	HAL_SETREG32(HAL_REG_SYS_MSF0, mfs0);
	HAL_SETREG32(HAL_REG_SYS_MSF1, mfs1);
	HAL_SETREG32(HAL_REG_SYS_MSF2, mfs2);
	HAL_SETREG32(HAL_REG_SYS_MSF3, mfs3);
	HAL_SETREG32(HAL_REG_SYS_MSF4, mfs4);
	HAL_SETREG32(HAL_REG_SYS_MSF5, mfs5);
}
EXPORT_SYMBOL_GPL(halSysSetMultiFunctionalPin);

void halSysPLLIsMaster(int master_flag)
{
	unsigned int reg;

	reg = UMVP_READ_REG(HAL_REG_SYS_OSCCtrl);
	if(master_flag)
		reg |= 0x1 << 16;
	else
		reg &= ~(0x1 << 16);
	UMVP_WRITE_REG(HAL_REG_SYS_OSCCtrl, reg);
}
EXPORT_SYMBOL_GPL(halSysPLLIsMaster);

unsigned int PLL_GetFout(int pll_no, unsigned int Fin)
{
	pll_ctrl_t reg;
	unsigned int Fout;

	if(pll_no == 0)
		*(unsigned int *)&reg = UMVP_READ_REG(HAL_REG_PLL0_CTRL);
	else
		*(unsigned int *)&reg = UMVP_READ_REG(HAL_REG_PLL1_CTRL);

	Fout = Fin / (reg.nr+1) * (reg.nf+1) / (1 << reg.op_div);

	return Fout;
}

EXPORT_SYMBOL_GPL(PLL_GetFout);

unsigned int PLL_GetAHBClock(unsigned int Fin)
{
	pll_ctrl_t reg;
	unsigned int Fout;

	*(unsigned int *)&reg = UMVP_READ_REG(HAL_REG_PLL0_CTRL);

	Fout = Fin / (reg.nr+1) * (reg.nf+1) / (1 << reg.op_div);

	return Fout/2;
}
EXPORT_SYMBOL_GPL(PLL_GetAHBClock);

unsigned int PLL_GetAPBClock(unsigned int Fin)
{
	return (PLL_GetAHBClock(Fin) / 2);
}
EXPORT_SYMBOL_GPL(PLL_GetAPBClock);


unsigned int halSysGetDeviceClockSrc(unsigned int Fin, int DeviceID, unsigned int *divider)
{
	volatile unsigned int reg;
	volatile sys_div_cnt0_t *cnt0_reg = (sys_div_cnt0_t *)&reg;
	volatile sys_div_cnt1_t *cnt1_reg = (sys_div_cnt1_t *)&reg;
	volatile sys_div_cnt2_t *cnt2_reg = (sys_div_cnt2_t *)&reg;
	volatile sys_div_cnt3_t *cnt3_reg = (sys_div_cnt3_t *)&reg;
//	volatile sys_div_cnt4_t *cnt4_reg = (sys_div_cnt4_t *)&reg;
	volatile sys_div_cnt5_t *cnt5_reg = (sys_div_cnt5_t *)&reg;

	if(divider != NULL) *divider = 0; // default set to 0

	switch(DeviceID)
	{
		case HAL_MFP_WDT:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt5);
//			if(divider != NULL) *divider = cnt5_reg->wdt_cntx+1;
//			return hal_wdt_clk_src/(cnt5_reg->wdt_cntx+1);	// if using external clk
			return hal_wdt_clk_src;
		case HAL_MFP_TMR:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt5);
//			if(divider != NULL) *divider = cnt5_reg->tmr_cntx+1;
//			return hal_tmr_clk_src/(cnt5_reg->tmr_cntx+1); // if using external clk
			return hal_tmr_clk_src;
		case HAL_MFP_SSI:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt5);
			if(divider != NULL) *divider = cnt5_reg->ssi_cntx+1;
			return hal_ssi_clk_src/(cnt5_reg->ssi_cntx+1);
		case HAL_MFP_I2S:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt5);
			if(divider != NULL) *divider = cnt5_reg->i2s_cntx+1;
			return hal_i2s_clk_src/(cnt5_reg->i2s_cntx+1);
		case HAL_MFP_UART0:case HAL_MFP_UART1:case HAL_MFP_UART2:case HAL_MFP_UART3:case HAL_MFP_UART4:
			return hal_uart_clk_src;
		case HAL_MFP_SENSOR:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt3);
			if(divider != NULL) *divider = cnt3_reg->sensor_cntx+1;
			return hal_sensor_clk_src/(cnt3_reg->sensor_cntx+1);
		case HAL_MFP_SENSOR_DIV:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt3);
			if(divider != NULL) *divider = cnt3_reg->sensor_cntx_div+1;
			return hal_sensor_clk_div_src/(cnt3_reg->sensor_cntx_div+1);
		case HAL_MFP_LCD:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt3);
			if(divider != NULL) *divider = cnt3_reg->lcd_cntx+1;
			return hal_lcd_clk_src/(cnt3_reg->lcd_cntx+1);
		case HAL_MFP_H8290:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt2);
			if(divider != NULL) *divider = cnt2_reg->h8290_cntx+1;
			return hal_h8290_clk_src/(cnt2_reg->h8290_cntx+1);
		case HAL_MFP_H9170:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt2);
			if(divider != NULL) *divider = cnt2_reg->h9170_cntx+1;
			return hal_h9170_clk_src/(cnt2_reg->h9170_cntx+1);
		case HAL_MFP_ISP:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt1);
			if(divider != NULL) *divider = cnt1_reg->isp_cntx+1;
			return hal_isp_clk_src/(cnt1_reg->isp_cntx+1);
		case HAL_MFP_VCAP:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt1);
			if(divider != NULL) *divider = cnt1_reg->vcap_cntx+1;
			return hal_vcap_clk_src/(cnt1_reg->vcap_cntx+1);
		case HAL_MFP_SMC:
			return hal_smc_clk_src;
		case HAL_MFP_I2C0:case HAL_MFP_I2C1:case HAL_MFP_I2C2:
			return hal_i2c_clk_src;
		case HAL_MFP_SDC:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt0);
			if(divider != NULL) *divider = cnt0_reg->sdc_cntx+1;
			return hal_sdc_clk_src/(cnt0_reg->sdc_cntx+1);
		case HAL_MFP_USB:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt0);
			if(divider != NULL) *divider = cnt0_reg->usb_cntx+1;
			return hal_usb_clk_src/(cnt0_reg->usb_cntx+1);
		case HAL_MFP_NAND:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt0);
			if(divider != NULL) *divider = cnt0_reg->nand_cntx+1;
			return hal_nand_clk_src/(cnt0_reg->nand_cntx+1);
	}
	return 0;
}
EXPORT_SYMBOL_GPL(halSysGetDeviceClockSrc);


unsigned int halSysSetDeviceClockDivider(unsigned int Fin, int DeviceID, unsigned int DividerValue)
{
	unsigned int ahb_gate, apb_gate;
	volatile unsigned int reg;
	volatile sys_div_cnt0_t *cnt0_reg = (sys_div_cnt0_t *)&reg;
	volatile sys_div_cnt1_t *cnt1_reg = (sys_div_cnt1_t *)&reg;
	volatile sys_div_cnt2_t *cnt2_reg = (sys_div_cnt2_t *)&reg;
	volatile sys_div_cnt3_t *cnt3_reg = (sys_div_cnt3_t *)&reg;
//	volatile sys_div_cnt4_t *cnt4_reg = (sys_div_cnt4_t *)&reg;
	volatile sys_div_cnt5_t *cnt5_reg = (sys_div_cnt5_t *)&reg;

	if(DividerValue <= 0)
		return halSysGetDeviceClockSrc(Fin, DeviceID, NULL);

	ahb_gate = UMVP_READ_REG(HAL_REG_SYS_AHBGateCtrl);
	apb_gate = UMVP_READ_REG(HAL_REG_SYS_APBGateCtrl);
	UMVP_WRITE_REG(HAL_REG_SYS_AHBGateCtrl, 0x87F7EFBF);
	UMVP_WRITE_REG(HAL_REG_SYS_APBGateCtrl, 0xFFF3FFFF);

	switch(DeviceID)
	{
		case HAL_MFP_WDT:
			reg= UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt5);
			cnt5_reg->wdt_cntx = DividerValue - 1;
			UMVP_WRITE_REG(HAL_REG_SYS_CLKDivCnt5, reg);
			break;
		case HAL_MFP_TMR:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt5);
			cnt5_reg->tmr_cntx = DividerValue - 1;
			UMVP_WRITE_REG(HAL_REG_SYS_CLKDivCnt5, reg);
			break;
		case HAL_MFP_SSI:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt5);
			cnt5_reg->ssi_cntx = DividerValue - 1;
			UMVP_WRITE_REG(HAL_REG_SYS_CLKDivCnt5, reg);
			break;
		case HAL_MFP_I2S:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt5);
			cnt5_reg->i2s_cntx = DividerValue - 1;
//			halUARTPrintf("set I2S divider 0x%08x 0x%08x 0x%08x \n\r", reg, cnt5_reg->i2s_cntx, DividerValue);
			UMVP_WRITE_REG(HAL_REG_SYS_CLKDivCnt5, reg);
			break;
		case HAL_MFP_UART0:case HAL_MFP_UART1:case HAL_MFP_UART2:case HAL_MFP_UART3:case HAL_MFP_UART4:
			break;
		case HAL_MFP_LCD:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt3);
			cnt3_reg->lcd_cntx = DividerValue - 1;
			UMVP_WRITE_REG(HAL_REG_SYS_CLKDivCnt3, reg);
			break;
		case HAL_MFP_SENSOR:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt3);
			cnt3_reg->sensor_cntx = DividerValue - 1;
//			halUARTPrintf("set sensor divider 0x%08x\n\r", *(unsigned int *)&cnt3_reg);
			UMVP_WRITE_REG(HAL_REG_SYS_CLKDivCnt3, reg);
			break;
		case HAL_MFP_SENSOR_DIV:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt3);
			cnt3_reg->sensor_cntx_div = DividerValue - 1;
			UMVP_WRITE_REG(HAL_REG_SYS_CLKDivCnt3, reg);
			break;
		case HAL_MFP_H8290:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt2);
			cnt2_reg->h8290_cntx = DividerValue - 1;
			UMVP_WRITE_REG(HAL_REG_SYS_CLKDivCnt2, reg);
			break;
		case HAL_MFP_H9170:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt2);
			cnt2_reg->h9170_cntx = DividerValue - 1;
			UMVP_WRITE_REG(HAL_REG_SYS_CLKDivCnt2, reg);
			break;
		case HAL_MFP_ISP:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt1);
			cnt1_reg->isp_cntx = DividerValue - 1;
			UMVP_WRITE_REG(HAL_REG_SYS_CLKDivCnt1, reg);
			break;
		case HAL_MFP_VCAP:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt1);
			cnt1_reg->vcap_cntx = DividerValue - 1;
			UMVP_WRITE_REG(HAL_REG_SYS_CLKDivCnt1, reg);
			break;
		case HAL_MFP_SMC:
			break;
		case HAL_MFP_I2C0:case HAL_MFP_I2C1:case HAL_MFP_I2C2:
			break;
		case HAL_MFP_SDC:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt0);
			cnt0_reg->sdc_cntx = DividerValue - 1;
			UMVP_WRITE_REG(HAL_REG_SYS_CLKDivCnt0, reg);
			break;
		case HAL_MFP_USB:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt0);
			cnt0_reg->usb_cntx = DividerValue - 1;
			UMVP_WRITE_REG(HAL_REG_SYS_CLKDivCnt0, reg);
			break;
		case HAL_MFP_NAND:
			reg = UMVP_READ_REG(HAL_REG_SYS_CLKDivCnt0);
			cnt0_reg->nand_cntx = DividerValue - 1;
			UMVP_WRITE_REG(HAL_REG_SYS_CLKDivCnt0, reg);
			break;
	}
	UMVP_WRITE_REG(HAL_REG_SYS_AHBGateCtrl, ahb_gate);
	UMVP_WRITE_REG(HAL_REG_SYS_APBGateCtrl, apb_gate);
	return halSysGetDeviceClockSrc(Fin, DeviceID, NULL);
}
EXPORT_SYMBOL_GPL(halSysSetDeviceClockDivider);

