/*
 * ALSA PCM interface for UMVP platform
 *
 * Author:      Betta Lin, <Betta.Lin@acti.com>
 * Copyright:   (C) 2010 ACTI, Inc., 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/dma-mapping.h>
#include <linux/kernel.h>

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>

//#include <asm/dma.h>
#include <asm/arch-umvp/dma.h>

#include <linux/interrupt.h> /* For tasklet_hi_schedule */

#include "umvp-pcm.h"
#include "umvp-i2s.h"

//#define DBG

#ifdef DBG
#define	KDUMP(x...)	printk(x)
#else
#define	KDUMP(x...)	do { } while (0)
#endif

static void umvp_dump(void)
{

	

	KDUMP("DAI_MODE = 0x%04X\n", __raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_MODE) );
	
	KDUMP("DAI_RX_CTRL = 0x%04X\n", __raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_RX_CTRL) );
	
	KDUMP("DAI_TX_CTRL = 0x%04X\n", __raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_TX_CTRL) );
	
	KDUMP("DAI_WLEN = 0x%04X\n", __raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_WLEN) );
	
	KDUMP("DAI_WPOS = 0x%04X\n", __raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_WPOS) );
	
	KDUMP("DAI_SLOT = 0x%04X\n", __raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_SLOT) );
	
	KDUMP("DAI_TX_FIFO_LTH = 0x%04X\n", __raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_TX_FIFO_LTH) );
	
	KDUMP("DAI_RX_FIFO_GTH = 0x%04X\n", __raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_RX_FIFO_GTH) );
	
	KDUMP("DAI_CLOCK = 0x%04X\n", __raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_CLOCK) );
	
	KDUMP("DAI_INIT = 0x%04X\n", __raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_INIT) );
	
	KDUMP("DAI_TX_FIFO_FLAG = 0x%04X\n", __raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_TX_FIFO_FLAG) );
	
	KDUMP("DAI_TX_LEFT_DATA = 0x%04X\n", __raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_TX_LEFT_DATA) );
	
	KDUMP("DAI_TX_RIGHT_DATA = 0x%04X\n", __raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_TX_RIGHT_DATA) );
	
	KDUMP("DAI_RX_FIFO_FLAG = 0x%04X\n", __raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_RX_FIFO_FLAG) );
	
	KDUMP("DAI_RX_DATA = 0x%04X\n", __raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_RX_DATA) );
	
	KDUMP("DAI_TX_FIFO_CNTR = 0x%04X\n", __raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_TX_FIFO_CNTR) );
	
	KDUMP("DAI_RX_FIFO_CNTR = 0x%04X\n", __raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_RX_FIFO_CNTR) );
	
	KDUMP("DAI_CLOCKOUT_CTRL = 0x%04X\n", __raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_CLOCKOUT_CTRL) );
	
	KDUMP("DAI_CLOCKOUT_DIV = 0x%04X\n", __raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_CLOCKOUT_DIV) );
	
}


static struct tasklet_struct  tasklet_write, tasklet_read;

static struct snd_pcm_hardware umvp_pcm_hardware = {
	.info = (SNDRV_PCM_INFO_INTERLEAVED | SNDRV_PCM_INFO_BLOCK_TRANSFER |
		 SNDRV_PCM_INFO_MMAP | SNDRV_PCM_INFO_MMAP_VALID |
		 SNDRV_PCM_INFO_PAUSE),
	.formats = (SNDRV_PCM_FMTBIT_S16_LE),
	.rates = (SNDRV_PCM_RATE_8000 | SNDRV_PCM_RATE_16000 |
		  SNDRV_PCM_RATE_22050 | SNDRV_PCM_RATE_32000 |
		  SNDRV_PCM_RATE_44100 | SNDRV_PCM_RATE_48000 |
		  SNDRV_PCM_RATE_88200 | SNDRV_PCM_RATE_96000 |
		  SNDRV_PCM_RATE_KNOT),
	.rate_min = 8000,
	.rate_max = 96000,
	.channels_min = 1,
	.channels_max = 2,
	.buffer_bytes_max = 128 * 1024,
	.period_bytes_min = 32,
	.period_bytes_max = 64 * 1024,
	.periods_min = 2,
	.periods_max = 255,
	.fifo_size = 0,
};

struct umvp_runtime_data {
	spinlock_t lock;
	int period;		/* current DMA period */
	struct umvp_pcm_dma_params *params;	/* DMA params */
};


static void umvp_pcm_enqueue_dma(struct snd_pcm_substream *substream)
{
	struct umvp_runtime_data *prtd = substream->runtime->private_data;
	struct snd_pcm_runtime *runtime = substream->runtime;

	unsigned int period_size;
	unsigned int dma_offset;
	dma_addr_t dma_pos;

//snd_pcm_lib_buffer_bytes
	period_size = snd_pcm_lib_period_bytes(substream);
	dma_offset = prtd->period * period_size;
	dma_pos = runtime->dma_addr + dma_offset;

	
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {

			prtd->params->dmap.transf_type = memory_to_peripheral;
			prtd->params->dmap.peripheral_type = umvp_dai_tx;
            
			//if (bIsPlayingStereo)
			if (prtd->params->chn_num == 2)
				prtd->params->dmap.transf_width = transf_width_word;
			else
				prtd->params->dmap.transf_width = transf_width_halfword;
            
			prtd->params->dmap.src_addr_inc = 1; /* src is memory; memory address is incremental */
			prtd->params->dmap.dst_addr_inc = 0; /* dst is DAI-TX; memory address is not incremental */
			prtd->params->dmap.transf_bytes = period_size;
			prtd->params->dmap.src_addr_phy = dma_pos;
			prtd->params->dmap.dst_addr_phy = UMVP_DAI_BASE + DAI_TX_LEFT_DATA;
	} else {
				/* start DMA to fill one fragment */
			prtd->params->dmap.transf_type = peripheral_to_memory;
			prtd->params->dmap.peripheral_type = umvp_dai_rx;
			prtd->params->dmap.transf_width = transf_width_halfword;
			prtd->params->dmap.src_addr_inc = 0; /* src is DAI-RX; memory address is not incremental */
			prtd->params->dmap.dst_addr_inc = 1; /* dst is memory; memory address is incremental */
			prtd->params->dmap.transf_bytes = period_size;
			prtd->params->dmap.src_addr_phy = UMVP_DAI_BASE + DAI_RX_DATA;
			prtd->params->dmap.dst_addr_phy = dma_pos;
	}
	
	prtd->period++;
	if (unlikely(prtd->period >= runtime->periods))
		prtd->period = 0;
}


static void write_task(unsigned long arg)
{
	//if (unlikely(ch_status != DMA_COMPLETE))
		//return;
	struct snd_pcm_substream *substream = (struct snd_pcm_substream *)arg;
	struct umvp_runtime_data *prtd = substream->runtime->private_data;

	if (snd_pcm_running(substream)) {
		/*
		update the pcm status for the next period 
		This function is called from the interrupt handler 
		when the PCM has processed the period size. 
		It will update the current pointer, wake up sleepers, etc. 
		*/
		spin_lock(&prtd->lock);
		umvp_pcm_enqueue_dma(substream);
		umvp_dmap_start(prtd->params->channel , &prtd->params->dmap);
		spin_unlock(&prtd->lock);
		snd_pcm_period_elapsed(substream);
	}
	
	KDUMP("tx\n");
}


static void isr_for_tx(unsigned long data)
{

//	printk("I2S TX FIFO status=0x%08x\n", (*((volatile unsigned int *) (IO_ADDRESS(0x9a400000) + 0x28))));
	tasklet_hi_schedule(&tasklet_write);
}

void read_task(unsigned long arg)
{
	struct snd_pcm_substream *substream = (struct snd_pcm_substream *)arg;
	struct umvp_runtime_data *prtd = substream->runtime->private_data;

	if (snd_pcm_running(substream)) {
		spin_lock(&prtd->lock);
		umvp_pcm_enqueue_dma(substream);
		umvp_dmap_start(prtd->params->channel , &prtd->params->dmap);
		spin_unlock(&prtd->lock);
		snd_pcm_period_elapsed(substream);
	}
	
	KDUMP("rx\n");
}


static void isr_for_rx(unsigned long data)
{
	tasklet_hi_schedule(&tasklet_read);

}

static int umvp_pcm_dma_request(struct snd_pcm_substream *substream)
{
	struct umvp_runtime_data *prtd = substream->runtime->private_data;
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct umvp_pcm_dma_params *dma_data = rtd->dai->cpu_dai->dma_data;

	if (!dma_data)
		return -ENODEV;

	prtd->params = dma_data;
	
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
	{
		tasklet_init(&tasklet_write, write_task, (unsigned long) substream);
		
		prtd->params->channel = umvp_dmap_request_some_channel(1, isr_for_tx,0);
//		prtd->params->channel = umvp_dmap_request_channel(isr_for_tx,0);

		if (prtd->params->channel == -1)
		{
			KDUMP("fail to request a DMA channel for TX\n");
			return -ENODEV;
		}
	}
	else
	{
		tasklet_init(&tasklet_read, read_task, (unsigned long)  substream);
#if 1
		prtd->params->channel = umvp_dmap_request_some_channel(0, isr_for_rx,0);
#else
		prtd->params->channel = umvp_dmap_request_channel(isr_for_rx,0);
#endif
		if (prtd->params->channel == -1)
		{
			KDUMP("fail to request a DMA channel for RX\n");
			return -ENODEV;
		}
		KDUMP("dma chn = %d\n", prtd->params->channel);
	}

	return 0;
}

static int umvp_pcm_trigger(struct snd_pcm_substream *substream, int cmd)
{
	struct umvp_runtime_data *prtd = substream->runtime->private_data;
	int ret = 0;

	spin_lock(&prtd->lock);

	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
	case SNDRV_PCM_TRIGGER_RESUME:
	case SNDRV_PCM_TRIGGER_PAUSE_RELEASE:
	/* if dai(i2s) has not not enabled , do not enable dma */
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) 
		{
//betta
			if (__raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_INIT) & (1 << 2) )
			{
				umvp_dmap_start(prtd->params->channel, &prtd->params->dmap);
			}
		}
		else
		{
			if (__raw_readl(IO_ADDRESS(UMVP_DAI_BASE) + DAI_INIT) & (1 << 1) )	
			{
				umvp_dmap_start(prtd->params->channel, &prtd->params->dmap);
				KDUMP("umvp_dmap_start for rx, dma chn = %d\n", prtd->params->channel);
				
			}
		}
		KDUMP("umvp_dmap_is_busy = %d\n", umvp_dmap_is_busy(prtd->params->channel) );
		umvp_dump();
		break;
	case SNDRV_PCM_TRIGGER_STOP:
	case SNDRV_PCM_TRIGGER_SUSPEND:
	case SNDRV_PCM_TRIGGER_PAUSE_PUSH:
		
		//prtd->params->fg_i2s_enable = 0;
		umvp_dmap_stop(prtd->params->channel);
		
		break;
	default:
		ret = -EINVAL;
		break;
	}

	spin_unlock(&prtd->lock);

	return ret;
}

static int umvp_pcm_prepare(struct snd_pcm_substream *substream)
{
	struct umvp_runtime_data *prtd = substream->runtime->private_data;

	prtd->period = 0;
	umvp_pcm_enqueue_dma(substream);

	return 0;
}


static snd_pcm_uframes_t umvp_pcm_pointer(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct umvp_runtime_data *prtd = runtime->private_data;
	
	snd_pcm_uframes_t offset;

	offset = prtd->period * runtime->period_size;
	if (offset >= runtime->buffer_size)
		offset = 0;
	return offset;
}

static int umvp_pcm_open(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct umvp_runtime_data *prtd;
	int ret = 0;
	

	snd_soc_set_runtime_hwparams(substream, &umvp_pcm_hardware);

	prtd = kzalloc(sizeof(struct umvp_runtime_data), GFP_KERNEL);
	if (prtd == NULL)
		return -ENOMEM;

	spin_lock_init(&prtd->lock);

	runtime->private_data = prtd;

	ret = umvp_pcm_dma_request(substream);
	if (ret) {
		KDUMP("umvp_pcm: Failed to get dma channels\n");
		kfree(prtd);
	}

	return ret;
}

static int umvp_pcm_close(struct snd_pcm_substream *substream)
{
	struct snd_pcm_runtime *runtime = substream->runtime;
	struct umvp_runtime_data *prtd = runtime->private_data;


	umvp_dmap_release_channel(prtd->params->channel);
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) 
		tasklet_kill(&tasklet_write);
	else
		tasklet_kill(&tasklet_read);

	
	kfree(prtd);

	return 0;
}

static int umvp_pcm_hw_params(struct snd_pcm_substream *substream,
				 struct snd_pcm_hw_params *hw_params)
{
	struct umvp_runtime_data *prtd = substream->runtime->private_data;
	
	switch (params_channels(hw_params)) {
	case 1:
		prtd->params->chn_num = 1;
		break;
	case 2:
		prtd->params->chn_num = 2;
		break;
	default:
		printk(KERN_WARNING "umvp-i2s: unsupported channel numbers\n");
		return -EINVAL;
	}
	
	return snd_pcm_lib_malloc_pages(substream,
					params_buffer_bytes(hw_params));
}

static int umvp_pcm_hw_free(struct snd_pcm_substream *substream)
{
	return snd_pcm_lib_free_pages(substream);
}

static int umvp_pcm_mmap(struct snd_pcm_substream *substream,
			    struct vm_area_struct *vma)
{
	struct snd_pcm_runtime *runtime = substream->runtime;

	return dma_mmap_writecombine(substream->pcm->card->dev, vma,
				     runtime->dma_area,
				     runtime->dma_addr,
				     runtime->dma_bytes);
				     
}

struct snd_pcm_ops umvp_pcm_ops = {
	.open = 	umvp_pcm_open,
	.close = 	umvp_pcm_close,
	.ioctl = 	snd_pcm_lib_ioctl,
	.hw_params = 	umvp_pcm_hw_params,
	.hw_free = 	umvp_pcm_hw_free,
	.prepare = 	umvp_pcm_prepare,
	.trigger = 	umvp_pcm_trigger,
	.pointer = 	umvp_pcm_pointer,
	.mmap = 	umvp_pcm_mmap,
};

static int umvp_pcm_preallocate_dma_buffer(struct snd_pcm *pcm, int stream)
{
	struct snd_pcm_substream *substream = pcm->streams[stream].substream;
	struct snd_dma_buffer *buf = &substream->dma_buffer;
	size_t size = umvp_pcm_hardware.buffer_bytes_max;

	buf->dev.type = SNDRV_DMA_TYPE_DEV;
	buf->dev.dev = pcm->card->dev;
	buf->private_data = NULL;
	buf->area = dma_alloc_writecombine(pcm->card->dev, size,
					   &buf->addr, GFP_KERNEL);

	KDUMP("umvp_pcm: preallocate_dma_buffer: area=%p, addr=%p, "
		"size=%d\n", (void *) buf->area, (void *) buf->addr, size);

	if (!buf->area)
		return -ENOMEM;

	buf->bytes = size;
	return 0;
}

static void umvp_pcm_free(struct snd_pcm *pcm)
{
	struct snd_pcm_substream *substream;
	struct snd_dma_buffer *buf;
	int stream;

	for (stream = 0; stream < 2; stream++) {
		substream = pcm->streams[stream].substream;
		if (!substream)
			continue;

		buf = &substream->dma_buffer;
		if (!buf->area)
			continue;

		dma_free_writecombine(pcm->card->dev, buf->bytes,
				      buf->area, buf->addr);
		buf->area = NULL;
	}
}

static u64 umvp_pcm_dmamask = 0xffffffff;

static int umvp_pcm_new(struct snd_card *card,
			   struct snd_soc_dai *dai, struct snd_pcm *pcm)
{
	int ret;

	if (!card->dev->dma_mask)
		card->dev->dma_mask = &umvp_pcm_dmamask;
	if (!card->dev->coherent_dma_mask)
		card->dev->coherent_dma_mask = 0xffffffff;

	if (dai->playback.channels_min) {
		ret = umvp_pcm_preallocate_dma_buffer(pcm,
			SNDRV_PCM_STREAM_PLAYBACK);
		if (ret)
			return ret;
	}

	if (dai->capture.channels_min) {
		ret = umvp_pcm_preallocate_dma_buffer(pcm,
			SNDRV_PCM_STREAM_CAPTURE);
		if (ret)
			return ret;
	}

	return 0;
}

struct snd_soc_platform umvp_soc_platform = {
	.name = 	"umvp-audio",
	.pcm_ops = 	&umvp_pcm_ops,
	.pcm_new = 	umvp_pcm_new,
	.pcm_free = 	umvp_pcm_free,
};
EXPORT_SYMBOL_GPL(umvp_soc_platform);

static int __init umvp_soc_platform_init(void)
{
	return snd_soc_register_platform(&umvp_soc_platform);
}
module_init(umvp_soc_platform_init);

static void __exit umvp_soc_platform_exit(void)
{
	snd_soc_unregister_platform(&umvp_soc_platform);
}
module_exit(umvp_soc_platform_exit);

MODULE_AUTHOR("Betta Lin");
MODULE_DESCRIPTION("UMVP PCM DMA module");
MODULE_LICENSE("GPL");
