//================================================================================
// FILE NAME:      umvp_wm8940.c
//--------------------------------------------------------------------------------
// Reason-ID       Reason
//--------------------------------------------------------------------------------
// R:2013114-00:   Create
//================================================================================
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/moduleparam.h>

//#include <linux/delay.h>  //mdeley() msleep()
#include <linux/fs.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/string.h>

#include <linux/slab.h>
#include <linux/init.h>
#include <linux/i2c.h>
#include <linux/i2c-dev.h>

#include "umvp_audio_codec.h"
#include "umvp_wm8940.h"

/* -----------------------------------------------------------------------
    define debug macro
 ----------------------------------------------------------------------- */
static u_int debug_ctrl = 1;
#define MSG_PFX         "[wm8940]"
#define dprintk(num, format, args...) \
    do { \
        if (debug_ctrl >= num) \
            printk(MSG_PFX format, ##args); \
    } while (0)


/*
 * Bias levels
 *
 * @ON:      Bias is fully on for audio playback and capture operations.
 * @PREPARE: Prepare for audio operations. Called before DAPM switching for
 *           stream start and stop operations.
 * @STANDBY: Low power standby state when no playback/capture operations are
 *           in progress. NOTE: The transition time between STANDBY and ON
 *           should be as fast as possible and no longer than 10ms.
 * @OFF:     Power Off. No restrictions on transition times.
 */
enum snd_soc_bias_level {
    SND_SOC_BIAS_ON,
    SND_SOC_BIAS_PREPARE,
    SND_SOC_BIAS_STANDBY,
    SND_SOC_BIAS_OFF,
};
enum {_OFF=0, _ON};

static int au_wm8940_set_output_signal(int onoff);
static int au_wm8940_set_bias_level( enum snd_soc_bias_level level);
static int au_wm8940_write(u_int reg, u_int value);

static u16 wm8940_reg_defaults[] = {
    0x8940, /* 0: Soft Reset */
    0x0000, /* 1: Power 1 */
    0x0000, /* 2: Power 2 */
    0x0000, /* 3: Power 3 */
    0x0010, /* 4: Interface Control */
    0x0000, /* 5: Companding Control */
    0x0140, /* 6: Clock Control */
    0x0000, /* 7: Additional Controls */
    0x0000, /* 8: GPIO Control */
    0x0002, /* 9: Auto Increment Control */
    0x0000, /* 10:DAC Control */
    0x00FF, /* 11:DAC Volume */
    0,      /* 12: */
    0,      /* 13: */
    0x0100, /* 14:ADC Control */
    0x00FF, /* 15:ADC Volume */
    0x0000, /* 16:Notch Filter 1 Control 1 */
    0x0000, /* 17:Notch Filter 1 Control 2 */
    0x0000, /* 18:Notch Filter 2 Control 1 */
    0x0000, /* 19:Notch Filter 2 Control 2 */
    0x0000, /* 20:Notch Filter 3 Control 1 */
    0x0000, /* 21:Notch Filter 3 Control 2 */
    0x0000, /* 22:Notch Filter 4 Control 1 */
    0x0000, /* 23:Notch Filter 4 Control 2 */
    0x0032, /* 24:DAC Limit Control 1 */
    0x0000, /* 25:DAC Limit Control 2 */
    0,      /* 26: */
    0,      /* 27: */
    0,      /* 28: */
    0,      /* 29: */
    0,      /* 30: */
    0,      /* 31: */
    0x0038, /* 32:ALC Control 1 */
    0x000B, /* 33:ALC Control 2 */
    0x0032, /* 34:ALC Control 3 */
    0x0000, /* 35:Noise Gate */
    0x0041, /* 36:PLLN */
    0x000C, /* 37:PLLK1 */
    0x0093, /* 38:PLLK2 */
    0x00E9, /* 39:PLLK3 */
    0,      /* 40: */
    0,      /* 41: */
    0x0030, /* 42:ALC Control 4 */
    0,      /* 43: */
    0x0002, /* 44:Input Control */
    0x0050, /* 45:PGA Gain */
    0,      /* 46: */
    0x0000, /* 47:ADC Boost Control  */
    0,      /* 48: */
    0x0002, /* 49:Output Control */
    0x0000, /* 50:Speaker Mixer Control */
    0,      /* 51: */
    0,      /* 52: */
    0,      /* 53: */
    0x0079, /* 54:Speaker Volume */
    0,      /* 55: */
    0x0000, /* 56:Mono Mixer Control */
};

#if 0
static unsigned int au_wm8940_read(u_int r)
{
    /* address is 8 bits , data is 16 bits */
    struct  i2c_client *client = au_codec.client;
    struct  i2c_msg xfer[2];
    u_char  reg = r & 0xff;
    u_short data;
    int     ret;

    /* Write register */
    xfer[0].addr = client->addr;
    xfer[0].flags = 0;
    xfer[0].len   = 1;
    xfer[0].buf   = &reg;
    ret = i2c_transfer(client->adapter, &xfer[0], 1);

    /* Read data */
    xfer[1].addr = client->addr;
    xfer[1].flags = I2C_M_RD;
    xfer[1].len   = 2;
    xfer[1].buf   = (u_char *)&data;
    ret = i2c_transfer(client->adapter, &xfer[1], 1);
    if (ret < 0) {
        dprintk(1,KERN_ERR "Error: i2c_transfer() returned(%d)\n", ret);
        return 0;
    }

    return (data >> 8) | ((data & 0xff) << 8);
}
void au_wm8940_reg_dump(void)
{
    int reg; u16 value;

    for (reg = 0; reg < ARRAY_SIZE(wm8940_reg_defaults); reg++) {
         value = au_wm8940_read(reg) & 0xFFFF;
         dprintk(0,KERN_INFO "reg %d = 0x%04X\n", reg, value);
    }

}
#endif

/*-----------------------------------------------------------------------------*/
/*
 * wm8940 register cache
 * We can't read the wm8940 register space when we are
 * using 2 wire for device control, so we cache them instead.
 * There is no point in caching the reset register
 */

static inline u_int au_wm8940_read_cache(u_int reg)
{
    u16 *cache = au_codec.reg_cache;

    if (reg == WM8940_SOFTRESET)
        return 0;

    if (reg >= ARRAY_SIZE(wm8940_reg_defaults))
        return -EINVAL;

    dprintk(4,KERN_INFO "ssm2602_read_cache: reg %d = 0x%04X\n", reg, cache[reg]);
    return cache[reg];

}

static inline void au_wm8940_write_cache(u_int reg, u_int value)
{
    u16 *cache = au_codec.reg_cache;

    if (reg >= ARRAY_SIZE(wm8940_reg_defaults))
        return;

    cache[reg] = value;
}

static int au_wm8940_write(u_int reg, u_int value)
{
    struct i2c_client *client = au_codec.client;
    u_char data[3];

    dprintk(4,KERN_INFO "wm8940_write: reg %d = 0x%04X\n", reg, value);

    data[0] = reg;
    data[1] = (value >> 8) & 0xff;
    data[2] = value & 0xff;

    if (i2c_master_send(client, data, 3) == 3) {
        au_wm8940_write_cache(reg, value);
        return 0;
    }else
        return -EIO;

}

int au_wm8940_involume(int value)
{
    u16 pgaVolm = au_wm8940_read_cache(WM8940_PGAGAIN) & ~0x003f;

    /* default ADCVOL is 0dB (0x10) */
    /* set input PGA gain 0~63, -12dB ~ 35.25dB*/
    if ((value < 0)||(value > 63))
        return -EINVAL;

    pgaVolm |= value;
    return au_wm8940_write(WM8940_PGAGAIN, pgaVolm);
}

int au_wm8940_pga_boost(int active)
{
    int ret;
    u16 boost = au_wm8940_read_cache(WM8940_ADCBOOST) & ~0x0100;
    u16 bias_voltage = au_wm8940_read_cache(WM8940_INPUTCTL) & ~0x0100;
    u16 bias_enable  = au_wm8940_read_cache(WM8940_POWER1)   & ~0x0010;

    if (!active){
        boost |= 0x0100;
        bias_voltage |= 0x0100;
        bias_enable  |= 0x0010;
    }
    ret = au_wm8940_write(WM8940_ADCBOOST, boost);
    if (ret < 0) {
        dprintk(0,KERN_ERR "Failed to set ADC boost\n");
    }
    ret = au_wm8940_write(WM8940_INPUTCTL, bias_voltage);
    ret = au_wm8940_write(WM8940_POWER1, bias_enable);
    if (ret < 0) {
        dprintk(0,KERN_ERR "Failed to set MIC bias\n");
        return ret;
    }
    return 0;
}

/* input  path */
int au_wm8940_pga_mute(int mute)
{
    u16 value = au_wm8940_read_cache(WM8940_PGAGAIN) & ~0x0040;

    if (mute)
        value |= 0x40;

    return au_wm8940_write(WM8940_PGAGAIN, value);
}

int au_wm8940_outvolume(int value)
{
    /* Speaker volume adjust */
    u16 volm = au_wm8940_read_cache(WM8940_SPKVOL) & ~0x003F;

    /* default ADCVOL is 0dB (0x39), 1 dB steps */
    /* Speaker volume 0~63, -57dB ~ +6dB*/
    if ((value < 0)||(value > 63))
        return -EINVAL;

    volm |= value;
    return au_wm8940_write(WM8940_SPKVOL, volm);
}

/* output path */
int au_wm8940_dac_mute(int mute)
{
    u16 value = au_wm8940_read_cache(WM8940_DAC) & ~0x0040;

    if (mute)
        value |= 0x40;

    return au_wm8940_write(WM8940_DAC, value);
}

void au_wm8940_stop(void)
{
    dprintk(1,KERN_INFO "%s\n", __func__);
    au_wm8940_dac_mute(_ON);
    /* disable output signal path */
    au_wm8940_set_output_signal(_OFF);
    au_wm8940_set_bias_level(SND_SOC_BIAS_OFF);
}

void au_wm8940_start(void)
{
    dprintk(1,KERN_INFO "%s\n", __func__);
    au_wm8940_set_bias_level(SND_SOC_BIAS_ON);
    /* This is one way so mute dac.  DACMU = 0*/
    au_wm8940_dac_mute(_ON);
}

/* --- Audio interface ----------------------------------------------------------- */
static int au_wm8940_set_dai_clkdiv(int div_id, int div)
{
    u16 reg;
    int ret = 0;

    switch (div_id) {
    case WM8940_BCLKDIV:
        reg = au_wm8940_read_cache (WM8940_CLOCK) & 0xFFE3;
        ret = au_wm8940_write(WM8940_CLOCK, reg | (div << 2));
        break;
    case WM8940_MCLKDIV:
        reg = au_wm8940_read_cache (WM8940_CLOCK) & 0xFF1F;
        ret = au_wm8940_write(WM8940_CLOCK, reg | (div << 5));
        break;
    case WM8940_OPCLKDIV:
        reg = au_wm8940_read_cache (WM8940_GPIO) & 0xFFCF;
        ret = au_wm8940_write(WM8940_GPIO, reg | (div << 4));

        break;
    }
    return ret;
}

static int au_wm8940_set_mclk_ctrl(void)
{
    u16 reg;

    /* Turn off PLL */
    reg = au_wm8940_read_cache(WM8940_POWER1) & 0x1df;
    au_wm8940_write(WM8940_POWER1, reg);

    /* set MCLKDIV divide by 6 */
    au_wm8940_set_dai_clkdiv(WM8940_MCLKDIV, WM8940_MCLKDIV_6);

    /* Clock CODEC directly from MCLK */
    reg = au_wm8940_read_cache(WM8940_CLOCK) & 0x0ff;
    au_wm8940_write(WM8940_CLOCK, reg);

    /* Pll power down */
    reg = au_wm8940_read_cache(WM8940_PLLN) & 0x07f;
    au_wm8940_write(WM8940_PLLN, reg | (1 << 7));

    return 0;
}

static int au_wm8940_set_dai_fmt(hw_params_t *param)
{
    u16 iface = au_wm8940_read_cache(WM8940_IFACE) & 0xFE67;
    u16 clk   = au_wm8940_read_cache(WM8940_CLOCK) & 0xfffe;

    /* set master or slave audio interface */
    switch (param->clk_fmt) {
    case A1_SOC_DAIFMT_CBM_CFM:
         /* cpu is slave; codec clk & frame master */
         /* clk & frame are output generated by WM8940(master) */
         clk |= 1;
         break;
    case A1_SOC_DAIFMT_CBS_CFS:
         /* codec is slave. clk & frm are inputs */
         break;
    default:
         return -EINVAL;
    }
    au_wm8940_write(WM8940_CLOCK, clk);

    /* set audio interface to I2S */
    switch (param->iface_fmt & A1_SOC_DAIFMT_FORMAT_MASK) {
    case A1_SOC_DAIFMT_I2S:
         iface |= (2 << 3);
         break;
    case A1_SOC_DAIFMT_LEFT_J:
         iface |= (1 << 3);
         break;
    case A1_SOC_DAIFMT_RIGHT_J:
         break;
    case A1_SOC_DAIFMT_DSP_A:
         iface |= (3 << 3);
         break;
    case A1_SOC_DAIFMT_DSP_B:
         iface |= (3 << 3) | (1 << 7);
         break;
    default:
        return -EINVAL;
    }

    /* set BCLK polarity */
    switch (param->iface_fmt  & A1_SOC_DAIFMT_INV_MASK) {
    case A1_SOC_DAIFMT_NB_NF:
         break;
    case A1_SOC_DAIFMT_NB_IF:
         iface |= (1 << 7);
         break;
    case A1_SOC_DAIFMT_IB_NF:
         iface |= (1 << 8);
         break;
    case A1_SOC_DAIFMT_IB_IF:
         iface |= (1 << 8) | (1 << 7);
         break;
    default:
        return -EINVAL;
    }
    au_wm8940_write(WM8940_IFACE, iface);
    dprintk(3,KERN_INFO "%s:iface(%x)\n", __func__, iface);

    return 0;
}

static int au_wm8940_hw_params(hw_params_t *param)
{
    u16 iface      = au_wm8940_read_cache(WM8940_IFACE) & 0xFD9F;   /* word lrngth = 16 bits */
    u16 addcntrl   = au_wm8940_read_cache(WM8940_ADDCNTRL) & 0xFFF1;
    u16 companding = au_wm8940_read_cache(WM8940_COMPANDINGCTL) & 0xFFDF;
    int ret;

    /* set approximate sample rate */
    switch (param->rate) {
    case 8000:
        addcntrl |= (0x5 << 1);
        break;
    case 11025:
        addcntrl |= (0x4 << 1);
        break;
    case 16000:
        addcntrl |= (0x3 << 1);
        break;
    case 22050:
        addcntrl |= (0x2 << 1);
        break;
    case 32000:
        addcntrl |= (0x1 << 1);
        break;
    case 44100:
    case 48000:
        break;
    }
    ret = au_wm8940_write(WM8940_ADDCNTRL, addcntrl);
    if (ret)
        goto error_ret;

    /* LoutR control: input mono channel data output on left and right channels */
     iface |= (1 << 9);

    /* interface bit size format */
    switch (param->format) {
    case A1_SOC_FORMAT_S8:
         /* WM8940 support 8 Bit word length */
         companding = companding | (1 << 5);
         break;
    case A1_SOC_FORMAT_S16_LE:
         break;
    case A1_SOC_FORMAT_S20_LE:
         iface |= (1 << 5);
         break;
    case A1_SOC_FORMAT_S24_LE:
         iface |= (2 << 5);
         break;
    case A1_SOC_FORMAT_S32_LE:
         iface |= (3 << 5);
         break;
    }
    ret = au_wm8940_write(WM8940_COMPANDINGCTL, companding);
    if (ret)
        goto error_ret;

    ret = au_wm8940_write(WM8940_IFACE, iface);
    return ret;

error_ret:
    dprintk(0,KERN_ERR "%s: Failed to set hw params(%x) sample rate(%x)\n", __func__, iface, addcntrl);
    return ret;
}

static int au_wm8940_set_sysclk(u_int freq)
{
    au_codec_t  *codec = &au_codec;

    dprintk(1,KERN_INFO "%s:freq(%u)\n", __func__, freq);

    switch (freq) {
        case 11289600:
        case 12000000:
        case 12288000:
        case 16934400:
        case 18432000:
             codec->sysclk = freq;
             return 0;
    }
    return -EINVAL;
}


static int au_wm8940_set_bias_level( enum snd_soc_bias_level level)
{
    u16 val;
    u16 pwr_reg = au_wm8940_read_cache(WM8940_POWER1) & 0x1F0;
    int ret = 0;

    switch (level) {
    case SND_SOC_BIAS_ON:
        /* ensure bufioen and biasen */
        pwr_reg |= (1 << 2) | (1 << 3);
        /* Enable thermal shutdown */
        val = au_wm8940_read_cache (WM8940_OUTPUTCTL);
        ret = au_wm8940_write(WM8940_OUTPUTCTL, val | 0x2);
        //dprintk(0,KERN_INFO "%s: ret(%d)\n", __func__, ret);
        if (ret)
            break;
        /* set vmid to 50k */
        ret = au_wm8940_write(WM8940_POWER1, pwr_reg | 0x1);
        break;
    case SND_SOC_BIAS_PREPARE:
        /* ensure bufioen and biasen */
        pwr_reg &= 0x0FF;  //R:20140319
        pwr_reg |= (1 << 2) | (1 << 3);
        ret = au_wm8940_write(WM8940_POWER1, pwr_reg | 0x1);
        break;
    case SND_SOC_BIAS_STANDBY:
        /* ensure bufioen and biasen */
        pwr_reg |= (1 << 2) | (1 << 3);
        /* set vmid to 250k for standby */
        ret = au_wm8940_write(WM8940_POWER1, pwr_reg | 0x2);
        break;
    case SND_SOC_BIAS_OFF:
        ret = au_wm8940_write(WM8940_POWER1, pwr_reg);
        break;
    }

    return ret;
}

static int au_wm8940_set_input_signal(int on_off)
{
    enum {_LO=0, _HI};
    int ret;
    u16 reg;

    if (on_off == _OFF) {
        /* disable ADC control and PGA */
        reg = au_wm8940_read_cache (WM8940_POWER2) & ~0x01;
        ret = au_wm8940_write(WM8940_POWER2, reg);
        if (ret < 0) {
            dprintk(0,KERN_ERR "Failed to set non-VMID and level shifters.\n");
            return ret;
        }
        return 0;
    }

    /* enable ADC control and PGA */
    reg  = au_wm8940_read_cache (WM8940_POWER2) & ~0x15;
    reg |= WM8940_ADCEN | WM8940_INPPGATE | WM8940_BOOSTEN;
    ret  = au_wm8940_write(WM8940_POWER2, reg);
    if (ret < 0) {
        dprintk(0,KERN_ERR "Failed to enable and PGA gate\n");
        return ret;
    }

    /* set input ctrl */
    reg  = au_wm8940_read_cache(WM8940_INPUTCTL) & ~0x103;
    reg |= WM8940_MICP2INPPGA | WM8940_MICN2INPPGA | WM8940_MBVSEL;
    ret  = au_wm8940_write(WM8940_INPUTCTL, reg);
    if (ret < 0) {
        dprintk(0,KERN_ERR "Failed to set input control\n");
        return ret;
    }

    /* PGA not mute */
    au_wm8940_pga_mute(_OFF);

    /* Input PGA gain 0~63, -12dB ~ 35.25dB*/
    /* set default PGA gain to 0dB */
    au_wm8940_involume(0x10);

    /* set input boost PGA gain = 0dB */
    au_wm8940_pga_boost(_LO);

    return 0;
}

static int au_wm8940_set_output_signal(int onoff)
{
    int ret;
    u16 reg;

    if (onoff == _OFF) {
        /* After software reset, the DAC, SPK, MONO out is disable */
        reg = au_wm8940_read_cache (WM8940_POWER3) & 0x00;
        ret = au_wm8940_write(WM8940_POWER3, reg);
        if (ret < 0) {
            dprintk(0,KERN_ERR "Failed to set non-VMID and level shifters.\n");
            return ret;
        }
        return 0;
    }

    /* enable dac mute DACMU = 1*/
    au_wm8940_dac_mute(_ON);

    /* enable DAC and Speak */
    reg  = au_wm8940_read_cache (WM8940_POWER3) & ~0x0065;
    reg |= WM8940_DACEN | WM8940_SPKMIXEN | WM8940_SPKPEN | WM8940_SPKNEN;
    ret  = au_wm8940_write(WM8940_POWER2, reg);
    if (ret < 0) {
        dprintk(0,KERN_ERR "Failed to enable dac and speaker.\n");
        return ret;
    }

    /* Output of DAC to speaker mixer input */
    reg  = au_wm8940_read_cache (WM8940_SPKMIX) & ~0x0001;
    ret  = au_wm8940_write(WM8940_SPKMIX, reg | 0x0001);
    if (ret < 0) {
        dprintk(0,KERN_ERR "Failed to select Output of DAC to speaker mixer input.\n");
    }

    /* set SPKVOL to 0 dB */
    au_wm8940_outvolume(53);

    /* disable dac mute DACMU = 0*/
     au_wm8940_dac_mute(_OFF);

    return 0;
}

int au_wm8940_init(void)
{
    au_codec_t  *codec = &au_codec;
    hw_params_t *param = &codec->param;
    int ret;

    dprintk(0,KERN_INFO "%s\n", __func__);

    codec->reg_cache_size = sizeof(wm8940_reg_defaults);
    codec->reg_cache      = kmemdup(wm8940_reg_defaults, sizeof(wm8940_reg_defaults), GFP_KERNEL);

    if (codec->reg_cache == NULL)
        return -ENOMEM;

    ret = au_wm8940_write(WM8940_SOFTRESET, 0);
    if (ret < 0) {
        dprintk(0,KERN_ERR "Failed to issue reset\n");
        return ret;
    }

    /* VMID_OP_EN = 1, LVLSHIFT_EN = 1 MICBEN = 1 */
    ret = au_wm8940_write(WM8940_POWER1, 0x190);
    if (ret < 0) {
        dprintk(0,KERN_ERR "Failed to set non-VMID and level shifters.\n");
        return ret;
    }

    ret = au_wm8940_set_bias_level(SND_SOC_BIAS_STANDBY);

    /* set input signal path */
    au_wm8940_set_input_signal(_ON);

    /* disable output signal path */
    au_wm8940_set_output_signal(_OFF);

    au_wm8940_set_sysclk(A1_SOC_SYSCLK_DEFAULT);
    /* set default sampling rate and bit size */
    param->rate     = A1_SOC_RATE_DEFAULT;
    param->format   = A1_SOC_FORMAT_S16_LE;
    au_wm8940_hw_params(param);

    /* clk & frame are output generated by WM8940(master) */
    param->clk_fmt  = A1_SOC_DAIFMT_CBM_CFM;
    /* set dai pcm interface and inverted clocks */
    param->iface_fmt = A1_SOC_DAIFMT_I2S | A1_SOC_DAIFMT_NB_NF;
    au_wm8940_set_dai_fmt(param);

    /* set Clock CODEC directly from MCLK */
    au_wm8940_set_mclk_ctrl();

    /* BCLK=MCLK/4 */
    au_wm8940_set_dai_clkdiv(WM8940_BCLKDIV, WM8940_BCLKDIV_4);

    //*********************************************************
    /* move to start ???? */
    ret = au_wm8940_set_bias_level(SND_SOC_BIAS_ON);

    return 0;
}

#if 0
int  au_wm8940_release(void)
{
    au_wm8940_dac_mute(_ON);
    /* disable output signal path */
    au_wm8940_set_output_signal(_OFF);
    au_wm8940_set_bias_level(SND_SOC_BIAS_OFF);
    return 0;
}
#endif
