/*
 *  linux/drivers/video/umvp_fb.c -- framebuffer driver for GUC UMVP3000
 *
 * Copyright (C) 2010 Global UniChip Corp.
 *
 * This file is based on the Virtual frame buffer device (vfb.c):
 *
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file COPYING in the main directory of this archive
 * for more details.
 *
 *  
 */


#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/tty.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <mach/hardware.h>
#include <mach/general.h>
#include <asm/uaccess.h>
#include <linux/fb.h>
#include <linux/init.h>
#include <linux/dma-mapping.h>
#include <linux/autoconf.h>
#include <asm/pgtable.h>



#define HAL_REG_SYS_BASE                (IO_ADDRESS(UMVP_SYSTEM_BASE))
#define HAL_REG_SYS_ModeSts             (HAL_REG_SYS_BASE+0x1C)

#define HAL_REG_LCD_BASE                (IO_ADDRESS(UMVP_LCD_BASE))
#define HAL_REG_LCD_HCtrlR1             (HAL_REG_LCD_BASE+0x00)
#define HAL_REG_LCD_HCtrlR2             (HAL_REG_LCD_BASE+0x04)
#define HAL_REG_LCD_VCtrlR1             (HAL_REG_LCD_BASE+0x08)
#define HAL_REG_LCD_VCtrlR2             (HAL_REG_LCD_BASE+0x0C)
#define HAL_REG_LCD_IFormR              (HAL_REG_LCD_BASE+0x10)
#define HAL_REG_LCD_DOutR               (HAL_REG_LCD_BASE+0x14)
#define HAL_REG_LCD_TCtrlR              (HAL_REG_LCD_BASE+0x18)
#define HAL_REG_LCD_YBaseR              (HAL_REG_LCD_BASE+0x1C)
#define HAL_REG_LCD_CBBaseR             (HAL_REG_LCD_BASE+0x20)
#define HAL_REG_LCD_IntCtrlR            (HAL_REG_LCD_BASE+0x24)
#define HAL_REG_LCD_IntStsR             (HAL_REG_LCD_BASE+0x28)
#define HAL_REG_LCD_TModeR              (HAL_REG_LCD_BASE+0x38)
#define HAL_REG_LCD_IMGCropR            (HAL_REG_LCD_BASE+0x3C)
#define HAL_REG_LCD_IMGPosR             (HAL_REG_LCD_BASE+0x40)
#define HAL_REG_LCD_IMGStrideR          (HAL_REG_LCD_BASE+0x44)
#define HAL_REG_LCD_BCOLORR             (HAL_REG_LCD_BASE+0x4C)
#define HAL_REG_LCD_OSDBaseR            (HAL_REG_LCD_BASE+0x50)
#define HAL_REG_LCD_OSDCropR            (HAL_REG_LCD_BASE+0x54)
#define HAL_REG_LCD_OSDPosR             (HAL_REG_LCD_BASE+0x58)
#define HAL_REG_LCD_OSDStrideR          (HAL_REG_LCD_BASE+0x5C)
#define HAL_REG_LCD_OSDTModeR           (HAL_REG_LCD_BASE+0x60)


#define HAL_SETREG(r,v) 		(*((volatile unsigned int *)(r))=((unsigned int)(v)))
#define HAL_GETREG(r)			(*((volatile unsigned int *)(r)))
#define HAL_SETREG32(r,v)		(*((volatile unsigned int *)(r)) = ((unsigned int)(v)))
#define HAL_GETREG32(r)			(*((volatile unsigned long *)(r)))

#define halLCDEnable(void)		(HAL_SETREG( HAL_REG_LCD_DOutR, 3 )) 


#define BITS_PER_PIXEL			32


#ifdef CONFIG_UMVP_LCD_640x480
	#define HAL_PANEL_WIDTH  640
	#define HAL_PANEL_HEIGHT 480
#endif

#ifdef CONFIG_UMVP_LCD_800x480
	//#define	OSD_WIDTH		200
	//#define	OSD_HEIGHT		100
	//#define	OSD_START_X		300
	//#define	OSD_START_Y		190
	#define	OSD_WIDTH		512
	#define	OSD_HEIGHT		320
	#define	OSD_START_X		144
	#define	OSD_START_Y		80
#endif

#ifdef CONFIG_UMVP_LCD_800x600
	#define HAL_PANEL_WIDTH  800
	#define HAL_PANEL_HEIGHT 600
#endif



#ifdef CONFIG_UMVP_LCD_640x480
	#define VIDEOMEMSIZE		(4*640*480)	
#endif

#ifdef CONFIG_UMVP_LCD_800x480
	#define VIDEOMEMSIZE		(4 * OSD_WIDTH * OSD_HEIGHT)
#endif

#ifdef CONFIG_UMVP_LCD_800x600
	#define VIDEOMEMSIZE		(4*800*600)	
#endif


unsigned int OSD_CROP_WIDTH		= OSD_WIDTH;
unsigned int OSD_CROP_HEIGHT		= OSD_HEIGHT;
unsigned int OSD_STRIDE_WIDTH		= OSD_WIDTH;
unsigned int OSD_POS_X			= OSD_START_X; 
unsigned int OSD_POS_Y			= OSD_START_Y; 
unsigned int OSD_EN			= 1;
unsigned int OSD_GLOBAL_ALPHA_EN	= 0;
unsigned int OSD_GLOBAL_ALPHA_VALUE	= 10;


static u64 		osd_fb_dma_mask = ~(u64)0;
static void 		*osd_videomemory;
static u_long 		osd_videomemorysize = VIDEOMEMSIZE;


module_param( osd_videomemorysize, ulong, 0 );


typedef enum halDisplay_E {
    halcTV  = 1,
    halcLCD = 2,
    halcVGA = 3
} halDisplay_E;


static struct fb_var_screeninfo osd_fb_default __initdata = {
	.xres 		= OSD_WIDTH,
	.yres		= OSD_HEIGHT,
	.xres_virtual	= OSD_WIDTH,
	.yres_virtual	= OSD_HEIGHT,
	.xoffset	= 0,
	.yoffset	= 0,
	.bits_per_pixel	= BITS_PER_PIXEL,
	.grayscale	= 0,
#if     (BITS_PER_PIXEL == 32)
	.red		= {16, 8, 0},
	.green		= {8, 8, 0},
	.blue		= {0, 8, 0},
	.transp		= {24, 8, 0},

#elif   (BITS_PER_PIXEL == 16)
	.red		= {11, 5, 0},
	.green		= {5, 6, 0},
	.blue		= {0, 5, 0},
	.transp		= {0, 0, 0},

#elif   (BITS_PER_PIXEL == 8)
	.red		= {0, 8, 0},
	.green		= {0, 8, 0},
	.blue		= {0, 8, 0},
	.transp		= {0, 0, 0},
#elif   (BITS_PER_PIXEL == 12)
#else
#error  INVALID PIXEL FORMAT!!!
#endif
	.nonstd		= 0,
	.activate	= FB_ACTIVATE_NOW,
	.height		= -1,
	.width		= -1,
	.accel_flags	= FB_ACCEL_NONE,
	.pixclock	= 79440,
	.left_margin	= 16,
	.right_margin	= 16,
	.upper_margin	= 16,
	.lower_margin	= 5,
	.hsync_len	= 48,
	.vsync_len	= 1,
	.sync		= 0,
	.vmode		= FB_VMODE_DOUBLE,
};


static struct fb_fix_screeninfo osd_fb_fix __initdata = {
	.id		= "OSD FB",
	.type		= FB_TYPE_PACKED_PIXELS,
	.visual		= FB_VISUAL_PSEUDOCOLOR,
	.xpanstep	= 1,
	.ypanstep	= 1,
	.ywrapstep	= 1,
	.accel		= FB_ACCEL_NONE,
};


static int osd_fb_enable __initdata = 0;	/* disabled by default */
module_param( osd_fb_enable, bool, 0 );

static int osd_fb_check_var( struct fb_var_screeninfo *var, struct fb_info *info );
static int osd_fb_set_par( struct fb_info *info );
static int osd_fb_setcolreg( u_int regno, u_int red, u_int green, u_int blue, u_int transp, struct fb_info *info );
static int osd_fb_pan_display( struct fb_var_screeninfo *var, struct fb_info *info );


static struct fb_ops osd_fb_ops = {
	.fb_check_var	= osd_fb_check_var,
	.fb_set_par	= osd_fb_set_par,
	.fb_setcolreg	= osd_fb_setcolreg,
	.fb_pan_display	= osd_fb_pan_display,
};

    /*
     *  Internal routines
     */

static u_long get_line_length( int xres_virtual, int bpp )
{
	u_long length;

	length = xres_virtual * bpp;
	length = (length + 31) & ~31;
	length >>= 3;
	return (length);
}

    /*
     *  Setting the video mode has been split into two parts.
     *  First part, xxxfb_check_var, must not write anything
     *  to hardware, it should only verify and adjust var.
     *  This means it doesn't alter par but it does use hardware
     *  data from it to check this var. 
     */

static int osd_fb_check_var( struct fb_var_screeninfo *var, struct fb_info *info )
{
	u_long line_length;

	/*
	 *  FB_VMODE_CONUPDATE and FB_VMODE_SMOOTH_XPAN are equal!
	 *  as FB_VMODE_SMOOTH_XPAN is only used internally
	 */

	if (var->vmode & FB_VMODE_CONUPDATE) {
		var->vmode |= FB_VMODE_YWRAP;
		var->xoffset = info->var.xoffset;
		var->yoffset = info->var.yoffset;
	}

	/*
	 *  Some very basic checks
	 */
	if (!var->xres)
		var->xres = 1;
	if (!var->yres)
		var->yres = 1;
	if (var->xres > var->xres_virtual)
		var->xres_virtual = var->xres;
	if (var->yres > var->yres_virtual)
		var->yres_virtual = var->yres;
	if (var->bits_per_pixel <= 1)
		var->bits_per_pixel = 1;
	else if (var->bits_per_pixel <= 8)
		var->bits_per_pixel = 8;
	else if (var->bits_per_pixel <= 16)
		var->bits_per_pixel = 16;
	else if (var->bits_per_pixel <= 24)
		var->bits_per_pixel = 24;
	else if (var->bits_per_pixel <= 32)
		var->bits_per_pixel = 32;
	else
		return -EINVAL;

	if (var->xres_virtual < var->xoffset + var->xres)
		var->xres_virtual = var->xoffset + var->xres;
	if (var->yres_virtual < var->yoffset + var->yres)
		var->yres_virtual = var->yoffset + var->yres;

	/*
	 *  Memory limit
	 */
	line_length = get_line_length(var->xres_virtual, var->bits_per_pixel);
	if (line_length * var->yres_virtual > osd_videomemorysize)
		return -ENOMEM;

	/*
	 * Now that we checked it we alter var. The reason being is that the video
	 * mode passed in might not work but slight changes to it might make it 
	 * work. This way we let the user know what is acceptable.
	 */
	switch (var->bits_per_pixel) {
	case 1:
	case 8:
		var->red.offset = 0;
		var->red.length = 8;
		var->green.offset = 0;
		var->green.length = 8;
		var->blue.offset = 0;
		var->blue.length = 8;
		var->transp.offset = 0;
		var->transp.length = 0;
		break;
	case 16:		/* RGBA 5551 */
		if (var->transp.length) {
			var->red.offset = 0;
			var->red.length = 5;
			var->green.offset = 5;
			var->green.length = 5;
			var->blue.offset = 10;
			var->blue.length = 5;
			var->transp.offset = 15;
			var->transp.length = 1;
		} else {	/* RGB 565 */
			var->red.offset = 0;
			var->red.length = 5;
			var->green.offset = 5;
			var->green.length = 6;
			var->blue.offset = 11;
			var->blue.length = 5;
			var->transp.offset = 0;
			var->transp.length = 0;
		}
		break;
	case 24:		/* RGB 888 */
		var->red.offset = 0;
		var->red.length = 8;
		var->green.offset = 8;
		var->green.length = 8;
		var->blue.offset = 16;
		var->blue.length = 8;
		var->transp.offset = 0;
		var->transp.length = 0;
		break;
	case 32:		/* RGBA 8888 */
		var->red.offset = 0;
		var->red.length = 8;
		var->green.offset = 8;
		var->green.length = 8;
		var->blue.offset = 16;
		var->blue.length = 8;
		var->transp.offset = 24;
		var->transp.length = 8;
		break;
	}

	var->red.msb_right = 0;
	var->green.msb_right = 0;
	var->blue.msb_right = 0;
	var->transp.msb_right = 0;

	return 0;
}

/* This routine actually sets the video mode. It's in here where we
 * the hardware state info->par and fix which can be affected by the 
 * change in par. For this driver it doesn't do much. 
 */
static int osd_fb_set_par( struct fb_info *info )
{
	info->fix.line_length = get_line_length( info->var.xres_virtual, info->var.bits_per_pixel );
	return 0;
}

    /*
     *  Set a single color register. The values supplied are already
     *  rounded down to the hardware's capabilities (according to the
     *  entries in the var structure). Return != 0 for invalid regno.
     */

static int osd_fb_setcolreg( u_int regno, u_int red, u_int green, u_int blue, u_int transp, struct fb_info *info )
{
	if (regno >= 256)	/* no. of hw registers */
		return 1;
	/*
	 * Program hardware... do anything you want with transp
	 */

	/* grayscale works only partially under directcolor */
	if (info->var.grayscale) {
		/* grayscale = 0.30*R + 0.59*G + 0.11*B */
		red = green = blue =
		    (red * 77 + green * 151 + blue * 28) >> 8;
	}

	/* Directcolor:
	 *   var->{color}.offset contains start of bitfield
	 *   var->{color}.length contains length of bitfield
	 *   {hardwarespecific} contains width of RAMDAC
	 *   cmap[X] is programmed to (X << red.offset) | (X << green.offset) | (X << blue.offset)
	 *   RAMDAC[X] is programmed to (red, green, blue)
	 * 
	 * Pseudocolor:
	 *    uses offset = 0 && length = RAMDAC register width.
	 *    var->{color}.offset is 0
	 *    var->{color}.length contains widht of DAC
	 *    cmap is not used
	 *    RAMDAC[X] is programmed to (red, green, blue)
	 * Truecolor:
	 *    does not use DAC. Usually 3 are present.
	 *    var->{color}.offset contains start of bitfield
	 *    var->{color}.length contains length of bitfield
	 *    cmap is programmed to (red << red.offset) | (green << green.offset) |
	 *                      (blue << blue.offset) | (transp << transp.offset)
	 *    RAMDAC does not exist
	 */
#define CNVT_TOHW(val,width) ((((val)<<(width))+0x7FFF-(val))>>16)
	switch (info->fix.visual) {
	case FB_VISUAL_TRUECOLOR:
	case FB_VISUAL_PSEUDOCOLOR:
		red = CNVT_TOHW(red, info->var.red.length);
		green = CNVT_TOHW(green, info->var.green.length);
		blue = CNVT_TOHW(blue, info->var.blue.length);
		transp = CNVT_TOHW(transp, info->var.transp.length);
		break;
	case FB_VISUAL_DIRECTCOLOR:
		red = CNVT_TOHW(red, 8);	/* expect 8 bit DAC */
		green = CNVT_TOHW(green, 8);
		blue = CNVT_TOHW(blue, 8);
		/* hey, there is bug in transp handling... */
		transp = CNVT_TOHW(transp, 8);
		break;
	}
#undef CNVT_TOHW
	/* Truecolor has hardware independent palette */
	if (info->fix.visual == FB_VISUAL_TRUECOLOR) {
		u32 v;

		if (regno >= 16)
			return 1;

		v = (red << info->var.red.offset) |
		    (green << info->var.green.offset) |
		    (blue << info->var.blue.offset) |
		    (transp << info->var.transp.offset);
		switch (info->var.bits_per_pixel) {
		case 8:
			break;
		case 16:
			((u32 *) (info->pseudo_palette))[regno] = v;
			break;
		case 24:
		case 32:
			((u32 *) (info->pseudo_palette))[regno] = v;
			break;
		}
		return 0;
	}
	return 0;
}

    /*
     *  Pan or Wrap the Display
     *
     *  This call looks only at xoffset, yoffset and the FB_VMODE_YWRAP flag
     */

static int osd_fb_pan_display( struct fb_var_screeninfo *var, struct fb_info *info )
{
	if (var->vmode & FB_VMODE_YWRAP) {
		if (var->yoffset < 0
		    || var->yoffset >= info->var.yres_virtual
		    || var->xoffset)
			return -EINVAL;
	} else {
		if (var->xoffset + var->xres > info->var.xres_virtual ||
		    var->yoffset + var->yres > info->var.yres_virtual)
			return -EINVAL;
	}
	info->var.xoffset = var->xoffset;
	info->var.yoffset = var->yoffset;
	if (var->vmode & FB_VMODE_YWRAP)
		info->var.vmode |= FB_VMODE_YWRAP;
	else
		info->var.vmode &= ~FB_VMODE_YWRAP;
	return 0;
}


    /*
     *  Initialisation
     */
static void osd_fb_platform_release(struct device *device)
{
	// This is called when the reference count goes to zero.
}


#ifndef MODULE
static int __init osd_fb_setup(char *options)
{
	char *this_opt;


	// osd crop size
	HAL_SETREG32( HAL_REG_LCD_OSDCropR, (OSD_CROP_HEIGHT<<16) + OSD_CROP_WIDTH );

	// osd start position
	HAL_SETREG32( HAL_REG_LCD_OSDPosR, (OSD_POS_Y<<16) + OSD_POS_X );

	// osd stride width                                   
	HAL_SETREG32( HAL_REG_LCD_OSDStrideR, OSD_STRIDE_WIDTH );

	// input image type / osd enable, use global alpha, half transparant effect
	HAL_SETREG32( HAL_REG_LCD_IFormR, (OSD_GLOBAL_ALPHA_VALUE<<8) + (OSD_GLOBAL_ALPHA_EN<<5) + (OSD_EN<<4) );


	osd_fb_enable = 1;

	if (!options || !*options)
		return 1;

	while ((this_opt = strsep(&options, ",")) != NULL) {
		if (!*this_opt)
			continue;

		if (!strncmp(this_opt, "disable", 7))
			osd_fb_enable = 0;
	}
	return 1;
}
#endif  /*  MODULE  */



static int __init osd_fb_probe( struct platform_device *dev )
{
	struct fb_info *info;
	int retval = -ENOMEM;
	unsigned long page;

	dma_addr_t dma;

	info = framebuffer_alloc( sizeof( u32 ) * 256, &dev->dev );

	if( ! info ) {
		goto err;
	}

#ifdef CONFIG_ARCH_CPU_V6
	//info->screen_base = (char __iomem *)UMVP_OSD_FB_VA_BASE;
	//dma = (dma_addr_t)UMVP_OSD_FB_BASE;
	info->screen_base = dma_alloc_coherent( &dev->dev, VIDEOMEMSIZE, &dma, GFP_KERNEL );
#else
	info->screen_base = dma_alloc_coherent( &dev->dev, VIDEOMEMSIZE, &dma, GFP_KERNEL );
#endif
	memset( info->screen_base, 0, VIDEOMEMSIZE );

	for( page = *info->screen_base;
		page < PAGE_ALIGN(*info->screen_base + VIDEOMEMSIZE); page += PAGE_SIZE ) {

		SetPageReserved( virt_to_page( page ) );
	}

	if( ! info->screen_base ) {
		printk( KERN_ERR "CLCD: unable to map framebuffer\n" );
		return -ENOMEM;
	}

	info->fbops = &osd_fb_ops;
	retval = fb_find_mode( &info->var, info, NULL, NULL, 0, NULL, 32 );

	info->var = osd_fb_default;
	info->fix = osd_fb_fix;
	info->pseudo_palette = info->par;
	info->par = NULL;
	info->flags = FBINFO_FLAG_DEFAULT;

	retval = fb_alloc_cmap( &info->cmap, 256, 0 );
	if( retval < 0 ) {
		goto err1;
	}

	info->fix.smem_start = dma;
	info->fix.smem_len = VIDEOMEMSIZE;
	info->fix.line_length = OSD_WIDTH * BITS_PER_PIXEL << 2;

	retval = register_framebuffer( info );
	HAL_SETREG32( HAL_REG_LCD_OSDBaseR, info->fix.smem_start );



	if( retval < 0 ) {
		goto err2;
	}

	printk( KERN_ERR "umvp_fb probe ok! screen_base=0x%08x, 0x%08x\n",
		(unsigned int)(info->screen_base), dma );


	__lcm_row_clean( 1 );
	__lcm_display( 1, 0, "LCD OK!");


	return 0;
err2:
	fb_dealloc_cmap( &info->cmap );
	printk( KERN_ERR "err2\n" );
err1:
	framebuffer_release( info );
	printk( KERN_ERR "err1\n" );
err:
	framebuffer_release( info );
	printk( KERN_ERR "err\n" );

	return retval;
}




static int osd_fb_remove( struct platform_device *dev )
{
	struct fb_info *info = platform_get_drvdata( dev );

	if( info ) {
		unregister_framebuffer( info );
		vfree( osd_videomemory );
		framebuffer_release( info );
	}
	return 0;
}


static struct platform_driver osd_fb_driver = {
	
	.driver		= { .name = "osd_fb", },
	.probe		= osd_fb_probe,
	.remove		= osd_fb_remove,
	
};

static struct platform_device osd_fb_device = {
	.name	= "osd_fb",
	.id	= 0,
	.dev	= {
		.dma_mask		= &osd_fb_dma_mask,
 		.coherent_dma_mask	= 0xffffffff,
		.release		= osd_fb_platform_release,
		}
};


static int __init osd_fb_init( void )
{
	int ret = 0;

#ifndef MODULE
	osd_fb_setup( NULL );
#endif

	if ( ! osd_fb_enable )
		return -ENXIO;

	ret = platform_driver_register( &osd_fb_driver );

	if( ! ret ) {
 		 ret = platform_device_register( &osd_fb_device );
		if (ret) {
			platform_driver_unregister( &osd_fb_driver );
		}
	}

	return ret;
}


module_init( osd_fb_init );


#ifdef MODULE
static void __exit osd_fb_exit( void )
{
	platform_device_unregister( &osd_fb_device );
	platform_driver_unregister( &osd_fb_driver );
}

module_exit( osd_fb_exit );

MODULE_LICENSE("GPL");
#endif				/* MODULE */



