/*
 * Port from IT8212 driver
 * linux/drivers/scsi/atlas.c
 *
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 */
 
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/time.h>
#include <linux/proc_fs.h>
#include <linux/sched.h>
#include <linux/ioport.h>
// henry 10142008
#include <linux/blkdev.h>
#include <linux/hdreg.h>
#include <linux/string.h>
#include <linux/delay.h>
#include <linux/reboot.h>
#include <linux/spinlock.h>
#include <linux/fs.h>
// henry 10142008
#include <linux/interrupt.h>

#include <asm/errno.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/uaccess.h>

#include <scsi/sg.h>  // #include "sd.h"
#include "scsi.h"
// henry 10142008
#include "initio.h"
#include <scsi/scsi_host.h>  // #include "hosts.h"

#define ATLAS_DEBUG	0
#define ATLAS_INOUT_DBG	0  /* Use ATLAS inport/outport debug */
#include "atlas.h"

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,10)
MODULE_LICENSE("GPL");
#endif

#define	PRD_BYTES		8	/* PRD table size		*/
#define PRD_ENTRIES		(PAGE_SIZE / (2 * PRD_BYTES))
#define RW_RETRY_COUNT		3
//Jason
     #define UMVP_IDE_MAJOR 79

struct Scsi_Host * atlas_vhost = 0;	/* SCSI virtual host		*/
Scsi_Cmnd *	   atlas_req_last = 0;	/* SRB request list		*/

unsigned int  NumAdapters = 0;		/* Adapters number		*/
PATLAS_ADAPTER atlas_adapters[2]={0};	/* How many adapters support	*/

#if ATLAS_DEBUG
int system_dbg  = 1;
int hexdump_dbg = 1;
#else
int system_dbg  = 0;
int hexdump_dbg = 0;
#endif

#if ATLAS_INOUT_DBG
int inout_dbg = 1;
#endif

static void internal_done(Scsi_Cmnd * SCpnt);

/************************************************************************
 * Notifier blockto get a notify on system shutdown/halt/reboot.
 ************************************************************************/
static int atlas_halt(struct notifier_block *nb, ulong event, void *buf);

struct notifier_block atlas_notifier =
{
 atlas_halt,
 NULL,
 0
};


static spinlock_t queue_request_lock	= SPIN_LOCK_UNLOCKED;
static spinlock_t io_request_lock	= SPIN_LOCK_UNLOCKED;

#if 0
//Jason
static struct file_operations   atlas_fops = {
	owner:		THIS_MODULE,
	/* poll:		atlas_poll, */
//	ioctl:		atlas_scsi2ide_ioctl,
//	open:		atlas_open,
	release:	atlas_release,
};
#endif

/************************************************************************
 * Dump buffer.
 ************************************************************************/
//#if ATLAS_DEBUG
static void
HexDump(unsigned char *buf, int length)
{

 unsigned int		i = 0;
 unsigned int		j = 0;

 if(!hexdump_dbg) return;
 
 printk("\n");
 for (i = 0; i < length; i += 16)
    {
     printk("%04X    ", i);
     for (j = i; (j < i + 8) && (j < length); j++)
	{
	 printk(" %02X", buf[j]);
	}
     if (j == i + 8) printk("-");
     for (j = i + 8; (j < i + 16) && (j < length); j++)
	{
	 printk("%02X ", buf[j]);
	}
     printk("\n");
    }

} /* end HexDump */
//#endif

void Atlas_Register_Dump(PChannel pChan)
{
	
  if(!hexdump_dbg) return;	
	
  printk("0x00: 0x%08x\n", ATLAS_INL(pChan->io_ports[IDE_FEATURE_OFFSET]+0x00));
  printk("0x04: 0x%08x\n", ATLAS_INL(pChan->io_ports[IDE_FEATURE_OFFSET]+0x04));
  printk("0x08: 0x%08x\n", ATLAS_INL(pChan->io_ports[IDE_FEATURE_OFFSET]+0x08));
  printk("0x0C: 0x%08x\n", ATLAS_INL(pChan->io_ports[IDE_FEATURE_OFFSET]+0x0C));
  printk("0x10: 0x%08x\n", ATLAS_INL(pChan->io_ports[IDE_FEATURE_OFFSET]+0x10));
  printk("0x14: 0x%08x\n", ATLAS_INL(pChan->io_ports[IDE_FEATURE_OFFSET]+0x14));
  printk("0x18: 0x%08x\n", ATLAS_INL(pChan->io_ports[IDE_FEATURE_OFFSET]+0x18));
  printk("0x1C: 0x%08x\n", ATLAS_INL(pChan->io_ports[IDE_FEATURE_OFFSET]+0x1C));
  printk("0x20: 0x%08x\n", ATLAS_INL(pChan->io_ports[IDE_FEATURE_OFFSET]+0x20));
  printk("0x24: 0x%08x\n", ATLAS_INL(pChan->io_ports[IDE_FEATURE_OFFSET]+0x24));
  printk("0x28: 0x%08x\n", ATLAS_INL(pChan->io_ports[IDE_FEATURE_OFFSET]+0x28));
  printk("0x2C: 0x%08x\n", ATLAS_INL(pChan->io_ports[IDE_FEATURE_OFFSET]+0x2C));	
	
}
#if 0
// henry start 10152008
u32
sg_to_virt(struct scatterlist *sg)
{
  if(sg->dma_address)
   return (u32)phys_to_virt(sg->dma_address);
  else
   return (u32)phys_to_virt(page_to_phys(sg->page) + sg->offset);  		
}
#endif
// Jason
u32
sg_to_virt(struct scatterlist *sg)
{
  struct page *page = sg_page(sg);
  
  if(sg->dma_address)
   return (u32)phys_to_virt(sg->dma_address);
  else{
    return (u32)phys_to_virt(page_to_phys(page) + sg->offset);  		
  }
}

// henry end 10152008
/************************************************************************
 * This routine maps ATAPI and IDE errors to specific SRB statuses.
 ************************************************************************/
u8
MapError(PChannel pChan, PSCSI_REQUEST_BLOCK Srb)
{

 u8			errorByte;
 u8			srbStatus;
 u8			scsiStatus;

 /*
  * Read the error register.
  */
 errorByte = ATLAS_INB(pChan->io_ports[IDE_ERROR_OFFSET]);
 dprintk("MapError: error register is %x\n", errorByte);

#if FAULT_RETRY_EMULATION
 if(Srb->emulate_fault)
  errorByte = IDE_ERROR_BAD_BLOCK; // CRC Error
#endif

 /*
  * If this is ATAPI error.
  */
 if (pChan->DeviceFlags[Srb->TargetId & 1] & DFLAGS_ATAPI_DEVICE)
    {
     switch (errorByte >> 4)
	{
	 case SCSI_SENSE_NO_SENSE:
	     printk("ATAPI: no sense information\n");
	     scsiStatus = SCSISTAT_CHECK_CONDITION;
	     srbStatus  = SRB_STATUS_ERROR;
	     break;

	 case SCSI_SENSE_RECOVERED_ERROR:
	     printk("ATAPI: recovered error\n");
	     scsiStatus = 0;
	     srbStatus  = SRB_STATUS_SUCCESS;
	     break;

	 case SCSI_SENSE_NOT_READY:
	     printk("ATAPI: device not ready\n");
	     scsiStatus = SCSISTAT_CHECK_CONDITION;
	     srbStatus  = SRB_STATUS_ERROR;
	     break;

	 case SCSI_SENSE_MEDIUM_ERROR:
	     printk("ATAPI: media error\n");
	     scsiStatus = SCSISTAT_CHECK_CONDITION;
	     srbStatus  = SRB_STATUS_ERROR;
	     break;

	 case SCSI_SENSE_HARDWARE_ERROR:
	     printk("ATAPI: hardware error\n");
	     scsiStatus = SCSISTAT_CHECK_CONDITION;
	     srbStatus  = SRB_STATUS_ERROR;
	     break;

	 case SCSI_SENSE_ILLEGAL_REQUEST:
	     printk("ATAPI: illegal request\n");
	     scsiStatus = SCSISTAT_CHECK_CONDITION;
	     srbStatus  = SRB_STATUS_ERROR;
	     break;

	 case SCSI_SENSE_UNIT_ATTENTION:
	     printk("ATAPI: unit attention\n");
	     scsiStatus = SCSISTAT_CHECK_CONDITION;
	     srbStatus  = SRB_STATUS_ERROR;
	     break;

	 case SCSI_SENSE_DATA_PROTECT:
	     printk("ATAPI: data protect\n");
	     scsiStatus = SCSISTAT_CHECK_CONDITION;
	     srbStatus  = SRB_STATUS_ERROR;
	     break;

	 case SCSI_SENSE_BLANK_CHECK:
	     printk("ATAPI: blank check\n");
	     scsiStatus = SCSISTAT_CHECK_CONDITION;
	     srbStatus  = SRB_STATUS_ERROR;
	     break;

	 case SCSI_SENSE_ABORTED_COMMAND:
	     printk("ATAPI: command Aborted\n");
	     scsiStatus = SCSISTAT_CHECK_CONDITION;
	     srbStatus  = SRB_STATUS_ERROR;
	     break;

	 default:
	     printk("ATAPI: invalid sense information\n");
	     scsiStatus = 0;
	     srbStatus  = SRB_STATUS_ERROR;
	     break;
	}
    }
 /*
  * If this is IDE error.
  */
 else
    {
     scsiStatus = 0;
     srbStatus  = SRB_STATUS_ERROR;

     /*
      * Save errorByte, to be used by SCSIOP_REQUEST_SENSE.
      */
     pChan->ReturningMediaStatus = errorByte;

     if (errorByte & IDE_ERROR_MEDIA_CHANGE_REQ)
	{
	 printk("IDE: media change\n");
	 scsiStatus = SCSISTAT_CHECK_CONDITION;
	 srbStatus  = SRB_STATUS_ERROR;
	}
     else if (errorByte & IDE_ERROR_COMMAND_ABORTED)
	{
	 printk("IDE: command abort\n");
	 srbStatus  = SRB_STATUS_ABORTED;
	 scsiStatus = SCSISTAT_CHECK_CONDITION;

	 if (Srb->SenseInfoBuffer)
	    {
	     PSENSE_DATA senseBuffer = (PSENSE_DATA)Srb->SenseInfoBuffer;

	     senseBuffer->ErrorCode = 0x70;
	     senseBuffer->Valid = 1;
	     senseBuffer->AdditionalSenseLength = 0xB;
	     senseBuffer->SenseKey = SCSI_SENSE_ABORTED_COMMAND;
	     senseBuffer->AdditionalSenseCode = 0;
	     senseBuffer->AdditionalSenseCodeQualifier = 0;

	     srbStatus |= SRB_STATUS_AUTOSENSE_VALID;
	    }
	 /*
	  * pChan->ErrorCount++;
	  */
	}
     else if (errorByte & IDE_ERROR_END_OF_MEDIA)
	{
	 printk("IDE: end of media\n");

	 scsiStatus = SCSISTAT_CHECK_CONDITION;
	 srbStatus = SRB_STATUS_ERROR;

	 if (!(pChan->DeviceFlags[Srb->TargetId & 1] & DFLAGS_MEDIA_STATUS_ENABLED))
	    {
	     /*
	      * pChan->ErrorCount++;
	      */
	    }
	}
     else if (errorByte & IDE_ERROR_ILLEGAL_LENGTH)
	{
	 printk("IDE: illegal length\n");
	 srbStatus = SRB_STATUS_INVALID_REQUEST;
	}
     else if (errorByte & IDE_ERROR_BAD_BLOCK)
	{
	 printk("IDE: bad block\n");

	 // CRC error, change to request retry
	 //srbStatus = SRB_STATUS_ERROR;
	 srbStatus  = SRB_STATUS_NEED_REQUEUE;
	 scsiStatus = SCSISTAT_CHECK_CONDITION;

	 if (Srb->SenseInfoBuffer)
	    {
	     PSENSE_DATA senseBuffer = (PSENSE_DATA)Srb->SenseInfoBuffer;

	     senseBuffer->ErrorCode = 0x70;
	     senseBuffer->Valid = 1;
	     senseBuffer->AdditionalSenseLength = 0xB;
	     senseBuffer->SenseKey = SCSI_SENSE_MEDIUM_ERROR;
	     senseBuffer->AdditionalSenseCode = 0;
	     senseBuffer->AdditionalSenseCodeQualifier = 0;

	     srbStatus |= SRB_STATUS_AUTOSENSE_VALID;
	    }
	}
     else if (errorByte & IDE_ERROR_ID_NOT_FOUND)
	{
	 printk("IDE: id not found\n");

	 srbStatus = SRB_STATUS_ERROR;
	 scsiStatus = SCSISTAT_CHECK_CONDITION;

	 if (Srb->SenseInfoBuffer)
	    {
	     PSENSE_DATA senseBuffer = (PSENSE_DATA)Srb->SenseInfoBuffer;

	     senseBuffer->ErrorCode = 0x70;
	     senseBuffer->Valid = 1;
	     senseBuffer->AdditionalSenseLength = 0xB;
	     senseBuffer->SenseKey = SCSI_SENSE_MEDIUM_ERROR;
	     senseBuffer->AdditionalSenseCode = 0;
	     senseBuffer->AdditionalSenseCodeQualifier = 0;

	     srbStatus |= SRB_STATUS_AUTOSENSE_VALID;
	    }
	 /*
	  * pChan->ErrorCount++;
	  */
	}
     else if (errorByte & IDE_ERROR_MEDIA_CHANGE)
	{
	 printk("IDE: media change\n");

	 scsiStatus = SCSISTAT_CHECK_CONDITION;
	 srbStatus = SRB_STATUS_ERROR;

	 if (Srb->SenseInfoBuffer)
	    {
	     PSENSE_DATA senseBuffer = (PSENSE_DATA)Srb->SenseInfoBuffer;

	     senseBuffer->ErrorCode = 0x70;
	     senseBuffer->Valid = 1;
	     senseBuffer->AdditionalSenseLength = 0xB;
	     senseBuffer->SenseKey = SCSI_SENSE_UNIT_ATTENTION;
	     senseBuffer->AdditionalSenseCode = SCSI_ADSENSE_MEDIUM_CHANGED;
	     senseBuffer->AdditionalSenseCodeQualifier = 0;

	     srbStatus |= SRB_STATUS_AUTOSENSE_VALID;
	    }
	}
     else if (errorByte & IDE_ERROR_DATA_ERROR)
	{
	 printk("IDE: data error\n");

	 scsiStatus = SCSISTAT_CHECK_CONDITION;
	 srbStatus = SRB_STATUS_ERROR;

	 if (!(pChan->DeviceFlags[Srb->TargetId & 1] & DFLAGS_MEDIA_STATUS_ENABLED))
	    {
	     /*
	      * pChan->ErrorCount++;
	      */
	    }

	 /*
	  * Build sense buffer.
	  */
	 if (Srb->SenseInfoBuffer)
	    {
	     PSENSE_DATA senseBuffer = (PSENSE_DATA)Srb->SenseInfoBuffer;

	     senseBuffer->ErrorCode = 0x70;
	     senseBuffer->Valid = 1;
	     senseBuffer->AdditionalSenseLength = 0xB;
	     senseBuffer->SenseKey = SCSI_SENSE_MEDIUM_ERROR;
	     senseBuffer->AdditionalSenseCode = 0;
	     senseBuffer->AdditionalSenseCodeQualifier = 0;

	     srbStatus |= SRB_STATUS_AUTOSENSE_VALID;
	    }
	}
    }

 /*
  * Set SCSI status to indicate a check condition.
  */
 Srb->ScsiStatus = scsiStatus;

 return srbStatus;

} /* end MapError */

/************************************************************************
 * Just get the higest bit value.
 ************************************************************************/
static u8
GetHighestBit(u8 Number)
{

 char			bit;

 for (bit = 7; bit > 0; bit--)
    {
     if (Number & (1 << bit)) return bit;
    }

 return 0xFF;

} /* end GetHighestBit */


/************************************************************************
 * Reset IDE controller or ATAPI device.
 ************************************************************************/
static u8
AtapiResetController(PATLAS_ADAPTER pAdap, PChannel pChan)
{

 u8			i;
 u8			resetResult;
 unsigned long		dma_base;
 PSCSI_REQUEST_BLOCK	Srb;
 

 dprintk("AtapiResetController\n");
 
 disable_irq(ATLAS_AHB_RESOURCE_IRQ);
 
 dma_base = pChan->dma_base;
 resetResult = TRUE;

 /*
  * Check and see if we are processing an internal srb.
  */
 if (pChan->OriginalSrb)
    {
     pChan->CurrentSrb = pChan->OriginalSrb;
     pChan->OriginalSrb = NULL;
    }

 /*
  * To avoid unexpected interrupts occurs during reset procedure.
  *
  * 1. Stop bus master operation.
  */
 ATLAS_RAWB(~ATLAS_CONTROL_START, pChan->io_ports[ATLAS_CONTROL]);
 
 /*
  * Check if request is in progress.
  */
 if (pChan->CurrentSrb)
    {
     /*
      * Complete outstanding request with SRB_STATUS_BUS_RESET.
      */
     Srb = pChan->CurrentSrb; 
     Srb->SrbStatus = SRB_STATUS_BUS_RESET;

     /*
      * Clear request tracking fields.
      */
     pChan->CurrentSrb = NULL;
     pChan->WordsLeft  = 0;
     pChan->DataBuffer = NULL;

     /*
      * Indicate ready for next request.
      */
     TaskDone(pChan, Srb);
    }

 /*
  * Clear expecting interrupt flag.
  */
 pChan->ExpectingInterrupt = FALSE;
 pChan->RDP = FALSE;

  resetResult = AtlasResetAdapter(pAdap);

 /*
  * Set transfer modes after resetting the adapter.
  */

 /*
  * Reenable interrupts.
  */
 for (i = 0; i < 2; i++)
    {
     if(pChan->DeviceFlags[i] & DFLAGS_DEVICE_PRESENT)
     {
      ATLAS_OUTB((u8)((i << 4) | 0xA0), pChan->io_ports[ATAPI_SELECT_OFFSET]);
      AtlasSelectDevice(i);
      ATLAS_OUTB(IDE_DC_REENABLE_CONTROLLER, pChan->io_ports[ATAPI_CONTROL_OFFSET]);
     }
    }

 IRQ_EDGE_CLEAR(ATLAS_AHB_RESOURCE_IRQ);
 enable_irq(ATLAS_AHB_RESOURCE_IRQ);
 
 dprintk("AtapiResetController Done\n"); 
 return TRUE;

} /* end AtapiResetController */

/************************************************************************
 * IDE start read/write transfer.
 ************************************************************************/
void
IdeStartTransfer
(
 PChannel		pChan,
 PSCSI_REQUEST_BLOCK	Srb,
 u32			startingSector,
 u32			SectorNumber
)
{

 u8			DiskId;
 u8			drvSelect;

 DiskId = (u8) Srb->TargetId;

 /*
  * 48-bit support.
  */
 if (((startingSector + SectorNumber) > 0x0FFFFFFF) || (SectorNumber > 0x100))
    {
     /*
      * Select drive and set LBA mode.
      */     
     drvSelect = (u8) (startingSector >> 24);
     drvSelect = drvSelect | (((u8) DiskId & 0x1) << 4) | 0x40 | 0xA0;

      ATLAS_OUTB((u8) (((DiskId & 0x1) << 4) | 0xA0 | 0x40),
	     pChan->io_ports[IDE_SELECT_OFFSET]);	     
     /* Atlas 28bit, 48bit slection */
     ATLAS_ROWW(ATLAS_CONTROL_48BIT, pChan->io_ports[ATLAS_CONTROL]);
     	
     if(SectorNumber == 0x10000)
      SectorNumber = 0;
     	     
     /*
      * Sector count register.
      */
     ATLAS_OUTB((u8) (SectorNumber >> 8), pChan->io_ports[IDE_NSECTOR_OFFSET]+1); // High
     ATLAS_OUTB((u8) SectorNumber, pChan->io_ports[IDE_NSECTOR_OFFSET]); // Low
     
     /* Atlas sector count */
     ATLAS_OUTW(SectorNumber, pChan->io_ports[ATLAS_XFER_SEC]);

     /*
      * LBA low register.
      */
     ATLAS_OUTB((u8) (startingSector >> 24), pChan->io_ports[IDE_LOCYL_OFFSET]+1);
     ATLAS_OUTB((u8) startingSector, pChan->io_ports[IDE_LOCYL_OFFSET]);

     /*
      * LBA mid register.
      */
     ATLAS_OUTB((u8) 0, pChan->io_ports[IDE_MIDCYL_OFFSET]+1);
     ATLAS_OUTB((u8) (startingSector >> 8), pChan->io_ports[IDE_MIDCYL_OFFSET]);

     /*
      * LBA high register.
      */
     ATLAS_OUTB((u8) 0, pChan->io_ports[IDE_HCYL_OFFSET]+1);
     ATLAS_OUTB((u8) (startingSector >> 16), pChan->io_ports[IDE_HCYL_OFFSET]);


     /*
      * Start the IDE read/write PIO / DMA command.
      */     
     if(pChan->TraType[DiskId & 0x1] == PIO_FLOW_CONTROL) // PIO
     {
           if (Srb->SrbFlags & SRB_FLAGS_DATA_IN)
	    ATLAS_OUTB(IDE_COMMAND_READ_SECTOR_EXT, pChan->io_ports[IDE_COMMAND_OFFSET]);
           else if (Srb->SrbFlags & SRB_FLAGS_DATA_OUT)
	    ATLAS_OUTB(IDE_COMMAND_WRITE_SECTOR_EXT, pChan->io_ports[IDE_COMMAND_OFFSET]);     	     	
     }
     else // MDMA or UDMA
     {
           if (Srb->SrbFlags & SRB_FLAGS_DATA_IN)
	    ATLAS_OUTB(IDE_COMMAND_READ_DMA_EXT, pChan->io_ports[IDE_COMMAND_OFFSET]);
           else if (Srb->SrbFlags & SRB_FLAGS_DATA_OUT)
	    ATLAS_OUTB(IDE_COMMAND_WRITE_DMA_EXT, pChan->io_ports[IDE_COMMAND_OFFSET]);
     }     
	
    }
 /*
  * 28-bit addressing.
  */
 else
    {
     /*
      * Select drive and set LBA mode.
      */     
     drvSelect = (u8) (startingSector >> 24);
     drvSelect = drvSelect | (((u8) DiskId & 0x1) << 4) | 0x40 | 0xA0;
   
     
     /* Atlas 28bit, 48bit slection */
     ATLAS_RAWW(~ATLAS_CONTROL_48BIT, pChan->io_ports[ATLAS_CONTROL]);
     
     /* Atlas sector count */
     ATLAS_OUTW(SectorNumber, pChan->io_ports[ATLAS_XFER_SEC]);

     ATLAS_OUTB(drvSelect,		      pChan->io_ports[IDE_SELECT_OFFSET]);
     
     if(SectorNumber == 0x100)
      ATLAS_OUTB((u8) 0,	      pChan->io_ports[IDE_NSECTOR_OFFSET]);
     else
      ATLAS_OUTB((u8) SectorNumber,	      pChan->io_ports[IDE_NSECTOR_OFFSET]);
       
     ATLAS_OUTB((u8) startingSector,	      pChan->io_ports[IDE_LOCYL_OFFSET]);
     ATLAS_OUTB((u8)(startingSector >> 8),  pChan->io_ports[IDE_MIDCYL_OFFSET]);
     ATLAS_OUTB((u8)(startingSector >> 16), pChan->io_ports[IDE_HCYL_OFFSET]);

  
      /*
      * Start the IDE read/write PIO / DMA command.
      */ 
         
     if(pChan->TraType[DiskId & 0x1] == PIO_FLOW_CONTROL) // PIO
     {
           if (Srb->SrbFlags & SRB_FLAGS_DATA_IN)
	    ATLAS_OUTB(IDE_COMMAND_READ_SECTOR, pChan->io_ports[IDE_COMMAND_OFFSET]);
           else if (Srb->SrbFlags & SRB_FLAGS_DATA_OUT)
	    ATLAS_OUTB(IDE_COMMAND_WRITE_SECTOR, pChan->io_ports[IDE_COMMAND_OFFSET]);     	     	
     }
     else // MDMA or UDMA
     {
           if (Srb->SrbFlags & SRB_FLAGS_DATA_IN)
	    ATLAS_OUTB(IDE_COMMAND_READ_DMA, pChan->io_ports[IDE_COMMAND_OFFSET]);
           else if (Srb->SrbFlags & SRB_FLAGS_DATA_OUT)
	    ATLAS_OUTB(IDE_COMMAND_WRITE_DMA, pChan->io_ports[IDE_COMMAND_OFFSET]);
     }

    }

     /*
      * Command Data Type
      */
     switch(pChan->TraType[DiskId & 0x1])
     {     	
     	case MULTIWORD_DMA:
     	    ATLAS_ROWW(ATLAS_CONTROL_XFER_MDMA, pChan->io_ports[ATLAS_CONTROL]);
     	   break;
     	case ULTRA_DMA:
     	    ATLAS_ROWW(ATLAS_CONTROL_XFER_UDMA, pChan->io_ports[ATLAS_CONTROL]);     	   
	   break;	   
	case PIO_FLOW_CONTROL:
	    ATLAS_ROWW(ATLAS_CONTROL_XFER_PIO, pChan->io_ports[ATLAS_CONTROL]);
	   break;
     }

 /*
  * Indicate expecting an interrupt.
  */
 pChan->ExpectingInterrupt = TRUE;

 /*
  * Setup PRD table physical address.
  */
 ATLAS_OUTL(pChan->dmatable_dma, pChan->io_ports[ATLAS_PRD]);

 /*
  * Clear INTR and ERROR flags.
  */
 ATLAS_ROWW(ATLAS_STATUS_INT|ATLAS_STATUS_DESC_ERR|
            ATLAS_STATUS_TX_ERR|ATLAS_STATUS_RX_ERR, 
            pChan->io_ports[ATLAS_STATUS]);
            
 /* Sync Dev Ctrl */
 ATLAS_OUTB(ATLAS_INB(pChan->io_ports[IDE_CONTROL_OFFSET]), pChan->io_ports[IDE_CONTROL_OFFSET]);


 /*
  * Start DMA read/write.
  */
 if (Srb->SrbFlags & SRB_FLAGS_DATA_IN)
  ATLAS_ROWB(ATLAS_CONTROL_RW|ATLAS_CONTROL_START, pChan->io_ports[ATLAS_CONTROL]);
 else if (Srb->SrbFlags & SRB_FLAGS_DATA_OUT)
  ATLAS_OUTB(((ATLAS_INB(pChan->io_ports[ATLAS_CONTROL])&(~ATLAS_CONTROL_RW))|ATLAS_CONTROL_START), pChan->io_ports[ATLAS_CONTROL]);
  

} /* end IdeStartTransfer */

/************************************************************************
 * Setup the PRD table.
 ************************************************************************/
static int
IdeBuildSglist(PChannel pChan, PSCSI_REQUEST_BLOCK Srb)
{

 int		nents		= 0;
 u32		bytesRemaining	= Srb->DataTransferLength;
 unsigned char * virt_addr	= Srb->DataBuffer;
 struct scatterlist * sg	= pChan->sg_table;

 if (Srb->SrbFlags & SRB_FLAGS_DATA_IN)
    {
     pChan->sg_dma_direction = PCI_DMA_FROMDEVICE;
    }
 else
    {
     pChan->sg_dma_direction = PCI_DMA_TODEVICE;
    }

 /*
  * The upper layer will never give the memory more than 64K bytes.
  */
 memset(&sg[nents], 0, sizeof(*sg));
 sg[nents].dma_address = (dma_addr_t)virt_addr;  //  sg[nents].address = virt_addr;
 sg[nents].length  = bytesRemaining;
 nents++;

 return pci_map_sg(pChan->pPciDev, sg, nents, pChan->sg_dma_direction);

} /* end IdeBuildSglist */

/************************************************************************
 * Prepares a dma request.
 ************************************************************************/
static int
IdeBuildDmaTable(PChannel pChan, PSCSI_REQUEST_BLOCK Srb)
{

 unsigned long *	table = pChan->dmatable_cpu;
 unsigned int		count = 0;
 int			i;
 struct scatterlist *	sg;

 i = IdeBuildSglist(pChan, Srb);

 sg = pChan->sg_table;
 while (i && sg_dma_len(sg))
    {
     u32	cur_len;
     u32	cur_addr;

     cur_addr = sg_dma_address(sg);
     cur_len  = sg_dma_len(sg);

     /*
      * Fill in the dma table, without crossing any 64kB boundaries.
      */
  	while (cur_len)
  	{  		
	    if (count++ >= PRD_ENTRIES)
	    {
	     printk(KERN_WARNING "@@DMA table too small\n");
	    }  	
  	    else
  	    {
  	      u32 xcount =  0x8000;
  	      u32 bcount;
  	      	
  	      if(cur_len<=xcount)
  	       bcount = cur_len;
  	      else
  	       bcount = xcount;
  	       	
  	      *table++ = cpu_to_le32(cur_addr);
  	      *table++ = cpu_to_le32(bcount);

	     cur_addr += bcount;
	     cur_len  -= bcount;
  	       	
  	    }
   	}
	
     sg++;
     i--;
    }

 if (count)
    {
     *--table |= cpu_to_le32(0x80000000);
     return count;
    }
 else
   {
    printk(KERN_WARNING "Empty DMA table?\n");
   }

 return count;

} /* end IdeBuildDmaTable */

/************************************************************************
 * Prepares a dma scatter/gather request.
 ************************************************************************/
//static void
// henry 10142008   viod
static int
IdeBuildDmaSgTable(PChannel pChan, PSCSI_REQUEST_BLOCK Srb)
{

 int			use_sg = 0;
 int			i;
 PPRD_TABLE_ENTRY	pSG = (PPRD_TABLE_ENTRY)pChan->dmatable_cpu;
 struct scatterlist *	sg = (struct scatterlist *)Srb->DataBuffer;

 if (Srb->SrbFlags & SRB_FLAGS_DATA_IN)
    {
     pChan->sg_dma_direction = PCI_DMA_FROMDEVICE;
    }
 else
    {
     pChan->sg_dma_direction = PCI_DMA_TODEVICE;
    }

 use_sg = pci_map_sg(pChan->pPciDev, Srb->DataBuffer, Srb->UseSg, pChan->sg_dma_direction);

 for (i = 0; i < use_sg; i++)
    {
     pSG[i].PhysicalBaseAddress	= sg_dma_address(&sg[i]);
     pSG[i].ByteCount		= sg_dma_len(&sg[i]);
     pSG[i].EndOfTable		= (i == use_sg - 1) ? SG_FLAG_EOT : 0;
    }
return use_sg;   // return;

} /* end IdeBuildDmaSgTable */

/************************************************************************
 * Setup DMA table for channel.
 ************************************************************************/
static void
IdeSetupDma(PChannel pChan, unsigned long dma_base, unsigned short num_ports)
{

 dprintk("Channel[%d] BM-DMA at 0x%lX-0x%lX\n", pChan->channel, dma_base,
	dma_base + num_ports - 1);

 /*
  * Allocate IDE DMA buffer.
  */
 pChan->dmatable_cpu = pci_alloc_consistent(pChan->pPciDev,
	 PRD_ENTRIES * PRD_BYTES, &pChan->dmatable_dma);

 if (pChan->dmatable_cpu == NULL)
    {
     printk("IdeSetupDma: allocate PRD table failed.\n");
     return;
    }

 memset(pChan->dmatable_cpu, 0, PRD_ENTRIES * PRD_BYTES);

 /*
  * Allocate SCATTER/GATHER table buffer.
  */
 pChan->sg_table = kmalloc(sizeof(struct scatterlist) * PRD_ENTRIES,
				 GFP_KERNEL);

 if (pChan->sg_table == NULL)
    {
     printk("IdeSetupDma: allocate sg_table failed.\n");
     pci_free_consistent(pChan->pPciDev, PRD_ENTRIES * PRD_BYTES,
		pChan->dmatable_cpu, pChan->dmatable_dma);
     return;
    }

 return;

} /* end IdeSetupDma */


/************************************************************************
 * Replace to Atlas specific PIO data access
 ************************************************************************/
static int
AtlasPIOData(PChannel pChan, void *buffer, u32 len, u16 direction)
{
  u32	cur_addr, cur_len, dmaphy = 0, count = 0;
  u32   *table;
  void  *dmavirt;
  

  if(len%512>0)
   return -2; // Length must align to sector size
  
  if(len<=512)
  {
   dmavirt = pChan->PIOVirt;
   dmaphy  = pChan->PIOPhy;
  }
  else
   dmavirt = pci_alloc_consistent(pChan->pPciDev, len, &dmaphy);
    
  cur_addr = dmaphy;
  cur_len  = len;
  table = (u32 *)pChan->dmatable_cpu;
  
  if(!dmavirt) return -1;
  
  if(!direction) // Write Action
   memcpy(dmavirt, buffer, len);
   
  pChan->pio_access = 1;  // Hold command access
 
   while(pChan->CurrentSrb||pChan->ExpectingInterrupt) 
   { udelay(100); } // Wait queue complete
     
     /* 
      * Build PRD
      * Fill in the dma table, without crossing any 64kB boundaries.
      */
  	while (cur_len)
  	{  		
	    if (count++ >= PRD_ENTRIES)
	    {
	     printk(KERN_WARNING "@@DMA table too small\n");
	     goto err;
	    }  	
  	    else
  	    {
  	      u32 xcount =  0x8000;
  	      u32 bcount;
  	      	
  	      if(cur_len<=xcount)
  	       bcount = cur_len;
  	      else
  	       bcount = xcount;
  	       	
  	      *table++ = cpu_to_le32(cur_addr);
  	      *table++ = cpu_to_le32(bcount);

	     cur_addr += bcount;
	     cur_len  -= bcount;
  	       	
  	    }
   	}
    
      if (count)
       *--table |= cpu_to_le32(0x80000000); // EOT
      else
      {
       printk(KERN_WARNING "Empty DMA table?\n");
       goto err;
      }
	     
	     /*
	      * Setup PRD table physical address.
	      */
	      ATLAS_OUTL(pChan->dmatable_dma, pChan->io_ports[ATLAS_PRD]);

	     /*
	      * Setup data sector count.
	      */
	      ATLAS_OUTW(len/512, pChan->io_ports[ATLAS_XFER_SEC]);
	      
	     /*
	      * Setup direction, PIO, and Start Transfer
	      */
    	      ATLAS_OUTW(ATLAS_CONTROL_START|ATLAS_CONTROL_XFER_PIO|direction, 
    	                 pChan->io_ports[ATLAS_CONTROL]);
    	 
      	     /*
    	      * Wait transfer complete
    	      */
    	      
    	      WaitForAtlasCommandComplete(pChan);


  
  if(direction) // Read Action
   memcpy(buffer, dmavirt, len); 
    	      	  
err:
   
   pChan->pio_access = 0;  // Release command access
   
   if(dmavirt != pChan->PIOVirt)
    pci_free_consistent(pChan->pPciDev, len, dmavirt, dmaphy);  
   
   return 0;
}


/************************************************************************
 * Reset the controller.
 ************************************************************************/
static u8
AtlasResetAdapter(PATLAS_ADAPTER pAdap)
{

 u8			resetChannel[2];
 u8			channel;
 u8			device, result;
 u8			status[4]={0};
 int			i;
 PChannel		pChan;

 /*
  * First, perform ATAPI soft reset if ATAPI devices are attached.
  */
 for (channel = 0; channel < pAdap->num_channels; channel++)
    {
     pChan = &pAdap->IDEChannel[channel];
     resetChannel[channel] = FALSE;
     for (device = 0; device < 2; device++)
	{
	 if (pChan->DeviceFlags[device] & DFLAGS_DEVICE_PRESENT)
	    {
	     if (pChan->DeviceFlags[device] & DFLAGS_ATAPI_DEVICE)
		{
		 dprintk("AtlasResetAdapter: perform ATAPI soft reset (%d, %d)\n",
			channel, device);
		 AtapiSoftReset(pChan, device);
		}
	     else
		{
		 printk("%s\n", __FUNCTION__);
		 IdeHardReset(pChan, result);	
		 resetChannel[channel] = TRUE;
		}
	    }
	}
    }

 /*
  * If ATA device is present on this channel, perform channel reset.
  */
 for (channel = 0; channel < pAdap->num_channels; channel++)
    {
     pChan = &pAdap->IDEChannel[channel];
     if (resetChannel[channel])
	{
	 dprintk("AtlasResetAdapter: reset channel %d\n", channel);
	 ControllerReset(pChan);	
	}
    }
 

 /*
  * Check device status after reset.
  */
 ATLAS_INOUT_DBG_DIS;
 for (i = 0; i < 1000; i++)
    {
     for (channel = 0; channel < pAdap->num_channels; channel++)
	{
	 pChan = &pAdap->IDEChannel[channel];
	 for (device = 0; device < 2; device++)
	    {
	     if (pChan->DeviceFlags[device] & DFLAGS_DEVICE_PRESENT)
		{
		 ATLAS_OUTB((u8)((device << 4) | 0xA0), pChan->io_ports[IDE_SELECT_OFFSET]);
		 AtlasGetDeviceCBR(device);
		 status[(channel * 2) + device] = ATLAS_INB(pChan->io_ports[IDE_STATUS_OFFSET]);
		}
	     else
		{
		 status[(channel * 2) + device] = 0;
		}
	    }
	}

     /*
      * ATA device should present status 0x50 after reset.
      * ATAPI device should present status 0 after reset.
      */
     if ((status[0] != IDE_STATUS_IDLE && status[0] != 0x0) ||
	 (status[1] != IDE_STATUS_IDLE && status[1] != 0x0) ||
	 (status[2] != IDE_STATUS_IDLE && status[2] != 0x0) ||
	 (status[3] != IDE_STATUS_IDLE && status[3] != 0x0))
	{
	 udelay(30);
	 dprintk("Reset status: 0x%x, \n!\n", status[0]);
	}
     else
	{
	 break;
	}
    }
 ATLAS_INOUT_DBG_EN;
 
 if (i == 1000)
    {
     dprintk("AtlasResetAdapter Fail!\n");
     dprintk("Device status after reset = [0x%x, 0x%x, 0x%x, 0x%x]\n",
		status[0], status[1], status[2], status[3]);
     return FALSE;
    }
 else
    {
     dprintk("AtlasResetAdapter Success!\n");
     return TRUE;
    }

} /* end AtlasResetAdapter */


/************************************************************************
 * Switch to PIO/DMA mode if necessary. *
 ************************************************************************/
void
AtlasSwitchPioDmaMode(PChannel pChan, u8 DeviceId)
{
 u8			channel;
 u8			device;

 /*
  * These tables are for performance issue. Better formance than lots
  * of "Shifts".
  */


 static const u16	pioMode[5] = { 300, 290, 290, 90, 70 }; // PIO Mode 0-4
 
 static const u16	mdmaMode[3][3] = { { 50, 220, 260 },  // MDMA 0 Param 1-3
 					   { 30,  80,  70 },  // MDMA 1 Param 1-3
 					   { 25,  70,  50 }}; // MDMA 2 Param 1-3
 					   
 static const u16	udmaMode[6][4] = { { 20,  70, 120, 70 },  // UDMA 0 Param 1-4
 					   { 20,  48,  80, 48 },  // UDMA 1 Param 1-4
 					   { 20,  31,  60, 31 },  // UDMA 2 Param 1-4
 					   { 20,  21,  45, 20 },  // UDMA 3 Param 1-4
 					   { 20,   7,  30,  7 },  // UDMA 4 Param 1-4
 					   { 20,   5,  20, 10 }}; // UDMA 5 Param 1-4
 					    					    					   
 /*
  * channel --> 0-1; device --> 0-1; DeviceId --> 0-3;
  */
 channel = DeviceId >> 1;
 device  = DeviceId & 1;

 /*
  * Do nothing if the mode switch is unnecessary.
  */
 if (!pChan->DoSwitch || pChan->ActiveDevice == DeviceId)
    {
     dprintk("AtlasSwitchDmaMode: do not need to switch mode!\n");
     return;
    }
 else
     dprintk("Switch mode!!\n");   
 
 /* Timing Parameter 2 */
 //ATLAS_OUTL((ROUNDUP(pioMode[pChan->PioMode[device]])<<8)|ATLAS_WAIT_PARM,
 ATLAS_OUTL((ROUNDUP(pioMode[pChan->PioMode[device]])<<8)|0x7F,
 	    pChan->io_ports[ATLAS_TP2]);
 	     	    

 /*
  * Running on MULTIWORD_DMA mode.
  */
 if (pChan->TraType[device] == MULTIWORD_DMA)
    {
      ATLAS_OUTL((ROUNDUP(mdmaMode[pChan->DmaMode[device]][0])<<0)|
                 (ROUNDUP(mdmaMode[pChan->DmaMode[device]][1])<<8)|
                 (ROUNDUP(mdmaMode[pChan->DmaMode[device]][2])<<16),
 	          pChan->io_ports[ATLAS_TP1]);
    }
 /*
  * Running on ULTRA DMA mode.
  */
 else if(pChan->TraType[device] == ULTRA_DMA)
    {
      u32 udmamode;
      udmamode = ((ROUNDUP(udmaMode[pChan->DmaMode[device]][0])<<0) |\
         	  (ROUNDUP(udmaMode[pChan->DmaMode[device]][1])<<8)| \
                  (ROUNDUP(udmaMode[pChan->DmaMode[device]][2])<<16)| \
                  (ROUNDUP(udmaMode[pChan->DmaMode[device]][3])<<24));

      // When Roundup to 0, ex. 50MHz clock UDMA5, it meanless 
      if(udmamode==0)
       ATLAS_OUTL(0x00010000, pChan->io_ports[ATLAS_TP1]);	 
      else
       ATLAS_OUTL(udmamode, pChan->io_ports[ATLAS_TP1]);

      //ATLAS_OUTL((ROUNDUP(pioMode[pChan->PioMode[1]])<<8)|0x7F,
            //         pChan->io_ports[ATLAS_TP2]);


	  ATLAS_OUTL((ROUNDUP(pioMode[0])<<8)|0x7F, pChan->io_ports[ATLAS_TP2]);

    }

 /*
  * Record the Active device on this channel
  */
 pChan->ActiveDevice = device;

} /* end AtlasSwitchPioDmaMode */

/************************************************************************
 * Atlas read/write routine.
 ************************************************************************/
u32
AtlasReadWrite(PChannel pChan, PSCSI_REQUEST_BLOCK Srb)
{

 u8			statusByte = 0;
 u32			startingSector;
 u32			sectorNumber;
 u32			capacity;
 PATLAS_ADAPTER		pAdap;

 
 if (Srb->TargetId >= 4)
    {
     pAdap = atlas_adapters[1];
     if (Srb->TargetId < 6) pChan = &pAdap->IDEChannel[0];
     else		    pChan = &pAdap->IDEChannel[1];
    }
 else
    {
     pAdap = atlas_adapters[0];
     if (Srb->TargetId < 2) pChan = &pAdap->IDEChannel[0];
     else		    pChan = &pAdap->IDEChannel[1];
    }

 /*
  * Return error if overrun.
  */
 startingSector = ((PCDB)Srb->Cdb)->CDB10.LogicalBlockByte3	   |
		  ((PCDB)Srb->Cdb)->CDB10.LogicalBlockByte2 << 8   |
		  ((PCDB)Srb->Cdb)->CDB10.LogicalBlockByte1 << 16  |
		  ((PCDB)Srb->Cdb)->CDB10.LogicalBlockByte0 << 24;

 sectorNumber = (unsigned short) ((Srb->DataTransferLength + 0x1FF) / 0x200);
 

 capacity = pChan->IdentifyData[Srb->TargetId & 0x1].UserAddressableSectors;
 if (capacity == 0x0FFFFFFF)
    {
     capacity = pChan->IdentifyData[Srb->TargetId & 0x1].Capacity_48bit_LOW;
    }
    

 if ((startingSector + sectorNumber - 1) > capacity)
    {
     printk("AtlasReadWrite: disk[%d] over disk size.\n", Srb->TargetId);
     printk("capacity: %d. starting sector: %d. sector number: %d\n",
		capacity, startingSector, sectorNumber);
     return SRB_STATUS_ERROR;
    }

 /*
  * Select device.
  */
 AtlasGetDeviceCBR((Srb->TargetId & 0x1));

 /*
  * Wait for device ready (Not Busy and Not DRQ).
  */
 WaitForDeviceReady(pChan, statusByte);
 if ((statusByte & IDE_STATUS_BUSY) || (statusByte & IDE_STATUS_DRQ) ||
     (statusByte == 0))
    {
     printk("AtlasReadWrite: disk[%d] not ready. status=0x%x\n",
		Srb->TargetId, statusByte);
     if(++(pChan->RWRetry)>=RW_RETRY_COUNT)
     {
       pChan->RWRetry = 0; // Add retry count
       return SRB_STATUS_BUS_RESET;
     }
     else
      return SRB_STATUS_BUSY;
    }

 /*
  * First, switch to DMA or UDMA mode if running on bypass mode.
  */
  AtlasSwitchPioDmaMode(pChan,	Srb->TargetId);

 /*
  * Check the SCATTER/GATHER count. The upper will give the different
  * memory address depend on whether use_sg is used or not.
  */
 if (Srb->UseSg == 0)
    {
     IdeBuildDmaTable(pChan, Srb);
    }
 else
    {
     IdeBuildDmaSgTable(pChan, Srb);
    }
 
 /*
  * Start transfer the data.
  */
 IdeStartTransfer(pChan, Srb, startingSector, sectorNumber);

 /*
  * Wait for interrupt.
  */
 return SRB_STATUS_PENDING;

} /* end AtlasReadWrite */

/************************************************************************
 * Setup the transfer mode.
 ************************************************************************/
void
AtlasSetTransferMode
(
 PChannel		pChan,
 u32			DiskId,
 u8			TransferMode,
 u8	 		ModeNumber
)
{

 u8			statusByte = 0;

 /*
  * Select device.
  */
 ATLAS_OUTB((u8) ((DiskId & 0x1) << 4 | 0xA0), pChan->io_ports[IDE_SELECT_OFFSET]);

 /*
  * Wait for device ready (Not Busy and Not DRQ).
  */
 WaitForDeviceReady(pChan, statusByte);
 if ((statusByte & IDE_STATUS_BUSY) || (statusByte & IDE_STATUS_DRQ))
    {
     printk("AtlasSetTransferMode: disk[%d] not ready. status=0x%x\n",
		DiskId, statusByte);
     return;
    }

 /*
  * Feature number ==> 03
  *
  * Mode contained in Sector Count Register.
  *
  * Bits(7:3)	Bits(2:0)	Mode
  *
  * 00000	000		PIO default mode
  * 00000	001		PIO default mode, disable IORDY
  * 00001	mode		PIO flow control transfer mode
  * 00010	mode		Single Word DMA mode
  * 00100	mode		Multi-word DMA mode
  * 01000	mode		Ultra DMA mode
  */
 TransferMode |= ModeNumber;
 dprintk("%s: Set Transfer Mode 0x%x\n", __FUNCTION__, TransferMode);

 ATLAS_OUTB(0x03,			pChan->io_ports[IDE_FEATURE_OFFSET]);
 ATLAS_OUTB(TransferMode,		pChan->io_ports[IDE_NSECTOR_OFFSET]);
 ATLAS_OUTB(0,			pChan->io_ports[IDE_HCYL_OFFSET]);
 ATLAS_OUTB(0,			pChan->io_ports[IDE_MIDCYL_OFFSET]);
 ATLAS_OUTB(IDE_COMMAND_SET_FEATURE,	pChan->io_ports[IDE_COMMAND_OFFSET]);

 /* Trigger Atlas to start transfer */
 ATLAS_OUTW(ATLAS_CONTROL_START|ATLAS_CONTROL_XFER_NONE|ATLAS_CONTROL_RW, 
    	             pChan->io_ports[ATLAS_CONTROL]);
    	             
 /*
  * Check error.
  */
 WaitForAtlasCommandCompleteStatus(pChan, statusByte);

 if ((statusByte != IDE_STATUS_IDLE) && (statusByte != (IDE_STATUS_IDLE|IDE_STATUS_INDEX)) && (statusByte != 0))
    {
     printk("AtlasSetTransferMode: disk[%d] ", DiskId);
     printk("return unexpected status after issue command. 0x%x\n",
		statusByte);
    }

} /* end AtlasSetTransferMode */



int piomode = -1;
int dmamode = -1;

// henry 10142008
module_param(piomode, int, 0);    //  MODULE_PARM(piomode,    "i");
module_param(dmamode, int, 0);    //  MODULE_PARM(dmamode,    "i");

/************************************************************************
 * Set the best transfer mode for device.
 ************************************************************************/
void
AtlasSetBestTransferMode(PATLAS_ADAPTER pAdap, PChannel pChan, u8 channel)
{

 u8			i;
 u8			transferMode;
 u8			modeNumber;
 u8			device;
 PIDENTIFY_DATA2	ideIdentifyData;



 /*
  * 2003/07/24
  * If running on Firmware mode, get cable status from it.
  */

 for (i = 0; i < 2; i++)
    {
     /*
      * channel -->0 to 1.
      * device  -->0 or 1.
      */
     pChan->UseDma[i] = TRUE;
     device	      = i & 1;

     if (!(pChan->DeviceFlags[i] & DFLAGS_DEVICE_PRESENT) ||
	  (pChan->DeviceFlags[i] & DFLAGS_CONFIG_CHANGED))
	 continue;

     /*
      * Set PIO Mode.
      */
     ideIdentifyData = &pChan->IdentifyData[i];
     if ((!(ideIdentifyData->ValidFieldIndicator & 0x02)) || (ideIdentifyData->AdvancedPIOModes == 0))
	{
	 transferMode = PIO_FLOW_CONTROL;
	 modeNumber = 2;
	}
     else
	{
	 transferMode = PIO_FLOW_CONTROL;
	 modeNumber = GetHighestBit((u8) ideIdentifyData->AdvancedPIOModes) + 3;
	}

     if(piomode>=0) modeNumber=piomode; // Debug
     
     
     /* Selected Trasfer type & mode */     
     pChan->TraType[i] = PIO_FLOW_CONTROL;
     
     pChan->PioMode[i] = modeNumber;
     pChan->BestPioMode[i] = modeNumber;
     
     //AtlasSetTransferMode(pChan, i, transferMode, modeNumber);
     

     pChan->BestMdmaMode[i] = 2;
     pChan->BestUdmaMode[i] = 0;

     /*
      * Get the best transfer mode (maybe Ultra DMA or Multi-Word DMA).
      */
     ideIdentifyData = &pChan->IdentifyData[i];
     if ((!(ideIdentifyData->ValidFieldIndicator & 0x04)) || (ideIdentifyData->UltraDMASupport == 0))
	{
	 /*
	  * UltraDMA is not valid.
	  */
	 transferMode = MULTIWORD_DMA;
	 modeNumber = GetHighestBit(ideIdentifyData->MultiWordDMASupport);

	 // Device not support DMA, like DOM
	 if(modeNumber==0xff) goto bad_dma;

	 pChan->BestMdmaMode[i] = modeNumber;
	 
	 printk("The best transfer mode of Device[%d] is DMA-%d\n", i, modeNumber);
	}
     else
	{ 
	 transferMode = ULTRA_DMA;
	 modeNumber = GetHighestBit(ideIdentifyData->UltraDMASupport);


	 if(ATLAS_MAC_CLOCK==50 && modeNumber>4)
	 {
	  modeNumber = 4;
	  printk("50MHz clock can't work on Ultra-5, down to Ultra-4!\n");
	 }
 
	 if(modeNumber>ATLAS_MAX_UDMA)
	 {
	  modeNumber = ATLAS_MAX_UDMA;
	  printk("Atlas controller is fixed on Ultra-%d!\n", ATLAS_MAX_UDMA);	 	 	
  	  printk("Atlas controller is fixed on PIO-%d!\n", ATLAS_MAX_UDMA);	 	 	
	 }

	 pChan->BestUdmaMode[i] = modeNumber;
	 printk("The best transfer mode of Device[%d] is Ultra-%d\n", i, modeNumber);
	 printk("The best transfer mode of Device[%d] is PIO-%d\n", i, modeNumber);

	 /*
	  * If this is 40-pin cable. Limit to Ultra DMA mode 2.
	  */
	 if ((pChan->Cable80[i] == CABLE_40_PIN) && (modeNumber > 2))
	    {
	     printk("Limit transfer mode of Device[%d] to Ultra-2 for 40pin cable.\n", i);
	     modeNumber = 0x02;
	     pChan->BestUdmaMode[i] = modeNumber;
	    }
	}

     
     if(dmamode>=0) modeNumber=dmamode; // Debug
//Jason     
      pChan->TraType[i] = transferMode;
      pChan->DmaMode[i] = modeNumber;
     //AtlasSetTransferMode(pChan, i, transferMode, modeNumber);
     
bad_dma:
     atlas_set_piodma_mode(pChan, i, transferMode, modeNumber);
     
    }

     /*
      * Check shall we do switch between the two devices
      */
       for (i = 0; i < 1; i++)
 	{
	 pChan->DoSwitch = TRUE;

	 /*
	  * Master is not present
	  */
	 if (!(pChan->DeviceFlags[i] & DFLAGS_DEVICE_PRESENT) ||
	      (pChan->DeviceFlags[i] & DFLAGS_CONFIG_CHANGED))
	    {
	     printk("Channel %x: master is not present. No switch mode.\n", channel);
	     pChan->DoSwitch = FALSE;
	     continue;
	    }

	 /*
	  * Slave is not present
	  */
	 if (!(pChan->DeviceFlags[i + 1] & DFLAGS_DEVICE_PRESENT) ||
	      (pChan->DeviceFlags[i + 1] & DFLAGS_CONFIG_CHANGED))
	    {
	     printk("Channel %x: slave is not present. No switch mode.\n", channel);
	     pChan->DoSwitch = FALSE;
	     continue;
	    }

	 /*
	  * No switch if the two devices are running on the same mode.
	  */
	 if ((pChan->TraType[i] == pChan->TraType[i + 1]) &&
	     (pChan->PioMode[i] == pChan->PioMode[i + 1]) &&
	     (pChan->DmaMode[i] == pChan->DmaMode[i + 1]))
	    {
	     printk("Channel %x: two dev run on the same mode. No switch mode.\n", channel);
	     pChan->DoSwitch = FALSE;
	     continue;
	    }

	 printk("Channel %x: switch mode is %sneeded.\n", channel, pChan->DoSwitch?"":"not ");
	}

} /* end AtlasSetBestTransferMode */


/************************************************************************
 * This is the interrupt service routine for ATAPI IDE miniport driver.
 * TURE if expecting an interrupt.
 ************************************************************************/
//Jason
irqreturn_t
AtlasInterrupt(PChannel pChan)
{

 u8			statusByte;
 u8			status;
 u16			atlas_status;
 u32			i;
 unsigned long		bmbase;
 PSCSI_REQUEST_BLOCK	Srb;

 bmbase	  = pChan->dma_base;
 Srb	  = pChan->CurrentSrb;

ATLAS_INOUT_DBG_DIS;

 /*
  * To handle share IRQ condition. If the interrupt is not ours, just
  * return FALSE.
  */
 status = ATLAS_INB(pChan->io_ports[ATLAS_STATUS]);
 if ((status & ATLAS_STATUS_INT) == 0)
    {
     dprintk("AtlasInterrupt: suspicious interrupt (int bit is not on)\n");
     printk("AtlasInterrupt: suspicious interrupt (int bit is not on)\n");
     return FALSE;
    }
    
 /* Clear IRQ */
 ClearAtlasInt(pChan);
 IRQ_EDGE_CLEAR(ATLAS_AHB_RESOURCE_IRQ);

 if (Srb == 0 || pChan->ExpectingInterrupt == 0)
    {
     dprintk("AtlasInterrupt: suspicious interrupt!\n");
     printk("AtlasInterrupt: suspicious interrupt!\n");
     /*
      * Clear interrupt by reading status register.
      */
     return FALSE;
    }

ATLAS_INOUT_DBG_EN;

      dprintk("AtlasInterrupt: Received our interrupt!\n");

 /*
  * Get status.
  */
 AtlasGetDeviceCBR((Srb->TargetId & 1));
 GetBaseStatus(pChan, statusByte);
 GetAtlasStatus(pChan, atlas_status);

 /*
  * Handle ATAPI interrupt.
  */
 if (pChan->DeviceFlags[Srb->TargetId & 1] & DFLAGS_ATAPI_DEVICE)
    {
     return AtapiInterrupt(pChan);
    }

 pChan->ExpectingInterrupt = FALSE;

 if ((statusByte & IDE_STATUS_BUSY) || (statusByte & IDE_STATUS_DRQ))
    {
     /*
      * Ensure BUSY and DRQ is non-asserted.
      */
 ATLAS_INOUT_DBG_DIS;     
     for (i = 0; i < 100; i++)
	{
	 AtlasGetDeviceCBR((Srb->TargetId & 1));	
	 GetBaseStatus(pChan, statusByte);
	 GetAtlasStatus(pChan, atlas_status);
	 if (!(statusByte & IDE_STATUS_BUSY) && !(statusByte & IDE_STATUS_DRQ))
	    {
	     break;
	    }
	 mdelay(5);
	}

     if (i == 100)
	{
	 printk("AtlasInterrupt: disk[%x] return busy or drq status. status = 0x%x\n",
		Srb->TargetId, statusByte);
	 return FALSE;
	}
 ATLAS_INOUT_DBG_EN;	
     }

 if (statusByte & IDE_STATUS_ERROR)
    {
     /*
      * Stop bus master operation.
      */
     u8 err = ATLAS_INB(pChan->io_ports[IDE_ERROR_OFFSET]); 
     //ATLAS_OUTB(bmbase, 0);
	 
     printk("AtlasInterrupt Error: Disk[%d] status: %x, err: %x\n", Srb->TargetId, statusByte, err);

	 if(err&(1<<7))
	  printk("CRC error, cable problem?\n");

     /*
      * Map error to specific SRB status and handle request sense.
      */
     Srb->SrbStatus = MapError(pChan, Srb);
    }
 else if(atlas_status & (ATLAS_STATUS_DESC_ERR|ATLAS_STATUS_TX_ERR|ATLAS_STATUS_RX_ERR))
    {
     printk("AtlasInterrupt: atlas status error! status: %x\n", atlas_status);
     /*
      * Map error to specific SRB status and handle request sense.
      */
     Srb->SrbStatus = MapError(pChan, Srb);     
    }
 else
    {
     // Sync receiving data, if transaction is successful.
// henry 10142008
//     if (Srb->SrbFlags & SRB_FLAGS_DATA_IN)
//      pci_dma_sync_sg(pChan->pPciDev, Srb->DataBuffer, Srb->UseSg, PCI_DMA_FROMDEVICE);
    	
     Srb->SrbStatus = SRB_STATUS_SUCCESS;
     dprintk("AtlasInterrupt: atlas 0x%08x, ide 0x%08x\n", atlas_status, statusByte);
    }
    

#if FAULT_RETRY_EMULATION    
 get_random_bytes((char *)&Srb->emulate_fault, 4);   
 if(Srb->emulate_fault%100==1)
  Srb->SrbStatus = MapError(pChan, Srb);
#endif
 	   

 pChan->CurrentSrb = NULL;

 TaskDone(pChan, Srb);

 return TRUE;

} /* end AtlasInterrupt */

/************************************************************************
 * This is the interrupt service routine for ATAPI IDE miniport driver.
 * TRUE if expecting an interrupt. Remember the ATAPI io registers are
 * different from IDE io registers and this is for each channel not for
 * entire controller.
 ************************************************************************/
static irqreturn_t
AtapiInterrupt(PChannel pChan)
{

 u32			wordCount;
 u32			wordsThisInterrupt;
 u32			status;
 u32			i;
 u8			statusByte;
 u8			interruptReason;
 u8			target_id;
 PSCSI_REQUEST_BLOCK	srb;
 PATLAS_ADAPTER		pAdap;

 wordCount	    = 0;
 wordsThisInterrupt = 256;
 srb		    = pChan->CurrentSrb;
 target_id	    = srb->TargetId;

 if (target_id >= 4)
    {
     pAdap = atlas_adapters[1];
     if (target_id < 6) pChan = &pAdap->IDEChannel[0];
     else		pChan = &pAdap->IDEChannel[1];
    }
 else
    {
     pAdap = atlas_adapters[0];
     if (target_id < 2) pChan = &pAdap->IDEChannel[0];
     else		pChan = &pAdap->IDEChannel[1];
    }

 /*
  * Clear interrupt by reading status.
  */
 GetBaseStatus(pChan, statusByte);

 dprintk("AtapiInterrupt: entered with status (%x)\n", statusByte);

 if (statusByte & IDE_STATUS_BUSY)
    {
     /*
      * Ensure BUSY is non-asserted.
      */
 ATLAS_INOUT_DBG_DIS;     
     for (i = 0; i < 10; i++)
	{
	 GetBaseStatus(pChan, statusByte);
	 if (!(statusByte & IDE_STATUS_BUSY))
	    {
	     break;
	    }
	 mdelay(5);
	}
 ATLAS_INOUT_DBG_EN;
     if (i == 10)
	{
	 printk("AtapiInterrupt: BUSY on entry. Status %x\n", statusByte);
	 return FALSE;
	}
    }

 /*
  * Check for error conditions.
  */
 if (statusByte & IDE_STATUS_ERROR)
    {
     if (srb->Cdb[0] != SCSIOP_REQUEST_SENSE)
	{
	 /*
	  * Fail this request.
	  */
	 status = SRB_STATUS_ERROR;
	 goto CompleteRequest;
	}
    }

 /*
  * Check reason for this interrupt.
  */
 interruptReason = (ATLAS_INB(pChan->io_ports[ATAPI_INTREASON_OFFSET]) & 0x3);
 wordsThisInterrupt = 256;

 if (interruptReason == 0x1 && (statusByte & IDE_STATUS_DRQ))
    {
     /*
      * Write the packet.
      */
     dprintk("AtapiInterrupt: writing Atapi packet.\n");

     /*
      * Send CDB to device.
      */
     WriteBuffer(pChan, (unsigned short *)srb->Cdb, 6);

     return TRUE;
    }
 else if (interruptReason == 0x0 && (statusByte & IDE_STATUS_DRQ))
    {
     /*
      * Write the data.
      */

     /*
      * Pick up bytes to transfer and convert to words.
      */
     wordCount = ATLAS_INB(pChan->io_ports[ATAPI_LCYL_OFFSET]);
     wordCount |= ATLAS_INB(pChan->io_ports[ATAPI_HCYL_OFFSET]) << 8;

     /*
      * Covert bytes to words.
      */
     wordCount >>= 1;

     if (wordCount != pChan->WordsLeft)
	{
	 dprintk("AtapiInterrupt: %ud words requested; %ud words xferred\n",
		pChan->WordsLeft, wordCount);
	}

     /*
      * Verify this makes sense.
      */
     if (wordCount > pChan->WordsLeft)
	{
	 wordCount = pChan->WordsLeft;
	}

     /*
      * Ensure that this is a write command.
      */
     if (srb->SrbFlags & SRB_FLAGS_DATA_OUT)
	{
	 dprintk("AtapiInterrupt: write interrupt\n");
	 WaitOnBusy(pChan, statusByte);
	 WriteBuffer(pChan, pChan->DataBuffer, wordCount);
	}
     else
	{
	 printk("AtapiInterrupt: int reason %x, but srb is for a write %p.\n",
		interruptReason, srb);

	 /*
	  * Fail this request.
	  */
	 status = SRB_STATUS_ERROR;
	 goto CompleteRequest;
	}

     /*
      * Advance data buffer pointer and bytes left.
      */
     pChan->DataBuffer += wordCount;
     pChan->WordsLeft -= wordCount;

     return TRUE;
    }
 else if (interruptReason == 0x2 && (statusByte & IDE_STATUS_DRQ))
    {
     /*
      * Pick up bytes to transfer and convert to words.
      */
     wordCount = ATLAS_INB(pChan->io_ports[ATAPI_LCYL_OFFSET]);
     wordCount |= ATLAS_INB(pChan->io_ports[ATAPI_HCYL_OFFSET]) << 8;

     /*
      * Covert bytes to words.
      */
     wordCount >>= 1;
     if (wordCount != pChan->WordsLeft)
	{
	 dprintk("AtapiInterrupt: %ud words requested; %ud words xferred\n",
		pChan->WordsLeft, wordCount);
	}

     /*
      * Verify this makes sense.
      */
     if (wordCount > pChan->WordsLeft)
	{
	 wordCount = pChan->WordsLeft;
	}

     /*
      * Ensure that this is a read command.
      */
     if (srb->SrbFlags & SRB_FLAGS_DATA_IN)
	{
	 WaitOnBusy(pChan, statusByte);
	 ReadBuffer(pChan, pChan->DataBuffer, wordCount);

	 /*
	  * From Windows DDK
	  * You should typically set the ANSI-approved Version field,
	  * in the INQUIRY response, to at least 2.
	  */
	 if (srb->Cdb[0] == SCSIOP_INQUIRY)
	    {
	     /*
	      * Maybe it's not necessary in Linux driver.
	      */
	     *((unsigned char *)pChan->DataBuffer + 2) = 2;
	    }
	}
     else
	{
	 printk("AtapiInterrupt: int reason %x, but srb is for a read %p.\n",
		interruptReason, srb);

	 /*
	  * Fail this request.
	  */
	 status = SRB_STATUS_ERROR;
	 goto CompleteRequest;
	}

     /*
      * Advance data buffer pointer and bytes left.
      */
     pChan->DataBuffer += wordCount;
     pChan->WordsLeft -= wordCount;

     /*
      * Check for read command complete.
      */
     if (pChan->WordsLeft == 0)
	{
	 /*
	  * Work around to make many atapi devices return correct sector size
	  * of 2048. Also certain devices will have sector count == 0x00, check
	  * for that also.
	  */
	 if ((srb->Cdb[0] == 0x25) &&
	    ((pChan->IdentifyData[srb->TargetId & 1].GeneralConfiguration >> 8) & 0x1F) == 0x05)
	    {
	     pChan->DataBuffer -= wordCount;
	     if (pChan->DataBuffer[0] == 0x00)
		{
		 *((u32 *) &(pChan->DataBuffer[0])) = 0xFFFFFF7F;
		}

	     *((u32 *) &(pChan->DataBuffer[2])) = 0x00080000;
	     pChan->DataBuffer += wordCount;
	    }
	}
     return TRUE;
    }
 else if (interruptReason == 0x3 && !(statusByte & IDE_STATUS_DRQ))
    {
     /*
      * Command complete.
      */
     if (pChan->WordsLeft)
	{
	 status = SRB_STATUS_DATA_OVERRUN;
	}
     else
	{
	 status = SRB_STATUS_SUCCESS;
	}

CompleteRequest:

     if (status == SRB_STATUS_ERROR)
	{
	 /*
	  * Map error to specific SRB status and handle request sense.
	  */
	 printk("AtapiInterrupt error\n");

	 status = MapError(pChan, srb);

	 /*
	  * Try to recover it.... 2003/02/27
	  */


	 pChan->RDP = FALSE;
	}
     else
	{
	 /*
	  * Wait for busy to drop.
	  */
	 ATLAS_INOUT_DBG_DIS; 
	 for (i = 0; i < 30; i++)
	    {
	     GetStatus(pChan, statusByte);
	     if (!(statusByte & IDE_STATUS_BUSY))
		{
		 break;
		}
	     udelay(500);
	    }
         ATLAS_INOUT_DBG_EN;
	 if (i == 30)
	    {
	     /*
	      * Reset the controller.
	      */
	     printk("AtapiInterrupt: resetting due to BSY still up - %x.\n", statusByte);
	     AtapiResetController(pAdap, pChan);
	     return TRUE;
	    }

	 /*
	  * Check to see if DRQ is still up.
	  */
	 if (statusByte & IDE_STATUS_DRQ)
	    {
	     ATLAS_INOUT_DBG_DIS;	
	     for (i = 0; i < 500; i++)
		{
		 GetStatus(pChan, statusByte);
		 if (!(statusByte & IDE_STATUS_DRQ))
		    {
		     break;
		    }
		 udelay(100);
		}
	     ATLAS_INOUT_DBG_EN;
	     if (i == 500)
		{
		 /*
		  * Reset the controller.
		  */
		 printk("AtapiInterrupt: resetting due to DRQ still up - %x\n",
			statusByte);
		 AtapiResetController(pAdap, pChan);
		 return TRUE;
		}
	    }
	}

     /*
      * Clear interrupt expecting flag.
      */
     pChan->ExpectingInterrupt = FALSE;

     /*
      * Sanity check that there is a current request.
      */
     if (srb != NULL)
	{
	 /*
	  * Set status in SRB.
	  */
	 srb->SrbStatus = (u8)status;

	 /*
	  * Check for underflow.
	  */
	 if (pChan->WordsLeft)
	    {
	     /*
	      * Subtract out residual words and update if filemark hit,
	      * setmark hit , end of data, end of media...
	      */
	     if (!(pChan->DeviceFlags[srb->TargetId & 1] & DFLAGS_TAPE_DEVICE))
		{
		 if (status == SRB_STATUS_DATA_OVERRUN)
		    {
		     srb->DataTransferLength -= pChan->WordsLeft * 2;
		    }
		 else
		    {
		     srb->DataTransferLength = 0;
		    }
		}
	     else
		{
		 srb->DataTransferLength -= pChan->WordsLeft * 2;
		}
	    }

	 GetBaseStatus(pChan, statusByte);
	 if (pChan->RDP && !(statusByte & IDE_STATUS_DSC))
	    {
	     printk("-@@-\n");
	    }
	 else
	    {
	     /*
	      * Clear current SRB. Indicate ready for next request.
	      */
	     pChan->CurrentSrb = NULL;
	     TaskDone(pChan, srb);
	    }
	}
     else
	{
	 printk("AtapiInterrupt: no SRB!\n");
	}

     return TRUE;
    }
 else
    {
     /*
      * Unexpected int.
      */
     printk("AtapiInterrupt: unexpected interrupt. interruptReason %x. status %x.\n",
		interruptReason, statusByte);
     return FALSE;
    }

 return TRUE;

} /* end AtapiInterrupt */

/************************************************************************
 * IRQ handler.
 ************************************************************************/
//Jason
static irqreturn_t
//Irq_Handler(int irq, void *dev_id, struct pt_regs *regs)
Irq_Handler(int irq, void *dev_id)
{

 u8			i;
 u8			j;
 unsigned long		flags;
 PATLAS_ADAPTER		pAdap;
//Jason 
 irqreturn_t		ret = IRQ_NONE;   // henry 10142008
 //*handler	ret;
 

 spin_lock_irqsave(&io_request_lock, flags);

 /*
  * Scan for interrupt to process.
  */
 for (i = 0; i < NumAdapters; i++)
    {
     pAdap = atlas_adapters[i];
     if (pAdap->irq != irq) continue;

     for (j = 0; j < pAdap->num_channels; j++)
	{
// henry 10142008
	 ret = AtlasInterrupt(&pAdap->IDEChannel[j]);  // ret=
	}
    }

 spin_unlock_irqrestore(&io_request_lock, flags);

 //Jason 
 ret = IRQ_HANDLED;
 return ret;    // henry 10142008
 
} /* end Irq_Handler */

/************************************************************************
 * This routine handles IDE Verify.
 ************************************************************************/
u8
IdeVerify(PChannel pChan, PSCSI_REQUEST_BLOCK Srb)
{

 u32		startingSector;
 u32		sectors;
 u32		endSector;
 u32		sectorCount;
 u8		drvSelect;
 u8		statusByte = 0;

 /*
  * Select device
  */
 ATLAS_OUTB((u8)((Srb->TargetId & 0x1) << 4 | 0xA0), pChan->io_ports[IDE_SELECT_OFFSET]);

 /*
  * Wait for device ready (Not BUSY and Not DRQ)
  */
 WaitForDeviceReady(pChan, statusByte);
 if ((statusByte & IDE_STATUS_BUSY) || (statusByte & IDE_STATUS_DRQ) ||
     (statusByte == 0))
    {
     printk("IdeVerify: disk[%d] not ready. status=0x%x\n",
		Srb->TargetId, statusByte);
     return SRB_STATUS_BUSY;
    }

 /*
  * Get the starting sector number from CDB.
  */
 startingSector = ((PCDB)Srb->Cdb)->CDB10.LogicalBlockByte3 |
		  ((PCDB)Srb->Cdb)->CDB10.LogicalBlockByte2 << 8 |
		  ((PCDB)Srb->Cdb)->CDB10.LogicalBlockByte1 << 16 |
		  ((PCDB)Srb->Cdb)->CDB10.LogicalBlockByte0 << 24;

 sectorCount = (u16)(((PCDB)Srb->Cdb)->CDB10.TransferBlocksMsb << 8 |
		     ((PCDB)Srb->Cdb)->CDB10.TransferBlocksLsb );

 endSector = startingSector + sectorCount;

 /*
  * Drive has these number sectors.
  *
  * 48-bit addressing.
  */
 if (endSector > 0x0FFFFFFF)
    {
     sectors = pChan->IdentifyData[Srb->TargetId & 0x01].Capacity_48bit_LOW;

     dprintk("IdeVerify (48-bit): starting sector %d, Ending sector %d\n",
		startingSector, endSector);
				

     if (endSector > sectors)
	{
	 /*
	  * Too big, round down.
	  */
	 dprintk("IdeVerify: truncating request to %x blocks\n", sectors - startingSector - 1);

	 ATLAS_OUTB((u8) ((sectors - startingSector - 1) >> 8), pChan->io_ports[IDE_NSECTOR_OFFSET]);
	 ATLAS_OUTB((u8) (sectors - startingSector - 1), pChan->io_ports[IDE_NSECTOR_OFFSET]+1);
	}
     else
	{
	 /*
	  * Set up sector count register. Round up to next block.
	  */
	 if (sectorCount > 0xFFFF)
	    {
	     sectorCount = (u16)0xFFFF;
	    }

	 ATLAS_OUTB((u8)(sectorCount >> 8), pChan->io_ports[IDE_NSECTOR_OFFSET]);
	 ATLAS_OUTB((u8) sectorCount, pChan->io_ports[IDE_NSECTOR_OFFSET]+1);
	}

     /*
      * Indicate expecting an interrupt.
      */
     pChan->ExpectingInterrupt = TRUE;

     /*
      * Set up LBA address
      */
     ATLAS_OUTB((u8) (startingSector >> 24), pChan->io_ports[IDE_LOCYL_OFFSET]);
     ATLAS_OUTB((u8) startingSector,	       pChan->io_ports[IDE_LOCYL_OFFSET]+1);
     ATLAS_OUTB((u8) 0,		       pChan->io_ports[IDE_MIDCYL_OFFSET]);
     ATLAS_OUTB((u8) (startingSector >> 8),  pChan->io_ports[IDE_MIDCYL_OFFSET]+1);
     ATLAS_OUTB((u8) 0,		       pChan->io_ports[IDE_HCYL_OFFSET]);
     ATLAS_OUTB((u8) (startingSector >> 16), pChan->io_ports[IDE_HCYL_OFFSET]+1);

     /*
      * Send verify command.
      */
     ATLAS_OUTB(IDE_COMMAND_READ_VERIFY_EXT, pChan->io_ports[IDE_COMMAND_OFFSET]);
     
     /* Atlas 28bit, 48bit slection */
     ATLAS_ROWW(ATLAS_CONTROL_48BIT, pChan->io_ports[ATLAS_CONTROL]);
          
    }
 /*
  * 28-bit addressing
  */
 else
    {
     sectors = pChan->IdentifyData[Srb->TargetId & 0x01].UserAddressableSectors;

     dprintk("IdeVerify: starting sector %d, ending sector %d\n",
		startingSector, endSector);

     if (endSector > sectors)
	{
	 /*
	  * Too big, round down.
	  */
	 dprintk("IdeVerify: truncating request to %d blocks\n", sectors - startingSector - 1);
	 ATLAS_OUTB((u8)(sectors - startingSector - 1), pChan->io_ports[IDE_NSECTOR_OFFSET]);
	}
     else
	{
	 /*
	  * Set up sector count register. Round up to next block.
	  */
	 if (sectorCount > 0xFF)
	    {
	     sectorCount = (u16)0xFF;
	    }
	 ATLAS_OUTB((u8)sectorCount, pChan->io_ports[IDE_NSECTOR_OFFSET]);
	}

     /*
      * Indicate expecting an interrupt.
      */
     pChan->ExpectingInterrupt = TRUE;

     /*
      * Set up LBA address
      */
     ATLAS_OUTB((u8) startingSector,	 pChan->io_ports[IDE_LOCYL_OFFSET]);
     ATLAS_OUTB((u8) (startingSector >> 8), pChan->io_ports[IDE_MIDCYL_OFFSET]);
     ATLAS_OUTB((u8) (startingSector >> 16),pChan->io_ports[IDE_HCYL_OFFSET]);

     /*
      * Select driver, set LBA mode, set LBA (27:27)
      */
     drvSelect = (u8) (startingSector >> 24);
     drvSelect = drvSelect | (((u8) Srb->TargetId & 0x1) << 4) | 0xA0 | 0x40;
     ATLAS_OUTB(drvSelect, pChan->io_ports[IDE_SELECT_OFFSET]);

     /*
      * Send verify command.
      */
     ATLAS_OUTB(IDE_COMMAND_READ_VERIFY, pChan->io_ports[IDE_COMMAND_OFFSET]);

     /* Atlas 28bit, 48bit slection */
     ATLAS_ROWW(~ATLAS_CONTROL_48BIT, pChan->io_ports[ATLAS_CONTROL]);     
     
    }

  /* Start command */
  ATLAS_ROWW(ATLAS_CONTROL_START|ATLAS_CONTROL_XFER_NONE, pChan->io_ports[ATLAS_CONTROL]);    

 /*
  * Wait for interrupt.
  */
 return SRB_STATUS_PENDING;

} /* end IdeVerify */

/************************************************************************
 * This function is used to copy memory with overlapped destination and
 * source. I guess ScsiPortMoveMemory cannot handle this well. Can it?
 ************************************************************************/
void
AtlasMoveMemory(unsigned char * DestAddr, unsigned char * SrcAddr, u32 ByteCount)
{

 long		i;

 dprintk("AtlasMoveMemory: DestAddr=0x%p, SrcAddr=0x%p, ByteCount=0x%x\n",
		DestAddr, SrcAddr, ByteCount);

 if (DestAddr > SrcAddr)
    {
     /*
      * If Destination Area is in the back of the Source Area, copy from
      * the end of the requested area.
      */
     for (i = (ByteCount - 1); i >= 0; i--)
	{
	 *(DestAddr + i) = *(SrcAddr + i);
	}
    }
 else if (DestAddr < SrcAddr)
    {
     /*
      * If Destination Area is in the front of the Source Area, copy from
      * the begin of the requested area.
      */
     for (i = 0; i < ByteCount; i++)
	{
	 *(DestAddr + i) = *(SrcAddr + i);
	}
    }

} /* end AtlasMoveMemory */

/************************************************************************
 * Convert SCSI packet command to Atapi packet command.
 ************************************************************************/
void
Scsi2Atapi(PChannel pChan, PSCSI_REQUEST_BLOCK Srb)
{

 /*
  * Change the cdb length.
  */
 Srb->CdbLength = 12;

 /*
  * Because the block descripter and the header translation, we must
  * adjust the requested length.
  */
 Srb->DataTransferLength -= 4;

 /*
  * Record the original CDB for later restore.
  */
 memcpy(pChan->TempCdb, Srb->Cdb, MAXIMUM_CDB_SIZE);

 /*
  * Indicate that we have performed Scsi2Atapi function. And we must
  * restore the CDB back once the command complete.
  */
 pChan->ConvertCdb = TRUE;

 switch (Srb->Cdb[0])
    {
     /*
      * Convert the command from SCSIOP_MODE_SENSE (0x1A) to
      * SCSIOP_MODE_SENSE10 (0x5A).
      */
     case SCSIOP_MODE_SENSE:
	{
	 PSCSI_MODE_SENSE10 modeSense10 = (PSCSI_MODE_SENSE10) Srb->Cdb;
	 PSCSI_MODE_SENSE6  modeSense6 = (PSCSI_MODE_SENSE6) pChan->TempCdb;

	 /*
	  * 1. Zero out the whole CDB.
	  */
	 memset((unsigned char *)modeSense10, 0, MAXIMUM_CDB_SIZE);

	 /*
	  * 2. Fill in command code (SCSI_MODE_SENSE10).
	  */
	 modeSense10->OperationCode = ATAPI_MODE_SENSE;
	 modeSense10->Dbd = modeSense6->Dbd;
	 modeSense10->PageCode = modeSense6->PageCode;
	 modeSense10->Pc = modeSense6->Pc;
	 modeSense10->SubpageCode = modeSense6->SubpageCode;
	 modeSense10->AllocationLengthLsb = modeSense6->AllocationLength;
	 modeSense10->Control = modeSense6->Control;

	 /*
	  * 3. Becasuse we will fake a block descripter (-8), and
	  * translate the header (+4), so the requested length
	  * should be modified. That is, -8+4=-4 bytes.
	  */
	 modeSense10->AllocationLengthLsb -= 4;

	 break;
	}

     /*
      * Convert the command from SCSIOP_MODE_SELECT (0x15) to
      * SCSIOP_MODE_SELECT10 (0x5A).
      */
     case SCSIOP_MODE_SELECT:
	{
	 u8		tempHeader[sizeof(PSCSI_MODE_PARAMETER_HEADER6)];
	 u16		byteCount;
	 PSCSI_MODE_PARAMETER_HEADER10 header10 = (PSCSI_MODE_PARAMETER_HEADER10)Srb->DataBuffer;
	 PSCSI_MODE_PARAMETER_HEADER6 header6 = (PSCSI_MODE_PARAMETER_HEADER6)tempHeader;
	 PSCSI_MODE_SELECT10 modeSelect10 = (PSCSI_MODE_SELECT10)Srb->Cdb;
	 PSCSI_MODE_SELECT6 modeSelect6 = (PSCSI_MODE_SELECT6)pChan->TempCdb;

	 /*
	  * First, convert the command block.
	  */

	 /*
	  * 1. Zero out the whole CDB.
	  */
	 memset((unsigned char *)modeSelect10, 0, MAXIMUM_CDB_SIZE);

	 /*
	  * 2. Fill in command code (SCSI_MODE_SENSE10).
	  */
	 modeSelect10->OperationCode = ATAPI_MODE_SELECT;
	 modeSelect10->SPBit = modeSelect6->SPBit;
	 modeSelect10->PFBit = modeSelect6->PFBit;
	 modeSelect10->ParameterListLengthLsb = modeSelect6->ParameterListLength;
	 modeSelect10->Control = modeSelect6->Control;

	 /*
	  * 3. Becasuse we will remove the block descripter (-8), and
	  * translate the header (+4), so the requested length should
	  * be modified. That is, -8+4=-4 bytes.
	  */
	 modeSelect10->ParameterListLengthLsb -= 4;

	 /*
	  * Second, convert the parameter page format from SCSI to ATAPI.
	  */

	 /*
	  * Remove the mode parameter data (except the header and the
	  * block descripter).
	  */
	 byteCount = modeSelect6->ParameterListLength -
		     sizeof(SCSI_MODE_PARAMETER_HEADER6) -
		     sizeof(SCSI_MODE_PARAMTER_BLOCK_DESCRIPTER);
	 if (byteCount > 0)
	    {
	     AtlasMoveMemory(
		(unsigned char *) header10 + sizeof(SCSI_MODE_PARAMETER_HEADER10),
		(unsigned char *) header10 + sizeof(SCSI_MODE_PARAMETER_HEADER6) +
		sizeof(SCSI_MODE_PARAMTER_BLOCK_DESCRIPTER),
		byteCount);
	    }

	 /*
	  * Keep the original header6 (4 bytes) in tempHeader for later use
	  */
	 memcpy(tempHeader, header10, sizeof(SCSI_MODE_PARAMETER_HEADER6));

	 /*
	  * Change the "mode parameter header(6)" to "mode parameter header(10)"
	  * Notice: Remove the block descripter in SCSI-2 command out. It
	  * won't be used in MMC.
	  */
	 memset((unsigned char *) header10, 0, sizeof(SCSI_MODE_PARAMETER_HEADER10));
	 header10->ModeDataLengthLsb = header6->ModeDataLength;
	 header10->MediumType = header6->MediumType;
	 header10->DeviceSpecificParameter = header6->DeviceSpecificParameter;
	 header10->BlockDescriptorLengthLsb = header6->BlockDescriptorLength;

	 /*
	  * ATAPI doesn't support block descripter, so remove it from the
	  * mode paramter.
	  */
	 header10->BlockDescriptorLengthLsb = 0;

	 break;
	}
    }

} /* end Scsi2Atapi */

/************************************************************************
 * Send ATAPI packet command to device.
 ************************************************************************/
u8
AtapiSendCommand(PChannel pChan, PSCSI_REQUEST_BLOCK Srb)
{

 u8			statusByte;
 u8			byteCountLow;
 u8			byteCountHigh;
 u8			useDMA;
 u8			RevisionID = 0;
 u32			flags;
 int			i;
 unsigned long		bmAddress = pChan->dma_base;


 /*
  * Default use PIO mode.
  */
 useDMA	= 0;
 pChan->ConvertCdb = FALSE;

 /*
  * Make sure command is to ATAPI device.
  */
 flags = pChan->DeviceFlags[Srb->TargetId & 1];
 if (flags & (DFLAGS_SANYO_ATAPI_CHANGER | DFLAGS_ATAPI_CHANGER))
    {
     if ((Srb->Lun) > (pChan->DiscsPresent[Srb->TargetId & 1] - 1))
	{
	 /*
	  * Indicate no device found at this address.
	  */
	 return SRB_STATUS_SELECTION_TIMEOUT;
	}
    }
 else if (Srb->Lun > 0)
    {
     return SRB_STATUS_SELECTION_TIMEOUT;
    }

 if (!(flags & DFLAGS_ATAPI_DEVICE))
    {
     return SRB_STATUS_SELECTION_TIMEOUT;
    }

 /*
  * Select device 0 or 1.
  */
 ATLAS_OUTB((u8)(((Srb->TargetId & 0x1) << 4) | 0xA0),
	pChan->io_ports[ATAPI_SELECT_OFFSET]);

 /*
  * Verify that controller is ready for next command.
  */
 GetStatus(pChan, statusByte);

 dprintk("AtapiSendCommand: entered with status %x\n", statusByte);

 if (statusByte & IDE_STATUS_BUSY)
    {
     dprintk("AtapiSendCommand: device busy (%x)\n", statusByte);
     return SRB_STATUS_BUSY;
    }

 if (statusByte & IDE_STATUS_ERROR)
    {
     if (Srb->Cdb[0] != SCSIOP_REQUEST_SENSE)
	{
	 dprintk("AtapiSendCommand: error on entry: (%x)\n", statusByte);

	 /*
	  * Read the error reg. to clear it and fail this request.
	  */
	 return MapError(pChan, Srb);
	}
    }

 /*
  * If a tape drive doesn't have DSC set and the last command is
  * restrictive, don't send the next command. See discussion of
  * Restrictive Delayed Process commands in QIC-157.
  */
 if ((!(statusByte & IDE_STATUS_DSC)) && (flags & DFLAGS_TAPE_DEVICE) && pChan->RDP)
    {
     mdelay(1);
     dprintk("AtapiSendCommand: DSC not set. %x\n", statusByte);
     return SRB_STATUS_BUSY;
    }

 if (statusByte & IDE_STATUS_DRQ)
    {
     dprintk("AtapiSendCommand: enter with status (%x). Attempt to recover.\n",
	statusByte);

     /*
      * Try to drain the data that one preliminary device thinks that it has
      * to transfer. Hopefully this random assertion of DRQ will not be present
      * in production devices.
      */
     ATLAS_INOUT_DBG_DIS; 
     for (i = 0; i < 0x10000; i++)
	{		
	 GetStatus(pChan, statusByte);

	 if (statusByte & IDE_STATUS_DRQ)
	    {
	     /*
	      * Note: The data register is always referenced as a 16-bit word.
	      */
	     //ATLAS_INW(pChan->io_ports[ATAPI_DATA_OFFSET]);
	     ;
	    }
	 else
	    {
	     break;
	    }
	}
     ATLAS_INOUT_DBG_EN;	

     if (i == 0x10000)
	{
	 dprintk("AtapiSendCommand: DRQ still asserted.Status (%x)\n", statusByte);
	 dprintk("AtapiSendCommand: issued soft reset to Atapi device. \n");

	 AtapiSoftReset(pChan, Srb->TargetId);

	 /*
	  * Re-initialize Atapi device.
	  */
	 IssueIdentify(pChan, (Srb->TargetId & 1), IDE_COMMAND_ATAPI_IDENTIFY);

	 /*
	  * Inform the port driver that the bus has been reset.
	  */

	 /*
	  * Clean up device extension fields that AtapiStartIo won't.
	  */
	 pChan->ExpectingInterrupt = FALSE;
	 pChan->RDP = FALSE;

	 return SRB_STATUS_BUS_RESET;
	}
    }

 if (flags & (DFLAGS_SANYO_ATAPI_CHANGER | DFLAGS_ATAPI_CHANGER))
    {
     /*
      * As the cdrom driver sets the LUN field in the cdb, it must be removed.
      */
     Srb->Cdb[1] &= ~0xE0;

     if ((Srb->Cdb[0] == SCSIOP_TEST_UNIT_READY) && (flags & DFLAGS_SANYO_ATAPI_CHANGER))
	{
	 /*
	  * Torisan changer. TUR's are overloaded to be platter switches.
	  */
	 Srb->Cdb[7] = Srb->Lun;
	}
    }

 /*
  * Convert SCSI to ATAPI commands if needed
  */
 switch (Srb->Cdb[0])
    {
     case SCSIOP_MODE_SENSE:
     case SCSIOP_MODE_SELECT:
	 if (flags & DFLAGS_ATAPI_DEVICE)
	    {
	     Scsi2Atapi(pChan, Srb);
	    }
	 break;
    }

 if (pChan->UseDma[Srb->TargetId & 1])
    {
     switch (Srb->Cdb[0])
	{
	 case SCSIOP_READ:			/* (0x28)		*/
	 case 0xA8:				/* READ(12)		*/
	 case SCSIOP_READ_CD:

	     if (Srb->DataTransferLength == 0)
		{
		 break;
		}

	     /*
	      * First, switch to DMA or UDMA mode if running on Bypass mode.
	      */
	      AtlasSwitchPioDmaMode(pChan, Srb->TargetId);

	     /*
	      * Check the SCATTER/GATHER count. The upper will give the
	      * different memory address depend on whether use_sg is used
	      * or not.
	      */
	     if (Srb->UseSg == 0)
		{
		 IdeBuildDmaTable(pChan, Srb);
		}
	     else
		{
		 IdeBuildDmaSgTable(pChan, Srb);
		}

	     useDMA = 1;

	     /*
	      * Setup PRD table physical address.
	      */
	     ATLAS_OUTL(pChan->dmatable_dma, pChan->io_ports[ATLAS_PRD]);

	     /*
	      * Clear the status.
	      */
	     ATLAS_ROWW(ATLAS_STATUS_INT|ATLAS_STATUS_DESC_ERR|
                        ATLAS_STATUS_TX_ERR|ATLAS_STATUS_RX_ERR, 
                        pChan->io_ports[ATLAS_STATUS]);

	     break;
	} /* end switch (Srb->Cdb[0]) */
    }

 /*
  * Set data buffer pointer and words left.
  */
 pChan->DataBuffer = (unsigned short *)Srb->DataBuffer;

 if (useDMA)
    {
     pChan->WordsLeft = 0;
    }
 else
    {
     pChan->WordsLeft = Srb->DataTransferLength / 2;
    }

 ATLAS_OUTB((u8)(((Srb->TargetId & 0x1) << 4) | 0xA0),
		pChan->io_ports[ATAPI_SELECT_OFFSET]);

 WaitOnBusy(pChan, statusByte);

 /*
  * Write transfer byte count to registers.
  */
 byteCountLow  = (u8)(Srb->DataTransferLength & 0xFF);
 byteCountHigh = (u8)(Srb->DataTransferLength >> 8);

 if (Srb->DataTransferLength >= 0x10000)
    {
     byteCountLow = byteCountHigh = 0xFF;
    }

 ATLAS_OUTB(byteCountLow,  pChan->io_ports[ATAPI_LCYL_OFFSET]);
 ATLAS_OUTB(byteCountHigh, pChan->io_ports[ATAPI_HCYL_OFFSET]);
 ATLAS_OUTB(0,	     pChan->io_ports[ATAPI_INTREASON_OFFSET]);
 ATLAS_OUTB(0,	     pChan->io_ports[ATAPI_UNUSED1_OFFSET]);
 ATLAS_OUTB(useDMA,	     pChan->io_ports[ATAPI_FEATURE_OFFSET]);

 WaitOnBusy(pChan, statusByte);

 if (flags & DFLAGS_INT_DRQ)
    {
     /*
      * This device interrupts when ready to receive the packet.
      *
      * Write ATAPI packet command.
      */
     ATLAS_OUTB(IDE_COMMAND_ATAPI_PACKET, pChan->io_ports[IDE_COMMAND_OFFSET]);

     dprintk("AtapiSendCommand: wait for int. to send packet. status (%x)\n",
		statusByte);

     pChan->ExpectingInterrupt = TRUE;

     return SRB_STATUS_PENDING;
    }
 else
    {
     /*
      * Write ATAPI packet command.
      */
     ATLAS_OUTB(IDE_COMMAND_ATAPI_PACKET, pChan->io_ports[IDE_COMMAND_OFFSET]);

     /*
      * Wait for DRQ.
      */
     WaitOnBusy(pChan, statusByte);
     WaitForDrq(pChan, statusByte);

     if (!(statusByte & IDE_STATUS_DRQ))
	{
	 dprintk("AtapiSendCommand: DRQ never asserted (%x)\n", statusByte);
	 return SRB_STATUS_ERROR;
	}
    }

 /*
  * Need to read status register.
  */
 GetBaseStatus(pChan, statusByte);

 /*
  * Send CDB to device.
  * After detecting DRQ, the host writes the 12 bytes(6 words) of Command
  * to the Data Register.
  */
 WaitOnBusy(pChan, statusByte);
 WriteBuffer(pChan, (unsigned short *)Srb->Cdb, 6);

 /*
  * If running on DMA mode, start BUS MASTER operation.
  */
 if (useDMA)
    {
     /*
      * If SCSIOP_READ command is sent to an Audio CD, error will be
      * returned. But the error will be blocked by our controller if bus
      * master operation started. So wait for a short period to check if
      * error occurs. If error occurs, don't start bus master operation.
      */
     if (RevisionID == 0x10)
	{
	 ATLAS_INOUT_DBG_DIS;	
	 for (i = 0; i < 500; i++)
	    {
	     udelay(1);
	     statusByte = ATLAS_INB(bmAddress + 2);
	     if (statusByte & BM_STAT_FLG_INT)
		{
		 /*
		  * If error occurs, give up this round.
		  */
		 dprintk("AtapiSendCommand: command failed. Don't start bus master.");
		 dprintk("status=%x, i=%d\n", statusByte, i);

		 pChan->ExpectingInterrupt = TRUE;
		 return SRB_STATUS_PENDING;
		}
	    }
	 ATLAS_INOUT_DBG_EN;   
	}

	if (Srb->SrbFlags & SRB_FLAGS_DATA_IN)
	 ATLAS_ROWB(ATLAS_CONTROL_RW|ATLAS_CONTROL_START, pChan->io_ports[ATLAS_CONTROL]);
	else if (Srb->SrbFlags & SRB_FLAGS_DATA_OUT)
         ATLAS_ROWB((~ATLAS_CONTROL_RW)|ATLAS_CONTROL_START, pChan->io_ports[ATLAS_CONTROL]);

    } /* end if (useDMA) */

 /*
  * Indicate expecting an interrupt and wait for it.
  */
 pChan->ExpectingInterrupt = TRUE;

 return SRB_STATUS_PENDING;

} /* end AtapiSendCommand */

/************************************************************************
 * Program ATA registers for IDE disk transfer.
 ************************************************************************/
static u32
IdeSendCommand(PChannel pChan, PSCSI_REQUEST_BLOCK Srb)
{

 u8			statusByte;
 u32			status;
 u32			i;
 Scsi_Cmnd *		pREQ;
 unsigned char *	request_buffer;
 PINQUIRYDATA		inquiryData;

 pREQ	    = Srb->pREQ;
 status	    = SRB_STATUS_SUCCESS;
 statusByte = 0;
// henry 10152008
 dprintk("IdeSendCommand: command %x\n", Srb->Cdb[0]);
 
 switch (Srb->Cdb[0])
    {
     case SCSIOP_INQUIRY:
	 /*
	  * Filter out all TIDs but 0 and 1 since this is an IDE interface
	  * which support up to two devices.
	  */
	 if ((pREQ->device->lun != 0) ||  // 	 if ((pREQ->lun != 0) ||
            (!pChan->DeviceFlags[pREQ->device->id & 1] & DFLAGS_DEVICE_PRESENT))   // pREQ->target
	    {
	     /*
	      * Indicate no device found at this address.
	      */
	     status = SRB_STATUS_INVALID_TARGET_ID;
	     break;
	    }
	 else
	    {
// henry start 10152008
	     if(Srb->UseSg)
	     {
	      struct scatterlist *sg = (struct scatterlist *)Srb->DataBuffer;
	      Srb->DataBuffer = (char *)sg_to_virt(sg);
	     }
// henry end 10152008
	     request_buffer = Srb->DataBuffer;
	     inquiryData    = Srb->DataBuffer;

	     /*
	      * Zero INQUIRY data structure.
	      */
	     memset(request_buffer, 0, Srb->DataTransferLength);

	     /*
	      * Standard IDE interface only supports disks.
	      */
	     inquiryData->DeviceType = DIRECT_ACCESS_DEVICE;

	     /*
	      * Device type modifer.
	      */
	     request_buffer[1] = 0;

	     /*
	      * No ANSI/ISO compliance.
	      */
	     request_buffer[2] = 0;

	     /*
	      * Additional length.
	      */
	     request_buffer[4] = 31;
	     memcpy(&request_buffer[8], "ATLAS     ", 8);
	     memcpy(&request_buffer[16], "Atlas          ", 16);
	     memcpy(&request_buffer[32], "0.1", 3);

	     /*
 	      * Set the removable bit, if applicable.
	      */
	     if (pChan->DeviceFlags[pREQ->device->id & 1] & DFLAGS_REMOVABLE_DRIVE)  // pREQ->target & 1
		{
		 inquiryData->RemovableMedia = 1;
		}

	     status = SRB_STATUS_SUCCESS;
	    }

	 break;

     case SCSIOP_MODE_SENSE:
	 status = SRB_STATUS_INVALID_REQUEST;
	 break;

     case SCSIOP_TEST_UNIT_READY:
	 status = SRB_STATUS_SUCCESS;
	 break;

     case SCSIOP_READ_CAPACITY:
// henry start 10152008
	     if(Srb->UseSg)
	     {
	      struct scatterlist *sg = (struct scatterlist *)Srb->DataBuffer;
	      Srb->DataBuffer = (char *)sg_to_virt(sg);
	     }   
// henry end 10152008
	 /*
	  * Claim 512 byte blocks (big-endian).
	  */
	 ((PREAD_CAPACITY_DATA)Srb->DataBuffer)->BytesPerBlock = 0x20000;

	 /*
	  * Calculate last sector.
	  */
	 if (pChan->IdentifyData[pREQ->device->id & 0x01].UserAddressableSectors == 0x0FFFFFFF)
	    {
	     i = pChan->IdentifyData[pREQ->device->id & 0x01].Capacity_48bit_LOW - 1;
	    }
	 else
	    {
	     i = pChan->IdentifyData[pREQ->device->id & 0x01].UserAddressableSectors - 1;
	    }

	 ((PREAD_CAPACITY_DATA)Srb->DataBuffer)->LogicalBlockAddress =
		(((unsigned char *)&i)[0] << 24) | (((unsigned char *)&i)[1] << 16) |
		(((unsigned char *)&i)[2] << 8)  | ((unsigned char *)&i)[3];

	 status = SRB_STATUS_SUCCESS;
	 break;

     case SCSIOP_VERIFY:
	 status = IdeVerify(pChan, Srb);
	 break;

     case SCSIOP_READ:
     case SCSIOP_WRITE:
	 status = AtlasReadWrite(pChan, Srb);
	 break;

     case SCSIOP_START_STOP_UNIT:
	 /*
	  * Determine what type of operation we should perform
	  */
	 status = SRB_STATUS_SUCCESS;
	 break;

     case SCSIOP_REQUEST_SENSE:
	 /*
	  * This function makes sense buffers to report the results
	  * of the original GET_MEDIA_STATUS command
	  */
	 status = SRB_STATUS_INVALID_REQUEST;
	 break;

     default:
	 printk("IdeSendCommand: unsupported command %x\n", Srb->Cdb[0]);
	 status = SRB_STATUS_INVALID_REQUEST;
    } /* end switch */

 return status;

} /* end IdeSendCommand */

/************************************************************************
 * This routine is called from the SCSI port driver synchronized with
 * the kernel to start an IO request. If the current SRB is busy, return
 * FALSE, else return TURE.
 ************************************************************************/
static void
AtapiStartIo(PChannel pChan, PSCSI_REQUEST_BLOCK Srb)
{

 u32			status = 0;

 /*
  * Determine which function.
  */
 switch (Srb->Function)
    {
     case SRB_FUNCTION_EXECUTE_SCSI:
	 /*
	  * Sanity check. Only one request can be outstanding on a
	  * controller.
 	  */
 	 if (pChan->CurrentSrb)
	    {
	     printk("AtapiStartIo: already have a request!\n");
	     status = SRB_STATUS_BUSY;
	     Srb->SrbStatus = SRB_STATUS_BUSY;
	     goto busy;
	    }

	 /*
	  * Indicate that a request is active on the controller.
	  */
	 pChan->CurrentSrb = Srb;
	 Srb->SrbStatus	   = SRB_STATUS_PENDING;

	 /*
 	  * Send command to device.
	  */
	 if (pChan->DeviceFlags[Srb->TargetId & 1] & DFLAGS_ATAPI_DEVICE)
	    {
	     /*
	      * If this is ATAPI device.
	      */
	     status = AtapiSendCommand(pChan, Srb);
	    }
	 else if (pChan->DeviceFlags[Srb->TargetId & 1] & DFLAGS_DEVICE_PRESENT)
	    {
	     /*
	      * If this is IDE device.
	      */
	     status = IdeSendCommand(pChan, Srb);
	    }
	 else
	    {
	     /*
	      * Nothing else.
	      */
	     dprintk("%s: dev%d SRB_STATUS_SELECTION_TIMEOUT\n", __FUNCTION__, (Srb->TargetId & 1)); 
	     status = SRB_STATUS_SELECTION_TIMEOUT;
	    }

	 break;

     case SRB_FUNCTION_IO_CONTROL:
	 /*
	  * IO control function.
	  */
	 printk("AtapiStartIo: IO control\n");
	 break;

    default:
	 /*
	  * Indicate unsupported command.
	  */
	 status = SRB_STATUS_INVALID_REQUEST;
	 break;

    } /* end switch */

 busy:
 if (status != SRB_STATUS_PENDING)
    {
     /*
      * Set status in SRB.
      */
     Srb->SrbStatus = (u8)status;
     dprintk("AtapiStartIo: status=%x\n", status);
     TaskDone(pChan, Srb);
    }

} /* end AtapiStartIo */

/************************************************************************
 * Convert Scsi_Cmnd structure to SCSI_REQUEST_BLOCK.
 ************************************************************************/
static void
MapRequest(Scsi_Cmnd * pREQ, PSCSI_REQUEST_BLOCK Srb)
{

 Srb->Length 	= sizeof(SCSI_REQUEST_BLOCK);
 Srb->CdbLength	= pREQ->cmd_len;
 Srb->TargetId	= pREQ->device->id;  // pREQ->target;
 Srb->Lun	= pREQ->device->lun;  // pREQ->lun;
//Jason
// Srb->UseSg	= pREQ->use_sg;
 Srb->UseSg	= scsi_sg_count(pREQ);
 	
 /*
  * Copy the actual command from Scsi_Cmnd to CDB.
  */
 memcpy(Srb->Cdb, pREQ->cmnd, Srb->CdbLength);

 /*
  * Always the SCSI_FUNCTION_EXECUTE_SCSI now.
  */
 Srb->Function = SRB_FUNCTION_EXECUTE_SCSI;

 Srb->SrbStatus		    = 0;
 Srb->ScsiStatus	    = 0;
 Srb->SenseInfoBufferLength = 16;

 /*
  * The CDB's first byte is operation code.
  */
 if ((Srb->Cdb[0] == SCSIOP_WRITE6) || (Srb->Cdb[0] == SCSIOP_WRITE) ||
     (Srb->Cdb[0] == SCSIOP_MODE_SELECT10))
    {
     Srb->SrbFlags = SRB_FLAGS_DATA_OUT;
    }
 else
    {
     Srb->SrbFlags = SRB_FLAGS_DATA_IN;
    }

 Srb->TimeOutValue    = 0;
 Srb->SenseInfoBuffer = pREQ->sense_buffer;

 if (Srb->Cdb[0] == SCSIOP_REQUEST_SENSE)
    {
     Srb->DataTransferLength = 0x40;
     Srb->DataBuffer	     = pREQ->sense_buffer;
    }
 else
    {
//Jason
// 	 Srb->DataTransferLength = pREQ->request_bufflen;
//	 Srb->DataBuffer	     = pREQ->request_buffer;
 	 Srb->DataTransferLength = scsi_bufflen(pREQ);
     Srb->DataBuffer	     = scsi_sglist(pREQ);
    }

//Jason
//	if (pREQ->use_sg)
	if (scsi_sg_count(pREQ))
    {
     Srb->WorkingFlags |= SRB_WFLAGS_USE_SG;
    }

 Srb->pREQ = pREQ;

} /* end MapRequest */
	
/************************************************************************
 * A task execution has been done. For OS request, we need to Notify OS
 * and invoke next take which wait at queue.
 ************************************************************************/
static void
TaskDone(PChannel pChan, PSCSI_REQUEST_BLOCK Srb)
{
 Scsi_Cmnd * pREQ = Srb->pREQ;

 pChan->RetryCount = 0;

 switch (SRB_STATUS(Srb->SrbStatus))
    {
     case SRB_STATUS_SUCCESS:
         pChan->RWRetry = 0; // Reset rw retry count
	 pREQ->result = (DID_OK << 16);
	 break;

     case SRB_STATUS_SELECTION_TIMEOUT:
	 pREQ->result = (DID_NO_CONNECT << 16);
	 break;

     case SRB_STATUS_ABORTED:
	 pREQ->result =	(DID_ABORT << 16);
	 break;

     case SRB_STATUS_BUSY:
	 pREQ->result =	(DID_BUS_BUSY << 16);
	 break;

     case SRB_STATUS_BUS_RESET:
	 pREQ->result =	(DID_RESET << 16);
	 break;

     case SRB_STATUS_INVALID_TARGET_ID:
     case SRB_STATUS_INVALID_PATH_ID:
     case SRB_STATUS_INVALID_LUN:
     case SRB_STATUS_NO_HBA:
	 pREQ->result = (DID_BAD_TARGET << 16);
	 break;

     case SRB_STATUS_NO_DEVICE:
	 pREQ->result = (DID_BAD_TARGET << 16);
	 break;

     case SRB_STATUS_ERROR:
	 pREQ->result = (DRIVER_SENSE << 24) | (DID_OK << 16) |
		 	(CHECK_CONDITION << 1);
	 break;

     // Request Retry
     case SRB_STATUS_NEED_REQUEUE:
	 pREQ->result = (DID_SOFT_ERROR << 16);
	 break;
    	 	 
    }

 dprintk("TaskDone(pChan=%p, pREQ=%p, result=%x)\n",
		pChan, pREQ, pREQ->result);

 /*
  * Notify OS that this OS request has been done.
  */
 pREQ->scsi_done(pREQ);

 pChan->CurrentSrb = NULL;
 
 /*
  * Check the queue again.
  */
 TaskQueue(); 
 

} /* end TaskDone */

/************************************************************************
 * Start a command, doing convert first.
 ************************************************************************/
static void
TaskStart(PChannel pChan, Scsi_Cmnd *pREQ)
{

 PSCSI_REQUEST_BLOCK	Srb;

 dprintk("TaskStart(pChan=%p, pREQ=%p)\n", pChan, pREQ);

 Srb = &pChan->_Srb;

 /*
  * Clear the SRB structure.
  */
 memset(Srb, 0, sizeof(SCSI_REQUEST_BLOCK));

 /*
  * Convert Scsi_Cmnd structure to SCSI_REQUEST_BLOCK.
  */
 MapRequest(pREQ, Srb);

 /*
  * Start IDE I/O command.
  */
 AtapiStartIo(pChan, Srb);

} /* end TaskStart */

/************************************************************************
 * Check if queue is empty. If there are request in queue, transfer the
 * request to HIM's request and execute the request.
 ************************************************************************/
static void
TaskQueue(void)
{

 unsigned long		flags;
 Scsi_Cmnd *		SCpnt;
 PChannel		pChan;
 PATLAS_ADAPTER		pAdap;

 check_next:

 
 if (atlas_req_last != NULL)
    {
     spin_lock_irqsave(&queue_request_lock, flags);
                   
     SCpnt = (Scsi_Cmnd *)atlas_req_last->SCp.ptr;

     if (atlas_req_last == SCpnt)
	{
	 atlas_req_last = NULL;
	}
     else
	{
	 atlas_req_last->SCp.ptr = (char *)SCpnt->SCp.ptr;
	}
     spin_unlock_irqrestore(&queue_request_lock, flags);

     /*
      * Check the command.
      */
     if (SCpnt->device->host)  // SCpnt->host
	{
// if ((SCpnt->channel != 0) || (SCpnt->target >= (4 * NumAdapters)))
	 if ((SCpnt->device->channel != 0) || (SCpnt->device->id >= (4 * NumAdapters)))
	    {
	     /*
	      * Returns that we have a bad target.
	      */
	     SCpnt->result = (DID_BAD_TARGET << 16);
	     SCpnt->scsi_done(SCpnt);
	     goto check_next;
	    }
	}

     if (SCpnt->device->id >= 4)  // SCpnt->target >= 4
	{
	 pAdap = atlas_adapters[1];
	 if (SCpnt->device->id < 6)	 pChan = &pAdap->IDEChannel[0];  // SCpnt->target < 6
	 else if(pAdap->num_channels < 2) // Only have one channel on this adapter
	 {
	     /*
	      * Returns that we have a bad target.
	      */
	     SCpnt->result = (DID_BAD_TARGET << 16);
	     SCpnt->scsi_done(SCpnt);
	     goto check_next;	  	
	 }	 	 
	 else			 pChan = &pAdap->IDEChannel[1];
	}
     else
	{
	 pAdap = atlas_adapters[0];
	 if (SCpnt->device->id < 2)	 pChan = &pAdap->IDEChannel[0];  // SCpnt->target < 2
	 else if(pAdap->num_channels < 2) // Only have one channel on this adapter
	 {
	     /*
	      * Returns that we have a bad target.
	      */
	     SCpnt->result = (DID_BAD_TARGET << 16);
	     SCpnt->scsi_done(SCpnt);
	     goto check_next;	  	
	 }
	 else			 pChan = &pAdap->IDEChannel[1];
	}

     TaskStart(pChan, SCpnt);
     return;
    }

} /* end TaskQueue */

/****************************************************************
 * Name:	atlas_queuecommand
 * Description:	Process a queued command from the SCSI manager.
 * Parameters:	SCpnt - Pointer to SCSI command structure.
 *		done  - Pointer to done function to call.
 * Returns:	Status code.
 ****************************************************************/
int
atlas_queuecommand(Scsi_Cmnd *SCpnt, void (*done)(Scsi_Cmnd *))
{

 unsigned long		flags;

 /*
  * Hooks the done routine.
  */
 SCpnt->scsi_done = (void *)done;

 spin_lock_irqsave(&queue_request_lock, flags);
 if (atlas_req_last == NULL)
    {
     SCpnt->SCp.ptr = (char *)SCpnt;
    }
 else
    {
     SCpnt->SCp.ptr = atlas_req_last->SCp.ptr;
     atlas_req_last->SCp.ptr = (char *)SCpnt;
    }
 atlas_req_last = SCpnt;
 spin_unlock_irqrestore(&queue_request_lock, flags);

 TaskQueue();

 return 0;

} /* end atlas_queuecommand */

/****************************************************************
 * Name:	internal_done :LOCAL
 * Description:	Done handler for non-queued commands
 * Parameters:	SCpnt - Pointer to SCSI command structure.
 * Returns:	Nothing.
 ****************************************************************/
static void
internal_done(Scsi_Cmnd * SCpnt)
{

 SCpnt->SCp.Status++;

} /* end internal_done */

/****************************************************************
 * Name:	atlas_command
 * Description:	Process a command from the SCSI manager.
 * Parameters:	SCpnt - Pointer to SCSI command structure.
 * Returns:	Status code.
 ****************************************************************/
int
atlas_command(Scsi_Cmnd *SCpnt)
{

 unsigned long		timeout;

 SCpnt->SCp.Status = 0;
 atlas_queuecommand(SCpnt, internal_done);

 /*
  * Should be longer than hard-reset time.
  */
 timeout = jiffies + 60 * HZ;
 while (!SCpnt->SCp.Status && time_before(jiffies, timeout))
    {
     barrier();
    }

 if (!SCpnt->SCp.Status) SCpnt->result = (DID_ERROR << 16);

 return SCpnt->result;

} /* end atlas_command */

/************************************************************************
 * Enables/disables media status notification.
 ************************************************************************/
#if 0
static void
IdeMediaStatus(u8 EnableMSN, PChannel pChan, u8 Device)
{

 u8			statusByte;
 u8			errorByte;

 statusByte = 0;

 if (EnableMSN == TRUE)
    {
     /*
      * If supported enable Media Status Notification support.
      */
     if ((pChan->DeviceFlags[Device] & DFLAGS_REMOVABLE_DRIVE))
	{
	 ATLAS_OUTB((u8) (0x95), pChan->io_ports[IDE_FEATURE_OFFSET]);
	 ATLAS_OUTB(IDE_COMMAND_ENABLE_MEDIA_STATUS, pChan->io_ports[IDE_COMMAND_OFFSET]);

	 WaitOnBaseBusy(pChan, statusByte);

	 if (statusByte & IDE_STATUS_ERROR)
	    {
	     /*
	      * Read the error register.
	      */
	     errorByte = ATLAS_INB(pChan->io_ports[IDE_ERROR_OFFSET]);

	     printk("IdeMediaStatus: error enabling media status. status %u, error byte %u\n",
		statusByte, errorByte);
	    }
	 else
	    {
	     pChan->DeviceFlags[Device] |= DFLAGS_MEDIA_STATUS_ENABLED;
	     printk("IdeMediaStatus: media status notification supported!\n");
	     pChan->ReturningMediaStatus = 0;
	    }
	}
    }
 else /* end if EnableMSN == TRUE */
    {
     /*
      * Disable if previously enabled.
      */
     if ((pChan->DeviceFlags[Device] & DFLAGS_MEDIA_STATUS_ENABLED))
	{
	 ATLAS_OUTB((u8)(0x31), pChan->io_ports[IDE_FEATURE_OFFSET]);
	 ATLAS_OUTB(IDE_COMMAND_ENABLE_MEDIA_STATUS,
		pChan->io_ports[IDE_COMMAND_OFFSET]);

	 WaitOnBaseBusy(pChan, statusByte);
	 pChan->DeviceFlags[Device] &= ~DFLAGS_MEDIA_STATUS_ENABLED;
	}
    }

} /* end IdeMediaStatus */
#endif

static void ide_fixstring (u8 *s, const int bytecount, const int byteswap)
{
	u8 *p = s, *end = &s[bytecount & ~1]; /* bytecount must be even */

	if (byteswap) {
		/* convert from big-endian to host byte order */
		for (p = end ; p != s;) {
			unsigned short *pp = (unsigned short *) (p -= 2);
			*pp = ntohs(*pp);
		}
	}
	/* strip leading blanks */
	while (s != end && *s == ' ')
		++s;
	/* compress internal blanks and strip trailing blanks */
	while (s != end && *s) {
		if (*s++ != ' ' || (s != end && *s && *s != ' '))
			*p++ = *(s-1);
	}
	/* wipe out trailing garbage */
	while (p != end)
		*p++ = '\0';
}

void AdjustIdentifyData(PIDENTIFY_DATA2 id)
{
  /* swap serial */
  ide_fixstring((char *)id->SerialNumber, sizeof(id->SerialNumber), 1);

  /* swap firmware */
  ide_fixstring((char *)id->FirmwareRevision, sizeof(id->FirmwareRevision), 1);

  /* swap model */
  ide_fixstring((char *)id->ModelNumber, sizeof(id->ModelNumber), 1);
  
  dprintk("Mode: %s\n", (char *)id->ModelNumber);
  dprintk("Firmware: %.8s\n", (char *)id->FirmwareRevision);
  dprintk("Serial: %s\n", (char *)id->SerialNumber);
  
}

/************************************************************************
 * Issue IDENTIFY command to a device.
 * Either the standard (EC) or the ATAPI packet (A1) IDENTIFY.
 ************************************************************************/
 
static u8
IssueIdentify(PChannel pChan, u8 DeviceNumber, u8 Command)
{

 u8			statusByte = 0;

 /*
  * Check that the status register makes sense.
  */
 GetBaseStatus(pChan, statusByte);

 if (Command == IDE_COMMAND_IDENTIFY)
    {
     /*
      * Mask status byte ERROR bits.
      */
     statusByte &= ~(IDE_STATUS_ERROR | IDE_STATUS_INDEX);

     dprintk("IssueIdentify: checking for IDE. status (%x)\n", statusByte);

     /*
      * Check if register value is reasonable.
      */
     if (statusByte != IDE_STATUS_IDLE)
	{
	 /*
	  * Reset the channel.
	  */
	 printk("IssueIdentify: resetting channel.\n");
	 IdeHardReset(pChan, statusByte);

	 ATLAS_OUTB((u8)((DeviceNumber << 4) | 0xA0), pChan->io_ports[IDE_SELECT_OFFSET]);
	 GetBaseStatus(pChan, statusByte);
	 statusByte &= ~IDE_STATUS_INDEX;

	 if (statusByte != IDE_STATUS_IDLE)
	    {
	     /*
	      * Give up on this.
	      */
	     printk("IssueIdentify(IDE): disk[%d] of channel %d not ready. status=0x%x\n",
		     DeviceNumber, pChan->channel, statusByte);
	     return FALSE;
	    }
	}
    }
 else
    {
     printk("IssueIdentify: checking for ATAPI. status (%x)\n", statusByte);
     if ((statusByte & IDE_STATUS_BUSY) || (statusByte & IDE_STATUS_DRQ))
	{
	 /*
	  * Reset the device.
	  */
	 printk("IssueIdentify: resetting device%d.\n", DeviceNumber);
	 AtapiSoftReset(pChan, DeviceNumber);

	 ATLAS_OUTB((u8)((DeviceNumber << 4) | 0xA0), pChan->io_ports[IDE_SELECT_OFFSET]);
	 GetBaseStatus(pChan, statusByte);
	 if (statusByte != 0)
	    {
	     /*
	      * Give up on this.
	      */
	     printk("IssueIdentify(ATAPI): disk[%d] not ready. status=0x%x\n", DeviceNumber, statusByte);
	     return FALSE;
	    }
	}
    } 
    
     /*
      * Wait for device ready (Not Busy and Not DRQ).
      */
     ATLAS_OUTB((u8)((DeviceNumber << 4) | 0xA0), pChan->io_ports[IDE_SELECT_OFFSET]);
     WaitForDeviceReady(pChan, statusByte);
     if ((statusByte & IDE_STATUS_BUSY) || (statusByte & IDE_STATUS_DRQ))
	{
	 printk("IssueIdentify: disk[%d] not ready. status=0x%x\n", DeviceNumber, statusByte);
	 return FALSE;
	}

     /*
      * Send IDENTIFY command.
      */
     ATLAS_OUTB(Command, pChan->io_ports[IDE_COMMAND_OFFSET]);
     ReadBuffer(pChan, (unsigned short *)&pChan->FullIdentifyData, 256);
          

 /*
  * Check out a few capabilities / limitations of the device.
  * 01/29/2003
  */
 if (pChan->FullIdentifyData.SpecialFunctionsEnabled & 1)
    {
     /*
      * Determine if this drive supports the MSN functions.
      */
     printk("Marking drive %x as removable. SFE = %x\n",
		DeviceNumber, pChan->FullIdentifyData.SpecialFunctionsEnabled);
     pChan->DeviceFlags[DeviceNumber] |= DFLAGS_REMOVABLE_DRIVE;
    }

 memcpy(&pChan->IdentifyData[DeviceNumber], &pChan->FullIdentifyData,
		sizeof(IDENTIFY_DATA2));


 AdjustIdentifyData(&pChan->IdentifyData[DeviceNumber]);
  
 HexDump((void *)&pChan->FullIdentifyData, 256);

 if (pChan->IdentifyData[DeviceNumber].GeneralConfiguration & 0x20 &&
        Command != IDE_COMMAND_IDENTIFY)
    {
     /*
      * This device interrupts with the assertion of DRQ after receiving
      * Atapi Packet Command.
      */
     pChan->DeviceFlags[DeviceNumber] |= DFLAGS_INT_DRQ;

     dprintk(KERN_NOTICE "Device interrupts on assertion of DRQ.\n");
    }
 else
    {
     dprintk(KERN_NOTICE "Device does't interrupt on assertion of DRQ.\n");
    }

 if (((pChan->IdentifyData[DeviceNumber].GeneralConfiguration & 0xF00)
	== 0x100) && Command != IDE_COMMAND_IDENTIFY)
    {
     /*
      * This is a tape.
      */
     pChan->DeviceFlags[DeviceNumber] |= DFLAGS_TAPE_DEVICE;
     printk(KERN_NOTICE "IssueIdentify: device is a tape drive.\n");
    }
 else
    {
     dprintk(KERN_NOTICE "IssueIdentify: device is not a tape drive.\n");
    }      

  return TRUE;
}
 


/************************************************************************
 * Check this is the IDE or ATAPI disk then identify it.
 ************************************************************************/
static u8
atlas_find_device(PChannel pChan, u8 channel)
{

 u8			deviceResponded = FALSE;
 u8			deviceNumber;
 u8			signatureLow;
 u8			signatureHigh;
 u8			statusByte = 0;
 u8			retry = 2;

detect_again:
 /*
  * Clear expecting interrupt flag and current SRB field.
  */
 pChan->ExpectingInterrupt = FALSE;
 pChan->CurrentSrb	   = NULL;
  pChan->DeviceFlags[0] = pChan->DeviceFlags[1] = 0;

 /*
  * Search for devices in each channel.
  */
 for (deviceNumber = 0; deviceNumber < 2; deviceNumber++)
    {
     /*
      * Select the device.
      */
        ATLAS_OUTB((u8)((deviceNumber << 4) | 0xA0), pChan->io_ports[IDE_SELECT_OFFSET]);

#if FIND_DEVICE_WITH_NOP

     /*
      * Send NOP command.
      */
        ATLAS_OUTB(IDE_COMMAND_NOP, pChan->io_ports[IDE_COMMAND_OFFSET]);
     
      /*
       * Setup direction, Protocol NONE, and Start Transfer
       */
    	ATLAS_OUTW(ATLAS_CONTROL_START|ATLAS_CONTROL_XFER_NONE|ATLAS_CONTROL_RW, 
    	           pChan->io_ports[ATLAS_CONTROL]);     
    	             

   	if(!WaitForAtlasCommandCompleteShort(pChan))
   	{
   	 // TODO: Get stuck	 
   	}    	

#else
	AtlasGetDeviceCBR(deviceNumber);
#endif
         
        // Device is not existence.  
        if(!(ATLAS_INB(pChan->io_ports[IDE_STATUS_OFFSET])&IDE_STATUS_DRDY))
        {
         printk("FindDevices: Device %d is not present!\n", deviceNumber);
         continue;
        }
        
             
     signatureLow  = ATLAS_INB(pChan->io_ports[IDE_MIDCYL_OFFSET]);
     signatureHigh = ATLAS_INB(pChan->io_ports[IDE_HCYL_OFFSET]);

     dprintk("%s: dev%d signature hi: 0x%x, low: 0x%x\n", __FUNCTION__, deviceNumber, signatureHigh, signatureLow);
     
     if (signatureLow == 0x14 && signatureHigh == 0xEB)
	{
	 /*
	  * ATAPI signature found. Issue ATAPI packet identify command.
	  */
	 if (IssueIdentify(pChan, deviceNumber, IDE_COMMAND_ATAPI_IDENTIFY))
	    {
	     /*
	      * Indicate ATAPI device.
	      */
	     dprintk("atlas_find_device: channel %x device %x is ATAPI.\n",
		 channel, deviceNumber);

	     pChan->DeviceFlags[deviceNumber] |= DFLAGS_ATAPI_DEVICE;
	     pChan->DeviceFlags[deviceNumber] |= DFLAGS_DEVICE_PRESENT;
	     pChan->DeviceFlags[deviceNumber] &= ~DFLAGS_CONFIG_CHANGED;

	     deviceResponded = TRUE;

	     GetStatus(pChan, statusByte);
	     if (statusByte & IDE_STATUS_ERROR)
		{
		 //AtapiSoftReset(pChan, deviceNumber);
		}
	    }
	 else
	    {
	     /*
	      * Indicate no working device.
	      */
	     printk("FindDevices: Channel %x device %x doesn't respond.\n",
		channel, deviceNumber);
	     pChan->DeviceFlags[deviceNumber] &= ~DFLAGS_DEVICE_PRESENT;
	    }
	}
     else
	{
	 /*
	  * Select the device.
	  */
	 ATLAS_OUTB((u8)((deviceNumber << 4) | 0xA0), pChan->io_ports[IDE_SELECT_OFFSET]);
	 AtlasGetDeviceCBR(deviceNumber);

	 /*
	  * Check here for some SCSI adapters that incorporate IDE emulation.
	  */
	 GetStatus(pChan, statusByte);
    
	 /*
	  * No Disk.
	  */
	 if (statusByte == 0xFF || statusByte == 0x7F || statusByte == 0x0)
	    {
	     printk("FindDevices: Cannot find IDE device[%d]. status = %x\n", deviceNumber, statusByte);
	     continue;
	    }

	 /*
	  * Issue IDE Identify. If an ATAPI device is actually present,
	  * the signature will be asserted, and the drive will be
	  * recognized as such.
	  */
	 if (IssueIdentify(pChan, deviceNumber, IDE_COMMAND_IDENTIFY))
	    {
	     /*
	      * IDE drive found.
	      */
	     printk(KERN_WARNING "FindDevices: Device %u is IDE\n",
		(channel * 2) + deviceNumber);

	     pChan->DeviceFlags[deviceNumber] |= DFLAGS_DEVICE_PRESENT;
	     pChan->DeviceFlags[deviceNumber] &= ~DFLAGS_ATAPI_DEVICE;
	     pChan->DeviceFlags[deviceNumber] &= ~DFLAGS_CONFIG_CHANGED;

	     deviceResponded = TRUE;
	    }
	 else
	    {
	     printk(KERN_WARNING "FindDevices: Device %u is not present\n",
		(channel * 2) + deviceNumber);
	     pChan->DeviceFlags[deviceNumber] &= ~DFLAGS_DEVICE_PRESENT;
	    }
	    
	}
    }


 if(!deviceResponded && retry)
 {
  atlas_controller_init(NULL, pChan);	
  retry--;
  goto detect_again;
 }
 
 return deviceResponded;

} /* end atlas_find_device */



/************************************************************************
 * Initialize atlas controller
 ************************************************************************/ 
static int
atlas_controller_init(PATLAS_ADAPTER pAdap, PChannel pChan)
{
     u8 status;
          
     // 0. HSRST
     ATLAS_OUTW(ATLAS_CONTROL_HSRST, pChan->io_ports[ATLAS_CONTROL]);
     mdelay(25);	 
     	 
     /* A. Two CRF wait */	
     mdelay(25);
     ATLAS_ROWB(ATLAS_STATUS_CRF, pChan->io_ports[ATLAS_STATUS]);
     mdelay(25);
     ATLAS_ROWB(ATLAS_STATUS_CRF, pChan->io_ports[ATLAS_STATUS]);
     
     /* B. Wait init OK */
     WaitOnAtlasInitial(pChan, status);

     /* C. Clear interrupt after two CRF */
     ClearAtlasInt(pChan);
     
     if(!(status&ATLAS_STATUS_INIT_OK)) return -1; // Never get initial ok flag...
    
     /* D. Sync Dev Ctrl */     
     ATLAS_OUTB(ATLAS_DC_NIEN, pChan->io_ports[IDE_CONTROL_OFFSET]); 
         
     /* E. Assign Cable Detection */
     pChan->Cable80[0] = pChan->Cable80[1] = ATLAS_MAC_STATUS(ATLAS_INB(pChan->io_ports[ATLAS_MAC]))&0x1;
    
  return 0;	
}

/************************************************************************
 * Initialize a adapter, return 0 means success.
 ************************************************************************/
static int
atlas_init(PATLAS_ADAPTER pAdap, struct pci_dev *pPciDev)
{

 u8			z;
 u8			i;
// u8			j;
 u8			set_irq;
 u8			found=0;
 unsigned long		base_addr;	/* IDE I/O port base address	*/
 PChannel		pChan = NULL;		/* Use for each channel		*/
//Jason
 //register int nResult;
 
 /*
  * Common settings.
  */
#if ATLAS_PCI  
 pAdap->pci_bus	 = pPciDev->bus->number;
#endif 
 pAdap->devfn	 = pPciDev->devfn;
//Jason  
 //pAdap->irq	 = pPciDev->irq;
 pAdap->irq	 = INT_IDE;
 pAdap->irqOwned = 0;
 /*
  * Allocate buffer for IDE channles (One Atlas supports two channels)
  */
 pAdap->IDEChannel =
 (PChannel)kmalloc(sizeof(Channel) * pAdap->num_channels, GFP_ATOMIC);

 if (pAdap->IDEChannel == 0) 
    {
     printk("atlas_init: pChan allocate failed.\n");
     return -1;
    }

 memset(pAdap->IDEChannel, 0, sizeof(Channel) * pAdap->num_channels);

 set_irq = 1;
 for (i = 0; i < NumAdapters; i++)
    {
     if (atlas_adapters[i]->irqOwned == pAdap->irq) set_irq = 0;
    }

 /*
  * Request the irq (share irq) and hook the interrupt service routine.
  */
 if (set_irq)
    {
     int irqerr;
/*
//Jason
	nResult = register_chrdev(UMVP_IDE_MAJOR, ATLAS_DEV_NAME, &atlas_fops);
	if (nResult < 0){
		printk("nResult: %x \n");
		return (nResult);
	}
	// setup interrupt properties 
	disable_irq(INT_IDE); 			
	vic_set_intr_trigger(INT_IDE, vicc_level_activeHigh);
	mdelay(1000);
*/	
	//if ((irqerr=request_irq(pAdap->irq, Irq_Handler, SA_SHIRQ, ATLAS_DEV_NAME, pAdap)) < 0)  
    if ((irqerr=request_irq(pAdap->irq, Irq_Handler, IRQF_SHARED, ATLAS_DEV_NAME, pAdap)) < 0)
	{ 
	 printk("atlas_init: unable to allocate IRQ %d for %s, err: %d\n", pAdap->irq, pAdap->name, irqerr);
	 kfree(pAdap->IDEChannel);
	 return -1;
	}
     else
    {
		 printk("atlas_init: success to allocate IRQ %d for %s, err: %d \n", pAdap->irq, pAdap->name, irqerr);
		 IRQ_SET_EDGE_TRIGGER(pAdap->irq);
         IRQ_SET_RISING_EDGE(pAdap->irq);
    }
	
     pAdap->irqOwned = pAdap->irq;
    }

 /*
  * Get the IDE port and DMA registers.
  */
 for (i = 0; i < pAdap->num_channels; i++)
    {
     pChan = &pAdap->IDEChannel[i];
     /*
      * Reference the book "LINUX DEVICE DRIVER 2nd", Page 484
      * unsigned long pci_resource_start(struct pci_dev *dev, int bar);
      */
     /* Atlas only use BAR */ 
#if ATLAS_PCI     
     pChan->phyaddr  = pci_resource_start(pPciDev, i * 2 + 4);
     pChan->addrlen  = pci_resource_len(pPciDev, i * 2 + 4);
#else // ATLAS_AHB
     pChan->phyaddr  = ATLAS_AHB_RESOURCE_ADDR;
     pChan->addrlen  = ATLAS_AHB_RESOURCE_LEN;
#endif     
#if ATLAS_MEMIO          
#if ATLAS_PCI
     pChan->virtaddr = (u32)ioremap(pChan->phyaddr, pChan->addrlen);
#else // ATLAS_AHB
     pChan->virtaddr = IO_ADDRESS(UMVP_IDE_BASE); // Fixed virtual address on BSP startup
#endif  
#endif     

     base_addr	     = ATLAS_BAR(pChan);
     pChan->dma_base = base_addr + 0x10;

     if(ATLAS_INL(base_addr+ATLAS_SIG_OFFSET)==ATLAS_SIGNATURE)
     {
      printk(KERN_NOTICE "Found IDE Controller: %s, Running @ %dMHz\n", pAdap->name, ATLAS_MAC_CLOCK);
      found = 1;

#if ATLAS_PCI     
     printk("PCI Address: 0x%08x - 0x%08x [%ud]\n", pChan->virtaddr, pChan->phyaddr, pChan->addrlen);
#else // ATLAS_AHB
     printk("AHB Address: 0x%08x - 0x%08x [%ud]\n", pChan->virtaddr, pChan->phyaddr, pChan->addrlen);
#endif  

     }
     
        /* Atlas register map */
     	pChan->io_ports[IDE_DATA_OFFSET]    = 0x00;  /* (0) Atlas doesn't have programmed I/O */
     	pChan->io_ports[IDE_FEATURE_OFFSET] = base_addr + 0x00; /* (1) */
     	pChan->io_ports[IDE_NSECTOR_OFFSET] = base_addr + 0x02; /* (2) */
     	pChan->io_ports[IDE_LOCYL_OFFSET]   = base_addr + 0x04; /* (3) */ 
     	pChan->io_ports[IDE_MIDCYL_OFFSET]  = base_addr + 0x06; /* (4) */   
     	pChan->io_ports[IDE_HCYL_OFFSET]    = base_addr + 0x08; /* (5) */   
     	pChan->io_ports[IDE_SELECT_OFFSET]  = base_addr + 0x0A; /* (6) */
     	pChan->io_ports[IDE_COMMAND_OFFSET] = base_addr + 0x0B; /* (7) */
     	pChan->io_ports[IDE_CONTROL_OFFSET] = base_addr + 0x1C; /* (8) */
     	//pChan->io_ports[IDE_IRQ_OFFSET]     = base_addr;       /* (9) */
     	pChan->io_ports[IDE_ERROR_OFFSET]   = base_addr + 0x21; /* (10) */
     	pChan->io_ports[IDE_STATUS_OFFSET]  = base_addr + 0x20; /* (11) */
     	pChan->io_ports[ATLAS_CONTROL]      = base_addr + 0x0C; /* (12) */
     	pChan->io_ports[ATLAS_STATUS]	    = base_addr + 0x0E; /* (13) */
     	pChan->io_ports[ATLAS_PRD]          = base_addr + 0x10; /* (14) */
     	pChan->io_ports[ATLAS_TP1]          = base_addr + 0x14; /* (15) */
     	pChan->io_ports[ATLAS_TP2]          = base_addr + 0x18; /* (16) */
     	pChan->io_ports[ATLAS_MAC]  	    = base_addr + 0x22; /* (17) */
     	pChan->io_ports[ATLAS_XFER_SEC]     = base_addr + 0x24; /* (18) */
     	pChan->io_ports[ATLAS_TEST_CTRL]    = base_addr + 0x28; /* (19) */
     	pChan->io_ports[ATLAS_TEST_DATA]    = base_addr + 0x2C; /* (20) */
     	pChan->io_ports[ATLAS_PKT_CTRL]     = base_addr + 0x30; /* (21) */
     	pChan->io_ports[ATLAS_SECTOR_SIZE]  = base_addr + 0x34; /* (22) */        	
    }
    
    if(!found)
    {
     printk("Not Found IDE Controller!!\n");
     goto err;
    }

 /*
  * Initialize channels.
  */
 for (z = 0; z < pAdap->num_channels; z++)
    {
     pChan	    = &pAdap->IDEChannel[z];
     pChan->pPciDev = pPciDev;
     pChan->channel = z;


     /*
      * Hardware initialize.
      */
     if(atlas_controller_init(pAdap, pChan))
     {     	
       printk("%s: Atlas controller initial failed!\n", __FUNCTION__ );       
       goto err;
     }
	 else
		printk("Atlas controller initial success ! \n");       
       
     /*
      * Set Scatter/Gather List buffer for the channel.
      */
     IdeSetupDma(pChan, pChan->dma_base, 1);
     
     pChan->PIOVirt = pci_alloc_consistent(pChan->pPciDev, 512, &pChan->PIOPhy);
     
     /*
      * Find and identify the IDE or ATAPI device.
      */
     if(!atlas_find_device(pChan, z))
     {
       printk("%s: Atlas controller find device failed!\n", __FUNCTION__ );       
       goto err;       	
     }
	 else 
		printk("Atlas controller find device success ! \n");
	 
     /*
      * Set the best transfer mode.
      */
     AtlasSetBestTransferMode(pAdap, pChan, z);
     
    }
 printk("running finished ! \n");
 return 0;

err:
#if ATLAS_MEMIO 
#if ATLAS_PCI         
     iounmap((void *)pChan->virtaddr);
	 printk("iounmap ! \n");
#endif     
#endif
	 printk("free ! \n");
     pci_free_consistent(pChan->pPciDev , 512, pChan->PIOVirt, pChan->PIOPhy);  
     free_irq(pAdap->irq, pAdap);	
     kfree(pAdap->IDEChannel);
     return -1;

} /* end atlas_init */

/************************************************************************
 * This function will find and initialize any cards.
 ************************************************************************/
// atlas_detect(Scsi_Host_Template *tpnt)

int
atlas_detect(struct scsi_host_template *tpnt)
{

 u8			i;
 u8			j;
 u8			pci_id;
 PChannel		pChan;
 PATLAS_ADAPTER		pAdap;
 struct pci_dev *	pPciDev;
 

 /*
  * Search ATLAS chip.
  */
 pPciDev = NULL;
 pci_id  = 0;

#if ATLAS_PCI 

 while ((pPciDev = pci_find_device(ATLAS_VENDOR_ID, ATLAS_DEVICE_ID, pPciDev)))
    {
     if (PCI_FUNC(pPciDev->devfn)) continue;
     
#else // ATLAS_AHB
 
    {
    	pPciDev = kmalloc(sizeof(*pPciDev), GFP_KERNEL);
    	
    	if(pPciDev==NULL)
    	 return 0;
    	 
    	memset(pPciDev, 0, sizeof(*pPciDev)); 
    	pPciDev->irq = ATLAS_AHB_RESOURCE_IRQ;
    	pPciDev->dma_mask = ATLAS_AHB_RESOURCE_MASK;
    	pPciDev->dev.coherent_dma_mask = pPciDev->dma_mask;
    	
#endif

     /*
      * Allocate memory for Adapter.
      */
     pAdap = (PATLAS_ADAPTER)kmalloc(sizeof(ATLAS_ADAPTER), GFP_ATOMIC);
     if (pAdap == NULL)
	{
	 printk("atlas_detect: pAdap allocate failed.\n");
#if ATLAS_PCI	 
	 continue;
#else // ATLAS_AHB
	 return 0;
#endif	 
	}

     memset(pAdap, 0, sizeof(ATLAS_ADAPTER));
     pAdap->name	 = CONTROLLER_NAME_Atlas;
     pAdap->num_channels = 1;
     pAdap->pci_dev	 = pPciDev;


     if (atlas_init(pAdap, pPciDev) == 0)
	{
	 atlas_adapters[NumAdapters++] = pAdap;
	}
     else
        {
	 dprintk("%s: atlas_init failed !!\n", __FUNCTION__);
	 kfree(pAdap);
         return 0;
	}

     pci_id++;
    }

 /*
  * Reenable interrupt after initialization. 2003/04/28
  */
 for (i = 0; i < NumAdapters; i++)
    {
     pAdap = atlas_adapters[i];
     for (j = 0; j < pAdap->num_channels; j++)
	{
	 pChan = &pAdap->IDEChannel[j];
	 ATLAS_OUTB(ATLAS_DC_ENABLE, pChan->io_ports[IDE_CONTROL_OFFSET]);
	}
    }

 dprintk("Number of Adapter: %d\n", NumAdapters);
 

 
 if (NumAdapters)
    {
      	
     /*
      * Register a virtual host.
      */
     atlas_vhost = scsi_register(tpnt, 0);

     atlas_vhost->io_port		= 0;
     atlas_vhost->n_io_port	= 0;
     atlas_vhost->max_channel	= 0;
     atlas_vhost->max_id		= MAX_DEVICES;
     atlas_vhost->max_lun		= 1;
#if (LINUX_VERSION_CODE <= KERNEL_VERSION(2,5,63))
     scsi_set_pci_device(atlas_vhost, pPciDev);
#endif

     /*
      * Register the Shutdown Notification hook in the kernel.
      */
     register_reboot_notifier(&atlas_notifier);

    }


 return 1;

} /* end atlas_detect() */

/************************************************************************
 * Name:	atlas_release
 * Description:	Release resources allocated for a single each adapter.
 * Parameters:	pshost - Pointer to SCSI command structure.
 * Returns:	zero.
 ************************************************************************/
int
atlas_release(struct Scsi_Host *pshost)
{

 u8			i;
 PATLAS_ADAPTER		pAdap;


 /*
  * Free irq and memory.
  */
  printk("release ! \n");
  
 for (i = 0; i < NumAdapters; i++)
    {
     pAdap = atlas_adapters[i];
 
#if ATLAS_MEMIO
#if ATLAS_PCI
	iounmap((void *)pAdap->IDEChannel[0].virtaddr);
#endif	
#endif	
     if (pAdap->irqOwned) free_irq(pAdap->irq, pAdap);
#if ATLAS_AHB     
     
     if (pAdap->IDEChannel[0].pPciDev != NULL) { 
     	pci_free_consistent(pAdap->IDEChannel[0].pPciDev , 512, pAdap->IDEChannel[0].PIOVirt, pAdap->IDEChannel[0].PIOPhy);
     	kfree(pAdap->IDEChannel[0].pPciDev); 
     }
#endif     
     if (pAdap->IDEChannel != NULL) { kfree(pAdap->IDEChannel); }
     if (pAdap		   != NULL) { kfree(pAdap); }
    }

 /*
  * Unregister the reboot notifier.
  */
 unregister_reboot_notifier(&atlas_notifier);

 /*
  * Tell kernel scsi-layer we are gone.
  */
 scsi_unregister(pshost);
 

 return 0;

} /* end atlas_Release */

/************************************************************************
 * Name:	atlas_old_abort
 * Description:	Process the Abort command from the SCSI manager.
 * Parameters:	SCpnt - Pointer to SCSI command structure.
 * Returns:	Always snooze(A short sleep).
 ************************************************************************/
int
atlas_old_abort(Scsi_Cmnd *SCpnt)
{

#if (0)
 return	SCSI_ABORT_SNOOZE;
#else
 return SCSI_ABORT_ERROR;
#endif

} /* end atlas_old_abort */

/************************************************************************
 * Name:	atlas_old_reset
 * Description:	Process the Reset command from the SCSI manager.
 * Parameters:	SCpnt - Pointer to SCSI command structure.
 *		flags - Flags about the reset command
 * Returns:	No active command at this time, so this means
 *		that each time we got some kind of response the
 *		last time through.  Tell the mid-level code to
 *		request sense information in order to decide what
 *		to do next.
 ************************************************************************/
int
atlas_old_reset(Scsi_Cmnd *SCpnt, unsigned int reset_flags)
{
 return SCSI_RESET_SUCCESS;
} /* end atlas_old_reset */

/************************************************************************
 * This is the new scsi eh reset function.
 ************************************************************************/
int
atlas_reset_eh(Scsi_Cmnd *SCpnt)
{
 u8			i;
 u8			j;
 PChannel		pChan;
 PATLAS_ADAPTER		pAdap;

 printk("%s\n", __FUNCTION__);
    
 if (SCpnt == NULL)
    {
     printk("atlas_reset_eh: invalid Scsi_Cmnd\n");
     return FAILED;
    }

 for (i = 0; i < NumAdapters; i++)
    {
     pAdap = atlas_adapters[i];
     for (j = 0; j < pAdap->num_channels; j++)
	{
	 pChan = &pAdap->IDEChannel[j];
	 if(AtapiResetController(pAdap, pChan)==FALSE)
	  return FAILED;	 
	}
    }

 return SUCCESS;

} /* end atlas_reset_eh */

/************************************************************************
 * The new error handling code.
 ************************************************************************/
int
atlas_abort_eh(Scsi_Cmnd *SCpnt)
{

 if (SCpnt == NULL)
    {
     printk("atlas_reset_eh: invalid Scsi_Cmnd\n");
     return FAILED;
    }

 return SUCCESS;

} /* end atlas_abort_eh */

/************************************************************************
 * Name:	atlas_biosparam
 * Description:	Process the biosparam request from the SCSI manager to
 *		return C/H/S data.
 * Parameters:	disk - Pointer to SCSI disk structure.
 *		dev  - Major/minor number from kernel.
 *		geom - Pointer to integer array to place geometry data.
 * Returns:	zero.
 ************************************************************************/
// int		atlas_biosparam (struct scsi_device *, struct block_device *, sector_t , int *);

int
atlas_biosparam(struct scsi_device *sdev, struct block_device *bdev, sector_t capacity, int *geom) 
// atlas_biosparam(Scsi_Disk *disk, kdev_t dev, int geom[])
{

 int			heads;
 int			sectors;
 int			cylinders;

 /*
  * Default heads (64) & sectors (32)
  * Handle extended translation size for logical drives > 1Gb
  */
 if (capacity >= 0x200000)  //  if (disk->capacity >= 0x200000)
    {
     heads   = 255;
     sectors = 63;
    }
 else
    {
     heads   = 64;
     sectors = 32;
    }

 cylinders = (unsigned long)capacity / (heads * sectors);
 // cylinders = disk->capacity / (heads * sectors);

 /*
  * Return result
  */
 geom[0] = heads;
 geom[1] = sectors;
 geom[2] = cylinders;

 return 0;

} /* end atlas_biosparam */

/************************************************************************
 * Shutdown routine.
 ************************************************************************/
static int
atlas_halt(struct notifier_block *nb, ulong event, void *buf)
{

 if (event != SYS_RESTART && event != SYS_HALT && event != SYS_POWER_OFF)
    {
     return NOTIFY_DONE;
    }

 unregister_reboot_notifier(&atlas_notifier);

 return NOTIFY_OK;

} /* end atlas_halt */

/************************************************************************
 * PROC information.
 ************************************************************************/
int
atlas_proc_info
(
struct Scsi_Host *	host,
char *			buffer,
char **			start,
off_t			offset,
int			length,
//int			host_no,
int			inout
)
{
	return 0;
} /* end atlas_proc_info */

/************************************************************************
 * IOCTL, handle IDE ioctl
 ************************************************************************/

int atlas_set_piodma_mode(PChannel pChan, int devid, int type, int mode)
{
  PIDENTIFY_DATA2	ideIdentifyData;
  int			status=0;
  u8			devCtrl;
  unsigned long		flags;

  // Sorry, we can't delay SCSI data transmission!!    
  if(atlas_req_last!=NULL || pChan->ExpectingInterrupt)
   return -1;
      
  spin_lock_irqsave(&queue_request_lock, flags);
    
  devid &= 0x1;
  devCtrl = ATLAS_INB(pChan->io_ports[IDE_CONTROL_OFFSET]);
  
  ATLAS_OUTB(ATLAS_DC_NIEN, pChan->io_ports[IDE_CONTROL_OFFSET]);

  
  /* Check disk present */
  if(!(pChan->DeviceFlags[devid] & DFLAGS_DEVICE_PRESENT))
  { status = -1; goto bad; }
  else
   ideIdentifyData = &pChan->IdentifyData[devid];



  /* Change current transfer type/mode and identify information */
  switch(type)
  {
    case PIO_FLOW_CONTROL:
         if(mode>pChan->BestPioMode[devid]) { status = -1; goto bad; }
         else
         {
          pChan->PioMode[devid]	= mode;
          pChan->TraType[devid] = PIO_FLOW_CONTROL; // Switch to PIO transfer type
          ideIdentifyData->MultiWordDMAActive = 0;
          ideIdentifyData->UltraDMAActive = 0;
         }
         break;
    case MULTIWORD_DMA:
         if(mode>pChan->BestMdmaMode[devid] || ideIdentifyData->MultiWordDMASupport==0) { status = -1; goto bad; }
         else
         {
    	   pChan->DmaMode[devid] = mode;
    	   pChan->TraType[devid] = MULTIWORD_DMA;
    	   ideIdentifyData->MultiWordDMAActive = (1<<mode);
    	   ideIdentifyData->UltraDMAActive = 0;
    	 }
    	 break;
    case ULTRA_DMA:
         if(mode>pChan->BestUdmaMode[devid] || ideIdentifyData->UltraDMASupport==0) { status = -1; goto bad; }
         else
         {
    	   pChan->DmaMode[devid] = mode;
    	   pChan->TraType[devid] = ULTRA_DMA;	
    	   ideIdentifyData->UltraDMAActive = (1<<mode);
    	   ideIdentifyData->MultiWordDMAActive = 0;    	   
  	 }
  	 break;
  }
    
  /* Set disk feature */
  AtlasSetTransferMode(pChan, devid, pChan->TraType[devid], 
  		      (pChan->TraType[devid]==PIO_FLOW_CONTROL)?pChan->PioMode[devid]:pChan->DmaMode[devid]);

  /* Force Adapter update timing parameter */
  pChan->DoSwitch = TRUE;
  pChan->ActiveDevice = !devid;
  AtlasSwitchPioDmaMode(pChan, devid);  		      
  	  	  	  	
  /* Finally setup DoSwitch variable */
  if(!(pChan->DeviceFlags[!devid] & DFLAGS_DEVICE_PRESENT) ||  // Only one device      
      // Configuration is the same
      ((pChan->TraType[0] == pChan->TraType[1]) &&
       (pChan->PioMode[0] == pChan->PioMode[1]) &&
       (pChan->DmaMode[0] == pChan->DmaMode[1]))
    )
     pChan->DoSwitch = FALSE;
  else
     pChan->DoSwitch = TRUE;	
	
bad:
 
  // Maintain devCtrl status
  ATLAS_OUTB(devCtrl, pChan->io_ports[IDE_CONTROL_OFFSET]); 
  
  spin_unlock_irqrestore(&queue_request_lock, flags);
 			
  return 0;
}

// atlas_scsi2ide_ioctl(Scsi_Device *dev, int cmd, void *arg)

static int
atlas_scsi2ide_ioctl(struct scsi_device *dev, int cmd, void *arg)
{
   
 PATLAS_ADAPTER		pAdap;
 PChannel		pChan;
 int			devid;
 u32			argval;
   
    devid = (dev->id&1);
    pAdap = (dev->id < 4)?atlas_adapters[0]:atlas_adapters[1];
    pChan = &pAdap->IDEChannel[dev->channel];

 switch(cmd)
 {
   /* Get Command */
   case HDIO_GET_IDENTITY:
    copy_to_user((char *)arg, (char *)&pChan->IdentifyData[devid], 512);
    return 0;

   case HDIO_GET_MULTCOUNT:
    put_user(pChan->IdentifyData[devid].MaximumBlockTransfer, (u8 *)arg);
    return 0;

   case HDIO_GET_32BIT:
    put_user(0, (u32 *)arg);
    arg = (void *)0;
    return 0;
    
   case HDIO_GET_UNMASKINTR:
    put_user(0, (u32 *)arg);
    return 0;
    
   case HDIO_GET_KEEPSETTINGS:
    put_user(0, (u32 *)arg);
    return 0;        
    
   case HDIO_GET_DMA:
    if(pChan->TraType[devid]!=PIO_FLOW_CONTROL)
     put_user((pChan->TraType[devid]==MULTIWORD_DMA)?
             (pChan->DmaMode[devid]+6):(pChan->DmaMode[devid]),
             (long *)arg);
    else
    put_user(0, (long *)arg);
    return 0; 
    
  /*   
   case HDIO_GET_PIO_MODE:
     put_user(pChan->PioMode[devid], (long *)arg); 
    return 0; 
   */      
     
   /* Set Command */

   case HDIO_DRIVE_CMD:
    put_user(0, (char *)arg);
    return 0;  
    
   case HDIO_SET_DMA:
     argval = (u32)arg;
    if(argval==0)
     return atlas_set_piodma_mode(pChan, devid, PIO_FLOW_CONTROL, pChan->PioMode[devid]);
    if(argval >= 6)
     return atlas_set_piodma_mode(pChan, devid, MULTIWORD_DMA, argval-6);
    else
     return atlas_set_piodma_mode(pChan, devid, ULTRA_DMA, argval);
    
   case HDIO_SET_PIO_MODE:
     argval = (u32)arg;
     return atlas_set_piodma_mode(pChan, devid, PIO_FLOW_CONTROL, argval);           


   default:
      ;
    dprintk("%s: unknown code 0x%x\n", __FUNCTION__, cmd);
 }

  return -EINVAL;
}

static int atlas_slave_config(struct scsi_device *sdev)
{
	PATLAS_ADAPTER pAdap;
	PChannel pChan;
	int sid = sdev->id;
	
    if (sid >= 4)
	{
	 pAdap = atlas_adapters[1];
	 if(sid < 6)
	  pChan = &pAdap->IDEChannel[0];
	 else
	  pChan = &pAdap->IDEChannel[1];
	}
    else
	{
	 pAdap = atlas_adapters[0];
	 if (sid < 2)
	  pChan = &pAdap->IDEChannel[0];
	 else 
	  pChan = &pAdap->IDEChannel[1];
	}

	//if(!(pChan->IdentifyData[sid & 0x1].CmdSetSupported&(1<<26)))
	if( 1 )
	{
	 printk("Device supports LBA28 ONLY!!\n");	
	 blk_queue_max_sectors(sdev->request_queue, LBA28_MAX_SECTORS);
	}
	else
	{
	 printk("Device supports LBA48!!\n");
	}

	return 0;
}

/*
 * To make our driver a SCSI module, we have to include some magic at the
 * end of the file. This generates the init_module and clean_module code
 * nedded for a SCSI device, rather than having the author replicate it
 * each time a new driver is written.
 */

// static Scsi_Host_Template driver_template = ATLAS_HOST;

static struct scsi_host_template driver_template =
{
	 .module				= THIS_MODULE,				
	 .name					= "atlas",					 
	 .proc_name 			= "atlas",					
	 .proc_info 			= atlas_proc_info,			
	 .detect				= atlas_detect, 			
	 .release				= atlas_release,			
	 .ioctl 				= atlas_scsi2ide_ioctl, 	
	 .queuecommand			= atlas_queuecommand,	
	 .eh_abort_handler		= atlas_abort_eh,		
	 .eh_host_reset_handler = atlas_reset_eh,	
	 .bios_param			= atlas_biosparam,
	 .slave_configure		= atlas_slave_config,		
	 .can_queue 			= 1,					
	 .this_id				= -1,					
	 .sg_tablesize			= 64,					
	 .cmd_per_lun			= 1,
	 .max_sectors			= LBA48_MAX_SECTORS,
	 .use_clustering		= DISABLE_CLUSTERING,	

};

#include "scsi_module.c"



