/*
 * =====================================================================================
 *
 *       Filename:  page_access.c
 *
 *    Description:  Function to illustrate how to access page
 *
 *        Version:  1.0
 *        Created:  04/12/11 14:13:50
 *       Revision:  none
 *       Compiler:  gcc
 *
 *         Author:  YOUR NAME (), 
 *        Company:  
 *
 * =====================================================================================
 */

#include <asm/div64.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/err.h>
#include <linux/mtd/mtd.h>
#include <linux/sched.h>

#define PRINT_PREF KERN_INFO "mtd_rw: "

static int dev;
module_param(dev, int, S_IRUGO);
MODULE_PARM_DESC(dev, "MTD device number to use");

static int _rw = 11;
module_param(_rw, int, S_IRUGO);
MODULE_PARM_DESC(_rw, "MTD device erase/read/write direction");

static int _block = 0;
module_param(_block, int, S_IRUGO);
MODULE_PARM_DESC(_block, "MTD device rw block#");

static int _page = 0;
module_param(_page, int, S_IRUGO);
MODULE_PARM_DESC(_page, "MTD device rw page#");

/* _rw = 0 for read, 1 for write */
static int _value = 0x88;
module_param(_value, int, S_IRUGO);
MODULE_PARM_DESC(_value, "MTD device write value");

static struct mtd_info *mtd;
static unsigned char *iobuf;
//static unsigned char *bbt;

static int pgsize;		// page size
static int ebcnt;		// blocks in a given mtd
static int pgcnt;		// pages in a block
/* End of declaration */

static void dump_a_page(void)
{
	char line[128];
	int i, j, n;

	printk(" Entering [%s], pgsize = %d\n", __FUNCTION__, pgsize);
	n = pgsize;
	//n = mtd->writesize;
	for (i = 0; i < n;) {
		char *p = line;

		p += sprintf(p, "%05x: ", i);
		for (j = 0; j < 32 && i < n; j++, i++)
			p += sprintf(p, "%02x", (unsigned int)iobuf[i]);
		printk(KERN_INFO "%s\n", line);
	}
}

static int erase_a_block(int ebnum)
{
	int err;
	struct erase_info ei;
	loff_t addr = ebnum * mtd->erasesize;

	printk(KERN_INFO" [%s], addr = %#llx\n", __FUNCTION__, addr);
	memset(&ei, 0, sizeof(struct erase_info));
	ei.mtd  = mtd;
	ei.addr = addr;
	ei.len  = mtd->erasesize;

	err = mtd->erase(mtd, &ei);
	if (err) {
		printk(PRINT_PREF "error %d while erasing EB %d\n", err, ebnum);
		return err;
	}

	if (ei.state == MTD_ERASE_FAILED) {
		printk(PRINT_PREF "some erase error occurred at EB %d\n",
		       ebnum);
		return -EIO;
	}

	return 0;
}

static int write_a_page(int _block, int _page)
{
	size_t written = 0;
	int i, err = 0;
	loff_t addr = _block * mtd->erasesize + _page * mtd->writesize;
	void *writebuf = iobuf;

	printk(KERN_INFO " [%s], addr = %#llx\n", __FUNCTION__, (long long)addr);

	for (i = 0; i < pgsize; i++) {
		iobuf[i] = _value;
	}

	err = mtd->write(mtd, addr, pgsize, &written, writebuf);

	if (err || written != pgsize) {
		printk(PRINT_PREF "error: write failed at %#llx\n",
		       (long long)addr);
		return err ? err : -1;
	}
}

static int read_a_page(int _block, int _page)
{
	size_t read = 0;
	int ret, err = 0;
	loff_t addr = _block * mtd->erasesize + _page * mtd->writesize;
	void *buf = iobuf;

	printk(KERN_INFO " [%s], addr = %#llx\n", __FUNCTION__, addr);

	ret = mtd->read(mtd, addr, pgsize, &read, buf);

	if (ret || read != pgsize) {
		printk(PRINT_PREF "error: read failed at %#llx\n",
		       (long long)addr);
		if (!err)
			err = ret;
		if (!err)
			err = -EINVAL;
	}
	
	dump_a_page();
}

static int do_access(void)
{
	switch (_rw) {
		case 11:
		printk("\n###### Read page test ######\n");
		printk( "block = %d, page = %d\n", _block, _page);

		read_a_page(_block, _page);
		break;

		case 22:
		printk("\n###### Write page test ######\n");
		printk( "block = %d, page = %d, w_value=0x%x\n", _block, _page, _value);

		write_a_page(_block, _page);
		break;

		case 33:
		printk("\n###### Erase block test ######\n");
		printk( "block = %d\n", _block);

		erase_a_block (_block);
		break;
	}
	// return negative to terminate the test, unload the module
	return -12;
}

static int __init mtd_rw_init(void)
{
	int err = 0;
	uint64_t tmp;

	printk(KERN_INFO "Compiled time %s\n", __TIME__);
	printk(KERN_INFO "=================================================\n");
	printk(PRINT_PREF "MTD device: %d\n", dev);

	mtd = get_mtd_device(NULL, dev);
	if (IS_ERR(mtd)) {
		err = PTR_ERR(mtd);
		printk(PRINT_PREF "error: cannot get MTD device\n");
		return err;
	}

	if (mtd->type != MTD_NANDFLASH) {
		printk(PRINT_PREF "this test requires NAND flash\n");
		goto out;
	}

	tmp = mtd->size;
	do_div(tmp, mtd->erasesize);
	ebcnt = tmp;
	pgcnt = mtd->erasesize / mtd->writesize;
	pgsize = mtd->writesize;

	printk(PRINT_PREF "MTD device size %llu, eraseblock size %u, "
	       "page size %u, count of eraseblocks %u, pages per "
	       "eraseblock %u, OOB size %u\n",
	       (unsigned long long)mtd->size, mtd->erasesize,
	       pgsize, ebcnt, pgcnt, mtd->oobsize);

	err = -ENOMEM;

	iobuf = kmalloc(mtd->erasesize, GFP_KERNEL);
	if (!iobuf) {
		printk(PRINT_PREF "error: cannot allocate memory\n");
		goto out;
	}

	err = do_access();

out:
	kfree(iobuf);
	put_mtd_device(mtd);

	printk(KERN_INFO "=================================================\n");
	return err;
}

module_init(mtd_rw_init);

static void __exit mtd_rw_exit(void)
{
	return;
}
module_exit(mtd_rw_exit);

MODULE_DESCRIPTION("NAND page test");
MODULE_AUTHOR("Adrian Hunter");
MODULE_LICENSE("GPL");
