/*
  libsmtp is a library to send mail via SMTP
     This is the MIME handling part
   
Copyright ?2001 Kevin Read <obsidian@berlios.de>

This software is available under the GNU Lesser Public License as described
in the COPYING file.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free
Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

Kevin Read <obsidian@berlios.de>
Thu Aug 16 2001 */

/* This will only be included when MIME is enabled */

/* #ifndef __G_LIB_H__ */
  #include <glib.h>
/* #endif */

#include "../config.h"

#include "libsmtp.h"
#include "libsmtp_mime.h"


/* #undef LIBSMTP_DEBUG */

/* We need some definitions here first. These arrays hold the name tags
   of all MIME settings */

char *libsmtp_mime_types[] = {
  "text", "message", "image", "audio", "video", \
  "application", "multipart", "custom!!"};

char *libsmtp_mime_subtypes0[] = {
  "plain", "html", "english", "richtext"};

char *libsmtp_mime_subtypes1[] = {
  "rfc822", "partial"};

char *libsmtp_mime_subtypes2[] = {
  "gif", "jpeg", "png", "tiff", "x-ms-bmp", "x-xbitmap", \
  "x-xpixmap", "x-portable-anymap", "x-portable-bitmap", \
  "x-portable-graymap", "x-portable-pixmap"};

char *libsmtp_mime_subtypes3[] = {
  "mpeg", "midi", "x-wav", "x-aiff"};

char *libsmtp_mime_subtypes4[] = {
  "mpeg", "x-ms-video", "quicktime", "fli"};

char *libsmtp_mime_subtypes5[] = {
  "rtf", "postscript", "pdf", "zip", "x-debian-package", \
      "x-executable", "x-gtar", "x-shellscript", "x-tar", "octet-stream"};

char *libsmtp_mime_subtypes6[] = {
  "mixed", "parallel", "digest", "alternative"};

char *libsmtp_mime_encodings[] = {
  "7bit", "8bit", "binary", "base64", "quoted-printable"};

char *libsmtp_mime_charsets[] = {
  "us-ascii", "iso-8859-1", "iso-8859-2", "iso-8859-3"};


static void libsmtp_parentpart_free(struct libsmtp_session_struct *libsmtp_session)
{
	if(libsmtp_session->PartNowNode->parent && libsmtp_session->PartNowNode->parent->data) {
		free(libsmtp_session->Parts->data);
	}
}

static struct libsmtp_part_struct *libsmtp_part_main_ready(int libsmtp_int_type, int libsmtp_int_subtype, \
			int libsmtp_int_encoding, int libsmtp_int_charset, char *libsmtp_int_desc, \
			struct libsmtp_session_struct *libsmtp_session)
{
    struct libsmtp_part_struct *libsmtp_int_part;

	libsmtp_int_part = (struct libsmtp_part_struct *)calloc (1, sizeof(struct libsmtp_part_struct));
           
    if (libsmtp_int_part == NULL)
      return NULL;
                 
    /* The GStrings must be initialized */
	memset(libsmtp_int_part, 0, sizeof(struct libsmtp_part_struct));
    
	if (libsmtp_int_desc && strlen(libsmtp_int_desc))
		strcpy(libsmtp_int_part->Description, libsmtp_int_desc);
    libsmtp_int_part->Type = libsmtp_int_type;
    libsmtp_int_part->Subtype=libsmtp_int_subtype;
    libsmtp_int_part->Encoding=libsmtp_int_encoding;
    libsmtp_int_part->Charset=libsmtp_int_charset;
    libsmtp_int_part->Base64_finished=0;

    if (libsmtp_int_check_part (libsmtp_int_part)) {
        libsmtp_session->ErrorCode = LIBSMTP_BADARGS;
        free (libsmtp_int_part);
        return NULL;
    }

    /* We adjust the counters */
    libsmtp_session->NumParts++;
  
	/* This is the main part. */
	libsmtp_session->Parts=g_node_new (libsmtp_int_part);

	return libsmtp_int_part;
}

/* This function creates a new body part, checks for conformance to RFC822
   and RFC 2045 and maybe attaches it to the session. It is taken care in here
   that only multipart and message parts can contain children! Charset is
   ignored unless you set a text or message part */

struct libsmtp_part_struct *libsmtp_part_new(struct libsmtp_part_struct *libsmtp_int_parent_part, \
			int libsmtp_int_type,int libsmtp_int_subtype, int libsmtp_int_encoding, int libsmtp_int_charset, \
			char *libsmtp_int_desc, struct libsmtp_session_struct *libsmtp_session)
{
    struct libsmtp_part_struct *libsmtp_int_part;
    GNode *libsmtp_int_temp_node;
  
    /* First we have to check if the session already has a main type */
  
    if (libsmtp_session->NumParts) {
      /* Yes, there already is a main part. Now lets see if he has passed
         us a non-NULL parent pointer */
      
      if (libsmtp_int_parent_part) {
        /* Ok, it is non-null. Now the parent part this pointer points to has
           to be some kind of multipart */
        
      	if ((libsmtp_int_parent_part->Type!=LIBSMTP_MIME_MULTIPART) &&
      		(libsmtp_int_parent_part->Type!=LIBSMTP_MIME_MESSAGE)) {
      		/* No, it isn't multipart. We can't append new parts to it. */
      		libsmtp_session->ErrorCode=LIBSMTP_NOMULTIPART;
      		return NULL;
      	}
      } else {
      	/* We already have a parent part but he passed a NULL pointer;
      	 This won't do!! */
      	libsmtp_session->ErrorCode = LIBSMTP_PART_EXISTS;
      	return NULL;
      }
    } else {
      /* This session hasn't got a main type yet. Lets see if he wants to
         define it. */
        if (libsmtp_int_parent_part) {
      	  /* He doesn't want to define the main part!! */
      	  libsmtp_session->ErrorCode=LIBSMTP_NOPARENT;
      	  return NULL;
        }
    }
      
    /* Ok. If we got so far the parent argument should be ok. */
           
    /* We use calloc here to clear the memory. GLists are initialized when
       they point to NULL, so it must be cleared. */

    libsmtp_int_part = (struct libsmtp_part_struct *)calloc (1, sizeof(struct libsmtp_part_struct));
           
    if (libsmtp_int_part == NULL)
      return NULL;
                 
    /* The GStrings must be initialized */
	memset(libsmtp_int_part, 0, sizeof(struct libsmtp_part_struct));
    
    libsmtp_int_part->Type = libsmtp_int_type;
    libsmtp_int_part->Subtype=libsmtp_int_subtype;
    libsmtp_int_part->Encoding=libsmtp_int_encoding;
	if (libsmtp_int_desc && strlen(libsmtp_int_desc))
		strcpy(libsmtp_int_part->Description, libsmtp_int_desc);
    libsmtp_int_part->Charset=libsmtp_int_charset;
    libsmtp_int_part->Base64_finished=0;

    if (libsmtp_int_check_part (libsmtp_int_part)) {
        libsmtp_session->ErrorCode = LIBSMTP_BADARGS;
        return NULL;
    }

    /* We adjust the counters */
    libsmtp_session->NumParts++;
  
    if (libsmtp_int_parent_part) {
        /* This is a sibling part. We search the N-Tree for the data */
        libsmtp_int_temp_node = g_node_find (libsmtp_session->Parts, \
            G_IN_ORDER, G_TRAVERSE_ALL, libsmtp_int_parent_part);
        g_node_append_data (libsmtp_int_temp_node, libsmtp_int_part);
        
        #ifdef LIBSMTP_DEBUG
          printf ("libsmtp_part_new: Me: %s\n", libsmtp_int_part->Description);
        #endif
        libsmtp_int_part=libsmtp_int_temp_node->data;
        
        #ifdef LIBSMTP_DEBUG
          printf ("libsmtp_part_new: Parent: %s\n", libsmtp_int_part->Description);
        #endif
    } else {
        /* This is the main part. */
        libsmtp_session->Parts=g_node_new (libsmtp_int_part);
        
        #ifdef LIBSMTP_DEBUG
          printf ("libsmtp_part_new: Parent: None\nlibsmtp_part_new: Me: %s\n", libsmtp_int_part->Description);
        #endif
    }

    return libsmtp_int_part;
}
                           
        
int libsmtp_mime_type_custom (char *libsmtp_int_custom_type, \
       struct libsmtp_part_struct *libsmtp_int_part)
{
  /* Is this a custom type ? */
  if (libsmtp_int_part->Type != LIBSMTP_MIME_CUSTOM)
    return LIBSMTP_BADMIME;

  strcpy(libsmtp_int_part->CustomType, libsmtp_int_custom_type);
  return LIBSMTP_NOERR;
}

int libsmtp_mime_subtype_custom (char *libsmtp_int_custom_subtype, \
       struct libsmtp_part_struct *libsmtp_int_part)
{
  /* Is this a custom subtype ? */
  if (libsmtp_int_part->Subtype != LIBSMTP_MIME_SUB_CUSTOM)
    return LIBSMTP_BADMIME;
    
  strcpy(libsmtp_int_part->CustomSubtype, libsmtp_int_custom_subtype);

  return LIBSMTP_NOERR;
}


/* We use this function internally to set the session to the next part to
   send. This function relies on the caller to check that all arguments
   are ok */
int libsmtp_int_nextpart (struct libsmtp_session_struct *libsmtp_session)
{
    GNode *libsmtp_temp_now;
    struct libsmtp_part_struct *libsmtp_temp_part;
    GString *libsmtp_temp_gstring=0;
    char *libsmtp_temp_string;
    int libsmtp_temp_int, libsmtp_loop_running=0, libsmtp_int_travel=0;
  
    libsmtp_temp_gstring = g_string_new (NULL);
  
    /* Are we in a part already? */
    if (libsmtp_session->PartNowNode == 0) {
        /* No, we start with the main part */
        libsmtp_session->PartNowNode = libsmtp_session->Parts;
        libsmtp_session->PartNow = libsmtp_session->PartNowNode->data;
        
        /* If this is a Multipart part, we must send the standard MIME blurb */
        if (libsmtp_session->PartNow->Type == LIBSMTP_MIME_MULTIPART) {
            g_string_assign (libsmtp_temp_gstring, \
               "This is a MIME multipart message. Your mail reader isn't MIME capable.\n\r"
				"You might not be able to read parts or all of this message.\r\n");
            
            if (libsmtp_int_send (libsmtp_temp_gstring, libsmtp_session)) {
				g_string_free(libsmtp_temp_gstring, 1);
				return LIBSMTP_ERRORSENDFATAL;
			}
        }
    } else {
        libsmtp_int_travel=1;
    }

	libsmtp_loop_running=1;
  
	/* This is the scanning loop. It scans for the next non-Multipart part */
	while (libsmtp_loop_running) {
		libsmtp_temp_part = (struct libsmtp_part_struct *)libsmtp_session->PartNow;

		/* Should we travel over this one anyway? */
    
		if (libsmtp_int_travel) {
		   /* Yes, so we'll have to take the next sibling. O Brother were art
		      thou? */
		   
		   libsmtp_int_travel=0;
			if (libsmtp_session->PartNowNode->next) {
				/* Yeah, so we take it. */
				libsmtp_session->PartNowNode=libsmtp_session->PartNowNode->next;
				libsmtp_session->PartNow=libsmtp_session->PartNowNode->data;
				  
				  /* Jump the gun, Mr. Ed */
				  continue;
			} else {
				/* No, so we need the parent */
				if (libsmtp_session->PartNowNode->parent) {
				  /* Ok, here it is */
				  libsmtp_session->PartNowNode=libsmtp_session->PartNowNode->parent;
				  libsmtp_session->PartNow=libsmtp_session->PartNowNode->data;
		
				  /* We will travel over this one, of course. It is already finished */
				  libsmtp_int_travel=1;
				  continue;
				} else {
				  /* No more parents here. We are finished. */
				  g_string_free(libsmtp_temp_gstring, 1);
				  return LIBSMTP_PARTSFINISHED;
				}
			}
		} else {
			/* Ok, we don't need to travel. Is this a multipart part? */
			if (libsmtp_temp_part->Type==LIBSMTP_MIME_MULTIPART) {

				/* Yes, is the boundary string set? */
				if (strlen(libsmtp_temp_part->Boundary)) {
					/* No, we really should set that */
					sprintf(libsmtp_temp_part->Boundary, "----_=_libsmtp_Nextpart__000_000007DA.3B95D19_%d", \
									  g_node_depth (libsmtp_session->PartNowNode));
					//g_string_sprintf (libsmtp_temp_part->Boundary, \
									  "----_=_libsmtp_Nextpart__000_000007DA.3B95D19_%d", \
									  g_node_depth (libsmtp_session->PartNowNode));
				}
        
				/* Set PartNowNode to the first child */
				libsmtp_session->PartNowNode=libsmtp_session->PartNowNode->children;
				libsmtp_session->PartNow=libsmtp_session->PartNowNode->data;
      
				#ifdef LIBSMTP_DEBUG
				  printf ("libsmtp_int_nextpart: Jumped to %s\n", libsmtp_session->PartNow->Description);
				#endif
      
				/* and continue to the next loop */
				continue;
			} else {
				/* No, this is our man! */
				libsmtp_session->PartNow = libsmtp_session->PartNowNode->data;
        
				/* Maybe we have to send out a boundary and next part data... */
				if (g_node_depth (libsmtp_session->PartNowNode) > 1) {
					libsmtp_temp_now=libsmtp_session->PartNowNode->parent;
					libsmtp_temp_part=libsmtp_temp_now->data;
          
					g_string_sprintf (libsmtp_temp_gstring, "\r\n\r\n\r\n--%s\r\n", \
									  libsmtp_temp_part->Boundary);
             
			#ifdef LIBSMTP_DEBUG
					printf ("libsmtp_mime_headers: %s", libsmtp_temp_gstring->str);
			#endif

					if (libsmtp_int_send (libsmtp_temp_gstring, libsmtp_session)) {
						g_string_free(libsmtp_temp_gstring, 1);
						return LIBSMTP_ERRORSENDFATAL;
					}
					libsmtp_temp_part=libsmtp_session->PartNow;

					/* We should check for valied MIME settings first */
					if ((libsmtp_temp_int=libsmtp_int_check_part (libsmtp_temp_part))) {
						libsmtp_session->ErrorCode=libsmtp_temp_int;
						g_string_free(libsmtp_temp_gstring, 1);
						return libsmtp_temp_int;
					}

			/* Then we look up the names of the MIME settings of the main body part 
					and send them as headers */


					if (strlen(libsmtp_temp_part->Description) == 0)
						g_string_sprintf (libsmtp_temp_gstring, "Content-Type: %s/%s", \
									  libsmtp_int_lookup_mime_type (libsmtp_temp_part), \
									  libsmtp_int_lookup_mime_subtype (libsmtp_temp_part));
					else 
						g_string_sprintf (libsmtp_temp_gstring, "Content-Type: %s/%s;name=\"%s\"", \
									  libsmtp_int_lookup_mime_type (libsmtp_temp_part), \
									  libsmtp_int_lookup_mime_subtype (libsmtp_temp_part), \
										  libsmtp_temp_part->Description);

			#ifdef LIBSMTP_DEBUG
					printf ("libsmtp_mime_headers: %s. Type: %d/%d\n", libsmtp_temp_gstring->str, \
							libsmtp_temp_part->Type, libsmtp_temp_part->Subtype);
			#endif
    
					if (libsmtp_int_send (libsmtp_temp_gstring, libsmtp_session)) {
						g_string_free(libsmtp_temp_gstring, 1);
						return LIBSMTP_ERRORSENDFATAL;
					}

					/* Text and message parts will have a charset setting */
					if ((libsmtp_temp_part->Type==LIBSMTP_MIME_TEXT) ||
						(libsmtp_temp_part->Type==LIBSMTP_MIME_MESSAGE))
						if ((libsmtp_temp_string=(gchar *)libsmtp_int_lookup_mime_charset(libsmtp_temp_part))) {
							g_string_sprintf (libsmtp_temp_gstring, ";\r\n charset=\"%s\"", \
											  libsmtp_temp_string);
       
							if (libsmtp_int_send (libsmtp_temp_gstring, libsmtp_session)) {
								g_string_free(libsmtp_temp_gstring, 1);
								return LIBSMTP_ERRORSENDFATAL;
							}

						#ifdef LIBSMTP_DEBUG
							printf ("libsmtp_mime_headers: %s", libsmtp_temp_gstring->str);
						#endif
						}
    
						/* We need a transfer encoding, too */
    
						g_string_sprintf (libsmtp_temp_gstring, "\r\nContent-Transfer-Encoding: %s\r\n\r\n", \
										  libsmtp_int_lookup_mime_encoding (libsmtp_temp_part));
    
					#ifdef LIBSMTP_DEBUG
						printf ("libsmtp_mime_headers: %s\n", libsmtp_temp_gstring);
					#endif
    
						if (libsmtp_int_send (libsmtp_temp_gstring, libsmtp_session)) {
							g_string_free(libsmtp_temp_gstring, 1);
							return LIBSMTP_ERRORSENDFATAL;
						}
				}
				g_string_free(libsmtp_temp_gstring, 1);
				return LIBSMTP_NOERR;
			}
		}
	}
	g_string_free(libsmtp_temp_gstring, 1);
	return LIBSMTP_NOERR;
}


/* This function returns the current part. It should be used directly after
   sending the headers to find out which part will be transmitted first. It
   can be used at any time to find out what part is currently being sent, of
   course. */
   
struct libsmtp_part_struct *libsmtp_part_query(struct libsmtp_session_struct *libsmtp_session)
{
  /* Are we in data stage? */
  if ((libsmtp_session->Stage < LIBSMTP_HEADERS_STAGE) ||
      (libsmtp_session->Stage > LIBSMTP_BODY_STAGE)) {
    libsmtp_session->ErrorCode = LIBSMTP_BADSTAGE;
    return NULL;
  }

  /* Check to see if we already are in a part */
  if (!libsmtp_session->PartNow) {
    /* We are not at the moment working on one part. Lets see if any parts
       are defined at all! */
    if (!libsmtp_session->Parts) {
      /* nope. bad mistake! */
      libsmtp_session->ErrorCode=LIBSMTP_NOPARENT;
      return NULL;
    }
    
    /* So we try to lookup the first part that might contain data */
    if (libsmtp_int_nextpart (libsmtp_session))
      return NULL;
  }
  
  /* Ok, return the current part */
  return libsmtp_session->PartNow;
}


/* libsmtp_int_check_part checks a part for correct settings */

int libsmtp_int_check_part (struct libsmtp_part_struct *libsmtp_int_part)
{
  
  /* Now we check if any invalid MIME arguments have been given.*/
  
  if ((libsmtp_int_part->Type < 0) || (libsmtp_int_part->Type > LIBSMTP_MAX_MIME))
    return LIBSMTP_BADARGS;
  
  /* Now the same for the subtype argument. This must correspond to the
     selected type */

  switch (libsmtp_int_part->Type) {
	  case (LIBSMTP_MIME_TEXT):
		  if ((libsmtp_int_part->Subtype < 0) || (libsmtp_int_part->Subtype > LIBSMTP_MAX_MIME_SUB0))
			  return LIBSMTP_BADMIME;
		  
		  /* Text types can have any encoding */
		  if ((libsmtp_int_part->Encoding < 0) || (libsmtp_int_part->Encoding > LIBSMTP_MAX_ENC)) 
			  return LIBSMTP_BADENCODING;
		  
		  /* Text types must have a valid charset */
		  if ((libsmtp_int_part->Charset < 0) || (libsmtp_int_part->Charset > LIBSMTP_MAX_CHARSET))
			  return LIBSMTP_BADCHARSET;
		  break;
	
	  case (LIBSMTP_MIME_MESSAGE):
		  if ((libsmtp_int_part->Subtype < 1000) || (libsmtp_int_part->Subtype > LIBSMTP_MAX_MIME_SUB1)) 
			  return LIBSMTP_BADMIME;
		  
		  /* Message types can have any encoding */
		  if ((libsmtp_int_part->Encoding < 0) || (libsmtp_int_part->Encoding > LIBSMTP_MAX_ENC))
			  return LIBSMTP_BADENCODING;
	
		  /* Message types must have a valid charset */
		  if ((libsmtp_int_part->Charset < 0) || (libsmtp_int_part->Charset > LIBSMTP_MAX_CHARSET))
			  return LIBSMTP_BADCHARSET;
		break;

    case (LIBSMTP_MIME_IMAGE):
      if ((libsmtp_int_part->Subtype < 2000) || (libsmtp_int_part->Subtype > LIBSMTP_MAX_MIME_SUB2)) 
        return LIBSMTP_BADMIME;

      /* Image types must be in a non-text encoding */
      if ((libsmtp_int_part->Encoding < LIBSMTP_ENC_BINARY) || (libsmtp_int_part->Encoding > LIBSMTP_MAX_ENC)) 
        return LIBSMTP_BADENCODING;
      
      /* Charset is set to -1 because it won't matter here */
      libsmtp_int_part->Charset = -1;
	  break;                         

    case (LIBSMTP_MIME_AUDIO):
      if ((libsmtp_int_part->Subtype < 3000) || (libsmtp_int_part->Subtype > LIBSMTP_MAX_MIME_SUB3))
		  return LIBSMTP_BADMIME;

      /* Audio types must be in a non-text encoding */
      if ((libsmtp_int_part->Encoding < LIBSMTP_ENC_BINARY) || (libsmtp_int_part->Encoding > LIBSMTP_MAX_ENC))
		  return LIBSMTP_BADENCODING;

      /* Charset is set to -1 because it won't matter here */
      libsmtp_int_part->Charset = -1;
	  break;

    case (LIBSMTP_MIME_VIDEO):
      if ((libsmtp_int_part->Subtype < 4000) || (libsmtp_int_part->Subtype > LIBSMTP_MAX_MIME_SUB4))
		  return LIBSMTP_BADMIME;

      /* Video types must be in a non-text encoding */
      if ((libsmtp_int_part->Encoding < LIBSMTP_ENC_BINARY) || (libsmtp_int_part->Encoding > LIBSMTP_MAX_ENC))
		  return LIBSMTP_BADENCODING;

      /* Charset is set to -1 because it won't matter here */
      libsmtp_int_part->Charset = -1;
	  break;

    case (LIBSMTP_MIME_APPLICATION):
      if ((libsmtp_int_part->Subtype < 5000) || (libsmtp_int_part->Subtype > LIBSMTP_MAX_MIME_SUB5))
		  return LIBSMTP_BADMIME;

      /* Application types must be in a non-text encoding */
      if ((libsmtp_int_part->Encoding < LIBSMTP_ENC_BINARY) || (libsmtp_int_part->Encoding > LIBSMTP_MAX_ENC))
		  return LIBSMTP_BADENCODING;

      /* Charset is set to -1 because it won't matter here */
      libsmtp_int_part->Charset = -1;
	  break;

    case (LIBSMTP_MIME_MULTIPART):
      if ((libsmtp_int_part->Subtype < 6000) || (libsmtp_int_part->Subtype > LIBSMTP_MAX_MIME_SUB6))
		  return LIBSMTP_BADMIME;

      /* Application types must be in a text encoding, and should only be
         7bit */
      if (libsmtp_int_part->Encoding != LIBSMTP_ENC_7BIT)
		  return LIBSMTP_BADENCODING;

      /* Charset is set to -1 because it won't matter here */
      libsmtp_int_part->Charset = -1;
	  break;

    case (LIBSMTP_MIME_CUSTOM):
      if (libsmtp_int_part->Subtype != LIBSMTP_MIME_SUB_CUSTOM)
		  return LIBSMTP_BADMIME;

      /* Custom type can have any encoding, of course */
      if ((libsmtp_int_part->Encoding < 0) || (libsmtp_int_part->Encoding > LIBSMTP_MAX_ENC))
		  return LIBSMTP_BADENCODING;

      /* Custom types must have a valid charset or NOCHARSET */
      if ((libsmtp_int_part->Charset < LIBSMTP_CHARSET_NOCHARSET) || (libsmtp_int_part->Charset > LIBSMTP_MAX_CHARSET))
		  return LIBSMTP_BADCHARSET;
	  break;
  }
  
  return 0;
}

/* These functions lookup the name of types, subtypes and encodings for a
   part. They only perform glancing checking of parameters, so you should
   check the mime settings beforehand with libsmtp_int_check_parts */

const char *libsmtp_int_lookup_mime_type (struct libsmtp_part_struct *libsmtp_int_part)
{
  if ((libsmtp_int_part->Type >= 0) && (libsmtp_int_part->Type < LIBSMTP_MAX_MIME))
  {
    if (libsmtp_int_part->Type == LIBSMTP_MIME_CUSTOM)
      return libsmtp_int_part->CustomType;
    else
      return libsmtp_mime_types[libsmtp_int_part->Type];
  }
  else
    return NULL;
}

const char *libsmtp_int_lookup_mime_subtype (struct libsmtp_part_struct *libsmtp_int_part)
{
    switch (libsmtp_int_part->Type)
    {
      case LIBSMTP_MIME_TEXT:
        return libsmtp_mime_subtypes0[libsmtp_int_part->Subtype];

      case LIBSMTP_MIME_MESSAGE:
        return libsmtp_mime_subtypes1[libsmtp_int_part->Subtype-1000];

      case LIBSMTP_MIME_IMAGE:
        return libsmtp_mime_subtypes2[libsmtp_int_part->Subtype-2000];

      case LIBSMTP_MIME_AUDIO:
        return libsmtp_mime_subtypes3[libsmtp_int_part->Subtype-3000];

      case LIBSMTP_MIME_VIDEO:
        return libsmtp_mime_subtypes4[libsmtp_int_part->Subtype-4000];

      case LIBSMTP_MIME_APPLICATION:
        return libsmtp_mime_subtypes5[libsmtp_int_part->Subtype-5000];

      case LIBSMTP_MIME_MULTIPART:
        return libsmtp_mime_subtypes6[libsmtp_int_part->Subtype-6000];

      case LIBSMTP_MIME_CUSTOM:
        return libsmtp_int_part->CustomSubtype;
      
      default:
        return NULL;
    }
}

const char *libsmtp_int_lookup_mime_charset (struct libsmtp_part_struct *libsmtp_int_part)
{
  /* Only textual parts can have a charset */
  if ((libsmtp_int_part->Type == LIBSMTP_MIME_TEXT) || \
      (libsmtp_int_part->Type == LIBSMTP_MIME_MESSAGE))
    {
    if ((libsmtp_int_part->Charset >= 0) && (libsmtp_int_part->Charset < LIBSMTP_MAX_CHARSET))
    {
      return libsmtp_mime_charsets[libsmtp_int_part->Charset];
    }
    else
      return NULL;
  }
  return NULL;
}

const char *libsmtp_int_lookup_mime_encoding (struct libsmtp_part_struct *libsmtp_int_part)
{
  if ((libsmtp_int_part->Encoding >= 0) && (libsmtp_int_part->Encoding <= LIBSMTP_MAX_ENC))
    return libsmtp_mime_encodings[libsmtp_int_part->Encoding];
  else
    return NULL;
}

int libsmtp_send_notify_message(char *message, struct libsmtp_session_struct *mailsession)
{
	struct libsmtp_part_struct *textpart;
	int ret = -1;

	textpart = libsmtp_part_new (NULL, LIBSMTP_MIME_TEXT, LIBSMTP_MIME_SUB_PLAIN, LIBSMTP_ENC_7BIT, LIBSMTP_CHARSET_USASCII,\
								 NULL, mailsession);
	if (textpart == NULL) {
	  printf ("Error adding part: %s\n", libsmtp_strerr (mailsession));
	  return ret;
	}

	if (libsmtp_mime_headers (mailsession)) {
	  return mailsession->ErrorCode;
	}

	if (libsmtp_part_send (message, strlen(message), mailsession)) {
	  return mailsession->ErrorCode;
	}
	return LIBSMTP_NOERR;
}

int libsmtp_multipart_header(struct libsmtp_session_struct *mailsession)
{
	/* Lets add some parts to the body. We gotta save the pointer to the
	   main part */
	mailsession->mainpart = libsmtp_part_main_ready(LIBSMTP_MIME_MULTIPART, LIBSMTP_MIME_SUB_MIXED, \
								 LIBSMTP_ENC_7BIT, LIBSMTP_CHARSET_NOCHARSET, NULL, mailsession);
	if (mailsession->mainpart == NULL) {
	  printf ("Error adding part: %s\n", libsmtp_strerr (mailsession));
	  return 1;
	}

	/* Now lets send the MIME headers */
	if (libsmtp_mime_headers (mailsession)) {
	  stat_message (mailsession);
	  return mailsession->ErrorCode;
	}
//	printf ("MIME headers sent.\n");

	return LIBSMTP_NOERR;

}

void libsmtp_multipart_end(struct libsmtp_session_struct *mailsession)
{
	libsmtp_parentpart_free(mailsession);
}

int libsmtp_send_multipart(int libsmtp_int_type,int libsmtp_int_subtype, 
						   int libsmtp_int_encoding, int libsmtp_int_charset, char *udname, 
						   char *snddata, int datasize, struct libsmtp_session_struct *mailsession)
{
	if (!libsmtp_part_new (mailsession->mainpart, libsmtp_int_type, libsmtp_int_subtype, \
						   libsmtp_int_encoding, libsmtp_int_charset, udname, mailsession)) {
	  printf ("Error adding part: %s\n", libsmtp_strerr (mailsession));
	  return 1;
	}

	/* This ends the body part */
	if (libsmtp_part_send_section(snddata, datasize, mailsession)) {
	  stat_message (mailsession);
		if(mailsession->PartNow)
			free(mailsession->PartNow);
	  return mailsession->ErrorCode;
	}
	if(mailsession->PartNow)
		free(mailsession->PartNow);
	return LIBSMTP_NOERR;
}

int libsmtp_send_bdata(int mime_type, int mime_subtype, char *udname, char *rawdata, 
					   int size, struct libsmtp_session_struct *mailsession)
{
	struct libsmtp_part_struct *mainpart;

	/* Lets add some parts to the body. We gotta save the pointer to the
	   main part */
	mainpart = libsmtp_part_main_ready(LIBSMTP_MIME_MULTIPART, LIBSMTP_MIME_SUB_MIXED, \
								 LIBSMTP_ENC_7BIT, LIBSMTP_CHARSET_NOCHARSET, NULL, mailsession);
	if (mainpart == NULL) {
	  printf ("Error adding part: %s\n", libsmtp_strerr (mailsession));
	  return 1;
	}
	if (!libsmtp_part_new (mainpart, mime_type, mime_subtype, LIBSMTP_ENC_BASE64, \
						   LIBSMTP_CHARSET_NOCHARSET, udname, mailsession)) {
	  printf ("Error adding part: %s\n", libsmtp_strerr (mailsession));
	  return 1;
	}

	/* Now lets send the MIME headers */
	if (libsmtp_mime_headers (mailsession)) {
	  stat_message (mailsession);
	  return mailsession->ErrorCode;
	}
	printf ("MIME headers sent.\n");

	printf ("Now in second body part.\n");

	if (libsmtp_part_send_section(rawdata, size, mailsession)) {
		if(mailsession->PartNow)
			free(mailsession->PartNow);
		libsmtp_parentpart_free(mailsession);
		stat_message (mailsession);
		return mailsession->ErrorCode;
	}
	if(mailsession->PartNow)
		free(mailsession->PartNow);
	libsmtp_parentpart_free(mailsession);

	return LIBSMTP_NOERR;

}
