/*******************************************************************
 * Routines for displaying/logging debug information.
 *
 * Licensed under the dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: xsup_debug.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: xsup_debug.c,v 1.33 2006/09/14 16:20:33 chessing Exp $
 * $Date: 2006/09/14 16:20:33 $
 * $Log: xsup_debug.c,v $
 * Revision 1.33  2006/09/14 16:20:33  chessing
 * Bug fix for needing to add fflush(stdout) to debug printing code. As reported in the bugs section.
 *
 * Revision 1.32  2006/06/02 21:49:18  chessing
 * More memory leak cleanups, and another patch from Carsten.
 *
 * Revision 1.31  2006/06/01 22:49:49  galimorerpg
 * Converted all instances of u_char to uint8_t
 * Fixed a bad #include in the generic frame handler.
 *
 * Revision 1.30  2006/05/30 03:53:21  chessing
 * Some fixes to IPC support.
 *
 * Revision 1.29  2006/05/29 02:25:49  chessing
 * Patches from Carsten Grohmann.
 *
 * Revision 1.28  2006/05/26 22:04:58  chessing
 * Fixed some memory access errors, and cleaned up some wext stuff that was causing issues with the madwifi driver in wext mode.
 *
 * Revision 1.27  2006/05/14 22:09:28  chessing
 * A few small changes in the Xsupplicant code.  Beginning of a configuration/monitor program can be found in gui_tools. (Very simple for now, with several known bugs, but no showstopper bugs know.)
 *
 * Revision 1.26  2006/04/25 01:17:42  chessing
 * LOTS of code cleanups, new error checking/debugging code added, and other misc. fixes/changes.
 *
 * Revision 1.25  2005/12/18 07:44:12  chessing
 * Added the ability to associate using IWAUTH options instead of setting a full IE.  This allows NDISwrapper to work with vanilla wireless extensions.  For some reason, we can't parse IWEVASSOCREQIE/IWEVASSOCRESPIE since we see the length as 0, even though iwevent sees the correct IE information. :-/  Need to figure out why.
 *
 * Revision 1.24  2005/11/22 19:49:42  chessing
 * When the desired interface isn't available when Xsupplicant starts, Xsupplicant will wait for it to show up before starting to authenticate.  In this case, the interface isn't 'available' when an ifconfig doesn't show that the interface exists.  It has *NOTHING* to do with the up/down state of the interface.  This is mostly useful for situations where a cardbus wireless card is plugged in before linux is booted.
 *
 * Revision 1.23  2005/10/17 03:56:54  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.22  2005/09/23 02:31:33  shaftoe
 * Log output cleanup for the benefit of syslog.
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.21  2005/09/19 01:58:33  shaftoe
 * - Send log messages to console if syslog fails.
 * - Append PID to syslog messages.
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.20  2005/08/25 03:34:05  chessing
 * Removed a bunch of functions from config.c that could be handled better in other ways.
 *
 * Revision 1.19  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <stdio.h>
#include <stdarg.h>
#include <strings.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <netinet/in.h>
#include <syslog.h>
#include "xsup_debug.h"
#include "xsupconfig.h"
#include "xsup_err.h"
#include "profile.h"
#include "xsup_ipc.h"

#ifdef USE_EFENCE
#include <efence.h>
#endif

unsigned char debug_level = 0;     // By default just show the "normal" stuff.
int isdaemon = 0;
int syslogging = 0;
FILE *logfile = NULL;

/*************************************************************************
 *
 * Check the value of 'tf'.  If it returns false, print out some debug
 * information and either return FALSE, or terminate.  (Depending on the
 * value of terminal.)  In general, this function should be called via the
 * xsup_assert() macro, so that the filename, line number, and function
 * name are automatically filled in.
 *
 *************************************************************************/
int xsup_assert_long(int tf, char *desc, int terminal, char *file, int line,
		     const char *function)
{
  if (!tf)
    {
      debug_printf(DEBUG_NORMAL, "Assertion '%s' failed in file %s, "
		   "function %s(), at line %d.\n", desc, file, function, line);

      if (terminal)
	{
	  debug_printf(DEBUG_NORMAL, "Cannot continue!\n");
	  exit(255);
	}
      return FALSE;
    }
  return TRUE;
}


/***************************************************
 *
 * Convert a string to be all lowercase.
 *
 ***************************************************/
void lowercase(char *instr)
{
  int i;

  for (i=0;i<strlen(instr);i++)
    {
      instr[i] = tolower(instr[i]);
    }
}

/***************************************************
 *
 * Return the logging facility the user asked us to use.
 *
 ***************************************************/
int xsup_debug_get_facility(char *facility_str)
{
  int facility_num = LOG_DAEMON;

  if (strcmp("cron", facility_str) == 0) facility_num = LOG_CRON;
  if (strcmp("daemon", facility_str) == 0) facility_num = LOG_DAEMON;
  if (strcmp("ftp", facility_str) == 0) facility_num = LOG_FTP;
  if (strcmp("kern", facility_str) == 0) facility_num = LOG_KERN;
  if (strcmp("local0", facility_str) == 0) facility_num = LOG_LOCAL0;
  if (strcmp("local1", facility_str) == 0) facility_num = LOG_LOCAL1;
  if (strcmp("local2", facility_str) == 0) facility_num = LOG_LOCAL2;
  if (strcmp("local3", facility_str) == 0) facility_num = LOG_LOCAL3;
  if (strcmp("local4", facility_str) == 0) facility_num = LOG_LOCAL4;
  if (strcmp("local5", facility_str) == 0) facility_num = LOG_LOCAL5;
  if (strcmp("local6", facility_str) == 0) facility_num = LOG_LOCAL6;
  if (strcmp("local7", facility_str) == 0) facility_num = LOG_LOCAL7;
  if (strcmp("lpr", facility_str) == 0) facility_num = LOG_LPR;
  if (strcmp("mail", facility_str) == 0) facility_num = LOG_MAIL;
  if (strcmp("news", facility_str) == 0) facility_num = LOG_NEWS;
  if (strcmp("user", facility_str) == 0) facility_num = LOG_USER;
  if (strcmp("uucp", facility_str) == 0) facility_num = LOG_UUCP;

  return facility_num;
}

/***************************************************
 *
 * Remove an old logfile, and create a new one.
 *
 ***************************************************/
int logfile_setup(char *logfilename)
{
  char *tempstr = NULL;
  int facility;
  int result;
  struct config_globals *globals;

  if (logfilename != NULL)
    {
      if (isdaemon != 2)
	{
	  tempstr = (char *)malloc(strlen(logfilename)+1);
	  if (tempstr == NULL)
	    {
	      printf("Couldn't allocate memory for temporary string! (%s:%d)\n",
		     __FUNCTION__, __LINE__);
	      return XEMALLOC;
	    }

	  strcpy(tempstr, logfilename);
	  lowercase(tempstr);
	  result = strcmp("syslog", tempstr);
	  free(tempstr);
 	  tempstr = NULL;
	  if (result == 0)
	    {
	      globals = config_get_globals();

	      if (globals == NULL)
		{
		  printf("No valid configuration globals available at %s!\n",
			 __FUNCTION__);
		  return XEMALLOC;
		}

	      tempstr = globals->log_facility;
	      lowercase(tempstr);

	      facility = xsup_debug_get_facility(tempstr);

	      openlog("Xsupplicant", LOG_CONS | LOG_PID | LOG_NDELAY, 
			      facility);
	      syslogging = 1;
	    } else {
	      unlink(logfilename);

	      logfile = fopen(logfilename, "w+");
	      if (!logfile)
		{
		  return XEMALLOC;
		}
	    }
	}
    }
  return XENONE;
}

/**************************************************
 *
 * Clean up our old logfile.
 *
 **************************************************/
void logfile_cleanup()
{
  if (logfile != NULL)
    {
      fclose(logfile);
    }
  if (syslogging == 1)
    {
      closelog();
    }
}

/*************************************************
 *
 * Set flags based on the numeric value that was passed in.
 *
 *************************************************/
void debug_set_flags(int new_flags)
{
  if (new_flags >= 1) debug_level |= DEBUG_CONFIG;
  if (new_flags >= 2) debug_level |= DEBUG_STATE;
  if (new_flags >= 3) debug_level |= DEBUG_AUTHTYPES;
  if (new_flags >= 4) debug_level |= DEBUG_INT;
  if (new_flags >= 5) debug_level |= DEBUG_SNMP;
  if (new_flags >= 6) debug_level |= DEBUG_EVERYTHING;
  if (new_flags >= 7) debug_level |= DEBUG_EXCESSIVE;
}

/*************************************************
 *
 * Set flags based on an ASCII string that was passed in.
 *
 *************************************************/
void debug_alpha_set_flags(char *new_flags)
{
  int i;

  debug_level = 0;

  for (i=0;i<strlen(new_flags);i++)
    {
      switch (new_flags[i])
	{
	case 'c':
	  debug_level |= DEBUG_CONFIG;
	  break;

	case 's':
	  debug_level |= DEBUG_STATE;
	  break;

	case 'a':
	  debug_level |= DEBUG_AUTHTYPES;
	  break;

	case 'i':
	  debug_level |= DEBUG_INT;
	  break;

	case 'n':
	  debug_level |= DEBUG_SNMP;
	  break;

	case 'e':
	  debug_level |= DEBUG_EVERYTHING;
	  break;

	case 'x':
	  debug_level |= DEBUG_EXCESSIVE;
	  break;

	case 'A':
	  debug_level |= 0xff;   // Set all flags.
	  break;
	}
    }
}

/*************************************************
 *
 * Depending on the value of fh, we will either print to the screen, or
 * a log file.
 *
 *************************************************/
void ufprintf(FILE *fh, char *instr, int level)
{

  // If xsup_debug.c is used outside the main Xsupplicant source, it will
  // get compiler errors because of the lack of xsup_ipc.*.  So, we should
  // define EXTERNAL_USE so that we don't attempt to build that part.
#ifndef EXTERNAL_USE
  // No matter what, we want to send it to connected gui interfaces.
  if (level == DEBUG_NORMAL)
    {
      xsup_ipc_send_log(level, instr);
    }
#endif

  // No decide where else to log to.
  if (((isdaemon == 2) || (fh == NULL)) && (syslogging != 1))
    {
      printf("%s", instr);
      fflush(stdout);
    } else if (syslogging ==1) {
      // XXX Consider ways of using other log levels.
      syslog(LOG_ALERT, "%s", instr);
    } else {
      fprintf(fh, "%s", instr);
      fflush(fh);
    }
}

/*************************************************
 *
 * Set the debug level.  This is a global value, and shouldn't be set per
 * interface.
 *
 *************************************************/
void debug_setdaemon(int xdaemon)
{
  isdaemon = xdaemon;

  if (xdaemon == TRUE)
    {
      close(0);
      close(1);
      close(2); 
    }
}

/*************************************************
 *
 * Get the debug level for debug situations where we can't use debug_printf
 * easily.
 *
 *************************************************/
int debug_getlevel()
{
  return debug_level;
}

static inline char to_hex_char(int val)
{
   return("0123456789abcdef"[val & 0xf]);
}

/*************************************************
 *
 * Dump hex values, without the ascii versions.
 *
 *************************************************/
void debug_hex_printf(unsigned char level, uint8_t *hextodump, int size)
{
  int i;
  int len = 0;
  char logstr[(size * 3) + 2];
  
  if ((!(debug_level & level)) && (level != 0))
    return;
  
  if (hextodump == NULL)
    return;
  
  for (i = 0; i < size; i++)
    {
      logstr[len++] = to_hex_char(hextodump[i] >> 4);
      logstr[len++] = to_hex_char(hextodump[i]);
      logstr[len++] = ' ';
    }
  
  logstr[len++] = '\n';
  logstr[len] = 0;
  ufprintf(logfile, logstr, level);
}

/*************************************************
 *
 * dump some hex values -- also
 * show the ascii version of the dump.
 *
 *************************************************/
void debug_hex_dump(unsigned char level, uint8_t *hextodump, int size)
{
  int i;
  char buf[80];
  int str_idx = 0;
  int chr_idx = 0;
  int count;
  int total;
  int tmp;
  
  if ((!(debug_level & level)) && (level != 0))
    return;
  
  if (hextodump == NULL)
    return;
  
  /* Initialize constant fields */
  memset(buf, ' ', sizeof(buf));
  buf[4]  = '|';
  buf[54] = '|';
  buf[72] = '\n';
  buf[73] = 0;
  
  count = 0;
  total = 0;
  for (i = 0; i < size; i++)
    {
      if (count == 0)
	{
          str_idx = 6;
          chr_idx = 56;
	  
          buf[0] = to_hex_char(total >> 8);
          buf[1] = to_hex_char(total >> 4);
          buf[2] = to_hex_char(total);
	}
      
      /* store the number */
      tmp = hextodump[i];
      buf[str_idx++] = to_hex_char(tmp >> 4);
      buf[str_idx++] = to_hex_char(tmp);
      str_idx++;
      
      /* store the character */
      buf[chr_idx++] = isprint(tmp) ? tmp : '.';
      
      total++;
      count++;
      if (count >= 16)
	{
          count = 0;
          ufprintf(logfile, buf, level);
	}
    }
  
  /* Print partial line if any */
  if (count != 0)
    {
      /* Clear out any junk */
      while (count < 16)
	{
          buf[str_idx]   = ' ';   /* MSB hex */
          buf[str_idx+1] = ' ';   /* LSB hex */
          str_idx += 3;
	  
          buf[chr_idx++] = ' ';
	  
          count++;
	}
      ufprintf(logfile, buf, level);
    }
}

/*************************************************
 *
 * Display some information.  But only if we are at a debug level that
 * should display it.
 *
 *************************************************/
void debug_printf(unsigned char level, char *fmt, ...)
{
  char dumpstr[2048], temp[2048];

  if (((level & debug_level) || (level == 0)) && (fmt != NULL))
    {
      va_list ap;
      va_start(ap, fmt);

      bzero((char *)&dumpstr, 2048);
      bzero((char *)&temp, 2048);

      // Print out a tag that identifies the type of debug message being used.
      switch (level)
	{
	case DEBUG_NORMAL:
	  break;   
	  
	case DEBUG_CONFIG:
	  strcpy((char *)&dumpstr, "[CONFIG] ");
	  break;

	case DEBUG_STATE:
	  strcpy((char *)&dumpstr, "[STATE] ");
	  break;

	case DEBUG_AUTHTYPES:
	  strcpy((char *)&dumpstr, "[AUTH TYPE] ");
	  break;
	  
	case DEBUG_INT:
	  strcpy((char *)&dumpstr, "[INT] ");
	  break;

	case DEBUG_EVERYTHING:
	  strcpy((char *)&dumpstr, "[ALL] ");
	  break;
	}

      vsnprintf((char *)&temp, 2048, fmt, ap);
      
      strcat((char *)&dumpstr, (char *)&temp);

      ufprintf(logfile, dumpstr, level);

      va_end(ap);
    }
}

/*************************************************
 *
 * Display some information.  But only if we are at a debug level that
 * should display it.
 *
 *************************************************/
void debug_printf_nl(unsigned char level, char *fmt, ...)
{
  char temp[2048];

  if ((((level & debug_level) != 0x00) || (level == 0)) && (fmt != NULL))
    {
      va_list ap;
      va_start(ap, fmt);

      vsnprintf((char *)&temp, 2048, fmt, ap);

      ufprintf(logfile, temp, level);
	  
      va_end(ap);
    }
}

