/*******************************************************************
 * File: wpa.c
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: wpa.c,v 1.30 2006/09/27 19:37:19 chessing Exp $
 * $Date: 2006/09/27 19:37:19 $
 * $Log: wpa.c,v $
 * Revision 1.30  2006/09/27 19:37:19  chessing
 * Fixed a potential problem in there was a partial configuration for a network.
 *
 * Revision 1.29  2006/08/25 23:37:17  chessing
 * Numerous patches that have come in over the last month or two.
 *
 * Revision 1.28  2006/06/23 02:15:18  chessing
 * Fixed some endianness issues in wpa.c.
 *
 * Revision 1.27  2006/06/20 21:51:12  chessing
 * Fix to the previous endianness fix. ;)
 *
 * Revision 1.26  2006/06/20 18:17:18  chessing
 * Fix some endianness problems with WPA and WPA2 IE parsing.
 *
 * Revision 1.25  2006/05/26 22:04:58  chessing
 * Fixed some memory access errors, and cleaned up some wext stuff that was causing issues with the madwifi driver in wext mode.
 *
 * Revision 1.24  2006/05/22 04:02:09  chessing
 * Some small changes/fixes.
 *
 * Revision 1.23  2006/05/13 03:44:46  chessing
 * Huge patch as a result of the downtime with SourceForge.  Fixed issues with WPA/WPA2.  Fixed a segfault when a scan returned a NULL.  Changed event handleing to use select() instead of older sleep/non-blocking socket resulting in a faster authentication.  Fixed a stack smashing bug that would cause random failures with WPA/WPA2.  Added status messages so that when we run in non-debug mode we have some idea of what is going on.  Various other code cleanups, bug fixes.
 *
 * Revision 1.22  2006/04/25 01:17:42  chessing
 * LOTS of code cleanups, new error checking/debugging code added, and other misc. fixes/changes.
 *
 * Revision 1.21  2006/01/20 06:53:48  chessing
 * Added support for large scan results.
 *
 * Revision 1.20  2006/01/19 23:01:43  chessing
 * Added a CHANGELOG file.  Fixed a couple of things in wpa.c
 *
 * Revision 1.19  2006/01/19 22:23:36  chessing
 * Added the ability to associate to an AP without having wpa_group_cipher, or wpa_pairwise_cipher defined in the config file.  If they are defined, then the value in the config file will be used no matter what.  Also, changed a piece in xsup_driver.c so that we print the capabilities that the cards reports on startup.  (This should help debug some problems that are almost certain to pop up with this new code. ;)
 *
 * Revision 1.18  2006/01/19 05:37:04  chessing
 * WPA2 is working correctly.  Added the ability to query the card to gather encryption/authentication capabilities.  1.2.3 is now ready to go.
 *
 * Revision 1.17  2005/12/24 04:51:51  chessing
 * Removed an unneeded file.  Updated scanning code to parse IWEVGENIE events correctly, and make use of them.
 *
 * Revision 1.16  2005/10/17 03:56:54  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.15  2005/08/25 03:34:05  chessing
 * Removed a bunch of functions from config.c that could be handled better in other ways.
 *
 * Revision 1.14  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <string.h>
#include <inttypes.h>
#include <unistd.h>
#include <sys/types.h>

#include "xsupconfig.h"
#include "profile.h"
#include "xsup_debug.h"
#include "wpa.h"
#include "cardif/cardif.h"
#include "config_ssid.h"
#include "wpa_common.h"
#include "xsup_err.h"

#ifdef USE_EFENCE
#include <efence.h>
#endif

const char wpaoui[3] = {0x00, 0x50, 0xf2};

/*************************************************************************
 *
 * Get the pairwise cipher that we want to use.  If one is configured in the
 * config file, then we will use that one.  Otherwise, we will make a decision
 * based on the information provided by the AP, and provided by the wireless
 * card.
 *
 *************************************************************************/
uint8_t wpa_get_pairwise_crypt(struct interface_data *ctx)
{
  struct config_network *network_data;
  uint8_t available_pair = 0, wpa_ie_len = 0;
  uint8_t *pairptr;
  uint8_t *wpa_ie = NULL;
  uint8_t i;
  uint16_t *ciphers;

  if (!xsup_assert((ctx != NULL), "ctx != NULL", FALSE))
    return XEMALLOC;

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return XEMALLOC;

  config_ssid_get_wpa_ie(&wpa_ie, &wpa_ie_len);

  if (wpa_ie == NULL)
    {
      debug_printf(DEBUG_NORMAL, "This SSID didn't return a WPA IE in our "
		   "scan!  We won't be able to connect!\n");
      return -1;
    }

  if ((wpa_ie_len <= 0) && (network_data->wpa_pairwise_crypt == 0))
    {
      debug_printf(DEBUG_NORMAL, "Insufficient information to build WPA "
		   "IE.  Please set 'wpa_pairwise_crypt' value in your network"
		   " clause for this network.\n");
      return -1;
    }

  // If the user has manually set a crypto type, then use it.
  if (network_data->wpa_pairwise_crypt != 0) 
    return network_data->wpa_pairwise_crypt;

  // Otherwise, see what the card has told us we can support, and compare it
  // to what the AP claims to support.
  ciphers = (uint16_t *)&wpa_ie[12];

  debug_printf(DEBUG_INT, "There are %d pairwise cipher(s) in this IE.\n",
	       *ciphers);
  pairptr = (uint8_t *)&wpa_ie[14];

  for (i=0;i<(*ciphers);i++)
    {
      if (memcmp(pairptr, &wpaoui, 3) != 0)
	{
	  debug_printf(DEBUG_NORMAL, "One of this AP's pairwise key settings "
		       "seems to be proprietary.  Skipping.\n");
	  pairptr += 4;  // Skip this.
	}
      else 
	{
	  pairptr += 3;
	  
	  if ((*pairptr) == CIPHER_WEP40) available_pair |= DOES_WEP40;
	  if ((*pairptr) == CIPHER_WEP104) available_pair |= DOES_WEP104;
	  if ((*pairptr) == CIPHER_TKIP) available_pair |= DOES_TKIP;
	  if ((*pairptr) == CIPHER_CCMP) available_pair |= DOES_CCMP;

	  pairptr++;
	}
    }

  // We want to test for cipher types from the best down to the worst, so
  // that we will select the best cipher possible.
  if (available_pair & DOES_CCMP)
    {
      if (ctx->enc_capa & DOES_CCMP)
	{
	  return CIPHER_CCMP;
	} 
      else 
	{
	  debug_printf(DEBUG_NORMAL, "The AP has requested that we use CCMP"
		       " for the pairwise cipher.  But, your card reports that"
		       " it doesn't support CCMP.  If you are sure that your"
		       " card supports CCMP, you should add "
		       "'wpa_pairwise_cipher = CCMP' to your network clause for"
		       " this network.\n");
	  return -1;
	}
    }

 if (available_pair & DOES_TKIP) 
    {
      if (ctx->enc_capa & DOES_TKIP)
	{
	  return CIPHER_TKIP;
	}
      else
	{
	  debug_printf(DEBUG_NORMAL, "The AP has requested that we use TKIP"
		       " for the pairwise cipher.  But, your card reports that"
		       " it doesn't support TKIP.  If you are sure that your"
		       " card supports TKIP, you should add "
		       "'wpa_pairwise_cipher = TKIP' to your network clause for"
		       " this network.\n");
	  return -1;
	}
    }

  if (available_pair & DOES_WEP104)
    {
      if (ctx->enc_capa & DOES_WEP104)
	{
	  return CIPHER_WEP104;
	}
      else
	{
	  debug_printf(DEBUG_NORMAL, "The AP has requested that we use WEP104"
		       " for the pairwise cipher.  But, your card reports that"
		       " it doesn't support WEP104.  If you are sure that your"
		       " card supports WEP104, you should add "
		       "'wpa_pairwise_cipher = WEP104' to your network clause for"
		       " this network.\n");
	  return -1;
	}
    }

 if (available_pair & DOES_WEP40)
   {
     if (ctx->enc_capa & DOES_WEP40)
       {
	 return CIPHER_WEP40;
       }
     else
       {
	 debug_printf(DEBUG_NORMAL, "The AP has requested that we use WEP40"
		      " for the pairwise cipher.  But, your card reports that"
		      " it doesn't support WEP40.  If you are sure that your"
		      " card supports WEP40, you should add "
		      "'wpa_pairwise_cipher = WEP40' to your network clause for"
		      " this network.\n");
	 return -1;
       }
   }

  // If we get here, then the AP has requested a cipher type we don't 
  // understand.
  debug_printf(DEBUG_NORMAL, "The AP has requested a cipher type that we "
	       "don't understand.\n");
  return -1;
}

/**********************************************************************
 *
 * Determine the group cipher that we should use.  If a value is set in the
 * configuration file, then we will make use of that.  Otherwise, we will
 * decide based on what the IE from the AP indicates, and the capabilities
 * that are supported by the card.
 *
 **********************************************************************/
uint8_t wpa_get_group_crypt(struct interface_data *ctx)
{
  struct config_network *network_data;
  uint8_t desired_group = -1, wpa_ie_len = 0;
  uint8_t *grpptr;
  uint8_t *wpa_ie;

  if (!xsup_assert((ctx != NULL), "ctx != NULL", FALSE))
    return XEMALLOC;

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return XEMALLOC;

  config_ssid_get_wpa_ie(&wpa_ie, &wpa_ie_len);

  if (!wpa_ie)
    {
      debug_printf(DEBUG_NORMAL, "This SSID didn't return an WPA IE in our "
		   "scan!  We won't be able to connect!\n");
      return -1;
    }

  if ((wpa_ie_len <= 0) && (network_data->wpa_group_crypt == 0))
    {
      debug_printf(DEBUG_NORMAL, "Insufficient information to build WPA "
		   "IE.  Please set 'wpa_group_crypt' value in your network"
		   " clause for this network.\n");
      return -1;
    }

  // If the user has manually set a crypto type, then use it.
  if (network_data->wpa_group_crypt != 0) 
    return network_data->wpa_group_crypt;

  // Otherwise, see what the card has told us we can support, and compare it
  // to what the AP claims to support.
  grpptr = (uint8_t *)&wpa_ie[8];
  if (memcmp(grpptr, &wpaoui, 3) != 0)
    {
      debug_printf(DEBUG_NORMAL, "AP's group key setting seems to be "
		   "proprietary.  This is unsupported.\n");
      return -1;
    }

  // Get the key type that is desired.
  desired_group = grpptr[3];

  if (desired_group == CIPHER_WEP40)
    {
      if (ctx->enc_capa & DOES_WEP40)
	{
	  return CIPHER_WEP40;
	}
      else
	{
	  debug_printf(DEBUG_NORMAL, "The AP has requested that we use WEP40"
		       " for the group cipher.  But, your card reports that"
		       " it doesn't support WEP40.  If you are sure that your"
		       " card supports WEP40, you should add "
		       "'wpa_group_cipher = WEP40' to your network clause for"
		       " this network.\n");
	  return -1;
	}
    }

  if (desired_group == CIPHER_WEP104)
    {
      if (ctx->enc_capa & DOES_WEP104)
	{
	  return CIPHER_WEP104;
	}
      else
	{
	  debug_printf(DEBUG_NORMAL, "The AP has requested that we use WEP104"
		       " for the group cipher.  But, your card reports that"
		       " it doesn't support WEP104.  If you are sure that your"
		       " card supports WEP104, you should add "
		       "'wpa_group_cipher = WEP104' to your network clause for"
		       " this network.\n");
	  return -1;
	}
    }

  if (desired_group == CIPHER_TKIP) 
    {
      if (ctx->enc_capa & DOES_TKIP)
	{
	  return CIPHER_TKIP;
	}
      else
	{
	  debug_printf(DEBUG_NORMAL, "The AP has requested that we use TKIP"
		       " for the group cipher.  But, your card reports that"
		       " it doesn't support TKIP.  If you are sure that your"
		       " card supports TKIP, you should add "
		       "'wpa_group_cipher = TKIP' to your network clause for"
		       " this network.\n");
	  return -1;
	}
    }

  if (desired_group == CIPHER_CCMP)
    {
      if (ctx->enc_capa & DOES_CCMP)
	{
	  return CIPHER_CCMP;
	} 
      else 
	{
	  debug_printf(DEBUG_NORMAL, "The AP has requested that we use CCMP"
		       " for the group cipher.  But, your card reports that"
		       " it doesn't support CCMP.  If you are sure that your"
		       " card supports CCMP, you should add "
		       "'wpa_group_cipher = CCMP' to your network clause for"
		       " this network.\n");
	  return -1;
	}
    }

  // If the desired group cipher is set to 0, then we should use the same
  // cipher as the pairwise cipher.
  if (desired_group == 0)
    {
      return wpa_get_pairwise_crypt(ctx);
    }

  // If we get here, then the AP has requested a cipher type we don't 
  // understand.
  debug_printf(DEBUG_NORMAL, "The AP has requested a cipher type that we "
	       "don't understand.  Type %d.\n", desired_group);
  return -1;
}

void wpa_gen_ie(struct interface_data *thisint, char *iedata)
{
  struct config_network *network_data;
  struct config_globals *globals;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return;

  if (!xsup_assert((iedata != NULL), "iedata != NULL", FALSE))
    return;

  network_data = config_get_network_config();

  if (!xsup_assert((network_data != NULL), "network_data != NULL", FALSE))
    return;

  globals = config_get_globals();
  
  if (!xsup_assert((globals != NULL), "globals != NULL", FALSE))
    return;

  // The first byte is the Element ID for WPA, which is 0xdd.
  iedata[0] = WPA_EID;

  // For WPA (without capabilities), the length value will always be 22.
  iedata[1] = 22;

  // For WPA, we need to add the "special" OUI before the version #.
  memcpy(&iedata[2], wpaoui, 3);
  iedata[5] = 0x01;

  // Set the version #
  iedata[6] = 0x01;
  iedata[7] = 0x00;

  // The group key cipher suite.
  memcpy(&iedata[8], wpaoui, 3);

  if ((iedata[11] = wpa_get_group_crypt(thisint)) == -1)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't build WPA IE!  (Error getting "
		   "group cipher information!\n");
      iedata[1] = 0;
      return;
    }

  debug_printf(DEBUG_CONFIG, "Using Group Cipher Suite : ");
  wpa_print_cipher_suite(DEBUG_CONFIG, iedata[11]);

  // We can only have 1 pairwise cipher suite!
  iedata[12] = 0x01;
  iedata[13] = 0x00;

  // The pairwise cipher suite.
  memcpy(&iedata[14], wpaoui, 3);

  //  iedata[17] = network_data->wpa_pairwise_crypt;
  if ((iedata[17] = wpa_get_pairwise_crypt(thisint)) == -1)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't build WPA IE!  (Error getting "
		   "pairwise cipher information!\n");
      iedata[1] = 0;
      return;
    }
      
  debug_printf(DEBUG_CONFIG, "Using Pairwise Cipher Suite : ");
  wpa_print_cipher_suite(DEBUG_CONFIG, iedata[17]);

  if ((network_data->wpa_group_crypt == CIPHER_TKIP) &&
      ((network_data->wpa_pairwise_crypt == CIPHER_WRAP) ||
       (network_data->wpa_pairwise_crypt == CIPHER_CCMP)))
    {

      if (!TEST_FLAG(globals->flags, CONFIG_GLOBALS_NO_FRIENDLY_WARNINGS))
	{
	  debug_printf(DEBUG_NORMAL, "WARNING : Group cipher is TKIP and "
		       "pairwise cipher is using AES.  Many wireless cards "
		       "have problems with this combination!\n");
	}
    }

  if (((network_data->wpa_group_crypt == CIPHER_WEP40) ||
       (network_data->wpa_group_crypt == CIPHER_WEP104)) &&
      ((network_data->wpa_pairwise_crypt == CIPHER_WRAP) ||
       (network_data->wpa_pairwise_crypt == CIPHER_CCMP)))
    {
      
      if (!TEST_FLAG(globals->flags, CONFIG_GLOBALS_NO_FRIENDLY_WARNINGS))
	{
	  debug_printf(DEBUG_NORMAL, "WARNING : Group cipher is WEP and "
		       "pairwise cipher is using AES.  Many wireless cards "
		       "have problems with this combination!\n");
	}
    }

  // For the authenticated key management suite, we can also only have 1.
  iedata[18] = 0x01;
  iedata[19] = 0x00;

  // The authenticated key management suite.
  memcpy(&iedata[20], wpaoui, 3);

  if (network_data->methods == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No valid EAP methods defined for this "
		   "network!\n");
      return;
    }

  if (network_data->methods->method_num == WPA_PSK)
    {
      iedata[23] = 2;  // WPA-PSK
      debug_printf(DEBUG_CONFIG, "Using WPA-PSK.\n");
    } else {
      iedata[23] = 1;  // WPA with 802.1X
      debug_printf(DEBUG_CONFIG, "Using WPA with 802.1X\n");
    }
}
 
void wpa_gen_ie_caps(struct interface_data *thisint, char *iedata)
{
  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return;

  if (!xsup_assert((iedata != NULL), "iedata != NULL", FALSE))
    return;

  wpa_gen_ie(thisint, iedata);

  iedata[1] = 24;
  iedata[24] = 0x00;
  iedata[25] = 0x00;
} 
  
/*************************************************************************
 *
 * Parse an information element.  Returns IE length  if it is a valid IE, 
 * and -1 if it isn't.
 *
 *************************************************************************/
int wpa_parse_ie(char *iedata)
{
  struct wpa_ie_struct *ie_struct;
  char wpa_oui[3] = {0x00, 0x50, 0xf2};
  char suite_id[4];
  int i, ieptr;
  uint16_t value16;

  if (!xsup_assert((iedata != NULL), "iedata != NULL", FALSE))
    return XEMALLOC;

  ie_struct = (struct wpa_ie_struct *)iedata;

  if (ie_struct->wpaid != 0xdd)
    {
      debug_printf(DEBUG_NORMAL, "IE is not a valid WPA IE! (Invalid vendor value!)\n");
      return -1;
    }

  debug_printf(DEBUG_INT, "--- WPA Data ---\n");
  
  if ((memcmp(ie_struct->oui, wpa_oui, 3) != 0) && (ie_struct->oui[3] != 0x01))
    {
      debug_printf(DEBUG_NORMAL, "IE is not a valid WPA IE! (Invalid OUI value!\n");
      return -1;
    }

  byte_swap(&ie_struct->wpa_ver);

  debug_printf(DEBUG_INT, "WPA Version : %d\n", ie_struct->wpa_ver);
  if (ie_struct->wpa_ver > MAX_WPA_VER)
    {
      debug_printf(DEBUG_NORMAL, "The requested WPA version of %d is greater"
		   " than the highest version we support of %d.\n",
		   ie_struct->wpa_ver, MAX_WPA_VER);
      return -1;
    }

  // From here, everything else is technically optional.
  if (ie_struct->wpalen <= 6)
    {
      // Nothing after the version #.
      debug_printf(DEBUG_NORMAL, "Short WPA IE.  Should assume TKIP/TKIP for "
		   "keying!\n");
      return ie_struct->wpalen;
    }  // Otherwise, we have a group cipher suite.

  debug_printf(DEBUG_INT, "Group Key Cipher Suite : ");
  wpa_print_cipher_suite(DEBUG_INT, ie_struct->group_cipher[3]);

  if (ie_struct->wpalen <= 10)
    {
      debug_printf(DEBUG_NORMAL, "Short WPA IE.  Should assume TKIP for "
		   "pairwise cipher.\n");
      return ie_struct->wpalen;
    }

  byte_swap(&ie_struct->pk_suite_cnt);

  debug_printf(DEBUG_INT, "Pairwise Key Cipher Suite Count : %d\n",
	       ie_struct->pk_suite_cnt);

  ieptr = sizeof(struct wpa_ie_struct);

  for (i=0;i<ie_struct->pk_suite_cnt;i++)
    {
      if (ie_struct->wpalen < (ieptr-2) + 4)
	{
	  debug_printf(DEBUG_NORMAL, "Invalid WPA IE!  The number of "
		       "cipher suites is too high for the length of the IE!"
		       "\n");
	  return ie_struct->wpalen;
	}

      debug_printf(DEBUG_INT, "Cipher Suite : ");
      memcpy((char *)&suite_id, (char *)&iedata[ieptr], 4);

      if (memcmp(suite_id, wpa_oui, 3) != 0)
	{
	  debug_printf(DEBUG_INT, "Proprietary\n");
	} else {
	  wpa_print_cipher_suite(DEBUG_INT, suite_id[3]);
	}

      ieptr+=4;
    }

  if (ie_struct->wpalen < (ieptr-2) + 2)
    {
      debug_printf(DEBUG_NORMAL, "Short IE.  Should assume an AKM of EAP.\n");
      return ie_struct->wpalen;
    }

  memcpy((char *)&value16, (char *)&iedata[ieptr], 2);
  ieptr+=2;
  debug_printf(DEBUG_INT, "Authenticated Key Management Suite Count : %d\n",
	       value16);

  for (i=0;i<value16;i++)
    {
      if (ie_struct->wpalen < (ieptr-2) + 4)
	{
	  debug_printf(DEBUG_NORMAL, "Truncated IE!  The length provided in "
		       "the IE isn't long enough to include the number of "
		       "Authenticated Key Management Suites claimed!\n");
	  return -1;
	}

      debug_printf(DEBUG_INT, "Authentication Suite : ");
      memcpy((char *)&suite_id, (char *)&iedata[ieptr], 4);

      if (memcmp(suite_id, wpa_oui, 3) != 0)
	{
	  debug_printf(DEBUG_INT, "Proprietary\n");
	} else {
	  wpa_print_auth_suite(DEBUG_INT, suite_id[3]);
	}
      ieptr+=4;
    }

  if ((ieptr-2) < ie_struct->wpalen)
    {
      memcpy((char *)&value16, (char *)&iedata[ieptr], 2);
      debug_printf(DEBUG_INT, "RSN Capabilities : %04X\n\n", value16);
    }

  return ie_struct->wpalen;
}

/* These functions are defined by 802.11i-D3.0 */
void wpa_STADisconnect()
{

}

void wpa_RemoveGTK()
{

}

