/*******************************************************************
 * Handle MIC routines.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE for more info.)
 *
 * File: mic.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: mic.c,v 1.21 2006/06/01 22:49:49 galimorerpg Exp $
 * $Date: 2006/06/01 22:49:49 $
 * $Log: mic.c,v $
 * Revision 1.21  2006/06/01 22:49:49  galimorerpg
 * Converted all instances of u_char to uint8_t
 * Fixed a bad #include in the generic frame handler.
 *
 * Revision 1.20  2006/06/01 20:46:25  chessing
 * More patches from Carsten Grohmann.
 *
 * Revision 1.19  2006/05/29 04:17:57  chessing
 * Fixes for some memory leaks.
 *
 * Revision 1.18  2006/05/29 02:21:41  chessing
 * Madwifi now works with WPA/WPA2 and the wext driver.
 *
 * Revision 1.17  2006/05/26 23:21:12  chessing
 * Fixed some memory leaks.
 *
 * Revision 1.16  2006/05/26 22:04:58  chessing
 * Fixed some memory access errors, and cleaned up some wext stuff that was causing issues with the madwifi driver in wext mode.
 *
 * Revision 1.15  2006/04/25 01:17:42  chessing
 * LOTS of code cleanups, new error checking/debugging code added, and other misc. fixes/changes.
 *
 * Revision 1.14  2006/01/15 06:56:29  chessing
 * Some fixes to WPA/WPA2.  WPA is working correctly now.  WPA2 needs a little work to correctly identify key frames that are for the GTK.
 *
 * Revision 1.13  2005/10/14 02:26:17  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.12  2005/08/09 01:39:14  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <openssl/hmac.h>
#include <string.h>

#include "xsup_debug.h"
#include "profile.h"
#include "eapol_key_type254.h"
#include "frame_structs.h"
#include "timer.h"
#include "cardif/cardif.h"
#include "mic.h"
#include "frame_structs.h"

#ifdef USE_EFENCE
#include <efence.h>
#endif

struct eapol_minus_type {
  uint8_t eapol_version;
  uint8_t eapol_type;
  uint16_t eapol_length;
} __attribute__((__packed__));

/************************************************************************
 *
 * Calculate the MIC value for a key packet, and return it.
 *
 ************************************************************************/
void mic_process(char *key, int keylen, char *datain, int insize, int version, 
		 char *mic)
{
  char *sha_hmac = NULL;
  struct eapol_minus_type *header;
  int i;
  uint16_t rlen;

  if (!xsup_assert((key != NULL), "key != NULL", FALSE))
    return;

  if (!xsup_assert((datain != NULL), "datain != NULL", FALSE))
    return;

  if ((insize < 0) || (insize > 1522))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data size of %d! (%s:%d)\n", insize,
		   __FUNCTION__, __LINE__);
      return;
    }

  if (!xsup_assert((mic != NULL), "mic != NULL", FALSE))
    return;

  debug_printf(DEBUG_INT, "Calculating MIC for Version %d!\n", version);

  // Some drivers may pass some extra stuff up that is beyond the end
  // of the data we are interested in.  So, we want to make sure we only
  // MIC the data that we should.
  header = (struct eapol_minus_type *)datain;
  rlen = ntohs(header->eapol_length);
  rlen+=4;

  switch (version)
    {
    case 1:
      /* Do an MD5 HMAC */
      HMAC(EVP_md5(), key, keylen, (uint8_t *) datain, rlen, (uint8_t *) mic, 
	   (u_int *) &i);
      break;

    case 2:
      /* Do an SHA1 HMAC  
       * Since the HMAC will be 20 bytes, and we only need 16, we must use
       * a temporary variable. */
      sha_hmac = (char *)malloc(20);
      if (!xsup_assert((sha_hmac != NULL), "sha_hmac != NULL", FALSE))
	return;

      HMAC(EVP_sha1(), key, keylen, (uint8_t *) datain, rlen, 
	   (uint8_t *)sha_hmac, (u_int *) &i);
      memcpy(mic, sha_hmac, 16);
      free(sha_hmac);
      sha_hmac = NULL;
      break;

    default:
      debug_printf(DEBUG_NORMAL, "Unknown MIC version!  (%d)\n", version);
      break;
    }
}

/*******************************************************************
 *
 * Given a frame, pull the MIC out, and check it.  If it matches,
 * return TRUE.  Otherwise, return FALSE.
 *
 *******************************************************************/
int mic_wpa_validate(char *inframe, int framesize, char *key, int keylen)
{
  struct wpa_key_packet *keydata;
  char *tempframe;
  char oldmic[16], newmic[16];
  uint16_t value16;
  int rc = FALSE;

  if (!xsup_assert((inframe != NULL), "inframe != NULL", FALSE))
    return FALSE;

  tempframe = (char *)malloc(framesize);
  if (tempframe == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory at %s line %d!\n",
		   __FUNCTION__, __LINE__);
      return FALSE;
    }

  memcpy(tempframe, inframe, framesize);

  /* First, get the keydata struct pointing to the correct place. */
  keydata = (struct wpa_key_packet *)&tempframe[OFFSET_TO_EAPOL+4];
  
  memcpy(oldmic, keydata->key_mic, 16);
  bzero(keydata->key_mic, 16);

  memcpy(&value16, keydata->key_information, 2);
  value16 = ntohs(value16);
  value16 &= WPA_KEYTYPE_MASK;

  mic_process(key, keylen, (char *)&tempframe[OFFSET_TO_EAPOL],
	      framesize - OFFSET_TO_EAPOL, value16, newmic);

  if (memcmp(oldmic, newmic, 16) == 0) rc = TRUE;

  free(tempframe);
  tempframe = NULL;

  return rc;
}

/*******************************************************************
 *
 * Given a frame, calculate the MIC, and stick it in the frame.
 *
 *******************************************************************/
void mic_wpa_populate(char *inframe, int framesize, char *key, int keylen)
{
  struct wpa_key_packet *keydata;
  char newmic[16];
  uint16_t value16;

  if (!xsup_assert((inframe != NULL), "inframe != NULL", FALSE))
    return;

  if (!xsup_assert((key != NULL), "key != NULL", FALSE))
    return;

  if ((framesize > 1522) || (framesize < 0))
    {
      debug_printf(DEBUG_NORMAL, "Invalid frame size of %d! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return;
    }

  /* First, get the keydata struct pointing to the correct place.  */
  keydata = (struct wpa_key_packet *)&inframe[OFFSET_TO_EAPOL+4];
  
  bzero(keydata->key_mic, 16);

  memcpy(&value16, &keydata->key_information, 2);
  value16 = ntohs(value16);
  value16 &= WPA_KEYTYPE_MASK;

  mic_process(key, keylen, &inframe[OFFSET_TO_EAPOL],
	      framesize - OFFSET_TO_EAPOL -4, value16, newmic);

  memcpy(keydata->key_mic, newmic, 16);
}


/*************************************
 *
 * When countermeasures have been enabled, this is the function that will
 * be called once they are ready to be disabled.
 *
 *************************************/
void mic_disable_countermeasures(struct interface_data *intdata)
{
  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return;

  debug_printf(DEBUG_NORMAL, "MIC countermeasures disabled.\n");
  
  cardif_countermeasures(intdata, FALSE);
  timer_cancel(COUNTERMEASURE_TIMER);
}
