/*******************************************************************
 * Event core implementation.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: event_core.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: event_core.c,v 1.13 2006/09/14 05:07:08 chessing Exp $
 * $Date: 2006/09/14 05:07:08 $
 * $Log: event_core.c,v $
 * Revision 1.13  2006/09/14 05:07:08  chessing
 * Fixed some problems with wired authentication.
 *
 * Revision 1.12  2006/05/26 23:21:12  chessing
 * Fixed some memory leaks.
 *
 * Revision 1.11  2006/05/26 22:04:58  chessing
 * Fixed some memory access errors, and cleaned up some wext stuff that was causing issues with the madwifi driver in wext mode.
 *
 * Revision 1.10  2006/05/22 22:29:17  chessing
 * Compiler warnings patches from Eric Evans.  (And one from me. ;)
 *
 * Revision 1.9  2006/05/18 01:08:16  chessing
 * One small change to Xsupplicant, major new stuff for the GUI application.
 *
 * Revision 1.8  2006/05/17 22:18:10  chessing
 * A couple of small changes to Xsupplicant, and some major changes to the GUI configuration/monitor tool.
 *
 * Revision 1.7  2006/05/17 02:40:06  chessing
 * Finished several things on the todo list.  Removed external event core stuff, added gui pings, clear keys and WPA IE when an SSID is set from outside Xsupplicant, etc.
 *
 * Revision 1.6  2006/05/14 22:09:28  chessing
 * A few small changes in the Xsupplicant code.  Beginning of a configuration/monitor program can be found in gui_tools. (Very simple for now, with several known bugs, but no showstopper bugs know.)
 *
 * Revision 1.5  2006/05/13 03:44:46  chessing
 * Huge patch as a result of the downtime with SourceForge.  Fixed issues with WPA/WPA2.  Fixed a segfault when a scan returned a NULL.  Changed event handleing to use select() instead of older sleep/non-blocking socket resulting in a faster authentication.  Fixed a stack smashing bug that would cause random failures with WPA/WPA2.  Added status messages so that when we run in non-debug mode we have some idea of what is going on.  Various other code cleanups, bug fixes.
 *
 * Revision 1.4  2006/04/25 01:17:42  chessing
 * LOTS of code cleanups, new error checking/debugging code added, and other misc. fixes/changes.
 *
 * Revision 1.3  2005/10/17 03:56:53  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.2  2005/09/14 02:50:44  chessing
 * Major updates.  Auto association now works.  Started to rewrite the rtnetlink pieces using iwlib from the wireless tools to avoid compatibility issues.  As a result, getting the WPA and RSN IEs via the IWEVGENIE event no longer works for some reason, but the old style using IWEVCUSTOM events should still work like a champ.
 *
 * Revision 1.1  2005/09/05 01:00:34  chessing
 * Major overhaul to most of the state machines in Xsupplicant.  Also added additional error messages to the TLS functions to try to debug the one of the problems reported on the list.  Basic testing shows this new code to be more stable than previous code, but it needs more testing.
 *
 *
 *******************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>

#include "profile.h"
#include "xsupconfig.h"
#include "wireless_sm.h"
#include "event_core.h"
#include "eapol.h"
#include "timer.h"
#include "xsup_debug.h"
#include "statemachine.h"

#ifdef USE_EFENCE
#include <efence.h>
#endif

typedef struct eventhandler_struct {
  int socket;
  char *name;
  int (*func_to_call)(struct interface_data *, int);
} eventhandler;

eventhandler events[MAX_EVENTS];  

int locate;

time_t last_check = 0;

/***********************************************************************
 *
 * Initialize the event core. -- This should make any generic calls needed
 * to set up the OS to provide us with events.
 *
 ***********************************************************************/
void event_core_init()
{
  int i;

  for (i=0; i < MAX_EVENTS; i++)
    {
      events[i].socket = -1;
      events[i].name = NULL;
      events[i].func_to_call = NULL;
    }

  time(&last_check);   // Get our starting clock position.
}

/***********************************************************************
 *
 * Deinit the event core. -- This should call deinit functions for anything
 * that was inited in event_core_init()
 *
 ***********************************************************************/
void event_core_deinit()
{
  int i;

  for (i=0;i < MAX_EVENTS; i++)
    {
      if (events[i].name)
	{
	  debug_printf(DEBUG_CONFIG, "Clearing handler '%s'.\n",
		       events[i].name);

	  free(events[i].name);
	  events[i].name = NULL;
	}
    }
}

/*************************************************************************
 *
 *  Register a socket, and a function to call when that socket has something
 * to read.  If "hilo" is set to 0, we will register this socket to the 
 * highest priority handler available.  If it is set to 2, we will register
 * this socket to the lowest priority handler available.  If it is set to 1,
 * we will register it to whatever is available.
 *
 * Returns :
 *   -1 -- if there are no more slots available
 *    0 -- on success
 *
 *************************************************************************/
int event_core_register(int sock, 
		int(*call_func)(struct interface_data *, int), 
		int hilo, char *name)
{
  int i = 0, done = FALSE;

  if (!xsup_assert((call_func != NULL), "call_func != NULL", FALSE))
    return -1;

  if (!xsup_assert((name != NULL), "name != NULL", FALSE))
    return -1;

  if (hilo == 0)
    {
      while ((i < MAX_EVENTS) && (done != TRUE))
	{
	  if (events[i].socket < 0)
	    {
              debug_printf(DEBUG_CONFIG, "Registered event handler '%s' in "
                           "slot %d, with socket %d.\n", name, i, sock);

	      events[i].socket = sock;
	      events[i].name = strdup(name);
	      events[i].func_to_call = call_func;
	      done = TRUE;
	    }
	  
	  i++;
	}
      
      if ((i >= MAX_EVENTS) && (done == FALSE))
	{
	  debug_printf(DEBUG_NORMAL, "Not enough event handler slots "
		       "available!\n");
	  return -1;
	}
    }
  else
    {
      i = MAX_EVENTS - 1;
      while ((i >= 0) && (done != TRUE))
        {
          if (events[i].socket < 0)
	    {
	      debug_printf(DEBUG_CONFIG, "Registered event handler '%s' in "
			   "slot %d, with socket %d.\n", name, i, sock);
              events[i].socket = sock;
       	      events[i].name = strdup(name);
              events[i].func_to_call = call_func;
              done = TRUE;
            }
	  
	  i--;
        }

      if ((i < 0) && (done == FALSE))
        {
          debug_printf(DEBUG_NORMAL, "Not enough event handler slots "
                       "available!\n");
	  return -1;
        }
    }

  return 0;
}

/***********************************************************************
 *
 * Deregister an event handler based on the socket id that we have.
 *
 ***********************************************************************/
void event_core_deregister(int sock)
{
  int i;

  for (i=0;i < MAX_EVENTS;i++)
    {
      if (events[i].socket == sock)
	{
	  debug_printf(DEBUG_CONFIG, "Deregistering event handler '%s' in "
		       "slot %d, with socket %d.\n", events[i].name, i, 
		       sock);

	  if (events[i].name) free(events[i].name);
	  events[i].name = NULL;
	  events[i].socket = -1;
	  events[i].func_to_call = NULL;
	}
    }
}

/***********************************************************************
 *
 * Process any events that we may have received.  This includes processing
 * frames that may have come in.  There should be *NOTHING* OS specific
 * in this code!!!  OS specific calls should be held under 
 * cardif/<OS>/<OS>_core.c/h!
 *
 ***********************************************************************/
void event_core(struct interface_data *ctx)
{
  int i, biggest = 0, result = 0;
  fd_set rfds;
  struct timeval timeout;
  time_t cur_time;

  xsup_assert((ctx != NULL), "ctx != NULL", TRUE);

  FD_ZERO(&rfds);

  for (i=0; i < MAX_EVENTS; i++)
    {
      if (events[i].socket > 0)
	{
	  FD_SET(events[i].socket, &rfds);

	  if (events[i].socket > biggest)
	    biggest = events[i].socket;
	}
    }

  timeout.tv_sec = 1;
  timeout.tv_usec = 0;

  result = select(biggest+1, &rfds, NULL, NULL, &timeout);

  if (result < 0)
    {
      // If we are sent a HUP, we will get an interrupted system call error.
      // Since this is normal, and nothing to worry about, don't display
      // it to the user. ;)
      if (errno != EINTR)
	{
	  debug_printf(DEBUG_NORMAL, "Error : %s\n", strerror(errno));
	} 
      else
	{
	  debug_printf(DEBUG_CONFIG, "Error : %s\n", strerror(errno));
	}
    }

  if (result > 0)
    {
      for (i=0; i < MAX_EVENTS; i++)
	{
	  if (events[i].socket > 0)
	    {
	      if (FD_ISSET(events[i].socket, &rfds))
		{
		  debug_printf(DEBUG_CONFIG, "Socket %d (%s) had an event!\n",
			       events[i].socket, events[i].name);

		  if (events[i].func_to_call)
		    {
		      events[i].func_to_call(ctx, events[i].socket);
		    }
		}
	    }
	}
    }

  // See if a second has elapsed.  (This is useful in the situation where
  // we have an event storm that keeps select from timing out.  It may
  // result is some situations where we don't wait a full second before
  // ticking off a second.  But, it is better than the alternative. ;)
  time(&cur_time);

  if ((result == 0) || (cur_time > last_check))
    {
      ctx->statemachine->tick = TRUE;
      ctx->tick = TRUE;
      last_check = cur_time;

      timer_tick(ctx);

      // If this is a wired interface, we are response for running the 
      // state machine.
      if (!TEST_FLAG(ctx->flags, IS_WIRELESS)) statemachine_run(ctx);
    }
}

/**************************************************************************
 *
 * Reset our locator index to 0.
 *
 **************************************************************************/
void event_core_reset_locate()
{
  locate = 0;
}

/**************************************************************************
 *
 *  Find the next socket that matches the locate string, and return it.  
 *
 *  Returns :
 *
 *  > 0 -- on success
 *  -1  -- if there are no more sockets to be found, or an error happens.
 *
 **************************************************************************/
int event_core_locate(char *matchstr)
{
  int retsock, done = FALSE;

  // If we search after returning an "all done", then we have a nasty bug
  // that is likely to cause future problems.
  xsup_assert((locate < MAX_EVENTS), "locate < MAX_EVENTS", TRUE);

  if (!xsup_assert((matchstr != NULL), "matchstr != NULL", FALSE))
    return -1;

  // Start lookin!
  while (done != TRUE)
    {
      if (locate >= MAX_EVENTS)
	{
	  done = TRUE;
	}
      else
	if ((events[locate].name != NULL) && (strcmp(matchstr, 
						     events[locate].name) == 0))
	  {
	    done = TRUE;
	  }
	else
	  {
	    locate++;
	  }
    }

  if (locate < MAX_EVENTS)
    {
      // We have a winner!
      debug_printf(DEBUG_INT, "Found socket match at index %d, socket %d!\n", 
		   locate, events[locate].socket);
      retsock = events[locate].socket;
      locate++;   // Move on so we don't loop forever.
      return retsock;
    }
  
  // Otherwise, we ran out of options.
  return -1;
}
