/*******************************************************************
 * EAPTLS (RFC 2716) Function implementations
 * 
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: tls_funcs.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: tls_funcs.c,v 1.58 2006/06/09 23:25:44 chessing Exp $
 * $Date: 2006/06/09 23:25:44 $
 * $Log: tls_funcs.c,v $
 * Revision 1.58  2006/06/09 23:25:44  chessing
 * More GUI work.
 *
 * Revision 1.57  2006/06/05 23:27:36  chessing
 * Fixed a couple bugs introduced in the TTLS-EAP-MD5 code.  Cleaned up some other bugs introduced in this morning's commit.
 *
 * Revision 1.56  2006/06/05 19:56:43  chessing
 * Fixed a 16k memory leak in the config parse code.  Cleaned up some of the TLS code.
 *
 * Revision 1.55  2006/06/05 01:40:41  chessing
 * Various small cleanups.
 *
 * Revision 1.54  2006/06/02 21:49:18  chessing
 * More memory leak cleanups, and another patch from Carsten.
 *
 * Revision 1.53  2006/06/01 22:49:50  galimorerpg
 * Converted all instances of u_char to uint8_t
 * Fixed a bad #include in the generic frame handler.
 *
 * Revision 1.52  2006/06/01 04:03:04  chessing
 * Fixed a problem where we would segfault on a reauth.  (We probably need to get this fixed up a bit and release 1.2.6 quickly as a bug fix release.)  Added support for TTLS-EAP-MD5
 *
 * Revision 1.51  2006/05/29 04:17:58  chessing
 * Fixes for some memory leaks.
 *
 * Revision 1.50  2006/05/13 03:44:46  chessing
 * Huge patch as a result of the downtime with SourceForge.  Fixed issues with WPA/WPA2.  Fixed a segfault when a scan returned a NULL.  Changed event handleing to use select() instead of older sleep/non-blocking socket resulting in a faster authentication.  Fixed a stack smashing bug that would cause random failures with WPA/WPA2.  Added status messages so that when we run in non-debug mode we have some idea of what is going on.  Various other code cleanups, bug fixes.
 *
 * Revision 1.48  2006/04/25 01:17:44  chessing
 * LOTS of code cleanups, new error checking/debugging code added, and other misc. fixes/changes.
 *
 * Revision 1.47  2006/03/28 21:16:07  chessing
 * Fixes to EAP-AKA, and PEAP.  PEAP now handles inner success messages, and works with Funk SBR.
 *
 * Revision 1.46  2005/11/18 19:14:47  chessing
 * Fixes to the OpenSSL detection settings in the autoconf file.  Added some checks for SHA changes in OpenSSL 0.9.8
 *
 * Revision 1.45  2005/11/12 23:59:52  chessing
 * Some typo/debug patches from Pekka Savola.
 *
 * Revision 1.44  2005/10/17 03:56:54  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.43  2005/10/14 02:26:18  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.42  2005/09/14 03:34:54  chessing
 * Small cosmetic changes.  Default association mode is now auto instead of manual. Fixes for bug IDs #1290449 & #1290323.
 *
 * Revision 1.41  2005/09/08 16:27:02  chessing
 * Some small updates to the new state machine code.  First attempt at an auto association mode.  (It mostly works. ;)
 *
 * Revision 1.40  2005/09/05 01:00:37  chessing
 * Major overhaul to most of the state machines in Xsupplicant.  Also added additional error messages to the TLS functions to try to debug the one of the problems reported on the list.  Basic testing shows this new code to be more stable than previous code, but it needs more testing.
 *
 * Revision 1.39  2005/08/12 03:34:06  chessing
 * Fix to the TLS implementation, should help avoid some of the weird 'block cipher pad' errors.  Also includes a partial implementation of the ability to use static WEP keys based on the SSID in use.
 *
 * Revision 1.38  2005/08/09 01:39:18  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <openssl/ssl.h>
#include <openssl/engine.h>
#include <openssl/ui.h>
#include <openssl/err.h>
#include <openssl/rand.h>
#include <openssl/hmac.h>
#include <openssl/md5.h>
#include <netinet/in.h>
#include <string.h>
#include <inttypes.h>
#include <unistd.h>
#include "xsupconfig.h"
#include "profile.h"
#include "eap.h"
#include "tls_funcs.h"
#include "tls_crypt.h"
#include "xsup_debug.h"
#include "xsup_err.h"

#ifdef USE_EFENCE
#include <efence.h>
#endif

// In OpenSSL 0.9.8 we need to explicitly include the SHA header.
#ifndef SHA_DIGEST_LENGTH
#include <openssl/sha.h>
#endif

// If it *STILL* isn't around, then just define it.
#ifndef SHA_DIGEST_LENGTH
#define SHA_DIGEST_LENGTH 20
#endif

int engine_load_dynamic_opensc(struct smartcard *sc);
void set_smartcard_pin(char *pin);
UI_METHOD *UI_noninteractive(void);

char *get_cert_common_name(SSL *ssl_ctx)
{
  char *commonName = NULL;
  X509 *server_cert;

  if (!xsup_assert((ssl_ctx != NULL), "ssl_ctx != NULL", FALSE))
    return NULL;

  // Get our certificate.
  server_cert = SSL_get_peer_certificate(ssl_ctx);

  if (!server_cert) return NULL;

  commonName = (char *)malloc(512);
  if (commonName == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory to hold the common name!\n");
      return NULL;
    }

  if (X509_NAME_get_text_by_NID(X509_get_subject_name(server_cert),
				NID_commonName, commonName, 512) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't extract common name from server certificate!\n");
      return NULL;
    }

  debug_printf(DEBUG_AUTHTYPES, "Extracted common name of %s\n",commonName);
  return commonName;
}

void tls_funcs_process_error()
{
  unsigned long err;

  err = ERR_get_error();
  if (err != 0)
    {
      debug_printf(DEBUG_NORMAL, "OpenSSL Error -- %s\n", 
		   ERR_error_string(err, NULL));
      debug_printf(DEBUG_NORMAL, "Library  : %s\n", ERR_lib_error_string(err));
      debug_printf(DEBUG_NORMAL, "Function : %s\n", ERR_func_error_string(err));
      debug_printf(DEBUG_NORMAL, "Reason   : %s\n", ERR_reason_error_string(err));
    }
}

static int ssl_verify_callback(int preverify_ok, X509_STORE_CTX *ctx)
{
  char buf[256];
  X509 *err_cert;
  int err, depth;

  if (!xsup_assert((ctx != NULL), "ctx != NULL", FALSE))
    return XEMALLOC;

  err_cert = X509_STORE_CTX_get_current_cert(ctx);
  err = X509_STORE_CTX_get_error(ctx);
  depth = X509_STORE_CTX_get_error_depth(ctx);
  X509_NAME_oneline(X509_get_subject_name(err_cert), buf, 256);

  debug_printf(DEBUG_AUTHTYPES, "     --- SSL_verify : depth %d\n", depth);

  if (!preverify_ok)
    {
      debug_printf(DEBUG_AUTHTYPES, "     --- SSL_verify error : num=%d:%s:depth=%d:%s\n",
		   err, X509_verify_cert_error_string(err), depth, buf);

      if (err == 26) preverify_ok = 1;
    }

  return preverify_ok;
}

int tls_funcs_init(struct generic_eap_data *thisint)
{
  struct tls_vars *mytls_vars;
  struct config_eap_tls *userdata;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XETLSINIT;

  if (!xsup_assert((thisint->eap_data != NULL), "thisint->eap_data != NULL",
		   FALSE))
    return XETLSINIT;

  if (!xsup_assert((thisint->eap_conf_data != NULL),
		   "thisint->eap_conf_data != NULL", FALSE))
    return XETLSINIT;

  mytls_vars = (struct tls_vars *)thisint->eap_data;
  userdata = (struct config_eap_tls *)thisint->eap_conf_data;

  if (mytls_vars->ctx != NULL)
    {
      SSL_CTX_free(mytls_vars->ctx);
      mytls_vars->ctx = NULL;
      debug_printf(DEBUG_NORMAL, "mytls_vars->ctx != NULL!!!\n");
    }

  mytls_vars->ctx = SSL_CTX_new(TLSv1_method());
  if (mytls_vars->ctx == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't initialize OpenSSL TLS library!\n");
      tls_funcs_process_error();
      return XETLSINIT;
    }

  // XXX Clean up this code.
  if (thisint->eapNum == EAP_TYPE_TLS)
    {
      if ( userdata->sc.engine_id != NULL )
	{
	  if(!engine_load_dynamic_opensc(&userdata->sc))
	    {
	      debug_printf(DEBUG_NORMAL, "OpenSC Engine will be unavailable!\n");
	    }
	  
	  debug_printf(DEBUG_NORMAL, "Using Engine with ID \"%s\"\n", userdata->sc.engine_id);
	  
	  mytls_vars->engine = ENGINE_by_id(userdata->sc.engine_id);
	  if (!mytls_vars->engine)
	    {
	      debug_printf(DEBUG_NORMAL, "Engine not available!\n");
	      /* This error will be unrecoverable - XSupplicant should die */
	      return XETLSINIT;
	    }
	  
	  /* Now everything that can be done without having a smartcard plugged in
	   * is done. The next step is initializing the Engine. 
	   * This step should probably be done in some place where it can be 
	   * retried if it fails.
	   */
	  
	  if (!ENGINE_init(mytls_vars->engine))
	    {
	      debug_printf(DEBUG_NORMAL, "Can't initialize OpenSC Engine! "
			   "Is the smartcard plugged in?\n");
	      /* If we get this error there's probably no smartcard connected -
	       * we should be able to retry the call to ENGINE_init() */
	      return XETLSINIT;
	    }
	}
    }

  return XENONE;
}

int tls_funcs_build_new_session(struct tls_vars *mytls_vars)
{
  if (!xsup_assert((mytls_vars != NULL), "mytls_vars != NULL", FALSE))
    return XEMALLOC;

  if (mytls_vars->ssl)
    {
      SSL_free(mytls_vars->ssl);
      mytls_vars->ssl = NULL;
    }

  mytls_vars->ssl = SSL_new(mytls_vars->ctx);
  if (!mytls_vars->ssl)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't create SSL object!\n");
      tls_funcs_process_error();
      
      return XETLSSTARTFAIL;
    }

  return XENONE;
}
  
int tls_funcs_start(struct tls_vars *mytls_vars)
{
  SSL_SESSION *sess = NULL;
  unsigned long err;
  int counter, resval = XENONE;

  if (!xsup_assert((mytls_vars != NULL), "mytls_vars != NULL", FALSE))
    return XEMALLOC;

  mytls_vars->resuming = 0;

  if (!mytls_vars->ssl)
    {
      resval = tls_funcs_build_new_session(mytls_vars);
      if (resval != XENONE) 
	{
	  debug_printf(DEBUG_NORMAL, "Error building a new session!\n");
	  return resval;
	}
    } else {
      // We already established a connection, so we probably we need to
      // resume the session.
      if (mytls_vars->resume == RES_YES)
	{
	  sess = SSL_get_session(mytls_vars->ssl);
	  if (!sess)
	    {
	      debug_printf(DEBUG_AUTHTYPES, "Couldn't get session information!"
			   " We won't try to resume this session!\n");
	      mytls_vars->resuming = 0;

	      // Clear the old session data.
	      SSL_free(mytls_vars->ssl);

	      // Set up a new session.
	      resval = tls_funcs_build_new_session(mytls_vars);
	      if (resval != XENONE) return resval;
	    } else {
	      debug_printf(DEBUG_AUTHTYPES, "Got session information, trying "
			   "to resume session!\n");
	      mytls_vars->resuming = 1;

	      // We don't want to send an alert to the other end..  So do a 
	      // quiet shutdown.  This violates the TLS standard, but it is 
	      // needed to avoid confusing the other end of the connection 
	      // when we want to do a reconnect!
	      SSL_set_quiet_shutdown(mytls_vars->ssl, 1);
	      
	      // Now, close off our old session.
	      err = 0;
	      counter = 0;

	      SSL_shutdown(mytls_vars->ssl);

	      while ((err == 0) && (counter < 60))
		{
		  err = SSL_shutdown(mytls_vars->ssl);
		  if (err == 0)
		    {
		      sleep(1);
		      counter++;
		    }
		}

	      if (err < 0)
		{
		  debug_printf(DEBUG_NORMAL, "Error trying to shut down SSL "
			       "context data.\n");
		  tls_funcs_process_error();
		}
	    }
	}
    }

  mytls_vars->ssl_in = BIO_new(BIO_s_mem());
  if (!mytls_vars->ssl_in)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't create ssl_in!\n");
      tls_funcs_process_error();
      return XETLSSTARTFAIL;
    }

  if (BIO_reset(mytls_vars->ssl_in) < 1)
    {
      printf("Error : %s:%d\n", __FUNCTION__, __LINE__);
      tls_funcs_process_error();
    }

  mytls_vars->ssl_out = BIO_new(BIO_s_mem());
  if (!mytls_vars->ssl_out)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't create ssl_out!\n");
      tls_funcs_process_error();
      return XETLSSTARTFAIL;
    }

  if (BIO_reset(mytls_vars->ssl_out) < 0)
    {
      printf("Error : %s:%d\n", __FUNCTION__, __LINE__);
      tls_funcs_process_error();
    }

  SSL_set_bio(mytls_vars->ssl, mytls_vars->ssl_in, mytls_vars->ssl_out);

  if (sess != NULL)
    {
      // If we have session information, we need to use it to resume the 
      // session.
      debug_printf(DEBUG_AUTHTYPES, "Attempting to resume session...\n");
      if (SSL_set_session(mytls_vars->ssl, sess) <= 0)
	{
	  debug_printf(DEBUG_NORMAL, "There was an error attempting to resume "
		       "the session!\n");
	  tls_funcs_process_error();
	}
    }

  // Set this to SSL_VERIFY_NONE if we don't want to do anything with a failed
  // verification.

  SSL_set_verify(mytls_vars->ssl, mytls_vars->verify_mode, 
		 ssl_verify_callback);

  return XENONE;
}

int tls_funcs_parse(struct generic_eap_data *thisint, uint8_t *indata, 
		    int insize, char *outdata, int *outsize, int chunksize)
{
  int rc;
  BUF_MEM *retData;
  struct tls_vars *mytls_vars;
  char *retVal;
  uint32_t length;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  // indata can be NULL (at the start of a TLS conversation.

  if (!xsup_assert((outdata != NULL), "outdata != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((outsize != NULL), "outsize != NULL", FALSE))
    return XEMALLOC;

  if (insize > 1520)
    {
      debug_printf(DEBUG_NORMAL, "Packet passed in to tls_funcs_parse() is too big! Ignoring!\n");
      return XEBADPACKETSIZE;
    }

  if ((chunksize == 0) || (chunksize > 1398))
    {
      chunksize = 1398;
    }

  if (thisint->eap_data == NULL) 
    {
      debug_printf(DEBUG_NORMAL, "(TLS) eap_data has been destroyed, or not "
		   "allocated!\n");
      return XEMALLOC;
    }
  mytls_vars = (struct tls_vars *)thisint->eap_data;

  if (!mytls_vars)
    {
      debug_printf(DEBUG_NORMAL, "mytls_vars == NULL!\n");
      return XEMALLOC;
    }

  if (mytls_vars->tlsoutsize==0) 
    {
      if (indata != NULL)
	{
	  if (BIO_write(mytls_vars->ssl_in, indata, insize) < 2)
	    {
	      printf("Error : %s:%d\n", __FUNCTION__, __LINE__);
	      tls_funcs_process_error();
	    }
	} 

      if (BIO_reset(mytls_vars->ssl_out) < 0)
	{
	  printf("Error : %s:%d\n", __FUNCTION__, __LINE__);
	  tls_funcs_process_error();
	}

      if (mytls_vars->ssl == NULL) 
	{
	  debug_printf(DEBUG_NORMAL, "SSL context is NULL!!!!\n");
	  return XETLSNOCTX;
	}

      rc = SSL_connect(mytls_vars->ssl);
      if (rc < 0)
	{
	  tls_funcs_process_error();
	}

      BIO_get_mem_ptr(mytls_vars->ssl_out, &retData);
      
      mytls_vars->tlsoutdata = retData->data;
      mytls_vars->tlsoutsize = retData->length;
    }

  if (mytls_vars->tlsoutsize == 0) 
    {
      debug_printf(DEBUG_AUTHTYPES, "No data returned!\n");

      return XTLSNEEDDATA;
    }

  if ((mytls_vars->tlsoutsize - mytls_vars->tlsoutptr)>chunksize)
    {
      // Return a maximum sized chunk.
      
      if (mytls_vars->tlsoutptr == 0)  // This is our first chunk, include
	{                              // the length.
	  outdata[0] = EAPTLS_LENGTH_MORE;  // We will have a length value, and more.
	  length = htonl(mytls_vars->tlsoutsize);
	  memcpy(&outdata[1], &length, 4);
	  retVal = &outdata[5];
	  *outsize = chunksize+5; // To account for length.
	} else {
	  outdata[0] = EAPTLS_MORE_FRAGS;
	  retVal = &outdata[1];
	  *outsize = chunksize+1;
	}

      memcpy(retVal, &mytls_vars->tlsoutdata[mytls_vars->tlsoutptr], 
	     chunksize);
      mytls_vars->tlsoutptr += chunksize;

    } else {
      // Return what is left.

      if (mytls_vars->tlsoutptr == 0)  // This is our first chunk, include
	{                              // the length.
	  outdata[0] = EAPTLS_LENGTH_INCL;  // We will have a length value.
	  length = htonl(mytls_vars->tlsoutsize);
	  memcpy(&outdata[1], &length, 4);
	  retVal = &outdata[5];
	  *outsize = (mytls_vars->tlsoutsize - mytls_vars->tlsoutptr)+5;
	} else {
	  outdata[0] = EAPTLS_FINAL;
	  retVal = &outdata[1];
	  *outsize = (mytls_vars->tlsoutsize - mytls_vars->tlsoutptr)+1;
	}

      memcpy(retVal, &mytls_vars->tlsoutdata[mytls_vars->tlsoutptr], 
	     *outsize);
      
      // Clean out the data chunk.
      mytls_vars->tlsoutptr = 0;
      mytls_vars->tlsoutsize = 0;
    }

  return XENONE;
}

int tls_funcs_cn_check(struct tls_vars *mytls_vars)
{
  char *cnname = NULL;
  char *temp = NULL;

  if (!xsup_assert((mytls_vars != NULL), "mytls_vars != NULL", FALSE))
    return XEMALLOC;

  if (mytls_vars->cncheck != NULL)
    {
      cnname = get_cert_common_name(mytls_vars->ssl);

      debug_printf(DEBUG_AUTHTYPES, "Certificate CN : %s\n",cnname);

      // mytls_vars->cncheck == NULL, do nothing.
      debug_printf(DEBUG_AUTHTYPES, "Doing a CN Check!\n");

      if (mytls_vars->cnexact == 1)
	{
	  debug_printf(DEBUG_AUTHTYPES, "Looking for an exact match!\n");

	  if (cnname != NULL)
	    {
	      if (strcmp(mytls_vars->cncheck, cnname) != 0)
		{
		  debug_printf(DEBUG_AUTHTYPES, "Certificate CN didn't "
			       "match!\n");
		  free(cnname);
		  return XEBADCN;
		} else {
		  debug_printf(DEBUG_AUTHTYPES, "Certificate CN matched!\n");
		}
	    }
	} else {
	  debug_printf(DEBUG_AUTHTYPES, "Looking for a relative match!\n");

	  temp = mytls_vars->cncheck;
	  if (cnname != NULL)
	    {
	      if (strstr(cnname, temp) == NULL)
		{
		  debug_printf(DEBUG_AUTHTYPES, "Certificate CN didn't "
			       "match!\n");
		  free(cnname);
		  return XEBADCN;
		} else {
		  debug_printf(DEBUG_AUTHTYPES, "Certificate CN matched!\n");
		}
	    }
	}
    }
  if (cnname != NULL)
    {
      free(cnname);
      cnname = NULL;
    }

  return XENONE;
}

int tls_funcs_decode_packet(struct generic_eap_data *thisint, char *inframe, 
			    int insize, char *outframe, int *outsize,
			    phase2_call dophase2, int chunksize)
{
  unsigned long err;
  int rtnVal = XENONE, tlsindex;
  char *tlsptr;
  struct tls_vars *mytls_vars;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((inframe != NULL), "inframe != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((outframe != NULL), "outframe != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((outsize != NULL), "outsize != NULL", FALSE))
    return XEMALLOC;

  if (insize > 1520)
    {
      debug_printf(DEBUG_NORMAL, "Packet size too big in tls_funcs_decode_packet()!  Ignoring!\n");
      return XEBADPACKETSIZE;
    }

  debug_printf(DEBUG_AUTHTYPES, "Packet in (%d) :\n", insize);
  debug_hex_dump(DEBUG_AUTHTYPES, (uint8_t *) inframe, insize);

  // First, make sure we don't have any errors.
  err = ERR_get_error();
  if (err != 0)
    {
      debug_printf(DEBUG_NORMAL, "OpenSSL Error -- %s\n", 
		   ERR_error_string(err, NULL));
    }

  mytls_vars = (struct tls_vars *)thisint->eap_data;

  if (mytls_vars == NULL)
    {
      debug_printf(DEBUG_NORMAL, "EAP data is invalid in tls_funcs_decode_packet()!\n");
      return XEMALLOC;
    }

  *outsize = 0;

  // Set up a pointer to the start of the data.
  tlsindex = 1;
  tlsptr = &inframe[tlsindex];

  rtnVal = XENONE;

  // The first byte should tell us what to do.
  switch ((uint8_t)inframe[0])
    {
    case EAPTLS_START:
      if (tls_funcs_start(mytls_vars) != XENONE)
	{
	  debug_printf(DEBUG_NORMAL, "There was an error starting the TLS "
		       "handshake!\n");
	}

      if (mytls_vars->ssl == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "The SSL handle is invalid in tls_funcs_decode_packet()!\n");
	  return XETLSNOCTX;
	}
      
      rtnVal = tls_funcs_parse(thisint, NULL, 0, outframe, outsize, chunksize);
      if (rtnVal < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Failed to generate TLS data!\n");
	}
      break;

    case EAPTLS_LENGTH_MORE:
    case EAPTLS_LENGTH_INCL:
      // Skip the four octets that contain the length.  OpenSSL knows when
      // we are done.
      tlsptr+=4;
      tlsindex+=4;

      // DON'T BREAK HERE!  We want to do the next case!

    case EAPTLS_MORE_FRAGS:
    case EAPTLS_ACK:
      if (!mytls_vars) printf("mytls_vars is hosed!\n");
      if (!mytls_vars->ssl) printf("mytls_vars->ssl is hosed!\n");
      if ((SSL_is_init_finished(mytls_vars->ssl) != 0) && (dophase2 != NULL))
	{
	  // Handle the phase 2 piece.  We pass in the encrypted piece of
	  // the packet, and let phase 2 deal with it!

	  // But, before we do anything, verify the CN.
	  if (tls_funcs_cn_check(mytls_vars) != XENONE)
	    {
	      debug_printf(DEBUG_NORMAL, "Failed certificate common name "
			   "check!\n");
	      *outsize = 0;
	      return XEBADCN;
	    }

	  // We are in phase 2, so indicate it.
	  mytls_vars->phase = 2;
	  
	  if ((mytls_vars->resuming != 1) || (mytls_vars->quickResponse != TRUE))
	    {
	      rtnVal = (*dophase2)(thisint, (uint8_t *) tlsptr, 
				   (insize-tlsindex), outframe, outsize);
	      if (rtnVal != XINNERSUCCESS)
		{
		  if (rtnVal != XENONE)
		    {
		      debug_printf(DEBUG_NORMAL, "Phase 2 failure!\n");
		      return XEPHASE2FAILURE;
		    }
		}
	    } else {
	      if (*outsize == 0)
		{
		  debug_printf(DEBUG_AUTHTYPES, "Resumed session, ACKing ACK!\n");
		  tls_funcs_build_ack(outframe, outsize);
		  rtnVal = XENONE;	
		}
	    }	  
	} else {
	  rtnVal = tls_funcs_parse(thisint, (uint8_t *) tlsptr, (insize-tlsindex), outframe, outsize, chunksize);
	  if (rtnVal < 0)
	    {
	      debug_printf(DEBUG_NORMAL, "Couldn't parse TLS data.\n");
	    }
      	  if ((SSL_is_init_finished(mytls_vars->ssl) != 0) && 
	      (dophase2 != NULL) && 
	      (mytls_vars->quickResponse == TRUE))
	    {
	      
	      if (tls_funcs_cn_check(mytls_vars) != XENONE)
		{
		  debug_printf(DEBUG_NORMAL, "Failed certificate common "
			       "name check!\n");
		  *outsize = 0;
		  return XEBADCN;
		}

	      // We made it to phase 2.  So, indicate it.
	      mytls_vars->phase = 2;

	      if ((mytls_vars->resuming != 1) || (mytls_vars->quickResponse != TRUE))
		{
		  if ((*dophase2)(thisint, (uint8_t *) tlsptr, (insize-tlsindex), 
				  outframe, outsize) != XENONE)
		    {
		      debug_printf(DEBUG_NORMAL, "Phase 2 Failure.\n");
		      return XEPHASE2FAILURE;
		    }
		} else {
		  if (*outsize == 0)
		    {
		      debug_printf(DEBUG_AUTHTYPES, "Resumed session, ACKing ACK!\n");
		      tls_funcs_build_ack(outframe, outsize);
		      rtnVal = XENONE;
		    }
		}
	    } else if (rtnVal == XTLSNEEDDATA)
	      {
		tls_funcs_build_ack(outframe, outsize);
		rtnVal = XENONE;
	      } 
	}
      break;

    default:
      debug_printf(DEBUG_NORMAL, "Invalid TLS flags! (%02X)\n",(uint8_t)inframe[0]);
      rtnVal = XETLSBADFLAGS;
    }

  return rtnVal;
}

char *tls_funcs_gen_keyblock(struct generic_eap_data *thisint)
{
  struct tls_vars *mydata;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return NULL;

  mydata = (struct tls_vars *)thisint->eap_data;

  if (!xsup_assert((mydata != NULL), "mydata != NULL", FALSE))
    return NULL;

  return tls_crypt_gen_keyblock(thisint, mydata->sessionkeyconst,
				mydata->sessionkeylen);
}

int tls_funcs_build_ack(char *outframe, int *outsize)
{
  debug_printf(DEBUG_EVERYTHING, "Sending TLS ACK!\n");

  if (!xsup_assert((outframe != NULL), "outframe != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((outsize != NULL), "outsize != NULL", FALSE))
    return XEMALLOC;

  outframe[0] = 0x00;
  *outsize = 1;
  return XENONE;
}


static void ssl_info_callback(SSL *ssl, int w, int r)
{
  if (!xsup_assert((ssl != NULL), "ssl != NULL", FALSE))
    return;

  debug_printf(DEBUG_AUTHTYPES, "     --- SSL : %s\n", SSL_state_string_long(ssl));
  if (w & SSL_CB_ALERT)
    debug_printf(DEBUG_AUTHTYPES, "     --- ALERT : %s\n", SSL_alert_desc_string_long(r));
}


static int return_password(char *buf, int size, int rwflag, void *userdata)
{
  if (!xsup_assert((buf != NULL), "buf != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((userdata != NULL), "userdata != NULL", FALSE))
    return XEMALLOC;

  strncpy(buf, (char *)(userdata), size);
  buf[size-1] = '\0';
  return(strlen(buf));
}

int tls_funcs_load_root_certs(struct generic_eap_data *thisint, 
			      char *root_cert, char *root_dir, char *crl_dir)
{
  struct tls_vars *mytls_vars;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  mytls_vars = (struct tls_vars *)thisint->eap_data;

  if (!mytls_vars)
    {
      debug_printf(DEBUG_NORMAL, "Invalid EAP data was passed in to tls_funcs_load_root_certs()!\n");
      return XEMALLOC;
    }

  if ((!root_cert) && (!root_dir))
    {
      debug_printf(DEBUG_NORMAL, "Error loading cert!  Path to cert is NULL!\n");
      return XETLSCERTLOAD;
    } 

  if (mytls_vars->ctx == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid context in tls_funcs_load_root_certs()!\n");
      return XEMALLOC;
    }

  debug_printf(DEBUG_CONFIG, "Trying to load root certificate %s or "
	       "certificate directory %s\n", root_cert, root_dir);

  SSL_CTX_set_info_callback(mytls_vars->ctx, (void (*) ()) ssl_info_callback);
  
  if (SSL_CTX_load_verify_locations(mytls_vars->ctx, root_cert, root_dir) == 0)
    {
      debug_printf(DEBUG_NORMAL, "Failed to initialize path to root certificate!\n");
      tls_funcs_process_error();

      if(mytls_vars->ctx)
	{
	  SSL_CTX_free(mytls_vars->ctx);
	  mytls_vars->ctx = NULL;
	}
      return XETLSCERTLOAD;
    }

  debug_printf(DEBUG_CONFIG, "Loaded root certificate %s and directory %s\n",
		root_cert, root_dir);

  if (crl_dir) {
    if (SSL_CTX_load_verify_locations(mytls_vars->ctx, NULL, crl_dir) == 0)
      {
	debug_printf(DEBUG_NORMAL, "Failed to initalize path to CRLs!\n");
	tls_funcs_process_error();
	//debug_printf(DEBUG_NORMAL, "Error : %s\n", ERR_error_string(ERR_get_error(), NULL));
	if(mytls_vars->ctx)
	  {
	    SSL_CTX_free(mytls_vars->ctx);
	    mytls_vars->ctx = NULL;
	  }
	return XETLSCERTLOAD;
      }
  }
  

  /* Do we really want to pick up the default paths? */
  if (SSL_CTX_set_default_verify_paths(mytls_vars->ctx) == 0)
    {
      debug_printf(DEBUG_NORMAL, "Failed to initalize default paths for root certificates!\n");
      tls_funcs_process_error();

      if(mytls_vars->ctx)
	{
	  SSL_CTX_free(mytls_vars->ctx);
	  mytls_vars->ctx = NULL;
	}
      return XETLSCERTLOAD;
    }

  return XENONE;
}

int tls_funcs_load_random(struct generic_eap_data *thisint, char *random_file)
{
  char *default_random = "/dev/urandom", *file;
  struct tls_vars *mytls_vars; 

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((thisint->eap_data != NULL), "thisint->eap_data != NULL",
		   FALSE))
    return XEMALLOC;

  mytls_vars = thisint->eap_data;

  file = random_file == NULL ? default_random : random_file;

  if (RAND_load_file(file, 1024) < 0)
    {
      tls_funcs_process_error();
      if(mytls_vars->ctx)
	{
	  SSL_CTX_free(mytls_vars->ctx);
	  mytls_vars->ctx = NULL;
	}
	  
      debug_printf(DEBUG_NORMAL, "Couldn't load random data from %s\n", file);

      return -1;
    } 

  return XENONE;
}


int tls_funcs_load_user_cert(struct generic_eap_data *thisint, 
			     char *client_cert, char *key_file, char *password,
			     char *random_file)
{
  struct tls_vars *mytls_vars;
  struct config_eap_tls *userdata;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((client_cert != NULL), "client_cert != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((key_file != NULL), "key_file != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((password != NULL), "password != NULL", FALSE))
    return XEMALLOC;

  if (!thisint->eap_data)
    {
      debug_printf(DEBUG_NORMAL, "Invalid EAP data in tls_funcs_load_user_cert()!\n");
      return XEMALLOC;
    }

  mytls_vars = (struct tls_vars *)thisint->eap_data;
  userdata = (struct config_eap_tls *)thisint->eap_conf_data;

  SSL_CTX_set_default_passwd_cb_userdata(mytls_vars->ctx, password);
  SSL_CTX_set_default_passwd_cb(mytls_vars->ctx, return_password);

  if (SSL_CTX_use_certificate_file(mytls_vars->ctx, client_cert, 
				   SSL_FILETYPE_ASN1) != 1 &&
      SSL_CTX_use_certificate_file(mytls_vars->ctx, client_cert, 
				   SSL_FILETYPE_PEM) != 1 )
    {
      debug_printf(DEBUG_NORMAL, "Couldn't load client certificate data!\n");
      tls_funcs_process_error();
      if(mytls_vars->ctx)
	{
	  SSL_CTX_free(mytls_vars->ctx);
	  mytls_vars->ctx = NULL;
	}
      return XETLSCERTLOAD;
    }

  debug_printf(DEBUG_CONFIG, "Loading user Private Key from %s...\n", key_file);
  if (userdata->sc.engine_id)
    {
      EVP_PKEY *pkey;
      debug_printf(DEBUG_CONFIG, "Loading user Private Key with id %s from %s...\n", userdata->sc.key_id, userdata->sc.engine_id);
      set_smartcard_pin(password);
      pkey = ENGINE_load_private_key(mytls_vars->engine, userdata->sc.key_id,
		      UI_noninteractive(), NULL);
      SSL_CTX_use_PrivateKey(mytls_vars->ctx, pkey);
      //EVP_PKEY_free(pkey);
    }
  else if (SSL_CTX_use_PrivateKey_file(mytls_vars->ctx, key_file, 
				  SSL_FILETYPE_PEM) != 1 &&
           SSL_CTX_use_PrivateKey_file(mytls_vars->ctx, key_file, 
				  SSL_FILETYPE_ASN1) != 1) 
    {
      tls_funcs_process_error();
      if(mytls_vars->ctx)
	{
	  SSL_CTX_free(mytls_vars->ctx);
	  mytls_vars->ctx = NULL;
	}
      debug_printf(DEBUG_NORMAL, "Couldn't load client private key!\n");
      return XETLSCERTLOAD;
    }

  if (!SSL_CTX_check_private_key(mytls_vars->ctx))
    {
      debug_printf(DEBUG_NORMAL, "Private key isn't valid!\n");
      tls_funcs_process_error();
      return XETLSCERTLOAD;
    }

  SSL_CTX_set_options(mytls_vars->ctx, SSL_OP_NO_SSLv2 | SSL_OP_NO_SSLv3 |
		      SSL_OP_SINGLE_DH_USE);

  SSL_CTX_set_verify(mytls_vars->ctx, SSL_VERIFY_PEER | 
		     SSL_VERIFY_FAIL_IF_NO_PEER_CERT, NULL);

  if (tls_funcs_load_random(thisint, random_file))
    {
      return XETLSCERTLOAD;
    }
  debug_printf(DEBUG_CONFIG, "Loaded random data from %s...\n", random_file);

  // If we made it this far, our user cert is loaded, so indicate it.
  mytls_vars->cert_loaded = TRUE;

  return XENONE;
}

int tls_funcs_failed(struct generic_eap_data *thisint)
{
  struct tls_vars *mytls_vars;
  int res = 0;

  debug_printf(DEBUG_EVERYTHING, "(TLS-FUNCS) Cleaning up (possible after a failure)!\n");

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (!thisint->eap_data)
    {
      debug_printf(DEBUG_NORMAL, "Invalid EAP data in %s()!\n", __FUNCTION__);
      return XEMALLOC;
    }

  mytls_vars = (struct tls_vars *)thisint->eap_data;

  if (mytls_vars->ssl)
    {
      debug_printf(DEBUG_AUTHTYPES, "(EAP-TLS) Calling SSL_shutdown()\n");
      res = SSL_shutdown(mytls_vars->ssl);

      if (res == 0)
	{
	  /* Need to call it again to complete the shutdown. */
	  res = SSL_shutdown(mytls_vars->ssl);
	  
	  if (res == 0)
	    {
	      debug_printf(DEBUG_CONFIG, "Couldn't shut down SSL connection. "
			   "We will leak memory!\n");
	    }
	}
      SSL_free(mytls_vars->ssl);
      mytls_vars->ssl = NULL;
    }

  if (mytls_vars->ctx)
    {
      debug_printf(DEBUG_AUTHTYPES, "(EAP-TLS) Freeing mytls_vars->ctx!\n");
      SSL_CTX_free(mytls_vars->ctx);
      mytls_vars->ctx = NULL;
    }

  if (mytls_vars->sessionkeyconst != NULL)
    {
      debug_printf(DEBUG_AUTHTYPES, "(EAP-TLS) Freeing session key const!\n");
      free(mytls_vars->sessionkeyconst);
      mytls_vars->sessionkeyconst = NULL;
    }

  return XENONE;
}


int tls_funcs_cleanup(struct generic_eap_data *thisint)
{
  int err=XENONE;
  struct tls_vars *mytls_vars;

  debug_printf(DEBUG_EVERYTHING, "(TLS-FUNCS) Cleaning up!\n");

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((thisint->eap_data != NULL), "thisint->eap_data != NULL",
		   FALSE)) return XEMALLOC;

  mytls_vars = (struct tls_vars *)thisint->eap_data;

  err = tls_funcs_failed(thisint);

  if (mytls_vars->engine)
    {
      debug_printf(DEBUG_NORMAL, "Cleaning up OpenSSL Engine\n");
      ENGINE_finish(mytls_vars->engine);
      mytls_vars->engine = NULL;
      debug_printf(DEBUG_NORMAL, "Cleaning up OpenSSL Engine internal ressources\n");
      ENGINE_cleanup();
    }

  return err;
}

/* TLS PRF from rfc2246 pages 11-12 */
int
tls_funcs_PRF(uint8_t *secret, int secret_len, uint8_t *label, int label_len, 
	     uint8_t *seed, int seed_len, uint8_t *output, int outlen)
{
  int retVal = 0;
  int L_S1, L_S2;
  uint8_t *S1, *S2;
  uint8_t *P_MD5_buf, *P_SHA1_buf;
  uint8_t *P_seed;
  int P_seed_len;
  uint8_t A_MD5[MD5_DIGEST_LENGTH];
  uint8_t A_SHA1[SHA_DIGEST_LENGTH];
  int MD5_iterations, SHA1_iterations;
  int i, hashed_len;
  const EVP_MD *hash;
  HMAC_CTX ctx;

  if (!xsup_assert((secret != NULL), "secret != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((label != NULL), "label != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((seed != NULL), "seed != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((output != NULL), "output != NULL", FALSE))
    return XEMALLOC;

  /* determine the length of "half" the secret */
  if (secret_len % 2 == 0) {
    L_S1 = secret_len / 2;
  }
  else {
    L_S1 = secret_len / 2 + 1;
  }
  L_S2 = L_S1;
  S1 = secret; /* first L_S1 bytes of secret */
  S2 = secret + secret_len - L_S2;  /* last L_S2 bytes of secret */
  MD5_iterations = outlen / MD5_DIGEST_LENGTH;
  /* if there is anything left over, iterate 1 more time */
  MD5_iterations = outlen % MD5_DIGEST_LENGTH == 0 ? 
    MD5_iterations : MD5_iterations + 1;
  SHA1_iterations = outlen / SHA_DIGEST_LENGTH;
  SHA1_iterations = outlen % SHA_DIGEST_LENGTH == 0 ?
    SHA1_iterations : SHA1_iterations + 1;
  P_seed_len = label_len + seed_len;
  P_seed = (uint8_t *)malloc(sizeof(uint8_t) * P_seed_len);
  if (P_seed == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Error with malloc of P_seed in tls_funcs_PRF().\n");
      return XEMALLOC;
    }

  memcpy(P_seed, label, label_len);
  memcpy(P_seed+label_len, seed, seed_len);
  P_MD5_buf = (uint8_t *)malloc(sizeof(uint8_t) * 
			       MD5_iterations  * MD5_DIGEST_LENGTH);
  if (P_MD5_buf == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Error with malloc of P_MD5_buf in tls_funcs_PRF().\n");
      free(P_seed);
      P_seed = NULL;
      return XEMALLOC;
    }

  P_SHA1_buf = (uint8_t *)malloc(sizeof(uint8_t) *
				SHA1_iterations * SHA_DIGEST_LENGTH);
  if (P_SHA1_buf == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Error with malloc of P_SHA1_buf in tls_funcs_PRF().\n");
      free(P_seed);
      P_seed = NULL;
      free(P_MD5_buf);
      P_MD5_buf = NULL;
      return XEMALLOC;
    }

  /* P_MD5 */
  hash = EVP_md5();
  /* Initialize A_MD5 */
  HMAC(hash, S1, L_S1, P_seed, P_seed_len, A_MD5, (u_int *) &hashed_len);

  for (i = 0; i < MD5_iterations; i++) {
    HMAC_Init(&ctx, S1, L_S1, hash);
    HMAC_Update(&ctx, A_MD5, MD5_DIGEST_LENGTH);
    HMAC_Update(&ctx, P_seed, P_seed_len);
    HMAC_Final(&ctx, P_MD5_buf + i*(MD5_DIGEST_LENGTH), (u_int *) &hashed_len);
    HMAC_cleanup(&ctx);
    HMAC(hash, S1, L_S1, A_MD5, MD5_DIGEST_LENGTH,
	 A_MD5, (u_int *) &hashed_len);
  }
    

  /* do P_SHA1 */
  hash = EVP_sha1();
  /* Initialize A_SHA1 */
  HMAC(hash, S2, L_S2, P_seed, P_seed_len, A_SHA1, (u_int *) &hashed_len);

  for (i = 0; i < SHA1_iterations; i++) {
    HMAC_Init(&ctx, S2, L_S2, hash);
    HMAC_Update(&ctx, A_SHA1, SHA_DIGEST_LENGTH);
    HMAC_Update(&ctx, P_seed, P_seed_len);
    HMAC_Final(&ctx, P_SHA1_buf + i*(SHA_DIGEST_LENGTH), (u_int *) &hashed_len);
    HMAC_cleanup(&ctx);
    HMAC(hash, S2, L_S2, A_SHA1, SHA_DIGEST_LENGTH,
	 A_SHA1, (u_int *) &hashed_len);
  }
  /* XOR Them for the answer */
  for (i = 0; i < outlen; i++) {
    *(output + i) = P_MD5_buf[i] ^ P_SHA1_buf[i];
  }
  if (P_seed)
    {free(P_seed); P_seed = NULL;}
  if (P_MD5_buf) 
    {free(P_MD5_buf); P_MD5_buf = NULL;}
  if (P_SHA1_buf) 
    {free(P_SHA1_buf); P_SHA1_buf = NULL;}
  return retVal;
}

/* smartcard support */

#define OPENSC_ENGINE_SO_PATH "/usr/lib/opensc/engine_opensc.so"
#define OPENSC_ENGINE_ID      "opensc"

/* This function 
 * - loads OpenSSL's "dynamic" engine
 * - executes all the commands given in the pre array of strings
 *   These commands will usually load the shared object, do some 
 *   initialization and add the engine to OpenSSL's internal list of 
 *   Engines
 */
int engine_load_dynamic(char *pre[])
{
  char *engine_id = "dynamic";
  int rc;
  ENGINE *e;
  ENGINE_load_dynamic();
  e = ENGINE_by_id(engine_id);
  if(!e)
    {
      printf("can't find engine %s\n", engine_id);
      goto err;
    }
  while(pre && pre[0])
    {
      /*printf("\"%s\" \"%s\"\n", pre[0], pre[1]);*/
      rc = ENGINE_ctrl_cmd_string(e, pre[0], pre[1], 0);
      if(rc == 0)
        {
          printf("ctrl cmd_string failed: %s %s\n", 
					pre[0], pre[1]);
  	  goto err_pre;
        }
      pre += 2;
    }
  /* Free the reference to the "dynamic" engine
   * The OpenSC engine can still be looked up using 
   * ENGINE_by_id() */
  ENGINE_free(e);
  return 1;
err_pre:
  ENGINE_free(e);
err:
  ENGINE_cleanup();
  return 0;
}

/* This function
 *  - makes the opensc engine available to OpenSSL
 */
int engine_load_dynamic_opensc(struct smartcard *sc)
{
  if (xsup_assert((sc != NULL), "sc != NULL", FALSE))
    return XEMALLOC;

  debug_printf(DEBUG_NORMAL, "Loading opensc engine.\n");
  if(!sc->opensc_so_path)
    {
      /* use the default value */
      sc->opensc_so_path = OPENSC_ENGINE_SO_PATH;
    }
  char *pre_cmd[] = 
    {
      "SO_PATH", sc->opensc_so_path,
      "ID", OPENSC_ENGINE_ID,
      "LIST_ADD", "1",
      "LOAD", NULL,
      NULL, NULL
    };
  return engine_load_dynamic(pre_cmd);
}

/* provide a UI_METHOD that makes it possible to use a string as the
 * smartcard PIN */
char *smartcard_pin = NULL;

void set_smartcard_pin(char *pin)
{
  smartcard_pin = pin;
}

void unset_smartcard_pin()
{
  set_smartcard_pin(NULL);
}

int read_string(UI *ui, UI_STRING *uis)
{
  if(smartcard_pin)
    {
      UI_set_result(ui, uis, smartcard_pin);
      return 1;
    }
  return 0;
}

UI_METHOD *UI_noninteractive(void)
{
  UI_METHOD *ui_method;
  ui_method = UI_create_method("ui_noninteractive");
  UI_method_set_reader(ui_method, read_string);
  return ui_method;
}

