/*******************************************************************
 * The driver function for a Linux application layer EAPOL 
 * implementation
 * File: eapmd5.c
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: eapmd5.c,v 1.24 2006/06/01 22:49:50 galimorerpg Exp $
 * $Date: 2006/06/01 22:49:50 $
 * $Log: eapmd5.c,v $
 * Revision 1.24  2006/06/01 22:49:50  galimorerpg
 * Converted all instances of u_char to uint8_t
 * Fixed a bad #include in the generic frame handler.
 *
 * Revision 1.23  2006/05/29 04:17:58  chessing
 * Fixes for some memory leaks.
 *
 * Revision 1.22  2006/04/25 01:17:43  chessing
 * LOTS of code cleanups, new error checking/debugging code added, and other misc. fixes/changes.
 *
 * Revision 1.21  2005/10/17 03:56:54  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.20  2005/08/09 01:39:16  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <openssl/ssl.h>
#include <openssl/md5.h>
#include <string.h>
#include <strings.h>

#include "xsup_debug.h"
#include "xsup_err.h"
#include "frame_structs.h"
#include "profile.h"
#include "xsupconfig.h"   
#include "eap.h"
#include "eapmd5.h"

#ifdef USE_EFENCE
#include <efence.h>
#endif

#define MD5_LENGTH    0x10

/*****************************************************
 *
 * Setup to handle MD5 EAP requests
 *
 * This function is called each time we recieve a packet of the EAP type MD5.
 * At a minimum, it should check to make sure it's stub in the structure 
 * exists, and if not, set up any variables it may need.  Since MD5 doesn't
 * have any state that needs to survive successive calls, we don't need to 
 * do anything here.
 *
 *****************************************************/
int eapmd5_setup(struct generic_eap_data *thisint)
{
  // Do anything special that might be needed for this EAP type to work.
  debug_printf(DEBUG_EVERYTHING, "Initalized EAP-MD5!\n");

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  thisint->eap_data = (int *)malloc(sizeof(int));
  if (thisint->eap_data == NULL) return XEMALLOC;

  return XENONE;
}


/*****************************************************
 *
 * Process MD5 EAP Requests
 *
 ******************************************************/
int eapmd5_process(struct generic_eap_data *thisint, uint8_t *dataoffs, 
		   int insize, uint8_t *outframe, int *outsize)
{
  struct md5_values *md5data, *md5out;
  struct config_eap_md5 *userdata;
  uint8_t md5_result[16];
  uint8_t *tohash;
  char *username;
  int tohashlen;
  int *processReady;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((thisint->eap_data != NULL), "thisint->eap_data != NULL",
		   FALSE)) return XEMALLOC;

  if (!xsup_assert((dataoffs != NULL), "dataoffs != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((outframe != NULL), "outframe != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((outsize != NULL), "outsize != NULL", FALSE))
    return XEMALLOC;

  processReady = (int *)thisint->eap_data;
  userdata = (struct config_eap_md5 *)thisint->eap_conf_data;

  if (!xsup_assert((userdata != NULL), "userdata != NULL", FALSE))
    return XEMALLOC;

  debug_printf(DEBUG_EVERYTHING, "(EAP-MD5) Processing.\n");

  if ((thisint->tempPwd == NULL) && (userdata->password == NULL))
    {
      thisint->need_password = 1;
      thisint->eaptype = strdup("EAP-MD5");
      thisint->eapchallenge = NULL;

      *outsize = 0;
      return XENONE;
    }

  // Make sure we have something to process...
  if (dataoffs == NULL) return XENONE;

  if (userdata->username == NULL)
    {
      username = thisint->identity;
    } else {
      username = userdata->username;
    }

  if ((userdata->password == NULL) && (thisint->tempPwd != NULL))
    {
      userdata->password = thisint->tempPwd;
      thisint->tempPwd = NULL;
      /*      debug_printf(DEBUG_AUTHTYPES, "Passed in password : %s\n",
	      userdata->password);*/
    }

  // Actually process, and respond to challenges.
  md5data = (struct md5_values *)dataoffs;
  md5out = (struct md5_values *)outframe;
  
  if (md5data->length != MD5_LENGTH)
    {
      debug_printf(DEBUG_NORMAL, "(EAP-MD5) Incorrect length value for MD5 random value.\n");
      return XEMD5LEN;
    }

  tohashlen = (1+md5data->length+strlen(userdata->password));
  tohash = (uint8_t *)malloc(tohashlen);
  if (tohash == NULL)
    {
      debug_printf(DEBUG_NORMAL, "(EAP-MD5) Couldn't allocate memory for building hash source!\n");
      return XEMALLOC;
    }

  // Make sure we clean out the memory space.
  bzero(tohash, tohashlen);
  
  // Build the information we need to hash. Start with the EAP identifier.
  tohash[0] = thisint->eapid;

  // Then, we need the password.
  memcpy(&tohash[1], userdata->password, strlen(userdata->password));
  
  // Then the random value sent to us.
  memcpy(&tohash[1+strlen(userdata->password)], &md5data->randval, MD5_LENGTH);

  // Now, run it through the hash routine.
  MD5(tohash, tohashlen, &md5_result[0]);

  // We are done with tohash, so free it.
  free(tohash);

  // Set up our response frame.
  md5out->length = MD5_LENGTH;
  memcpy(&md5out->randval[0], &md5_result[0], MD5_LENGTH);

  memcpy(&outframe[sizeof(struct md5_values)], username, strlen(username));

  *outsize = (sizeof(struct md5_values)+strlen(username));

  return XENONE;
}

/*******************************************************
 *
 * Set our keys, if we can.
 *
 *******************************************************/
int eapmd5_get_keys(struct interface_data *thisint)
{
  // We don't key, so return -1.  (We return 0 if we set a key.)
  return -1;
}

/*******************************************************
 *
 * Clean up after ourselves.  This will get called when we get a packet that
 * needs to be processed requests a different EAP type.  It will also be 
 * called on termination of the program.
 *
 *******************************************************/
int eapmd5_cleanup(struct generic_eap_data *thisint)
{
  // Clean up after ourselves.

  debug_printf(DEBUG_AUTHTYPES, "(EAP-MD5) Cleaning up.\n");

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (thisint->eap_data != NULL)
    {
      free(thisint->eap_data);
      thisint->eap_data = NULL;
    }

  return XENONE;
}

/*******************************************************
 *
 * If we fail an authentication, we will call this routine.  It should clean
 * up anything that shouldn't live in to the next authentication attempt.
 *
 *******************************************************/
int eapmd5_failed(struct generic_eap_data *thisint)
{
  struct config_eap_md5 *userdata;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((thisint->eap_conf_data != NULL),
		   "thisint->eap_conf_data != NULL", FALSE))
    return XEMALLOC;

  userdata = (struct config_eap_md5 *)thisint->eap_conf_data;

  // If configure was passed the no password reset flag, we shouldn't do
  // anything!
#ifndef NO_PWD_RESET
  /*
  if (userdata->password != NULL)
    {
      free(userdata->password);
      userdata->password = NULL;
    }
  */
#endif

  return XENONE;
}
