/*******************************************************************
 * Implementation for handling SSID related config pieces.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: config_ssid.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: config_ssid.c,v 1.29 2006/06/01 22:49:49 galimorerpg Exp $
 * $Date: 2006/06/01 22:49:49 $
 * $Log: config_ssid.c,v $
 * Revision 1.29  2006/06/01 22:49:49  galimorerpg
 * Converted all instances of u_char to uint8_t
 * Fixed a bad #include in the generic frame handler.
 *
 * Revision 1.28  2006/05/26 23:21:12  chessing
 * Fixed some memory leaks.
 *
 * Revision 1.27  2006/05/26 22:04:58  chessing
 * Fixed some memory access errors, and cleaned up some wext stuff that was causing issues with the madwifi driver in wext mode.
 *
 * Revision 1.26  2006/05/23 23:38:39  chessing
 * Patch to allow the madwifi driver to associate and authenticate using the regular wireless extensions.  Should this start working, we are dumpping the madwifi specific driver crap! ;)
 *
 * Revision 1.25  2006/04/25 01:17:42  chessing
 * LOTS of code cleanups, new error checking/debugging code added, and other misc. fixes/changes.
 *
 * Revision 1.24  2006/03/02 04:52:41  chessing
 * Added the ability to load a config based on the hints that are included in an EAP request identity message.  Should consider having a way to override listening to these messages.
 *
 * Revision 1.23  2006/01/23 20:33:47  chessing
 * Added support for the replay counters in WPA/WPA2.  BSSID is now selected based on the signal quality information returned during the scan.  (Assuming signal quality information is returned during a scan. ;)
 *
 * Revision 1.22  2006/01/23 05:28:37  chessing
 * Fixed a few settings that were causing errors with IOCTLs on some cards.  Updated WPA2 code to properly process group key packets. We now record quality, signal level, and noise level from scan results, so that we can start to make better decisions on which AP to associate to.
 *
 * Revision 1.21  2006/01/19 22:23:36  chessing
 * Added the ability to associate to an AP without having wpa_group_cipher, or wpa_pairwise_cipher defined in the config file.  If they are defined, then the value in the config file will be used no matter what.  Also, changed a piece in xsup_driver.c so that we print the capabilities that the cards reports on startup.  (This should help debug some problems that are almost certain to pop up with this new code. ;)
 *
 * Revision 1.20  2005/12/18 07:44:12  chessing
 * Added the ability to associate using IWAUTH options instead of setting a full IE.  This allows NDISwrapper to work with vanilla wireless extensions.  For some reason, we can't parse IWEVASSOCREQIE/IWEVASSOCRESPIE since we see the length as 0, even though iwevent sees the correct IE information. :-/  Need to figure out why.
 *
 * Revision 1.19  2005/10/17 03:56:53  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.18  2005/10/14 02:26:17  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.17  2005/10/13 18:46:47  chessing
 * Fixed to the Madwifi driver to allow it to do dynamic WEP again.  Fixed up the wext driver to behave correctly again. ;)
 *
 * Revision 1.16  2005/09/19 21:45:46  chessing
 * Fix for the madwifi driver when it connects to certain APs to do WEP.  (Currently the only known one with this problem is the Trapeze APs when they are running MSS 4.x+.)
 *
 * Revision 1.15  2005/09/14 02:50:44  chessing
 * Major updates.  Auto association now works.  Started to rewrite the rtnetlink pieces using iwlib from the wireless tools to avoid compatibility issues.  As a result, getting the WPA and RSN IEs via the IWEVGENIE event no longer works for some reason, but the old style using IWEVCUSTOM events should still work like a champ.
 *
 * Revision 1.14  2005/09/08 16:27:01  chessing
 * Some small updates to the new state machine code.  First attempt at an auto association mode.  (It mostly works. ;)
 *
 * Revision 1.13  2005/09/05 01:00:34  chessing
 * Major overhaul to most of the state machines in Xsupplicant.  Also added additional error messages to the TLS functions to try to debug the one of the problems reported on the list.  Basic testing shows this new code to be more stable than previous code, but it needs more testing.
 *
 * Revision 1.12  2005/08/25 02:20:20  chessing
 * Some cleanup in xsup_debug.c, added the ability to wait for an interface to come up if it is down when Xsupplicant is started.  Roughed in the ability to choose between having Xsupplicant auto associate you, or allow you to set the ssid manually.  The stale key timer can now be set in the config file.  The association timeout can be set in the config file, and will also be used once the functionality is in place to attempt to guess the key settings needed for association, as well as the functionality to auto associate.
 *
 * Revision 1.11  2005/08/09 01:39:13  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 * Revision 1.10  2005/05/01 22:09:07  chessing
 * Small updates that are needed prior to the 1.2-pre1 release.
 *
 *******************************************************************/

#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include "xsup_debug.h"
#include "profile.h"
#include "xsupconfig.h"
#include "config_ssid.h"
#include "cardif/cardif.h"
#include "xsup_err.h"

#ifdef USE_EFENCE
#include <efence.h>
#endif

struct found_ssids *ssids=NULL, *active_ssid=NULL, *working = NULL;

/**************************************
 *
 * Determine which SSID structure is the better choice based on the signal
 * quality information that we get from a scan.  We want to rely first on 
 * the signal and noise measurements, and then on the quality measurement.
 *
 **************************************/
struct found_ssids *config_ssid_best_signal(struct found_ssids *one,
					    struct found_ssids *two)
{
  uint8_t one_lb, two_lb;

  if (!xsup_assert((one != NULL), "one != NULL", FALSE))
    return NULL;

  if (!xsup_assert((two != NULL), "two != NULL", FALSE))
    return NULL;

  if ((one->noise != 0) && (two->noise != 0))
    {
      one_lb = one->noise - one->signal;
      two_lb = two->noise - two->signal;
    }
  else if ((one->noise == 0) && (two->noise != 0))
    {
      return two;
    }
  else if ((one->noise == 0) && (two->noise == 0))
    {
      // We will have to rely on the quality value.
      if ((one->quality != 0) && (two->quality != 0))
	{
	  if (one->quality > two->quality)
	    {
	      return one;
	    }
	  else
	    {
	      return two;
	    }
	}
    }

  // If we get here, then we aren't sure which one is better.  So, return
  // nothing.
  return NULL;
}

/**************************************
 *
 * Init a new node in the ESSID list.
 *
 **************************************/
void config_ssid_init_new_node()
{
  if (!ssids)
    {
      // We don't have a first node allocated yet.
      ssids = (struct found_ssids *)malloc(sizeof(struct found_ssids));
      if (ssids == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "Insufficient memory at %s:%d!\n",
		       __FUNCTION__, __LINE__);
	  
	  // XXX Replace the _exit calls with a call to clean up before exiting.
	  _exit(1);
	}

      working = ssids;
    } else {
      
      working = ssids;

      while (working->next)
	{
	  working = working->next;
	}

      working->next = (struct found_ssids *)malloc(sizeof(struct found_ssids));
      if (!working->next)
	{
	  debug_printf(DEBUG_NORMAL, "Insufficient memory at %s:%d!\n",
		       __FUNCTION__, __LINE__);
	  _exit(1);
	}

      working = working->next;
    }

  // Initialize the new structure.
  bzero(working, sizeof(struct found_ssids));
}
 
/**************************************
 *
 * Add a newly discovered SSID.
 *
 **************************************/
void config_ssid_add_ssid_name(char *newssid)
{
  if ((!working) || (working->ssid_name))
    {
      debug_printf(DEBUG_CONFIG, "Found new ESSID, adding...\n");
      config_ssid_init_new_node();
    }

  // We need to make a copy.  We can't assume that the allocated memory
  // will always be there!
  working->ssid_name = strdup(newssid);
}

/**************************************
 *
 * Add a WPA IE to an SSID node.
 *
 **************************************/
void config_ssid_add_wpa_ie(uint8_t *wpa_ie, uint8_t len)
{
  if ((!working) || (working->wpa_ie))
    {
      debug_printf(DEBUG_CONFIG, "Found new ESSID block, adding...\n");
      config_ssid_init_new_node();
    }

  working->wpa_ie = (uint8_t *)malloc(len);
  if (!working->wpa_ie)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory to store the WPA "
		   "IE!  Xsupplicant will not be able to use this ESSID! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return;
    }

  memcpy(working->wpa_ie, wpa_ie, len);
  working->wpa_ie_len = len;
}

/**************************************
 *
 * Add an RSN IE to a SSID node.
 *
 **************************************/
void config_ssid_add_rsn_ie(uint8_t *rsn_ie, uint8_t len)
{
  if ((!working) || (working->rsn_ie))
    {
      debug_printf(DEBUG_CONFIG, "Found new ESSID block, adding...\n");
      config_ssid_init_new_node();
    }

  working->rsn_ie = (uint8_t *)malloc(len);
  if (!working->rsn_ie)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory to store the RSN "
		   "IE!  Xsupplicant will not be able to use this ESSID! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return;
    }

  memcpy(working->rsn_ie, rsn_ie, len);
  working->rsn_ie_len = len;
}

/**************************************
 *
 * Add the BSSID to the ESSID node.
 *
 **************************************/
void config_ssid_add_bssid(char *newmac)
{
  char empty_mac[6] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

  if (!xsup_assert((newmac != NULL), "newmac != NULL", FALSE))
    return;

  if (!working) 
    {
      config_ssid_init_new_node();
    }
  else
    {
      if (memcmp(working->mac, empty_mac, 6) != 0)
	{
	  debug_printf(DEBUG_CONFIG, "Found new ESSID block, adding...\n");
	  config_ssid_init_new_node();
	}
    }

  memcpy(working->mac, newmac, 6);
}

/**************************************
 *
 * Add flags to the ESSID node.
 *
 **************************************/
void config_ssid_update_abilities(uint8_t newabil)
{
  // If we get here, and don't have a valid structure, then we are doomed.
  xsup_assert((working != NULL), "working != NULL", TRUE);

  working->abilities |= newabil;
}

/**************************************
 *
 * Add the frequency to the ESSID node.
 *
 **************************************/
void config_ssid_add_freq(unsigned int newfreq)
{
  if ((!working) || (working->freq != 0))
    {
      debug_printf(DEBUG_CONFIG, "Found new ESSID block, adding...\n");
      config_ssid_init_new_node();
    }

  working->freq = newfreq;
}

/**************************************
 *
 * Add signal quality information to the ESSID node.
 *
 **************************************/
void config_ssid_add_qual(unsigned char qual, char signal, char noise)
{
  if ((!working) || (working->quality != 0) || (working->signal != 0) ||
      (working->noise != 0))
    {
      debug_printf(DEBUG_CONFIG, "Found new ESSID block, adding...\n");
      config_ssid_init_new_node();
    }

  working->quality = qual;
  working->signal = signal;
  working->noise = noise;
}

/**************************************
 *
 *  Dump all of the information about the SSID we are looking at.
 *
 **************************************/
void config_ssid_dump()
{
  struct found_ssids *cur;

  cur = ssids;

  debug_printf(DEBUG_EVERYTHING, "\n\n\nDumpping SSIDs:\n\n");

  while (cur != NULL)
    {
      if (cur == NULL)
	{
	  debug_printf(DEBUG_EVERYTHING, "NO VALID SSID INFORMATION AVAILABLE "
		       "TO DUMP!!\n");
	  return;
	}
      
      if (cur->ssid_name != NULL)
	{
	  debug_printf(DEBUG_EVERYTHING, "ESSID : %s\n", cur->ssid_name);
	  debug_printf(DEBUG_EVERYTHING, "Abilities : %02X\n", cur->abilities);
	  debug_printf(DEBUG_EVERYTHING, "Quality : %d     ", cur->quality);
	  if (cur->signal > 0)
	    {
	      debug_printf_nl(DEBUG_EVERYTHING, "Signal level : %d dBm",
			      cur->signal);
	    }

	  if (cur->noise > 0)
	    {
	      debug_printf_nl(DEBUG_EVERYTHING, "Noise level : %d dBm",
			      cur->noise);
	    }

	  debug_printf_nl(DEBUG_EVERYTHING, "\n");

	  if ((cur->wpa_ie != NULL) && ((cur->wpa_ie_len > 0) && (cur->wpa_ie_len < 255)))
	    {
	      debug_printf(DEBUG_EVERYTHING, "WPA IE (%d) : ", cur->wpa_ie_len);
	      debug_hex_printf(DEBUG_EVERYTHING, cur->wpa_ie, cur->wpa_ie_len);
	      debug_printf(DEBUG_EVERYTHING, "\n");
	    }
	  
	  if ((cur->rsn_ie != NULL) && ((cur->rsn_ie_len > 0) && (cur->rsn_ie_len < 255)))
	    {
	      debug_printf(DEBUG_EVERYTHING, "RSN IE (%d) : ", cur->rsn_ie_len);
	      debug_hex_printf(DEBUG_EVERYTHING, cur->rsn_ie, cur->rsn_ie_len);
	      debug_printf(DEBUG_EVERYTHING, "\n");
	    }
	}
      cur = cur->next;
    }
}

/**************************************
 *
 * Clear out the list of SSIDs that we know about.  This will usually be 
 * called at the time that a scan is started.  To allow for new data to
 * be compiled.
 *
 **************************************/
void config_ssid_clear()
{
  struct found_ssids *cur, *next;

  cur = ssids;

  while (cur != NULL)
    {
      if (cur->ssid_name != NULL)
	{
	  free(cur->ssid_name);
	  cur->ssid_name = NULL;
	}

      if (cur->wpa_ie != NULL)
	{
	  free(cur->wpa_ie);
	  cur->wpa_ie = NULL;
	}

      if (cur->rsn_ie != NULL)
	{
	  free(cur->rsn_ie);
	  cur->rsn_ie = NULL;
	}

      next = cur->next;
      free(cur);
      cur = next;
    }

  ssids = NULL;
  active_ssid = NULL;
  working = NULL;
}

/******************************************
 *
 * Search through our list of known SSIDs, and see if we know the one
 * specified in ssid_name.
 *
 ******************************************/
struct found_ssids *config_ssid_find_by_name(char *ssid_name)
{
  struct found_ssids *cur;

  if (!xsup_assert((ssid_name != NULL), "ssid_name != NULL", FALSE))
    return NULL;

  // Start at the top of the list.
  cur = ssids;

  while ((cur != NULL) && (strcmp(ssid_name, cur->ssid_name) != 0))
    {
      cur = cur->next;
    }

  if (cur != NULL)
    {
      debug_printf(DEBUG_CONFIG, "Found SSID with name of '%s'!\n",
		   cur->ssid_name);
      active_ssid = cur;
    }
  return cur;
}

/*******************************************
 *
 * Given an SSID name, set the pointer to the active SSID configuration.
 * (NOTE : If there is more than one AP visible, this won't necessarily find
 * the one that we are really connected to.  We mostly need this to have useful
 * information about the SSID, which should be the same between APs.)
 *
 *******************************************/
void config_ssid_set_active_ssid(char *ssid_name)
{
  struct found_ssids *cur;

  if (!xsup_assert((ssid_name != NULL), "ssid_name != NULL", FALSE))
    return;

  cur = config_ssid_find_by_name(ssid_name);
}

/******************************************
 *
 * See if the ESSID requested is one that we know about in our list.
 *
 ******************************************/
int config_ssid_ssid_known(char *ssid)
{
  if (!xsup_assert((ssid != NULL), "ssid != NULL", FALSE))
    return FALSE;

  if (config_ssid_find_by_name(ssid) != NULL)
    {
      return TRUE;
    } else {
      return FALSE;
    }
}

/******************************************
 *
 * Go through the list of SSIDs that we found in the scan, and see which one
 * is the "best" based on the priority.  If there are two SSIDs with the same
 * priority, then we will return the first one that was found.
 *
 ******************************************/
struct found_ssids *config_ssid_find_best_ssid()
{
  struct found_ssids *cur, *best = NULL, *temp = NULL;
  uint8_t best_pri = 0xff, cur_pri;
  
  // Start at the top of our list of SSIDs.
  cur = ssids;

  // We want to go through every single SSID we have in the list.
  while (cur != NULL)
    {
      cur_pri = config_get_network_priority(cur->ssid_name);
      debug_printf(DEBUG_CONFIG, "Checking %s with Priority %d\n",
		   cur->ssid_name, cur_pri);

      if ((best) && (strcmp(cur->ssid_name, best->ssid_name) == 0))
	{
	  if ((temp = config_ssid_best_signal(best, cur)) != NULL)
	    {
	      // We found the same SSID with a better signal, so select that
	      // one.
	      best = temp;
	    }
	  // Otherwise, we were unable to determine which one was better, so
	  // just use the first one that showed up.
	}
      else if (cur_pri < best_pri)
	{
	  best = cur;
	  best_pri = cur_pri;
	}

      cur = cur->next;
    }

  if (best)
    {
      debug_printf(DEBUG_CONFIG, "Best SSID appears to be '%s'\n",
		   best->ssid_name);
      debug_printf(DEBUG_CONFIG, "    Signal : %d   Noise : %d   Quality : "
		   "%d\n", best->signal, best->noise, best->quality);
    }
  return best;
}

/******************************************
 *
 * Return the SSID name for the SSID we want to connect to.
 *
 ******************************************/
char *config_ssid_get_desired_ssid(struct interface_data *ctx)
{
  struct config_globals *globals;
  char cur_ssid[99];

  if (!xsup_assert((ctx != NULL), "ctx != NULL", FALSE))
    return NULL;

  // We don't know anything about the current SSID, so see if
  // we are in auto associate mode, and if so, find the SSID that the user
  // has specified with the highest priority.
  globals = config_get_globals();

  if (!xsup_assert((globals != NULL), "globals != NULL", FALSE))
    return NULL;

  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_ASSOC_AUTO))
    {
      active_ssid = config_ssid_find_best_ssid();

      if (active_ssid == NULL) return NULL;

      return active_ssid->ssid_name;
    } else {
      bzero(cur_ssid, 99);

      // Get our current SSID
      if (cardif_GetSSID(ctx, cur_ssid) == XENONE)
	{
	  active_ssid = config_ssid_find_by_name(cur_ssid);
	  
	  if (active_ssid != NULL) 
	    {
	      debug_printf(DEBUG_NORMAL, "Returning SSID '%s'\n", 
			   active_ssid->ssid_name);
	      return active_ssid->ssid_name;
	    } else {
	      debug_printf(DEBUG_NORMAL, "You are in manual association mode"
			   ", and the current ESSID was not found.  Will"
			   " scan again.\n");
	    }
	}
    }

  return NULL;
}

/******************************************
 *
 * Return an unsigned char with bitflags that indicate what capabilities this
 * SSID has. (Supporting of WEP, WPA, 802.11I, etc.)
 *
 ******************************************/
uint8_t config_ssid_get_ssid_abilities()
{
  if (active_ssid == NULL) return 0x00;

  return active_ssid->abilities;
}

/******************************************
 *
 * Return if we are using WEP or not.
 *
 ******************************************/
int config_ssid_using_wep()
{
  int abil;

  abil = config_ssid_get_ssid_abilities();

  if ((abil & WPA_IE) || (abil & RSN_IE)) return FALSE;

  return TRUE;
}

/******************************************
 *
 * Return the wpa_ie and the wpa_ie_len for the selected SSID.
 *
 ******************************************/
void config_ssid_get_wpa_ie(uint8_t **wpa_ie, uint8_t *wpa_ie_len)
{
  if (!xsup_assert((wpa_ie != NULL), "wpa_ie != NULL", FALSE))
    return;

  if (active_ssid == NULL) 
    {
      *wpa_ie = NULL;
      wpa_ie_len = 0;
      return;
    }

  *wpa_ie = active_ssid->wpa_ie;
  *wpa_ie_len = active_ssid->wpa_ie_len;
}

/*******************************************************
 *
 * Return the rsn_ie and the rsn_ie_len for the selected SSID.
 *
 *******************************************************/
void config_ssid_get_rsn_ie(uint8_t **rsn_ie, uint8_t *rsn_ie_len)
{
  if (!xsup_assert((rsn_ie != NULL), "rsn_ie != NULL", FALSE))
    return;

  if (active_ssid == NULL)
    {
      *rsn_ie = NULL;
      rsn_ie_len = 0;
      return;
    }

  *rsn_ie = active_ssid->rsn_ie;
  *rsn_ie_len = active_ssid->rsn_ie_len;
}

/*******************************************************
 *
 * Return the frequency that this AP is on.
 *
 *******************************************************/
unsigned int config_ssid_get_freq()
{
  return ssids->freq;
}

/*******************************************************
 *
 * Return the MAC address of the AP for this SSID.
 *
 *******************************************************/
uint8_t *config_ssid_get_mac()
{
  if (ssids == NULL)
    {
      // This may not actually be a problem.  Rather, it could be that the
      // user wanted to use an SSID that we didn't find in a scan.
      debug_printf(DEBUG_INT, "Invalid SSID struct!  (%s:%d)\n", 
		   __FUNCTION__, __LINE__);
      return NULL;
    }

  return (uint8_t *)&active_ssid->mac;
}
