/*******************************************************************
 * Linux card interface implementation.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: cardif_linux.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: cardif_linux.c,v 1.143 2006/10/08 03:42:53 chessing Exp $
 * $Date: 2006/10/08 03:42:53 $
 * $Log: cardif_linux.c,v $
 * Revision 1.143  2006/10/08 03:42:53  chessing
 * Last batch of changes before 1.2.8.
 *
 * Revision 1.142  2006/08/28 04:31:54  chessing
 * A couple of fixes.
 *
 * Revision 1.141  2006/08/25 23:37:18  chessing
 * Numerous patches that have come in over the last month or two.
 *
 * Revision 1.140  2006/06/23 00:45:19  chessing
 * Completed basic implementation of operstate support.  Probably need to consider extending it to be used to populate the up/down state of the interface in the statemachine.
 *
 * Revision 1.139  2006/06/22 23:24:21  chessing
 * Added more operstate support.  It now works with the dhcpclient from http://www.flamewarmaster.de/software/dhcpclient/
 *
 * Revision 1.138  2006/06/22 04:33:10  chessing
 * More bits added for the operstate pieces.
 *
 * Revision 1.137  2006/06/20 18:17:18  chessing
 * Fix some endianness problems with WPA and WPA2 IE parsing.
 *
 * Revision 1.136  2006/06/01 22:49:49  galimorerpg
 * Converted all instances of u_char to uint8_t
 * Fixed a bad #include in the generic frame handler.
 *
 * Revision 1.135  2006/05/29 20:31:00  chessing
 * Removed madwifi specific driver code.  (Wext works with the madwifi driver now.)
 *
 * Revision 1.134  2006/05/28 22:32:21  chessing
 * More fixes to make madwifi behave itself.
 *
 * Revision 1.133  2006/05/26 22:04:58  chessing
 * Fixed some memory access errors, and cleaned up some wext stuff that was causing issues with the madwifi driver in wext mode.
 *
 * Revision 1.132  2006/05/23 23:07:01  chessing
 * Cleaned up the previous patch for timer_tick.  (There was a cleaner way of doing it.)  Fixed up some stuff that was causing issues getting associated/dynamic WEP authenticated with the madwifi driver.  Madwifi seems to be happily working (with dynamic WEP) now.
 *
 * Revision 1.131  2006/05/23 00:53:08  chessing
 * Code cleanup.
 *
 * Revision 1.130  2006/05/22 22:29:17  chessing
 * Compiler warnings patches from Eric Evans.  (And one from me. ;)
 *
 * Revision 1.129  2006/05/17 02:40:07  chessing
 * Finished several things on the todo list.  Removed external event core stuff, added gui pings, clear keys and WPA IE when an SSID is set from outside Xsupplicant, etc.
 *
 * Revision 1.128  2006/05/14 22:09:28  chessing
 * A few small changes in the Xsupplicant code.  Beginning of a configuration/monitor program can be found in gui_tools. (Very simple for now, with several known bugs, but no showstopper bugs know.)
 *
 * Revision 1.127  2006/05/13 05:56:44  chessing
 * Removed last pieces of code that relied on SIGALRM.  Active scan timeout is now configurable so that people that wish to hammer on their cards now have the option to do that. ;)
 *
 * Revision 1.126  2006/05/13 03:44:46  chessing
 * Huge patch as a result of the downtime with SourceForge.  Fixed issues with WPA/WPA2.  Fixed a segfault when a scan returned a NULL.  Changed event handleing to use select() instead of older sleep/non-blocking socket resulting in a faster authentication.  Fixed a stack smashing bug that would cause random failures with WPA/WPA2.  Added status messages so that when we run in non-debug mode we have some idea of what is going on.  Various other code cleanups, bug fixes.
 *
 * Revision 1.125  2006/04/25 01:17:43  chessing
 * LOTS of code cleanups, new error checking/debugging code added, and other misc. fixes/changes.
 *
 * Revision 1.124  2006/04/16 21:20:42  chessing
 * Added initial TNC support via patch from Mike McCauley.  Requires libtnc in order to work, which isn't yet available to the public.
 *
 * Revision 1.123  2006/03/21 18:22:10  chessing
 * Fixes to EAP-AKA code.  Changed a few defaults to disable Xsupplicant based roaming, and passive scanning for this release.  (It would currently cause more problems than it would solve.
 *
 * Revision 1.122  2006/02/23 22:26:53  chessing
 * Fix for bug id #1415020.  'Building Xsupplicant 1.2.3 Fails on FC4'.
 *
 * Revision 1.121  2006/01/24 04:42:26  chessing
 * A few more fixes to WPA code, along with a fix to the scan reaping code.
 *
 * Revision 1.120  2006/01/23 05:28:37  chessing
 * Fixed a few settings that were causing errors with IOCTLs on some cards.  Updated WPA2 code to properly process group key packets. We now record quality, signal level, and noise level from scan results, so that we can start to make better decisions on which AP to associate to.
 *
 * Revision 1.119  2006/01/19 05:37:04  chessing
 * WPA2 is working correctly.  Added the ability to query the card to gather encryption/authentication capabilities.  1.2.3 is now ready to go.
 *
 * Revision 1.118  2006/01/12 17:54:03  chessing
 * WPA almost works again on cards that use the GENIE setting.  Added a small fix to allow Xsupplicant to build on Open SuSE 10.
 *
 * Revision 1.117  2005/12/24 04:51:51  chessing
 * Removed an unneeded file.  Updated scanning code to parse IWEVGENIE events correctly, and make use of them.
 *
 * Revision 1.116  2005/12/24 02:39:48  chessing
 * Fixes to autoconf script to correctly identify the number of arguments that iw_extract_event_stream() takes, along with checking iwlib.h compiles correctly.
 *
 * Revision 1.115  2005/12/03 22:18:51  chessing
 * Added an include to fix some problems when compiling with WE19.
 *
 * Revision 1.114  2005/11/30 00:42:29  chessing
 * Cleaned up some warnings during the build process.
 *
 * Revision 1.113  2005/11/22 19:49:42  chessing
 * When the desired interface isn't available when Xsupplicant starts, Xsupplicant will wait for it to show up before starting to authenticate.  In this case, the interface isn't 'available' when an ifconfig doesn't show that the interface exists.  It has *NOTHING* to do with the up/down state of the interface.  This is mostly useful for situations where a cardbus wireless card is plugged in before linux is booted.
 *
 * Revision 1.112  2005/11/14 17:27:39  chessing
 * Removed the driver specific code for hostap, ipw, and ndiswrapper, since they all support WE18 now.
 *
 * Revision 1.111  2005/11/10 04:56:54  chessing
 * Added patch from Ben Gardner to add support for setting a specific WEP key prior to attempting to associte.  (With a few slight modifications by me to make it fit in the current CVS code, and get it supported in config-parse.)  Added patch from Pekka Savola to fix some header ordering issues, and a potential buffer overflow.
 *
 * Revision 1.110  2005/10/17 03:56:54  chessing
 * Updates to the libxsupconfig library.  It no longer relies on other source from the main tree, so it can be used safely in other code with problems.
 *
 * Revision 1.109  2005/10/14 02:26:18  shaftoe
 * - cleanup gcc 4 warnings
 * - (re)add support for a pid in the form of /var/run/xsupplicant.<iface>.pid
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.108  2005/09/25 15:03:36  shaftoe
 * Document syslog functionality in example config file.
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.107  2005/09/25 00:39:48  shaftoe
 * - More work to pretty up syslog output.
 *
 * -- Eric Evans <eevans@sym-link.com>
 *
 * Revision 1.106  2005/09/19 21:48:05  chessing
 * Added a line that will indicate that the interface is scanning when running in normal (non-debug) mode.
 *
 * Revision 1.105  2005/09/19 21:45:46  chessing
 * Fix for the madwifi driver when it connects to certain APs to do WEP.  (Currently the only known one with this problem is the Trapeze APs when they are running MSS 4.x+.)
 *
 * Revision 1.104  2005/09/14 03:34:54  chessing
 * Small cosmetic changes.  Default association mode is now auto instead of manual. Fixes for bug IDs #1290449 & #1290323.
 *
 * Revision 1.103  2005/09/14 02:50:44  chessing
 * Major updates.  Auto association now works.  Started to rewrite the rtnetlink pieces using iwlib from the wireless tools to avoid compatibility issues.  As a result, getting the WPA and RSN IEs via the IWEVGENIE event no longer works for some reason, but the old style using IWEVCUSTOM events should still work like a champ.
 *
 * Revision 1.102  2005/09/08 16:27:01  chessing
 * Some small updates to the new state machine code.  First attempt at an auto association mode.  (It mostly works. ;)
 *
 * Revision 1.101  2005/09/05 01:00:37  chessing
 * Major overhaul to most of the state machines in Xsupplicant.  Also added additional error messages to the TLS functions to try to debug the one of the problems reported on the list.  Basic testing shows this new code to be more stable than previous code, but it needs more testing.
 *
 * Revision 1.100  2005/08/25 03:34:05  chessing
 * Removed a bunch of functions from config.c that could be handled better in other ways.
 *
 * Revision 1.99  2005/08/25 02:20:20  chessing
 * Some cleanup in xsup_debug.c, added the ability to wait for an interface to come up if it is down when Xsupplicant is started.  Roughed in the ability to choose between having Xsupplicant auto associate you, or allow you to set the ssid manually.  The stale key timer can now be set in the config file.  The association timeout can be set in the config file, and will also be used once the functionality is in place to attempt to guess the key settings needed for association, as well as the functionality to auto associate.
 *
 * Revision 1.98  2005/08/20 19:06:54  chessing
 * Patch from Carsten Grohmann to fix a few things in xsup_get_state.c.  Also added the ability to define an empty network clause, that will set the card in to encryption disabled mode.  From there, anything short of changing the SSID will be ignored by Xsupplicant.
 *
 * Revision 1.97  2005/08/09 01:39:15  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#ifdef LINUX_FRAMER

// We want to use kernel headers to build.
#define HEADERS_KERNEL

#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/compiler.h>
#include <iwlib.h>
#include <linux/if_packet.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <linux/rtnetlink.h>
#include <linux/wireless.h>

#include "xsupconfig.h"
#include "profile.h"
#include "config_ssid.h"
#include "cardif/linux/cardif_linux_wext.h"
#include "cardif/cardif.h"
#include "xsup_common.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "snmp.h"
#include "statemachine.h"
#include "wireless_sm.h"
#include "cardif/linux/cardif_linux.h"
#include "cardif/linux/cardif_atmel_driver.h"
#include "cardif/linux/cardif_linux_rtnetlink.h"
#include "timer.h"

#ifdef USE_EFENCE
#include <efence.h>
#endif

#ifndef ETH_P_EAPOL
#define ETH_P_EAPOL 0x888e
#endif

// Define this, so the compiler doesn't complain.
extern unsigned int if_nametoindex(const char *);

// This contains a pointer to the functions needed for wireless.  
struct cardif_funcs *wireless;

// Store values about what state our interface was in before we start messing
// with it.
struct int_starting_data *startup;

/********************************************************************
 *
 * Clear all keys, and accept unencrypted traffic again.
 *
 ********************************************************************/
void cardif_linux_clear_keys(struct interface_data *ctx)
{
  int i;

  if (!xsup_assert((ctx != NULL), "ctx != NULL", FALSE))
    return;

  debug_printf(DEBUG_INT, "Clearing keys!\n");
  
  // Clear the PTK.
  cardif_delete_key(ctx, 0, 1);

  for (i=0;i<4;i++)
    {
      cardif_delete_key(ctx, i, 0);
    }

  // Accept unencrypted frames again.
  cardif_drop_unencrypted(ctx, 0);
}

/***********************************************
 *
 * Determine if we are currently associated. 
 *
 ***********************************************/
int cardif_check_associated(struct interface_data *intdata)
{
  char newmac[6], curbssid[6];

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  // If we are wired, this function doesn't do anything useful.
  if (!TEST_FLAG(intdata->flags, IS_WIRELESS)) return XENONE;

  cardif_GetBSSID(intdata, curbssid);

  memset(newmac, 0x00, 6);
  if (memcmp(newmac, curbssid, 6) == 0)
    {
      return IS_UNASSOCIATED;
    }

  memset(newmac, 0x44, 6);
  if (memcmp(newmac, curbssid, 6) == 0)
    {
      return IS_UNASSOCIATED;
    }

  memset(newmac, 0xFF, 6);
  if (memcmp(newmac, curbssid, 6) == 0)
    {
      return IS_UNASSOCIATED;
    }

  //Otherwise, we are associated.
  return IS_ASSOCIATED;
}

/***********************************************
 *
 * Set up the wireless cardif_funcs structure to the driver that the user
 * has requested.
 *
 ***********************************************/
void cardif_set_driver(char driver)
{
  switch (driver)
    {
    case DRIVER_NONE:
      wireless = NULL;
      break;

    case DRIVER_WEXT:
      wireless = &cardif_linux_wext_driver;
      break;

    case DRIVER_ATMEL:
      wireless = &cardif_atmel_driver;
      break;
    }
}

/***********************************************
 *
 * Do whatever is needed to get the interface in to a state that we can send
 * and recieve frames on the network.  Any information that we need to later
 * use should be stored in the interface_data structure.
 *
 ***********************************************/
int cardif_init(struct interface_data *thisint, char driver)
{
  struct ifreq ifr;
  struct lin_sock_data *sockData;
  int retval;
  struct config_globals *globals;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  // Get the information about the global settings from the config file.
  globals = config_get_globals();

  if (!xsup_assert((globals != NULL), "globals != NULL", FALSE))
    return XEGENERROR;

  debug_printf(DEBUG_INT, "Initializing socket for interface %s..\n",
	       thisint->intName);

  // Keep track of which driver we were assigned.
  thisint->driver_in_use = driver;

  // Find out what the interface index is.
  thisint->intIndex = if_nametoindex(thisint->intName);
  if (thisint->intIndex < 0)
    {
      debug_printf(DEBUG_NORMAL, "Invalid value for interface index! (%d)\n",
		   thisint->intIndex);
      return XENOSOCK;
    }

  // Allocate memory for the things we need.
  thisint->sockData = (void *)malloc(sizeof(struct lin_sock_data));
  if (thisint->sockData == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Error allocating memory!\n");
      return XEMALLOC;
    }

  sockData = thisint->sockData;

  // Establish a socket handle.
  sockData->sockInt = socket(PF_PACKET, SOCK_RAW, htons(ETH_P_EAPOL));
  if (sockData->sockInt < 0)
    {
      debug_printf(DEBUG_NORMAL, 
		   "Couldn't initialize raw socket for interface %s!\n",
		   thisint->intName);
      return XENOSOCK;
    }        

  // Build our link layer socket struct, so we can bind it to a specific
  // interface.
  sockData->sll.sll_family = PF_PACKET;
  sockData->sll.sll_ifindex = thisint->intIndex;
  sockData->sll.sll_protocol = htons(ETH_P_EAPOL);

  // Bind to the interface.
  retval = bind(sockData->sockInt, (const struct sockaddr *)&sockData->sll, 
		sizeof(struct sockaddr_ll));
  if (retval < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error binding raw socket to interface %s!\n",
		   thisint->intName);
      return XESOCKOP;
    }

  memset(&ifr, 0x00, sizeof(ifr));
  ifr.ifr_ifindex = thisint->intIndex;

  // Tell the ifreq struct which interface we want to use.
  Strncpy((char *)&ifr.ifr_name, thisint->intName, sizeof(ifr.ifr_name));

  // Get our MAC address.  (Needed for sending frames out correctly.)
  retval = ioctl(sockData->sockInt, SIOCGIFHWADDR, &ifr);
  if (retval < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error getting hardware (MAC) address for interface %s!\n",
		   thisint->intName);
      return XENOTINT;
    }

  // Store a copy of our source MAC for later use.
  memcpy((char *)&thisint->source_mac[0], (char *)&ifr.ifr_hwaddr.sa_data[0], 6);

  // Check if we want ALLMULTI mode, and enable it.
  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_ALLMULTI))
    {
      // Tell the ifreq struct which interface we want to use.
      strncpy((char *)&ifr.ifr_name, thisint->intName, sizeof(ifr.ifr_name));

      if (ioctl(sockData->sockInt, SIOCGIFFLAGS, &ifr) < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't determine if ALLMULTI is enabled!\n");
	} else {
	  if (ifr.ifr_flags & IFF_ALLMULTI)
	    {
	      debug_printf(DEBUG_INT, "Allmulti mode is already enabled on this device!\n");
	      thisint->flags |= ALLMULTI;
	    } else {
	      debug_printf(DEBUG_INT, "Allmulti is currently disabled on this device!\n");
	      thisint->flags &= ~ALLMULTI;
	    }
	}

      ifr.ifr_flags |= IFF_ALLMULTI;
      if (ioctl(sockData->sockInt, SIOCSIFFLAGS, &ifr) < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't set ALLMULTI mode on this interface!  We will continue anyway!\n");
	}
    }

  // Set up wireless card drivers.
  cardif_set_driver(driver);

  if (cardif_int_is_wireless(thisint->intName) == TRUE)
    {
      // If we have our destination set to AUTO, then preset our destination
      // address.
      if (globals->destination == DEST_AUTO)
	{
	  cardif_GetBSSID(thisint, thisint->dest_mac);
	}
    }

  // Initialize our event handler.
  cardif_linux_rtnetlink_init(thisint);
	  
  return XENONE;
}

/**************************************************************
 *
 * Tell the wireless card to start scanning for wireless networks.
 *
 **************************************************************/
int cardif_do_wireless_scan(struct interface_data *thisint, char passive)
{
  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (wireless == NULL) 
    {
      debug_printf(DEBUG_INT, "No valid wireless calls struct! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return -1;
    }

  // If we are already scanning, then we shouldn't get here, but go ahead 
  // and ignore it anyway.
  if (TEST_FLAG(thisint->flags, SCANNING) )
    {
      debug_printf(DEBUG_INT, "Got a request to start a new scan when one is"
		   " already in progress!  Ignoring!\n");
      return XENONE;
    }

  SET_FLAG(thisint->flags, SCANNING);
  config_ssid_clear();

  return wireless->scan(thisint, passive);
}

/**************************************************************
 *
 * Send a disassociate message.
 *
 **************************************************************/
int cardif_disassociate(struct interface_data *thisint, int reason_code)
{
  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (wireless == NULL) return -1;

  debug_printf(DEBUG_INT, "Called %s\n", __FUNCTION__);
  return wireless->disassociate(thisint, reason_code);
}

/**************************************************************
 *
 * Check to see if the BSSID value is valid.  If it is, return TRUE. If
 * it isn't return FALSE.
 *
 **************************************************************/
int cardif_valid_dest(struct interface_data *thisint)
{
  char baddest[6];

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if ((thisint->flags & IS_WIRELESS))
    {
      memset((char *)&baddest, 0x00, 6);
      if (memcmp(thisint->dest_mac, baddest, 6) == 0)
	{
	  debug_printf(DEBUG_INT, "All 0s for dest mac!\n");
	  return FALSE;

	}

      memset((char *)&baddest, 0x44, 6);
      if (memcmp(thisint->dest_mac, baddest, 6) == 0)
	{
	  debug_printf(DEBUG_INT, "All 4s for dest mac!\n");
	  return FALSE;
	}

      memset((char *)&baddest, 0xff, 6);
      if (memcmp(thisint->dest_mac, baddest, 6) == 0)
	{
	  debug_printf(DEBUG_INT, "All Fs for dest mac!\n");
	  return FALSE;
	}
    }  
  return TRUE;
}

/******************************************
 *
 * Return the socket number for functions that need it.
 *
 ******************************************/
int cardif_get_socket(struct interface_data *thisint)
{
  struct lin_sock_data *sockData;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  sockData = thisint->sockData;

  return sockData->sockInt;
}

/******************************************
 *
 * Clean up anything that was created during the initialization and operation
 * of the interface.  This will be called before the program terminates.
 *
 ******************************************/
int cardif_deinit(struct interface_data *thisint)
{
  struct ifreq ifr;
  uint16_t int16;
  struct lin_sock_data *sockData;
  uint8_t all0s[6] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;
  
  sockData = thisint->sockData;

  debug_printf(DEBUG_EVERYTHING, "Cleaning up interface %s...\n",thisint->intName);

  cardif_linux_rtnetlink_cleanup(thisint);

  if (TEST_FLAG(thisint->flags, IS_WIRELESS))
  {
    // Remove all of the keys that we have set.
    cardif_linux_clear_keys(thisint);

    debug_printf(DEBUG_INT, "Turning off WPA support/state.\n");

    // Clear the WPA IE.
    cardif_linux_wext_set_wpa_ie(thisint, NULL, 0);

    cardif_disable_wpa_state(thisint);

    // Reset the MAC address to all 0s.  (This tells the driver to
    // scan for new associations.
    cardif_setBSSID(thisint, all0s);
  }

  // Check if we want ALLMULTI mode, and enable it.
  if (TEST_FLAG(thisint->flags, ALLMULTI))
    {
      memset(&ifr, 0x00, sizeof(ifr));

      // Tell the ifreq struct which interface we want to use.
      strncpy((char *)&ifr.ifr_name, thisint->intName, sizeof(ifr.ifr_name));

      if (ioctl(sockData->sockInt, SIOCGIFFLAGS, &ifr) < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't get interface flags!\n");
	} else {
	  // Check if allmulti was disabled when we started.  If it was,
	  // then disable it again, so everything is good.
	  if (!(thisint->flags & ALLMULTI))
	    {
	      debug_printf(DEBUG_INT, "Turning off ALLMULTI mode!\n");

	      int16 = ifr.ifr_flags;

	      // ANDing the flags with 0xfdff will turn off the ALLMULTI flag.
	      ifr.ifr_flags = (int16 & 0xfdff);
	      if (ioctl(sockData->sockInt, SIOCSIFFLAGS, &ifr) < 0)
		{
		  debug_printf(DEBUG_NORMAL, "Couldn't set ALLMULTI mode on this interface!  We will continue anyway!\n");
		}
	    }
	}
    }

  close(sockData->sockInt);

  // Now clean up the memory.
  if (thisint->sockData != NULL)
    {
      free(thisint->sockData);
      thisint->sockData = NULL;
    }

  return XENONE;
}

/******************************************
 *
 * Set a WEP key.  Also, based on the index, we may change the transmit
 * key.
 *
 ******************************************/
int cardif_set_wep_key(struct interface_data *thisint, uint8_t *key, 
		       int keylen, int index)
{
  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((key != NULL), "key != NULL", FALSE))
    return XEMALLOC;

  if (wireless == NULL) return -1;

  return wireless->set_wep_key(thisint, key, keylen, index);
}

/**********************************************************
 *
 * Set a TKIP key. 
 *
 **********************************************************/
int cardif_set_tkip_key(struct interface_data *thisint, char *addr, 
			      int keyidx, int settx, char *key, int keylen)
{
  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (wireless == NULL) return -1;

  return wireless->set_tkip_key(thisint, (uint8_t *) addr, keyidx, settx, 
				key, keylen);
}

/**********************************************************
 *
 * Set a CCMP (AES) key
 *
 **********************************************************/
int cardif_set_ccmp_key(struct interface_data *thisint, char *addr, int keyidx,
			int settx, char *key, int keylen)
{
  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (wireless == NULL) return -1;

  return wireless->set_ccmp_key(thisint, (uint8_t *) addr, keyidx, settx,
				key, keylen);
}

/**********************************************************
 *
 * Delete a key
 *
 **********************************************************/
int cardif_delete_key(struct interface_data *intdata, int key_idx, int set_tx)
{
  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  if (wireless == NULL) return -1;

  return wireless->delete_key(intdata, key_idx, set_tx);
}

/******************************************
 *
 * Do whatever we need to do in order to associate based on the flags in
 * the ssids_list struct.
 *
 ******************************************/
void cardif_associate(struct interface_data *intdata)
{
  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return;

  wireless->associate(intdata);
}

/******************************************
 *
 * Ask the wireless card for the ESSID that we are currently connected to.  If
 * this is not a wireless card, or the information is not available, we should
 * return an error.
 *
 ******************************************/
int cardif_GetSSID(struct interface_data *thisint, char *ssid_name)
{
  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((ssid_name != NULL), "ssid_name != NULL", FALSE))
    return XEMALLOC;

  if (wireless == NULL) 
    {
      debug_printf(DEBUG_NORMAL, "No valid call to get SSID for this driver!"
		   "\n");
      return -1;
    }

  if ((thisint == NULL) || (ssid_name == NULL)) 
  {
    debug_printf(DEBUG_INT, "NULL value passed to %s!\n", __FUNCTION__);
    return -1;
  }

  return wireless->get_ssid(thisint, ssid_name);
}

/******************************************
 *
 * Get the Broadcast SSID (MAC address) of the Access Point we are connected 
 * to.  If this is not a wireless card, or the information is not available,
 * we should return an error.
 *
 ******************************************/
int cardif_GetBSSID(struct interface_data *thisint, char *bssid_dest)
{
  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((bssid_dest != NULL), "bssid_dest != NULL", FALSE))
    return XEMALLOC;

  if (wireless == NULL) return -1;

  if (thisint == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface data structure passed to %s!\n", __FUNCTION__);
      return -1;
    }

  if (bssid_dest == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid bssid_dest in %s!\n", __FUNCTION__);
      return -1;
    }

  return wireless->get_bssid(thisint, bssid_dest);
}

/******************************************
 *
 * Set the flag in the state machine that indicates if this interface is up
 * or down.  If there isn't an interface, we should return an error.
 *
 ******************************************/
int cardif_get_if_state(struct interface_data *thisint)
{
  int retVal;
  struct ifreq ifr;
  struct lin_sock_data *sockData;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  sockData = thisint->sockData;

  if (!xsup_assert((sockData != NULL), "sockData != NULL", FALSE))
    return XEMALLOC;

  memset(&ifr, 0x00, sizeof(ifr));
  strncpy(ifr.ifr_name, thisint->intName, sizeof(ifr.ifr_name));
  retVal = ioctl(sockData->sockInt, SIOCGIFFLAGS, &ifr);
  if (retVal < 0)
    {
      debug_printf(DEBUG_NORMAL, "Interface %s not found!\n", thisint->intName);
      return FALSE;
    }
  
  if ((ifr.ifr_flags & IFF_UP) == IFF_UP)
    {
      return TRUE;
    } else {
      SET_FLAG(thisint->flags, WAS_DOWN);
      return FALSE;
    }
  return XENONE;
}

/******************************************
 *
 * Send a frame out of the network card interface.  If there isn't an 
 * interface, we should return an error.  We should return a different error
 * if we have a problem sending the frame.
 *
 ******************************************/
int cardif_sendframe(struct interface_data *thisint)
{
  char nomac[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
  int retval;
  struct lin_sock_data *sockData;
  struct config_network *network_data;
  uint16_t pad;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  sockData = thisint->sockData;

  if (!xsup_assert((sockData != NULL), "sockData != NULL", FALSE))
    return XEMALLOC;

  if (thisint->send_size == 0) 
    {
      debug_printf(DEBUG_INT, "%s:%d -- Nothing to send!\n",
		   __FUNCTION__, __LINE__);
      return XENONE;
    }

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return XEBADCONFIG;
    }

  // The frame we are handed in shouldn't have a src/dest, so put it in.
  memcpy(&thisint->sendframe[0], &thisint->dest_mac[0], 6);
  memcpy(&thisint->sendframe[6], &thisint->source_mac[0], 6);

  if (memcmp(nomac, (char *)&network_data->dest_mac[0], 6) != 0)
    {
      debug_printf(DEBUG_INT, "Static MAC address defined!  Using it!\n");
      memcpy(&thisint->sendframe[0], &network_data->dest_mac[0], 6);
    }

  // Make sure the frame is large enough.
  if ((!TEST_FLAG(thisint->flags, IS_WIRELESS)) && (thisint->send_size < 64))
    {
      pad = 64 - thisint->send_size;
      debug_printf(DEBUG_INT, "Padding frame to 64 bytes by adding %d byte"
                   "(s).\n", pad);
      memset(&thisint->sendframe[thisint->send_size+1], 0x00, pad);
      thisint->send_size += pad;
    }

  debug_printf(DEBUG_EVERYTHING, "Frame to be sent (%d) : \n",
	       thisint->send_size);
  debug_hex_dump(DEBUG_EVERYTHING, thisint->sendframe, thisint->send_size);

  snmp_dot1xSuppEapolFramesTx();

  retval = sendto(sockData->sockInt, thisint->sendframe, thisint->send_size, 0,
		  NULL, 0);
  if (retval <= 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't send frame! (%s)\n", strerror(errno));
    }

  thisint->send_size = 0;
  
  // Clear out the receive buffer so we don't accidently try to process it
  // again.
  bzero(thisint->recvframe, 1520);
  thisint->recv_size = 0;

  return retval;
}

/******************************************
 * 
 * Get a frame from the network.  Make sure to check the frame, to determine 
 * if it is something we care about, and act accordingly.
 *
 ******************************************/
int cardif_getframe(struct interface_data *thisint)
{
  int newsize=0;
  char dot1x_default_dest[6] = {0x01, 0x80, 0xc2, 0x00, 0x00, 0x03};
  struct lin_sock_data *sockData;
  uint8_t resultframe[1520];
  int resultsize;
  struct config_globals *globals;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  globals = config_get_globals();

  if (!xsup_assert((globals != NULL), "globals != NULL", FALSE))
    return XEMALLOC;

  sockData = thisint->sockData;

  if (!xsup_assert((sockData != NULL), "sockData != NULL", FALSE))
    return XEMALLOC;

  errno = 0;
  resultsize = 1520; /* overflows resultframe if too large, or should we increase resultframe instead? */

  newsize = recvfrom(sockData->sockInt, resultframe, resultsize, 0, 0, 0);
  if (newsize <= 0)
    {
      if ((errno != EAGAIN) && (errno != ENETDOWN))
	{
	  debug_printf(DEBUG_NORMAL, "Error (%d) : %s  (%s:%d)\n", errno,
		       strerror(errno), __FUNCTION__, __LINE__);
	}
      return XENOFRAMES;
    } else {
      debug_printf(DEBUG_EVERYTHING, "Got Frame (%d) : \n", newsize);
      debug_hex_dump(DEBUG_EVERYTHING, resultframe, newsize);
    }

  snmp_dot1xSuppEapolFramesRx();

  // Make sure that the frame we got is for us..
  if ((memcmp(&thisint->source_mac[0], &resultframe[0], 6) == 0) ||
      ((memcmp(&resultframe[0], &dot1x_default_dest[0], 6) == 0) &&
       (memcmp(&resultframe[6], &thisint->source_mac[0], 6) != 0)))
    {
      // Since we now know this frame is for us, record the address it
      // came from.
      snmp_dot1xSuppLastEapolFrameSource((uint8_t *)&resultframe[6]);

      resultsize = newsize;

      switch (globals->destination)
	{
	case DEST_AUTO:
	  // If it is a wired interface, only change the destination if
	  // the recieved frame destination isn't the multicast address.
	  if (!TEST_FLAG(thisint->flags, IS_WIRELESS))
	    {
	      if (memcmp(&resultframe[0], dot1x_default_dest, 6) == 0)
		{
		  break;
		}
	      // Otherwise, fall through.
	    }
	case DEST_SOURCE:
	  if (memcmp(thisint->dest_mac, &resultframe[6], 6) != 0)
	    {
	      debug_printf(DEBUG_INT, "Changing destination mac to source.\n");
	    }
	  memcpy(thisint->dest_mac, &resultframe[6], 6);
	  break;

	case DEST_MULTICAST:
	  memcpy(thisint->dest_mac, dot1x_default_dest, 6);
	  break;

	case DEST_BSSID:
	  cardif_GetBSSID(thisint, thisint->dest_mac);
	  break;

	default:
	  debug_printf(DEBUG_NORMAL, "Unknown destination mode!\n");
	  break;
	}

      thisint->recv_size = newsize;

      memcpy(thisint->recvframe, resultframe, newsize);
      return newsize;
    }

  // Otherwise it isn't for us. 
  debug_printf(DEBUG_INT, "Got a frame, not for us.\n");
  return XENOFRAMES;
}

/**************************************************************
 *
 * Set the state needed to associate to a WPA enabled AP, and actually
 * do a WPA authentication.
 *
 **************************************************************/
int cardif_enable_wpa_state(struct interface_data *thisint)
{
  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (wireless == NULL) return -1;

  debug_printf(DEBUG_INT, "WPA: Enabling WPA state on interface %s.\n",thisint->intName);

  return wireless->wpa_state(thisint, TRUE);
}

/**************************************************************
 *
 * Clear the state needed to associate to a WPA enabled AP, and actually
 * do a WPA authentication.
 *
 **************************************************************/
int cardif_disable_wpa_state(struct interface_data *thisint)
{
  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (wireless == NULL) return -1;

  return wireless->wpa_state(thisint, FALSE);
}

/**************************************************************
 *
 * Enable WPA (if it is supported.)
 *
 **************************************************************/
int cardif_enable_wpa(struct interface_data *thisint)
{
  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (wireless == NULL) return -1;

  debug_printf(DEBUG_INT, "WPA: Enabling WPA on interface %s.\n",thisint->intName);

  return wireless->wpa(thisint, TRUE);
}

/**************************************************************
 *
 * Call this when we roam to a different AP, or disassociate from an AP.
 *
 **************************************************************/
int cardif_wep_associate(struct interface_data *thisint, int zeros)
{
  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  if (wireless == NULL) return -1;

  if (!config_ssid_using_wep())
    {
      debug_printf(DEBUG_INT, "Doing WPA/WPA2 mode! Not "
		   "setting/unsetting keys.\n");
      return 0;
    }

  return wireless->wep_associate(thisint, zeros); 
}

/******************************************
 * 
 * Return true if there is a frame in the queue to be processed.
 *
 ******************************************/
int cardif_frameavail(struct interface_data *thisint)
{
  int newsize=0;
  char resultframe[1520];
  struct lin_sock_data *sockData;

  if (!xsup_assert((thisint != NULL), "thisint != NULL", FALSE))
    return XEMALLOC;

  sockData = thisint->sockData;

  if (!xsup_assert((sockData != NULL), "sockData != NULL", FALSE))
    return XEMALLOC;

  newsize = recvfrom(sockData->sockInt, &resultframe, 1520, MSG_PEEK, 0, 0);
  if (newsize > 0) return TRUE;

  return FALSE;
}

/******************************************
 *
 * Validate an interface, based on if it has a MAC address.
 *
 ******************************************/
int cardif_validate(char *interface)
{
  int sd, res;
  struct ifreq ifr;

  if (!xsup_assert((interface != NULL), "interface != NULL", FALSE))
    return XEMALLOC;

  memset(&ifr, 0x00, sizeof(ifr));
  strncpy(ifr.ifr_name, interface, sizeof(interface)+1);

  sd = socket(PF_PACKET, SOCK_RAW, 0);
  if (sd < 0)
    return FALSE;
  res = ioctl(sd, SIOCGIFHWADDR, &ifr);
  close(sd);
  if (res < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't get information for interface %s!\n",interface);
    } else {
      switch (ifr.ifr_hwaddr.sa_family)
	{
	case ARPHRD_ETHER:
	case ARPHRD_IEEE80211:
	  return TRUE;
	}
    }
  return FALSE;
}

/******************************************
 *
 * (en)/(dis)able countermeasures on this interface.
 *
 ******************************************/
int cardif_countermeasures(struct interface_data *intdata, char endis)
{
  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  if (wireless == NULL) return -1;

  return wireless->countermeasures(intdata, endis);
}

/******************************************
 *
 * (en)/(dis)able receiving of unencrypted frames on this interface.
 *
 ******************************************/
int cardif_drop_unencrypted(struct interface_data *intdata, char endis)
{
  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  if (wireless == NULL) return -1;

  if (config_ssid_using_wep()) return XENONE;
  
  return wireless->drop_unencrypted(intdata, endis);
}

/******************************************
 *
 * Get the name of an interface, based on an index value.
 *
 ******************************************/
#define PROC_DEV_FILE  "/proc/net/dev"

int cardif_get_int(int index, char *retInterface)
{
  FILE *fp;
  int hits;
  char line[1000], *lineptr;

  if (!xsup_assert((retInterface != NULL), "retInterface != NULL", FALSE))
    return XEMALLOC;

  hits = 0;

  fp = fopen(PROC_DEV_FILE, "r");
  if (fp == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't access /proc/net/dev!\n");
      exit(250);
    }

  bzero(line, 1000);

  while ((hits <= index) && (fgets(line, 999, fp) != NULL))
    { 
      lineptr = strchr(line, ':');
      
      if (lineptr == NULL) continue;

      *lineptr = '\0';
      lineptr = &line[0];

      while (*lineptr == ' ') lineptr++;  // Strip out blanks.
      
      strcpy(retInterface, lineptr);
      hits++;
    }

  if (hits <= index)
    {
      debug_printf(DEBUG_INT, "No more interfaces to look at!\n");
      return XNOMOREINTS;
    }

  debug_printf(DEBUG_INT, "Found interface : %s\n",retInterface);

  fclose(fp);

  return XENONE;   // No errors.
}


/*******************************************************
 *
 * Check to see if an interface is wireless.  On linux, we look in
 * /proc/net/wireless to see if the interface is registered with the
 * wireless extensions.
 *
 *******************************************************/
#define PROC_WIRELESS_FILE  "/proc/net/wireless"

int cardif_int_is_wireless(char *interface)
{
  FILE *fp;
  char line[1000], *lineptr=NULL;
  int done;

  if (!xsup_assert((interface != NULL), "interface != NULL", FALSE))
    return XEMALLOC;

  done = FALSE;

  fp = fopen(PROC_WIRELESS_FILE, "r");
  if (fp == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't access /proc/net/wireless!  (You probably don't have wireless extensions enabled!)\n");
      return -1;
    }

  bzero(line, 1000);

  while ((!done) && (fgets(line, 999, fp) != NULL))
    { 
      lineptr = strchr(line, ':');
      
      if (lineptr != NULL)
	{
	  
	  *lineptr = '\0';
	  lineptr = &line[0];
	  
	  while (*lineptr == ' ') lineptr++;  // Strip out blanks.
	  if (lineptr != NULL)
	    {
	      if (strcmp(lineptr, interface) == 0) done=TRUE;
	    }
	}
    }
  fclose(fp);
  
  if ((lineptr != NULL) && (strcmp(lineptr, interface) == 0))
    {
      debug_printf(DEBUG_INT, "Interface %s is wireless!\n",interface);
      return TRUE;
    } else {
      debug_printf(DEBUG_INT, "Interface %s is NOT wireless!\n",interface);
      return FALSE;
    }
  return XENONE;   // No errors.
}

int cardif_get_wpa_ie(struct interface_data *intdata, char *iedata, int *ielen)
{
  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((iedata != NULL), "iedata != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((ielen != NULL), "ielen != NULL", FALSE))
    return XEMALLOC;

  return wireless->get_wpa_ie(intdata, iedata, ielen);
}

int cardif_get_wpa2_ie(struct interface_data *intdata, char *iedata, int *ielen)
{
  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((iedata != NULL), "iedata != NULL", FALSE))
    return XEMALLOC;

  if (!xsup_assert((ielen != NULL), "ielen != NULL", FALSE))
    return XEMALLOC;

  if (!wireless)
    {
      debug_printf(DEBUG_NORMAL, "Invalid wireless function pointers.\n");
      return XEMALLOC;
    }

  if (iedata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid bucket for IE data! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  if (!wireless->get_wpa2_ie)
    {
      debug_printf(DEBUG_NORMAL, "No valid function to get WPA2 IE!\n");
      return XEMALLOC;
    }

  return wireless->get_wpa2_ie(intdata, iedata, ielen);
}

/**************************************************************
 *
 * This function should clear out all keys that have been applied to the card.
 * It should be indepentant of the type (WEP/TKIP/CCMP) of key that was
 * applied.
 *
 **************************************************************/
int cardif_clear_keys(struct interface_data *intdata)
{
  int retVal = 0, i;

  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  // Clear the TX key.
  retVal = cardif_delete_key(intdata, 0, 1);
  if (retVal != XENONE) 
    {
      debug_printf(DEBUG_NORMAL, "Error clearing default TX key!\n");
      return retVal;
    }

  // Now, clear out the broadcast/multicast/group key(s).
  for (i=0;i<4;i++)
    {
      retVal = cardif_delete_key(intdata, i, 0);
      if (retVal != XENONE)
	{
	  debug_printf(DEBUG_NORMAL, "Error clearing key %d!\n", i);
	  return retVal;
	}
    }

  return XENONE;
}

/*******************************************************************
 *
 * Attempt to reassociate to the network we were previously connected to.
 *
 *******************************************************************/
void cardif_reassociate(struct interface_data *intiface, uint8_t reason)
{
  if (!xsup_assert((intiface != NULL), "intiface != NULL", FALSE))
    return;

  if (!config_ssid_using_wep())
    {
      debug_printf(DEBUG_NORMAL, "SSID '%s' is WPA/WPA2 capable. WPA/WPA2 is "
		   "enabled on this connection.\n", intiface->cur_essid);
      
      // Since we are doing WPA/WPA2, we need to disassociate from 
      // the network, and reassociate with WPA/WPA2 set up.
      cardif_enable_wpa(intiface);
      cardif_enable_wpa_state(intiface);

      cardif_clear_keys(intiface);
    }

  cardif_associate(intiface);  
}

/*****************************************************************
 *
 * Disable encryption on the card.  (Set the interface to open mode.)
 *
 *****************************************************************/
int cardif_enc_disable(struct interface_data *intdata)
{
  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return XEMALLOC;

  return wireless->enc_disable(intdata);
}

/******************************************************************
 *
 * Determine what abilities this card has.  (WPA, WPA2, TKIP, CCMP, WEP40, 
 * etc.)
 *
 ******************************************************************/
void cardif_get_abilities(struct interface_data *intdata)
{
  if (!xsup_assert((intdata != NULL), "intdata != NULL", FALSE))
    return;

  if (!wireless->enc_capabilities)
    {
      intdata->enc_capa = 0;
      return;
    }

  wireless->enc_capabilities(intdata);
}

/******************************************************************
 *
 * Change the BSSID that we are currently connected to.
 *
 ******************************************************************/
void cardif_setBSSID(struct interface_data *ctx, uint8_t *new_bssid)
{
  if (!xsup_assert((ctx != NULL), "ctx != NULL", FALSE))
    return;

  if (!xsup_assert((new_bssid != NULL), "new_bssid != NULL", FALSE))
    return;

  if (!wireless->setbssid)
    {
      return;
    }

  wireless->setbssid(ctx, new_bssid);
}

/*************************************************************************
 *
 * Change the operstate of the interface.
 *
 *************************************************************************/
void cardif_operstate(struct interface_data *ctx, uint8_t newstate)
{
  if (!xsup_assert((ctx != NULL), "ctx != NULL", FALSE))
    return;

  if (!wireless->set_operstate)
    {
      debug_printf(DEBUG_INT, "No function defined to set operstate. (This "
		   "is probably nothing to worry about.\n");
      return;
    }

  wireless->set_operstate(ctx, newstate);
}

/******************************************************************
 *
 * Wait for an interface to be "attached" to the system before starting to
 * attempt authentication with it.  This is a blocking call that should
 * *ONLY* return when the interface is available.  (Note : 'available'
 * does not mean that the interface is UP.  The 802.1X state machine will
 * deal with the interface if it is down.  We just need to wait for an
 * interface to exist so that we can use it.
 *
 ******************************************************************/
void cardif_wait_for_int(char *intname)
{
  int idx = -1;

  if (!xsup_assert((intname != NULL), "intname != NULL", FALSE))
    return;

  idx = if_nametoindex(intname);
  if (idx < 1)
    {
      debug_printf(DEBUG_NORMAL, "Waiting for interface to be inserted, or "
		   "driver to be loaded.\n");
      while (if_nametoindex(intname) < 1)
	{
	  sleep(1);
	}
    }
}

/********************************************************************
 *
 * The passive scan timer expried.  So, we need to issue a scan request,
 * and reset our timer to recheck the scan results periodically.
 *
 ********************************************************************/
void cardif_passive_scan_timeout(struct interface_data *ctx)
{
  struct config_globals *globals;
  uint8_t *mac;
  char *ssid;

  if (!xsup_assert((ctx != NULL), "ctx != NULL", FALSE))
    return;

#warning FINISH!  We get scan data results, but we still need to do something with them.

  if (!TEST_FLAG(ctx->flags, PASV_SCANNING))
    {
     if (!TEST_FLAG(ctx->flags, SCANNING))
	{
	  timer_reset_timer_count(PASSIVE_SCAN_TIMER, 5);
	  cardif_do_wireless_scan(ctx, 1);
	  SET_FLAG(ctx->flags, PASV_SCANNING);
	}
      else
	{
	  debug_printf(DEBUG_NORMAL, "Got a request to start a new passive scan "
		       "when a previous one has not completed!\n");
	}
    }
  else
    {
      // If the scanning flag is no longer set, then we need to make a decision
      // about how to associate.
      debug_printf(DEBUG_NORMAL, "Looking for the best network to connect to.\n");
      // Clear the passive scanning flag.
      UNSET_FLAG(ctx->flags, PASV_SCANNING);

      // Reset the timer so that we scan again.
      
      globals = config_get_globals();
      
      if (!globals)
	{
	  debug_printf(DEBUG_NORMAL, "No global data!  Passive scanning will"
		       " be broken until the next time an authentication "
		       "completes.\n");
	}
      else
	{
	  debug_printf(DEBUG_EVERYTHING, "Resetting passive scan timer.\n");
	  timer_reset_timer_count(PASSIVE_SCAN_TIMER, globals->passive_timeout);
	  
	}

      ssid = config_ssid_get_desired_ssid(ctx);

      if (strcmp(ssid, ctx->cur_essid) != 0)
	{
	  debug_printf(DEBUG_NORMAL, "The best AP to connect to appears to be"
		       " in a different ESSID!  It is likely that your card"
		       " doesn't support the needed passive scanning flags."
		       "\n");
	  // Don't do anything with the result.
	} 
      else
	{
	  // We got a valid result.  So, see if it is a different AP.  If it
	  // is, then jump to it.
	  mac = config_ssid_get_mac();
	  if (memcmp(ctx->dest_mac, mac, 6) != 0)
	    {
	      debug_printf(DEBUG_INT, "Jumpping to a BSSID with a better "
			   "signal.  (BSSID : ");
	      debug_hex_printf(DEBUG_INT, mac, 6);
	      debug_printf_nl(DEBUG_INT, ")\n");

	      // Then change our BSSIDs.
	      cardif_setBSSID(ctx, mac);
	    }
	  else
	    {
	      debug_printf(DEBUG_EVERYTHING, "We are connected to the best "
			   "BSSID already.\n");
	    }
	}
    }
}

#endif
