/*******************************************************************
 * Card driver specific interface.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: cardif.h
 *
 * Authors: Chris.Hessing@utah.edu
 *
 *******************************************************************/

#ifndef _CARDIF_H_
#define _CARDIF_H_

#include "../profile.h"

// Alg methods to use when doing WPA.
#define WPA_NONE 0
#define WPA_WEP  1
#define WPA_TKIP 2
#define WPA_CCMP 3

// Reconsider if these are needed.
#define ENC       0x02
#define WPA_IE    0x04
#define RSN_IE    0x08

#define DRIVER_NONE        0
#define DRIVER_WEXT        1
#define DRIVER_ATMEL       3

// Reasons for a disassociation.
// Valid sections include 7.3.1.7 in 802.11i-2004.
#define DISASSOC_UNSPECIFIED          1     // Unspecified reason
#define DISASSOC_INVALID_IE           13    // Invalid information element
#define DISASSOC_MIC_FAILURE          14    // MIC failure
#define DISASSOC_4WAY_TIMEOUT         15    // 4-way handshake timeout
#define DISASSOC_GROUPKEY_TIMEOUT     16    // group key handshake timeout
#define DISASSOC_DIFF_IE              17    // IEs in assoc, and 4-way differ.
#define DISASSOC_INVALID_GROUP_CIPHER 18    // Invalid group cipher
#define DISASSOC_INVALID_PAIR_CIPHER  19    // Invalid pairwise cipher
#define DISASSOC_INVALID_AKMP         20    // Invalid AKMP
#define DISASSOC_BAD_RSN_VERSION      21    // Unsupported RSN version
#define DISASSOC_INVALID_RSN_CAPS     22    // Invalid RSN capabilities
#define DISASSOC_DOT1X_FAILURE        23    // 802.1X authentication failure.
#define DISASSOC_CIPHER_REJECT        24    // Cipher suite rejected.

// Return values for associated, and unassociated.
#define IS_ASSOCIATED                 1
#define IS_UNASSOCIATED               2

// Different encryption/authentication modes that are supported.  (Since
// any interface should be able to do 802.1X, we don't flag that here.)
#define DOES_WPA                      0x00000001
#define DOES_WPA2                     0x00000002
#define DOES_WEP40                    0x00000004
#define DOES_WEP104                   0x00000008
#define DOES_TKIP                     0x00000010
#define DOES_CCMP                     0x00000020

/* RFC 2863 operational status (using different names to avoid compile problems) */
enum {
  XIF_OPER_UNKNOWN,
  XIF_OPER_NOTPRESENT,
  XIF_OPER_DOWN,
  XIF_OPER_LOWERLAYERDOWN,
  XIF_OPER_TESTING,
  XIF_OPER_DORMANT,
  XIF_OPER_UP,
};

enum {
  XIF_LINK_MODE_DEFAULT,
  XIF_LINK_MODE_DORMANT,
};

struct cardif_funcs {

  // Issue a scan request.
  int (*scan)(struct interface_data *, char);

  // Send a disassociate message to the AP
  int (*disassociate)(struct interface_data *, int);

  // Set a WEP key
  int (*set_wep_key)(struct interface_data *, uint8_t *, int, int);

  // Set a TKIP key.
  int (*set_tkip_key)(struct interface_data *, unsigned char *, int, int, 
		      char *, int);

  // Set a CCMP (AES) key.
  int (*set_ccmp_key)(struct interface_data *, unsigned char *, int, int,
		      char *, int);

  // Delete a key.
  int (*delete_key)(struct interface_data *, int, int);

  // Tell the card to associate to a specific SSID.
  void (*associate)(struct interface_data *);

  // Request the SSID for this card.
  int (*get_ssid)(struct interface_data *, char *);

  // Request the BSSID.
  int (*get_bssid)(struct interface_data *, char *);

  // Enable the WPA state.  (Set the IEs, etc.)
  int (*wpa_state)(struct interface_data *, char);

  // Enable WPA in the card.
  int (*wpa)(struct interface_data *, char);

  // Set WEP keys needed to connect to a new AP.
  int (*wep_associate)(struct interface_data *, int);

  // Tell the driver to enable countermeasures.
  int (*countermeasures)(struct interface_data *, char);

  // Tell the driver to drop all unencrypted frames.
  int (*drop_unencrypted)(struct interface_data *, char);

  // Return the WPA IE we need for this driver.
  int (*get_wpa_ie)(struct interface_data *, char *, int *);

  // Return the WPA2 IE we need for this driver.
  int (*get_wpa2_ie)(struct interface_data *, char *, int *);

  // Disable encryption.
  int (*enc_disable)(struct interface_data *);

  // Get the encryption capabilities of the card. (WEP, WPA, WPA2, TKIP, CCMP)
  void (*enc_capabilities)(struct interface_data *);

  // Change the bssid that we are connected to.
  int (*setbssid)(struct interface_data *, uint8_t *);

  // Notify the kernel (and any listeners) of the RFC2863 state of the 
  // interface.
  void (*set_operstate)(struct interface_data *, uint8_t);

  // Notify the kernel that the link should start in dormant.
  void (*set_linkmode)(struct interface_data *, uint8_t);
};

// Stuff needed by both wired, and wireless interfaces.

int cardif_init(struct interface_data *, char);
int cardif_deinit(struct interface_data *);
int cardif_get_socket(struct interface_data *);
int cardif_sendframe(struct interface_data *);
int cardif_getframe(struct interface_data *);
void cardif_clock_tick(struct interface_data *);
int cardif_get_if_state(struct interface_data *);
int cardif_int_is_valid(char *);
int cardif_check_dest(struct interface_data *);
int cardif_validate(char *);
char *cardif_get_search_ssid();
void cardif_set_search_ssid(char *);

// Stuff needed by wireless interfaces.  (If wireless isn't supported they
// should either return XENOTWIRELSS, or just return (in the case of a void)

int cardif_enable_wpa(struct interface_data *);
int cardif_do_wireless_scan(struct interface_data *, char);
int cardif_set_wep_key(struct interface_data *, uint8_t *, int, int);
int cardif_set_tkip_key(struct interface_data *, char *, int, int, char *, 
			int);
int cardif_set_ccmp_key(struct interface_data *, char *, int, int, char *, 
			int);
int cardif_delete_key(struct interface_data *, int, int);
void cardif_associate(struct interface_data *);
int cardif_disassociate(struct interface_data *, int);
int cardif_GetSSID(struct interface_data *, char *);
int cardif_check_ssid(struct interface_data *);
int cardif_GetBSSID(struct interface_data *, char *);
void cardif_setBSSID(struct interface_data *, uint8_t *);
int cardif_get_int(int, char *);
int cardif_int_is_wireless(char *);
int cardif_wep_associate(struct interface_data *, int);
int cardif_disable_wpa_state(struct interface_data *);
int cardif_enable_wpa_state(struct interface_data *);
int cardif_drop_unencrypted(struct interface_data *, char);
int cardif_countermeasures(struct interface_data *, char);
int cardif_get_wpa_ie(struct interface_data *, char *, int *);
int cardif_get_wpa2_ie(struct interface_data *, char *, int *);
int cardif_clear_keys(struct interface_data *);
int cardif_check_associated(struct interface_data *);
void cardif_reassociate(struct interface_data *, uint8_t);
void cardif_association_timeout_expired(struct interface_data *);
int cardif_enc_disable(struct interface_data *);
void cardif_get_abilities(struct interface_data *);
void cardif_wait_for_int(char *);
void cardif_passive_scan_timeout(struct interface_data *);
void cardif_operstate(struct interface_data *, uint8_t);
void cardif_linkmode(struct interface_data *, uint8_t);
#endif
