/**
* @constructor <p>aui.ui.Input</p>
* @param {object} options
*/
aui.ui.Input = function(options) {
    var me = this;
    /* merge options */
    this.options = {
        className: "aui-ui-input",
        target: null,
        required: false,
        validator: null,
        filter: null,
        defaultValue: ""
    }

    this.options = aui.lang.merge(this.options, options || {});
    this._appendToNode = this.options.target || document.body;
    this.object = null;
    this.options.filter = pvtMergeFilter(this.options.filter);

    this._min = null;
    this._max = null;

    function pvtMergeFilter(value) {
        var obj = [], cc = aui.ui.Input.charCodeCollection;
        for (var i in value) {
            var cur = value[i];
            if (typeof cur == "string") obj = obj.concat(cc[cur]);
            else if (YAHOO.lang.isArray(cur)) obj.concat(cur);
        }
        return obj.length > 0 ? new RegExp("^(" + obj.join("|") + "){1}$") : null;
    }
    
    /* custom events */
    this.onValidated = new aui.lang.util.CustomEvent("onValidated", this);
}

aui.ui.Input.prototype = {
    /** @lends aui.ui.Input.prototype */
    validator: null,
    validate: function() { },
    getValue: function() { },
    setValue: function() { },
    setMaxLength: function(value) {
        this.object.maxlength = value;
    },
    setWidth: function(width) {
        width = width.toString().replace("px", "");
        width = Number(width) + 2;

        if (width <= 10) return;
        this.object.style.width = (width - 10) + "px";
    },
    setHeight: function(height) {
        height = height.toString().replace("px", "");
        this.object.style.height = ((parseInt(height, 10) > 0) ? height : 0) + "px";
    },
    render: function(nodeToAppend) {
        this._appendToNode = nodeToAppend || this._appendToNode;
        this._appendToNode.appendChild(this.object);
    },
    unrender: function() {
        this._appendToNode.removeChild(this.object);
    },
    enable: function(value) {
        //	    this.object.disabled = !value;
        this.object.readOnly = !value;
        if (!value) {
            YAHOO.util.Dom.addClass(this.object, "disable");
            YAHOO.util.Dom.removeClass(this.object, "error");
        } else { YAHOO.util.Dom.removeClass(this.object, "disable"); }
    },
    getEnable: function() {
        return !this.object.readOnly;
    },

    error: function(value) {
        if (value) YAHOO.util.Dom.addClass(this.object, "error");
        else YAHOO.util.Dom.removeClass(this.object, "error");
    },

    getValue: function() {
        return aui.text.String.trim(this.object.value);
    },

    setValue: function(value) {
        this.object.value = ((value == null) ? "" : value);
    },

    show: function() {
        this.object.style.display = "";
    },

    hide: function() {
        this.object.style.display = "none";
    },

    focus: function() {
        if (this.object.focus) {
            try {
                this.object.focus();
            } catch (e) { }
        }
    },

    select: function() {
        if (this.object.select) {
            try {
                this.object.select();
            } catch (e) { }
        }
    },

    setLimit: function(min, max) {
        this._min = min;
        this._max = max;
    },

    setCaretPosition: function(pos) {
        if (this.object.setSelectionRange) {
            this.focus();
            this.object.setSelectionRange(pos,pos);
        }
    },

    getCaretPosition: function() {
        var caretPos = 0;
        var sel = document.selection.createRange();
        sel.moveStart ('character', -this.object.value.length);
        caretPos = sel.text.length;
        return (caretPos);
    }
}

/**
* aui.ui.Input.charCodeCollection
* @static
*/

aui.ui.Input.charCodeCollection = {
    alpha: [65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90],
    all_alpha: [65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,80,81,82,83,84,85,86,87,88,89,90,
            97,98,99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122], //lower / upper case alpha
    numeric: [48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 96, 97, 98, 99, 100, 101, 102, 103, 104, 105], /* 0-9 */
    control: [16, 17, 18, 8, 46, 35, 36, 9, 13], /* ctrl,shift,alt. backspace,delete. home,end. tab,enter */
    arrowkey: [37, 38, 39, 40]				/* left,up,right,down */
}
aui.ui.Input.charCodeCollection.alphaNumeric = [].concat(aui.ui.Input.charCodeCollection.alpha).concat(aui.ui.Input.charCodeCollection.numeric);

/**
* @constructor <p>aui.ui.TextInput</p>
* @param {object} options
*/

aui.ui.TextInput = function(options) {
    var me = this;
    aui.ui.TextInput.superclass.constructor.call(this, options);

    this.pvtValidator = this.options.validator || /.*/;
    this.object = document.createElement("input");
    this.object.className = this.options.className;
    this.object.type = "text";
    this.object.value = this.options.defaultValue;
    /* key filter */
    //	this.object.attachEvent("onkeydown", function(){
    //		if(me.options.filter){
    //			//window.status = YAHOO.util.Event.getCharCode(YAHOO.util.Event.getEvent());
    //			return me.validate( me.options.filter, YAHOO.util.Event.getCharCode(YAHOO.util.Event.getEvent()), true );
    //		}
    //	});

    this.object.onmousedown = function() { event.cancelBubble = true; };    //Albert add, UI 3.0.01.13-006
    this.object.onkeydown = function() {
        var result = false;
        
        if(me.options.filter){
            result = me.validate(me.options.filter, YAHOO.util.Event.getCharCode(YAHOO.util.Event.getEvent()), true);
            
            if(result) me.onKeyDown.fire();
            
            return result;
        }
        if (event.keyCode == 9) { //Albert Added, Bug 17040, TabkeyupLkvalidateD
            me.validate();
        }
    };
    this.object.onkeyup = function() {
        
        if(event.keyCode != 9){ // ignore tab button.
            if (typeof me._max == "number") {
                if (parseInt(me.getValue()) > me._max) me.setValue(me._max);
            }
            else {
                me.validate();
            }
            
            me.onKeyUp.fire();
        }
    };
    /*
    this.object.attachEvent("onkeyup", function() {
    if (typeof me._max == "number") {
    if (parseInt(me.getValue()) > me._max)
    me.setValue(me._max);
    } else {
    me.validate();
    }
    me.onKeyUp.fire();
    });
    */
    /* methods */
    this.setMaxLength = function(value) { me.object.maxLength = value; }
    this.selectText = function() { me.object.select(); }

    this.onClick = new aui.lang.util.CustomEvent('onClick', this);
    this.onChange = new aui.lang.util.CustomEvent('onChange', this);
    this.onBlur = new aui.lang.util.CustomEvent('onBlur', this);
    this.onKeyDown = new aui.lang.util.CustomEvent('onKeyDown', this);
    this.onKeyUp = new aui.lang.util.CustomEvent('onKeyUp', this);
    this.onFocus = new aui.lang.util.CustomEvent('onFocus', this);
    this.object.onclick = function() { me.onClick.fire() };
    this.object.onchange = function() { me.onChange.fire() };
    this.object.onfocus = function() {  //Albert Add, Bug 11908
        if (this.readOnly) this.blur();
        me.onFocus.fire();
    }
    this.object.onblur = function() {
        if (typeof me._min == "number") {
            if (parseInt(me.getValue()) < me._min || me.getValue() == "")
                me.setValue(me._min);
        }
        if (me.options.defaultValue) {
            if (me.getValue() == "")
                me.setValue(me.options.defaultValue);
        }
        me.onBlur.fire();
    };
}

aui.lang.Class.extend(aui.ui.TextInput, aui.ui.Input, {
    validate: function(validator, value, noevent) {

        var result = true;

        validator = validator || this.pvtValidator;
        noevent = noevent || false;
        value = value || this.object.value;

        if (!value || (value.toString().replace(/\s/g, "") == "")) {

            if (this.getEnable()) {
                result = !this.options.required;
            }
            else {
                result = true;
            }

            if (typeof validator == "function") result = validator(value);
        }
        else {
            result = (typeof validator == "function")
                ? validator(value)
                : validator.test(value);
        }

        if (!noevent) this.onValidated.fire(result, value);

        return result;
    }
});

/**
* @constructor <p>aui.ui.NumericInput</p>
*/

aui.ui.NumericInput = function(options) {
    options = options || {};
    options.filter = ["numeric", "control", "arrowkey"];
    
    var me = input = new aui.ui.TextInput(options);
    
    input.object.onkeyup = function() {
        
        if(event.keyCode != 9){ // ignore tab button.
            var element = (event.srcElement) ? event.srcElement : this;
            
            var numeric = /^[0-9]+$/;
            
            if(!numeric.test(element.value)) element.value = element.value.replace(/\D/g, '');
            
            if(typeof me._max == "number"){
                if(parseInt(me.getValue()) > me._max) me.setValue(me._max);
            }
            
            me.validate();
            me.onKeyUp.fire();
        }
    };
    input.object.onkeydown = function() {
        var result = false;
        
        if (!event.shiftKey) {
            result = me.validate(me.options.filter, YAHOO.util.Event.getCharCode(YAHOO.util.Event.getEvent()), true);
        }
        else{
            result = false;
        }
        
        if(result) me.onKeyDown.fire();
        
        return result;
    };
    input.object.onblur = function() {  //Albert Add, Bug 11542
        if (me.options.defaultValue) {
            if (me.getValue() == "")
                me.setValue(me.options.defaultValue);
        }
        if (typeof me._min == "number") {
            if (parseInt(me.getValue()) < me._min || me.getValue() == "")
                me.setValue(me._min);
        }
        
        if (!options.keepText) { /* Judy modified for Bug 11586, make converting text to integer a optional process */
            if (!isNaN(me.getValue()) && me.getValue() != '') {
                me.setValue(parseInt(me.getValue()));
            }
        }
        me.onBlur.fire();
    };
    
    /*
    input.object.style.imeMode = 'disable';
    
    input.object.onpropertychange = function() {
        var numeric = /^[0-9]+$/;
        
        if(!numeric.test(me.object.value)) me.object.value = me.object.value.replace(/\D/, '');
    }
    */
    
    return input;
}

/**
* @namespace <p>aui.ui.PortInput</p>
* Available Ports: 0 through 65535.
*/

aui.ui.PortInput = function(options) {
    options = options || {};
    options.filter = ["numeric", "control", "arrowkey"];
    
    var me = input = new aui.ui.TextInput(options);
    
    input.object.style.imeMode = 'disabled';    //Albert added, Bug 19823, disable IME.
    input.object.onkeyup = function() {
        
        if(event.keyCode != 9){ // ignore tab button.
            var element = (event.srcElement) ? event.srcElement : this;
            
            var numeric = /^[0-9]+$/;
            
            if(!numeric.test(element.value)) element.value = element.value.replace(/\D/g, '');
            
            if(element.value > 65535) element.value = 65535;
            
            me.validate();
            me.onKeyUp.fire();
        }
    };
    input.object.onkeydown = function() {
        var result = false;
        
        if (!event.shiftKey) {
            result = me.validate(me.options.filter, YAHOO.util.Event.getCharCode(YAHOO.util.Event.getEvent()), true);
        }
        else{
            result = false;
        }
        
        if(result) me.onKeyDown.fire();
        
        return result;
    };
    input.object.onblur = function() {
        this.value = parseInt(this.value);
        if (this.value < 1 || isNaN(this.value)) this.value = 1;
        me.onBlur.fire();
    }
    
    /*
    input.object.style.imeMode = 'disable';
    
    input.object.onpropertychange = function() {
        var numeric = /^[0-9]+$/;
        
        if(!numeric.test(me.object.value)) me.object.value = me.object.value.replace(/\D/, '');
    }
    */
    
    return input;
}

/**
* @constructor <p>aui.ui.AlphaNumericInput</p>
*/

aui.ui.AlphaNumericInput = function(options) {
    options = options || {};
    options.filter = ["numeric", "alpha", "control", "arrowkey"];
    options.validator = function(value) {
        var alpha_numeric = /(\W)|\_/;
        
        if(!alpha_numeric.test(value)){
            return true;
        }
        else{
            return false;
        }
    }

    var me = input = new aui.ui.TextInput(options);
    
    input.object.onkeyup = function() {
        
        if(event.keyCode != 9){ // ignore tab button.
            var element = (event.srcElement) ? event.srcElement : this;
            
            var alpha_numeric = /(\W)|\_/;
            
            if(alpha_numeric.test(element.value)) element.value = element.value.replace(/(\W)|\_/g, '');
            
            me.validate();
            me.onKeyUp.fire();
        }
    };
    input.object.onkeydown = function() {
        var type = 'alpha';
        
        if(me.options.filter.test(event.keyCode)){
            
            for(var i in aui.ui.Input.charCodeCollection.numeric){
                if(event.keyCode == aui.ui.Input.charCodeCollection.numeric[i]) type = 'numeric';
            }
            
            if(type == 'numeric'){
                if (!event.shiftKey) {
                    result = me.validate(me.options.filter, YAHOO.util.Event.getCharCode(YAHOO.util.Event.getEvent()), true);
                }
                else{
                    result = false;
                }
            }
            else{
                result = me.validate(me.options.filter, YAHOO.util.Event.getCharCode(YAHOO.util.Event.getEvent()), true);
            }
            
            if(result) me.onKeyDown.fire();
            
            return result;
        }
        else{
            return false;
        }
    };
    
    return input;
}

/**
* @namespace <p>aui.ui.NetworkAddressInput</p>
*/

aui.ui.NetworkAddressInput = function(options) {
    options = options || {};
    options.validator = /^(25[0-5]|2[0-4][0-9]|[0-1]{1}[0-9]{2}|[1-9]{1}[0-9]{1}|[1-9])\.(25[0-5]|2[0-4][0-9]|[0-1]{1}[0-9]{2}|[1-9]{1}[0-9]{1}|[1-9]|0)\.(25[0-5]|2[0-4][0-9]|[0-1]{1}[0-9]{2}|[1-9]{1}[0-9]{1}|[1-9]|0)\.(25[0-5]|2[0-4][0-9]|[0-1]{1}[0-9]{2}|[1-9]{1}[0-9]{1}|[0-9])$/;
    return new aui.ui.TextInput(options);
}

/**
* @constructor <p>aui.ui.EmailAddressInput</p>
*/

aui.ui.EmailAddressInput = function(options) {
    options = options || {};
    options.validator = /^[a-zA-Z0-9_\-\.\@]+$/;
    return new aui.ui.TextInput(options);
}

/**
* @constructor <p>aui.ui.DateTimeInput</p>
*/

aui.ui.DateTimeInput = function(options) {
    options = options || {};
    //options.validator = /^(19[0-9]{2}|[2-9][0-9]{3})[-/]((0(1|3|5|7|8)|10|12)[-/](0[1-9]|1[0-9]|2[0-9]|3[0-1])|(0(4|6|9)|11)[-/](0[1-9]|1[0-9]|2[0-9]|30)|(02)[-/](0[1-9]|1[0-9]|2[0-9]))\x20(0[0-9]|1[0-9]|2[0-3])(:[0-5][0-9]){2}$/;
    options.validator = function(value) {
        return new Date(value).valueOf() > 0;
    }
    return new aui.ui.TextInput(options);
}

/**
* @constructor <p>aui.ui.IDCardInput</p>
*/

aui.ui.IDCardInput = function(options) {
    options = options || {};
    options.filter = ["alphaNumeric", "control", "arrowkey"];
    options.validator = function(value) {
        var regexp = /^[A-Z][12][0-9]{8}$/i;
        if (regexp.test(value)) {
            var x = 10 + "abcdefghjklmnpqrstuvxywzio".indexOf(value.charAt(0).toLowerCase());
            var chksum = (x - (x % 10)) / 10 + (x % 10) * 9;
            for (var i = 1; i < 9; i++) chksum += value.charAt(i) * (9 - i);
            chksum = (10 - chksum % 10) % 10;
            if (chksum == value.charAt(9)) return true;
        } return false;
    }
    return new aui.ui.TextInput(options);
}

/**
* @constructor <p>aui.ui.TimeDuration</p>
*/

aui.ui.TimeDuration = function(options) {
    options = options || {};
    options.validator = /[0-9]{4}(\/[0-9]{1,2}(\/[0-9]{1,2}( [0-9]{1,2}\:[0-9]{1,2}(\:[0-9]{1,2})?)?)?)?/gi;
    var me = input = new aui.ui.TextInput(options);
    input.validate = function(validator, value) {
        validator = validator || input.pvtValidator;
        value = value || me.object.value;
        var result = ((me.options.required == false && value == "") || (me.startTime && me.endTime)) ? true : false;
        me.onValidated.fire(result, value);
        return result;
    }
    /* functions */
    function analyze() {     /* analyze input date and set into startTime / endTime */
        var stTemp, etTemp;
        me.startTime = null, me.endTime = null;
        var temp = me.object.value.match(options.validator);
        if (temp && temp.length > 0) me.startTime = (stTemp = aui.lang.Date(temp[0])) ? temp[0] : null;
        if (temp && temp.length > 1) me.endTime = (etTemp = aui.lang.Date(temp[1])) ? temp[1] : null;
        if (stTemp && etTemp && etTemp < stTemp) { var temp2 = me.endTime; me.endTime = me.startTime; me.startTime = temp2; }
    }

    function rewrite() {     /* rewrite input value to rule */
        if (me.startTime && me.endTime) {
            me.pvtStoredValue = me.object.value;
            me.object.value = "from " + me.startTime + " to " + me.endTime;
        }
        me.validate();
    }
    /* events */
    input.object.onpropertychange = function() {
        if (event.propertyName == "value") analyze();
    }
    input.object.onblur = function() {
        rewrite();
    }
    input.object.onfocus = function() { if (me.pvtStoredValue) event.srcElement.value = me.pvtStoredValue; }
    /* methods */
    input.getValue = function() { return me.startTime + "-" + me.endTime; }
    input.setValue = function(value) {
        me.object.value = value == null ? "" : value;
        //analyze();
        rewrite();
    }
    input.getStartTime = function() { return me.startTime }
    input.getEndTime = function() { return me.endTime }

    return input;
};


/**
* @constructor <p>aui.ui.UserNameInput</p>
*/

aui.ui.UserNameInput = function(options) {
    options = options || {};
    /* 3-31 characters, start with alpha, else alphanumeric */
    //options.filter = ["alphaNumeric","control","arrowkey"];
    //options.validator = /^[a-z]([a-z]|[0-9]){2,30}$/i;
    return new aui.ui.TextInput(options);
}

/**
* @constructor <p>aui.ui.PasswordInput</p>
*/

aui.ui.PasswordInput = function(options) {
    options = options || {};
    /* 5-63 characters */
    //options.validator = /^.{5,63}$/;
    var obj = new aui.ui.TextInput(options);
    
    obj.changed = false;
    obj.object.type = "password";
    
    obj.object.onblur = function() {        
        obj.onChange.fire();
    }
    obj.object.onkeydown = function() {
        if (event.keyCode == 9) { //Albert Added, Bug 17040, TabkeyupLkvalidateD
            obj.realValue = this.value;
            obj.validate();
        }
        obj.onKeyDown.fire();
    }
    obj.object.onkeyup = function() {
        if(obj.getEnable() && event.keyCode != 9){ // ignore tab button.
            obj.realValue = this.value;
            obj.validate();
            obj.onKeyUp.fire();
        }
    }
    obj.object.onfocus = function() {
        if(obj.getEnable()){
            this.value = '';
            obj.changed = true;
        }
    }
    
    obj.getValue = function() {  //Albert Modify, Bug 11664,11666 password allow blank char.
        return this.realValue; 
    }
    obj.setValue = function(value) {
        this.realValue = value;
        if (value != '') 
            this.object.value = '******';
        else
            this.object.value = '';
    }
    obj.onChange.subscribe(function(type, args) {
        if(this.changed){
            if(this.object.value != '') this.realValue = this.object.value;
            if(this.realValue != '') this.object.value = '******';
            this.changed = false;
        }
    });
    return obj;
}


/**
* @constructor <p>aui.ui.TextAreaInput</p>
* @param {object} options
*/

aui.ui.TextAreaInput = function(options) {
    var me = this;
    /* merge options */
    this.options = {
        className: "aui-ui-TextAreaInput" 
    }
    this.options = aui.lang.merge(this.options, options || {});
    aui.ui.TextAreaInput.superclass.constructor.call(this, this.options);

    this.pvtValidator = this.options.validator || /.*/;
    this.object = document.createElement("textarea");
    this.object.className = this.options.className;
    /* key filter */
    this.object.attachEvent("onkeydown", function() {
        if (me.options.filter) {
            //window.status = YAHOO.util.Event.getCharCode(YAHOO.util.Event.getEvent());
            return me.validate(me.options.filter, YAHOO.util.Event.getCharCode(YAHOO.util.Event.getEvent()), true);
        }
    });
    this.object.attachEvent("onkeyup", function() {
        me.validate();
    });
    this.object.onfocus = function() {  //Albert Add, Bug 11908
        if (this.readOnly) this.blur();
    }
    this.object.onblur = function() {
        me.onblur.fire();
    }
    /* methods */
    this.getValue = function() { return me.object.innerText }
    this.setValue = function(value) { me.object.innerText = value == null ? "" : value }
    this.onblur = new aui.lang.util.CustomEvent("onblur", this); 
}

aui.lang.Class.extend(aui.ui.TextAreaInput, aui.ui.Input, {
    validate: function(validator, value, noevent) {
        validator = validator || this.pvtValidator;
        noevent = noevent || false;
        value = value || this.object.value;
        //        var result = (this.options.required == false && value == "") ? true :
        //			((typeof validator == "function") ? validator(value) : validator.test(value));
        var result = true;
        if (!value || (value.toString().replace(/\s/g, "") == "")) {

            if (this.getEnable()) {
                result = !this.options.required;
            }
            else {
                result = true;
            }

            if (typeof validator == "function") result = validator(value);
        }
        else {
            result = (typeof validator == "function")
                ? validator(value)
                : validator.test(value);
        }
        if (!noevent) this.onValidated.fire(result, value);
        return result;
    },
    setMaxLength: function(max) { /*]TextAreaϥMaxLengthݩʵLġAQjsMaxLength.(20120917 Leo add)*/
        this.object.onkeyup = function() {
            if (event.keyCode != 37 && event.keyCode != 39) //[Bug 15708, 15709]LokeyboardkAϨi@
                if (this.value.length > max) //[Bug 15928]fixed WX]wȤ~slice
                    this.value = this.value.slice(0, max);

        };
    }
});


/**
* @constructor <p>aui.ui.OpenFileInput</p>
* @param {object} options
*/

aui.ui.OpenFileInput = function(options) {
    var me = this;
    /* merge options */
    this.options = {
        className: "aui-ui-OpenFileInput",
        isSaveFile: false,
        fileName: ""
    }
    this.options = aui.lang.merge(this.options, options || {});
    aui.ui.OpenFileInput.superclass.constructor.call(this, this.options);

    this.pvtValidator = this.options.validator || /.*/;
    /* container */
    this.object = document.createElement("div");
    this.object.className = this.options.className;
    /* elements */
    this.input = document.createElement("input");
    this.input.className = "aui-ui-input";
    this.input.readOnly = true;
    this.input.onmousedown = function() {
        document.body.focus();
        return false;
    }
    this.object.appendChild(this.input);
    this.button = new aui.nvr.ui.CheckButton({ caption: Lang("MSG_BROWSE") });
    this.button.setMaxWidth(100);
    this.button.render(this.object);
    this.button.onclick.subscribe(function() {
        /* Judy modified for Bug 10375, maintain value after user click on cancel */
        if (!me.options.isSaveFile)
            var value = aui.nvr.Utilities.openFileDialog(me.options.def, me.options.types, me.options.fileName);
        else
            var value = aui.nvr.Utilities.saveFileDialog(me.options.def, me.options.types, me.options.fileName); //Albert Added for saveFileDialog.

        if (value != "") {
            me.input.value = value;
        }
        me.validate();
    });
    /* key filter */
    this.object.attachEvent("onkeydown", function() {
        if (me.options.filter) {
            //window.status = YAHOO.util.Event.getCharCode(YAHOO.util.Event.getEvent());
            return me.validate(me.options.filter, YAHOO.util.Event.getCharCode(YAHOO.util.Event.getEvent()), true);
        }
    });
    this.object.attachEvent("onkeyup", function() {
        me.validate();
    });
    /* methods */
    this.getValue = function() { return me.input.value }
    this.setValue = function(value) { me.input.value = value == null ? "" : value }
    this.setDefaultFileName = function(fileName) { me.options.fileName = fileName; }
}

aui.lang.Class.extend(aui.ui.OpenFileInput, aui.ui.Input, {
    validate: function(validator, value, noevent) {
        validator = validator || this.pvtValidator;
        noevent = noevent || false;
        value = value || this.input.value;
        var result = (this.options.required == false && value == "") ? true :
			((typeof validator == "function") ? validator(value) : validator.test(value));
        if (!noevent) this.onValidated.fire(result, value);
        return result;
    },
    enable: function(value) {
        if (!value) {
            YAHOO.util.Dom.addClass(this.input, "disable");
            YAHOO.util.Dom.removeClass(this.input, "error");
        } else { YAHOO.util.Dom.removeClass(this.input, "disable"); }
        this.button.setEnable(value);
    }
});


/**
* @constructor <p>aui.ui.OpenFolderInput</p>
* @param {object} options
*/

aui.ui.OpenFolderInput = function(options) {
    var me = this;
    /* merge options */
    this.options = {
        className: "aui-ui-OpenFileInput"
    }
    this.options = aui.lang.merge(this.options, options || {});
    aui.ui.OpenFileInput.superclass.constructor.call(this, this.options);

    this.pvtValidator = this.options.validator || /.*/;
    /* container */
    this.object = document.createElement("div");
    this.object.className = this.options.className;
    /* elements */
    this.input = document.createElement("input");
    this.input.className = "aui-ui-input";
    this.input.readOnly = true;
    this.input.onmousedown = function() {
        document.body.focus();
        return false;
    }
    this.object.appendChild(this.input);
    this.button = new aui.nvr.ui.CheckButton({ caption: Lang("MSG_BROWSE") });
    this.button.setMaxWidth(150);
    this.button.render(this.object);
    this.button.onclick.subscribe(function() {
        /* Judy modified for Bug 10375, maintain value after user click on cancel */
        var value = aui.nvr.Utilities.openFolderDialog();
        if (value != "")
            me.input.value = value;
    });
    /* key filter */
    this.object.attachEvent("onkeydown", function() {
        if (me.options.filter) {
            //window.status = YAHOO.util.Event.getCharCode(YAHOO.util.Event.getEvent());
            return me.validate(me.options.filter, YAHOO.util.Event.getCharCode(YAHOO.util.Event.getEvent()), true);
        }
    });
    this.object.attachEvent("onkeyup", function() {
        me.validate();
    });
    /* methods */
    this.getValue = function() { return me.input.value }
    this.setValue = function(value) { me.input.value = value == null ? "" : value }
}

aui.lang.Class.extend(aui.ui.OpenFolderInput, aui.ui.Input, {
    validate: function(validator, value, noevent) {
        validator = validator || this.pvtValidator;
        noevent = noevent || false;
        value = value || this.input.value;
        var result = (this.options.required == false && value == "") ? true :
			((typeof validator == "function") ? validator(value) : validator.test(value));
        if (!noevent) this.onValidated.fire(result, value);
        return result;
    }
});
