﻿/**
* @namespace
*/
aui.ui.Dialog = function() { }

/**
* @static
* @class
*/
aui.ui.Dialog.Icon = {
    None: 'None',
    Question: 'Question',
    Confirm: 'Confirm',
    Stop: 'Stop',
    Error: 'Error',
    Warning: 'Warning',
    Information: 'Information'
}

/**
* @static
* @class
*/
aui.ui.Dialog.ButtonSet = {
    None: [],
    OK: ['OK'],
    YES: ['YES'],
    NO: ['NO'],
    Cancel: ['CANCEL'],
    Close: ['CLOSE'],
    OKCancel: ['OK', 'CANCEL'],
    CancelOK: ['CANCEL', 'OK'],
    CancelApply: ['CANCEL', 'APPLY'],
    CancelExport: ['CANCEL', 'EXPORT'],
    AbortRetryIgnore: ['ABORT', 'RETRY', 'IGNORE'],
    YesNoCancel: ['YES', 'NO', 'CANCEL'],
    YesNo: ['YES', 'NO'],
    NoYes: ['NO', 'YES'],
    RetryCancel: ['RETRY', 'CANCEL'],
    ResetCancelApply: ['RESET', 'CANCEL', 'APPLY'],
    LicenselistClose: ['LICENSE_LIST', 'CLOSE'],
    Licenselist: ['LICENSE_LIST'],
    CancelDelete: ['CANCEL', 'DELETE'],
    CloseDelete: ['CLOSE', 'DELETE']
}

/**
* @static
* @class
*/
aui.ui.Dialog.DialogResult = {
    /**
    * Represent the "OK" button been clicked.
    * @static
    * @field
    */
    OK: 2,

    /**
    * Represent the "Yes" button been clicked.
    * @static
    * @field
    */
    YES: 4,

    /**
    * Represent the "Apply" button been clicked.
    * @static
    * @field
    */
    APPLY: 8,

    /**
    * Represent the "Export" button been clicked.
    * @static
    * @field
    */
    EXPORT: 8,

    /**
    * Represent the "Confirm" button been clicked.
    * @static
    * @field
    */
    CONFIRM: 16,

    /**
    * Represent the "Retry" button been clicked.
    * @static
    * @field
    */
    RETRY: 32,

    /**
    * Represent the "Reset" button been clicked.
    * @static
    * @field
    */
    RESET: 32,

    /**
    * Represent the "Ignore" button been clicked.
    * @static
    * @field
    */
    IGNORE: 64,

    /**
    * Represent the "Abort" button been clicked.
    * @static
    * @field
    */
    ABORT: 128,
    
    /**
    * Represent the "License List" button been clicked.
    * @static
    * @field
    */
    LICENSE_LIST: 256,
    
    /**
    * Represent the "Delete" button been clicked.
    * @static
    * @field
    */
    DELETE: 512,
    
    /**
    * Represent the "Cancel" button been clicked.
    * @static
    * @field
    */
    CANCEL: 0,

    /**
    * Represent the "No" button been clicked.
    * @static
    * @field
    */
    NO: 0,
    
    /**
    * Represent the "No" button been clicked.
    * @static
    * @field
    */
    CLOSE: 0
};


/**
* @constructor
*/
aui.ui.Dialog.DialogBase = function(options) {
    this.options = {
        className: 'aui-ui-Dialog-DialogBase',
        width: null,
        height: null,
        destructor: false,
        hotkey: false
    }
    var me = this;

    this.options = aui.lang.merge(this.options, options || {});
    this.className = this.options.className;
    this.width = this.options.width;
    this.height = this.options.height;
    this.hotkey = this.options.hotkey;

    this.pvtResult = aui.ui.Dialog.DialogResult.NO;
    this.pvtCallback = null;

    this.dialogBG = NewObj("div", "aui-ui-Dialog-DialogBackground");
    this.dialogContent = NewObj("div", this.className);

    this.dialogContent.onkeyup = function checkSubmit() {
        if (event.srcElement.tagName.toLowerCase() == 'input' ||
           event.srcElement.tagName.toLowerCase() == 'textarea' ||
           event.srcElement.contentEditable == "true")
            return;
        if (me.hotkey) {
            switch (event.keyCode) {
                case 13: // Enter press
                    if (me.defaultButton != undefined)
                        me.setResult(me.defaultButton);
                    break;
                case 27: // ESC press
                    me.setResult(aui.ui.Dialog.DialogResult.CANCEL);
                    break;
            }
        }
    }
    if (this.height) this.dialogContent.style.height = Math.max(this.height, 0) + "px";
    if (this.width) this.dialogContent.style.width = Math.max(this.width, 0) + "px";

    var dialogFrame = NewObj("iframe", "dialog-frame")
    this.dialogBGColor = NewObj("div", "dialog-bgcolor");

    this.dialogBG.appendChild(dialogFrame);
    this.dialogBG.appendChild(this.dialogBGColor);

    /**
    /PREPARE DIALOG
    **/
    this.prepareDialog(this.dialogContent);
    
    this.dialogBG.style.display = "none";
    this.dialogContent.style.display = "none";
    
    document.body.appendChild(this.dialogBG);
    document.body.appendChild(this.dialogContent);
};

aui.ui.Dialog.Manager = {
    dialogList: new aui.collections.ArrayList(),
    dialogIndex: 990,
    add: function(dia) {
        // Landy added for bug 14606, to avoid that double adding dialog causes dialogList uncompleted removing
        // if dialogList removes incompletely, zIndex will continually increase cause ComboDropdownList couldn't show sub items
        if (aui.ui.Dialog.Manager.dialogList.contains(dia)) return;
        
        aui.ui.Dialog.Manager.dialogList.add(dia);
        //Albert Modified, Bug 15593, z-index will increase continually
        //Judy added, reset z-index to dialog, Bug 14288
        //dia.setZIndex(aui.ui.Dialog.Manager.dialogIndex++);
        dia.setZIndex(aui.ui.Dialog.Manager.dialogIndex + aui.ui.Dialog.Manager.dialogList.getCount() + 1);
    },
    remove: function(dia) {
        aui.ui.Dialog.Manager.dialogList.remove(dia);
        
        if (aui.ui.Dialog.Manager.dialogList.getCount() == 0)
            aui.ui.Dialog.Manager.dialogIndex = 990;
    },

    refresh: function() {
        aui.ui.Dialog.Manager.dialogList.each(function(each) {
            each.item.refresh();
        });
    }
}

aui.ui.Dialog.DialogBase.prototype = {
    /**
    * Get the dialog result from user operation.
    * @function
    * @return {aui.ui.Dialog.DialogResult}
    * Return the result set by user. The default 
    * value is {aui.ui.Dialog.DialogResult.NO}.
    */
    getResult: function() {
        return this.pvtResult;
    },

    /**
    * Set a new dialog result to dialog. Once the dialog
    * result is set, the dialog will hide and call the
    * callback function provided when calling showDialog().
    * exception note: click RESET will keep dialog and call the
    * callback function
    * @function
    * @param {aui.ui.Dialog.DialogResult} result
    * The new result for this dialog.
    */
    setResult: function(result) {
        this.pvtResult = result;

        if (this.pvtResult == aui.ui.Dialog.DialogResult.APPLY) {
            if (this.pvtValidate()) {
                if (this.pvtCallback) {
                    if (this.pvtCallback(result) != false)
                        this.hideDialog();
                } else {
                    this.hideDialog();
                }
            }
        } else {
            if (this.pvtCallback) {
                if (this.pvtResult == aui.ui.Dialog.DialogResult.RESET) {
                    this.pvtCallback(result);
                    return;
                }
                if (this.pvtCallback(result) != false)
                    this.hideDialog();
            } else {
                this.hideDialog();
            }
        }

        //destructor dialog after using it.
        if (this.options.destructor) this.destructor();
    },

    refresh: function() {
        if (!this.config) return;

        YAHOO.util.Dom.setStyle(this.dialogBG, "height", aui.applications.getHeight() + "px"); //document.body.clientHeight
        YAHOO.util.Dom.setStyle(this.dialogBG, "width", aui.applications.getWidth() + "px"); //document.body.clientWidth

        YAHOO.util.Dom.setStyle(this.dialogContent, "top", ((aui.applications.getHeight() - 27 - this.dialogContent.offsetHeight) / 2) + "px");
        YAHOO.util.Dom.setStyle(this.dialogContent, "left", ((aui.applications.getWidth() - this.dialogContent.offsetWidth) / 2) + "px");

        if (this.config.position.indexOf('top') >= 0)
            YAHOO.util.Dom.setStyle(this.dialogContent, "top", 0);

        if (this.config.position.indexOf('bottom') >= 0)
            YAHOO.util.Dom.setStyle(this.dialogContent, "top", (aui.applications.getHeight() - this.dialogContent.offsetHeight) + "px");

        if (this.config.position.indexOf('left') >= 0)
            YAHOO.util.Dom.setStyle(this.dialogContent, "left", 0);

        if (this.config.position.indexOf('right') >= 0)
            YAHOO.util.Dom.setStyle(this.dialogContent, "left", (aui.applications.getWidth() - this.dialogContent.offsetWidth) + "px");
    },
    /**
    * Show dialog.
    * @function
    * @param {object} config
    */
    showDialog: function(config) {
        this.config = {
            showBG: true,
            position: "center", //top_left . top . top_right . left . center . right . bottom_left . bottom . bottom_right
            onResult: null,
            onValidate: function() { return true; },
            onShow: function() { }
        }

        this.config = aui.lang.merge(this.config, config || {});

//        if (this.config.showBG)
//            document.body.appendChild(this.dialogBG);
        if (this.config.showBG)
            this.dialogBG.style.display = "block";

//        document.body.appendChild(this.dialogContent);
        this.dialogContent.style.visibility = "hidden";
        this.dialogContent.style.display = "block";

        this.refresh();

        this.dialogContent.style.visibility = "visible";

        this.dialogContent.focus();
        if (this.config) {
            this.pvtCallback = this.config.onResult;
            this.pvtValidate = this.config.onValidate;
        }

        this.config.onShow();
        aui.ui.Dialog.Manager.add(this);
    },

    /**
    * Hide dialog.
    * @function
    */
    hideDialog: function() {
        var me = this;
        aui.ui.Dialog.Manager.remove(this);
        Try.all(
//			function() { document.body.removeChild(me.dialogBG) }
//			function() { document.body.removeChild(me.dialogContent) }
		)
        this.dialogBG.style.display = "none";
        this.dialogContent.style.display = "none";
    },

    pvtValidate: function() {
        return false;
    },

    _lastX: null,
    _lastY: null,

    moveToLastX: function(duration, callback) {
        if (this._lastX === null) return;
        this.moveToX(this._lastX, duration, callback);
    },

    moveToLastY: function(duration, callback) {
        if (this._lastY === null) return;
        this.moveToY(this._lastY, duration, callback);
    },

    moveToX: function(X, duration, callback) {
        if (this.dialogContent.style.left == X + "px") return;

        this._lastX = parseInt(this.dialogContent.style.left.replace("px", ""));

        if (duration) {
            var myAnim = new YAHOO.util.Anim(this.dialogContent, {
                left: { to: X }
            }, duration);

            if (callback) myAnim.onComplete.subscribe(callback);

            myAnim.animate();
        } else {
            this.dialogContent.style.left = X + "px";
            if (callback) callback();
        }
    },

    moveToY: function(Y, duration, callback) {
        if (this.dialogContent.style.top == Y + "px") return;

        this._lastY = parseInt(this.dialogContent.style.top.replace("px", ""));

        if (duration) {
            var myAnim = new YAHOO.util.Anim(this.dialogContent, {
                top: { to: Y }
            }, duration);

            if (callback) myAnim.onComplete.subscribe(callback);

            myAnim.animate();
        } else {
            this.dialogContent.style.top = Y + "px";
            if (callback) callback();
        }
    },

    setWidth: function(width, duration, callback) {
        if (width < 0 || isNaN(width)) return; //Judy modified for Bug 12766, IE8 does not support negative number
        
        if (this.dialogContent.style.width == width + "px") return;
        this.width = width;

        if (duration) {
            var myAnim = new YAHOO.util.Anim(this.dialogContent, {
                width: { to: this.width }
            }, duration);

            if (callback) myAnim.onComplete.subscribe(callback);

            myAnim.animate();
        } else {
            this.dialogContent.style.width = this.width + "px";
            if (callback) callback();
        }
    },

    getWidth: function() {
        return this.width;
    },

    setHeight: function(height) { //Judy modified for Bug 12766, IE8 does not support negative number
        if (height < 0 || isNaN(height)) return;
        
        this.height = height;
        this.dialogContent.style.height = this.height + "px";
    },

    getHeight: function() {
        return this.height;
    },

    getX: function() {
        return parseInt(this.dialogContent.style.left.replace("px", ""));
    },

    getY: function() {
        return parseInt(this.dialogContent.style.top.replace("px", ""));
    },

    destructor: function() {
        for (var key in this)
            this[key] = null;
        //this = null;
    },
    
    setBGColorClass: function(className){
        this.dialogBGColor.className = className;
    },
    
    setZIndex: function(num) {
        this.dialogContent.style.zIndex = num;
        this.dialogBG.style.zIndex = num;
    }
}
