#include <stdlib.h>
#include <string.h>
#include <stddef.h>

#include "sys.h"
#include "sysHashTable.h"




sysHashTable_S *
sysHashTableInit(int storageStructSize,
               int storageNextPtrOffset,
               int bucketNum,
               sysHashCompare_FT compFun,
               sysHashHash_FT    hashFun,
               sysHashDumpOne_FT dumpOneFun)
{
    sysHashTable_S  *pHashTable;

    pHashTable = (sysHashTable_S *) sysMalloc(sizeof(sysHashTable_S)); 
    memset(pHashTable, 0, sizeof(sysHashTable_S));

    pHashTable->storageStructSize = storageStructSize;
    pHashTable->storageNextPtrOffset = storageNextPtrOffset;
    pHashTable->bucketNum = bucketNum;
    pHashTable->compFun = compFun;
    pHashTable->hashFun = hashFun;
    pHashTable->dumpOneFun = dumpOneFun;

    /* buckets */
    pHashTable->buckets = (sysHashBucket_T *) sysMalloc(sizeof(sysHashBucket_T)*bucketNum);
    memset(pHashTable->buckets, 0, sizeof(sysHashBucket_T)*bucketNum);

    return (pHashTable);
}

void sysHashTableExit(sysHashTable_S *pHashTable)
{
    sysFree(pHashTable->buckets);
    sysFree(pHashTable);
}

void sysHashTableDump(sysHashTable_S *pHashTable)
{
    int i;
    void *pElement;

    if (!pHashTable)
        return;

    for (i = 0; i < pHashTable->bucketNum; ++i)
    {
        pElement = pHashTable->buckets[i];
        while (pElement)
        {
            pHashTable->dumpOneFun(pElement); 
            memcpy(&pElement, (void *) (((char *) pElement) + pHashTable->storageNextPtrOffset), sizeof(void *));
        }
    }
}

void * sysHashTableProbeInBuchet_i(sysHashTable_S *pHashTable, int bucketIdx, void *pStruct)
{
    void *pElement;

    pElement = pHashTable->buckets[bucketIdx];
    while (pElement)
    {
        if ((pHashTable->compFun)(pElement, pStruct))
            return (pElement);

        memcpy(&pElement, (void *) (((char *) pElement) + pHashTable->storageNextPtrOffset), sizeof(void *));
    }

    return (NULL);
}

void * sysHashTableProbe(sysHashTable_S *pHashTable, void *pStruct)
{
    int hashValue;

    hashValue = (pHashTable->hashFun)(pStruct) % pHashTable->bucketNum;
    return (sysHashTableProbeInBuchet_i(pHashTable, hashValue, pStruct));
}

bool_T sysHashTableFind(sysHashTable_S *pHashTable, void *pStruct)
{
    if (sysHashTableProbe(pHashTable, pStruct))
        return (TRUE);
    else
        return (FALSE);
}

bool_T sysHashTableAdd(sysHashTable_S *pHashTable, void *pStruct)
{
    int hashValue;

/*
    if (sysHashTableFind(pHashTable, pStruct))
        return (FALSE);
*/

    hashValue = ((pHashTable->hashFun)(pStruct)) % pHashTable->bucketNum;
    
    memcpy((((char *) pStruct) + pHashTable->storageNextPtrOffset),
           &(pHashTable->buckets[hashValue]), sizeof(void *));
    memcpy(&(pHashTable->buckets[hashValue]), &pStruct, sizeof(void *));
    return (TRUE);
}

bool_T sysHashTableDelete(sysHashTable_S *pHashTable, void *pStruct)
{
    /* pElement->qElement */

    int hashValue;
    void *pElement, *qElement;

    hashValue = ((pHashTable->hashFun)(pStruct)) % pHashTable->bucketNum;

    pElement = pHashTable->buckets[hashValue];
    if (!pElement)
        return (FALSE);

    /* examine the head of the bucket */
    if ((pHashTable->compFun)(pElement, pStruct))
    {
        memcpy(&(pHashTable->buckets[hashValue]),
                (void *) (((char *) pElement) + pHashTable->storageNextPtrOffset), sizeof(void *));
        return (TRUE);
    }
    else
    { /* examine all the others */
        memcpy(&qElement, (void *) (((char *) pElement) + pHashTable->storageNextPtrOffset), sizeof(void *));

        if (qElement == NULL)
            return (FALSE);


        while (qElement)
        {
            if ((pHashTable->compFun)(qElement, pStruct))
            {
                memcpy((void *) (((char *) pElement) + pHashTable->storageNextPtrOffset),
                       (void *) (((char *) qElement) + pHashTable->storageNextPtrOffset), sizeof(void *));
                return (TRUE);
            }
            else
            {
                pElement = qElement;

                memcpy(&qElement, (void *) (((char *) qElement) + pHashTable->storageNextPtrOffset), sizeof(void *));
            }
        }

        return (FALSE);
    }
}


/*-------------------------------------------------------------------*/

sysDoubleHashTable_S *
sysDoubleHashTableInit(int storageStructSize,
						int storageNextPtrOffset,
						int bucket1Num,
						int bucket2Num,
						sysHashCompare_FT	compFun,
						sysHashHash_FT		hashFun1,
						sysHashHash_FT		hashFun2,
						sysHashDumpOne_FT	dumpOneFun)
{
	int i;
	sysDoubleHashTable_S *pTable;

	pTable = (sysDoubleHashTable_S *) sysMalloc(sizeof(sysDoubleHashTable_S));
	memset(pTable, 0, sizeof(sysDoubleHashTable_S));
	
	pTable->bucket1Num = bucket1Num;
	pTable->hashFun1 = hashFun1;
	

	pTable->buckets_1 = (sysHashTable_S **) sysMalloc(sizeof(sysHashTable_S *) * bucket1Num);  
	for (i = 0; i < bucket1Num; ++i)
	{
		pTable->buckets_1[i] = sysHashTableInit(storageStructSize,
											storageNextPtrOffset,
											bucket2Num,
											compFun,
											hashFun2,
											dumpOneFun);
	}
		
	
	return (pTable);
}

void sysDoubleHashTableExit(sysDoubleHashTable_S *pTable)
{
	int i;

	for (i = 0; i < pTable->bucket1Num; ++i)
		sysHashTableExit(pTable->buckets_1[i]);

	sysFree(pTable->buckets_1);
	sysFree(pTable);
}

void sysDoubleHashTableDump(sysDoubleHashTable_S *pTable)
{
	int i;

	for (i = 0; i < pTable->bucket1Num; ++i)
		sysHashTableDump(pTable->buckets_1[i]);
}

void * sysDoubleHashTableProbe(sysDoubleHashTable_S *pTable, void *pStruct)
{
	int hashValue = (pTable->hashFun1)(pStruct) % pTable->bucket1Num;
	return (sysHashTableProbe(pTable->buckets_1[hashValue], pStruct));
}

bool_T sysDoubleHashTableFind(sysDoubleHashTable_S *pTable, void *pStruct)
{
    if (sysDoubleHashTableProbe(pTable, pStruct))
        return (TRUE);
    else
        return (FALSE);
}

bool_T sysDoubleHashTableAdd(sysDoubleHashTable_S *pTable, void *pStruct)
{
	int hashValue = (pTable->hashFun1)(pStruct) % pTable->bucket1Num;
	return (sysHashTableAdd(pTable->buckets_1[hashValue], pStruct));
}

bool_T sysDoubleHashTableDelete(sysDoubleHashTable_S *pTable, void *pStruct)
{
	int hashValue = (pTable->hashFun1)(pStruct) % pTable->bucket1Num;
	return (sysHashTableDelete(pTable->buckets_1[hashValue], pStruct));
}



#if 0
/* --------------------- example ----------------------- */
typedef struct myElem
{
    int i;
    struct myElem *next;
} myElem;

#include "stdio.h"
void myDumpOne(myElem *p)
{
    printf("i = %d\n", p->i);
}

int myCompare(myElem *p, myElem *q)
{
    return ((p->i == q->i) ? 1 : 0);
}

int myHash(myElem *p)
{
    return (p->i);
}

int main()
{
#define SIZE  10
    int i;
    struct sysHashTable_S *pMyHashTable;
    myElem myElements[SIZE];

    for (i = 0; i <SIZE; ++i)
        myElements[i].i = SIZE - i -1;

    pMyHashTable = UHASH_TABLE_INIT(myElem,
                                    next,
                                    5,
                                    (sysHashCompare_FT) myCompare,
                                    (sysHashHash_FT)    myHash,
                                    (sysHashDumpOne_FT) myDumpOne);
                                  
    printf("add %d elements\n", SIZE);
    for (i = 0; i < SIZE; ++i)
        sysHashTableAdd(pMyHashTable, &myElements[i]);
    sysHashTableDump(pMyHashTable);

    printf("delete elements 0.. 4\n");
    for (i = 0; i < 5; ++i)
    {
        printf("delete -> ");
        myDumpOne(&myElements[i]);
        sysHashTableDelete(pMyHashTable, &myElements[i]);
    }
    sysHashTableDump(pMyHashTable);

    sysHashTableExit(pMyHashTable);
}
#endif

