#include "sys.h"
#include "hal_mmp2.h"
#include "hal.h"
#include "halVcap.h"
#include "10x16font.h"


char_font_map_t char_font_10x16_map_tbl[] =
{
	/* { 'character', font idex in  font_code[]} */
	{'0', 0},{'1', 1},{'2', 2},{'3', 3},{'4', 4},{'5', 5},{'6', 6},{'7', 7},{'8', 8},{'9', 9},{'A', 10},{'B', 11},{'C', 12},
	{'D', 13},{'E', 14},{'F', 15},{'G', 16},{'H', 17},{'I', 18},{'J', 19},{'K', 20},{'L', 21},{'M', 22},{'N', 23},{'O', 24},
	{'P', 25},{'Q', 26},{'R', 27},{'S', 28},{'T', 29},{'U', 30},{'V', 31},{'W', 32},{'X', 33},{'Y', 34},{'Z', 35},{' ', 36},
	{'=', 37},	{',', 38}, {'-', 39},	{'/', 40},	{':', 41},
};

/* font with size 10x16  */
unsigned short font_10x16_code[] =
{
//Char:[0]   0
0x0000,0x0000,0x00e0,0x01b0,0x0318,0x0318,0x0318,0x0318,0x0318,0x0318,0x0318,0x0318,0x01b0,0x00e0,0x0000,0x0000,
//Char:[1]   1
0x0000,0x0000,0x01e0,0x0060,0x0060,0x0060,0x0060,0x0060,0x0060,0x0060,0x0060,0x0060,0x0060,0x01f8,0x0000,0x0000,
//Char:[2]   2
0x0000,0x0000,0x01f0,0x0318,0x0318,0x0318,0x0030,0x0060,0x00c0,0x0180,0x0100,0x0398,0x0358,0x0330,0x0000,0x0000,
//Char:[3]   3
0x0000,0x0000,0x03f8,0x0318,0x0330,0x0330,0x0060,0x00e0,0x0030,0x0018,0x0018,0x0318,0x0330,0x01e0,0x0000,0x0000,
//Char:[4]   4
0x0000,0x0000,0x0030,0x0070,0x0070,0x00f0,0x00b0,0x01b0,0x0130,0x0330,0x03f8,0x0030,0x0030,0x00f8,0x0000,0x0000,
//Char:[5]   5
0x0000,0x0000,0x03f8,0x0318,0x0318,0x0300,0x03e0,0x0330,0x0318,0x0018,0x0018,0x0318,0x0330,0x01e0,0x0000,0x0000,
//Char:[6]   6
0x0000,0x0000,0x00f0,0x0198,0x0318,0x0300,0x03e0,0x03b0,0x0318,0x0318,0x0318,0x0318,0x01b0,0x00e0,0x0000,0x0000,
//Char:[7]   7
0x0000,0x0000,0x03d8,0x0338,0x0318,0x0310,0x0030,0x0030,0x0030,0x0060,0x0060,0x0060,0x0060,0x0060,0x0000,0x0000,
//Char:[8]   8
0x0000,0x0000,0x00e0,0x01b0,0x0318,0x0318,0x01b0,0x00e0,0x01b0,0x0318,0x0318,0x0318,0x01b0,0x00e0,0x0000,0x0000,
//Char:[9]   9
0x0000,0x0000,0x00e0,0x01b0,0x0318,0x0318,0x0318,0x0318,0x01b8,0x00d8,0x0018,0x0318,0x0330,0x01e0,0x0000,0x0000,
//Char:[A]   10
0x0000,0x0000,0x01c0,0x0040,0x00e0,0x00e0,0x00a0,0x00a0,0x01b0,0x01f0,0x01b0,0x01b0,0x01b0,0x03b8,0x0000,0x0000,
//Char:[B]   11
0x0000,0x0000,0x03f0,0x0198,0x0198,0x0198,0x0198,0x01f0,0x0198,0x0198,0x0198,0x0198,0x0198,0x03f0,0x0000,0x0000,
//Char:[C]   12
0x0000,0x0000,0x00e8,0x01b8,0x0318,0x0308,0x0308,0x0300,0x0300,0x0300,0x0300,0x0308,0x0198,0x00f0,0x0000,0x0000,
//Char:[D]   13
0x0000,0x0000,0x03e0,0x01b0,0x0198,0x0198,0x0198,0x0198,0x0198,0x0198,0x0198,0x0198,0x01b0,0x03e0,0x0000,0x0000,
//Char:[E]   14
0x0000,0x0000,0x03f8,0x0198,0x0198,0x0180,0x0190,0x01f0,0x0190,0x0190,0x0180,0x0198,0x0198,0x03f8,0x0000,0x0000,
//Char:[F]   15
0x0000,0x0000,0x03f8,0x0198,0x0198,0x0180,0x0190,0x01f0,0x0190,0x0190,0x0180,0x0180,0x0180,0x03e0,0x0000,0x0000,
//Char:[G]   16
0x0000,0x0000,0x00d8,0x01b8,0x0318,0x0318,0x0300,0x0300,0x0378,0x0318,0x0318,0x0318,0x01b8,0x00d8,0x0000,0x0000,
//Char:[H]   17
0x0000,0x0000,0x0318,0x0318,0x0318,0x0318,0x0318,0x03f8,0x0318,0x0318,0x0318,0x0318,0x0318,0x0318,0x0000,0x0000,
//Char:[I]   18
0x0000,0x0000,0x03f0,0x00c0,0x00c0,0x00c0,0x00c0,0x00c0,0x00c0,0x00c0,0x00c0,0x00c0,0x00c0,0x03f0,0x0000,0x0000,
//Char:[J]   19
0x0000,0x0000,0x01f8,0x0060,0x0060,0x0060,0x0060,0x0060,0x0060,0x0060,0x0360,0x0360,0x0360,0x03c0,0x0000,0x0000,
//Char:[K]   20
0x0000,0x0000,0x03b8,0x0198,0x01b0,0x01b0,0x01e0,0x01c0,0x01e0,0x01b0,0x01b0,0x0198,0x0198,0x03b8,0x0000,0x0000,
//Char:[L]   21
0x0000,0x0000,0x03c0,0x0180,0x0180,0x0180,0x0180,0x0180,0x0180,0x0180,0x0198,0x0198,0x0198,0x03f8,0x0000,0x0000,
//Char:[M]   22
0x0000,0x0000,0x0318,0x0318,0x03b8,0x03b8,0x03b8,0x03f8,0x0358,0x0358,0x0358,0x0318,0x0318,0x0318,0x0000,0x0000,
//Char:[N]   23
0x0000,0x0000,0x03b8,0x0198,0x0198,0x01d8,0x01d8,0x01d8,0x01b8,0x01b8,0x01b8,0x0198,0x0198,0x03d8,0x0000,0x0000,
//Char:[O]   24
0x0000,0x0000,0x00e0,0x01b0,0x0318,0x0318,0x0318,0x0318,0x0318,0x0318,0x0318,0x0318,0x01b0,0x00e0,0x0000,0x0000,
//Char:[P]   25
0x0000,0x0000,0x03e0,0x01b0,0x0198,0x0198,0x0198,0x0198,0x01b0,0x01e0,0x0180,0x0180,0x0180,0x03e0,0x0000,0x0000,
//Char:[Q]   26
0x0000,0x0000,0x00e0,0x01b0,0x0318,0x0318,0x0318,0x0318,0x0318,0x0318,0x0358,0x03b8,0x01b0,0x00f0,0x0018,0x0000,
//Char:[R]   27
0x0000,0x0000,0x03f0,0x0198,0x0198,0x0198,0x0198,0x01b0,0x01e0,0x01b0,0x0198,0x0198,0x0198,0x03d8,0x0000,0x0000,
//Char:[S]   28
0x0000,0x0000,0x00e8,0x01b8,0x0318,0x0308,0x0308,0x01c0,0x0070,0x0218,0x0218,0x0318,0x03b0,0x02e0,0x0000,0x0000,
//Char:[T]   29
0x0000,0x0000,0x01f8,0x0168,0x0168,0x0168,0x0060,0x0060,0x0060,0x0060,0x0060,0x0060,0x0060,0x00f0,0x0000,0x0000,
//Char:[U]   30
0x0000,0x0000,0x03b8,0x0198,0x0198,0x0198,0x0198,0x0198,0x0198,0x0198,0x0198,0x0198,0x0198,0x00f0,0x0000,0x0000,
//Char:[V]   31
0x0000,0x0000,0x03b8,0x01b0,0x01b0,0x01b0,0x01b0,0x01b0,0x01b0,0x00e0,0x00e0,0x00e0,0x0040,0x0040,0x0000,0x0000,
//Char:[W]   32
0x0000,0x0000,0x0318,0x0318,0x0318,0x0318,0x0358,0x0358,0x0358,0x03f8,0x01f0,0x01b0,0x01b0,0x01b0,0x0000,0x0000,
//Char:[X]   33
0x0000,0x0000,0x03b8,0x03b8,0x0110,0x01b0,0x01b0,0x00e0,0x00e0,0x01b0,0x01b0,0x0110,0x03b8,0x03b8,0x0000,0x0000,
//Char:[Y]   34
0x0000,0x0000,0x03b8,0x0198,0x0198,0x0198,0x0198,0x00f0,0x0060,0x0060,0x0060,0x0060,0x0060,0x00f0,0x0000,0x0000,
//Char:[Z]   35
0x0000,0x0000,0x03f0,0x0330,0x0330,0x0360,0x0060,0x00c0,0x00c0,0x0180,0x01b0,0x0330,0x0330,0x03f0,0x0000,0x0000,
//Char:[ ]   36
0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
//Char:[=]   37
0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x03f8,0x0000,0x0000,0x03f8,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
//Char:[,]   38
0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x00c0,0x00c0,0x0040,0x0080,
//Char:[-]   39
0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x001e,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,0x0000,
//Char:[/]   40
0x0000,0x0000,0x0004,0x0004,0x0004,0x0008,0x0008,0x0008,0x0010,0x0010,0x0010,0x0020,0x0020,0x0020,0x0000,0x0000,
//Char:[:]   41
0x0000,0x0000,0x0000,0x0000,0x0006,0x0006,0x0000,0x0000,0x0000,0x0000,0x0006,0x0006,0x0000,0x0000,0x0000,0x0000,
};

int get_10x16_font_map_size()
{
	return sizeof(char_font_10x16_map_tbl) / sizeof(char_font_map_t);
}

int get_10x16_font_count()
{
	return sizeof(font_10x16_code) / (FONT_H * sizeof(short));
}

unsigned short get_font_idx(unsigned char *c, char_font_map_t *p, int tbl_size)
{
	int i;

	for (i=0; i<tbl_size; i++, p++)
		if (p->s==*c)
			return p->font_idx;
   return (FONT_NUM + 1);
}

void store_osdfont2fontRAM(void)
{
	unsigned short *pfont;
	unsigned short fontram_addr;
	unsigned short font_code_base;
	int font_idx, line;
	struct osdfont_s osdfont;

	for (font_idx=0; font_idx < CHAR_FONT_NUMBER; font_idx++)
	{
		font_code_base = font_idx*FONT_H;
		pfont = &FONT_CODE[font_code_base];
		fontram_addr = font_idx*24; /* RAM address to put font */
		for (line=0; line < 24; line++)
		{
			osdfont.data = (line<FONT_H) ? *pfont++ : 0;
			osdfont.addr = fontram_addr++;
#if 0
			if (font_idx == 15) // 'F'
			{
				printf("osdfont.data = %d, osdfont.addr = %d\n",osdfont.data, osdfont.addr);
			}
#endif
			if (write_osd_fontram(&osdfont)) return;
		}
	}
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////////

int vcap_reset(void)
{
	int i = 1000;
	int val = 1;

	HAL_SETREG(HAL_REG_VCAP_SYS_CTRL, 0x01);
	do
	{
		val = HAL_GETREG(HAL_REG_VCAP_SYS_CTRL)&0x01;
	}while( val && i-- );

	return (val)? -1 : 0;
}

void set_src_if(src_if_t src_if)
{
	int *src_if_data;

	src_if_data = (int *)&(src_if);

	HAL_SETREG(HAL_REG_VCAP_SRC_VIF, *src_if_data);
}

void set_vif_enable(int enable)
{
	HAL_SETREG(HAL_REG_VCAP_VIF_EN, enable);
}

void set_di_enable(int enable)
{
	HAL_SETREG(A1_REG_VCAP_DI_CTL, enable);
}

void set_lb_ctrl(lb_ctrl_t lb_ctrl)
{
	HAL_SETREG(A1_REG_VCAP_LINE_BUF_CTL, *(int *)&(lb_ctrl));
}

void set_src_size(src_size_t src_size)
{
	unsigned int src_size_data;
	src_size_data = src_size.src_w;
	src_size_data |= (src_size.src_h << 16);
	HAL_SETREG(HAL_REG_VCAP_SRC_SIZE, src_size_data);
}

void set_src_start(src_start_t src_start)
{
	unsigned int src_start_data;
	src_start_data = src_start.src_hstart;
	src_start_data |= (src_start.src_vstart << 16);
	HAL_SETREG(HAL_REG_VCAP_SRC_START, src_start_data);
}

void set_clip_size(clip_size_t clip_size)
{
	unsigned int clip_size_data;
	clip_size_data = clip_size.swc_w;
	clip_size_data |= (clip_size.swc_h << 16);
	HAL_SETREG(HAL_REG_VCAP_SRC_CLIP_SIZE, clip_size_data);
}

void set_clip_start(clip_start_t clip_start)
{
	unsigned int clip_start_data;
	clip_start_data = clip_start.swc_x;
	clip_start_data |= (clip_start.swc_y << 16);
	HAL_SETREG(HAL_REG_VCAP_SRC_CLIP_SIZE, clip_start_data);
}

void set_crop_size(ch_id_t chn, crop_size_t crop_size)
{
	unsigned int crop_size_data;
	crop_size_data = crop_size.crop_w;
	crop_size_data |= (crop_size.crop_h << 16);
	HAL_SETREG(HAL_REG_VCAP_CHm_CROP_SIZE(chn), crop_size_data);
}

void set_crop_offset(ch_id_t chn, crop_offset_t crop_offset)
{
	unsigned int crop_offset_data;
	crop_offset_data = crop_offset.crop_x;
	crop_offset_data |= (crop_offset.crop_y << 16);
	HAL_SETREG(HAL_REG_VCAP_CHm_CROP_OFFSET(chn), crop_offset_data);
}

void set_scaler(ch_id_t chn, scaler_t scaler)
{
	unsigned int data;

	HAL_SETREG(A1_REG_VCAP_CHm_SC_X(chn), ((4096*scaler.out_w) % scaler.src_w) ? ((4096*scaler.out_w) / scaler.src_w + 1) : ((4096*scaler.out_w) / scaler.src_w));
	HAL_SETREG(A1_REG_VCAP_CHm_SC_Y(chn), ((4096*scaler.out_h) % scaler.src_h) ? ((4096*scaler.out_h) / scaler.src_h + 1) : ((4096*scaler.out_h) / scaler.src_h));

	data = HAL_GETREG(A1_REG_VCAP_SC_CTRL);
	if(scaler.en)
		data |= 0x7 << (chn * 4);
	else
		data &= ~(0x7 << (chn * 4));
	HAL_SETREG(A1_REG_VCAP_SC_CTRL, data);
}

void set_src_blank(src_blank_t src_blank)
{
	unsigned int src_blank_data;
	src_blank_data = src_blank.hblank;
	src_blank_data |= (src_blank.vblank << 16);
	HAL_SETREG(HAL_REG_VCAP_SRC_BLANK, src_blank_data);
}
void set_channel_dma(ch_id_t chn, chn_dma_t chn_dma)
{
	HAL_SETREG(HAL_REG_VCAP_CHm_DMA(chn), *(int *)&(chn_dma));
}

void set_ybuf_addr(ch_id_t chn, fb_id_t fb, unsigned int addr)
{
	HAL_SETREG(HAL_REG_VCAP_CHm_FBn_DESTY(chn, fb), addr);
}

void set_uvbuf_addr(ch_id_t chn, fb_id_t fb, unsigned int addr)
{
	HAL_SETREG(HAL_REG_VCAP_CHm_FBn_DESTC(chn, fb), addr);
}

void cap_ctrl(unsigned short on_off)
{
	HAL_SETREG(HAL_REG_VCAP_CAP_CTRL ,on_off);
}

void reg_update(void)
{
	HAL_SETREG(HAL_REG_VCAP_REG_UPDATE ,0x00100000);
}

unsigned int query_int_sts(void)
{ 
	return HAL_GETREG(HAL_REG_VCAP_INT_STS);
}

void write_int_sts_clr(unsigned short bit)
{
	HAL_SETREG(HAL_REG_VCAP_INT_STS_CLR, 1<<bit);
}

unsigned int query_int_mask_sts(void)
{
	return HAL_GETREG(HAL_REG_VCAP_INT_MASK_STS);
}

/* Enable specific interrupt */
void unmask_int(unsigned short bit)
{
	HAL_SETREG(HAL_REG_VCAP_INT_MASK_SET, 1<<bit);
}

/* Disable all interrupts */
void mask_all_int(void)
{
	HAL_SETREG(HAL_REG_VCAP_INT_MASK_CLR, 0xFFFFFFFF);
}

void set_int_pri(unsigned short bit, int_pri_t priority)
{
	unsigned int val = HAL_GETREG(HAL_REG_VCAP_INT_PRI);
	HAL_SETREG(HAL_REG_VCAP_INT_PRI, (priority == high_pri)? val | (1<<bit) : val & (~(1<<bit)) );
}

void set_buf_own(ch_id_t chn, fb_id_t fb)
{
	unsigned int val = HAL_GETREG(HAL_REG_VCAP_CHm_DEST_OWN(chn));
	val |= (1<<fb);
	HAL_SETREG(HAL_REG_VCAP_CHm_DEST_OWN(chn), val);
}

short get_buf_own_value(ch_id_t chn, fb_id_t fb)
{
	/*  return either 0 or 1.
		0: Hardware has cleared it 
	*/
	return  (1<<fb) & HAL_GETREG(HAL_REG_VCAP_CHm_DEST_OWN(chn));
}


//////////////////////////////////////////////////////////////////////////////////
//// porting from Linux driver by Ted
//////////////////////////////////////////////////////////////////////////////////
#define WIN_NUM 8

typedef struct win_pos_order_s
{
        int x;
        int y;
        int id;
} win_pos_order_t;

void swap_order(win_pos_order_t *x, win_pos_order_t *y)
{
	win_pos_order_t temp;
	temp = *x;
	*x = *y;
	*y = temp;
}

void bublesort(win_pos_order_t win_pos[],  int n)
{
	int i, j;

	for(i=0;i<(n-1);i++)
	{
		for(j=0;j<(n-(i+1));j++)
		{
			if(win_pos[j].x > win_pos[j+1].x)
				swap_order(&win_pos[j],&win_pos[j+1]);
			else if (win_pos[j].x == win_pos[j+1].x)
				if (win_pos[j].y > win_pos[j+1].y)
					swap_order(&win_pos[j],&win_pos[j+1]);
		}
	}
}

//// OSG////////////////////////////////////////////////////////
//// OSG////////////////////////////////////////////////////////
//// OSG ///////////////////////////////////////////////////////
//// OSG ///////////////////////////////////////////////////////

int cfg_osg_win_cor(struct osg_win_cor_s *p)
{
#if 0
	unsigned int val = HAL_GETREG(A1_REG_OSG_WIN_COR(p->win_id) );
	val |= p->key_cr | (p->key_cb<<8) | (p->key_y<<16) |
	(p->colorkey_trans<<24) | (p->noncolorkey_trans<<28) | (p->colorkey_en<<26);
#else
	unsigned int val = p->key_cr | (p->key_cb<<8) | (p->key_y<<16) |
	(p->colorkey_trans<<24) | (p->noncolorkey_trans<<28) | (p->colorkey_en<<26);
#endif

	writel(val, A1_REG_OSG_WIN_COR(p->win_id) );

	return 0;
}

int cfg_osg_win_src(struct osg_win_src_s *p)
{
	writel(p->win_src, A1_REG_OSG_WIN_SRC(p->win_id) );
	return 0;
}

int cfg_osg_win_size(struct osg_win_size_s *p)
{

	writel( (p->win_w-1) | ( (p->win_h-1)<<16 ), A1_REG_OSG_WIN_SIZE(p->win_id) );

	KDUMP("win_w = %d, win_h = %d\n", p->win_w, p->win_h);
	return 0;
}

int cfg_osg_win_pos(int chn, struct osg_win_pos_s *p)
{
	win_pos_order_t win_pos_list[WIN_NUM];
	int i;
	unsigned val;

	writel( p->x_pos | (p->y_pos<<16), A1_REG_CHm_OSG_WOFFSETn(chn, p->win_id) );

	for (i=0; i<WIN_NUM; i++) /* here, i is equal to win id */
	{
		val = HAL_GETREG(A1_REG_CHm_OSG_WOFFSETn(chn, i));

		win_pos_list[i].id = i;
		win_pos_list[i].x = val & 0xFFF; /* bit0 ~ bit11 */
		win_pos_list[i].y = val >>16; /* bit16 ~ bit27 */

	}

	/* sort order of windows based on x/y position. */
	bublesort(win_pos_list, WIN_NUM);
	val = 0;
	for (i=0; i<WIN_NUM; i++) /* here, i is equal to order */
		val |= ( win_pos_list[i].id << (4*i) );

	writel(val, A1_REG_CHm_OSG_WIN_ORDER(chn));
	return 0;
}

int osg_on_off(int chn, struct osg_win_en_s *p)
{
	unsigned int val;

	if(p->en!=0)
		val = 1<<p->win_id;
	else
		val = 1<<(p->win_id + 8);
	writel(val , A1_REG_OSG_WIN_EN(chn) );

	return 0;
}

#if 0
int retrieve_osg_bufs(struct soc_camera_file *icf, struct osg_dma_buf_s *p)
{
	struct videobuf_queue *q = &icf->vb_vidq;
	/* dma_addr_t videobuf_to_dma_contig(struct videobuf_buffer *buf) */

	if (q->bufs[p->buf_id])
	{
		p->buf_phy = videobuf_to_dma_contig(q->bufs[p->buf_id]);
		KDUMP("p->buf_phy = 0x%0x\n", p->buf_phy);
	}
	else
		return -1;

	return 0;
}
#endif

//// OSD ///////////////////////////////////////////////////////
//// OSD ///////////////////////////////////////////////////////
//// OSD ///////////////////////////////////////////////////////
//// OSD ///////////////////////////////////////////////////////

int cfg_osd_font_size(struct osd_font_size_s *p)
{
#if 0
	if (p->font_w%2 == 0)
	{
		if (p->fspace_col == 0)	p->fspace_col = 2;
		if (p->fspace_col%2)	p->fspace_col--;
	}
	else
	{
		if (p->fspace_col == 0)	p->fspace_col = 1;
		if ( p->fspace_col%2 == 0 )	p->fspace_col--;
	}

	if (p->font_h%2 == 0)
	{
		if (p->fspace_row == 0)	p->fspace_row = 2;
		if (p->fspace_row%2)	p->fspace_row--;
	}
	else
	{
		if (p->fspace_row == 0)	p->fspace_row = 1;
		if ( p->fspace_row%2 == 0 )	p->fspace_row--;
	}
#else
	if (p->fspace_col == 0) p->fspace_col = 1;
	if (p->fspace_row == 0) p->fspace_row = 1;
#endif

	//writel(p->font_w | (p->font_h<<8) | (p->fspace_col<<16) | (p->fspace_row<<24)  ,A1_REG_OSD_FSIZE(p->win_id) );
	writel((p->font_w -1) | ( (p->font_h-1)<<8) | ( (p->fspace_col-1)<<16) | ( (p->fspace_row-1)<<24), A1_REG_OSD_FSIZE(p->win_id) );
	return 0;
}

int cfg_osd_win_size(struct osd_win_size_s *p)
{
	//writel(p->win_w | (p->win_h<<8) | (p->disaddr<<16)   ,A1_REG_OSD_WSIZE(p->win_id) );

	writel( (p->HDim - 1) | ( (p->VDim-1)<<8) | ( (0x1FF<<16) & HAL_GETREG(A1_REG_OSD_WSIZE(p->win_id)) )   ,A1_REG_OSD_WSIZE(p->win_id) );
	return 0;
}

int cfg_osd_win_cor(struct osd_win_cor_s *p)
{
	writel(p->fgColor | (p->bgColor<<4) |(p->fgtrans<<8) | (p->bgtrans<<10), A1_REG_OSDCOR(p->win_id) );
	return 0;
}

int cfg_osd_win_pos(int chn, struct osd_win_pos_s  *p)
{
	writel(p->x_pos | (p->y_pos<<16), A1_REG_CHm_OSD_WOFFSETn(chn, p->win_id));
	return 0;
}

typedef struct chn_win_bit_s
{
	int chn;
	int win_id;
	int value;
	unsigned int reg;
}	chn_win_bit_t;

int set_chn_win_bit(chn_win_bit_t chn_win_data)
{
	unsigned int val = HAL_GETREG(chn_win_data.reg);
	if (chn_win_data.value==1)
		val |=  CH_WIN_BIT(chn_win_data.chn, chn_win_data.win_id);
	else if (chn_win_data.value == 0)
		val &= (~CH_WIN_BIT(chn_win_data.chn,  chn_win_data.win_id));
	writel(val, chn_win_data.reg);
	return 0;
}

int osd_win_on_off(int chn, struct osd_win_en_s *p)
{
	//chn_win_bit_t	chn_win_data;
	 unsigned int val = CH_WIN_BIT(chn,  p->win_id);

	if ( p->en == 0)
	{
		writel(val, A1_REG_OSDDISABLE);
	}
	else
	{
#if 0
		chn_win_data.chn = chn;
		chn_win_data.win_id = p->win_id;
		chn_win_data.value = p->en;
		chn_win_data.reg = A1_REG_OSD_EN;
		return set_chn_win_bit(chn_win_data);
#endif
		writel(val, A1_REG_OSD_EN);
	}
	return 0;
}

int write_osd_fontram(struct osdfont_s *p)
{
	unsigned int val = p->data | (p->addr<<16) | 0x80000000;
	writel(val, A1_REG_OSDFONT);
	return 0;
}

int write_win_osd_string(struct osd_win_font_s *p)
{
	int i;
	unsigned short font_idx;
	unsigned int val;

	for (i=0; i<p->numChar; i++)
	{
//		if (memcpy (&font_idx, (void *) p->font_idx_p++, sizeof(unsigned short)) )
//			return -1;
		font_idx = p->font_idx_p[i];
		val = font_idx | ( (p->Start+i)<<16 ) | 0x80000000;
		//printk("font_idx = %d\n", font_idx);
		writel(val, A1_REG_OSDDISP);
	}
	writel(  (0xFFFF & HAL_GETREG(A1_REG_OSD_WSIZE(p->win_id)))  | ( (p->Start&0x1FF)<<16 )   ,A1_REG_OSD_WSIZE(p->win_id) );
	return 0;
}

int cfg_palette(struct osd_palette_s *p)
{
	writel( (p->Y<<16) | (p->Cb<<8) | p->Cr , A1_REG_PALETTE_COLOR(p->index) );
	return 0;
}

//// MASK ///////////////////////////////////////////////////////
//// MASK ///////////////////////////////////////////////////////
//// MASK ///////////////////////////////////////////////////////
//// MASK ///////////////////////////////////////////////////////

int mask_on_off(int chn, struct msken_s *p)
{
	(p->en == 1) ? a1_writel( CH_WIN_BIT(chn, p->win_id) , A1_REG_MSKEN ) : \
			a1_writel( CH_WIN_BIT(chn, p->win_id) , A1_REG_MSKDISABLE );

	return 0;
}

int cfg_mask_win_dim(int chn, struct msk_dim_s *p)
{
	a1_writel( p->x_pos | (p->y_pos<<16) , A1_REG_CHmMSKn_START(chn, p->win_id) );
	a1_writel( (p->x_width - 1) | (p->y_end<<16) , A1_REG_CHmMSKn_END(chn, p->win_id) );

	return 0;
}

int cfg_mask_win_mode(int chn, struct msk_mode_s *p)
{
	unsigned int val = 0;

	val = HAL_GETREG(A1_REG_MSK_MODE);
	val = (p->mode == 1) ? (val | CH_WIN_BIT(chn, p->win_id)) : (val & (~CH_WIN_BIT(chn,  p->win_id)));
	a1_writel(val, A1_REG_MSK_MODE);

	return 0;
}

int cfg_msk_win_cor(int chn, struct msk_color_s *p)
{
	unsigned int val =  ~(0x7 <<(p->win_id*4) );

	a1_writel( (p->palette_idx << (p->win_id*4)) | (val & HAL_GETREG(A1_REG_CHmMSKCOR(chn))), A1_REG_CHmMSKCOR(chn) );

	return 0;
}

int cfg_scramble_usrkey(int chn, struct scramble_usrkey_s *p)
{
	a1_writel(p->scramble_key , A1_REG_CHmSCRAMBLE_TAB(chn));

	return 0;
}

int cfg_msk(int chn, struct msk_setting_s *p)
{
	int i, bit;
	unsigned int en_reg, start_reg[8], end_reg[8], mode_reg, color_reg, temp_reg;

	en_reg = mode_reg = color_reg = 0;
	for(i = 0, bit = 1; i < 8; i++, bit <<= 1)
	{
		en_reg |= (p->win[i].en == 1) ? bit : 0;
		mode_reg |= (p->win[i].mode == 1) ? bit : 0;
		start_reg[i] = p->win[i].x_pos | (p->win[i].y_pos << 16);
		end_reg[i] = (p->win[i].x_width - 1) | (p->win[i].y_end << 16);

		color_reg |= p->win[i].palette_id  << (i * 4);
	}

	// mode
	temp_reg = HAL_GETREG(A1_REG_MSK_MODE);
	temp_reg &= ~(0xFF << (p->chn * 8));
	temp_reg |= mode_reg  << (p->chn * 8);
	a1_writel(temp_reg, A1_REG_MSK_MODE);

	// color
	a1_writel(color_reg, A1_REG_CHmMSKCOR(p->chn));

	// pos
	for(i = 0, bit = 1; i < 8; i++, bit <<= 1)
	{
		a1_writel( start_reg[i], A1_REG_CHmMSKn_START(p->chn, i) );
		a1_writel( end_reg[i] , A1_REG_CHmMSKn_END(p->chn, i) );
	}
	// scramble_key
	a1_writel(p->scramble_key , A1_REG_CHmSCRAMBLE_TAB(p->chn));

	// enable
	a1_writel(0xFF << (p->chn * 8), A1_REG_MSKDISABLE);
	a1_writel(en_reg << (p->chn * 8), A1_REG_MSKEN);

	return 0;
}

int cfg_frc_ctl(int chn, int fps)
{
	unsigned int val = 60<<16; /*  Max value 60 */

	 /* Step size */
	if (fps == 0)
	{
		writel(0 , A1_REG_FRC_CTL(chn));
		return 0;
	}
	else
		val |= fps * 2;

	if (chn==1) val |= (15<<8); /* init value */
	else if (chn==2) val |= (30<<8); /* init value */
	else if (chn==3) val |= (45<<8); /* init value */

	val |= (0x10000000);

	writel(val , A1_REG_FRC_CTL(chn));

	return 0;
}

/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////
vcap_set_palette(int y[], int u[], int v[])
{
	struct osd_palette_s pal;
	int i;

	for (i=0; i<8; i++)
	{
		pal.index = i;
		pal.Y = y[i];
		pal.Cb= u[i];
		pal.Cr= v[i];
		cfg_palette(&pal);
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//// OSD testing, porting from Linux workspace sample by Ted
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int vcap_set_OSD_string(struct osd_win_str_s *osdwin_str)
{
	unsigned short font_idx;
	int i;
	struct osd_win_font_s win_font;
	int tbl_size = CHAR_FONT_MAP_SIZE;
	int tempbuf[1024 * 10];

	win_font.Start = osdwin_str->Start;
	win_font.numChar = osdwin_str->numChar;
	win_font.win_id = osdwin_str->win_id;
	win_font.font_idx_p = tempbuf;

	for (i=0; i<win_font.numChar; i++)
	{
		font_idx = get_font_idx(&osdwin_str->str[i], CHAR_FONT_MAP_TBL, tbl_size);
		if (font_idx > FONT_NUM)
		{
			//halUARTPrintf("character %c has invalid font code\n",osdwin_str->str[i]);
			return -1;
		}
		else
			win_font.font_idx_p[i] = font_idx;
	}

	write_win_osd_string(&win_font);

	return 0;
}

int vcap_set_OSD_window(osd_properties_t property[8])
{
	struct osd_win_cor_s osd_color;
	struct osd_win_str_s osd_str;
	struct osd_font_size_s font_size;
	struct osd_win_size_s osd_size;
	int win, val, display_addr;


	store_osdfont2fontRAM();

	for (win=0; win<8; win++)
	{
		font_size.win_id = win;
		font_size.font_w = FONT_W;
		font_size.font_h = FONT_H ;
		font_size.fspace_col = property[win].font_column_space;
		font_size.fspace_row = property[win].font_row_space;
		cfg_osd_font_size(&font_size);

		osd_size.win_id = win;
		osd_size.HDim = property[win].osd_horizontal_width;
		osd_size.VDim = property[win].osd_vertical_height;
		cfg_osd_win_size(&osd_size);

		osd_color.win_id = win;
		osd_color.fgColor = property[win].foreground_color;
		osd_color.bgColor = property[win].background_color;
		osd_color.fgtrans = property[win].foreground_transparency;
		osd_color.bgtrans = property[win].background_transparency;
		cfg_osd_win_cor(&osd_color);
	}

	display_addr = 0;
	for (win=0; win<8; win++)
	{
		osd_str.win_id = win;
		osd_str.Start = display_addr;
		osd_str.numChar = property[win].osd_horizontal_width * property[win].osd_vertical_height;
		display_addr += osd_str.numChar;
		memcpy(osd_str.str, property[win].osd_str, 64);
		if(osd_str.numChar == 0) continue;
		if((val=vcap_set_OSD_string(&osd_str)))
		{
			//halUARTPrintf("win:%d set_osd_string: errorno= %d\n\r", win, val);
			return -1;
		}
	}
}

int vcap_set_OSD_channel(int chn, vcap_pos_t pos[8], int enable[8])
{
	struct osd_win_pos_s osd_pos;
	struct osd_win_en_s win_on_off;
	int win;


	for (win=0; win<8; win++)
	{
		osd_pos.x_pos = pos[win].x_pos;
		osd_pos.y_pos = pos[win].y_pos;
		osd_pos.win_id = win;
		cfg_osd_win_pos(chn, &osd_pos);
	}

	win_on_off.chn= chn;
	for (win=0; win<8; win++)
	{
		win_on_off.win_id = win;
		win_on_off.en = enable[win];
		osd_win_on_off(chn, &win_on_off);
	}
	return 0;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//// OSG testing, porting from Linux workspace sample by Ted
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
int vcap_set_OSG_window(char *buf, osg_properties_t property[8])
{
	struct osg_win_cor_s win_color;
	struct osg_win_size_s win_size;
	struct osg_win_src_s osgwin_src;
	int win, osg_buf_size = 0;

	for (win=0; win<8; win++)
	{
		win_color.win_id = win;
		win_color.key_cr = property[win].key_cr;
		win_color.key_cb = property[win].key_cb;
		win_color.key_y = property[win].key_y;
		win_color.colorkey_trans = property[win].colorkey_trans;
		win_color.colorkey_en = property[win].colorkey_en;
		win_color.noncolorkey_trans = property[win].noncolorkey_trans;
		cfg_osg_win_cor(&win_color);
	}

	for (win=0; win<8; win++)
	{
		win_size.win_id = win;
		win_size.win_w = property[win].win_w;
		win_size.win_h = property[win].win_h;
		cfg_osg_win_size(&win_size);
		osg_buf_size +=  (property[win].win_w * property[win].win_h * 2 / 32 + 1) * 32;
	}

	for (win=0; win<8; win++)
	{
		osgwin_src.win_id = win;
		osgwin_src.win_src = buf;
		cfg_osg_win_src(&osgwin_src);
		buf += (property[win].win_w*property[win].win_h * 2 / 32 + 1) * 32;
	}

	return 0;
}

int vcap_set_OSG_channel(int chn, vcap_pos_t pos[8], int enable[8])
{
	struct osg_win_pos_s osg_pos;
	struct osg_win_en_s win_en;
	int win;

	// use AF0/AF4/AF8/AFC as OSG windows enable registers
	HAL_SETREG(A1_REG_VCAP_OSG_WIN_EN_SEL, 0x0);

	win_en.chn = chn;
	for (win=0; win<8; win++)
	{
		win_en.win_id = win;
		win_en.en = 0; //Turn off all windows.
		osg_on_off(chn, &win_en);
	}

	osg_pos.chn = chn;
	for (win=0; win<8; win++)
	{
		osg_pos.x_pos = pos[win].x_pos;
		osg_pos.y_pos = pos[win].y_pos;
		osg_pos.win_id = win;
		cfg_osg_win_pos(chn, &osg_pos);
	}

	win_en.chn = chn;
	for (win=0; win<8; win++)
	{
		win_en.win_id = win;
		win_en.en = enable[win];
		osg_on_off(chn, &win_en);
	}
	return 0;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//// PMASK testing, porting from Linux workspace sample by Ted
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#define SCRAMBLE_UNIT_SIZE 	4
#define SKEY_SIZE				8
#define SKEY_USED				1
#define SKEY_NOT_USED			0

static unsigned int DEFAULT_SKEY[] = { 4, 0, 3, 5, 7, 2, 6, 1}; // why this? just random selected by Ted //

/* translate scramble table to hw scramble key */
static unsigned int skgen_trans_hwkey(unsigned int skey[])
{
	unsigned int i, hwkey=0;

	for(i = 0; i < SKEY_SIZE; i++)
	{
		hwkey += skey[i] << ((SKEY_SIZE-i-1) * 3);
	}
	return hwkey;
}

/* to check the scramble table is OK or not */
static int skgen_check_key_table(unsigned int skey[])
{
	unsigned int i, check;


	/* check whether too many position is unchanged */
	check = 0;
	for(i = 0; i < SKEY_SIZE; i++)
	{
		if(skey[i] == i)
			check++;
	}

	if(check > (SKEY_SIZE / 2))
		return FALSE;

	/* check whether the position is only shift, 01234...  765432... */
	check = TRUE;
	for(i = 1; i < SKEY_SIZE; i++)
	{
		if((skey[i-1] != ((skey[i] + 1) % SKEY_SIZE)) && (((skey[i-1]+1) % SKEY_SIZE) != skey[i]))
		{
			check = FALSE;
			break;
		}
	}
	if(check == TRUE)
		return FALSE;

	/* check another */
	return TRUE;

}

static unsigned long int skgen_random(unsigned long key)
{
	return 	(unsigned long int)(key * 1103515245 + 12345);
}

static void skgen_gen_table(unsigned long user_key, unsigned int skey[])
{
	unsigned int used[SKEY_SIZE];
	unsigned int i, key, index;
	unsigned long random_key;

	for(i = 0; i < SKEY_SIZE; i++)
		used[i] = SKEY_NOT_USED;

	random_key = user_key;
	for(i = 0; i < SKEY_SIZE; i++)
	{
		random_key = skgen_random(random_key);
		index = random_key % (SKEY_SIZE - i);

		for(key = 0; key < SKEY_SIZE; key++)
		{
			if(used[key] == SKEY_NOT_USED)
			{
				if(index == 0)
					break;
				index--;
			}
		}
		skey[i] = key;
		used[key] = SKEY_USED;
	}

	/* if the scramble table is not suitable, use the default table */
	if(skgen_check_key_table(skey) == FALSE)
		memcpy((void *)skey, (const void *)DEFAULT_SKEY, SKEY_SIZE * sizeof(unsigned int));

}

static int skgen(unsigned long int user_key)
{
	unsigned int skey[SKEY_SIZE], hwkey;

	skgen_gen_table(user_key, skey);

	hwkey =skgen_trans_hwkey(skey);
	return hwkey;
}

int vcap_set_PMSK(int chn, vcap_pos_t pos[8], vcap_size_t size[8], int enable[8], int palette[8], int mode[8], int key)
{
	struct msk_setting_s msk_setting;
	struct msk_win_s *msk_win;
	int win;

	msk_setting.chn = chn;
	msk_setting.user_key = key;
//	msk_setting.scramble_key = skgen(msk_setting.user_key);
	msk_setting.scramble_key = (0 << 21) + (1 << 18) + (2 << 15) + (3 << 12) + (4 << 9) + (5 << 6) + (6 << 3) + (7 << 0);
	//halUARTPrintf("scramble: user key = 0x%08X ==> scramble key = 0x%08X\n\r",msk_setting.user_key, msk_setting.scramble_key);

	for (win=0; win<8; win++)
	{
		msk_win = &msk_setting.win[win];
		msk_win->en = enable[win];
		msk_win->mode = mode[win];
		msk_win->palette_id = palette[win];
		msk_win->x_pos = pos[win].x_pos;
		msk_win->y_pos = pos[win].y_pos;
		msk_win->x_width = size[win].w;
		msk_win->y_end = msk_win->y_pos + size[win].h - 1;
	}
	cfg_msk(chn, &msk_setting);

	return 0;
}
