#include "hal.h"

#include "sdcard.h"

#ifdef SDB_UBOOT //added by Ted for UBoot
#include <asm/arch/hardware.h>
#define halUARTPrintf printf
#define halDelay	mdelay
#define set926EJ_S_CleanInvalidateDCacheAll()	{asm ("mrc	p15, 0, r15, c7, c14, 3");}
#define set926EJ_S_CleanInvalidateDCacheEntry(x) {asm ("mrc	p15, 0, r15, c7, c14, 3");}
#else
extern void set926EJ_S_CleanInvalidateDCacheAll(void);
extern void set926EJ_S_CleanInvalidateDCacheEntry(unsigned int addr);
#endif

#define SD_PRINT_ENABLE			0x80000000
#define DEFAULT_INFO				0x001
#define SD_COMMAND_INFO			0x002
#define SD_RW_INFO					0x004
#define SD_HW_INFO					0x008
#define SD_ERR_INFO					0x010
#define PRINT_SD_DEBUG				(SD_ERR_INFO | SD_PRINT_ENABLE)
//#define PRINT_SD_DEBUG				(SD_ERR_INFO | SD_PRINT_ENABLE | SD_RW_INFO)
//#define PRINT_SD_DEBUG				(0xFFFFFFFF)
#define SDDebug(x, y)				if((x & PRINT_SD_DEBUG) != 0) y


#define SDC_DMA_ADDR_ALIGNMENT			0x00000003		// 32-bit alignment
#define DEFAULT_BLK_SIZE    						0x200
#define MAX_BLK_COUNT						8

#define WAIT_CMD_DONE						0x1
#define WAIT_CMD_RESPONSE					0x2
#define WAIT_DATA_DONE						0x3
#define DEFAULT_DEBNCE_VAL     			0x0FFFFF


typedef struct DmaDescStruct {
	unsigned int desc0;   	 /* control and status information of descriptor */
	unsigned int desc1;   	 /* buffer sizes                                 */
	unsigned int desc2;   	 /* physical address of the buffer 1             */
	unsigned int desc3;    	 /* physical address of the buffer 2             */
}DmaDesc;

DmaDesc	CurrDmaDesc __attribute__((aligned(0x10)));

unsigned char TestRegion[DEFAULT_BLK_SIZE * MAX_BLK_COUNT]__attribute__((aligned(0x100))) = {0xFF, };

/* control register */
#define SDCtrlR			(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x00)
#define SDPwrCtrlR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x04)
#define SDTimeoutR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x14)
#define SDCardTypeR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x18)
#define SDFIFOThreshR	(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x4c)
#define SDCardDetectR	(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x50)
#define SDCardWPR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x54)
#define SDGpioR			(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x58)
#define SDDebounR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x64)
#define SDBusModeR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x80)

/* status register */
#define SDTCBCntR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x5c)
#define SDTTBBCntR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x60)
#define SDUsrIDR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x68)
#define SDVersionR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x6c)
#define SDHwConfigR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x70)

/* clock */
#define SDClkDivR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x08)
#define SDClkSrcR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x0C)
#define SDClkEnR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x10)

/* SD transfer registers */
#define SDArgR			(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x28)
#define SDCmdR			(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x2c)
#define SDRespCmdR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x2c)
#define SDResp0R		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x30)
#define SDResp1R		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x34)
#define SDResp2R		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x38)
#define SDResp3R		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x3c)

#define SDDatTmOutR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x14)

#define SDByteCntR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x20)

#define SDBlkSizeR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x1c)


#define SDStsR			(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x48)


/* interrupt registers */
#define SDIntrMaskR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x24)
#define SDIntrStsR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x40)
#define SDIntrRawR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x44)
#define SDIntrClrR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x44)

/* internal DMA registers */
#define SDDmaDLAddrtR	(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x88)
#define SDDmaStsR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x8c)
#define SDDmaIntEnR		(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x90)
#define SDDmaDescAddrR	(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x94)
#define SDDmaBufAddrR	(volatile unsigned int *)(HAL_REG_SDC_BASE + 0x98)


/******************************************************************************/
/* register bit mapping */
/********************************************************************************/
/* control register */
#define CTRL_CTRL_RESET			0x00000001
#define CTRL_FIFO_RESET			0x00000002
#define CTRL_DMA_RESET				0x00000004
#define CTRL_INT_ENABLE			0x00000010
#define CTRL_USE_IDMAC	       	0x02000000


/* clock & power settings */
#define CLOCK_ON  					0x00000001
#define CLOCK_LOW_POWER			0x00010000
#define PWR_ON      				0x00000001

/* CMD Register Defines */
#define CMD_VOLT_SW_BIT         	0x10000000
#define CMD_RESP_EXP_BIT	    	0x00000040
#define CMD_RESP_LENGTH_BIT	  	0x00000080
#define CMD_CHECK_CRC_BIT	   	0x00000100
#define CMD_DATA_EXP_BIT	    	0x00000200
#define CMD_RW_BIT		        	0x00000400
#define CMD_TRANSMODE_BIT	    	0x00000800
#define CMD_SENT_AUTO_STOP_BIT	0x00001000
#define CMD_WAIT_PRV_DAT_BIT		0x00002000
#define CMD_ABRT_CMD_BIT	    	0x00004000
#define CMD_SEND_INIT_BIT 	  	0x00008000
#define CMD_SEND_CLK_ONLY	    	0x00200000
#define CMD_READ_CEATA		    	0x00400000
#define CMD_CCS_EXPECTED	    	0x00800000
#define CMD_DONE_BIT		    	0x80000000
#define CLK_ONLY_CMD	   		0x80200000
#define CMD_START 					0x80000000

/* INT status */
#define INT_CardDetect				0x00000001
#define INT_ResponseErr			0x00000002
#define INT_CommandDone			0x00000004
#define INT_DataTxOver				0x00000008
#define INT_TxFifoDataReq			0x00000010
#define INT_RxFifoDataReq			0x00000020
#define INT_ResponseCRCErr		0x00000040
#define INT_DataCRCErr				0x00000080
#define INT_ResponseTimeout		0x00000100
#define INT_DataReadTimeout		0x00000200
#define INT_ACD						0x00004000
#define INT_EBE						0x00008000

/* descriptor */
#define DescOwnByDma          		0x80000000
#define DescCardErrSummary    		0x40000000
#define DescEndOfRing         		0x00000020
#define DescSecAddrChained    		0x00000010
#define DescFirstDesc         		0x00000008
#define DescLastDesc          		0x00000004
#define DescDisInt            		0x00000002

/* Bus Mode Register Bit Definitions */
#define BMOD_SWR 						0x00000001	// Software Reset: Auto cleared after one clock cycle                                0
#define BMOD_FB 						0x00000002	// Fixed Burst Length: when set SINGLE/INCR/INCR4/INCR8/INCR16 used at the start     1
#define BMOD_DE 						0x00000080	// Idmac Enable: When set IDMAC is enabled                                           7
#define BMOD_DSL_MSK					0x0000007C	// Descriptor Skip length: In Number of Words                                      6:2
#define BMOD_DSL_Shift					2	        	// Descriptor Skip length Shift value
#define BMOD_DSL_ZERO          		0x00000000	// No Gap between Descriptors
#define BMOD_DSL_TWO           		0x00000008	// 2 Words Gap between Descriptors
#define BMOD_PBL						0x00000400	// MSIZE in FIFOTH Register
/* Internal DMAC Status Register IDSTS Bit Definitions
 * Internal DMAC Interrupt Enable Register Bit Definitions */
#define IDMAC_AI			        	0x00000200   // Abnormal Interrupt Summary Enable/ Status                                       9
#define IDMAC_NI						0x00000100   // Normal Interrupt Summary Enable/ Status                                         8
#define IDMAC_CES						0x00000020   // Card Error Summary Interrupt Enable/ status                                     5
#define IDMAC_DU						0x00000010   // Descriptor Unavailabe Interrupt Enable /Status                                  4
#define IDMAC_FBE						0x00000004   // Fata Bus Error Enable/ Status                                                   2
#define IDMAC_RI						0x00000002   // Rx Interrupt Enable/ Status                                                     1
#define IDMAC_TI						0x00000001   // Tx Interrupt Enable/ Status                                                     0

#define IDMAC_EN_INT_ALL   			0x00000337   // Enables all interrupts

#define IDMAC_HOST_ABORT_TX     		0x00000400   // Host Abort received during Transmission                                     12:10
#define IDMAC_HOST_ABORT_RX     		0x00000800   // Host Abort received during Reception                                        12:10

/* IDMAC FSM States */
#define IDMAC_DMA_IDLE          0x00000000   // DMA is in IDLE state
#define IDMAC_DMA_SUSPEND       0x00002000   // DMA is in SUSPEND state
#define IDMAC_DESC_RD           0x00004000   // DMA is in DESC READ or FETCH State
#define IDMAC_DESC_CHK          0x00006000   // DMA is checking the Descriptor for Correctness
#define IDMAC_DMA_RD_REQ_WAIT   0x00008000   // DMA is in this state till dma_req is asserted (Read operation)
#define IDMAC_DMA_WR_REQ_WAIT   0x0000A000   // DMA is in this state till dma_req is asserted (Write operation)
#define IDMAC_DMA_WR            0x0000E000   // DMA is in Write mode
#define IDMAC_DESC_CLOSE        0x00010000   // DMA is closing the Descriptor


#define FIFOTH_MSIZE_1		0x00000000   // Multiple Trans. Size is 1
#define FIFOTH_MSIZE_4		0x10000000   // Multiple Trans. Size is 4
#define FIFOTH_MSIZE_8   	0x20000000   // Multiple Trans. Size is 8
#define FIFOTH_MSIZE_16	0x30000000   // Multiple Trans. Size is 16
#define FIFOTH_MSIZE_32	0x40000000   // Multiple Trans. Size is 32
#define FIFOTH_MSIZE_64	0x50000000   // Multiple Trans. Size is 64
#define FIFOTH_MSIZE_128	0x60000000   // Multiple Trans. Size is 128
#define FIFOTH_MSIZE_256	0x70000000   // Multiple Trans. Size is 256

#define FIFOTH_MSIZE_8_REG   	0x20070008   // Multiple Trans. Size is 8
//#define SD_DMA_BURST_SIZE		FIFOTH_MSIZE_1 // Ted, OK
#define SD_DMA_BURST_SIZE		FIFOTH_MSIZE_8_REG // Ted
//#define SD_DMA_BURST_SIZE		FIFOTH_MSIZE_256 // ???????


int SD_CLK_RATE;

void sd_compute_capacity(int cardtype);
int get_timeValue(int time);
void sd_get_csd_timer(unsigned int rspREG);

bool_T SdInit(void);
bool_T SdPhyMReadNSector(unsigned int SectNo, unsigned int ColNo,unsigned int  nSize, unsigned char *data);
bool_T mmcPhySReadNSector(unsigned int SectNo, unsigned int ColNo, unsigned int  nSize,unsigned char *data);
bool_T SdPhyWriteNSector(unsigned int SectNo, unsigned int ColNo, unsigned int  nSize,unsigned char *data);
bool_T mmcPhyWriteNSector(unsigned int SectNo, unsigned int ColNo, unsigned int  nSize,unsigned char *data);
int wait_event_complete(unsigned int cmd_index, unsigned int cmdtype);

struct card_handle cardhandle = {FALSE, };

int SD_Send_Command(unsigned int CMDIndex, unsigned int arg, unsigned int *resp_buf)
 {
	unsigned int CMDreg, response = 0, cmdtype;
	int i;

	SDDebug(SD_COMMAND_INFO, halUARTPrintf("SD_Send_Command(CMD=%d, arg=%08X): ", CMDIndex, arg));

	CMDreg = CMDIndex & 0x3F;

	cmdtype = WAIT_CMD_DONE;
	switch(CMDIndex)
	{
		/* no response */
		case CMD0:
			CMDreg |= 0x0;
			CMDreg &= ~CMD_ABRT_CMD_BIT | ~CMD_RESP_EXP_BIT | ~CMD_RESP_LENGTH_BIT;
			response = 0;
			break;
		/* long response */
		case CMD2: case CMD9:
			CMDreg |= CMD_RESP_EXP_BIT | CMD_RESP_LENGTH_BIT;
			CMDreg &= ~CMD_ABRT_CMD_BIT;
			response = 4;
			cmdtype = WAIT_CMD_RESPONSE;
			break;
		/* short response */
		case ACMD41: case ACMD13: case ACMD23: case ACMD6:
		case CMD3: case CMD5: case CMD8: case CMD12: case CMD55: case CMD7:
		case CMD32: case CMD33: case CMD38: case CMD13: case CMD16:
			CMDreg |= CMD_RESP_EXP_BIT;
			CMDreg &= ~CMD_RESP_LENGTH_BIT;
			response = 1;
			cmdtype = WAIT_CMD_RESPONSE;
			break;
		/* short response + data read */
		case CMD17: case CMD18:
			CMDreg |= CMD_RESP_EXP_BIT;
			CMDreg &= ~CMD_RESP_LENGTH_BIT;

			CMDreg |= CMD_DATA_EXP_BIT | CMD_WAIT_PRV_DAT_BIT;
			CMDreg &= ~CMD_SENT_AUTO_STOP_BIT;
			response = 1;
			cmdtype = WAIT_DATA_DONE;
			break;

		/* short response + data write */
		case CMD19: case CMD24: case CMD25: case CMD42:
			CMDreg |= CMD_RESP_EXP_BIT;
			CMDreg &= ~CMD_RESP_LENGTH_BIT;

			CMDreg |= CMD_DATA_EXP_BIT | CMD_WAIT_PRV_DAT_BIT | CMD_RW_BIT;
			if((CMDIndex == CMD24) || (CMDIndex == CMD42))
				CMDreg &= ~CMD_SENT_AUTO_STOP_BIT;
			response = 1;
			cmdtype = WAIT_DATA_DONE;
			break;
		/* special command */
		default:
			response = 0;
			break;
	}

	/* clear all int */
	*SDIntrClrR = 0xFFFFFFFF;

	CMDreg |= CMD_DONE_BIT;
	*SDArgR = arg;
	*SDCmdR = CMDreg;

	SDDebug(SD_COMMAND_INFO, halUARTPrintf("\tCMD%d command reg = %08X \n\r", CMDIndex, CMDreg));

    if(!wait_event_complete(CMDIndex, cmdtype))
	   return FALSE;

    if(!response)
    	return TRUE;

	SDDebug(SD_COMMAND_INFO, halUARTPrintf("\tresponse = "));
    for(i = 0; i < response; i++)
    {
		resp_buf[i] = *(SDResp0R + i);
		SDDebug(SD_COMMAND_INFO, halUARTPrintf("%08X ", resp_buf[i]));
    }
	SDDebug(SD_COMMAND_INFO, halUARTPrintf("\n\r"));
    return TRUE;
 }

int SD_Send_Cmd(unsigned int CMDIndex, unsigned int arg, unsigned int *resp_buf)
{
	unsigned int respR1;

	SDDebug(DEFAULT_INFO, halUARTPrintf("SD_Send_Cmd(CMD=%d, arg=%08X) \n\r", CMDIndex, arg));

	if(CMDIndex > APP_CMD_OFFSET)
	{
		if(!SD_Send_Command(CMD55, cardhandle.RCA, &respR1))
			return FALSE;
	}

	return SD_Send_Command(CMDIndex, arg, resp_buf);
}

#define WAIT_CMD_LOOP_COUNT 0x800000

int wait_cmd_done(unsigned int cmd_index)
{
	unsigned int waitcnt = 0;

	SDDebug(DEFAULT_INFO, halUARTPrintf("\twait_cmd_done: "));

	// wait command done
	while ( waitcnt++ < WAIT_CMD_LOOP_COUNT)
	{
		if ( (*SDIntrRawR & INT_CommandDone) == INT_CommandDone)
			break;
	}

	if(waitcnt >= WAIT_CMD_LOOP_COUNT)
	{
		SDDebug(SD_ERR_INFO, halUARTPrintf("\tCMD%d command done expired\n\r", cmd_index));
		return FALSE;
	}

	SDDebug(DEFAULT_INFO, halUARTPrintf(" done\n\r "));
	return TRUE;
}

int wait_cmd_response(unsigned int cmd_index)
{
	unsigned int waitcnt = 0;
	unsigned int IntStatus;

	SDDebug(DEFAULT_INFO, halUARTPrintf("\twait_cmd_response: "));

	// wait command done
	while ( waitcnt++ < WAIT_CMD_LOOP_COUNT)
	{
		if ( (*SDIntrRawR & INT_CommandDone) == INT_CommandDone)
			break;
	}

	if(waitcnt >= WAIT_CMD_LOOP_COUNT)
	{
		SDDebug(SD_ERR_INFO, halUARTPrintf("\tCMD%d command done expired\n\r", cmd_index));
		return FALSE;
	}

  	IntStatus = *SDIntrRawR;
	if(((IntStatus & INT_ResponseErr) != 0) ||
			((IntStatus & INT_ResponseCRCErr) != 0) ||
			((IntStatus & INT_ResponseTimeout) != 0))
	{
		SDDebug(SD_ERR_INFO, halUARTPrintf("\tCMD%d command done expired(int=%08X)\n\r", cmd_index, *SDIntrRawR));
		return FALSE;
	}

	SDDebug(DEFAULT_INFO, halUARTPrintf(" done\n\r "));
	return TRUE;
}

#define WAIT_DATA_DONE_LOOP_COUNT WAIT_CMD_LOOP_COUNT

int wait_data_done(unsigned int cmd_index)
{
	unsigned int waitcnt = 0;
	unsigned int IntStatus;

	SDDebug(DEFAULT_INFO, halUARTPrintf("\twait_data_done: "));

	// wait command done
	while ( waitcnt++ < WAIT_CMD_LOOP_COUNT)
	{
		if ( (*SDIntrRawR & INT_CommandDone) == INT_CommandDone)
			break;
	}

	if(waitcnt >= WAIT_CMD_LOOP_COUNT)
	{
		SDDebug(SD_ERR_INFO, halUARTPrintf("\tCMD%d command done expired(int=%08X)\n\r", cmd_index, *SDIntrRawR));
		return FALSE;
	}

  	IntStatus = *SDIntrRawR;
	if(((IntStatus & INT_ResponseErr) != 0) ||
			((IntStatus & INT_ResponseCRCErr) != 0) ||
			((IntStatus & INT_ResponseTimeout) != 0))
	{
		SDDebug(SD_ERR_INFO, halUARTPrintf("\tCMD%d response error (int=%08X)\n\r", cmd_index, IntStatus));
		return FALSE;
	}

	// wait data done
	waitcnt = 0;
	while ( waitcnt++ < WAIT_DATA_DONE_LOOP_COUNT)
	{
		if ( (*SDIntrRawR & INT_DataTxOver) == INT_DataTxOver)
			break;
		//halDelay(1);
	}

	if(waitcnt >= WAIT_DATA_DONE_LOOP_COUNT)
	{
		SDDebug(SD_ERR_INFO, halUARTPrintf("\tCMD%d data done expired(int=%08X)\n\r", cmd_index, *SDIntrRawR));
		return FALSE;
	}

 	IntStatus = *SDIntrRawR;
	if(((IntStatus & INT_DataCRCErr) != 0) ||
//			((IntStatus & INT_TxFifoDataReq) != 0) ||
//			((IntStatus & INT_RxFifoDataReq) != 0) ||
			((IntStatus & INT_DataReadTimeout) != 0))
	{
		SDDebug(SD_ERR_INFO, halUARTPrintf("\tCMD%d data error(int=%08X)\n\r", cmd_index, IntStatus));
		return FALSE;
	}

	SDDebug(DEFAULT_INFO, halUARTPrintf(" done\n\r "));
	return TRUE;
}

int wait_event_complete(unsigned int cmd_index, unsigned int cmdtype)
{
	unsigned int sts = TRUE;

	switch (cmdtype)
	{
		default:
		case WAIT_CMD_DONE: // do not need response
			sts = wait_cmd_done(cmd_index);
			break;;
		case WAIT_CMD_RESPONSE: // need response
			sts = wait_cmd_response(cmd_index);
			break;;
		case WAIT_DATA_DONE:
			sts = wait_data_done(cmd_index);
//			SDDebug(SD_RW_INFO, halUARTPrintf("\tCMD%d INT sts=%08X\n\r", cmd_index, *SDIntrRawR));
			break;;
	}

	*SDIntrRawR = 0xFFFFFFFF;

	return sts;
}


/****************************************************************************************/
/****************************************************************************************/
/****************************************************************************************/
/****************************************************************************************/
/****************************************************************************************/
/****************************************************************************************/
/****************************************************************************************/
/****************************************************************************************/
/****************************************************************************************/
/****************************************************************************************/
void SdEnableInt(unsigned int mask)
{
	*SDIntrMaskR = mask;
}

unsigned int SdClearInt(void)
{
	unsigned int sts;

	sts = *SDIntrStsR;
	*SDIntrClrR = 0xFFFFFFFF;
	return sts;
}

bool_T SdSetInternalDMA_read(DmaDesc *desc, unsigned char *buf, int size)
{
	if((((int)buf) & SDC_DMA_ADDR_ALIGNMENT) != 0)
	{
		desc->desc2 = (unsigned int)(TestRegion);
		halUARTPrintf("\tSdSetInternalDMA(): desc=%08X buf=%08X\n\r", desc, desc->desc2);
	}
	else
	{
		desc->desc2 = (unsigned int)buf;
	}

	SDDebug(SD_RW_INFO, halUARTPrintf("\tSdSetInternalDMA(): desc=%08X buf=%08X\n\r", desc, desc->desc2));

	desc->desc1 = size;
	desc->desc3 = 0;
	desc->desc0 = DescOwnByDma | DescFirstDesc | DescLastDesc;

	*SDDmaDLAddrtR = (unsigned int)desc;

	*SDCtrlR |= CTRL_USE_IDMAC;									// Set control register

	*SDBusModeR = BMOD_DE | BMOD_DSL_ZERO;

	*SDDmaIntEnR = IDMAC_EN_INT_ALL;
	*SDDmaStsR = 0xffffffff;	//clear all

	*SDFIFOThreshR = SD_DMA_BURST_SIZE;

	*SDByteCntR = size;
	*SDBlkSizeR = DEFAULT_BLK_SIZE;

	set926EJ_S_CleanInvalidateDCacheAll();

	return TRUE;
}

bool_T SdGetInternalStatus_read(DmaDesc *desc, unsigned char *buf, int size)
{
	int i;
	volatile unsigned int *desc0;

	SDDebug(SD_RW_INFO, halUARTPrintf("\tSdGetInternalDMA(): desc=%08X buf=%08X ", desc, desc->desc2));
//	SDDebug(0xffffffff, halUARTPrintf("\tSdGetInternalDMA(): desc=%08X buf=%08X ", desc, desc->desc2));

	desc0 = (unsigned int *)desc;


#define DESCRIPTOR_LOOP_COUNT		0x100000
	for(i = 0; i < DESCRIPTOR_LOOP_COUNT; i++)
	{
		set926EJ_S_CleanInvalidateDCacheEntry((unsigned int)desc);
		if((*desc0 & DescOwnByDma) == 0)
			break;
	//	halDelay(1);
	}

	if(i == DESCRIPTOR_LOOP_COUNT)
	{
		SDDebug(SD_ERR_INFO, halUARTPrintf("SD internal DMA descriptor own bit checking expired. sts = 0x%08X\n\r", *desc0));
		return FALSE;
	}


	if((*desc0 & DescCardErrSummary) != 0)
	{
		SDDebug(SD_ERR_INFO, halUARTPrintf("SD internal DMA descriptor err bit checking failed.\n\r"));
		return FALSE;
	}

	set926EJ_S_CleanInvalidateDCacheAll();

	if((((int)buf) & SDC_DMA_ADDR_ALIGNMENT) != 0)
	{
#if 1
		memcpy(buf, TestRegion, size);
#else
		for(i = 0; i < size; i++)
			buf[i] = TestRegion[i];
#endif
	}
	SDDebug(SD_RW_INFO, halUARTPrintf("done.\n\r"));
	return TRUE;
}

bool_T SdSetInternalDMA_write(DmaDesc *desc, unsigned char *buf, int size)
{
	int i;

	if(((int)buf & SDC_DMA_ADDR_ALIGNMENT) != 0)
	{
		desc->desc2 = (unsigned int)(TestRegion);
#if 1
		memcpy(TestRegion, buf, size);
#else
		for(i = 0; i < size; i++)
			TestRegion[i] = buf[i];
#endif
//		halUARTPrintf("\tSdSetInternalDMA(): desc=%08X buf=%08X tempbuf=%08X\n\r", desc, buf, desc->desc2);
	}
	else
	{
		desc->desc2 = (unsigned int)buf;
//		halUARTPrintf("\tSdSetInternalDMA(): desc=%08X buf=%08X\n\r", desc, desc->desc2);
	}

	SDDebug(SD_RW_INFO, halUARTPrintf("\tSdSetInternalDMA(): desc=%08X buf=%08X\n\r", desc, desc->desc2));

	desc->desc1 = size;
	desc->desc3 = 0;
	desc->desc0 = DescOwnByDma | DescFirstDesc | DescLastDesc;

	*SDDmaDLAddrtR = (unsigned int)desc;

	*SDCtrlR |= CTRL_USE_IDMAC;									// Set control register

	*SDBusModeR = BMOD_DE | BMOD_DSL_ZERO;

	*SDDmaIntEnR = IDMAC_EN_INT_ALL;
	*SDDmaStsR = 0xffffffff;	//clear all

	*SDFIFOThreshR = SD_DMA_BURST_SIZE;

	*SDByteCntR = size;
	*SDBlkSizeR = DEFAULT_BLK_SIZE;

	set926EJ_S_CleanInvalidateDCacheAll();


	return TRUE;
}

bool_T SdGetInternalStatus_write(DmaDesc *desc, unsigned char *buf, int size)
{
	int i;

	volatile unsigned int *desc0;

	SDDebug(SD_RW_INFO, halUARTPrintf("\tSdGetInternalDMA(): desc=%08X buf=%08X ", desc, desc->desc2));

	desc0 = (unsigned int *)desc;

#define DESCRIPTOR_LOOP_COUNT		0x100000
	for(i = 0; i < DESCRIPTOR_LOOP_COUNT; i++)
	{
		set926EJ_S_CleanInvalidateDCacheEntry((unsigned int)desc);
		if((*desc0 & DescOwnByDma) == 0)
			break;
	//	halDelay(1);
	}

	if(i == DESCRIPTOR_LOOP_COUNT)
	{
		SDDebug(SD_ERR_INFO, halUARTPrintf("SD internal DMA descriptor own bit checking expired. sts = 0x%08X\n\r", *desc0));
		return FALSE;
	}

	if((*desc0 & DescCardErrSummary) != 0)
	{
		SDDebug(SD_ERR_INFO, halUARTPrintf("SD internal DMA descriptor err bit (%08X) checking failed.\n\r", *desc0));
		return FALSE;
	}

	SDDebug(SD_RW_INFO, halUARTPrintf("done.\n\r"));
	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////
/* SD parameter & information APIs */
/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////

bool_T initialrwproc(unsigned int RCA,unsigned int BL_LEN,unsigned int nEBLEN)
{
	unsigned int resp;

	SDDebug(DEFAULT_INFO, halUARTPrintf("initialrwproc(RCA = %08X) \n\r", RCA));

	/* set bus width to 4 bit */
	if(!SD_Send_Cmd(ACMD6, 0x2, &resp))
		return FALSE;
	
	/* set card type to 4-bit mode */
	*SDCardTypeR = 1;

	/* set block length */
	if(!SD_Send_Cmd(CMD16, BL_LEN, &resp))
		return FALSE;

	 return (TRUE);
}

//#define ACMD41_RETRY_COUNT 	0x10000
//#define CMD5_RETRY_COUNT		0x10000
//#define CMD3_RETRY_COUNT		0x10000
//#define CMD2_RETRY_COUNT		0x10000

#define ACMD41_RETRY_COUNT 	0x10000
#define CMD5_RETRY_COUNT		0x100
#define CMD3_RETRY_COUNT		0x10000
#define CMD2_RETRY_COUNT		0x10000

bool_T sdio_init_flow(void)
{
	unsigned int new_ocr, resp = 0;
	unsigned int count;

	SDDebug(DEFAULT_INFO, halUARTPrintf("Tesging SDIO card.\n\r"));

	count = CMD5_RETRY_COUNT;
	while (count--)
	{
		new_ocr = resp & IO_R4_OCR_MSK;
		/* send CMD5 to check if it is SDIO card */
		if((SD_Send_Cmd(CMD5, new_ocr, &resp)) && (resp & IO_R4_READY_BIT))
			break;
	}
	if (count == 0xFFFFFFFF) {
		cardhandle.card_type = CARD_NOCARD;
		return FALSE;
	}

	cardhandle.card_type = CARD_IO;
	if (resp & IO_R4_MEM_PRESENT_BIT)
		cardhandle.card_type = CARD_IOMEM;

	/* still has some init need to do, but stop here so far */

	if(cardhandle.card_type == CARD_IO)
		{SDDebug(SD_PRINT_ENABLE, halUARTPrintf("\tFind SDIO card.\n\r"));}
	else
		{SDDebug(SD_PRINT_ENABLE, halUARTPrintf("\tFind SDIO with MEM card.\n\r"));}

	return TRUE;
}

bool_T sdhc_init_flow(void)
{
	unsigned int resp, new_ocr;
	int count;

	SDDebug(DEFAULT_INFO, halUARTPrintf("Tesging SDHC card.\n\r"));

	cardhandle.card_type = CARD_NOCARD;

	/* CMD0 */
	if(!SD_Send_Cmd(CMD0, 0, 0))
		return FALSE;

	/* set voltage */
	count = ACMD41_RETRY_COUNT;
	new_ocr = OCR_27TO36 | OCR_POWER_UP_STATUS | OCR_HCS | OCR_XPC | OCR_S18R;
	while(count--)
	{
		if((SD_Send_Cmd(ACMD41, new_ocr, &resp)) && (resp & OCR_POWER_UP_STATUS) && ((resp & OCR_27TO36) == OCR_27TO36))
//		if((SD_Send_Cmd(ACMD41, new_ocr, &resp)) && ((resp & OCR_27TO36) == OCR_27TO36))
			break;
	}
	if (count == 0xFFFFFFFF)
		return FALSE;

	if(resp & OCR_CCS)
	{
		cardhandle.card_type = CARD_SDHC;
		SDDebug(SD_PRINT_ENABLE, halUARTPrintf("\tFind SDHC card.\n\r"));
	}
	else
	{
		cardhandle.card_type = CARD_SD;
		SDDebug(SD_PRINT_ENABLE, halUARTPrintf("\tFind SD card(from SDHC).\n\r"));
	}
	return TRUE;
}

bool_T sd_init_flow(void)
{
	unsigned int resp, new_ocr;
	int count;

	SDDebug(DEFAULT_INFO, halUARTPrintf("Tesging SD card.\n\r"));

	cardhandle.card_type = CARD_NOCARD;

	/* CMD0 */
	if(!SD_Send_Cmd(CMD0, 0, 0))
		return FALSE;

	/* set voltage */
	count = ACMD41_RETRY_COUNT;
	new_ocr = OCR_27TO36 | OCR_POWER_UP_STATUS;
	while(count--)
	{
		if((SD_Send_Cmd(ACMD41, new_ocr, &resp)) && (resp & OCR_POWER_UP_STATUS) && ((resp & OCR_27TO36) == OCR_27TO36))
			break;
	}
	if (count == 0xFFFFFFFF)
		return FALSE;

	cardhandle.card_type = CARD_SD;

	SDDebug(SD_PRINT_ENABLE, halUARTPrintf("\tFind SD card.\n\r"));

	return TRUE;
}

bool_T mmc_init_flow(void)
{
	unsigned int resp, new_ocr;
	unsigned int count;


	SDDebug(DEFAULT_INFO, halUARTPrintf("Tesging MMC card.\n\r"));

	/* CMD0 */
	if(!SD_Send_Cmd(CMD0, 0, 0))
		return FALSE;

	/* set voltage */
	count = ACMD41_RETRY_COUNT;
	new_ocr = OCR_27TO36 | OCR_POWER_UP_STATUS;
	while(count--)
	{
		if((SD_Send_Cmd(ACMD41, new_ocr, &resp)) && (resp & OCR_POWER_UP_STATUS) && ((resp & OCR_27TO36) == OCR_27TO36))
			break;
	}
	if (count == 0xFFFFFFFF)
		return FALSE;

	cardhandle.card_type = CARD_MMC;

	SDDebug(SD_PRINT_ENABLE, halUARTPrintf("\tFind MMC card.\n\r"));
	return TRUE;
}

int GetCardType()
{
	unsigned int resp;

	cardhandle.card_type = CARD_NOCARD;

#if 0
	/* send CMD5 to check if it is SDIO card */
	if(SD_Send_Cmd(CMD5, 0, &resp))
		return (sdio_init_flow() ? CARD_IO : CARD_NOCARD);
#endif

	/* CMD0 */
	if(!SD_Send_Cmd(CMD0, 0, 0))
		return CARD_NOCARD;

	/* CMD8 for SDHC */
	if(SD_Send_Cmd(CMD8, 0x000001AA, &resp))
		return (sdhc_init_flow() ? CARD_SDHC : CARD_NOCARD);

	if(SD_Send_Cmd(ACMD41, 0, &resp))
		return (sd_init_flow() ? CARD_SD : CARD_NOCARD);

	return (mmc_init_flow() ? CARD_MMC : CARD_NOCARD);

	return CARD_NOCARD;
}


int GetCardInform(int cardtype)
{
	int count;

	/* get CID */
	count = CMD2_RETRY_COUNT;
	while(count--)
	{
		if(SD_Send_Cmd(CMD2, 0, cardhandle.CID))
			break;
	}
	if (count == 0xFFFFFFFF)
		return FALSE;

	/* set RCA */
	count = CMD3_RETRY_COUNT;
	while(count--)
	{
		if(SD_Send_Cmd(CMD3, 0, &cardhandle.RCA))
			break;
	}
	if (count == 0xFFFFFFFF)
		return FALSE;

	if(!SD_Send_Cmd(CMD9, cardhandle.RCA, cardhandle.CSD))
		return FALSE;

	return TRUE;
}

bool_T SdInit(void)
{
	unsigned int resp;
	int cdetect, cardtype;
	
	SDDebug(DEFAULT_INFO, halUARTPrintf("\nSdInit.....\n\r"));

	cardhandle.RCA = 0;
	/***************************/
	/* sw reset */
	/***************************/
	halUARTPrintf("\nReset card.\n\r");
	*SDCtrlR = CTRL_CTRL_RESET;									// Set control register
	while((*SDCtrlR & CTRL_CTRL_RESET) != 0);					// wait for reset complete

#if 0
	halUARTPrintf("\nReset DMA.\n\r");
	*SDCtrlR = CTRL_DMA_RESET;									// Set DMA register
	while((*SDCtrlR & CTRL_DMA_RESET) != 0);					// wait for reset complete

	halUARTPrintf("\nReset FIFO.\n\r");
	*SDCtrlR = CTRL_FIFO_RESET;									// Set FIFO register
	while((*SDCtrlR & CTRL_FIFO_RESET) != 0);					// wait for reset complete
#endif

	/***************************/
	/* init */
	/***************************/
	*SDCardTypeR = 0x0; 							/* Now make CTYPE to default i.e, all the cards connected will work in 1 bit mode initially*/

	/* Power up the card
	    - Shut-down the card/device once wait for some time
	    - Enable the power to the card/Device. wait for some time
	*/
	*SDPwrCtrlR	= 0x0;  							// clear power control
	halDelay(HAL_TICKS_PER_SEC * 5); 				/* sleep for 1 seconds */

	*SDPwrCtrlR	= 0x1;  							// clear power control
	halDelay(HAL_TICKS_PER_SEC * 5); 				/* sleep for 1 seconds */

	/* Set up the interrupt mask.
 	   - Clear the interrupts in any pending Wrinting 1's to RINTSTS
	   - Enable all the interrupts other than ACD in INTMSK register
	   - Enable global interrupt in control register
	*/
	*SDIntrClrR	= 0xffffffff;  					// clear interrupt status
	*SDIntrMaskR = 0xffffffff  & ~INT_ACD;;
	*SDCtrlR	|= CTRL_INT_ENABLE;
	
	/* Set Data and Response timeout to Maximum Value*/
	*SDTimeoutR = 0xffffffff;

	/* Enable the clocks to the card
	   - Note this command is to CIU of host controller ip
	   - the command is not sent on the command bus
	   - it emplys the polling method to accomplish the task
	   - it also emplys wait prev data complete in CMD register
        */
	*SDClkEnR	 = CLOCK_ON;								// enable clock & disable low power
	*SDClkDivR = cardhandle.CLK_DIV;					// set all 4 clock divider
	*SDClkSrcR =	0x0; 								// using clock divider 0

//	halUARTPrintf("\Send clk only cmd.\n\r");
	/* send clock only command & wait command complete */
	*SDCmdR = CLK_ONLY_CMD | CMD_WAIT_PRV_DAT_BIT;
	while((*SDCmdR & CMD_DONE_BIT) == CMD_DONE_BIT);

	/* Set the card Debounce to allow the CDETECT fluctuations to settle down*/
	*SDDebounR = DEFAULT_DEBNCE_VAL;
	
//	halUARTPrintf("\Start Card Detection.\n\r");

	/* card detect */
	cdetect = ((*SDCardDetectR & 0x1) == 0x1) ? 0 : 1;

	if(!cdetect)
	{
		halUARTPrintf("\nCan not detect any card.\n\r");
		return FALSE;
	}

	/* set TX RX FIFO watermark */

	/********************************/
	/********************************/
	/********************************/
	/* check card type */
	cardtype = GetCardType();

	if(cardtype == CARD_NOCARD)
	{
		SDDebug(SD_PRINT_ENABLE, halUARTPrintf("\nCan not find SD Card.\n\r"));
		return FALSE;
	}

	if((cardtype == CARD_IO) || (cardtype == CARD_IOMEM))
		return FALSE;

	if(!GetCardInform(cardtype))
		return FALSE;

   //CACULATE sd card capacity
   sd_compute_capacity(cardtype);
   
   
   //let card enter tranmit state
   if(!SD_Send_Cmd(CMD7, cardhandle.RCA, &resp))
	   return FALSE;

   if (initialrwproc(cardhandle.RCA, DEFAULT_BLK_SIZE, 1))
	   return TRUE;
   else
	   return FALSE;
}


bool_T GetSdCardStatus(unsigned int *resp)
{
	if(!SD_Send_Cmd(CMD13, cardhandle.RCA, resp))
			return FALSE;

	SDDebug(SD_PRINT_ENABLE, halUARTPrintf("SD Card Status = %08X\n\r", *resp));
	return TRUE;
}

#define WAIT_TRANSFER_STATE_LOOP_COUNT		0x10000
int WaitSdCardTransferState(void)
{
	int i = 0;
	unsigned int resp;

	while(i++ < WAIT_TRANSFER_STATE_LOOP_COUNT)
	{
		if(!SD_Send_Cmd(CMD13, cardhandle.RCA, &resp))
				return FALSE;

		if(R1CS_CURRENT_STATE(resp) == SD_TRANSFER_STATE)
			return TRUE;
	}
	SDDebug(SD_ERR_INFO, halUARTPrintf("Can not enter transfer state.\n\r"));
	return FALSE;
}

void sd_compute_capacity(int cardtype)
{
	unsigned int read_block_size, write_block_size;
	unsigned int card_size;
	unsigned int blocknr, blocklen;

//	if (CSD_TAAC(cardhandle.CSD) > cardhandle.max_TAAC_value)
		cardhandle.max_TAAC_value =  CSD_TAAC(cardhandle.CSD);

//	if (CSD_NSAC(cardhandle.CSD) > cardhandle.max_NSAC_value)
		cardhandle.max_NSAC_value = CSD_NSAC(cardhandle.CSD);


	read_block_size  = 1 << (CSD_READ_BL_LEN((cardhandle.CSD)));
	write_block_size = 1 << (CSD_WRT_BL_LEN((cardhandle.CSD)));

	/* See section 5.3 of the 4.1 revision of the MMC specs for
	   an explanation for the calculation of these values
	 */
	blocknr = (CSD_C_SIZE(cardhandle.CSD) + 1) * (1 << (CSD_C_SIZE_MULT(cardhandle.CSD) + 2));
	blocklen = read_block_size;
	card_size = blocknr * blocklen;
	/* read/write block size */
	cardhandle.card_write_blksize = (write_block_size > DEFAULT_BLK_SIZE) ? DEFAULT_BLK_SIZE : write_block_size;
	cardhandle.orig_card_read_blksize = read_block_size;
	cardhandle.card_read_blksize = (read_block_size > DEFAULT_BLK_SIZE) ? DEFAULT_BLK_SIZE : read_block_size;
	cardhandle.orig_card_write_blksize = write_block_size;

	cardhandle.card_size = card_size;


	if(cardtype == CARD_MMC)
	{
		if (CSD_SPEC_VERS(cardhandle.CSD) > CSD_SPEC_VER_3)
		{
			cardhandle.card_type = CARD_MMC42;
			/* need to do something else */
		}
    }

#if 0
	switch(cardtype)
	{
		case CARD_SD:
		case CARD_SDHC:
		case CARD_MMC:
			/* Set the card capacity */
			cardhandle.card_size = blocknr;
			break;
	}
#endif

	SDDebug(DEFAULT_INFO,halUARTPrintf("Card Information:\n\r"));
	SDDebug(DEFAULT_INFO,halUARTPrintf("\tSize=%d WR_BLK:%d RD_BLK:%d\n\r", cardhandle.card_size, cardhandle.card_write_blksize, cardhandle.orig_card_read_blksize));

}

int get_timeValue(int time)
{
	int ret [] = {0, 10, 12, 13, 15, 20, 25, 30, 35, 40, 45, 50, 55, 60, 70, 80};

	if(time <= 0xf)
		return ret[time];
	else
		return 0;
}

void sd_get_csd_timer(unsigned int rspREG)
{
	char tmpTACC,tmpTimeUNIT,tmpTimeValue;
	char tmpNSAC;
	int TACC = 0;

	tmpTACC = (rspREG & 0x00ff0000) >> 16;
	tmpTimeUNIT =  tmpTACC & 0x7;
	tmpTimeValue =  get_timeValue( (tmpTACC & 0x78) >> 3 );

	switch(tmpTimeUNIT)
	{
		case 0:
			TACC = tmpTimeValue * 1;
			break;
		case 1:
			TACC = tmpTimeValue * 10;
			break;
		case 2:
			TACC = tmpTimeValue * 100;
			break;
		case 3:
			TACC = tmpTimeValue * 1000;
			break;
		case 4:
			TACC = tmpTimeValue * 10000;
			break;
		case 5:
			TACC = tmpTimeValue * 100000;
			break;
		case 6:
			TACC = tmpTimeValue * 1000000;
			break;
		case 7:
			TACC = tmpTimeValue * 10000000;
			break;
	}

	tmpNSAC = (rspREG & 0xff00) >> 8;

	cardhandle.NSAC =( tmpNSAC * 100) * 100;

	cardhandle.TACC = (	TACC / (SD_CLK_RATE * 10)) * 100;

	*SDDatTmOutR = cardhandle.TACC  +  cardhandle.NSAC + 10000;
}

/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////
/* sector read/write APIs */
/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////
bool_T sd_erase_block(int start_addr, int end_addr)
{
	unsigned int resp;

	SDDebug(SD_RW_INFO, halUARTPrintf("sd_erase_block(): start=%d end=%d \n\r", start_addr, end_addr));

	if(!SD_Send_Cmd(CMD32, start_addr, &resp))
		return FALSE;

	if(!SD_Send_Cmd(CMD33, end_addr, &resp))
		return FALSE;

	if(!SD_Send_Cmd(CMD38, 0, &resp))
		return FALSE;

	return TRUE;
}

bool_T SdPhyWriteNSector(unsigned int SectNo, unsigned int ColNo, unsigned int  nSize, unsigned char *data)
{
	unsigned int Erasecnt, resp;

	SDDebug(SD_RW_INFO, halUARTPrintf("SdPhyWriteNSector(): Sector=%d, size=%d, buf=%08X\n\r", SectNo, nSize, (unsigned int)data));

	WaitSdCardTransferState();

	/*reset FIFO */
	*SDCtrlR |= CTRL_FIFO_RESET;									// Set FIFO register
	while((*SDCtrlR & CTRL_FIFO_RESET) != 0);					// wait for reset complete

	if (cardhandle.card_type == CARD_MMC)
		return mmcPhyWriteNSector( SectNo,  ColNo, nSize,data);	

	Erasecnt = nSize / DEFAULT_BLK_SIZE;
   
	//set erase block length
	if(!SD_Send_Cmd(ACMD23, Erasecnt, &resp))
		return FALSE;

	SdSetInternalDMA_write(&CurrDmaDesc, data, nSize);

	WaitSdCardTransferState();

   //issue muti write command
	if(!SD_Send_Cmd(CMD25, SectNo, &resp))
		return FALSE;

	if(SdGetInternalStatus_write(&CurrDmaDesc, data, nSize))
	{
		SDDebug(SD_RW_INFO, halUARTPrintf("\tSdPhyWriteNSector(): done\n\r"));
		if(!SD_Send_Cmd(CMD12, 0, &resp))
			return FALSE;
		return TRUE;
	}
	else
	{
		SDDebug(SD_ERR_INFO, halUARTPrintf("\tSdPhyWriteNSector(): failed\n\r"));
		if(!SD_Send_Cmd(CMD12, 0, &resp))
			return FALSE;
		return FALSE;

	}

	return TRUE;
}


bool_T mmcPhyWriteNSector(unsigned int SectNo, unsigned int ColNo,unsigned int  nSize,unsigned char *data)
{
	unsigned int Erasecnt, resp;

	SDDebug(SD_RW_INFO, halUARTPrintf("mmcPhyWriteNSector(): Sector=%d, size=%d, buf=%08X\n\r", SectNo, nSize, (unsigned int)data));

	Erasecnt = nSize / DEFAULT_BLK_SIZE;
   
	//set erase block length
	if(!SD_Send_Cmd(ACMD23, Erasecnt, &resp))
		return FALSE;

	SdSetInternalDMA_write(&CurrDmaDesc, data, nSize);

   //issue muti write command
	if(!SD_Send_Cmd(CMD25, SectNo, &resp))
		return FALSE;

	if(SdGetInternalStatus_write(&CurrDmaDesc, data, nSize))
	{
		SDDebug(DEFAULT_INFO, halUARTPrintf("\tmmcPhyWriteNSector(): done\n\r"));
		if(!SD_Send_Cmd(CMD12, 0, &resp))
			return FALSE;
		return TRUE;
	}
	else
	{
		SDDebug(DEFAULT_INFO, halUARTPrintf("\tmmcPhyWriteNSector(): failed\n\r"));
		if(!SD_Send_Cmd(CMD12, 0, &resp))
			return FALSE;
		return FALSE;

	}

	return TRUE;
}

bool_T SdPhyMReadNSector(unsigned int SectNo, unsigned int ColNo,unsigned int  nSize,unsigned char *data)
{
	unsigned int resp;

	SDDebug(SD_RW_INFO, halUARTPrintf("SdPhyMReadNSector(): Sector=%d Size=%d buf=%08X\n\r", SectNo, nSize, data));

	WaitSdCardTransferState();

	/*reset FIFO */
	*SDCtrlR |= CTRL_FIFO_RESET;									// Set FIFO register
	while((*SDCtrlR & CTRL_FIFO_RESET) != 0);					// wait for reset complete

	if (cardhandle.card_type == CARD_MMC)
		return mmcPhySReadNSector( SectNo,  ColNo,  nSize,data);

	SdSetInternalDMA_read(&CurrDmaDesc, data, nSize);

 	if(!SD_Send_Cmd(CMD18, SectNo, &resp))
  		return FALSE;

	if(SdGetInternalStatus_read(&CurrDmaDesc, data, nSize))
		{SDDebug(DEFAULT_INFO, halUARTPrintf("\tSdPhyMReadNSector(): done\n\r"));}
	else
		{SDDebug(DEFAULT_INFO, halUARTPrintf("\tSdPhyMReadNSector(): failed\n\r"));}

  	if(!SD_Send_Cmd(CMD12, 0, &resp))
  		return FALSE;

//	GetSdCardStatus(&resp);
	return TRUE;
}


bool_T mmcPhySReadNSector(unsigned int SectNo, unsigned int ColNo,unsigned int  nSize,unsigned char *data)
{
	unsigned int resp;

	SDDebug(SD_RW_INFO, halUARTPrintf("mmcPhySReadNSector(): Sector=%d Size=%d buf=%08X\n\r", SectNo, nSize, data));

	SdSetInternalDMA_read(&CurrDmaDesc, data, nSize);

 //	if(!SD_Send_Cmd(CMD18, SectNo, &resp))
	if(!SD_Send_Cmd(CMD17, SectNo, &resp))
  		return FALSE;

 // 	if(!SD_Send_Cmd(CMD12, 0, &resp))
  ///		return FALSE;

	if(SdGetInternalStatus_read(&CurrDmaDesc, data, nSize))
		{SDDebug(DEFAULT_INFO, halUARTPrintf("\tmmcPhySReadNSector(): done\n\r"));}
	else
		{SDDebug(DEFAULT_INFO, halUARTPrintf("\tmmcPhySReadNSector(): failed\n\r"));}

//	GetSdCardStatus(&resp);
	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////
/* exposed APIs */
/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////

bool_T halSDCardExist(void)
{
	unsigned int cdetect;

	SDDebug(DEFAULT_INFO, halUARTPrintf("halSDCardExist()\n\r"));

	/* card detect */
	cdetect = ((*SDCardDetectR & 0x1) == 0x1) ? 0 : 1;

	if(!cdetect)
		return FALSE;

	return TRUE;
}

unsigned int halSDGetCapacity(void)
{
	SDDebug(DEFAULT_INFO, halUARTPrintf("halSDGetCapacity()\n\r"));
	return (cardhandle.card_size);
}

volatile bool_T bSDCIntrOccur_gi;
volatile unsigned int sd_int_sts = 0;

static void SD_ISR(void)
{
	bSDCIntrOccur_gi ++ ;
	sd_int_sts = SdClearInt();
}


bool_T halSDInstall(int sd_cclk)
{
	SDDebug(DEFAULT_INFO, halUARTPrintf("halSDInstall()\n\r"));

	SD_CLK_RATE = sd_cclk;
//	Ccardhandle.LK_DIV = sd_cclk / 52;
	cardhandle.CLK_DIV = 1;

//	halVICSetIntrTrigMode(halc_sdc_intr, halc_level_activeHigh);
//	halRegisterISR(halcIRQMode, halc_sdc_intr, SD_ISR);
	if(SdInit())
	{
//		sdbvPrintf_g("SD Int count = %d \n\r", bSDCIntrOccur_gi);
//		bSDCIntrOccur_gi = 0;
//		halUnregisterISR(halcIRQMode, halc_sdc_intr);
		cardhandle.sd_ready = 1;
		return (TRUE);
	}
	else
	{
//		halUnregisterISR(halcIRQMode, halc_sdc_intr);
		SDDebug(DEFAULT_INFO, halUARTPrintf("SdInit failed\n\r"));
		cardhandle.sd_ready = 0;
		return (FALSE);
	}
}

bool_T halSDIsInstall(void)
{
	SDDebug(DEFAULT_INFO, halUARTPrintf("halSDIsInstall()\n\r"));

	return cardhandle.sd_ready;
}

bool_T halSDUninstall(void)
{
	SDDebug(DEFAULT_INFO, halUARTPrintf("halSDUninstall()\n\r"));

	{
		cardhandle.sd_ready = 1;
		return (TRUE);
	}
	cardhandle.sd_ready = 0;
	return (TRUE);
}

bool_T halSDReadNSector(unsigned int nStartSect, unsigned int nSectNum, void *pBuf)
{
//	SDDebug(DEFAULT_INFO, halUARTPrintf("halSDReadNSector(): Sector=%d num=%d buf = %08X\n\r", nStartSect, nSectNum, pBuf));
	SDDebug(DEFAULT_INFO, halUARTPrintf("Read Sec=%d num=%d\n\r", nStartSect, nSectNum));

//	return SdPhyMReadNSector(nStartSect * DEFAULT_BLK_SIZE, 0, nSectNum * DEFAULT_BLK_SIZE, pBuf);
	while(nSectNum > MAX_BLK_COUNT)
	{
		if(SdPhyMReadNSector(nStartSect * DEFAULT_BLK_SIZE, 0, MAX_BLK_COUNT * DEFAULT_BLK_SIZE, pBuf) == FALSE)
			return FALSE;
		nSectNum -= MAX_BLK_COUNT;
		nStartSect += MAX_BLK_COUNT;
		pBuf = (void *)((unsigned int)pBuf + MAX_BLK_COUNT * DEFAULT_BLK_SIZE);
	}
	if(nSectNum > 0) return SdPhyMReadNSector(nStartSect * DEFAULT_BLK_SIZE, 0, nSectNum * DEFAULT_BLK_SIZE, pBuf);
	return TRUE;
}

bool_T halSDWriteNSector(unsigned int nStartSect, unsigned int nSectNum, void *pBuf)
{
	SDDebug(DEFAULT_INFO, halUARTPrintf("halSDWriteNSector(): Sector=%d num=%d pBUf=0x%08x\n\r", nStartSect, nSectNum, pBuf));

	//return SdPhyWriteNSector(nStartSect * DEFAULT_BLK_SIZE, 0, nSectNum * DEFAULT_BLK_SIZE, pBuf);

	while(nSectNum > MAX_BLK_COUNT)
	{
		if(SdPhyWriteNSector(nStartSect * DEFAULT_BLK_SIZE, 0, MAX_BLK_COUNT * DEFAULT_BLK_SIZE, pBuf) == FALSE)
			return FALSE;
		nSectNum -= MAX_BLK_COUNT;
		nStartSect += MAX_BLK_COUNT;
		pBuf = (void *)((unsigned int)pBuf + MAX_BLK_COUNT * DEFAULT_BLK_SIZE);
	}
	if(nSectNum > 0) return SdPhyWriteNSector(nStartSect * DEFAULT_BLK_SIZE, 0, nSectNum * DEFAULT_BLK_SIZE, pBuf);
	return TRUE;

}

