#include "sys.h"

#include "dosFS.h"
#include "dosFSi.h"

static bool_T dosFATReadCache_ii(dosBS_S *, int nFATSector);

static int dosFAT12ReadEntry_ii(dosBS_S *, int nNThEntry);
static bool_T dosFAT12WriteEntry_ii(dosBS_S *, int nNThEntry, int nFEC);

static int dosFAT16ReadEntry_ii(dosBS_S *, int nNThEntry);
static bool_T dosFAT16WriteEntry_ii(dosBS_S *, int nNThEntry, int nFEC);

static int dosFATReadEntry_ii(dosBS_S *, int nNThEntry);
static bool_T dosFATWriteEntry_ii(dosBS_S *, int nNThEntry, int nFEC);



void dosFATCacheFlush_i(dosBS_S *pBS)
{
	int nIdx;
	int nAbsSector;

	if (pBS->p1bs.FATCache.isValid && pBS->p1bs.FATCache.isDirty)
	{
		for (nIdx = 0; nIdx < pBS->p1bs.numFATs; ++nIdx)
		{
			nAbsSector = pBS->p1bs.nFATStartSector +
						nIdx * pBS->p1bs.sectorsPerFAT +
						pBS->p1bs.FATCache.nFATSector;

//2006.06.12
#if 0
			dosWriteMediaSectorNByte_i(pBS, nAbsSector, 0,
						DOS_BYTES_PER_SEC, pBS->p1bs.FATCache.cacheBuf);
#else
			dosWriteMediaFATSector_i(pBS, nAbsSector);
#endif
		}

		pBS->p1bs.FATCache.isDirty = FALSE;
	}
}

static bool_T dosFATReadCache_ii(dosBS_S *pBS, int nFATSector)
{
	int nAbsSector;

	if (pBS->p1bs.FATCache.isValid &&
		nFATSector == pBS->p1bs.FATCache.nFATSector)
	{
		return (TRUE);
	}

	if (pBS->p1bs.FATCache.isValid && pBS->p1bs.FATCache.isDirty)
		dosFATCacheFlush_i(pBS);

	/* read the 0th FAT will be okay */
	nAbsSector = pBS->p1bs.nFATStartSector + nFATSector;

//2006.06.12
#if 0
	if (dosReadMediaSectorNByte_i(pBS, nAbsSector,
			0, DOS_BYTES_PER_SEC, pBS->p1bs.FATCache.cacheBuf))
#else
	if (dosReadMediaFATSector_i(pBS, nAbsSector))
#endif
	{
		pBS->p1bs.FATCache.nFATSector = nFATSector;
		pBS->p1bs.FATCache.isValid = TRUE;
		pBS->p1bs.FATCache.isDirty = FALSE;

		return (TRUE);
	}
	else
	{
		return (FALSE);
	}
}



/*
 *	FAT12 alignment:
 *
 *
 *		+---- N-th byte
 *		|
 *		|                +----(N+1)-th byte
 *		|                |
 *		|                |                +----(N+2)
 *		|                |                |
 *	   \|/              \|/              \|/
 *	    V                V                V
 *		+----------------+----------------+----------------+
 *		|        12 bits         ||        12 bits         |
 *		+----------------+----------------+----------------+
 *		kind 1                    kind 2
 *
 *
 *      kind 1 : 1st byte + half low byte of the 2nd byte
 *
 *      kind 2 : half high byte of the 1st byte + 2nd byte
 *
 *		p.s. 1. read 2 bytes for each FAT entry
 */

static int dosFAT12ReadEntry_ii(dosBS_S *pBS, int nNThEntry)
{
	unsigned char *pCacheBuf;
	int nFirstByte, nSecondByte;
	int nFATSector;
	int nFEC;

	pCacheBuf = pBS->p1bs.FATCache.cacheBuf;
	nFirstByte = (12 * nNThEntry)/8;
	nSecondByte= nFirstByte + 1;

	if (nSecondByte % DOS_BYTES_PER_SEC == 0)
	{
		/* the two bytes are located at different sectors */

		nFEC = 0;

		if ((12 * nNThEntry) % 8 == 0)
		{
			/* kind 1 
				8bit byte is in the first sector and
				4bit half byte is in the second sector
			*/

			nFATSector = nFirstByte / DOS_BYTES_PER_SEC;
			/* if the FATSector isn't in cache, read it */
			if (nFATSector != pBS->p1bs.FATCache.nFATSector)
			{
				dosFATCacheFlush_i(pBS);
				if (!dosFATReadCache_ii(pBS, nFATSector))
					return (-1);	/* make the program crazy */
			}
			nFEC = pCacheBuf[511];
            
            
			nFATSector += 1;
			dosFATCacheFlush_i(pBS);
			if (!dosFATReadCache_ii(pBS, nFATSector))
				return (-1);	/* make the program crazy */
			nFEC += (pCacheBuf[0] & 0x0F) << 8;
		}
		else
		{
			/* kind 2
				4bit half byte is in the first sector and
				8bit byte is in the second sector
			*/

			nFATSector = nFirstByte / DOS_BYTES_PER_SEC;
			/* if the FATSector isn't in the cache, read it */
			if (nFATSector != pBS->p1bs.FATCache.nFATSector)
			{
				dosFATCacheFlush_i(pBS);
				if (!dosFATReadCache_ii(pBS, nFATSector))
					return (-1);
			}
			nFEC = (pCacheBuf[511] & 0xF0) >> 4;

			nFATSector += 1;
			dosFATCacheFlush_i(pBS);
			if (!dosFATReadCache_ii(pBS, nFATSector))
				return (-1);
			nFEC += pCacheBuf[0] << 4;
		}
	}
	else
	{
		/* the two bytes are located at the same sector */

		nFATSector = nFirstByte / DOS_BYTES_PER_SEC;
		nFirstByte = nFirstByte % DOS_BYTES_PER_SEC;
		nSecondByte= nSecondByte% DOS_BYTES_PER_SEC;

		/* if the FATSector isn't in cache, read it */
		if (nFATSector != pBS->p1bs.FATCache.nFATSector)
		{
			dosFATCacheFlush_i(pBS);
			if (!dosFATReadCache_ii(pBS, nFATSector))
				return (-1);	/* make the program crazy */
		}

		nFEC = 0;
		if ((12 * nNThEntry) % 8 == 0)
		{
			/* kind 1, refer to FAT12 alignment */
			nFEC = pCacheBuf[nFirstByte];
			nFEC += (pCacheBuf[nSecondByte] & 0x0F) << 8;
		}
		else
		{
			/* kind 2, refer to FAT12 alignment */
			nFEC = (pCacheBuf[nFirstByte] & 0xF0) >> 4;
			nFEC += pCacheBuf[nSecondByte] << 4;
		}
	}

	return (nFEC);
}

static bool_T dosFAT12WriteEntry_ii(dosBS_S *pBS, int nNThEntry, int nFEC)
{
	unsigned char *pCacheBuf;
	int nFirstByte, nSecondByte;
	int nFATSector;

	pCacheBuf = pBS->p1bs.FATCache.cacheBuf;
	nFirstByte = (12 * nNThEntry)/8;
	nSecondByte= nFirstByte + 1;

	if (nSecondByte % DOS_BYTES_PER_SEC == 0)
	{
		/* the two bytes are located at different sectors */

		if ((12 * nNThEntry) % 8 == 0)
		{
			/* kind 1
				8bit byte is in the first sector and
				4bit half byte is in the second sector
			*/

			/* write bit 0 1 2 3 4 5 6 7 */
			nFATSector = nFirstByte / DOS_BYTES_PER_SEC;
			/* if the FATSector isn't in cache, read it */
			if (nFATSector != pBS->p1bs.FATCache.nFATSector)
			{
				dosFATCacheFlush_i(pBS);
				if (!dosFATReadCache_ii(pBS, nFATSector))
					return (FALSE);
			}
			pCacheBuf[511] = (nFEC & 0xFF);
			pBS->p1bs.FATCache.isDirty = TRUE;
			dosFATCacheFlush_i(pBS);


			/* write bit 8 9 10 11 */
			nFATSector += 1;
			if (!dosFATReadCache_ii(pBS, nFATSector))
				return (FALSE);
			pCacheBuf[0] &= 0xF0; /* clear lower half byte */
			pCacheBuf[0] += ((nFEC & 0xF00) >> 8); /* write to lower half byte */
			pBS->p1bs.FATCache.isDirty = TRUE;
		}
		else
		{
			/* kind 2 
				4bit half byte is in the first sector and
				8bit byte is in the second sector 
			*/

			/* write bit 0 1 2 3 */
			nFATSector = nFirstByte / DOS_BYTES_PER_SEC;
			if (nFATSector != pBS->p1bs.FATCache.nFATSector)
			{
				dosFATCacheFlush_i(pBS);
				if (!dosFATReadCache_ii(pBS, nFATSector))
					return (FALSE);
			}
			pCacheBuf[511] &= 0x0F; /* clear high half byte */
			pCacheBuf[511] += ((nFEC & 0x00F) << 4); /* write to high half byte */
			pBS->p1bs.FATCache.isDirty = TRUE;
			dosFATCacheFlush_i(pBS);


			/* write bit 4 5 6 7 8 9 10 11 */
			nFATSector += 1;
			if (!dosFATReadCache_ii(pBS, nFATSector))
				return (FALSE);
			pCacheBuf[0] = ((nFEC & 0xFF0) >> 4);
			pBS->p1bs.FATCache.isDirty = TRUE;
		}
	}
	else
	{
		/* the two bytes are located at the same sector */

		nFATSector = nFirstByte / DOS_BYTES_PER_SEC;
		nFirstByte = nFirstByte % DOS_BYTES_PER_SEC;
		nSecondByte= nSecondByte% DOS_BYTES_PER_SEC;

		/* Is the entry in the cache? */
		if (nFATSector != pBS->p1bs.FATCache.nFATSector)
		{
			dosFATCacheFlush_i(pBS);
			if (!dosFATReadCache_ii(pBS, nFATSector))
				return (FALSE);
		}
		
		pBS->p1bs.FATCache.isDirty = TRUE;

		/* write! */
		if ((12 * nNThEntry) % 8 == 0)
		{
			/* kind 1, refer to FAT12 alignment */

			pCacheBuf[nFirstByte]  = (nFEC & 0xFF);
			pCacheBuf[nSecondByte] &= 0xF0; /* clear low half-byte */
				/* the low half-byte is part of this fat entry and
					the high half-byte belongs to next fat entry */
			pCacheBuf[nSecondByte] += (unsigned char) ((nFEC & 0xF00) >> 8);
		}
		else
		{
			/* kind 2, refer to FAT12 alignment */

			pCacheBuf[nFirstByte] &= 0x0F;   /* clear high half-byte */
			pCacheBuf[nFirstByte] += ((nFEC & 0x00F) << 4);
			pCacheBuf[nSecondByte] = ((nFEC & 0xFF0) >> 4);
		}
	}

	return (TRUE);
}

static int dosFAT1632ReadEntry_ii(dosBS_S *pBS, int nNThEntry,
									int nFATEntrySize)
{
	unsigned int nFEC;
	int nFATEntriesPerSector;
	int nFATSector;
	int nOffset;


	nFATEntriesPerSector = DOS_BYTES_PER_SEC / nFATEntrySize;

	/* calculate the location in which the entry will be read */
	nFATSector = nNThEntry/nFATEntriesPerSector;
	nOffset = (nNThEntry%nFATEntriesPerSector) * nFATEntrySize;

	/* Is the entry not in the cache? */
	if (nFATSector != pBS->p1bs.FATCache.nFATSector)
	{
		dosFATCacheFlush_i(pBS);
		if (!dosFATReadCache_ii(pBS, nFATSector))
			return (-1);	/* make the program crazy */
	}

	dosvDecodeBufRoutine_g(pBS->p1bs.FATCache.cacheBuf+nOffset,
				nFATEntrySize, &nFEC);

	return ((int) nFEC);
}

static bool_T dosFAT1632WriteEntry_ii(dosBS_S *pBS, int nNThEntry,
								int nFEC, int nFATEntrySize)
{
	int nFATEntriesPerSector;
	int nFATSector;
	int nOffset;

	nFATEntriesPerSector = DOS_BYTES_PER_SEC / nFATEntrySize;

	/* calculate the location in which the entry will be read */
	nFATSector = nNThEntry/nFATEntriesPerSector;
	nOffset = (nNThEntry%nFATEntriesPerSector) * nFATEntrySize;

	/* Is the entry in the cache? */
	if (nFATSector != pBS->p1bs.FATCache.nFATSector)
	{
		dosFATCacheFlush_i(pBS);
		if (!dosFATReadCache_ii(pBS, nFATSector))
			return (FALSE);
	}

	dosvEncodeBufRoutine_g(pBS->p1bs.FATCache.cacheBuf+nOffset,
				nFATEntrySize, nFEC);
	pBS->p1bs.FATCache.isDirty = TRUE;

	return (TRUE);
}

static int dosFAT16ReadEntry_ii(dosBS_S *pBS, int nNThEntry)
{
	return (dosFAT1632ReadEntry_ii(pBS, nNThEntry, DOS_FAT16_ENTRY_SIZE));
}

static bool_T dosFAT16WriteEntry_ii(dosBS_S *pBS, int nNThEntry, int nFEC)
{
	return (dosFAT1632WriteEntry_ii(pBS, nNThEntry, nFEC, DOS_FAT16_ENTRY_SIZE));
}

static int dosFAT32ReadEntry_ii(dosBS_S *pBS, int nNThEntry)
{
	return (dosFAT1632ReadEntry_ii(pBS, nNThEntry, DOS_FAT32_ENTRY_SIZE));
}

static bool_T dosFAT32WriteEntry_ii(dosBS_S *pBS, int nNThEntry, int nFEC)
{
	return (dosFAT1632WriteEntry_ii(pBS, nNThEntry, nFEC, DOS_FAT32_ENTRY_SIZE));
}

static int dosFATReadEntry_ii(dosBS_S *pBS, int nNThEntry)
{
	switch (pBS->p1bs.fatSystem)
	{
	case DOS_FAT_BITS_12:
		return (dosFAT12ReadEntry_ii(pBS, nNThEntry));

	case DOS_FAT_BITS_16:
		return (dosFAT16ReadEntry_ii(pBS, nNThEntry));

	case DOS_FAT_BITS_32:
		return (dosFAT32ReadEntry_ii(pBS, nNThEntry));
	}

	return (0);
}

static bool_T dosFATWriteEntry_ii(dosBS_S *pBS, int nNThEntry, int nFEC)
{
	switch (pBS->p1bs.fatSystem)
	{
	case DOS_FAT_BITS_12:
		return (dosFAT12WriteEntry_ii(pBS, nNThEntry, nFEC));

	case DOS_FAT_BITS_16:
		return (dosFAT16WriteEntry_ii(pBS, nNThEntry, nFEC));

	case DOS_FAT_BITS_32:
		return (dosFAT32WriteEntry_ii(pBS, nNThEntry, nFEC));
	}

	return (FALSE);
}



bool_T dosFATIsNilFEC_i(dosBS_S *pBS, int nFATEntryCode)
{
	/*
		examine the FAT entry code to determine whether
		it is the "file's last cluster" code
	*/

	register int code;

	switch (pBS->p1bs.fatSystem)
	{
	case DOS_FAT_BITS_12:
		code = DOS_F12EC_LAST_CLUSTER_IN_FILE;	/* 0xFFF */
		break;

	case DOS_FAT_BITS_16:
		code = DOS_F16EC_LAST_CLUSTER_IN_FILE;	/* 0xFFFF */
		break;

	case DOS_FAT_BITS_32:
		code = DOS_F32EC_LAST_CLUSTER_IN_FILE;	/* 0x0FFF_FFFF */
		break;
	}

	/*
	 *         FF8 ~        FFF
	 *        FFF8 ~       FFFF
	 *   0FFF_FFF8 ~  0FFF_FFFF
	 */
	if (nFATEntryCode >= code-7)
		return (TRUE);
	else
		return (FALSE);
}

bool_T dosFATIsFreeFEC_i(dosBS_S *pBS, int nFATEntryCode)
{
	if (nFATEntryCode == DOS_FEC_FREE)
		return (TRUE);
	else
		return (FALSE);
}

/*
2006.06.27: not used by any other function
bool_T dosFATIsReservedFEC_i(dosBS_S *pBS, int nFATEntryCode)
{
	int code;

	if (pBS->p1bs.fatSystem == DOS_FAT_BITS_12)
		code = DOS_F12EC_RESERVED;
	else
		code = DOS_F16EC_RESERVED;

	if (nFATEntryCode == code)
		return (TRUE);
	else
		return (FALSE);
}
*/

int dosFATGetFEC_i(dosBS_S *pBS, int nNThEntry)
{
	return (dosFATReadEntry_ii(pBS, nNThEntry));
}

int dosFATGetNextCluster_i(dosBS_S *pBS, int nCurrent)
{
	return (dosFATReadEntry_ii(pBS, nCurrent));
}

int dosFATGetLatestCluster_i(dosBS_S *pBS, int nCurrent, int *pnRelativeIdx)
{
	/* find out the latest cluster in the "cluster linked list";
		The relative index is counted from 0 and nCurrent is 
		the 0th cluster. */

	int nLastCluster;
	int nNextCluster;

	*pnRelativeIdx = 0;
	nLastCluster = nCurrent;
	nNextCluster = nCurrent;
	for ( ; ; )
	{
		nLastCluster = nNextCluster;
		nNextCluster = dosFATGetNextCluster_i(pBS, nLastCluster);
		if (dosFATIsNilFEC_i(pBS, nNextCluster))
			break;
		*pnRelativeIdx += 1;
	}

	return (nLastCluster);
}

bool_T dosFATUpdateClusterEntry_i(dosBS_S *pBS, int nCurrent, int value)
{
	return (dosFATWriteEntry_ii(pBS, nCurrent, value));
}

int dosFATGetFreeCluster_i(dosBS_S *pBS)
{
	int nFEC;
	int nIdx;
	static int nSearchStart = 0;

	
	/* test the statistics variable whether there is any
		free FAT entry (free cluster)
	*/
	if (pBS->p1bs.numFreeFATEntries == 0)
	{
		return (0);
	}


	if (nSearchStart%pBS->p1bs.numPhyEntriesPerFAT == 0)
		nSearchStart = 2;	/* searching from the 2nd cluster */

	for (nIdx = nSearchStart +1;
		nIdx != nSearchStart; /* has searched all FAT entries */
		++nIdx) 
	{
		nFEC = dosFATReadEntry_ii(pBS, nIdx);
		if (dosFATIsFreeFEC_i(pBS, nFEC))
		{
			nSearchStart = nIdx;

			/* one cluster used */
			--(pBS->p1bs.numFreeFATEntries);

			return (nIdx);
		}
	}

	return (0);
}

void dosFATRecycleFileClusters_i(dosBS_S *pBS, int nFirstCluster)
{
	int nCurrentCluster;
	int nNextCluster;

	nCurrentCluster = nFirstCluster;
	for ( ; ; )
	{
		nNextCluster = dosFATGetNextCluster_i(pBS, nCurrentCluster);

		dosFATWriteEntry_ii(pBS, nCurrentCluster, DOS_FEC_FREE);


		/* one cluster released */
		++(pBS->p1bs.numFreeFATEntries);


		/*************************************************************
			The nCurrentCluster FAT entry's content is nNextCluster. This
			means that nCurrentCluster cluster points to nNextCluster cluster.
			If nNextCluster is a nil symbol, nCurrentCluster would be last
			cluster.
		**************************************************************/

		if (dosFATIsNilFEC_i(pBS, nNextCluster))
			break;

		nCurrentCluster = nNextCluster;
	}

	dosFATCacheFlush_i(pBS);
}

void dosFATCountFreeEntries_i(dosBS_S *pBS)
{
	register int nIdx;



	/* Don't count the free FAT entries too precisely!
		If there are n free FAT entries existing in the
		disk, report (n-m) free FAT entries. This can
		warn the application program that the disk is
		near to be full in advance.
	*/


	pBS->p1bs.numFreeFATEntries = 0;
	for (nIdx = 2; nIdx < pBS->p1bs.numPhyEntriesPerFAT; ++nIdx)
	{
		if (DOS_FEC_FREE == dosFATReadEntry_ii(pBS, nIdx))
			++(pBS->p1bs.numFreeFATEntries);
	}
}

