#include <common.h>
#include <nand.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/nand.h>
#include <asm/io.h>
#include <asm/errno.h>
#include <asm/arch/nand/nf_types.h>
#include <asm/arch/nand/nf_controller.h>
#include <asm/arch/nand/nf_memory.h>
#include <asm/arch/nand/nf_error.h>
#include <asm/arch/nand/nf_utils.h>
#include <asm/arch/hardware.h>

struct mtd_info *a1_mtd;

#define NAND_HW_CTRL_SIZE	0x400
#define NAND_CTRL_BUF_SIZE	8192

nf_controller_t 	a1_host;
nf_memory_t 		used_flash;

nf_controller_t *pcontroller 	= &a1_host;
nf_memory_t 	*nfmem 			= &used_flash;

static struct nand_ecclayout A1_OOB_16 = {
	.eccbytes = 4,
	.eccpos = {0, 1, 2, 3},
	.oobfree = {
		{.offset = 6,
		 .length = 10} },
};

static struct nand_ecclayout A1_OOB_64 = {
	.eccbytes = 28,
	.eccpos = {         32, 33, 34, 35, 36, 37, 38, 39,
				40, 41, 42, 43, 44, 45, 46, 47, 48, 49,
				50, 51, 52, 53, 54, 55, 56, 57, 58, 59},
	.oobfree = {
		{.offset = 1,
		 .length = 31} },
};

static struct nand_ecclayout A1_OOB_218 = {
	.eccbytes = 56,
	.eccpos = {      160, 161, 162, 163, 164, 165, 166, 167, 168, 169,
				170, 171, 172, 173, 174, 175, 176, 177, 178, 179,
				180, 181, 182, 183, 184, 185, 186, 187, 188, 189,
				190, 191, 192, 193, 194, 195, 196, 197, 198, 199,
				200, 201, 202, 203, 204, 205, 206, 207, 208, 209,
				210, 211, 212, 213, 214, 215	},
	.oobfree = {
		{.offset = 1,
		 .length = 159} },
};

static uint8_t scan_ff_pattern[] = { 0xff, };

static struct nand_bbt_descr A1_small_bbt = {
	.options = NAND_BBT_SCAN2NDPAGE,
	.offs = 5,
	.len = 1,
	.pattern = scan_ff_pattern
};

static struct nand_bbt_descr A1_large_bbt = {
	.options = NAND_BBT_SCAN2NDPAGE,
	.offs = 0,
	.len = 1,
	.pattern = scan_ff_pattern
};

/* Generic flash bbt decriptors
*/
static uint8_t bbt_pattern[] = {'A', 'C', 'T', 'i' };
static uint8_t mirror_pattern[] = {'i', 'T', 'C', 'A' };

static struct nand_bbt_descr small_bbt_main_descr = {
	.options = NAND_BBT_LASTBLOCK | NAND_BBT_CREATE | NAND_BBT_WRITE
		| NAND_BBT_2BIT | NAND_BBT_VERSION | NAND_BBT_PERCHIP,
	.offs =	6,
	.len = 4,
	.veroffs = 10,
	.maxblocks = 4,
	.pattern = bbt_pattern
};

static struct nand_bbt_descr small_bbt_mirror_descr = {
	.options = NAND_BBT_LASTBLOCK | NAND_BBT_CREATE | NAND_BBT_WRITE
		| NAND_BBT_2BIT | NAND_BBT_VERSION | NAND_BBT_PERCHIP,
	.offs =	6,
	.len = 4,
	.veroffs = 10,
	.maxblocks = 4,
	.pattern = mirror_pattern
};

static struct nand_bbt_descr bbt_main_descr = {
	.options = NAND_BBT_LASTBLOCK | NAND_BBT_CREATE | NAND_BBT_WRITE
		| NAND_BBT_2BIT | NAND_BBT_VERSION | NAND_BBT_PERCHIP,
	.offs =	1,
	.len = 4,
	.veroffs = 5,
	.maxblocks = 4,
	.pattern = bbt_pattern
};

static struct nand_bbt_descr bbt_mirror_descr = {
	.options = NAND_BBT_LASTBLOCK | NAND_BBT_CREATE | NAND_BBT_WRITE
		| NAND_BBT_2BIT | NAND_BBT_VERSION | NAND_BBT_PERCHIP,
	.offs =	1,
	.len = 4,
	.veroffs = 5,
	.maxblocks = 4,
	.pattern = mirror_pattern
};

static void a1_hwcontrol(struct mtd_info *mtd, int cmd, unsigned int bitmask)
{
	//TRACE();
}

uint8_t a1_nand_read_byte(struct mtd_info *mtd)
{
//	struct nand_chip *chip = mtd->priv;
	uint8_t data8;

	if(pcontroller->last_cmd == NAND_CMD_READID) {
		data8 = (pcontroller->dma_buffer[pcontroller->pio_bufptr++]);
		return data8;
	}
	// FIXME, hack
	else if(pcontroller->last_cmd == NAND_CMD_READOOB) {
		data8 = (pcontroller->dma_buffer[pcontroller->pio_bufptr++]);
		//printf( ">>>>   read_oob return %x\n", data8);
		return data8;
	}
	else if(pcontroller->last_cmd == NAND_CMD_STATUS) {
		return NAND_STATUS_WP;
	}
	else {
		printf ( "%s, last_cmd----[0x%x], return 0\n", 
			__FUNCTION__, pcontroller->last_cmd);
		return 0;
	}
}

static int a1_device_ready(struct mtd_info *mtd)
{
	return 1;
}

/**
 * nand_command - [DEFAULT] Send command to NAND device
 * @mtd:	MTD device structure
 * @command:	the command to be sent
 * @column:	the column address for this command, -1 if none
 * @page_addr:	the page address for this command, -1 if none
 *
 * Send command to NAND device. This function is used for small page
 * devices (256/512 Bytes per page)
 */

/* Used by the upper layer to write command to NAND Flash for
 * different operations to be carried out on NAND Flash */
void a1_nand_command(struct mtd_info *mtd, unsigned command,
				int column, int page_addr)
{
	int status = 0;
	unsigned int tmpbuf;

	A1_set_pin_mux (HAL_MFP_NAND);

	pcontroller->last_cmd = command;

	switch (command) {

		case NAND_CMD_READID:		
			pcontroller->ecc_enabled = 0;
			nf_mem_read_id(nfmem, pcontroller->dma_buffer);
			/* myhack, record last command and reset bufer index to the beginning */
			pcontroller->pio_bufptr = 0;
			return;

		case NAND_CMD_CACHEDPROG:	
			printf( "	...[NAND_CMD_CACHEDPROG]	FIXME!!	\n"); 		
			return;

		case NAND_CMD_PAGEPROG:		
			printf( "	...[NAND_CMD_PAGEPROG]		FIXME!!	\n"); 		
			return;


		case NAND_CMD_SEQIN:		
			/* we don't need to handle it */
			// just save the page, column address
			pcontroller->column = column;
			pcontroller->page_addr = page_addr;
			return;

		case NAND_CMD_ERASE1:		
			/*
			 * Basically, upper layer driver will divide a erase into two phases.
			 * step-1: send command "NAND_CMD_ERASE1", and "page address"
			 * step-2: send command "NAND_CMD_ERASE2", only
			 * soluton: we just need skip "NAND_CMD_ERASE2" command
			 */
			//printf( "	...[NAND_CMD_ERASE1]	page_shit=%d		\n", nfmem->pages_per_block_shift);
			pcontroller->ecc_enabled = 0;
			nf_mem_block_erase (nfmem, page_addr >> nfmem->pages_per_block_shift);
			break;
		case NAND_CMD_ERASE2:		
			/* we don't need to handle it */
		case NAND_CMD_RNDIN:		
		case NAND_CMD_STATUS:		
		case NAND_CMD_DEPLETE1:		
			/*
			 * read error status commands require only a short delay
			 */
		case NAND_CMD_STATUS_ERROR:
		case NAND_CMD_STATUS_ERROR1:
		case NAND_CMD_STATUS_ERROR2:
		case NAND_CMD_STATUS_ERROR3:
			return;
			
		case NAND_CMD_READOOB:
			pcontroller->pio_bufptr = 0;
			pcontroller->ecc_enabled = 0;
			nf_mem_page_read_oob(nfmem, page_addr, mtd->writesize, pcontroller->dma_buffer, mtd->oobsize);
			return;
	
		case NAND_CMD_RESET:
			nf_mem_reset(nfmem);
			return;
	
		case NAND_CMD_RNDOUT:
			return;
	
		case NAND_CMD_READ0:
			// we don't need to hook READ command here, upper will invoke chip->ecc.read_page()
			// printf( "	...[NAND_CMD_READ0]\n"); 
			pcontroller->column = column;
			pcontroller->page_addr = page_addr;
			return;

			/* This applies to read commands */
		default:
			/*
			 * If we don't have access to the busy pin, we apply the given
			 * command delay
			 */
			printf( "	...[default catch!!!!]\n"); 
	}
}

static int a1_nand_wait(struct mtd_info *mtd, struct nand_chip *chip)
{
		return 0;
}

static void a1_nand_select(struct mtd_info *mtd, int chipnr)
{
	if (chipnr > 0) {
		nf_ctrl_select_target(pcontroller, chipnr);
	}
}

/**
 * nand_write_page - [REPLACEABLE] write one page
 * @mtd:	MTD device structure
 * @chip:	NAND chip descriptor
 * @buf:	the data to write
 * @page:	page number to write
 * @cached:	cached programming
 * @raw:	use _raw version of write_page
 */
int a1_nand_write_page(struct mtd_info *mtd, struct nand_chip *chip,
			   const uint8_t *buf, int page, int cached, int raw)
{
	// save page to host, mtd don't send "page" info to    chip->ecc.write_page_XXX 
	pcontroller->page_addr = page;

	if (unlikely(raw))
		chip->ecc.write_page_raw(mtd, chip, buf);
	else
		chip->ecc.write_page(mtd, chip, buf);
	return 0;
}

/**
 * a1_hwecc_read_page- [REPLACABLE] hardware ecc based page read function
 * @mtd:	mtd info structure
 * @chip:	nand chip info structure
 * @buf:	buffer to store read data
 * @page:	page number to read
 */
int a1_hwecc_read_page(struct mtd_info *mtd, 
		struct nand_chip *chip, uint8_t *buf)
{
	int status = 0;
	uint32_t page_number = pcontroller->page_addr;

	A1_set_pin_mux (HAL_MFP_NAND);

	pcontroller->ecc_enabled = 0;

	if( nf_mem_page_empty(nfmem, page_number) == 0)
		pcontroller->ecc_enabled = 1;

	status = nf_mem_page_read(nfmem, page_number, 0, buf, mtd->writesize);

	// FIXME : TODO
	// Need to fill up following two variable ???????

	// mtd->ecc_stats.failed++;
	// mtd->ecc_stats.corrected += stat;

	if (status == 0)
		return 0;
	else 
	{
		printf(" FIXME: <<%s>>, pg_no = %d\n", __FUNCTION__, page_number);
		return -EIO;
	}
}

void a1_hwecc_write_page(struct mtd_info *mtd,
					      struct nand_chip *chip,
					      const uint8_t *buf)
{
	int status = 0;
	uint32_t page_number = pcontroller->page_addr;
	uint32_t	tmp_buf = pcontroller->dma_buffer + pcontroller->page_size;
	uint8_t *p = (uint8_t *)tmp_buf;
	uint32_t *eccpos = chip->ecc.layout->eccpos;
	int i;

	A1_set_pin_mux (HAL_MFP_NAND);

	pcontroller->ecc_enabled = 1;
	status = nf_mem_page_program(nfmem, page_number, 0, buf, mtd->writesize);

	if (status != 0) {
		printf ( "%s: status = 0x%x\n", __FUNCTION__, status);
	}

	// read oob
	nf_mem_page_read_oob(nfmem, page_number, nfmem->page_size, tmp_buf, mtd->oobsize);

	// combined oob with ecc
	for (i = 0; i < chip->ecc.total; i++)
		chip->oob_poi[eccpos[i]] = p[eccpos[i]];

	// write oob back
	status = nf_mem_page_program_oob(nfmem, page_number, nfmem->page_size, chip->oob_poi, mtd->oobsize);
	if (status != 0) {
		printf ( "%s: status = 0x%x\n", __FUNCTION__, status);
	}
}

/**
 * a1_nand_write_oob - [REPLACABLE] the most common OOB data write function
 * @mtd:	mtd info structure
 * @chip:	nand chip info structure
 * @page:	page number to write
 */
int a1_nand_write_oob(struct mtd_info *mtd, struct nand_chip *chip,
			      int page)
{
	int status = 0;
	const uint8_t *buf = chip->oob_poi;
	int length = mtd->oobsize;

	A1_set_pin_mux (HAL_MFP_NAND);

	status = nf_mem_page_program_oob(nfmem, page, nfmem->page_size, (uint8_t *)buf, length);
	return (status == 0) ? 0 : -EIO ;
}
/**
 * a1_nand_read_oob - [REPLACABLE] the most common OOB data read function
 * @mtd:	mtd info structure
 * @chip:	nand chip info structure
 * @page:	page number to read
 * @sndcmd:	flag whether to issue read command or not
 */
int a1_nand_read_oob(struct mtd_info *mtd, struct nand_chip *chip,
			     int page, int sndcmd)
{
	int status = 0;
	const uint8_t *buf = chip->oob_poi;
	int length = mtd->oobsize;

	pcontroller->ecc_enabled = 0;
	status = nf_mem_page_read_oob(nfmem, page, nfmem->page_size, (uint8_t *)buf, length);
	if (status != 0) {
		printf ( "%s: status = 0x%x\n", __FUNCTION__, status);
	}
	return 1;
}

void init_NF_MEM(char id_buf[])
{
	struct nand_chip	*chip;
	chip = (struct nand_chip *)a1_mtd->priv;

	pcontroller->nandid = *((unsigned long *)id_buf);
	nfmem->manufacturer_id = id_buf[0];
	nfmem->nandid = *((unsigned long *)id_buf);

	// 1G, K9K8G08U0B
//	printf(" %s: %d	a1_mtd->writesize = 0x%x\n", __FUNCTION__, __LINE__, a1_mtd->writesize);
	nfmem->page_size = a1_mtd->writesize;

//	printf(" %s: %d	a1_mtd->erasesize = 0x%x\n", __FUNCTION__, __LINE__, a1_mtd->erasesize);
	nfmem->pages_per_block = (a1_mtd->erasesize / a1_mtd->writesize);
//	printf(" %s: %d	nfmem->pages_per_block = 0x%x\n", __FUNCTION__, __LINE__, nfmem->pages_per_block);

//	printf(" %s: %d	a1_mtd->oobsize = 0x%x\n", __FUNCTION__, __LINE__, a1_mtd->oobsize);
	nfmem->spare_area_size = a1_mtd->oobsize;

	nfmem->blocks_count = (chip->chipsize / a1_mtd->erasesize);
//	printf(" %s: %d	nfmem->blocks_count = 0x%x\n", __FUNCTION__, __LINE__, nfmem->blocks_count);

	nfmem->cache_prog_supp = 0;
	nfmem->copy_back_support = 0;
	nfmem->page_shift = chip->page_shift;
	nfmem->pages_per_block_shift = get_fsb(nfmem->pages_per_block);
	nfmem->current_work_mode = MEMORY_ASYNCHRONOUS_MODE;

#if 0  //R:20121025 change nand addr cycle from 5 to 4
	nfmem->addr_cyc_cnt = (nfmem->page_size <= 512) ? 3 : 5;
#else
	nfmem->addr_cyc_cnt = (nfmem->page_size <= 512) ? 3 : 4;
#endif
	printf(" %s: %d	nfmem->addr_cyc_cnt = 0x%x\n", __FUNCTION__, __LINE__, nfmem->addr_cyc_cnt);

#if 1      
    /* RE# or WE# high hold time */
    nfmem->timings.trwh_ns = 10+5;
    /* RE# or WE# pulse width */
    nfmem->timings.trwp_ns = 15+5;
    /* WE# high to RE# low time */
    nfmem->timings.twhr_ns = 60+5;
    /* RE# high to WE# low time */
    nfmem->timings.trhw_ns = 100+5;
    /* ALE to data start time */
    //nfmem->timings.tadl_ns = 100;
    nfmem->timings.tadl_ns = 100+5;
    /* Change column setup */
    //nfmem->timings.tccs_ns = 200;
    nfmem->timings.tccs_ns = 100+0;
    #if 0
    /* Busy time for interface change.*/
	    nfmem->timings.twb_ns = 100 + 0;
    #else
	    /* Busy time for interface change.*/
	    nfmem->timings.twb_ns = 100 - 10;      //R:20121025 100 is max
    #endif

    /* Read high to Read low. */
    nfmem->timings.trr_ns = 20+5;
#endif
//#if 0
//    /* RE# or WE# high hold time */
//    nfmem->timings.trwh_ns = 20;
//    /* RE# or WE# pulse width */
//    nfmem->timings.trwp_ns = 60;
//    /* WE# high to RE# low time */
//    nfmem->timings.twhr_ns = 60;
//    /* RE# high to WE# low time */
//    nfmem->timings.trhw_ns = 100;
//    /* ALE to data start time */
//    //nfmem->timings.tadl_ns = 100;
//    nfmem->timings.tadl_ns = 100;
//    /* Change column setup */
//    //nfmem->timings.tccs_ns = 200;
//    nfmem->timings.tccs_ns = 200;
//    /* Busy time for interface change.*/
//    nfmem->timings.twb_ns = 100;
//    /* Read high to Read low. */
//    nfmem->timings.trr_ns = 20;
//#endif
}


/* Initialize nand memory */
int scan_and_init_NF_MEM ()
{
	char id_buf[8];

	A1_set_pin_mux (HAL_MFP_NAND);

	// Reset First, that is necessary for Large Page device
	nf_mem_reset(nfmem);

	// Read ID
	if (nf_mem_read_id(nfmem, id_buf) != NF_ERR_NO_ERRORS) {
		printf("    : Failed to detect nand chip\n");
		return -1;
	}
//	printf("  NAND Chip Signature: %x-%x-%x-%x\n", id_buf[0], id_buf[1], id_buf[2], id_buf[3]);
	init_NF_MEM (id_buf);
	return 0;
}

/**
 * nand_write_page_raw - [Intern] raw page write function
 * @mtd:	mtd info structure
 * @chip:	nand chip info structure
 * @buf:	data buffer
 *
 * Not for syndrome calculating ecc controllers, which use a special oob layout
 */
void a1_nand_write_page_raw(struct mtd_info *mtd, struct nand_chip *chip,
			      uint8_t *buf)
{
	uint32_t page_number = pcontroller->page_addr;
	unsigned int tmpbuf;

	// raw read page first without ecc
	pcontroller->ecc_enabled = 0;
	nf_mem_page_program(nfmem, page_number, 0, buf, mtd->writesize);
	nf_mem_page_program_oob(nfmem, page_number, nfmem->page_size, chip->oob_poi, mtd->oobsize);
}


/**
 * nand_read_page_raw - [Intern] read raw page data without ecc
 * @mtd:	mtd info structure
 * @chip:	nand chip info structure
 * @buf:	buffer to store read data
 * @page:	page number to read

	chip->read_buf(mtd, buf, mtd->writesize);
	chip->read_buf(mtd, chip->oob_poi, mtd->oobsize);
 *
 * Not for syndrome calculating ecc controllers, which use a special oob layout
 */
int a1_nand_read_page_raw(struct mtd_info *mtd, struct nand_chip *chip,
			      uint8_t *buf)
{
	int status = 0;
	uint32_t page_number = pcontroller->page_addr;
	unsigned int tmpbuf;

	// switch MFS to SMC
	A1_set_pin_mux (HAL_MFP_NAND);

	// raw read page first without ecc
	if (pcontroller->last_cmd == NAND_CMD_READ0) {
		pcontroller->ecc_enabled = 0;
		nf_mem_page_read(nfmem, page_number, 0, buf, mtd->writesize);
		nf_mem_page_read_oob(nfmem, page_number, nfmem->page_size, chip->oob_poi, mtd->oobsize);
		return 0;
	}
	else {
		printf("Help!! ......[%s:%d]BUG!!\n", __FUNCTION__, __LINE__);
		return -EIO;
	}
}

/*
 * Register nand_chip's callbacks.
 */
void nand_chip_init_fn(struct nand_chip *chip)
{
	chip->options &= ~NAND_BUSWIDTH_16;	// we support 8 bits only
	chip->cmdfunc	 = a1_nand_command;
	chip->cmd_ctrl	 = a1_hwcontrol;
	chip->read_byte  = a1_nand_read_byte;
	chip->dev_ready  = a1_device_ready;
	chip->waitfunc	 = a1_nand_wait;
	chip->select_chip = a1_nand_select;
/** function to write a page according to the ecc generator requirements **/
	chip->write_page = a1_nand_write_page;	
//	chip->block_bad = a1_nand_block_bad;

	//FIXME: following ecc related function
	chip->ecc.mode   = NAND_ECC_HW;
	chip->ecc.read_page = a1_hwecc_read_page;
	chip->ecc.write_page = a1_hwecc_write_page;
	chip->ecc.write_oob = a1_nand_write_oob;
	chip->ecc.read_oob = a1_nand_read_oob;
	chip->ecc.read_page_raw = a1_nand_read_page_raw;
	chip->ecc.write_page_raw = a1_nand_write_page_raw;
}

 
void nand_chip_init_ecc(struct nand_chip *chip)
{
	// Specify ecc_layout
	chip->ecc.size = nfmem->page_size;		

	if (nfmem->page_size <= 512) {
		chip->ecc.layout = &A1_OOB_16;
		chip->badblock_pattern = &A1_small_bbt;
		chip->bbt_td = &small_bbt_main_descr;
		chip->bbt_md = &small_bbt_mirror_descr;
		chip->ecc.bytes = 4;	// 2-bit correction
		chip->options   |= (NAND_USE_FLASH_BBT | NAND_NO_READRDY |NAND_NO_AUTOINCR );
		return;
	}
	else if (nfmem->page_size == 2048)
		chip->ecc.layout = &A1_OOB_64;

	else if (nfmem->page_size == 4096)
		chip->ecc.layout = &A1_OOB_218;

	chip->badblock_pattern = &A1_large_bbt;
	chip->bbt_td = &bbt_main_descr;
	chip->bbt_md = &bbt_mirror_descr;
	chip->ecc.bytes = 7;	// 4-bit correction

	/* The NAND_SKIP_BBTSCAN option is necessary for 'overridesize' */
	/* and 'badblocks' parameters to work */
	chip->options   |= (NAND_USE_FLASH_BBT | NAND_NO_READRDY |NAND_NO_AUTOINCR );
}

int board_nand_init(struct nand_chip *chip)
{
	/* MTD structure for NAND controller */
	static int chip_nr = 0;
	int retval = -ENOMEM;

	A1_set_pin_mux (HAL_MFP_NAND);

	a1_mtd = &nand_info[chip_nr++];
	a1_mtd->priv = chip;

	// Link hw-resources to instance
	pcontroller->reg_offset = (uint32_t)CONFIG_SYS_NAND_BASE;
	pcontroller->dma_buffer = (uint32_t)CONFIG_A1_NAND_DMA_BUFFER;
	pcontroller->dma_buff_address = (uint32_t)CONFIG_A1_NAND_DMA_BUFFER;

	// Do flash memory instance
	nfmem->pcontroller = pcontroller;
	nfmem->ce = 0;

	// initialize controller
	nf_ctrl_initialize(pcontroller);

	//Register nand_chip's callbacks.
	nand_chip_init_fn(chip);

	nand_scan_ident(a1_mtd, 1);

	/* use nf_mem_read_id library to get basic nand chip information first */
	retval = scan_and_init_NF_MEM();
	if (retval != 0) {
		printf("unable to detected nand chip\n");
		goto a1_exit;
	}

    // apply timing settings
    nf_ctrl_configure_timings(pcontroller, &(nfmem->timings));  

	nand_chip_init_ecc(chip);
	
	nf_mem_select_target(nfmem);

a1_exit:	
	return  retval;

}

