/*
 * Basic I2C functions
 *
 * Copyright (c) 2004 Texas Instruments
 *
 * This package is free software;  you can redistribute it and/or
 * modify it under the terms of the license found in the file
 * named COPYING that should have accompanied this file.
 *
 * THIS PACKAGE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 * Author: Jian Zhang jzhang@ti.com, Texas Instruments
 *
 * Copyright (c) 2003 Wolfgang Denk, wd@denx.de
 * Rewritten to fit into the current U-Boot framework
 *
 * Adapted for OMAP2420 I2C, r-woodruff2@ti.com
 *
 */

#include <common.h>

#include <asm/arch/hardware.h>
#include <asm/arch/i2c.h>
#include <asm/io.h>

/* Default I2C0 */
static A1_i2c i2c[3] = { 
		{A1_I2C0_BASE, 0, CONFIG_SYS_I2C_SPEED, CONFIG_SYS_I2C0_SLAVE},
		{A1_I2C1_BASE, 0, CONFIG_SYS_I2C_SPEED, CONFIG_SYS_I2C1_SLAVE},
		{A1_I2C2_BASE, 0, CONFIG_SYS_I2C_SPEED, -1} 
};

u32	i2c_base = A1_I2C0_BASE;

static int current_bus = 0;

inline void i2c_disable(void)
{
	writel (IIC_DISABLE, i2c_base + ENABLE_REG);
}

inline void i2c_enable(void)
{
	writel (IIC_ENABLE, i2c_base + ENABLE_REG);
}

/* Bus busy wait */
inline void bb_wait()
{
	int timeout = 20;

	/* Busy waiting for IF */
	while ( ((readl (i2c_base + CONTROL_REG) & 0x3) != IIC_CTRL_IF) && timeout--) {
		udelay(1000);
	}

}

void inline i2c_addr(u8 devaddr, u8 direction)
{
	/* Send I2C slave address first, direction [W] */
	writel ((devaddr << 1) | direction, (i2c_base + DATA_REG));

	/* START */
	writel (IIC_CTRL_START, (i2c_base + CONTROL_REG));

	bb_wait();
}

void inline i2c_select_reg(u8 regoffset)
{
	/* Send register offset */
	writel (regoffset, (i2c_base + DATA_REG));

	/* clear interrupt and send data */
	writel (0, (i2c_base + CONTROL_REG));

	bb_wait();
}

void inline i2c_stop()
{
	/* STOP */
	writel (IIC_CTRL_STOP, (i2c_base + CONTROL_REG));
}
/*
 *	programming sequence :
 *
 *	Start -> Slave address (W) -> ACK-s -> 
 *	slave sub-address (M bytes) -> -> ACK-s
 *	-> repeated Start (Sr) -> SLAVE ADDRESS (R) -> ACK-s
 *	-> DATA (N Bytes) -> ACK-m -> STOP
 */
static int i2c_read_byte (u8 devaddr, u8 regoffset, u8 * value)
{
	/************************************************************
	 * Setup Phase
	 * 		Write slave address,
	 *  	Write subreg address
	 ***********************************************************/
	i2c_addr(devaddr, IIC_WRITE);

	i2c_select_reg(regoffset);

	/************************************************************
	 * Data Phase 
	 *		Read data back
	 ***********************************************************/

	/* Send I2C slave address first, direction [R] */
	i2c_addr(devaddr, IIC_READ);

	writel (IIC_CTRL_ACK, (i2c_base + CONTROL_REG));

	bb_wait();

	/* retrieve data from data register */
	*value = (u8) readl (i2c_base + DATA_REG);

	i2c_stop();

	return 0;
}

/*
 *	programming sequence :
 *
 *	START -> slave address -> slave sub-address (M bytes) ->
 *	data (N bytes) -> STOP
 *
 */

/*
 *	While wirting Ctrl as 0, the following actions are done:
 *		1. clear interrupt
 *		2. the value in Data register is transmitted.
 */
static int i2c_write_byte (u8 devaddr, u8 regoffset, u8 value)
{

	/************************************************************
	 * Setup Phase
	 * 		Write slave address,
	 *  	Write subreg address
	 ***********************************************************/
	i2c_addr(devaddr, IIC_WRITE);

	i2c_select_reg(regoffset);

	/************************************************************
	 * Data Phase 
	 *		Read data back
	 ***********************************************************/
	writel (value, i2c_base + DATA_REG);

	/* clear interrupt and send data */
	writel (0, (i2c_base + CONTROL_REG));

	bb_wait();

	i2c_stop();

	return 0;
}


/**
 * i2c_read: - Read multiple bytes from an i2c device
 *
 * The higher level routines take into account that this function is only
 * called with len < page length of the device (see configuration file)
 *
 * @chip:	address of the chip which is to be read
 * @addr:	i2c data address within the chip
 * @alen:	length of the i2c data address (1..2 bytes)
 * @buffer:	where to write the data
 * @len:	how much byte do we want to read
 * @return:	0 in case of success
 */

int i2c_read (uchar chip, uint addr, int alen, uchar * buffer, int len)
{
	int i;

	iic_trace();

	if (alen > 1) {
		printf ("I2C read: addr len %d not supported\n", alen);
		return 1;
	}

	if (addr + len > 256) {
		printf ("I2C read: address out of range\n");
		return 1;
	}

	for (i = 0; i < len; i++) {
		if (i2c_read_byte (chip, addr + i, &buffer[i])) {
			printf ("I2C read: I/O error\n");
			return 1;
		}
	}

	return 0;
}

/**
 * i2c_write: -  Write multiple bytes to an i2c device
 *
 * The higher level routines take into account that this function is only
 * called with len < page length of the device (see configuration file)
 *
 * @chip:	address of the chip which is to be written
 * @addr:	i2c data address within the chip
 * @alen:	length of the i2c data address (1..2 bytes)
 * @buffer:	where to find the data to be written
 * @len:	how much byte do we want to read
 * @return:	0 in case of success
 */

int i2c_write (uchar chip, uint addr, int alen, uchar * buffer, int len)
{
	int i;

	iic_trace();

	if (alen > 1) {
		printf ("I2C read: addr len %d not supported\n", alen);
		return 1;
	}

	if (addr + len > 256) {
		printf ("I2C read: address out of range\n");
		return 1;
	}

	for (i = 0; i < len; i++) {
		if (i2c_write_byte (chip, addr + i, buffer[i])) {
			printf ("I2C read: I/O error\n");
			return 1;
		}
	}

	return 0;
}

/*
 * for slaveadd, we don't need it
 */
void i2c_init(int speed, int slaveadd)
{
	iic_trace();

//	if (i2c[current_bus].slaveaddr != slaveadd)
//		printf ("Warning: i2c_init() get slave addr differs from configuration\n");

	/* init global i2c_base */
	i2c_base = i2c[current_bus].base;

	/* Set I2C interface speed */
	i2c_set_bus_speed(speed);

	/* Enable it */
	i2c_enable ();

	i2c[current_bus].init = 1;
}


int i2c_set_bus_num(unsigned int bus)
{
	iic_trace();
	if ((bus < 0) || (bus >= I2C_BUS_MAX)) {
		printf("Bad bus: %d\n", bus);
		return -1;
	}

	current_bus = bus;
	i2c_base = i2c[current_bus].base;

	if(!i2c[bus].init)
		i2c_init(CONFIG_SYS_I2C_SPEED, (int)i2c[current_bus].slaveaddr);

	return 0;
}

int i2c_get_bus_num(void)
{
	iic_trace();
	return (int) current_bus;
}

/**
 * i2c_set_bus_speed - set i2c bus speed
 *	@speed: bus speed (in HZ)
 *  @return: success 0, faild != 0
 */
int i2c_set_bus_speed(unsigned int speed)
{
	int nPCLK = (CONFIG_SYS_CLK_FREQ / 2);
	int scale;
	
	iic_trace();

	if (!speed)
		speed = i2c[current_bus].speed;

	scale = nPCLK / (4 * speed);
	
	writel (scale & 0xFF, i2c_base + PSCALE_LO_REG);
	writel ((scale >> 8) & 0xFF, i2c_base + PSCALE_HI_REG);

	i2c[current_bus].speed = speed;

	return 0;
}

/**
 * i2c_get_bus_speed - get i2c bus speed
 *	@speed: bus speed (in HZ)
 */
unsigned int i2c_get_bus_speed(void)
{
	return i2c[current_bus].speed;
}

/**
 * i2c_probe: - Test if a chip answers for a given i2c address
 *
 * @chip:	address of the chip which is searched for
 * @return:	0 if a chip was found, -1 otherwhise
 */

int i2c_probe (uchar chip)
{
	int timeout = 20;
	int res = -1; /* default = fail */

	i2c_addr(chip, IIC_WRITE);


	/* Busy waiting for IF */
	while ( ((readl (i2c_base + CONTROL_REG) & 0x3) != IIC_CTRL_IF) && timeout--) {
		udelay(1000);
	}

	i2c_stop();

	return (timeout <=0 ) ? -1 : 0 ;

	return res;
}


