/*------------------------------------------------------------------------------
--                                                                            --
--       This software is confidential and proprietary and may be used        --
--        only as expressly authorized by a licensing agreement from          --
--                                                                            --
--                            Hantro Products Oy.                             --
--                                                                            --
--                   (C) COPYRIGHT 2006 HANTRO PRODUCTS OY                    --
--                            ALL RIGHTS RESERVED                             --
--                                                                            --
--                 The entire notice above must be reproduced                 --
--                  on all copies and should not be removed.                  --
--                                                                            --
--------------------------------------------------------------------------------
--
--  Description : System Wrapper Layer for Linux using IRQs
--
------------------------------------------------------------------------------
--
--  Version control information, please leave untouched.
--
--  $RCSfile: dwl_x170_linux_irq.c,v $
--  $Revision: 1.13 $
--  $Date: 2009/04/08 14:03:08 $
--
------------------------------------------------------------------------------*/

#include "basetype.h"
#include "dwl.h"
#include "dwl_linux.h"

//#include <assert.h>
#define assert(...)  /* no assertion */

//#include <stdio.h>
//#include <stdlib.h>
//#include <string.h>

#ifdef USE_EFENCE
#include "efence.h"
#endif


#ifdef _DWL_HW_PERFORMANCE
#define HW_PERFORMANCE_LOG "hw_performance.log"
#define HW_PERFORMANCE_RESOLUTION 0.00001   /* 10^-5 */
static FILE *hw_performance_log = NULL;
static u32 dec_pic_counter = 0;
static u32 pp_pic_counter = 0;
u32 dec_time;                /* decode time in milliseconds */
struct timeval dec_start_time;
struct timeval pp_start_time;
u32 pp_time /* pp time in milliseconds */ ;
#endif

#define X170_SEM_KEY 0x8070


enum {
	DWL_DEC_HW_READY = 0x1,
	DWL_PP_HW_READY  = 0x2,
};


//volatile unsigned int irq_count = 0;


/*------------------------------------------------------------------------------
    Function name   : DWLInit
    Description     : Initialize a DWL instance

    Return type     : const void * - pointer to a DWL instance

    Argument        : void * param - not in use, application passes NULL
------------------------------------------------------------------------------*/
const void *DWLInit(DWLInitParam_t * param)
{
    hX170dwl_t *dec_dwl;
    unsigned long base;

    dec_dwl = (hX170dwl_t *) DWLmalloc(sizeof(hX170dwl_t));

    DWL_DEBUG("DWL: INITIALIZE\n\r");
    if(dec_dwl == NULL) {
        DWL_DEBUG("DWL: failed to alloc hX170dwl_t struct\n\r");
        return NULL;
    }

	dec_dwl->clientType = param->clientType;

    switch (dec_dwl->clientType) {
    case DWL_CLIENT_TYPE_H264_DEC:
    case DWL_CLIENT_TYPE_MPEG4_DEC:
    case DWL_CLIENT_TYPE_JPEG_DEC:
    case DWL_CLIENT_TYPE_VC1_DEC:
    case DWL_CLIENT_TYPE_MPEG2_DEC:
    case DWL_CLIENT_TYPE_VP6_DEC:
    case DWL_CLIENT_TYPE_RV_DEC:
    case DWL_CLIENT_TYPE_PP:
		break;
    default:
		goto err;
		break;
    }

    /* map the hw registers to user space */
    dec_dwl->pRegBase =
        DWLMapRegisters(dec_dwl->fd_mem, base, dec_dwl->regSize, 1);

    return dec_dwl;

err:
    DWLRelease(dec_dwl);

    return NULL;
}

/*------------------------------------------------------------------------------
    Function name   : DWLRelease
    Description     : Release a DWl instance

    Return type     : i32 - 0 for success or a negative error code

    Argument        : const void * instance - instance to be released
------------------------------------------------------------------------------*/
i32 DWLRelease(const void *instance)
{
    hX170dwl_t *dec_dwl = (hX170dwl_t *) instance;

    assert(dec_dwl != NULL);

    if(dec_dwl->pRegBase != NULL) {
        DWLUnmapRegisters((void *) dec_dwl->pRegBase, dec_dwl->regSize);
	}

    DWLfree(dec_dwl);

    return (DWL_OK);
}

/*------------------------------------------------------------------------------
    Function name   : DWLWaitDecHwReady
    Description     : Wait until hardware has stopped running
                        and Decoder interrupt comes.
                      Used for synchronizing software runs with the hardware.
                      The wait could succed, timeout, or fail with an error.

    Return type     : i32 - one of the values DWL_HW_WAIT_OK
                                              DWL_HW_WAIT_TIMEOUT
                                              DWL_HW_WAIT_ERROR

    Argument        : const void * instance - DWL instance
------------------------------------------------------------------------------*/
i32 DWLWaitDecHwReady(const void *instance, u32 timeout)
{
	return DWLWaitPpHwReady(instance, timeout);
#if 0
    hX170dwl_t *dec_dwl = (hX170dwl_t *) instance;

    DWL_DEBUG("DWLWaitDecHwReady: Start\n\r");

    /* Check invalid parameters */
    if(dec_dwl == NULL) {
        assert(0);
        return DWL_HW_WAIT_ERROR;
    }

	while ((irq_count & DWL_DEC_HW_READY) == 0) { ; } // Waiting for the interrupt
	irq_count &= ~DWL_DEC_HW_READY; // Restore the ISR flag

    DWL_DEBUG("DWLWaitDecHwReady: OK\n\r");
    return DWL_HW_WAIT_OK;
#endif // 0
}

/*------------------------------------------------------------------------------
    Function name   : DWLWaitPpHwReady
    Description     : Wait until hardware has stopped running
                      and pp interrupt comes.
                      Used for synchronizing software runs with the hardware.
                      The wait could succed, timeout, or fail with an error.

    Return type     : i32 - one of the values DWL_HW_WAIT_OK
                                              DWL_HW_WAIT_TIMEOUT
                                              DWL_HW_WAIT_ERROR

    Argument        : const void * instance - DWL instance
------------------------------------------------------------------------------*/
i32 DWLWaitPpHwReady(const void *instance, u32 timeout)
{
	unsigned int i;
	hX170dwl_t *dec_dwl = (hX170dwl_t *) instance;
	volatile u32 irq_stats;

	u32 irqRegOffset;

	assert(dec_dwl != NULL);

    if (dec_dwl->clientType == DWL_CLIENT_TYPE_PP)
        irqRegOffset = HX170PP_REG_START;   /* pp ctrl reg offset */
    else
        irqRegOffset = HX170DEC_REG_START;  /* decoder ctrl reg offset */

    /* wait for decoder */
    irq_stats = DWLReadReg(dec_dwl, irqRegOffset);
    irq_stats = (irq_stats >> 12) & 0xFF;

    if (irq_stats != 0) {
		DWL_DEBUG("DWLWaitHwReady: 0x%08x\n", irq_stats);
        return DWL_HW_WAIT_OK;
    } else {
		int loop = 1;
		i32 ret = DWL_HW_WAIT_TIMEOUT;

		do {
			// Have CPU Spin here
			for (i = 0; i < 50; i++) ;

			irq_stats = DWLReadReg(dec_dwl, irqRegOffset);
			irq_stats = (irq_stats >> 12) & 0xFF;

			if (irq_stats != 0) {
				DWL_DEBUG("DWLWaitHwReady: 0x%08x\n", irq_stats);
				ret = DWL_HW_WAIT_OK;
				loop = 0;   /* end the polling loop */
            }
		} while (loop);

        return ret;
    }
#if 0
    hX170dwl_t *dec_dwl = (hX170dwl_t *) instance;

    DWL_DEBUG("DWLWaitPpHwReady: Start\n\r");

    /* Check invalid parameters */
    if(dec_dwl == NULL) {
        assert(0);
        return DWL_HW_WAIT_ERROR;
    }

	while ((irq_count & DWL_PP_HW_READY) == 0) { ; } // Waiting for the interrupt
	irq_count &= ~DWL_PP_HW_READY; // Restore the ISR flag

    DWL_DEBUG("DWLWaitPpHwReady: OK\n\r");
    return DWL_HW_WAIT_OK;
#endif // 0
}

/* HW locking */

/*------------------------------------------------------------------------------
    Function name   : DWLReserveHw
    Description     :
    Return type     : i32
    Argument        : const void *instance
------------------------------------------------------------------------------*/
i32 DWLReserveHw(const void *instance)
{
    //i32 ret;
    //hX170dwl_t *dec_dwl = (hX170dwl_t *) instance;

    DWL_DEBUG("DWL: success\n\r");

	return DWL_OK;
}
