/*******************************************************************************
 (c) Copyright 2010, ACTi Corporation, Inc. ALL RIGHTS RESERVED

 All software are Copyright 2010 by ACTi Corporation. ALL RIGHTS RESERVED.
 Redistribution and use in source and binary forms, with or
 without modification, are strictly prohibited.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY EXPRESS
 OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

#include <common.h>
#include <command.h>

#include <codec/hx9170/include/mp4decapi.h>


static char* _progress = "|/-\\";

extern void timer2_start(unsigned int interval/* in ms */);
extern void timer2_stop(void);
extern unsigned int timer2_now(void);

extern void uapc2700_init(unsigned int panel);
extern void uapc2700_set_display(unsigned int width, unsigned height);
extern void uapc2700_set_fb(unsigned char* fb, unsigned int width, unsigned height);
extern void uapc2700_pause(unsigned int pause);


extern void  dma_memorypool_reset(void);
extern void* dma_memory_alloc(unsigned int size);


static void _print_decoder_info(MP4DecInfo* decInfo)
{
	printf("MPEG4 decoder information:\n");
	printf("    picWidth: %d\n", decInfo->frameWidth);
	printf("    picHeight: %d\n", decInfo->frameHeight);
	printf("    outputFormat: %d\n", decInfo->outputFormat);
}

int do_mpeg4dec_playback(cmd_tbl_t *cmdtp, int flag, int argc, char** argv)
{
	unsigned int current;
	unsigned int picDecodeNumber = 0; /* decoded picture ID */

	MP4DecRet ret;

	MP4DecInst    decoder;
	MP4DecInput   decIn;
	MP4DecInfo    decInfo;
	MP4DecOutput  decOut;
	MP4DecPicture decPic;

	unsigned int   fps;
	unsigned int   size;
	unsigned int   loop;
	unsigned char* input;
	unsigned int   panel;

	unsigned int width = 0;
	unsigned int height = 0;

	if (argc != 6) {
		printf("Usage:\n\t%s%s", cmdtp->usage, cmdtp->help);
		return 1;
	}

	panel = simple_strtoul(argv[1], NULL, 10);
	input = (unsigned char*)simple_strtoul(argv[2], NULL, 16);
	size  = simple_strtoul(argv[3], NULL, 10);
	fps   = simple_strtoul(argv[4], NULL, 10);
	loop  = simple_strtoul(argv[5], NULL, 10);
	printf("Decoding MPEG4 bitstream from: 0x%08x(%u bytes)\n", (unsigned int)input, size);

	memset(&decoder, 0, sizeof(decoder));
	memset(&decIn,   0, sizeof(decIn));
	memset(&decOut,  0, sizeof(decOut));
	memset(&decPic,  0, sizeof(decPic));
	memset(&decInfo, 0, sizeof(decInfo));

	timer2_start(1000 / fps);

	dma_memorypool_reset();

	uapc2700_init(panel);

	/* Configure the decoder input structure for the first decoding unit */
	decIn.dataLen = size;
	decIn.pStream = (void*)input;
	decIn.streamBusAddress = (unsigned int)input;

	/* Decoder initialization, output reordering disabled */
	ret = MP4DecInit(&decoder, 0/* sorensonSpark */, 0/* useVideoFreezeConcealment */);
	if(ret != MP4DEC_OK) {
		printf("Failed to initialize HX9170 MPEG4 decoder\n");
		goto error;
	}

	do {
		/* Picture ID is the picture number in decoding order */
		decIn.picId = picDecodeNumber;

		/* decode the stream data */
		ret = MP4DecDecode(decoder, &decIn, &decOut);
		switch(ret) {
		case MP4DEC_HDRS_RDY:
			/* read stream info */
			ret = MP4DecGetInfo(decoder, &decInfo);
			if (ret == MP4DEC_OK) {
				if (width == 0 && height == 0) {
					_print_decoder_info(&decInfo);

					width  = decInfo.frameWidth;
					height = decInfo.frameHeight;

					uapc2700_set_display(width, height);
				}
			} else {
				printf("Failed to get decoder information\n");
				goto failed;
			}
			break;
		case MP4DEC_DP_HDRS_RDY:
			/* Data partitioning used in the stream.
			 * The decoder has to reallocate resources */
			//printf("MP4DEC_DP_HDRS_RDY\n");
			break;
		case MP4DEC_PIC_DECODED:
			/* a picture was decoded */
			/* Increment decoding number after every decoded picture */
			while (MP4DecNextPicture(decoder, &decPic, 0) == MP4DEC_PIC_RDY) {
				printf("\rframe#%u decoded %c",
					   picDecodeNumber, _progress[picDecodeNumber & 0x3]);
				uapc2700_pause(1);
				uapc2700_set_fb((unsigned char*)decPic.pOutputPicture,
								width, height);
				uapc2700_pause(0);
			}

			/* Increment decoding number after every decoded picture */
			picDecodeNumber++;

			current = timer2_now();
			while (timer2_now() == current) ;

			ret = MP4DEC_OK;

			break;
		case MP4DEC_STRM_PROCESSED:
			/* Input stream was processed but no picture is ready */
			//printf("MP4DEC_STRM_PROCESSED\n");
			break;
		case MP4DEC_STRM_ERROR:
			printf("\nMP4DEC_STRM_ERROR\n");
			break;
		default:
			/* all other cases are errors where decoding cannot continue */
			printf("\nFailed to decode video stream: %d\n", ret);
			goto failed;
		}

		if (decOut.dataLeft == 0) {
			if (loop) {
				decIn.dataLen = size;
				decIn.pStream = (void*)input;
				decIn.streamBusAddress = (unsigned int)input;
			} else {
				decIn.dataLen = decOut.dataLeft;
			}
		} else {
			/* data left undecoded */
			decIn.dataLen = decOut.dataLeft;
			decIn.pStream = decOut.pStrmCurrPos;
			decIn.streamBusAddress = decOut.strmCurrBusAddress;
		}

		/* keep decoding until all data from input stream buffer consumed */
	} while (decIn.dataLen > 0);

failed:
	MP4DecRelease(decoder);

error:
	timer2_stop();

	return 0;
}


U_BOOT_CMD(
	mpeg4dec_playback,	CONFIG_SYS_MAXARGS,	1,	do_mpeg4dec_playback,
	" To playback MPEG4 bitstream\n",
	"[panel] [address] [stream size] [fps] [loop]\n"
);
