/*******************************************************************************
 (c) Copyright 2010, ACTi Corporation, Inc. ALL RIGHTS RESERVED

 All software are Copyright 2010 by ACTi Corporation. ALL RIGHTS RESERVED.
 Redistribution and use in source and binary forms, with or
 without modification, are strictly prohibited.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY EXPRESS
 OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

#include <common.h>
#include <command.h>

#include <codec/hx8290/include/jpegencapi.h>


extern void  dma_memorypool_reset(void);
extern void* dma_memory_alloc(unsigned int size);


static void _init_encoder_config(JpegEncCfg* pCfg,
								 unsigned int width, unsigned int height);
static int  _open_jpeg_encoder(JpegEncCfg* pCfg, JpegEncInst * pEnc);
static int  _allocate_resources(JpegEncInst* pEnc, JpegEncCfg* pCfg,
								JpegEncIn* pEncIn, JpegEncOut* pEncOut,
								unsigned char* rawdata);
static int  _start_jpeg_encode(JpegEncInst * pEnc,
								JpegEncIn* pEncIn, JpegEncOut* pEncOut);


int do_jpegenc(cmd_tbl_t *cmdtp, int flag, int argc, char** argv)
{
	int rc;
	JpegEncInst encoder;
	JpegEncCfg config;
	JpegEncIn encIn;
	JpegEncOut encOut;

	unsigned int width;
	unsigned int height;
	unsigned char* rawdata;

	if (argc != 4) {
		printf("Usage:\n\t%s%s", cmdtp->usage, cmdtp->help);
		return 1;
	}

	rawdata = (unsigned char*)simple_strtoul(argv[1], NULL, 16);
	width   = simple_strtoul(argv[2], NULL, 10);
	height  = simple_strtoul(argv[3], NULL, 10);
	printf("Encoding JPEG frame(%ux%u) from: 0x%08x\n",
		   width, height, (unsigned int)rawdata);

	memset(&encoder, 0, sizeof(encoder));
	memset(&config,  0, sizeof(config));
	memset(&encIn,   0, sizeof(encIn));
	memset(&encOut,  0, sizeof(encOut));

	dma_memorypool_reset();

	//
	// Test case: JPEG encoding
	//

	// Step 1: Initial encoder
	_init_encoder_config(&config, width, height);
	rc = _open_jpeg_encoder(&config, &encoder);
	if (rc != 0) {
		printf("Failed to open JPEG encoder\n");
		goto error;
	}

	// Step 2: Allocate resource
	rc = _allocate_resources(&encoder, &config, &encIn, &encOut, rawdata);
	if (rc != 0) {
		printf("Failed to open allocate resources for JPEG encoder\n");
		goto error;
	}

	// Step 3: Encode a frame
	rc = _start_jpeg_encode(&encoder, &encIn, &encOut);
	if (rc != 0) goto error;

	printf("JPEG encoded at 0x%08x(%u bytes)\n",
		   (unsigned int)encIn.pOutBuf, encOut.jfifSize);

	JpegEncRelease(encoder);

	return 0;

error:
	printf("Failed to encode a JPEG frame\n");

	return 0;
}


U_BOOT_CMD(
	jpegenc, CONFIG_SYS_MAXARGS, 0, do_jpegenc,
	"Encodes a JPEG frame\n",
	"[loadAddress] [width] [height]"
);


static void _init_encoder_config(JpegEncCfg* pCfg,
								 unsigned int width, unsigned int height)
{
	pCfg->qLevel = 8;
	pCfg->frameType = JPEGENC_YUV420_PLANAR;
	pCfg->markerType = JPEGENC_SINGLE_MARKER;
	pCfg->unitsType = JPEGENC_DOTS_PER_INCH;
	pCfg->xDensity = 72;
	pCfg->yDensity = 72;
	pCfg->inputWidth = width;
	pCfg->inputHeight = height;
	pCfg->codingWidth = width;
	pCfg->codingHeight = height;
	pCfg->xOffset = 0;
	pCfg->yOffset = 0;
	pCfg->rotation = JPEGENC_ROTATE_0;
	pCfg->codingType = JPEGENC_WHOLE_FRAME;
	pCfg->restartInterval = 0;
}

static int  _open_jpeg_encoder(JpegEncCfg* pCfg, JpegEncInst * pEnc)
{
	JpegEncRet ret;

	if ((ret = JpegEncInit(pCfg, pEnc)) != JPEGENC_OK) {
		return -1;
	}

	return 0;
}

static int  _allocate_resources(JpegEncInst* pEnc, JpegEncCfg* pCfg,
								JpegEncIn* pEncIn, JpegEncOut* pEncOut,
								unsigned char* rawdata)
{
	unsigned char* output;

	size_t outbufSize = pCfg->inputWidth * pCfg->inputHeight;
	unsigned int lumaSize = pCfg->inputWidth * pCfg->inputHeight;
	unsigned int chromaSize = pCfg->inputWidth * pCfg->inputHeight / 4;

	output = dma_memory_alloc(outbufSize);
	if (output == NULL) return -1;
	memset(output, 0, outbufSize);

	/* Setup encoder input */
	pEncIn->pOutBuf = (u8*)output;
	pEncIn->busOutBuf = (u32)output;
	pEncIn->outBufSize = outbufSize;
	pEncIn->frameHeader = 1;

	/* Set Full Resolution mode */
	if (JpegEncSetPictureSize(*pEnc, pCfg) != JPEGENC_OK) return 1;

	/* Bus addresses of input picture, used by hardware encoder */
	pEncIn->busLum = (u32)rawdata;
	pEncIn->busCb  = (u32)(rawdata + lumaSize);
	pEncIn->busCr  = (u32)(rawdata + lumaSize + chromaSize);

	/* Virtual addresses of input picture, used by software encoder */
	pEncIn->pLum   = rawdata;
	pEncIn->pCb    = rawdata + lumaSize;
	pEncIn->pCr    = rawdata + lumaSize + chromaSize;

	return 0;
}

static int  _start_jpeg_encode(JpegEncInst * pEnc,
								JpegEncIn* pEncIn, JpegEncOut* pEncOut)
{
	int ret;

	ret = JpegEncEncode(*pEnc, pEncIn, pEncOut);

	return (ret == JPEGENC_FRAME_READY) ? 0 : 1;
}
