/*******************************************************************************
 (c) Copyright 2010, ACTi Corporation, Inc. ALL RIGHTS RESERVED

 All software are Copyright 2010 by ACTi Corporation. ALL RIGHTS RESERVED.
 Redistribution and use in source and binary forms, with or
 without modification, are strictly prohibited.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY EXPRESS
 OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

#include <common.h>
#include <command.h>

#include <codec/hx9170/include/jpegdecapi.h>


static char* _progress = "|/-\\";


extern void timer2_start(unsigned int interval/* in ms */);
extern void timer2_stop(void);
extern unsigned int timer2_now(void);

extern void  dma_memorypool_reset(void);
extern void* dma_memory_alloc(unsigned int size);


int do_jpegdec_bm(cmd_tbl_t *cmdtp, int flag, int argc, char** argv)
{
	unsigned int count;

	unsigned int start;
	unsigned int stop;

	JpegDecRet rc = JPEGDEC_ERROR;

	JpegDecInst      decoder;
	JpegDecInput     decIn;
	JpegDecOutput    decOut;
	JpegDecImageInfo imageInfo;

	unsigned int width  = 0;
	unsigned int height = 0;

	unsigned int size;
	unsigned char* input;
	unsigned char* output;
	unsigned int   lastPic;

	if (argc != 4) {
		printf("Usage:\n\t%s%s", cmdtp->usage, cmdtp->help);
		return 1;
	}

	timer2_start(10);

	input = (unsigned char*)simple_strtoul(argv[1], NULL, 16);
	size  = simple_strtoul(argv[2], NULL, 10);
	lastPic = simple_strtoul(argv[3], NULL, 10);
	printf("Decoding JPEG frame from: 0x%08x(%u bytes)\n", (unsigned int)input, size);

	memset(&decoder,   0, sizeof(decoder));
	memset(&imageInfo, 0, sizeof(imageInfo));
	memset(&decIn,     0, sizeof(decIn));
	memset(&decOut,    0, sizeof(decOut));

	width = 0;
	height = 0;

	dma_memorypool_reset();

	/* Jpeg initialization */
	rc = JpegDecInit(&decoder);
	if (rc != JPEGDEC_OK) {
		printf("Failed to initialize JPEG decoder\n");
		goto error;
	}

	start = timer2_now();

	for (count = 0; count < lastPic; ++count) {
		/* Pointer to the input JPEG */
		decIn.streamBuffer.pVirtualAddress = (void*)input;
		decIn.streamBuffer.busAddress = (unsigned int)input;
		decIn.streamLength = size; /* JPEG length in bytes */
		decIn.bufferSize = 0; /* input buffering (0 == not used) */

		/* Get image information of the JPEG image */
		rc = JpegDecGetImageInfo(decoder, &decIn, &imageInfo);
		if (rc != JPEGDEC_OK) {
			printf("\nFailed to get information of JPEG frame\n");
			break;
			//goto failed;
		}

		if (width == 0 && height == 0) {
			width = imageInfo.displayWidth;
			height = imageInfo.displayHeight;

			output = dma_memory_alloc(width * height * 3 / 2);
			if (output == NULL) {
				rc = JPEGDEC_ERROR;
				printf("\nFailed to allocate memory block for output buffer\n");
				break;
				//goto failed;
			}
		}

		/* Luma Bus */
		decIn.pictureBufferY.pVirtualAddress = (void*)output;
		decIn.pictureBufferY.busAddress = (unsigned int)decIn.pictureBufferY.pVirtualAddress;

		/* Chroma Bus */
		decIn.pictureBufferCbCr.pVirtualAddress = (void*)(output + width * height);
		decIn.pictureBufferCbCr.busAddress = (unsigned int)decIn.pictureBufferCbCr.pVirtualAddress;

		/* Decode JPEG */
		rc = JpegDecDecode(decoder, &decIn, &decOut);
		if (rc == JPEGDEC_FRAME_READY) {
			printf("\rframe#%u decoded %c", count, _progress[count & 0x3]);
			//printf("JPEG frame is decoded to 0x%08x\n", (unsigned int)output);
			rc = JPEGDEC_OK;
		}
	}

	stop = timer2_now();

	printf("\nTotal %u frames decoded\n", count);
	printf("Start time: %u, End time: %u\n", start, stop);
	printf("Average frame rate: %u\n", count * 100 / (stop - start));

//failed:
	/* Release JPEG Decoder */
	JpegDecRelease(decoder);

error:
	timer2_stop();

	return 0;
}


U_BOOT_CMD(
	jpegdec_benchmark,	CONFIG_SYS_MAXARGS,	1,	do_jpegdec_bm,
	"To benchmark JPEG decoder",
	"[address] [size] [last pic]"
);
