/*******************************************************************************
 (c) Copyright 2010, ACTi Corporation, Inc. ALL RIGHTS RESERVED

 All software are Copyright 2010 by ACTi Corporation. ALL RIGHTS RESERVED.
 Redistribution and use in source and binary forms, with or
 without modification, are strictly prohibited.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY EXPRESS
 OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

#include <common.h>
#include <command.h>

#include <codec/hx8290/include/h264encapi.h>


enum _properties {
	FRAME_WIDTH  = 144,
	FRAME_HEIGHT = 96,
	OUTBUF_SIZE  = FRAME_WIDTH * FRAME_HEIGHT,

	LUMA_SIZE = FRAME_WIDTH * FRAME_HEIGHT,
	CHROMA_SIZE = FRAME_WIDTH * FRAME_HEIGHT / 4,
	PIC_SIZE = FRAME_WIDTH * FRAME_HEIGHT * 3 / 2,
};


extern void  dma_memorypool_reset(void);
extern void* dma_memory_alloc(unsigned int size);
extern void  dma_memory_free(void* ptr);
extern unsigned char colorbar_144x96[];


int do_h264enc_selftest(cmd_tbl_t *cmdtp, int flag, int argc, char** argv)
{
	H264EncRet ret;

	H264EncInst   encoder;
	H264EncConfig config;
	H264EncIn     encIn;
	H264EncOut    encOut;

	int next, last;

	unsigned char* outbuf;

	unsigned int   length = 0;
	unsigned char* bitstream;

	unsigned char* pattern;

	memset(&encoder, 0, sizeof(encoder));
	memset(&config,  0, sizeof(config));
	memset(&encIn,   0, sizeof(encIn));
	memset(&encOut,  0, sizeof(encOut));

	dma_memorypool_reset();

	/* Step 1: Initialize an encoder instance */
	/* 30 fps frame rate */
	config.frameRateDenom = 1;
	config.frameRateNum = 30; /* this is the time resolution also */

	/* VGA resolution */
	config.width  = FRAME_WIDTH;
	config.height = FRAME_HEIGHT;

	/* Stream type */
	config.streamType = H264ENC_BYTE_STREAM;
	config.level = H264ENC_LEVEL_3;

	if ((ret = H264EncInit(&config, &encoder)) != H264ENC_OK) {
		printf("Failed to open H.264 encoder\n");
		goto error;
	}

	/* Step 3: Allocate linear memory resources. */
	pattern = dma_memory_alloc(PIC_SIZE);
	if (pattern == NULL) {
		ret = H264ENC_ERROR;
		printf("Failed to allocate memory block for pattern\n");
		goto h264_error;
	}
	memcpy(pattern, colorbar_144x96, PIC_SIZE);

	bitstream = dma_memory_alloc(OUTBUF_SIZE);
	if (bitstream == NULL) {
		ret = H264ENC_ERROR;
		printf("Failed to allocate memory block for bitstream\n");
		goto h264_error;
	}

	outbuf = dma_memory_alloc(OUTBUF_SIZE);
	if (outbuf == NULL) {
		ret = H264ENC_ERROR;
		printf("Failed to allocate memory block for output buffer\n");
		goto h264_error;
	}

	encIn.pOutBuf = (void*)outbuf;
	encIn.busOutBuf = (unsigned int)outbuf;
	encIn.outBufSize = OUTBUF_SIZE; /* bytes */

	/* Step 4: Start the stream */
	ret = H264EncStrmStart(encoder, &encIn, &encOut);
	if (ret != H264ENC_OK) {
		printf("Failed to start H.264 encoder\n");
		goto h264_error;
	}

	/* Save the generated stream. */
	memcpy(bitstream + length, encIn.pOutBuf, encOut.streamSize);
	length += encOut.streamSize;

	/* Step 5: Encode the video frames */
	ret = H264ENC_FRAME_READY;
	next = 0; last = 1;

	while((next < last) &&
		  (ret == H264ENC_FRAME_READY ||
		   ret == H264ENC_OUTPUT_BUFFER_OVERFLOW))
	{
		if (next == 0) {
			encIn.timeIncrement = 0; /* start time is 0 */
		} else {
			encIn.timeIncrement = 1; /* time units between frames */
		}

		/* Select frame type */
		if (next == 0) {
			encIn.codingType = H264ENC_INTRA_FRAME; /* First frame must be intra */
		} else {
			encIn.codingType = H264ENC_PREDICTED_FRAME;
		}

		/* Read next frame and get the bus address. */
		//ReadYuv420PlanarFrame(pict_bus_address);

		/* we assume one linear buffer for the input image in */
		/* planar YCbCr 4:2:0 format */
		encIn.busLuma = (unsigned int)pattern;
		encIn.busChromaU = encIn.busLuma + config.width * config.height;
		encIn.busChromaV = encIn.busChromaU + (config.width / 2) * (config.height / 2);

		ret = H264EncStrmEncode(encoder, &encIn, &encOut);
		switch (ret) {
		case H264ENC_FRAME_READY:
			memcpy(bitstream + length, encIn.pOutBuf, encOut.streamSize);
			length += encOut.streamSize;
			break;
		default:
			printf("Failed to encode a frame into H.264\n");
			goto h264_error;
		}

		next++;
	} /* End of main encoding loop */

	/* Step 6: End stream */
	ret = H264EncStrmEnd(encoder, &encIn, &encOut);
	if (ret != H264ENC_OK) {
		printf("Failed to stop H.264 encoder\n");
		goto h264_error;
	} else {
		memcpy(bitstream + length, encIn.pOutBuf, encOut.streamSize);
		length += encOut.streamSize;
	}

	printf("H.264 I-frame encoded at 0x%08x(%u bytes)\n",
		   (unsigned int)bitstream, length);

	/* Free all resources */
	dma_memory_free(outbuf);
	dma_memory_free(bitstream);
	dma_memory_free(pattern);

h264_error:
	/* Last Step: Release the encoder instance */
	H264EncRelease(encoder);

error:
	if (ret != H264ENC_OK) {
		printf("Failed to encode a H.264 frame\n");
	}

	return 0;
}


U_BOOT_CMD(
	h264enc_selftest,	CONFIG_SYS_MAXARGS,	1,	do_h264enc_selftest,
	"Perform selftest of H.264 encoder",
	"\n"
	"	- To Encode a built-in color bar into an I frame."
);
