/*******************************************************************************
 (c) Copyright 2010, ACTi Corporation, Inc. ALL RIGHTS RESERVED

 All software are Copyright 2010 by ACTi Corporation. ALL RIGHTS RESERVED.
 Redistribution and use in source and binary forms, with or
 without modification, are strictly prohibited.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY EXPRESS
 OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

//#include <asm/arch/vic.h>

#include <common.h>
#include <command.h>

#include <codec/hx8290/include/h264encapi.h>
#include <codec/hx8290/include/h264encapi_ext.h>


extern void  dma_memorypool_reset(void);
extern void* dma_memory_alloc(unsigned int size);
extern void  dma_memory_free(void* ptr);

extern void timer2_start(unsigned int interval/* in ms */);
extern void timer2_stop(void);
extern unsigned int timer2_now(void);


static char* _progress = "|/-\\";


int do_h264enc_bm4(cmd_tbl_t *cmdtp, int flag, int argc, char** argv)
{
	H264EncRet ret;

	unsigned int start;
	unsigned int stop;

	H264EncInst       encoder;
	H264EncConfig     config;
	H264EncCodingCtrl codingCfg;
	H264EncIn         encIn;
	H264EncOut        encOut;

	unsigned int   next;
	unsigned int   offset;

	unsigned char* outbuf;
	unsigned int   bufferSize;

	unsigned int   length;
	unsigned char* bitstream;

	unsigned int   qmv;
	unsigned int   burn;
	unsigned int   width;
	unsigned int   height;
	unsigned int   lastPic;
	unsigned char* rawdata;
	unsigned int   frameSize;
	unsigned int   frameCount;

	if (argc < 6 || argc > 8) {
		printf("Usage:\n\t%s%s", cmdtp->usage, cmdtp->help);
		return 1;
	}

	timer2_start(100);

	rawdata     = (unsigned char*)simple_strtoul(argv[1], NULL, 16);
	width       = simple_strtoul(argv[2], NULL, 10);
	height      = simple_strtoul(argv[3], NULL, 10);
	frameCount  = simple_strtoul(argv[4], NULL, 10);
	lastPic     = simple_strtoul(argv[5], NULL, 10);
	printf("Encoding H.264 bitstream(%ux%u, %u frames) from 0x%08x(%u frames)\n",
		   width, height, lastPic, (unsigned int)rawdata, frameCount);

	if (argc >= 7) {
		burn = simple_strtoul(argv[6], NULL, 10);
	} else {
		burn = 0;
	}

	if (argc == 8) {
		qmv = simple_strtoul(argv[7], NULL, 10);
	} else {
		qmv = 0;
	}

	frameSize = width * height * 3 / 2;

	outbuf = NULL;
	length = 0;
	bitstream = NULL;
	memset(&encoder,   0, sizeof(encoder));
	memset(&config,    0, sizeof(config));
	memset(&codingCfg, 0, sizeof(codingCfg));
	memset(&encIn,     0, sizeof(encIn));
	memset(&encOut,    0, sizeof(encOut));

	dma_memorypool_reset();

	/* Step 1: Initialize an encoder instance */
	/* 30 fps frame rate */
	config.frameRateDenom = 1;
	config.frameRateNum = 30; /* this is the time resolution also */

	/* VGA resolution */
	config.width  = width;
	config.height = height;

	/* Stream type */
	config.streamType = H264ENC_BYTE_STREAM;
	config.level = H264ENC_LEVEL_5_1;

	if ((ret = H264EncInit(&config, &encoder)) != H264ENC_OK) {
		printf("Failed to open H.264 encoder\n");
		goto error;
	}

	if ((ret = H264EncGetCodingCtrl(encoder, &codingCfg)) != H264ENC_OK) {
		printf("Failed to get coding control of H.264 encoder\n");
		goto error;
	}

	// Set stream to High profile - adaptive 8x8 transform
	codingCfg.enableCabac = 1;
	codingCfg.transform8x8Mode = 1;

	if ((ret = H264EncSetCodingCtrl(encoder, &codingCfg)) != H264ENC_OK) {
		printf("Failed to set coding control of H.264 encoder\n");
		goto error;
	}

	H264EncSetQuarterPixelMv(encoder, qmv);

	/* Step 3: Allocate linear memory resources. */
	if (burn == 0) {
		bitstream = 0x5b000000;
		//bitstream = memory_alloc(frameSize * lastPic / 10);
		if (bitstream == NULL) {
			ret = H264ENC_ERROR;
			printf("Failed to allocate memory block for bitstream\n");
			goto h264_error;
		}
	}

	bufferSize = frameSize;
	outbuf = dma_memory_alloc(bufferSize);
	if (outbuf == NULL) {
		ret = H264ENC_ERROR;
		printf("Failed to allocate memory block for output buffer\n");
		goto h264_error;
	}

	encIn.pOutBuf = (void*)outbuf;
	encIn.busOutBuf = (unsigned int)outbuf;
	encIn.outBufSize = bufferSize; /* bytes */

	/* Step 4: Start the stream */
	ret = H264EncStrmStart(encoder, &encIn, &encOut);
	if (ret != H264ENC_OK) {
		printf("Failed to start H.264 encoder\n");
		goto h264_error;
	}

	/* Save the generated stream. */
	if (burn == 0) {
		memcpy(bitstream + length, encIn.pOutBuf, encOut.streamSize);
	}
	length += encOut.streamSize;

	/* Step 5: Encode the video frames */
	ret = H264ENC_FRAME_READY;

	start = timer2_now();

	next = 0;
	while((next < lastPic) &&
		  (ret == H264ENC_FRAME_READY ||
		   ret == H264ENC_OUTPUT_BUFFER_OVERFLOW))
	{
		if (next == 0) {
			encIn.timeIncrement = 0; /* start time is 0 */
		} else {
			encIn.timeIncrement = 1; /* time units between frames */
		}

		/* Select frame type */
		if ((next & 0x1f) == 0) {
		//if (next == 0) {
			encIn.codingType = H264ENC_INTRA_FRAME; /* First frame must be intra */
		} else {
			encIn.codingType = H264ENC_PREDICTED_FRAME;
		}

		/* Read next frame and get the bus address. */
		//ReadYuv420PlanarFrame(pict_bus_address);

		encIn.pOutBuf = (void*)outbuf;
		encIn.busOutBuf = (unsigned int)outbuf;
		encIn.outBufSize = bufferSize; /* bytes */

		/* we assume one linear buffer for the input image in */
		/* planar YCbCr 4:2:0 format */
		offset = next % frameCount;
		encIn.busLuma = (unsigned int)rawdata + offset * frameSize;
		encIn.busChromaU = encIn.busLuma + config.width * config.height;
		encIn.busChromaV = encIn.busChromaU + (config.width / 2) * (config.height / 2);

		ret = H264EncStrmEncode(encoder, &encIn, &encOut);
		switch (ret) {
		case H264ENC_FRAME_READY:
			if (burn == 0) {
				memcpy(bitstream + length, encIn.pOutBuf, encOut.streamSize);
			}
			length += encOut.streamSize;

			printf("%c\r", _progress[next & 0x3]);
			//printf("frame#%5u encoded\r", next);
			//printf("H.264 frame(%5u) encoded: %6u bytes\r",
			//	   next, encOut.streamSize);
			break;
		case H264ENC_OUTPUT_BUFFER_OVERFLOW:
			printf("Failed to encode H.264 frame(%5u): %6u/%6u bytes\n",
				   next, encOut.streamSize, bufferSize);
			next++;
			continue;
		default:
			printf("\nFailed to encode a frame into H.264\n");
			next++;
			goto h264_end;
		}

		next++;
	} /* End of main encoding loop */
	printf("\n");

h264_end:
	stop = timer2_now();

	/* Step 6: End stream */
	ret = H264EncStrmEnd(encoder, &encIn, &encOut);
	if (ret != H264ENC_OK) {
		printf("Failed to stop H.264 encoder\n");
		goto h264_error;
	} else {
		if (burn == 0) {
			memcpy(bitstream + length, encIn.pOutBuf, encOut.streamSize);
		}
		length += encOut.streamSize;
	}

	if (burn == 0) {
		printf("Total H.264 frames(%u) encoded at 0x%08x, %u bytes\n",
				next, (unsigned int)bitstream, length);
	} else {
		printf("Total H.264 frames(%u) encoded: %u bytes\n",
				next, length);
	}

	printf("Average frame rate: %u\n",
		   next * 10 / (stop - start));
//		   (float)(next * 10 / (stop - start)));

h264_error:
	/* Last Step: Release the encoder instance */
	H264EncRelease(encoder);

error:
	if (ret != H264ENC_OK) {
		printf("Failed to encode a H.264 frame\n");
	}

	timer2_stop();

	return 0;
}


U_BOOT_CMD(
	h264enc_benchmark4,	CONFIG_SYS_MAXARGS,	0,	do_h264enc_bm4,
	"Benchmark for H.264 encoder",
	"[loadAddress] [width] [height] [frameCount] [lastPic] [burn(optional)]"
);
