/*******************************************************************************
 (c) Copyright 2010, ACTi Corporation, Inc. ALL RIGHTS RESERVED

 All software are Copyright 2010 by ACTi Corporation. ALL RIGHTS RESERVED.
 Redistribution and use in source and binary forms, with or
 without modification, are strictly prohibited.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY EXPRESS
 OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

#include <asm/arch/vic.h>

#include <common.h>
#include <command.h>

#include <codec/hx9170/include/h264decapi.h>


#define TimerCnt2 2
static volatile unsigned int _tick = 0;
static char* _progress = "|/-\\";


static void tmr2_isr(void)
{
	vic_cleanirq(tmr2_intr);

	++_tick;
}


extern unsigned int GET_APBCLOCK();

extern void uapc2700_init(unsigned int panel);
extern void uapc2700_set_display(unsigned int width, unsigned height);
extern void uapc2700_set_fb(unsigned char* fb, unsigned int width, unsigned height);
extern void uapc2700_pause(unsigned int pause);


extern void  dma_memorypool_reset(void);
extern void* dma_memory_alloc(unsigned int size);


static void _print_decoder_info(H264DecInfo* decInfo)
{
	printf("H.264 decoder information:\n");
	printf("    picWidth: %d\n", decInfo->picWidth);
	printf("    picHeight: %d\n", decInfo->picHeight);
	printf("    outputFormat: 0x%08x\n", decInfo->outputFormat);
}

int do_h264dec_playback(cmd_tbl_t *cmdtp, int flag, int argc, char** argv)
{
	unsigned int current;
	unsigned int picDecodeNumber = 0; /* decoded picture ID */

	H264DecRet  ret;
	H264DecInst decoder;

	H264DecInput   decIn;
	H264DecOutput  decOut;
	H264DecPicture decPic;
	H264DecInfo    decInfo;

	unsigned int   fps;
	unsigned int   size;
	unsigned int   loop;
	unsigned char* input;
	unsigned int   panel;

	unsigned int width = 0;
	unsigned int height = 0;
	//unsigned char* output = NULL;

	if (argc != 6) {
		printf("Usage:\n\t%s%s", cmdtp->usage, cmdtp->help);
		return 1;
	}

	panel = simple_strtoul(argv[1], NULL, 10);
	input = (unsigned char*)simple_strtoul(argv[2], NULL, 16);
	size  = simple_strtoul(argv[3], NULL, 10);
	fps   = simple_strtoul(argv[4], NULL, 10);
	loop  = simple_strtoul(argv[5], NULL, 10);
	printf("Decoding H.264 bitstream from: 0x%08x(%u bytes) on panel(%u)\n",
		   (unsigned int)input, size, panel);

	memset(&decoder, 0, sizeof(decoder));
	memset(&decIn,   0, sizeof(decIn));
	memset(&decOut,  0, sizeof(decOut));
	memset(&decPic,  0, sizeof(decPic));
	memset(&decInfo, 0, sizeof(decInfo));

	/* disable timer (counter) interrupts */
	vic_disirq(tmr2_intr);

	/* register timer service routines */
	reg_interrupt(USE_IRQ, tmr2_intr, tmr2_isr);

	/* Setup .... */
	_tick = 0;
	EnableTimer(TimerCnt2, 0 , 0, GET_APBCLOCK() / fps);

	/* clear pending interrupts and start it */
	vic_cleanirq(tmr2_intr);
	vic_enirq(tmr2_intr);

	dma_memorypool_reset();

	uapc2700_init(panel);

	/* Configure the decoder input structure for the first decoding unit */
	decIn.dataLen = size;
	decIn.pStream = (void*)input;
	decIn.streamBusAddress = (unsigned int)input;

	/* Decoder initialization, output reordering disabled */
	ret = H264DecInit(&decoder, 1, 0);
	if (ret != H264DEC_OK) {
		printf("Failed to initialize HX9170 H.264 decoder\r\n");
		goto error;
	}

	do {
		/* Picture ID is the picture number in decoding order */
		decIn.picId = picDecodeNumber;

		/* decode the stream data */
		ret = H264DecDecode(decoder, &decIn, &decOut);
		switch (ret) {
		case H264DEC_HDRS_RDY:
			/* read stream info */
			ret = H264DecGetInfo(decoder, &decInfo);
			if (ret == H264DEC_OK) {
				//if (output != NULL) break;
				if (width == 0 && height == 0) {
					_print_decoder_info(&decInfo);

					width  = decInfo.picWidth;
					height = decInfo.picHeight;

					//output = memory_alloc(width * height * 3 / 2);
					//if (output == NULL) {
					//	ret = H264DEC_PARAM_ERROR;
					//	printf("Failed to allocate memory block for output buffer\n");
					//	goto failed;
					//}

					uapc2700_set_display(width, height);
				}
			} else {
				printf("Failed to get decoder information\n");
				goto failed;
			}
			break;
		case H264DEC_ADVANCED_TOOLS:
			/* Arbitrary Slice Order/multiple slice groups noticed */
			/* in the stream. The decoder has to reallocate resources */
			printf("H264DEC_ADVANCED_TOOLS\n");
			break;
		case H264DEC_PIC_DECODED: /* a picture was decoded */
			/* The decoder output is ready, now obtain pictures in */
			/* display order, if any available */
			while (H264DecNextPicture(decoder, &decPic, 0) == H264DEC_PIC_RDY) {
				printf("\rframe#%u decoded %c",
					   picDecodeNumber, _progress[picDecodeNumber & 0x3]);
				uapc2700_pause(1);
				//memcpy(output, decPic.pOutputPicture,
				//	   decPic.picWidth * decPic.picHeight * 3 / 2);
				uapc2700_set_fb((unsigned char*)decPic.pOutputPicture,
								width, height);
				uapc2700_pause(0);
			}

			/* Increment decoding number after every decoded picture */
			picDecodeNumber++;

			current = _tick;
			while (_tick == current) ;

			ret = H264DEC_OK;

			break;
		case H264DEC_STRM_PROCESSED:
			/* Input stream was processed but no picture is ready */
			//printf("H264DEC_STRM_PROCESSED\n");
			break;
		default:
			/* all other cases are errors where decoding cannot continue */
			printf("Failed to decode video stream: %d\n", ret);
			goto failed;
		}

		if (decOut.dataLeft == 0) {
			if (loop) {
				decIn.dataLen = size;
				decIn.pStream = (void*)input;
				decIn.streamBusAddress = (unsigned int)input;
			} else {
				decIn.dataLen = decOut.dataLeft;
			}
		} else {
			/* data left undecoded */
			decIn.dataLen = decOut.dataLeft;
			decIn.pStream = decOut.pStrmCurrPos;
			decIn.streamBusAddress = decOut.strmCurrBusAddress;
		}

		/* keep decoding until all data from input stream buffer consumed */
	} while (decIn.dataLen > 0);

	//printf("H.264 bitstream is decoded to 0x%08x(%u frames)\n",
	//	   (unsigned int)output, picDecodeNumber);

failed:
	H264DecRelease(decoder);

error:
	DisableTimer(TimerCnt2);
	unregister_interrupt(tmr2_intr);

	return 0;
}


U_BOOT_CMD(
	h264dec_playback,	CONFIG_SYS_MAXARGS,	1,	do_h264dec_playback,
	"To playback H.264 bitstream\n",
	"[panel] [address] [stream size] [fps] [loop]\n"
);
