/*******************************************************************************
 (c) Copyright 2010, ACTi Corporation, Inc. ALL RIGHTS RESERVED

 All software are Copyright 2010 by ACTi Corporation. ALL RIGHTS RESERVED.
 Redistribution and use in source and binary forms, with or
 without modification, are strictly prohibited.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY EXPRESS
 OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

#include <common.h>
#include <command.h>

#include <linux/list.h>

#include <lwip/netif.h>
#include <lwip/init.h>
//#include <lwip/dhcp.h>
#include <lwip/tcp.h>
#include <lwip/tcp_impl.h>
#include <lwip/ip_frag.h>
#include <netif/etharp.h>

#include <json.h>

#include <h264-coder.h>
#include <h264-coder-utils.h>


//extern void  dma_memorypool_reset(void);
extern void* dma_memory_alloc(unsigned int size);
extern void* dma_memory_calloc(unsigned int size, unsigned int number);
extern void dma_memory_free(void* ptr);
extern void* memory_set(void *p, int c, unsigned int n);

extern void timer2_start(unsigned int interval/* in ms */);
extern void timer2_stop(void);
extern unsigned int timer2_now(void);

extern err_t umvp2500_if_init(struct netif *netif);
extern void umvp2500_if_poll(struct netif *netif);


enum {
	TIMER_INTERVAL         = 10,
	ETHARP_TIMER_INTERVAL  = ARP_TMR_INTERVAL / TIMER_INTERVAL,
	TCP_TIMER_INTERVAL     = TCP_TMR_INTERVAL / TIMER_INTERVAL,
	IPREASS_TIMER_INTERVAL = IP_TMR_INTERVAL / TIMER_INTERVAL,
	FRAMESERVER_PORT       = 8080,
	FS_CLIENT_INITIALIZED  = 0,
	FS_CLIENT_CONFIGURED   = 1,
	FS_CLIENT_START        = 2,
	FS_CLIENT_ENCODE       = 3,
	FS_CLIENT_STOP         = 4,
};

struct frame_server
{
	struct netif   net_dev;
	struct ip_addr ip;
	struct ip_addr gw;
	struct ip_addr mask;

	unsigned int ts_etharp;
	unsigned int ts_tcp;
	unsigned int ts_ipreass;

	struct tcp_pcb *tcp;

	char* status;
};

enum
{
	REQUEST_TYPE_CONFIGURE = 0,
	REQUEST_TYPE_START     = 1,
	REQUEST_TYPE_ENCODE    = 2,
	REQUEST_TYPE_STOP      = 3,
};

struct client_request_header
{
	unsigned int     id;
	unsigned int     length;
};

struct client_request
{
	struct list_head list;
	unsigned int     received;
	unsigned int     remains;

	void* priv;
	int (*execute)(struct client_request* req);

	// Header of Packet buffer
	unsigned int     id;
	unsigned int     length;
	char             content[0];
};

enum
{
	RESPONSE_TYPE_OK     = 0,
	RESPONSE_TYPE_FAILED = 1,
};

struct client_response
{
	struct list_head list;
	unsigned int     deliveried;
	unsigned int     remains;

	// Header of Packet buffer
	unsigned int     id;
	unsigned int     length;
	char             content[0];
};

struct frame_client
{
	struct tcp_pcb *connection;

	struct json_object* properties;

	struct list_head requests;
	struct list_head responses;

	struct h264_coder_opts* options;
	struct h264_coder*      coder;

	unsigned int processedFrames;
	unsigned int codedFrames;
	unsigned int intraPeriod;
	unsigned int streamSize;
	unsigned int bitrate;
};


static char* _SERVER_STATUS[] =
{
	"\rWaiting for connection %c",
	"\rClient connected       %c",
	"\rProperties committed   %c",
	"\rH.264 encoder starts   %c",
//	"\rEncoding H.264 frame   %c",
};
static char*        _display = NULL;
static char*        _roller0 = "|/-\\";
static unsigned int _roller1 = 0;
static unsigned int _roller2 = 0;
#define DUPE_H264_STREAM
#ifdef DUPE_H264_STREAM
static unsigned char* _output = (unsigned char*)0x5b000000;
#endif /* DUPE_H264_STREAM */

static int _do_client_configure(struct client_request* req);
static int _do_client_start(struct client_request* req);
static int _do_client_encode(struct client_request* req);
static int _do_client_stop(struct client_request* req);


static inline void  ADD_REQUEST(struct frame_client* client,
								struct client_request* req)
{
	list_add_tail(&req->list, &client->requests);
}

static inline struct client_request* GET_FIRST_REQUEST(struct frame_client* client)
{
	if (list_empty(&client->requests)) return NULL;

	return list_first_entry(&client->requests, struct client_request, list);
}

#define list_last_entry(ptr, type, member) \
	list_entry((ptr)->prev, type, member)

static inline struct client_request* GET_LAST_REQUEST(struct frame_client* client)
{
	if (list_empty(&client->requests)) return NULL;

	return list_last_entry(&client->requests, struct client_request, list);
}

static inline unsigned char* REQUEST_PAYLOAD(struct client_request* req)
{
	return (unsigned char*)(&req->id);
}

static struct client_request* _client_request_new(struct frame_client* client,
												  unsigned int id,
												  unsigned int length)
{
	struct client_request* req = NULL;

	req = dma_memory_calloc(1, sizeof(*req) + length);
	if (req == NULL) return NULL;

	INIT_LIST_HEAD(&req->list);
	req->received = 0;
	req->remains  = sizeof(struct client_request_header) + length;

	req->id       = id;
	req->length   = length;

	req->priv = client;
	switch (id) {
	case REQUEST_TYPE_CONFIGURE:
		req->execute = _do_client_configure;
		break;
	case REQUEST_TYPE_START:
		req->execute = _do_client_start;
		break;
	case REQUEST_TYPE_ENCODE:
		req->execute = _do_client_encode;
		break;
	case REQUEST_TYPE_STOP:
		req->execute = _do_client_stop;
		break;
	}

	return req;
}

static void _client_request_free(struct client_request* req)
{
	if (req == NULL) return;

	dma_memory_free(req);
}

static inline void ADD_RESPONSE(struct frame_client* client,
								struct client_response* resp)
{
	list_add_tail(&resp->list, &client->responses);
}

static inline struct client_response* GET_RESPONSE(struct frame_client* client)
{
	if (list_empty(&client->responses)) return NULL;

	return list_first_entry(&client->responses, struct client_response, list);
}

static inline unsigned char* RESPONSE_PAYLOAD(struct client_response* resp)
{
	return (unsigned char*)(&resp->id);
}

static struct client_response* _client_response_new(unsigned int size)
{
	struct client_response* resp = NULL;

	resp = dma_memory_calloc(1, sizeof(*resp) + size);
	if (resp == NULL) return NULL;

	INIT_LIST_HEAD(&resp->list);
	resp->deliveried = 0;
	resp->remains    = size + 8;

	resp->id     = 0;
	resp->length = 0;

	return resp;
}

static void _client_response_free(struct client_response* resp)
{
	if (resp == NULL) return;

	dma_memory_free(resp);
}

static int _frame_server_initialize(struct frame_server* server)
{
	// Add our netif to LWIP (netif_add calls our driver initialization function)
	IP4_ADDR(&server->ip,   172, 16, 14, 87);
	IP4_ADDR(&server->gw,   172, 16, 14, 253);
	IP4_ADDR(&server->mask, 255, 255, 255, 0);
	if (netif_add(&server->net_dev,
		&server->ip, &server->mask, &server->gw, NULL,
		umvp2500_if_init, ethernet_input) == NULL)
	{
        printf("netif_add() failed\n");
        return -1;
    }

	netif_set_default(&server->net_dev);
	netif_set_up(&server->net_dev);

	server->ts_etharp = server->ts_tcp = server->ts_ipreass = timer2_now();

	server->tcp = tcp_new();
	if (server->tcp == NULL) return -1;

	tcp_arg(server->tcp, server);

	return 0;
}

static int _frame_server_finalize(struct frame_server* server)
{
	if (server->tcp != NULL) {
		tcp_close(server->tcp);
	}

	return 0;
}

static int _frame_client_initialize(struct frame_client* client,
									struct tcp_pcb *connection)
{
	if (client == NULL) return 0;

	client->connection = connection;
	client->properties = NULL;

	INIT_LIST_HEAD(&client->requests);
	INIT_LIST_HEAD(&client->responses);

	client->options                  = NULL;
	client->coder                    = NULL;

	client->processedFrames          = 0;
	client->codedFrames              = 0;
	client->intraPeriod              = 0;
	client->streamSize               = 0;
	client->bitrate                  = 0;

	return 0;
}

static int _frame_client_finalize(struct frame_client* client)
{
	struct client_request*  req;
	struct client_response* resp;

	if (client == NULL) return 0;

	printf("\n");

	if (client->coder != NULL) {
		//h264_coder_stop(client->coder);
		h264_coder_close(client->coder);
		dma_memory_free(client->coder);
	}

	if (client->options != NULL) {
		dma_memory_free(client->options);
	}

	if (client->properties != NULL) {
		json_object_put(client->properties);
	}

	while ((req = GET_FIRST_REQUEST(client)) != NULL) {
		list_del_init(&req->list);
		_client_request_free(req);
	}

	while ((resp = GET_RESPONSE(client)) != NULL) {
		list_del_init(&resp->list);
		_client_response_free(resp);
	}

	return 0;
}

static err_t _on_client_accepted(void *arg, struct tcp_pcb *newpcb, err_t err);

static int _frame_server_listen(struct frame_server* server)
{
	struct tcp_pcb *connection;

	if (tcp_bind(server->tcp, IP_ADDR_ANY, FRAMESERVER_PORT) != ERR_OK) {
        printf("Failed to bind to port: %d\n", FRAMESERVER_PORT);
        return -1;
	}

	connection = tcp_listen(server->tcp);
	if (connection == NULL) {
        printf("Failed to listen on port: %d\n", FRAMESERVER_PORT);
		return -1;
	}
	server->tcp = connection;

	tcp_accept(server->tcp, _on_client_accepted);

	return 0;
}

static int _frame_server_poll(struct frame_server* server)
{
	unsigned int now;

	// Call network interface to process incoming packets and do housekeeping
	umvp2500_if_poll(&server->net_dev);

	// Processes lwip network-related timers.
	now = timer2_now();
	if ((now - server->ts_etharp) >= ETHARP_TIMER_INTERVAL) {
		etharp_tmr();
		server->ts_etharp = now;
	}
	if ((now - server->ts_tcp) >= TCP_TIMER_INTERVAL) {
		tcp_tmr();
		server->ts_tcp = now;
	}
	if ((now - server->ts_ipreass) >= IPREASS_TIMER_INTERVAL) {
		ip_reass_tmr();
		server->ts_ipreass = now;
	}

	if ((now - _roller2) >= 5) {
		if (_display != NULL) {
			printf(_display, _roller0[++_roller1 & 0x3]);
		}
		_roller2 = now;
	}

	return 0;
}

static unsigned int _do_packet_send(struct frame_client* client,
									struct client_response* resp)
{
	unsigned int sending;
	unsigned int available;

	available = tcp_sndbuf(client->connection);
	while (available > 0 && resp != NULL) {
	//while (available > 0) {
		if (available > resp->remains) {
			sending = resp->remains;
		} else {
			sending = available;
		}

		tcp_write(client->connection,
				  RESPONSE_PAYLOAD(resp) + resp->deliveried,
				  sending,
				  TCP_WRITE_FLAG_COPY);

		resp->deliveried += sending;
		resp->remains    -= sending;

		available        -= sending;

		if (resp->remains == 0) {
			list_del_init(&resp->list);
			_client_response_free(resp);
			resp = GET_RESPONSE(client);
			printf("Response queue[head/first]: 0x%08x/0x%08x\n",
				   (unsigned int)&client->responses,
				   (unsigned int)resp);
			if (resp != NULL) {
				printf("Starts sending response(0x%08x/%u)\n",
					   (unsigned int)RESPONSE_PAYLOAD(resp),
					   resp->length + 8);
			} else {
				printf("All responses sent\n");
			}
		}
	}

	tcp_output(client->connection);

	return 0;
}

static void _append_response(struct frame_client* client,
							 struct client_response* resp)
{
	struct client_response* head;

	printf("\nAppends a response to queue: 0x%08x\n", (unsigned int)resp);

	ADD_RESPONSE(client, resp);

	printf("Response@0x%08x = (ID: %u, length: %u)\n",
		   (unsigned int)resp, resp->id, resp->length);

	head = GET_RESPONSE(client);
	printf("Response queue[head/first/new]: 0x%08x/0x%08x/0x%08x\n",
		   (unsigned int)&client->responses,
		   (unsigned int)head, (unsigned int)resp);

	if (head == resp) {
		printf("Starts sending response(0x%08x/%u)\n",
			   (unsigned int)RESPONSE_PAYLOAD(head), head->length + 8);
		//printf("\nSending response(0x%08x)\n", (unsigned int)resp);
		_do_packet_send(client, head);
	}
}

static err_t _on_packet_sent(void *arg, struct tcp_pcb *tpcb, u16_t len)
{
	struct client_response* resp;
	struct frame_client* client = (struct frame_client*)arg;

	printf("\n_on_packet_sent: %u bytes sent\n", len);

	resp = GET_RESPONSE(client);
	if (resp != NULL) {
		_do_packet_send(client, resp);
	}

	return ERR_OK;
}

static err_t _on_packet_received(void *arg, struct tcp_pcb *tpcb,
								 struct pbuf *p, err_t err)
{
	unsigned int size;
	struct client_request* request;
	struct frame_client* client = (struct frame_client*)arg;

	if (p == NULL) {
		//printf("Client disconnected\n");
		_frame_client_finalize(client);
		dma_memory_free(client);

		//printf("\n");
		//_display = _SERVER_STATUS[0];

		return ERR_OK;
	}

	request = GET_LAST_REQUEST(client);
	if ((request == NULL) || (request->remains == 0)) {
		struct client_request_header header;

		if (p->len < sizeof(header)) {
			// Not enough data
			// TODO: Should response error
		}

		memcpy(&header, p->payload, sizeof(header));
		request = _client_request_new(client, header.id, header.length);
		if (request == NULL) goto failback;
		ADD_REQUEST(client, request);
	}

	size = p->len;
	if (size > request->remains) {
		printf("\nPacket buffer overflow(%u/%u). %d @ %s\n",
			   size, request->remains, __LINE__, __FILE__);
		// Shrink it
		size = request->remains;
	}

	memory_copy(REQUEST_PAYLOAD(request) + request->received,
				p->payload, size);
	request->received += size;
	request->remains  -= size;

	//printf("\nclient->inPacketBytesRemaining = %d\n", client->inPacketBytesRemaining);
	printf("\rReceiving(0x%08x): %u/%u",
		   (unsigned int)REQUEST_PAYLOAD(request),
		   request->received, request->length + 8);

	if (request->remains == 0) {
		request->execute(request);
	}

failback:
	tcp_recved(tpcb, p->len);

	return ERR_OK;
}

static err_t _on_client_accepted(void *arg, struct tcp_pcb* incomming, err_t err)
{
	struct frame_client* client = NULL;
	struct frame_server* server = (struct frame_server*)arg;

	//printf("\nClient accepted\n");

	client = dma_memory_calloc(sizeof(*client), 1);
	if (client == NULL) {
		tcp_abort(incomming);
		return ERR_ABRT;
	}

	_frame_client_initialize(client, incomming);

	tcp_arg(incomming, client);
	tcp_sent(incomming, _on_packet_sent);
	tcp_recv(incomming, _on_packet_received);

	tcp_accepted(server->tcp);

	//printf("\n");
	//_display = _SERVER_STATUS[1];

	return ERR_OK;
}

static void _read_int_property(struct json_object* obj, char* field, int* val)
{
	struct json_object* property;

	property = json_object_object_get(obj, field);
	if (property != NULL &&
		json_object_get_type(property) == json_type_int)
	{
		*val = json_object_get_int(property);
		//printf("Property: %s = %d\n", field, *val);
	} else {
		*val = -999;
	}
}

static void _read_str_property(struct json_object* obj, char* field, char** val)
{
	struct json_object* property;

	property = json_object_object_get(obj, field);
	if (property != NULL &&
		json_object_get_type(property) == json_type_string)
	{
		*val = (char*)json_object_get_string(property);
	} else {
		*val = "invalid";
	}
}

static int _print_properties(struct json_object* properties)
{
	int value;
	char* string;
	struct json_object* property;

	property = properties;
/*
	property = json_object_object_get(properties, "encoders");
	if (property == NULL ||
		json_object_get_type(property) != json_type_array)
	{
		printf("Type of 'encoders' is not array\n");
		return 0;
	}
	printf("  Count of encoders: %d\n", json_object_array_length(property));

	property = json_object_array_get_idx(property, 0);
*/

	#define READ_STR_PROPERTY(prop) \
	_read_str_property(property, prop, &string); \
	printf("    %s: %s\n", prop, string);

	#define READ_INT_PROPERTY(prop) \
	_read_int_property(property, prop, &value); \
	printf("    %s: %d\n", prop, value);

	READ_STR_PROPERTY("input");
	READ_STR_PROPERTY("output");
	READ_INT_PROPERTY("lastPic");
	READ_INT_PROPERTY("inputWidth");
	READ_INT_PROPERTY("inputHeight");
	READ_INT_PROPERTY("outputWidth");
	READ_INT_PROPERTY("outputHeight");
	READ_INT_PROPERTY("xOffset");
	READ_INT_PROPERTY("yOffset");
	READ_INT_PROPERTY("inputRateNumer");
	READ_INT_PROPERTY("inputRateDenom");
	READ_INT_PROPERTY("level");
	READ_INT_PROPERTY("hrdConformance");
	READ_INT_PROPERTY("intraPicRate");
	READ_INT_PROPERTY("qpHdr");
	READ_INT_PROPERTY("qpMin");
	READ_INT_PROPERTY("qpMax");
	READ_INT_PROPERTY("bitPerSecond");
	READ_INT_PROPERTY("picRc");
	READ_INT_PROPERTY("mbRc");
	READ_INT_PROPERTY("picSkip");
	READ_INT_PROPERTY("rotation");
	READ_INT_PROPERTY("inputFormat");
	READ_INT_PROPERTY("chromaQpOffset");
	READ_INT_PROPERTY("trans8x8");
	READ_INT_PROPERTY("enableCabac");
	READ_INT_PROPERTY("quarterPixelMv");
	READ_INT_PROPERTY("byteStream");
	READ_INT_PROPERTY("intraQpDelta");
	READ_INT_PROPERTY("fixedIntraQp");
	READ_INT_PROPERTY("mbQpAdjustment");
	READ_INT_PROPERTY("mvOutput");
}

static int _do_client_configure(struct client_request* request)
{
	struct frame_client*    client;
	struct client_response* response;

	client = (struct frame_client*)request->priv;

	client->properties = json_tokener_parse(request->content);
	if (client->properties != NULL) {
		client->options = dma_memory_calloc(1, sizeof(*client->options));
		if (client->options != NULL) {
			h264_coder_opts_create_from_json(client->properties,
											 client->options);
			h264_coder_opts_print(client->options);

			_print_properties(client->properties);

			response = _client_response_new(3);
			if (response == NULL) goto failback;

			response->id = RESPONSE_TYPE_OK;
			response->length = 3;
			response->content[0] = 'O';
			response->content[1] = 'K';
			response->content[2] = '\0';

			_append_response(client, response);

			//_display = _SERVER_STATUS[2];
		} else {
			printf("\nFailed to allocate struct h264_coder_opts. %u@%s\n",
				   __LINE__, __FILE__);
		}
	} else {
		printf("\nFailed to create configuration file for H.264 encoder\n");
	}

failback:
	return 0;
}

static int _do_client_start(struct client_request* request)
{
	struct frame_client*    client;
	struct client_response* response;

	client = (struct frame_client*)request->priv;

	client->coder = dma_memory_calloc(1, sizeof(*client->coder));
	if (client->coder == NULL) {
		printf("\nFailed to create an instance of H.264 encoder\n");
		goto failback;
	}

	if (h264_coder_open(client->coder, client->options) != 0) {
		printf("\nFailed to open H.264 encoder\n");
		goto failback;
	}

	if (h264_coder_start(client->coder) != 0) {
		printf("\nFailed to start H.264 stream\n");
		goto failback;
	}
#ifdef DUPE_H264_STREAM
	memcpy(_output + client->streamSize,
		   client->coder->outputMem.virtualAddress,
		   client->coder->encOutput.streamSize);
#endif /* DUPE_H264_STREAM */

	/* First frame is always intra with time increment = 0 */
	client->coder->encInput.codingType = H264ENC_INTRA_FRAME;
	client->coder->encInput.timeIncrement = 0;
	client->intraPeriod = client->coder->options->intraPicRate;
	client->streamSize = client->coder->encOutput.streamSize;

	H264EncGetRateCtrl(client->coder->encoder, &client->coder->rateCtrl);

	/* Allocate a buffer for user data and read data from file */
	//pUserData = ReadUserData(encoder, cml->userData);

	printf("\n Pic | QP | Target | Type |   BR avg    MA(%3d) | ByteCnt (inst) |",
		   client->coder->ma.length);
//	if (cml->psnr) {
//		printf(" PSNR  | NALU sizes\n");
//	} else {
		printf(" NALU sizes\n");
//	}

	printf("--------------------------------------------------------------------------------\n");
	printf("     | %2d |   HDR  |                            | %7i %6i | ",
			client->coder->rateCtrl.qpHdr, client->streamSize, client->coder->encOutput.streamSize);
//	if (cml->psnr)
//		printf("      | ");
//	PrintNalSizes(encIn.pNaluSizeBuf, (u8 *) outbufMem.virtualAddress,
//				  encOut.streamSize, cml->byteStream);
	printf("\n");

	response = _client_response_new(client->coder->encOutput.streamSize);
	if (response == NULL) goto failback;

	response->id = RESPONSE_TYPE_OK;
	response->length = client->coder->encOutput.streamSize;
	memcpy(response->content,
		   client->coder->outputMem.virtualAddress,
		   client->coder->encOutput.streamSize);

	_append_response(client, response);

	//_display = _SERVER_STATUS[3];
	//_display = NULL;

failback:
	return 0;
}

static const char* _frame_to_string(H264EncPictureCodingType type)
{
	char* str;

	switch (type) {
	case H264ENC_INTRA_FRAME:
		str = "I";
		break;
	case H264ENC_PREDICTED_FRAME:
		str = "P";
		break;
	case H264ENC_NOTCODED_FRAME:
		str = "'skip'";
		break;
	default:
		str = "'Unknown'";
		break;
	}

	return str;
}

static int _do_client_encode(struct client_request* request)
{
	struct frame_client*    client;
	struct client_response* response;

	client = (struct frame_client*)request->priv;

	H264EncRet rc;
	int targetFrameType;
	int actualFrameType;

	#define coder client->coder
	//struct h264_coder* coder;

	if (_display != NULL) { printf("\n"); _display = NULL; }

/*
	if (coder->options->videoStab > 0) {
		// Swap picture buffer and stabilization picture buffer.
		// Thus, the latest frame is going to be stabilized.
		h264_coder_swap_linear_mem(&coder->pictureMem,
								   &coder->pictureStabMem);
		coder->encInput.busLumaStab = coder->pictureStabMem.busAddress;
	}
*/

	/* Select frame type */
	if ((coder->options->intraPicRate != 0) &&
		(client->intraPeriod >= coder->options->intraPicRate)) {
		client->intraPeriod = 0;
	}
	targetFrameType = (client->intraPeriod == 0) ?
						H264ENC_INTRA_FRAME : H264ENC_PREDICTED_FRAME;

	/* Setup encoder input */
	{
		unsigned int w = (coder->options->inputWidth + 15) & (~0x0f);

		coder->encInput.busLuma = (unsigned int)request->content;
		//coder->encInput.busLuma = coder->pictureMem.busAddress;
		coder->encInput.busChromaU = coder->encInput.busLuma +
								  (w * coder->options->inputHeight);
		coder->encInput.busChromaV = coder->encInput.busChromaU +
								  (((w + 1) >> 1) *
								  ((coder->options->inputHeight + 1) >> 1));
	}

	// Encode a frame
	rc = h264_coder_compress(coder, targetFrameType, &actualFrameType);
#ifdef DUPE_H264_STREAM
		memcpy(_output + client->streamSize,
			   coder->outputMem.virtualAddress,
			   coder->encOutput.streamSize);
#endif /* DUPE_H264_STREAM */

	client->streamSize += coder->encOutput.streamSize;

	/* Note: This will overflow if large output rate numerator is used */
	client->bitrate = 8 * ((client->streamSize / (client->processedFrames + 1)) *
					  (u32)coder->options->inputRateNumer /
					  (u32)coder->options->inputRateDenom);

	switch (rc) {
	case H264ENC_FRAME_READY:
		printf(" %3i | %2i | %s | %s | %9u %9u | %7i %6i |\n",
				client->processedFrames, coder->rateCtrl.qpHdr,
				coder->encInput.codingType == H264ENC_INTRA_FRAME ? "  I   " :
				coder->encInput.codingType == H264ENC_PREDICTED_FRAME ? "  P   " : " skip ",
				coder->encOutput.codingType == H264ENC_INTRA_FRAME ? " I  " :
				coder->encOutput.codingType == H264ENC_PREDICTED_FRAME ? " P  " : "skip",
				client->bitrate, h264_ma_calculate(&coder->ma),
				client->streamSize, coder->encOutput.streamSize);
		break;

	case H264ENC_OUTPUT_BUFFER_OVERFLOW:
            printf(" %3i | %2i | %s | %s | %9u %9u | %7i %6i | \n",
                client->processedFrames, coder->rateCtrl.qpHdr,
                coder->encInput.codingType == H264ENC_INTRA_FRAME ? "  I   " :
                coder->encInput.codingType == H264ENC_PREDICTED_FRAME ? "  P   " : " skip ",
				"lost",
				client->bitrate, h264_ma_calculate(&coder->ma),
				client->streamSize, coder->encOutput.streamSize);
            break;

	default:
		//PrintErrorValue("H264EncStrmEncode() failed.", ret);
		printf("Failed to encode %s frame. ", _frame_to_string(targetFrameType));
		if (targetFrameType != actualFrameType) {
			printf("targetFrameType(%s) != actualFrameType(%s)\n",
					_frame_to_string(targetFrameType),
					_frame_to_string(actualFrameType));
		}

		/* For debugging, can be removed */
		//WriteStrm(fout, outbufMem.virtualAddress, encOut.streamSize, 0);

		/* We try to continue encoding the next frame */
		break;
	}

	client->processedFrames++;

	if (actualFrameType != H264ENC_NOTCODED_FRAME) {
		client->intraPeriod++; client->codedFrames++;
	}

	response = _client_response_new(coder->encOutput.streamSize);
	if (response == NULL) goto failback;

	response->id = RESPONSE_TYPE_OK;
	response->length = coder->encOutput.streamSize;
	memcpy(response->content,
		   coder->outputMem.virtualAddress,
		   coder->encOutput.streamSize);

	_append_response(client, response);

	return 0;

failback:
	return 0;
}

static int _do_client_stop(struct client_request* request)
{
	struct frame_client*    client;
	struct client_response* response;

	client = (struct frame_client*)request->priv;

	//H264EncRet rc;

	#define coder client->coder
	//struct h264_coder* coder;

	/* End stream */
	if (h264_coder_stop(coder) == 0) {
#ifdef DUPE_H264_STREAM
		memcpy(_output + client->streamSize,
			   coder->outputMem.virtualAddress,
			   coder->encOutput.streamSize);
#endif /* DUPE_H264_STREAM */

        client->streamSize += coder->encOutput.streamSize;
        printf("     |    |   END  |                            | %7i %6i | ",
                client->streamSize, coder->encOutput.streamSize);
//        if (cml->psnr)
//            printf("      | ");
//        PrintNalSizes(encIn.pNaluSizeBuf, (u8 *) outbufMem.virtualAddress,
//                encOut.streamSize, cml->byteStream);
        printf("\n");

//		fwrite(coder->outputMem.virtualAddress, 1,
//			   coder->encOutput.streamSize, fout);

//        if(cml->byteStream == 0)
//        {
//            WriteNalSizesToFile(nal_sizes_file, encIn.pNaluSizeBuf,
//                                encIn.naluSizeBufSize);
//        }

		printf("\nBitrate target %d bps, actual %d bps (%d%%).\n",
			   coder->rateCtrl.bitPerSecond, client->bitrate,
			   (coder->rateCtrl.bitPerSecond) ?
					client->bitrate * 100 / coder->rateCtrl.bitPerSecond : 0);
		printf("Total of %d frames processed, %d frames encoded, %d bytes.\n",
			   client->processedFrames,
			   client->codedFrames,
			   client->streamSize);

		response = _client_response_new(coder->encOutput.streamSize);
		if (response == NULL) goto failback;

		response->id = RESPONSE_TYPE_OK;
		response->length = coder->encOutput.streamSize;
		memcpy(response->content,
			   coder->outputMem.virtualAddress,
			   coder->encOutput.streamSize);

		//printf("Sending Response@0x%08x = (ID: %u, length: %u)\n",
		//	   (unsigned int)response, response->id, response->length);

		_append_response(client, response);
    }

	return 0;

failback:
	return 0;
}

int do_frameserver(cmd_tbl_t *cmdtp, int flag, int argc, char** argv)
{
	struct frame_server *server;

	timer2_start(TIMER_INTERVAL);

	// Initialize LWIP
	lwip_init();

	server = dma_memory_calloc(sizeof(*server), 1);
	if (server == NULL) {
		printf("Failed to allocate struct frame_server. %d@%s\n",
			   __LINE__, __FILE__);
		return 0;
	}

	if (_frame_server_initialize(server) != 0) {
		printf("Failed to initialize frame server. %d@%s\n",
			   __LINE__, __FILE__);
		goto error;
	}

	if (_frame_server_listen(server) != 0) {
		goto error;
	}

	printf("Frame server listening on port: %u\n", FRAMESERVER_PORT);

	//_display = _SERVER_STATUS[0];
	while (_frame_server_poll(server) == 0) {
		//if (_frame_server_poll(server) != 0) break;
	}

error:
	_frame_server_finalize(server);
	dma_memory_free(server);

	timer2_stop();

	return 0;
}


U_BOOT_CMD(
	frameserver, CONFIG_SYS_MAXARGS, 0, do_frameserver,
	"",
	""
);
