#include <stdio.h>

#include "hal.h"

volatile int x = 0;

void halSysInit_i(void)
{
	/* TODO */
}

void halSysSelDisplay(halDisplay_E nDisplay)
{
}

unsigned int halSysGetHWVersion(void)
{
	register unsigned int r;

	r = HAL_GETREG32(HAL_REG_SYS_VerNum);
	return (r);
}

unsigned int halSysGetDIPSwitchValue(void)
{
	register unsigned int r;

	r = HAL_GETREG32(HAL_REG_SYS_JumpSts);
	return (r);
}

void halSysShowSysInfo(void)
{
//	unsigned int r;

	return;
}

void halSysPLLIsMaster(int master_flag)
{
	unsigned int reg;

	reg = HAL_GETREG32(HAL_REG_SYS_OSCCtrl);
	if(master_flag)
		reg |= 0x1 << 16;
	else
		reg &= ~(0x1 << 16);
	HAL_SETREG32(HAL_REG_SYS_OSCCtrl, reg);
}

void halSysInternalISPEnable(int enable)
{
	unsigned int reg;

	reg = HAL_GETREG32(HAL_REG_SYS_OSCCtrl);
	if(enable)
		reg |= 0x1 << 24;
	else
		reg &= ~(0x1 << 24);
	HAL_SETREG32(HAL_REG_SYS_OSCCtrl, reg);
}

int halSysGetJumperStatus(int Device)
{
	sys_jumper_t reg;

	*(unsigned int *)&reg  = HAL_GETREG32(HAL_REG_SYS_JumpSts);

	switch(Device)
	{
		case HAL_MFP_NAND:
			return (reg.nand_boot == 0 ? 1: 0);
			break;
		case HAL_MFP_SMC:
			return (reg.nand_boot == 1 ? 1: 0);
			break;
	}
	return 0;
}


void halSysSetMultiFunctionalPin(int Device)
{
	unsigned int mfs0, mfs1, mfs2, mfs3, mfs4, mfs5;

	mfs0 = HAL_GETREG32(HAL_REG_SYS_MSF0);
	mfs1 = HAL_GETREG32(HAL_REG_SYS_MSF1);
	mfs2 = HAL_GETREG32(HAL_REG_SYS_MSF2);
	mfs3 = HAL_GETREG32(HAL_REG_SYS_MSF3);
	mfs4 = HAL_GETREG32(HAL_REG_SYS_MSF4);
	mfs5 = HAL_GETREG32(HAL_REG_SYS_MSF5);

	switch(Device)
	{
		case HAL_MFP_I2C0:
			mfs4 = (mfs4 & 0xFFFFCFFF) | 0x00000000;
			break;
		case HAL_MFP_I2C1:
			mfs4 = (mfs4 & 0xFFFF3FFF) | 0x00000000;
			mfs5 = (mfs5 & 0xFFFFFCFF) | 0x00000000;
			break;
		case HAL_MFP_I2C2:
			mfs4 = (mfs4 & 0xF3FFFFFF) | 0x00000000;
			mfs5 = (mfs5 & 0xFFFFF3FF) | 0x00000000;
			break;
		case HAL_MFP_SSI:	// normal A1 SSI pin mux
			mfs4 = (mfs4 & ~0x01E00000) | 0x00000000;
			break;
		case HAL_MFP_SSI_ISP:	// used for ISP purpose
			mfs4 = (mfs4 & ~0x01E00000) | 0x00200000;	// set SSI CS pin to GPIO, requested by ISP team
			break;
		case HAL_MFP_NAND:
			mfs1 = (mfs1 & 0x000000FF) | 0x00000000;
			mfs2 = 0x00000000;
			mfs3 = (mfs3 & 0xFFFFF000) | 0x00000000;
			mfs4 = (mfs4 & 0xFDFFFFFF) | 0x00000000;
			break;
		case HAL_MFP_SDC:
			mfs3 = (mfs3 & 0xFFE00FFF) | 0x00000000;
			break;
		case HAL_MFP_SMC:
			mfs1 = (mfs1 & 0x000000FF) | 0x55555500;
			mfs2 = 0x55555555;
			mfs3 = (mfs3 & 0xFFFFF000) | 0x00000555;
			mfs4 |= 0x02000000;
			break;
		case HAL_MFP_LCD:
			mfs0 = (mfs0 & ~0x1FFFFFFF) | 0x0AAABFFF;
			mfs1 = (mfs1 & ~0x000000FF) | 0x000000FF;
			mfs5 = (mfs5 & ~0x000000FF) | 0x00000000;
			break;
		case HAL_MFP_WDT:
			mfs4 &= 0xEFFFFFFF;
			break;
		case HAL_MFP_SENSOR:
			mfs5 |= 0x80000000;
			break;
		case HAL_MFP_ISP:
			halSysInternalISPEnable(1);
			// CMOS interface
			mfs0 = (mfs0 & ~0x001FFFE7) | 0x00000000;
			mfs0 = (mfs0 & ~0x1FE00000) | 0x15400000;
			mfs1 = (mfs1 & ~0x000000FF) | 0x00000000;
			// I2C
			mfs4 = (mfs4 & ~0x0C00C000) | 0x00000000;
			mfs5 = (mfs5 & ~0x00000300) | 0x00000300;
			//SPI
			mfs4 = (mfs4 & ~0xE0000000) | 0x00000000;
			mfs5 = (mfs5 & ~0x00000400) | 0x00000400;
			//GPIO
			mfs5 = (mfs5 & ~0x7FFFF8FF) | 0x7FFFF8FF;
			mfs1 = (mfs1 & ~0x000000FF) | 0x00000000;
			mfs3 = (mfs3 & ~0x00000550) | 0x00000550;
			break;
		case HAL_MFP_ISP_DEBUG:
			halSysInternalISPEnable(1);
			// CMOS interface
			mfs0 = (mfs0 & ~0x001FFFE7) | 0x00000000;
			mfs0 = (mfs0 & ~0x1FE00000) | 0x15400000;
			mfs1 = (mfs1 & ~0x000000FF) | 0x00000000;
			// I2C
			mfs4 = (mfs4 & ~0x0C00C000) | 0x00000000;
			mfs5 = (mfs5 & ~0x00000300) | 0x00000300;
			//SPI
			mfs4 = (mfs4 & ~0xE0000000) | 0x00000000;
			mfs5 = (mfs5 & ~0x00000400) | 0x00000400;
			//GPIO
			mfs5 = (mfs5 & ~0x7FFFF8FF) | 0x7FFFF8FF;
			mfs1 = (mfs1 & ~0x000000FF) | 0x00000000;
			mfs3 = (mfs3 & ~0x00000550) | 0x00000550;
			// for YUV output, for debug
			mfs1 = (mfs1 & ~0x00FC0000) | 0x00FC0000;
			mfs2 = (mfs2 & ~0xFFFFFFF0) | 0xFFFFFFF0;
			mfs3 = (mfs3 & ~0x0000000F) | 0x0000000F;
			break;
		case HAL_MFP_EXTERNAL_ISP:
			halSysInternalISPEnable(0);
			mfs0 = (mfs0 & ~0x01FFFFE7) | 0x00000000;
			mfs1 = (mfs1 & ~0x000000FF) | 0x00000000;
			mfs5 = (mfs5 & ~0x000000FF) | 0x00000000;
			break;
		case HAL_MFP_GPIO_AP:
			//GPIO0:0,1,2,3,12,13,14,15
			mfs4 = (mfs4 & ~0x000001E0) | 0x000001E0;	// GPIO0:12,13,14,15
			mfs5 = (mfs5 & ~0x00078000) | 0x00000000; 	// GPIO0:0,1,2,3
			//GPIO2: 30, 31
			mfs0 = (mfs0 & ~0x00000018) | 0x00000000;
			//GPIO3: 25,26,27,28,30
			mfs3 = (mfs3 & ~0x80000000) | 0x80000000;	// GPIO3:25
			mfs4 = (mfs4 & ~0x00000017) | 0x00000017;	// GPIO3:26,27,28,30
			break;
		case HAL_MFP_GPIO_AP_POWER:
			mfs4 = (mfs4 & ~0x00000004) | 0x00000004;	// GPIO3:28, for Power LED only
			break;
		default:
			break;
	}
	HAL_SETREG32(HAL_REG_SYS_MSF0, mfs0);
	HAL_SETREG32(HAL_REG_SYS_MSF1, mfs1);
	HAL_SETREG32(HAL_REG_SYS_MSF2, mfs2);
	HAL_SETREG32(HAL_REG_SYS_MSF3, mfs3);
	HAL_SETREG32(HAL_REG_SYS_MSF4, mfs4);
	HAL_SETREG32(HAL_REG_SYS_MSF5, mfs5);
}

char *ClkGateInfo [64] =
{
		"apb_hclk",
		"ahbc_hclk",
		"",
		"nandc_hclk",
		"nandc_clkb",
		"isp_mclk",
		"vcap_mclk",
		"usb_uclk",
		"vic_hclk",
		"",
		"emac_hclk",
		"dmac_hclk",
		"sdc_hclk",
		"lmc_hclk",
		"lcd_hclk",
		"usb_hclk",
		"smc_hclk",
		"axibrg0_hclk",
		"axibrg1_hclk",
		"h8290_aclk",
		"h9170_aclk",
		"ddr2x_clk",
		"lcdc_clk",
		"cmos_pclk",
		"axibrg_cfg",
		"sdc_clk",
		"",
		"vcap0_aclk",
		"vcap1_aclk",
		"vcap2_aclk",
		"vcap3_aclk",
		"sensor_clk",

		"wdt_pclk",
		"uart0_pclk",
		"uart1_pclk",
		"uart2_pclk",
		"uart3_pclk",
		"uart4_pclk",
		"tmr_pclk",
		"i2c0_pclk",
		"i2c1_pclk",
		"i2c2_pclk",
		"gpio0_pclk",
		"gpio1_pclk",
		"gpio2_pclk",
		"gpio3_pclk",
		"gpio4_pclk",
		"i2s_pclk",
		"i2s_ext_bypass",
		"ssi_pclk",
		"h8290_pclk",
		"h9170_pclk",
		"uart0_clk",
		"uart1_clk",
		"uart2_clk",
		"uart3_clk",
		"uart4_clk",
		"i2s_clk",
		"tmr_eclk",
		"ssi_clk",
		"pll0out_mux",
		"pll1out_mux",
		"apbbrg_pclk",
		"isp_pclk",
};


void halSysGetDeviceClkGateStr(char *str)
{
	unsigned int i, gate;

	gate = HAL_GETREG32(HAL_REG_SYS_AHBGateCtrl);
	for(i = 0; i < 32; i++)
	{
		if((gate & 0x01) == 0)
		{
			if(ClkGateInfo[i] != "")
			{
				strcat(str, ClkGateInfo[i]);
				strcat(str, " ");
			}
		}
		gate >>= 1;
	}
	gate = HAL_GETREG32(HAL_REG_SYS_APBGateCtrl);
	for(i = 0; i < 32; i++)
	{
		if((gate & 0x01) == 0)
		{
			if(ClkGateInfo[i] != "")
			{
				strcat(str, ClkGateInfo[i+32]);
				strcat(str, " ");
			}
		}
		gate >>= 1;
	}
}

void halSysSetDeviceClkGate(int DeviceID, int enable)
{
	unsigned int gate;

	if(DeviceID < 32)
	{
		gate = HAL_GETREG32(HAL_REG_SYS_AHBGateCtrl);
		if(enable)
			gate |= (0x1 << DeviceID);
		else
			gate &= ~(0x1 << DeviceID);
		HAL_SETREG32(HAL_REG_SYS_AHBGateCtrl, gate);

	}
	else
	{
		gate = HAL_GETREG32(HAL_REG_SYS_APBGateCtrl);
		DeviceID -= 32;
		if(enable)
			gate |= (0x1 << DeviceID);
		else
			gate &= ~(0x1 << DeviceID);
		HAL_SETREG32(HAL_REG_SYS_APBGateCtrl, gate);
	}
}

/* clock source table */
#define hal_wdt_clk_src			(PLL_GetAHBClock(Fin)/2) //PLL_GetFout(1,Fin ) if using external clk
#define hal_tmr_clk_src			(PLL_GetAHBClock(Fin)/2) //PLL_GetFout(1, Fin) if using external clk
#define hal_ssi_clk_src			(PLL_GetAHBClock(Fin)/2)
#define hal_i2s_clk_src			PLL_GetFout(1, Fin)
#define hal_lcd_clk_src			PLL_GetFout(1, Fin)
#define hal_h8290_clk_src			PLL_GetFout(1, Fin)
#define hal_h9170_clk_src			PLL_GetFout(1, Fin)
#define hal_vcap_clk_src			PLL_GetAHBClock(Fin)
#define hal_isp_clk_src			PLL_GetFout(1, Fin)
#define hal_sensor_clk_src		PLL_GetFout(1, Fin)
#define hal_sensor_clk_div_src	halSysGetDeviceClockSrc(Fin, HAL_MFP_ISP, NULL)
#define hal_nand_clk_src			PLL_GetAHBClock(Fin)
#define hal_usb_clk_src			(480*1000*1000)
#define hal_sdc_clk_src			PLL_GetAHBClock(Fin)
#define hal_uart_clk_src			(PLL_GetAHBClock(Fin)/2)
#define hal_smc_clk_src			PLL_GetAHBClock(Fin)
#define hal_i2c_clk_src			(PLL_GetAHBClock(Fin)/2)

unsigned int halSysGetDeviceClockSrc(unsigned int Fin, int DeviceID, unsigned int *divider)
{
	volatile unsigned int reg;
	volatile sys_div_cnt0_t *cnt0_reg = (sys_div_cnt0_t *)&reg;
	volatile sys_div_cnt1_t *cnt1_reg = (sys_div_cnt1_t *)&reg;
	volatile sys_div_cnt2_t *cnt2_reg = (sys_div_cnt2_t *)&reg;
	volatile sys_div_cnt3_t *cnt3_reg = (sys_div_cnt3_t *)&reg;
	volatile sys_div_cnt4_t *cnt4_reg = (sys_div_cnt4_t *)&reg;
	volatile sys_div_cnt5_t *cnt5_reg = (sys_div_cnt5_t *)&reg;

	if(divider != NULL) *divider = 0; // default set to 0

	switch(DeviceID)
	{
		case HAL_MFP_WDT:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt5);
//			if(divider != NULL) *divider = cnt5_reg->wdt_cntx+1;
//			return hal_wdt_clk_src/(cnt5_reg->wdt_cntx+1);	// if using external clk
			return hal_wdt_clk_src;
		case HAL_MFP_TMR:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt5);
//			if(divider != NULL) *divider = cnt5_reg->tmr_cntx+1;
//			return hal_tmr_clk_src/(cnt5_reg->tmr_cntx+1); // if using external clk
			return hal_tmr_clk_src;
		case HAL_MFP_SSI:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt5);
			if(divider != NULL) *divider = cnt5_reg->ssi_cntx+1;
			return hal_ssi_clk_src/(cnt5_reg->ssi_cntx+1);
		case HAL_MFP_I2S:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt5);
			if(divider != NULL) *divider = cnt5_reg->i2s_cntx+1;
			return hal_i2s_clk_src/(cnt5_reg->i2s_cntx+1);
		case HAL_MFP_UART0:case HAL_MFP_UART1:case HAL_MFP_UART2:case HAL_MFP_UART3:case HAL_MFP_UART4:
			return hal_uart_clk_src;
		case HAL_MFP_SENSOR:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt3);
			if(divider != NULL) *divider = cnt3_reg->sensor_cntx+1;
			return hal_sensor_clk_src/(cnt3_reg->sensor_cntx+1);
		case HAL_MFP_SENSOR_DIV:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt3);
			if(divider != NULL) *divider = cnt3_reg->sensor_cntx_div+1;
			return hal_sensor_clk_div_src/(cnt3_reg->sensor_cntx_div+1);
		case HAL_MFP_LCD:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt3);
			if(divider != NULL) *divider = cnt3_reg->lcd_cntx+1;
			return hal_lcd_clk_src/(cnt3_reg->lcd_cntx+1);
		case HAL_MFP_H8290:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt2);
			if(divider != NULL) *divider = cnt2_reg->h8290_cntx+1;
			return hal_h8290_clk_src/(cnt2_reg->h8290_cntx+1);
		case HAL_MFP_H9170:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt2);
			if(divider != NULL) *divider = cnt2_reg->h9170_cntx+1;
			return hal_h9170_clk_src/(cnt2_reg->h9170_cntx+1);
		case HAL_MFP_ISP:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt1);
			if(divider != NULL) *divider = cnt1_reg->isp_cntx+1;
			return hal_isp_clk_src/(cnt1_reg->isp_cntx+1);
		case HAL_MFP_VCAP:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt1);
			if(divider != NULL) *divider = cnt1_reg->vcap_cntx+1;
			return hal_vcap_clk_src/(cnt1_reg->vcap_cntx+1);
		case HAL_MFP_SMC:
			return hal_smc_clk_src;
		case HAL_MFP_I2C0:case HAL_MFP_I2C1:case HAL_MFP_I2C2:
			return hal_i2c_clk_src;
		case HAL_MFP_SDC:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt0);
			if(divider != NULL) *divider = cnt0_reg->sdc_cntx+1;
			return hal_sdc_clk_src/(cnt0_reg->sdc_cntx+1);
		case HAL_MFP_USB:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt0);
			if(divider != NULL) *divider = cnt0_reg->usb_cntx+1;
			return hal_usb_clk_src/(cnt0_reg->usb_cntx+1);
		case HAL_MFP_NAND:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt0);
			if(divider != NULL) *divider = cnt0_reg->nand_cntx+1;
			return hal_nand_clk_src/(cnt0_reg->nand_cntx+1);
	}
	return 0;
}


unsigned int halSysSetDeviceClockDivider(unsigned int Fin, int DeviceID, unsigned int DividerValue)
{
	unsigned int ahb_gate, apb_gate;
	volatile unsigned int reg;
	volatile sys_div_cnt0_t *cnt0_reg = (sys_div_cnt0_t *)&reg;
	volatile sys_div_cnt1_t *cnt1_reg = (sys_div_cnt1_t *)&reg;
	volatile sys_div_cnt2_t *cnt2_reg = (sys_div_cnt2_t *)&reg;
	volatile sys_div_cnt3_t *cnt3_reg = (sys_div_cnt3_t *)&reg;
	volatile sys_div_cnt4_t *cnt4_reg = (sys_div_cnt4_t *)&reg;
	volatile sys_div_cnt5_t *cnt5_reg = (sys_div_cnt5_t *)&reg;

	if(DividerValue <= 0)
		return halSysGetDeviceClockSrc(Fin, DeviceID, NULL);

	ahb_gate = HAL_GETREG32(HAL_REG_SYS_AHBGateCtrl);
	apb_gate = HAL_GETREG32(HAL_REG_SYS_APBGateCtrl);
	HAL_SETREG32(HAL_REG_SYS_AHBGateCtrl, 0x87F7EFBF);
	HAL_SETREG32(HAL_REG_SYS_APBGateCtrl, 0xFFF3FFFF);

	switch(DeviceID)
	{
		case HAL_MFP_WDT:
			reg= HAL_GETREG32(HAL_REG_SYS_CLKDivCnt5);
			cnt5_reg->wdt_cntx = DividerValue - 1;
			HAL_SETREG32(HAL_REG_SYS_CLKDivCnt5, reg);
			break;
		case HAL_MFP_TMR:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt5);
			cnt5_reg->tmr_cntx = DividerValue - 1;
			HAL_SETREG32(HAL_REG_SYS_CLKDivCnt5, reg);
			break;
		case HAL_MFP_SSI:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt5);
			cnt5_reg->ssi_cntx = DividerValue - 1;
			HAL_SETREG32(HAL_REG_SYS_CLKDivCnt5, reg);
			break;
		case HAL_MFP_I2S:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt5);
			cnt5_reg->i2s_cntx = DividerValue - 1;
//			halUARTPrintf("set I2S divider 0x%08x 0x%08x 0x%08x \n\r", reg, cnt5_reg->i2s_cntx, DividerValue);
			HAL_SETREG32(HAL_REG_SYS_CLKDivCnt5, reg);
			break;
		case HAL_MFP_UART0:case HAL_MFP_UART1:case HAL_MFP_UART2:case HAL_MFP_UART3:case HAL_MFP_UART4:
			break;
		case HAL_MFP_LCD:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt3);
			cnt3_reg->lcd_cntx = DividerValue - 1;
			HAL_SETREG32(HAL_REG_SYS_CLKDivCnt3, reg);
			break;
		case HAL_MFP_SENSOR:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt3);
			cnt3_reg->sensor_cntx = DividerValue - 1;
//			halUARTPrintf("set sensor divider 0x%08x\n\r", *(unsigned int *)&cnt3_reg);
			HAL_SETREG32(HAL_REG_SYS_CLKDivCnt3, reg);
			break;
		case HAL_MFP_SENSOR_DIV:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt3);
			cnt3_reg->sensor_cntx_div = DividerValue - 1;
			HAL_SETREG32(HAL_REG_SYS_CLKDivCnt3, reg);
			break;
		case HAL_MFP_H8290:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt2);
			cnt2_reg->h8290_cntx = DividerValue - 1;
			HAL_SETREG32(HAL_REG_SYS_CLKDivCnt2, reg);
			break;
		case HAL_MFP_H9170:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt2);
			cnt2_reg->h9170_cntx = DividerValue - 1;
			HAL_SETREG32(HAL_REG_SYS_CLKDivCnt2, reg);
			break;
		case HAL_MFP_ISP:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt1);
			cnt1_reg->isp_cntx = DividerValue - 1;
			HAL_SETREG32(HAL_REG_SYS_CLKDivCnt1, reg);
			break;
		case HAL_MFP_VCAP:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt1);
			cnt1_reg->vcap_cntx = DividerValue - 1;
			HAL_SETREG32(HAL_REG_SYS_CLKDivCnt1, reg);
			break;
		case HAL_MFP_SMC:
			break;
		case HAL_MFP_I2C0:case HAL_MFP_I2C1:case HAL_MFP_I2C2:
			break;
		case HAL_MFP_SDC:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt0);
			cnt0_reg->sdc_cntx = DividerValue - 1;
			HAL_SETREG32(HAL_REG_SYS_CLKDivCnt0, reg);
			break;
		case HAL_MFP_USB:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt0);
			cnt0_reg->usb_cntx = DividerValue - 1;
			HAL_SETREG32(HAL_REG_SYS_CLKDivCnt0, reg);
			break;
		case HAL_MFP_NAND:
			reg = HAL_GETREG32(HAL_REG_SYS_CLKDivCnt0);
			cnt0_reg->nand_cntx = DividerValue - 1;
			HAL_SETREG32(HAL_REG_SYS_CLKDivCnt0, reg);
			break;
	}
	HAL_SETREG32(HAL_REG_SYS_AHBGateCtrl, ahb_gate);
	HAL_SETREG32(HAL_REG_SYS_APBGateCtrl, apb_gate);
	return halSysGetDeviceClockSrc(Fin, DeviceID, NULL);
}


////////////////////// POM Control Functions /////////////////////////////////////

extern int FCSIntrOccur_IRQ;
extern int FCSIntrOccur_FIQ;

int POM_SWReset()
{
	HAL_SETREG32(HAL_REG_POM_CTRL, POM_CTRL_SW_RESET);
}

int PLL_GetFCS(int pll_no, unsigned int *NR, unsigned int *NF, unsigned int *NO)
{
	pll_ctrl_t reg;

	if(pll_no == 0)
		*(unsigned int *)&reg = HAL_GETREG32(HAL_REG_PLL0_CTRL);
	else
		*(unsigned int *)&reg = HAL_GETREG32(HAL_REG_PLL1_CTRL);

	*NR = reg.nr;
	*NF = reg.nf;
	*NO = reg.op_div;

	return *(unsigned int *)&reg;
}

int PLL_SetFCS(int pll_no, unsigned int Fin, unsigned int NR, unsigned int NF, unsigned int NO)
{
	unsigned int pll_reg, BS, fvco;

	// set PLL0 Freq
	fvco = Fin * (NF+1) / (NR+1);
	if(fvco >= 500)
		BS = 2;
	else if(fvco >= 300)
		BS = 1;
	else if(fvco >= 100)
		BS = 0;
	else
		BS = 2;

	pll_reg = (NR << PLL_NR_SHIFT) | (NF << PLL_NF_SHIFT) | (NO << PLL_NO_SHIFT) | (BS << PLL_BS_SHIFT) | PLL_CTRL_UPDATE;
	if(pll_no == 0)
		HAL_SETREG32(HAL_REG_PLL0_CTRL, pll_reg);
	else
		HAL_SETREG32(HAL_REG_PLL1_CTRL, pll_reg);

	return pll_reg;
}

unsigned int PLL_GetFout(int pll_no, unsigned int Fin)
{
	pll_ctrl_t reg;
	unsigned int Fout;

	if(pll_no == 0)
		*(unsigned int *)&reg = HAL_GETREG32(HAL_REG_PLL0_CTRL);
	else
		*(unsigned int *)&reg = HAL_GETREG32(HAL_REG_PLL1_CTRL);

	Fout = Fin / (reg.nr+1) * (reg.nf+1) / (1 << reg.op_div);

	return Fout;
}

unsigned int PLL_GetAHBClock(unsigned int Fin)
{
	pll_ctrl_t reg;
	unsigned int Fout;


	*(unsigned int *)&reg = HAL_GETREG32(HAL_REG_PLL0_CTRL);

	Fout = Fin / (reg.nr+1) * (reg.nf+1) / (1 << reg.op_div);

	return Fout/2;
}

unsigned int PLL_GetAPBClock(unsigned int Fin)
{
	return (PLL_GetAHBClock(Fin) / 2);
}

int CPU_GetTurbo(void)
{
	return HAL_GETREG32(HAL_REG_SYS_CPUMode) & CPU_TURBO_BIT;
}

unsigned int Mem_PerformanceTest(int count)
{
#define MEM_SIZE	0x8000
	unsigned int Temp[MEM_SIZE], result = 0;
	unsigned int i;

	for (i = 0; i < MEM_SIZE ; i++)
		Temp[i] = i;

	while(count-- > 0)
	{
		for (i = 0; i < MEM_SIZE ; i++)
			result += Temp[i];
	}

	return result;
}

unsigned int CPU_PerformaceTest(int count)
{
	register i, x, y, z;

	x = y = z = 0x11223344;

	for(i = 0; i < count; i++)
	{
		x = (x+y) * (y+z) * (z+x);
		y = (x+y) * (y+z) * (z+x);
		z = (x+y) * (y+z) * (z+x);
	}
	return x;
}

void BackupDDR(unsigned int *SrcAddr, unsigned int *DestAddr, unsigned int size)
{
	size /= 4;

	for(; size > 0; size--, DestAddr++, SrcAddr++)
		*DestAddr = *SrcAddr;
	return;
}


////////////////////// POM Control Functions /////////////////////////////////////
////////////////////// POM Control Functions /////////////////////////////////////
////////////////////// POM Control Functions /////////////////////////////////////
#define DO_DELAY_P(y)		{delayloop = y;while(delayloop-- > 0); }

#define DDR_TEMP_BASE					0x20000000
#define DDR_PARAMETER_ADDR_BASE			(DDR_TEMP_BASE + (8 * 4))
#define DDR_PARAMETER_VALUE_BASE		(DDR_PARAMETER_ADDR_BASE + (16 * 4))
#define FCS_CODE_BASE					0x20000200

#define ASM_CONSTANT_DEFINE()				__asm__ __volatile__ (	\
												".equ		FCS_CODE_BASE, 				0x20000200\n\t" \
											);

int Set_WDT(unsigned int Fin, int seconds)
{
	if(seconds > 0)
	{
		halSysSetMultiFunctionalPin(HAL_MFP_WDT);
		halWDTReStartNew(PLL_GetAPBClock(Fin) * seconds, halcWDTClk_pclk, FALSE, FALSE, TRUE);
	}
	else
	{
		halWDTDisable();
	}
}

int PLL1_FCS(unsigned int Fin, unsigned int NR, unsigned int NF, unsigned int NO)
{
	unsigned int pom_reg, pom_status, sys_reg, ddr2_reg, ddrphyctrl_reg,  BS, fvco, vic_enable_reg, i;
	unsigned int *SrcAddr, *DestAddr, size, mSpeed, *ddr_p_addr, *ddr_p_value;
	register ddr_p_count,  delayloop, back_addr, cp15_reg, cp15_flag, pll_reg;
	DDRSetting_T DDRSettingTable;

	mSpeed = (Fin/1000/1000) * (NR+1) * (NF+1) / (NO+1);
	// calculate the new DDR2 parameter
	DDRSettingTable.count = 0;
	ddr_p_count = halGetDDRSetting(mSpeed, &DDRSettingTable);

#ifndef SDB_UBOOT //added by Ted for UBoot
	halUARTPrintf("set mspeed = %d \n\r", mSpeed);
	for(i=0; i < DDRSettingTable.count; i++)
		halUARTPrintf("0x%08x = 0x%08x \n\r", DDRSettingTable.addr[i], DDRSettingTable.value[i]);
#endif

	ASM_CONSTANT_DEFINE();

	//back-up all VIC enable settings
	vic_enable_reg = HAL_GETREG(HAL_REG_VIC1_IntEnable);
	//disable all VIC sources
	HAL_SETREG(HAL_REG_VIC1_IntEnableClear, 0xffffffff);

	//enable LMC for buffer of data & code
	halSysSetDeviceClkGate(halcGateLMC, 1);

	// backup DDR2 first 8 words, these will be updated by DDR2 data training
	SrcAddr = (unsigned int *)HAL_OMB_DDR2_BASE; DestAddr = (unsigned int *)DDR_TEMP_BASE;
	for(size = 0x8; size > 0; size--, DestAddr++, SrcAddr++)
		*DestAddr = *SrcAddr;

	// move ddr parameters into internal sram
	DestAddr = (unsigned int *)DDR_PARAMETER_ADDR_BASE;
	for(i=0; i < DDRSettingTable.count; DestAddr++, i++)
		*DestAddr = DDRSettingTable.addr[i];
	DestAddr = (unsigned int *)DDR_PARAMETER_VALUE_BASE;
	for(i=0; i < DDRSettingTable.count; DestAddr++, i++)
		*DestAddr = DDRSettingTable.value[i];


	fvco = Fin /1000/1000 * (NF+1) / (NR+1);
	if(fvco >= 500)
		BS = 2;
	else if(fvco >= 300)
		BS = 1;
	else if(fvco >= 100)
		BS = 0;
	else
		BS = 2;

	pll_reg = (NR << PLL_NR_SHIFT) | (NF << PLL_NF_SHIFT) | (NO << PLL_NO_SHIFT) | (BS << PLL_BS_SHIFT);

	Set_WDT(Fin, 5);

	__asm__ __volatile__ (
			"stmdb	sp!, {r0-r10}					\n\t" \
			"ldr		r0, =pll_fcs_start				\n\t" \
			"ldr		r1, =pll_fcs_end				\n\r" \
			"ldr		r2, =FCS_CODE_BASE					\n\t" \
			"move_loop:									\n\t" \
			"cmp		r0, r1							\n\t" \
			"ldmltia	r0!, {r3-r10}					\n\t" \
			"stmltia	r2!, {r3-r10}					\n\t" \
			"blt		move_loop							\n\t" \
			"tci_loop1:									\n\t" \
			"mrc		p15, 0, r15, c7, c14, 3		\n\t" \
			"bne		tci_loop1							\n\t" \
			"mov		r0, #0x00000005					@I/D Cache enable bit & MMU \n\t" \
			"orr		r0, r0, #0x00001000				\n\t" \
			"mvn		r0, r0							\n\t" \
			"mrc		p15, 0, r1, c1, c0, 0			@read CR \n\t" \
			"and		r1, r1, r0						@modify content \n\t" \
			"mcr		p15, 0, r1, c1, c0, 0			@write CR \n\t" \
			"ldmia	sp!, {r0-r10}					\n\r" \
			"ldr		%0, =pll_fcs_back				\n\r" \
			"ldr		pc, =FCS_CODE_BASE				\n\r" \
			"pll_fcs_start:								\n\r" \
			:"=r"(back_addr):
	);

	// program FCS Start to 1
	pom_reg = HAL_GETREG32(HAL_REG_POM_CTRL);
	HAL_SETREG32(HAL_REG_POM_CTRL, pom_reg | POM_CTRL_FCS_START);

#if 0
	// disable sdram ODT
	HAL_SETREG32(0x90d001f4, 0x0);
	// disable DDRC read/write ODT
	HAL_SETREG32(0x99000080, 0x000003ff);
	HAL_SETREG32(0x90d00098, 0x0);
#endif

	// issue enter SRF cmd
	ddr2_reg = HAL_GETREG32(HAL_REG_DDR2C_DRAMConfig);
	ddr2_reg &= ~DDR2_CMD_BITS;
	ddr2_reg |= DDR2_IN_SRF_CMD | DDR2_EXECUTE_CMD_BIT;

	// in order to make sure DDR can enter self-refresh successfully.
	for(i=0;i<0x10;i++)
		HAL_SETREG32(HAL_REG_DDR2C_DRAMConfig, ddr2_reg);

	// wait for SRF mode
	do{
		pom_reg = HAL_GETREG32(HAL_REG_POM_CTRL);
	}while((pom_reg & POM_CTRL_IN_SRF_MODE) == 0);

	// program FCS start to 0
	HAL_SETREG32(HAL_REG_POM_CTRL, pom_reg & (~POM_CTRL_FCS_START));

	// gate DDR2x clk
//	sys_reg = HAL_GETREG32(HAL_REG_SYS_APBGateCtrl);
//	sys_reg &= ~0x20000000;
//	HAL_SETREG32(HAL_REG_SYS_APBGateCtrl, sys_reg);
//	sys_reg = HAL_GETREG32(HAL_REG_SYS_AHBGateCtrl);
//	sys_reg &= ~0x79380000;
//	HAL_SETREG32(HAL_REG_SYS_AHBGateCtrl, sys_reg);

	// reset ddr phy, added by Ted
	ddrphyctrl_reg = HAL_GETREG32(HAL_REG_SYS_DDR2PhyCtrl);
	ddrphyctrl_reg &= 0xFFFF0000;
	HAL_SETREG32(HAL_REG_SYS_DDR2PhyCtrl, ddrphyctrl_reg);

	// set PLL1 Freq, due to HW design bug, software should poll whether PLL enter PLL update state
	while(1)
	{
		HAL_SETREG32(HAL_REG_PLL1_CTRL, pll_reg);
		HAL_SETREG32(HAL_REG_PLL1_CTRL, pll_reg | PLL_CTRL_UPDATE);

		//why? trail and error, should wait a short delay between 2 PLL1 control register programming
		DO_DELAY_P(0x1000);

		if((HAL_GETREG32(HAL_REG_SYS_PMUSts) & PMU_STATUS_PLL1_STATE_BITS) != PLL1_WAIT_UPDATE_STATE)
			break;
	}

	// wait PLL1 lock
	do {
		pll_reg = HAL_GETREG32(HAL_REG_PLL1_CTRL);
	}while((pll_reg & PLL_CTRL_LOCK_DET) == 0);

	// open DDR2x clk
//	sys_reg = HAL_GETREG32(HAL_REG_SYS_APBGateCtrl);
//	sys_reg |= 0x20000000;
//	HAL_SETREG32(HAL_REG_SYS_APBGateCtrl, sys_reg);
//	sys_reg = HAL_GETREG32(HAL_REG_SYS_AHBGateCtrl);
//	sys_reg |= 0x79380000;
//	HAL_SETREG32(HAL_REG_SYS_AHBGateCtrl, sys_reg);


	ddrphyctrl_reg |= 0x000003FF;
	HAL_SETREG32(HAL_REG_SYS_DDR2PhyCtrl, ddrphyctrl_reg);

	// reset sdram dll
	ddr2_reg = HAL_GETREG32(HAL_REG_DDR2C_ModeReg);
	ddr2_reg |= DDR2_RESET_SDRAM_DLL_BIT;;
	HAL_SETREG32(HAL_REG_DDR2C_ModeReg, ddr2_reg);

	// issue exit SRF cmd
	ddr2_reg = HAL_GETREG32(HAL_REG_DDR2C_DRAMConfig);
	ddr2_reg &= ~DDR2_CMD_BITS;
	ddr2_reg |= DDR2_EXIT_SRF_CMD | DDR2_EXECUTE_CMD_BIT;;
	HAL_SETREG32(HAL_REG_DDR2C_DRAMConfig, ddr2_reg);

	// add @2012/5/2 for better FCS flow
	DO_DELAY_P(0x10000);

	// wait for POM do data training
	do{
		pom_reg = HAL_GETREG32(HAL_REG_POM_CTRL);
	}while((pom_reg & POM_CTRL_DO_DATA_TRAIN) == 0);

	// setup new DDR parameter
	ddr_p_addr = (unsigned int *)DDR_PARAMETER_ADDR_BASE; ddr_p_value = (unsigned int *)DDR_PARAMETER_VALUE_BASE;
	for(i =0; i < ddr_p_count; ddr_p_addr++, ddr_p_value++, i++)
		HAL_SETREG32(*ddr_p_addr, *ddr_p_value);

#if 0
	// 	enable sdram ODT
	HAL_SETREG32(0x90d001f4, 0x044);
	// enable DDRC read/write ODT
	HAL_SETREG32(0x99000080, 0x00f503ff);
	HAL_SETREG32(0x90d00098, 0x000f000F);
#endif

	// do DDR2 data training
	ddr2_reg = HAL_GETREG32(HAL_REG_DDR2C_CtrlConfig);
	HAL_SETREG32(HAL_REG_DDR2C_CtrlConfig, ddr2_reg | DDR2_DO_DATA_TRAIN);

	// add @2012/5/2 for better FCS flow
	DO_DELAY_P(0x10000);

	// wait for POM ddr2 OK
	do{
		pom_reg = HAL_GETREG32(HAL_REG_POM_CTRL);
	}while((pom_reg & POM_CTRL_DDR_OK) == 0);

#if 1 //Ted: for sw workaround. To check DDR init flag to make sure DDR data training is completed
	do{
		ddr2_reg = HAL_GETREG32(HAL_REG_SYS_DDR2_INIT_STS);
	}while((ddr2_reg & DDR2_INIT_STS_BUSY) == 1);
#endif

	// restore DDR content, that is overwrite by DDR data training (in this procedure, can not use any function call)
	SrcAddr = (unsigned int *)DDR_TEMP_BASE; DestAddr = (unsigned int *)HAL_OMB_DDR2_BASE;
	for(size = 0x8; size > 0; size--, DestAddr++, SrcAddr++)
		*DestAddr = *SrcAddr;

#if 1 //Ted: for sw workaround. To check the data content
	SrcAddr = (unsigned int *)DDR_TEMP_BASE; DestAddr = (unsigned int *)HAL_OMB_DDR2_BASE;
	for(size = 0x8; size > 0; size--, DestAddr++, SrcAddr++)
		if(*DestAddr != *SrcAddr)
			while(1);
#endif

	__asm__ __volatile__ (
			"mov		%0, #0x00000005				@I/D Cache enable bit & MMU \n\t" \
			"orr		%0, %0, #0x00001000			\n\t" \
			"mrc		p15, 0, %1, c1, c0, 0		@read CR \n\t" \
			"orr		%1, %1, %0					@modify content \n\t" \
			"mcr		p15, 0, %1, c1, c0, 0		@write CR \n\t" \
			"mov		pc, %2		\n\t" \
			"pll_fcs_back:			\n\t" \
			:"=r"(cp15_flag), "=r"(cp15_reg):"r"(back_addr)
	);


	Set_WDT(Fin, 0);

	// restore VIC enable settings
	HAL_SETREG(HAL_REG_VIC1_IntEnable, vic_enable_reg);

	return TRUE;
}
void dummy(void)
{
	__asm__ __volatile__ (
		// issue enter SRF cmd
		"pll_fcs_end:					\n\t" \
	);
}


int PLL0_FCS(unsigned int Fin, unsigned int NR, unsigned int NF, unsigned int NO)
{
	unsigned int delayloop, pom_reg, pom_status, pll_reg, BS, fvco, vic_enable_reg, rc;

	//back-up all VIC enable settings
	vic_enable_reg = HAL_GETREG(HAL_REG_VIC1_IntEnable);
	//disable all VIC sources
	HAL_SETREG(HAL_REG_VIC1_IntEnableClear, 0xffffffff);

	//enable POM FIQ
	FCSIntrOccur_IRQ = FCSIntrOccur_FIQ = 0;
	pom_reg = HAL_GETREG32(HAL_REG_POM_CTRL);
	HAL_SETREG32(HAL_REG_POM_CTRL, pom_reg | POM_CTRL_CPU_FIRQ_EN);

	// set PLL0 Freq
	fvco = Fin/1000/1000 * (NF+1) / (NR+1);
	if(fvco >= 500)
		BS = 2;
	else if(fvco >= 300)
		BS = 1;
	else if(fvco >= 100)
		BS = 0;
	else
		BS = 2;

	pll_reg = (NR << PLL_NR_SHIFT) | (NF << PLL_NF_SHIFT) | (NO << PLL_NO_SHIFT) | (BS << PLL_BS_SHIFT);

	//enable WDT for HW PLL failed
	Set_WDT(Fin, 3);

	// set PLL0 Freq, due to HW design bug, software should poll whether PLL enter PLL update state
	while(1)
	{
		HAL_SETREG32(HAL_REG_PLL0_CTRL, pll_reg);
		HAL_SETREG32(HAL_REG_PLL0_CTRL, pll_reg | PLL_CTRL_UPDATE);

		DO_DELAY_P(0x1000);

		if((HAL_GETREG32(HAL_REG_SYS_PMUSts) & PMU_STATUS_PLL0_STATE_BITS) == PLL0_WAIT_UPDATE_STATE)
			break;
	}

	DO_DELAY_P(0x1000);
	if((HAL_GETREG32(HAL_REG_SYS_PMUSts) & PMU_STATUS_PLL0_STATE_BITS) != PLL0_WAIT_UPDATE_STATE)
	{
		while(1);
	}

	// CPU sleep
	__asm__ __volatile__ (
		"mcr p15, 0, r0, c7, c0, 4\n\t");

	if((FCSIntrOccur_IRQ != 1) && (FCSIntrOccur_FIQ != 1))
		rc = 0; // fail
	else
		rc = 1; // success

	Set_WDT(Fin, 0);

	// restore VIC enable settings
	HAL_SETREG(HAL_REG_VIC1_IntEnable, vic_enable_reg);


	return rc;
}

int CPU_SetTurbo(int turbo)
{
	unsigned int pom_reg, cpumode_reg, vic_enable_reg, x, y;
	int rc = -1;

	// hardware limitation: if CPU turbo is on CAN NOT on again, if CPU turbo is off CAN NOT off again.
	if(CPU_GetTurbo() == turbo)
		return 1;

	pom_reg = HAL_GETREG32(HAL_REG_POM_CTRL);
	pom_reg |= POM_CTRL_CPU_TURBO_ECO;
	HAL_SETREG32(HAL_REG_POM_CTRL, pom_reg);

	//back-up all VIC enable settings
	vic_enable_reg = HAL_GETREG(HAL_REG_VIC1_IntEnable);
	//disable all VIC sources
	HAL_SETREG(HAL_REG_VIC1_IntEnableClear, 0xffffffff);

	// enable POM IRQ
	FCSIntrOccur_IRQ = FCSIntrOccur_FIQ = 0;
	pom_reg = HAL_GETREG32(HAL_REG_POM_CTRL);
	HAL_SETREG32(HAL_REG_POM_CTRL, pom_reg & ~POM_CTRL_CPU_FIRQ_EN);
//	HAL_SETREG32(HAL_REG_POM_CTRL, pom_reg | POM_CTRL_CPU_FIRQ_EN);

	// enable or disable CPU Turbo
	cpumode_reg = HAL_GETREG32(HAL_REG_SYS_CPUMode);
	if(turbo == 1)
		HAL_SETREG32(HAL_REG_SYS_CPUMode, cpumode_reg | CPU_TURBO_BIT);
	else
		HAL_SETREG32(HAL_REG_SYS_CPUMode, cpumode_reg & (~CPU_TURBO_BIT));
#if 0
	__asm__ __volatile__ (
			"tci_loop_turbo:									\n\t" \
			"mrc		p15, 0, r15, c7, c14, 3					\n\t" \
			"bne		tci_loop_turbo							\n\t" \
			"mov		%0, #0x00000005							@I/D Cache enable bit & MMU \n\t" \
			"orr		%0, %0, #0x00001000						\n\t" \
			"mvn		%0, %0									\n\t" \
			"mrc		p15, 0, %1, c1, c0, 0					@read CR \n\t" \
			"and		%1, %1, %0								@modify content \n\t" \
			"mcr		p15, 0, %1, c1, c0, 0					@write CR \n\t" \
			:"=r"(x), "=r"(y):
	);
#endif
	// CPU sleep
	__asm__ __volatile__ (
		"mov r0, r0 \n\t" \
		"mcr p15, 0, r0, c7, c0, 4\n\t");

	if((FCSIntrOccur_IRQ != 1) && (FCSIntrOccur_FIQ != 1))
		rc = 0; // fail
	else
		rc = 1; // success
#if 0
	__asm__ __volatile__ (
			"mov		%0, #0x00000005				@I/D Cache enable bit & MMU \n\t" \
			"orr		%0, %0, #0x00001000			\n\t" \
			"mrc		p15, 0, %1, c1, c0, 0		@read CR \n\t" \
			"orr		%1, %1, %0					@modify content \n\t" \
			"mcr		p15, 0, %1, c1, c0, 0		@write CR \n\t" \
			"@tci_loop_turbo2:								\n\t" \
			"@mrc		p15, 0, r15, c7, c14, 3		\n\t" \
			"@bne		tci_loop_turbo2							\n\t" \
			:"=r"(x), "=r"(y)
	);
#endif
	// restore VIC enable settings
	HAL_SETREG(HAL_REG_VIC1_IntEnable, vic_enable_reg);

	return rc;
}
