#include <string.h>
#include <stddef.h>
#include <stdio.h>

#include "sys.h"

#include "dosFS.h"
#include "dosFSi.h"

static bool_T dosIsValidPathChar_ii(char c);

static void dosExtractExtension_ii(
				char *szName,		/* I */
				char *szFName,		/* O */
				char *szEName);		/* O */

static bool_T dosLocateFirstPathNameIdx_ii(
				char *szFullName,	/* I */
				int *pnDirEnd);		/* O */

static bool_T dosLocateFileNameIdx_ii(
				char *szFullName,	/* I */
				int *pnFileStart);	/* O */

static bool_T dosMarkRDLastestDirEntry16_ii(
				dosBS_Id,
				int nNThDirEntry);

/* mark the nth directory entry in cluster as
"lastest directory entry" */
static bool_T dosMarkClusterLatestDirEntry_ii(
				dosBS_Id,
				int nCluster,
				int nNThDirEntry);


/* similiar to probe under root directory */
static bool_T dosProbeFileUnderCluster_ii(
				dosBS_Id,	/* I */
				bool_T bProbeDir,	/* I */
				int nCluster,		/* I */
				char *szFileName,	/* I */
				unsigned char *pDirEntry,	/* O */
				int *pnDECluster,	/* O */
				int *pnDEOffset);	/* O */

static bool_T dosProbeHierFile_ii(
				dosBS_Id,				/* I */
				bool_T bProbeDir,			/* I, probe a directory? */
				int nCluster,				/* I, the first cluster of the directory */
				char *szFractionalFile,		/* I */
				unsigned char *pDirEntry,	/* O, directory entry */
				int *pnDECluster,			/* O */
				int *pnDEOffset);			/* O */

static bool_T dosProbeFileUnderRD16_ii(
				dosBS_Id,	/* I */
				bool_T bProbeDir,	/* I, probe a directory? */
				char *szFName,		/* I, file name */
				char *szExtName, 	/* I, file extension name */
				unsigned char *pDirEntry,	/* O, directory entry */
				int *pnDESector,	/* O */
				int *pnDEOffset);	/* O */

/* similiar to create under root directory */
static bool_T dosCreateFileUnderDirFirstCluster_ii(
				dosBS_Id,
				bool_T bCreateDir,
				int nCluster,
				char *szFileName);

static bool_T dosCreateHierFile_ii(
				dosBS_Id,
				bool_T bCreateDir,
				int nCluster,
				char *szFractionalFN);

static bool_T dosCreateFileUnderRD16_ii(
				dosBS_S *pBS,
				bool_T bCreateDir,	/* create a directory? */
				char *szFName,		/* file name */
				char *szExtName); 	/* file extension name */

static bool_T dosProbe_ii(
				dosBS_Id pBS,	/* I */
				bool_T bDir,		/* I */
				char *szName,		/* I */
				unsigned char *pDirEntry,	/* O */
				unsigned int *pDEBlockNum,	/* O */
				unsigned int *pDEOffset);	/* O */

static bool_T dosIsFileUnderRD16_ii(char *szName);



/* operate on directory entry */
static bool_T dosIsAttrROHFSFVN_ii(unsigned char *pDirEntry);

static bool_T dosIsAttrVN_ii(unsigned char *pDirEntry);

static bool_T dosIsDir_ii(unsigned char *pDirEntry);

static bool_T dosIsNormalFile_ii(unsigned char *pDirEntry);

static int dosGetFileFirstCluster16_ii(unsigned char *pDirEntry);

static int dosGetFileFirstCluster32_ii(unsigned char *pDirEntry);

static int dosGetFileSize_ii(unsigned char *pDirEntry);

static void dosSetFileSize_ii(unsigned char *pDirEntry, int nSize);

static void dosGetFileName_ii(unsigned char *pDirEntry, char *buffer);

static void dosMakeDirEntry16_ii(
				unsigned char *pDEntry,	/* O */
				bool_T bCreateDir,	/* I */
				char *szFName,		/* I */
				char *szExtName, 	/* I */
				int nFirstCluster);	/* I */

static void dosMakeDirEntry32_ii(
				unsigned char *pDEntry,	/* O */
				bool_T bCreateDir,	/* I */
				char *szFName,		/* I */
				char *szExtName, 	/* I */
				int nFirstCluster);	/* I */

static void dosMakeDeletedDirEntry_ii(unsigned char *pDirEntry);

static void dosSetDirEntryCluster16_ii(unsigned char *pDEntry, int nFirstCluster);

static void dosSetDirEntryCluster32_ii(unsigned char *pDEntry, int nFirstCluster);



static void dosSyncClusterByFilePtrPos_ii(dosFile_Id fId);




static bool_T dosIsValidPathChar_ii(char c)
{
	if (c == ' ')
		return (FALSE);

	if (c >= '!' && c <= '~')
		return (TRUE);

	//return (FALSE);
	return (TRUE);		//allow wider name space (e.g. chinese), 2005/06/07, stan
}

bool_T dosIsValidPath(char *szName)
{
	/*
		If true is returned, each directory name or file name seperated by
		the directory seperator is bounded within 8 characters and extension
		name is bounded with 3 characters.
	*/

#define STATE_SEPERATOR		1
#define STATE_FILENAME		2
#define STATE_EXTNAME		3

	unsigned short nState; /* FSM's state of this function */
	unsigned short nLen; /* length of the file name or the extension file name */


	if (!szName || !szName[0])
		return (TRUE);



	{
		/* suppose that there is always a '\\' at the beginning */

		if (*szName == '\\')
			szName += 1;

		nState = STATE_SEPERATOR;
		nLen = 0;
	}



	while (*szName)
	{
		if (*szName == '\\')
		{
			if (nState == STATE_SEPERATOR)
			{
				return (FALSE);
			}
			else
			{
				nState = STATE_SEPERATOR;
			}
		}
		else if (*szName == '.')
		{
			if (nState == STATE_EXTNAME)
			{
				return (FALSE);
			}
			else
			{
				nState = STATE_EXTNAME;
				nLen = 0;
			}
		}
		else if (dosIsValidPathChar_ii(*szName))
		{
			if (nState == STATE_SEPERATOR)
			{
				nState = STATE_FILENAME;
				nLen = 1;
			}
			else if (nState == STATE_FILENAME)
			{
				++nLen;

				if (nLen > 8)
					return (FALSE);
			}
			else if (nState == STATE_EXTNAME)
			{
				++nLen;

				if (nLen > 3)
					return (FALSE);
			}
		}
		else
		{
			return (FALSE);
		}

		szName += 1;
	}

	return (TRUE);

#undef STATE_SEPERATOR
#undef STATE_FILENAME
#undef STATE_EXTNAME
}

char * dosBeautyFileName(char *szWorkDir, char *szUserSpecifyPath)
{
	int nLen, nIdx;
	static char buffer[256];

	/*
		1. If szUserSpecifyPath begins with '\\', it dominate
		the file name.

	*/

	if (szUserSpecifyPath[0] == '\\')
	{
		strcpy(buffer, szUserSpecifyPath);

		if (strcmp("\\", buffer) != 0)
		{
			/* not the root directory */

			nLen = strlen(buffer);
			if (buffer[nLen-1] == '\\')
				buffer[nLen-1] = '\0';
		}
	}
	else if (strcmp("..", szUserSpecifyPath) == 0)
	{
		strcpy(buffer, szWorkDir);

		if (strcmp("\\", szWorkDir) != 0)
		{
			nLen = strlen(buffer);
            
			/*
                \aaaa\bbb\ccc 
                           /|\
                            |
                     <--    |
                            |
            
			*/
            
			for (nIdx = nLen-1; nIdx >= 0; --nIdx)
			{
				if (buffer[nIdx] == '\\')
				{
					if (nIdx == 0)
					{
						/* the case of "\abcdef" */
						buffer[1] = '\0';
					}
					else
					{
						/* the case of "\abcdef\ghi\jkl..." */
						buffer[nIdx] = '\0';
					}

					break;
				}
			}
		}
	}
	else if (strcmp(".", szUserSpecifyPath) == 0 ||
		strcmp(".\\", szUserSpecifyPath) == 0)
	{
		strcpy(buffer, szWorkDir);
	}
	else
	{
		strcpy(buffer, szWorkDir);

		if (strcmp(szWorkDir, "\\") != 0)
		{
			nLen = strlen(buffer);
			if (buffer[nLen-1] == '\\')
			{
				strcat(buffer, szUserSpecifyPath);
			}
			else
			{
				strcat(buffer, "\\");
				strcat(buffer, szUserSpecifyPath);
			}
		}
		else
		{
			strcat(buffer, szUserSpecifyPath);
		}

		nLen = strlen(buffer);
		if (buffer[nLen-1] == '\\')
			buffer[nLen-1] = '\0';

	}

	return (buffer);
}

static void dosExtractExtension_ii(
				char *szName,		/* I */
				char *szFName,		/* O */
				char *szEName)		/* O */
{
	/*
		szName = xxxxx.xxx = szFName.szEName

		The caller to this function should be responsible for
		the buffer's allocation.
	*/

	int i, j, n;

	j = strlen(szName);
	for (i = 0; i < j; ++i)
	{
		if (szName[i] == '.')
			break;

		szFName[i] = szName[i];
	}
	szFName[i] = '\0';
	n = i;
	++i;
	for ( ; i < j; ++i)
		szEName[i-n-1] = szName[i];
	szEName[i-n-1] = '\0';
}


static bool_T dosLocateFirstPathNameIdx_ii(
				char *szPartialName,/* I */
				int *pnDirEnd)		/* O */
{
	/* 
		    +------ *pnDirEnd = 3, which will be identified
		   \|/
		    V
		"xxxx\xxxx\xxxx" 

	     --->
	*/


	if (!szPartialName || !szPartialName[0])
		return (FALSE);

	for (*pnDirEnd = 0; szPartialName[*pnDirEnd] != '\0'; ++(*pnDirEnd))
	{
		if (szPartialName[*pnDirEnd] == '\\')
		{
			(*pnDirEnd) -= 1;
			return (TRUE);
		}
	}

	*pnDirEnd -= 1;
	return (TRUE);
}


static bool_T dosLocateFileNameIdx_ii(
				char *szFullName,	/* I */
				int *pnFileStart)	/* O */
{
	/* 
				   +------ *pnFileStart = 9, which will be identified
				  \|/
				   V
		"xxxx\xxxx\xxxx" 

				   <---
	*/

	int nLen;

	if (!szFullName || !szFullName[0])
		return (FALSE);

	nLen = strlen(szFullName);
	for (*pnFileStart = nLen - 1; *pnFileStart >=0; --(*pnFileStart))
	{
		if (szFullName[*pnFileStart] == '\\')
		{
			*pnFileStart += 1;
			return (TRUE);
		}
	}

	*pnFileStart = 0;
	return (TRUE);
}

static bool_T dosMarkRDLastestDirEntry16_ii(
				dosBS_Id pBS,
				int nNThDirEntry)
{
	/*
	 * For FAT12/16 only.
	 */

	unsigned int nRDEntriesPerSector;
	unsigned char dirEntry[DOS_DIR_ENTRY_SIZE];

	sysDAssert(pBS->p1bs.fatSystem != DOS_FAT_BITS_32);

	/* make an empty directory entry, (0th byte = 0 : lastest entry) */
	sysMemSet(dirEntry, 0, DOS_DIR_ENTRY_SIZE);

	/* place dirEntry's content to the nNThDirEntry directory entry in RD */
	nRDEntriesPerSector = pBS->p1bs.bytesPerSector/DOS_DIR_ENTRY_SIZE;
	dosWriteMediaSectorNByte_i(pBS,
					pBS->p1bs.nRDStartSector16 + nNThDirEntry/nRDEntriesPerSector,
					(nNThDirEntry%nRDEntriesPerSector) * DOS_DIR_ENTRY_SIZE,
					DOS_DIR_ENTRY_SIZE, dirEntry);

	return (TRUE);
}

/* mark the nth directory entry in cluster as
"lastest directory entry" */
static bool_T dosMarkClusterLatestDirEntry_ii(
				dosBS_Id pBS,
				int nCluster,
				int nNThDirEntry)
{
	unsigned int nDEntriesPerCluster;
	unsigned char dirEntry[DOS_DIR_ENTRY_SIZE];


	/* make an empty directory entry, (0th byte = 0 : lastest entry) */
	sysMemSet(dirEntry, 0, DOS_DIR_ENTRY_SIZE);

	/* place dirEntry's content to the nNThDirEntry directory
		entry in the cluster */
	nDEntriesPerCluster = pBS->p1bs.bytesPerCluster/DOS_DIR_ENTRY_SIZE;
	return (dosWriteDOSDAClusterNByte_i(pBS,
					&(pBS->DataCache),
					nCluster, 
					(nNThDirEntry%nDEntriesPerCluster)*DOS_DIR_ENTRY_SIZE, /* offset */
					DOS_DIR_ENTRY_SIZE, /* size */
					dirEntry)); /* content */
}

static bool_T dosProbeFileUnderCluster_ii(
				dosBS_Id pBS,
				bool_T bProbeDir,
				int nCluster,
				char *szFileName,
				unsigned char *pDirEntry,
				int *pnDECluster,
				int *pnDEOffset)
{
	unsigned char buffer[DOS_BYTES_PER_SEC];

	int nIdx1;
	unsigned int nDEntriesPerSector;
	unsigned int nDEntriesPerCluster;
	unsigned int nRelativeEntryInSector;
	int nFLen, nELen;
	char szFName[9], szEName[4];
	unsigned char *pEntryStart, cDEFirstChar;

	*pnDECluster = 0;
	*pnDEOffset	= 0;

	/* initialize the file name buffers */
	dosExtractExtension_ii(szFileName, szFName, szEName);
	nFLen = strlen(szFName);
	nELen = strlen(szEName);
	nFLen = (nFLen > 8 ? 8 : nFLen);
	nELen = (nELen > 3 ? 3 : nELen);
	
	szFName[8] = '\0';
	szEName[3] = '\0';
	for (nIdx1 = 7; nIdx1 >= nFLen; --nIdx1)
		szFName[nIdx1] = ' ';
	for (nIdx1 = 2; nIdx1 >= nELen; --nIdx1)
		szEName[nIdx1] = ' ';


	sysMemSet(pDirEntry, 0, DOS_DIR_ENTRY_SIZE);

	/* probe the specified cluster in the media */
	nDEntriesPerSector = DOS_BYTES_PER_SEC/DOS_DIR_ENTRY_SIZE;
	nDEntriesPerCluster = pBS->p1bs.bytesPerCluster/DOS_DIR_ENTRY_SIZE;
	do
	{
		for (nIdx1 = 0; nIdx1 < nDEntriesPerCluster; ++nIdx1)
		{
			nRelativeEntryInSector = nIdx1 % nDEntriesPerSector;

			if (nRelativeEntryInSector == 0)
			{
				if (!dosReadDOSDAClusterNByte_i(pBS,
							&(pBS->DataCache),
							nCluster,
							(nIdx1/nDEntriesPerSector)*DOS_BYTES_PER_SEC,
							DOS_BYTES_PER_SEC,
							buffer))
					goto LLL_PROBE_RESULT_FAIL;
			}


			pEntryStart = buffer + nRelativeEntryInSector*DOS_DIR_ENTRY_SIZE;
			cDEFirstChar = *pEntryStart;

			/* examine the directory entry */
			if (cDEFirstChar == '\0') /* lastest directory entry */
			{
				goto LLL_PROBE_RESULT_FAIL;
			}
			else if (cDEFirstChar == (unsigned char) 0xE5 || cDEFirstChar == '.')
			{
				/* file deleted */
				continue;
			}
			else /* find a directory entry; examine it! */
			{
				memcpy(pDirEntry, pEntryStart, DOS_DIR_ENTRY_SIZE);

				if (bProbeDir && !dosIsDir_ii(pDirEntry))
					continue;
				if (!bProbeDir && !dosIsNormalFile_ii(pDirEntry))
					continue;

				if (cDEFirstChar == (unsigned char) 0x05)
				{
					/* ASCII code 0xE5 */
					pDirEntry[0] = (unsigned char) 0xE5;
				}

				if (sysStrNCaseCmp((char *) pDirEntry, szFName, 8) == 0)
				{
					if ((unsigned char) pDirEntry[0] == (unsigned char) 0xE5)
					{
						pDirEntry[0] = (unsigned char) 0x05;
					}

					if (nELen == 0)
					{
						*pnDECluster = nCluster;
						*pnDEOffset = (nIdx1%nDEntriesPerCluster)*DOS_DIR_ENTRY_SIZE;
						goto LLL_PROBE_RESULT_SUCCESS;
					}
					else
					{
						if (sysStrNCaseCmp((char *) (pDirEntry+8), szEName, 3) == 0)
						{
							*pnDECluster = nCluster;
							*pnDEOffset = (nIdx1%nDEntriesPerCluster)*DOS_DIR_ENTRY_SIZE;
							goto LLL_PROBE_RESULT_SUCCESS;
						}
					}
				}
			}
		}

		nCluster = dosFATGetNextCluster_i(pBS, nCluster);
	} while (!dosFATIsNilFEC_i(pBS, nCluster));

LLL_PROBE_RESULT_FAIL:
	return (FALSE);

LLL_PROBE_RESULT_SUCCESS:
	return (TRUE);
}


static bool_T dosProbeHierFile_ii(
				dosBS_Id pBS,	/* I */
				bool_T bProbeDir,			/* I, probe a directory? */
				int nCluster,				/* I, the first cluster of the directory */
				char *szFileName,			/* I, file name */
				unsigned char *pDirEntry,	/* O, directory entry */
				int *pnDECluster,			/* O */
				int *pnDEOffset)			/* O */
{
	int nDirEndIdx, nNameLen;
	char *pFractionalFile;
	bool_T bProbeResult;

	/*
		    +------ nDirEndIdx = 3 (index)
		    |
		    |
		"xxxx\xxxxx\xxxxx" = szFileName
		     ||
			 ||
			 |+------ pFractionalFile (memory pointer)
			 |
			 +------- replace this character as '\0'
	*/


	while (TRUE)
	{
		nNameLen = strlen(szFileName);
		if (!dosLocateFirstPathNameIdx_ii(szFileName, &nDirEndIdx))
			return (FALSE);
    
		if (nDirEndIdx != nNameLen -1)
		{
			/* break the file name into 2 parts:
				dirname : pointed by szFileName
				fractional name : pointed by pFractionalFile
			*/
			szFileName[nDirEndIdx+1] = '\0';
			pFractionalFile = szFileName + nDirEndIdx + 2;
		}
		else
		{
			pFractionalFile = NULL;
		}
    
		if (!pFractionalFile)
		{
			bProbeResult = dosProbeFileUnderCluster_ii(pBS,
						bProbeDir, nCluster, szFileName, pDirEntry,
						pnDECluster, pnDEOffset);
		}
		else
		{
			bProbeResult = dosProbeFileUnderCluster_ii(pBS,
						TRUE, nCluster, szFileName, pDirEntry,
						pnDECluster, pnDEOffset);
		}
    
		if (bProbeResult)
		{
			if (!pFractionalFile) /* end of probing */
			{
				return (TRUE);
			}
			else
			{
				if (pBS->p1bs.fatSystem == DOS_FAT_BITS_32)
					nCluster = dosGetFileFirstCluster32_ii(pDirEntry);
				else
					nCluster = dosGetFileFirstCluster16_ii(pDirEntry);
    
				/* continue to probe deeper */
				szFileName = pFractionalFile;
			}
		}
		else
		{
			return (FALSE);
		}
	}
}

static bool_T dosProbeFileUnderRD16_ii(
			dosBS_Id pBS,
			bool_T bProbeDir,
			char *szFName,
			char *szExtName,
			unsigned char *pDirEntry,
			int *pnDESector,
			int *pnDEOffset)
{
	/* search file under root directory (FAT12/16 only) */

	/* prerequisite: dosvBSs_g[nMdHandle] has datum */

	unsigned char buffer[DOS_BYTES_PER_SEC];

	bool_T bResult;
	unsigned int nIdx1;
	unsigned int nRDEntriesPerSector;
	unsigned int nRelativeEntryInSector;
	int nFLen, nELen;
	char szFileName[9];
	char szExtensionName[4];

	*pnDESector = 0;
	*pnDEOffset = 0;

	/* initialize the file name buffers */
	memset(szFileName, ' ', 8);
	szFileName[8] = '\0';
	nFLen = strlen(szFName);
	nFLen = (nFLen > 8 ? 8 : nFLen);
	memcpy(szFileName, szFName, nFLen);
	
	memset(szExtensionName, ' ', 3);
	szExtensionName[3] = '\0';
	if (szExtName == NULL_STR || szExtName[0] == '\0')
	{
		nELen = 0;
	}
	else
	{
		nELen = strlen(szExtName);
		nELen = (nELen > 3 ? 3 : nELen);
		memcpy(szExtensionName, szExtName, nELen);
	}



	sysMemSet(pDirEntry, 0, DOS_DIR_ENTRY_SIZE);

	/* probe the root directory area in the media */
	nRDEntriesPerSector = DOS_BYTES_PER_SEC/DOS_DIR_ENTRY_SIZE;

	for (nIdx1 = 0; nIdx1 < pBS->p1bs.numEntriesInRD; ++nIdx1)
	{
		nRelativeEntryInSector = nIdx1 % nRDEntriesPerSector;

		if (nRelativeEntryInSector == 0)
		{
			/* read root directory entries from media to buffer */

			bResult = dosReadMediaSectorNByte_i(
						pBS,
						pBS->p1bs.nRDStartSector16 + (nIdx1/nRDEntriesPerSector),
						0,
						DOS_BYTES_PER_SEC,
						buffer);

			if (!bResult)
				return (FALSE);
		}


		/* retrive data stored in cache to a directory entry buffer */
		memcpy(pDirEntry,
			 buffer+nRelativeEntryInSector*DOS_DIR_ENTRY_SIZE,
			 DOS_DIR_ENTRY_SIZE);

		/* examine the directory entry */	
		if (pDirEntry[0] == '\0') /* lastest directory entry */
		{
			return (FALSE);
		}
		else if (pDirEntry[0] == (unsigned char) 0xE5)
		{
			/* file deleted */
			continue;
		}
		else
		{
			if (bProbeDir && !dosIsDir_ii(pDirEntry))
			{
				continue;
			}
			if (!bProbeDir && !dosIsNormalFile_ii(pDirEntry))
			{
				continue;
			}

			if (pDirEntry[0] == 0x05) /* ASCII code 0xE5 */
				pDirEntry[0] = (char) 0xE5;

			if (sysStrNCaseCmp((char *) pDirEntry, szFileName, 8) == 0 &&
				sysStrNCaseCmp((char *) (pDirEntry+8), szExtensionName, 3) == 0)
			{
				if (pDirEntry[0] == (unsigned char) 0xE5)
				{
					pDirEntry[0] = (unsigned char) 0x05;
				}

				*pnDESector = pBS->p1bs.nRDStartSector16 + nIdx1/nRDEntriesPerSector;
				*pnDEOffset = (nIdx1 * DOS_DIR_ENTRY_SIZE) % pBS->p1bs.bytesPerSector;
					
				return (TRUE);
			}
		}
	}

	return (FALSE);
}

static bool_T dosCreateFileUnderDirFirstCluster_ii(
				dosBS_Id pBS,
				bool_T bCreateDir,
				int nCluster,	/* the 1st cluster of the file */
				char *szFileName)
{
	int nIdx1;
	unsigned int nDEntriesPerCluster;
	int nFLen, nELen;
	int nCurrentCluster, nNextCluster;
	int nFreeCluster;
	int nTerminalClusterId;
	char szFName[9], szEName[4];
	char cFirstByte;
	unsigned char dirEntry[DOS_DIR_ENTRY_SIZE];


	/* initialize the file name buffers */
	dosExtractExtension_ii(szFileName, szFName, szEName);
	nFLen = strlen(szFName);
	nELen = strlen(szEName);
	szFName[8] = '\0';
	szEName[3] = '\0';
	for (nIdx1 = 7; nIdx1 >= nFLen; --nIdx1)
	{
		szFName[nIdx1] = ' ';
	}
	for (nIdx1 = 2; nIdx1 >= nELen; --nIdx1)
	{
		szEName[nIdx1] = ' ';
	}
	

	switch (pBS->p1bs.fatSystem)
	{
	case DOS_FAT_BITS_12:
		nTerminalClusterId = DOS_F12EC_LAST_CLUSTER_IN_FILE;
		break;

	case DOS_FAT_BITS_16:
		nTerminalClusterId = DOS_F16EC_LAST_CLUSTER_IN_FILE;
		break;

	case DOS_FAT_BITS_32:
		nTerminalClusterId = DOS_F32EC_LAST_CLUSTER_IN_FILE;
		break;
	}


	nDEntriesPerCluster = pBS->p1bs.bytesPerCluster/DOS_DIR_ENTRY_SIZE;
	sysMemSet(dirEntry, 0, DOS_DIR_ENTRY_SIZE);


	/* examine whether any file/directory with the same name
		has existed */
	nNextCluster = nCluster;
	nCurrentCluster = nCluster;
	for ( ; !dosFATIsNilFEC_i(pBS, nNextCluster); )
	{
		nCurrentCluster = nNextCluster;

		for (nIdx1 = 0; nIdx1 < nDEntriesPerCluster; ++nIdx1)
		{
			dosReadDOSDAClusterNByte_i(pBS, &(pBS->DataCache),
								nCurrentCluster,
								nIdx1*DOS_DIR_ENTRY_SIZE,
								DOS_DIR_ENTRY_SIZE,
								dirEntry);

			cFirstByte = dirEntry[0];
			if ((unsigned char) cFirstByte == (unsigned char) 0xE5)
			{
				continue;
			}
			else if (cFirstByte == 0)
			{
				goto LLL_START_TO_CREATE;
			}
			else
			{
				if (sysStrNCaseCmp((char *) dirEntry, szFName, 8) == 0 &&
					sysStrNCaseCmp((char *) (dirEntry+8), szEName, 3) == 0)
				{
					return (FALSE);
				}
			}
		}

		nNextCluster = dosFATGetNextCluster_i(pBS, nCurrentCluster);
	}





	/* create the file under the specified cluster */
LLL_START_TO_CREATE:
	nNextCluster = nCluster;
	nCurrentCluster = nCluster;
	for ( ; !dosFATIsNilFEC_i(pBS, nNextCluster); )
	{
LLL_START_TO_CREATE2:
		nCurrentCluster = nNextCluster;

		for (nIdx1 = 0; nIdx1 < nDEntriesPerCluster; ++nIdx1)
		{
			dosReadDOSDAClusterNByte_i(pBS, &(pBS->DataCache),
								nCurrentCluster,
								nIdx1*DOS_DIR_ENTRY_SIZE,
								DOS_DIR_ENTRY_SIZE,
								dirEntry);

			/* examine the directory entry */
			cFirstByte = dirEntry[0];

			if (cFirstByte == '\0' ||
				(unsigned char) cFirstByte == (unsigned char) 0xE5)
			{
				/*
				 * 2006.06.16
				 * For directory creation, allocate a cluster.
				 * For normal file creation, don't allocate any cluster.
				 */

				if (bCreateDir)
				{
					nFreeCluster = dosFATGetFreeCluster_i(pBS);
					if (dosFATIsNilFEC_i(pBS, nFreeCluster))
						goto LLL_CREATE_RESULT_FAIL;

					/* fill the free cluster as blank */
					dosWriteDOSDAClusterBlank_i(pBS, nFreeCluster);

					/* "." */
					if (pBS->p1bs.fatSystem == DOS_FAT_BITS_32)
						dosMakeDirEntry32_ii(dirEntry, TRUE, ".", "   ", nFreeCluster);
					else
						dosMakeDirEntry16_ii(dirEntry, TRUE, ".", "   ", nFreeCluster);
					dosWriteDOSDAClusterNByte_i(pBS, &(pBS->DataCache),
						nFreeCluster, 0, DOS_DIR_ENTRY_SIZE, dirEntry);

					/* ".." */
					if (pBS->p1bs.fatSystem == DOS_FAT_BITS_32)
					{
						if (nCluster == pBS->p1bs.nRDStartCluster32)
							nCluster = 0;

						dosMakeDirEntry32_ii(dirEntry, TRUE, "..", "   ", nCluster);
					}
					else
					{
						dosMakeDirEntry16_ii(dirEntry, TRUE, "..", "   ", nCluster);
					}
					dosWriteDOSDAClusterNByte_i(pBS, &(pBS->DataCache),
						nFreeCluster, DOS_DIR_ENTRY_SIZE, DOS_DIR_ENTRY_SIZE, dirEntry);
				}
				else
				{
					nFreeCluster = 0;
				}

				/* write a directory entry for the created file to the directory */
				if (pBS->p1bs.fatSystem == DOS_FAT_BITS_32)
					dosMakeDirEntry32_ii(dirEntry, bCreateDir, szFName, szEName, nFreeCluster);
				else
					dosMakeDirEntry16_ii(dirEntry, bCreateDir, szFName, szEName, nFreeCluster);
				dosWriteDOSDAClusterNByte_i(pBS, &(pBS->DataCache),
						nCurrentCluster, 
						nIdx1*DOS_DIR_ENTRY_SIZE,
						DOS_DIR_ENTRY_SIZE,
						dirEntry);

				/* update FATs: for the created file */
				if (bCreateDir)
					dosFATUpdateClusterEntry_i(pBS, nFreeCluster, nTerminalClusterId);

				/* update directory's content: its directory entry */
				if (cFirstByte == '\0')
				{
					if (nIdx1+1 == nDEntriesPerCluster) 
					{
						/* nothing to do; this case means that this "file
							creation" fills the cluster, nCurrentCluster, full */
					}
					else
					{
						/* mark the directory entry, nIdx1+1, which follows
							the nIdx1 as lastest one */
						dosMarkClusterLatestDirEntry_ii(pBS, nCurrentCluster, nIdx1+1);
					}
				}

				goto LLL_CREATE_RESULT_SUCCESS;
			}
			else
			{
				continue;
			}
		}

		nNextCluster = dosFATGetNextCluster_i(pBS, nCurrentCluster);
	}


	/* after the searching a "blank" directory entry
		on the existing clusters, which belongs to the file,
		no "blank" directory entry is found; it's necessary
		to allocate another cluster */
	nFreeCluster = dosFATGetFreeCluster_i(pBS);
	if (nFreeCluster == 0) /* no media space */
		goto LLL_CREATE_RESULT_FAIL;

	dosFATUpdateClusterEntry_i(pBS, nCurrentCluster, nFreeCluster);
	dosFATUpdateClusterEntry_i(pBS, nFreeCluster, nTerminalClusterId);
	dosWriteDOSDAClusterBlank_i(pBS, nFreeCluster);
	nNextCluster = nFreeCluster;
	goto LLL_START_TO_CREATE2;


LLL_CREATE_RESULT_FAIL:
	return (FALSE);

LLL_CREATE_RESULT_SUCCESS:
	return (TRUE);
}

static bool_T dosCreateHierFile_ii(
				dosBS_Id pBS,
				bool_T bCreateDir,
				int nCluster,	/* first cluster of a directory */
				char *szFileName)
{
	int nDirEndIdx, nNameLen;
	int nDECluster, nDEOffset;
	char *pFractionalFile;
	unsigned char dirEntry[DOS_DIR_ENTRY_SIZE];
	bool_T bResult;

	/*
		    +------ nDirEndIdx = 3 (index)
		    |
		    |
		"xxxx\xxxxx\xxxxx" = szFileName
		     ||
			 ||
			 |+------ pFractionalFile (memory pointer)
			 |
			 +------- replace this character as '\0'
	*/

	while (TRUE)
	{
		nNameLen = strlen(szFileName);
		if (!dosLocateFirstPathNameIdx_ii(szFileName, &nDirEndIdx))
			return (FALSE);

		if (nDirEndIdx != nNameLen -1)
		{
			/* break the file name into 2 parts:
				dirname : pointed by szFileName
				fractional name : pointed by pFractionalFile
			*/
			szFileName[nDirEndIdx+1] = '\0';
			pFractionalFile = szFileName + nDirEndIdx + 2;
		}
		else
		{
			pFractionalFile = NULL_STR;
		}
    
		if (pFractionalFile == NULL_STR)
		{
			bResult = dosCreateFileUnderDirFirstCluster_ii(pBS,
						bCreateDir, nCluster, szFileName);
			return (bResult);
		}
		else
		{
			bResult = dosProbeFileUnderCluster_ii(pBS,
						TRUE, nCluster, szFileName, dirEntry,
						&nDECluster, &nDEOffset);
			if (!bResult)
			{
				bResult = dosCreateFileUnderDirFirstCluster_ii(pBS,
						TRUE, nCluster, szFileName);
				if (!bResult)
					return (FALSE);
    
				bResult = dosProbeFileUnderCluster_ii(pBS,
						TRUE, nCluster, szFileName, dirEntry,
						&nDECluster, &nDEOffset);
				if (!bResult)
					return (FALSE);
			}
    
			nCluster = dosGetFileFirstCluster16_ii(dirEntry);
    
    
			/* change to use a non-recursive method */
			szFileName = pFractionalFile;
		}
	}
}

static bool_T dosCreateFileUnderRD16_ii(
				dosBS_S *pBS,
				bool_T bCreateDir,	/* create a directory? */
				char *szFName,		/* file name */
				char *szExtName) 	/* file extension name */
{
	/*
	 * For FAT12/16 only.
	 */

	unsigned int nRDEntriesPerSector;
	unsigned int nRelativeEntryInSector;
	unsigned char dirEntry[DOS_DIR_ENTRY_SIZE];
	unsigned char buffer[DOS_BYTES_PER_SEC];
	unsigned char cFirstByte;
	char *pDE;
	int nFreeCluster, nIdx1;
	int nTerminalClusterId;
	bool_T bResult;
	bool_T bEmptyDEExist;


	if (pBS->p1bs.fatSystem == DOS_FAT_BITS_12)
		nTerminalClusterId = DOS_F12EC_LAST_CLUSTER_IN_FILE;
	else
		nTerminalClusterId = DOS_F16EC_LAST_CLUSTER_IN_FILE;




	nRDEntriesPerSector = DOS_BYTES_PER_SEC/DOS_DIR_ENTRY_SIZE;


	/* examine whether any file/directory with the same name
		has existed */
	bEmptyDEExist = FALSE;
	for (nIdx1 = 0; nIdx1 < pBS->p1bs.numEntriesInRD; ++nIdx1)
	{
		nRelativeEntryInSector = nIdx1 % nRDEntriesPerSector;

		if (nRelativeEntryInSector == 0)
		{
			/* read root directory entries from media to buffer */

			bResult = dosReadMediaSectorNByte_i(
						pBS,
						pBS->p1bs.nRDStartSector16 + (nIdx1/nRDEntriesPerSector),
						0,
						DOS_BYTES_PER_SEC,
						buffer);

			if (!bResult)
				return (FALSE);
		}

		pDE = (char *) (buffer + nRelativeEntryInSector * DOS_DIR_ENTRY_SIZE);

		if (*pDE == 0xE5)
		{
			bEmptyDEExist = TRUE;
			continue;
		}
		else if (*pDE == 0)
		{
			bEmptyDEExist = TRUE;
			break;
		}
		else if (sysStrNCaseCmp(szFName,   pDE, 8) == 0 && 
				 sysStrNCaseCmp(szExtName, pDE+8, 3) == 0)
			return (FALSE);
	}
	if (nIdx1 == pBS->p1bs.numEntriesInRD && !bEmptyDEExist)
	{
		/* no space for any new file in RD */

		return (FALSE);
	}



	/* make a directory entry for this file or directory */
		/*
		 * 2006.06.16
		 * For directory creation, allocate a cluster.
		 * For normal file creation, don't allocate any cluster.
		 */
	if (bCreateDir)
	{
		nFreeCluster = dosFATGetFreeCluster_i(pBS);

		if (dosFATIsNilFEC_i(pBS, nFreeCluster))
			return (FALSE);

		/* make the data area of the directory's 1st cluster as all 0's
		and create "." and ".." directory entry on it */

		dosWriteDOSDAClusterBlank_i(pBS, nFreeCluster);

		/* "." */
		dosMakeDirEntry16_ii(dirEntry, TRUE, ".",  "   ", nFreeCluster);	
		dosWriteDOSDAClusterNByte_i(pBS, &(pBS->DataCache),
			nFreeCluster, 0, DOS_DIR_ENTRY_SIZE, dirEntry);

		/* ".." */
		dosMakeDirEntry16_ii(dirEntry, TRUE, "..", "   ", 0);	
		dosWriteDOSDAClusterNByte_i(pBS, &(pBS->DataCache),
			nFreeCluster, DOS_DIR_ENTRY_SIZE, DOS_DIR_ENTRY_SIZE, dirEntry);
	}
	else
	{
		nFreeCluster = 0;
	}
	dosMakeDirEntry16_ii(dirEntry, bCreateDir, szFName, szExtName, nFreeCluster);	




	/* write out the entry to the root directory */
	for (nIdx1 = 0; nIdx1 < pBS->p1bs.numEntriesInRD; ++nIdx1)
	{
		nRelativeEntryInSector = nIdx1 % nRDEntriesPerSector;

		if (nRelativeEntryInSector == 0)
		{
			/* read root directory entries from media to buffer */

			bResult = dosReadMediaSectorNByte_i(
						pBS,
						pBS->p1bs.nRDStartSector16 + (nIdx1/nRDEntriesPerSector),
						0,
						DOS_BYTES_PER_SEC,
						buffer);

			if (!bResult)
				return (FALSE);
		}

		cFirstByte = *(buffer + nRelativeEntryInSector * DOS_DIR_ENTRY_SIZE);

		if (cFirstByte == 0x00 || (unsigned char) cFirstByte == (unsigned char) 0xE5)
		{
			dosWriteMediaSectorNByte_i(
					pBS,
					pBS->p1bs.nRDStartSector16 + nIdx1/nRDEntriesPerSector,
					(nIdx1%nRDEntriesPerSector) * DOS_DIR_ENTRY_SIZE,
					DOS_DIR_ENTRY_SIZE, dirEntry);

			if (cFirstByte == 0x00 && nIdx1+1 != pBS->p1bs.numEntriesInRD)
			{
				/* mark next directory entry as the lastest one */
				dosMarkRDLastestDirEntry16_ii(pBS, nIdx1+1);
			}

			/* update FATs */
			if (nFreeCluster)
				dosFATUpdateClusterEntry_i(pBS, nFreeCluster, nTerminalClusterId);

			return (TRUE);
		}
	}


	return (FALSE);
}


static bool_T dosProbe_ii(
				dosBS_Id pBS,	/* I */
				bool_T bDir,		/* I */
				char *szName,		/* I */
				unsigned char *pDirEntry,	/* O */
				unsigned int *pDEBlockNum,	/* O */
				unsigned int *pDEOffset)	/* O */
{
	/*
	 * The value assigned to *pDEBlockNum might be a cluster or a sector number.
	 * The caller function should interpret this number itself.
	 */

	int nNameLen;
	int nFileStartIdx, nFirstPathEndIdx;
	char szMyFName[SYS_MAX_PATH_LEN], *pszMyFNameFract;
	char szFN[64], szEN[16];
	int nClusterNum;
	bool_T bResult;

	/*		  copy		   extract
		szName --> szMyFName --> szFN + szEN

		The purpose to copy szName to szMyFName is not to modify
		szName's content.
	*/

	if (!szName || !szName[0])
		return (FALSE);

	/* prune the first and last '\' character(s) if there is any */
	nNameLen = strlen(szName);
	if (szName[nNameLen-1] == '\\')
		szName[nNameLen-1] = '\0';
	if (*szName == '\\')
		szName += 1;

	if (!strcmp("", szName))
	{
		if (bDir)
			return (TRUE);		/* root directory */
		else
			return (FALSE);
	}

	strcpy(szMyFName, szName);
	if (!dosLocateFileNameIdx_ii(szMyFName, &nFileStartIdx))  /* xxx\xxx <- */
		return (FALSE);

	if (!dosLocateFirstPathNameIdx_ii(szMyFName, &nFirstPathEndIdx)) /* -> xxx\xxx */
		return (FALSE);

	/*
	 * (1) 
	 * szName = "abc\def\xyz"
	 *
	 *  ==> szMyFName = "abc"
	 *      pszMyFNameFract = "def\xyz"
	 *      szFN = "abc"
	 *      szEN = ""
	 *
	 * (2)
	 * szName = "uvwxyz.abc"
	 *
	 *  ==> szMyFName = "uvwxyz.abc"
	 *      pszMyFNameFract = ""
	 *      szFN = "uvwsyz"
	 *      szEN = "abc"
	 */
	szMyFName[nFirstPathEndIdx + 1] = (char) 0;
	pszMyFNameFract = szMyFName + nFirstPathEndIdx + 2;

	dosExtractExtension_ii(szMyFName, szFN, szEN);


	/* probe under the first level */
	if (pBS->p1bs.fatSystem == DOS_FAT_BITS_32)
	{
		if (nFileStartIdx == 0)
		{
			return (dosProbeFileUnderCluster_ii(pBS, bDir,
					pBS->p1bs.nRDStartCluster32, szMyFName,
					pDirEntry, (int *) pDEBlockNum, (int *) pDEOffset));
		}
		else
		{
			bResult = dosProbeFileUnderCluster_ii(pBS, TRUE,
					pBS->p1bs.nRDStartCluster32, szMyFName,
					pDirEntry, (int *) pDEBlockNum, (int *) pDEOffset);
		}
	}
	else
	{
		if (nFileStartIdx == 0)
		{
			return (dosProbeFileUnderRD16_ii(pBS, bDir, szFN, szEN,
						pDirEntry, (int *) pDEBlockNum, (int *) pDEOffset));
		}
		else
		{
			bResult = dosProbeFileUnderRD16_ii(pBS, TRUE, szFN, szEN,
						pDirEntry, (int *) pDEBlockNum, (int *) pDEOffset);
		}
	}
	 
	if (!bResult)
		return (FALSE);

	if (pBS->p1bs.fatSystem == DOS_FAT_BITS_32)
		nClusterNum = dosGetFileFirstCluster32_ii(pDirEntry);
	else
		nClusterNum = dosGetFileFirstCluster16_ii(pDirEntry);

	return (dosProbeHierFile_ii(pBS, bDir, nClusterNum,
				pszMyFNameFract, pDirEntry, (int *) pDEBlockNum, (int *) pDEOffset));
}

static bool_T dosIsFileUnderRD16_ii(char *szName)
{
	/*
		"abc"      ->    nDepth = 1
		"abc\"     ->    nDepth = 1
		"\abc"     ->    nDepth = 1
		"\abc\"    ->    nDepth = 1

		"abc\d"    ->    nDepth = 2
		"\abc\d"   ->    nDepth = 2
	*/

	int idx;
	int nDepth = 0;

	if (!szName || !szName[0])
		return (TRUE);

	if (szName[0] != '\\')
		nDepth = 1;

	for (idx = 0 ; ; ++idx)
	{
		if (szName[idx] == '\0')
			break;

		if (szName[idx] == '\\' && szName[idx+1] != '\0')
		{
			/* "xxxx\x..." is encountered */

			++nDepth;

			if (nDepth >= 2)
				return (FALSE);
		}
	}

	return (TRUE);
}

static bool_T dosIsAttrROHFSFVN_ii(unsigned char *pDirEntry)
{
	register unsigned char fileAttr;

	fileAttr = *(pDirEntry+0x0b);
	if ((fileAttr & (DOS_FA_RO | DOS_FA_HF | DOS_FA_SF | DOS_FA_VN)) ==
		(DOS_FA_RO | DOS_FA_HF | DOS_FA_SF | DOS_FA_VN))
		return (TRUE);
	else
		return (FALSE);
}

static bool_T dosIsAttrVN_ii(unsigned char *pDirEntry)
{
	register unsigned char fileAttr;

	fileAttr = *(pDirEntry+0x0b);
	if ((fileAttr &  DOS_FA_VN) == DOS_FA_VN)
		return (TRUE);
	else
		return (FALSE);
}

static bool_T dosIsDir_ii(unsigned char *pDirEntry)
{
	/* TODO: check more attributes */
	unsigned char fileAttr;

	fileAttr = *(pDirEntry+0x0b);
	if (fileAttr & DOS_FA_SD)
		return (TRUE);
	else
		return (FALSE);
}

static bool_T dosIsNormalFile_ii(unsigned char *pDirEntry)
{
	/* TODO: check more attributes */
	unsigned char fileAttr;

	fileAttr = *(pDirEntry+0x0b);
	if (fileAttr & DOS_FA_SD)
		return (FALSE);
	else
		return (TRUE);
}

static int dosGetFileFirstCluster16_ii(unsigned char *pDirEntry)
{
	/* for FAT12/16 only */

	int i;

	dosvDecodeBufRoutine_g(pDirEntry+0x1A, 2, (unsigned int *) &i);
	return (i);
}

static int dosGetFileFirstCluster32_ii(unsigned char *pDirEntry)
{
	/* for FAT12/16 only */

	unsigned int nLo, nHi;

	dosvDecodeBufRoutine_g(pDirEntry+0x14, 2, &nHi);
	dosvDecodeBufRoutine_g(pDirEntry+0x1A, 2, &nLo);

	return ((nHi << 16) + nLo);
}

static int dosGetFileSize_ii(unsigned char *pDirEntry)
{
	int i;

	dosvDecodeBufRoutine_g(pDirEntry+0x1C, 4, (unsigned int *) &i);
	return (i);
}

static void dosSetFileSize_ii(unsigned char *pDirEntry, int nSize)
{
	dosvEncodeBufRoutine_g(pDirEntry+0x1C, 4, nSize);
}

static void dosGetFileName_ii(unsigned char *pDirEntry, char *buffer)
{
	unsigned short nIdx1;
	unsigned short nIdx2;

	for (nIdx1 = 0; nIdx1 < 8; ++nIdx1)
	{
		if (pDirEntry[nIdx1] == (unsigned char) ' ')
		{
			break;
		}

		buffer[nIdx1] = (char) (pDirEntry[nIdx1]);
	}

	if (pDirEntry[8] == (unsigned char) ' ')
	{
		buffer[nIdx1] = '\0';
		return;
	}
	else
	{
		buffer[nIdx1] = '.';
	}

	for (nIdx2 = 0; nIdx2 < 3; ++nIdx2)
	{
		if (pDirEntry[8+nIdx2] == (unsigned char) ' ')
		{
			buffer[nIdx1+1+nIdx2] = '\0';
			return;
		}

		buffer[nIdx1+1+nIdx2] = (char) pDirEntry[8+nIdx2];
	}

	buffer[nIdx1+1+nIdx2] = '\0';
}

static void dosMakeDirEntry16_ii(
				unsigned char *pDEntry,
				bool_T bCreateDir,
				char *szFName,
				char *szExtName,
				int nFirstCluster)
{
	/*
		directory entry content:

		start	field					size

		0x00	file name				8
		0x08	file extension			3
		0x0B	file attribute			1
		0x0C	reserved				10
		0x16	time of lastest change	2
		0x18	date of lastest change	2
		0x1A	first cluster of file	2
		0x1C	file size				4

				M             L
		attr =	S 6 5 4 3 2 1 S
				B             B
				| | | | | | | |
				| | | | | | | +-- 1 = read only (RO)
				| | | | | | +---- 1 = hidden file (HF)
				| | | | | +------ 1 = system file (SF)
				| | | | +-------- 1 = volume name (VN)
				| | | +---------- 1 = subdirectory (SD)
				| | +------------ 1 = archive bit (AB)
				| +-------------- (reserved)
				+---------------- (reserved)

		time = F E D C B A 9 8 7 6 5 4 3 2 1 0
			    -------   ---------   -------
				   |           |         |
				   |           |         +--- second/2 (0~4)
				   |           +------------- minute (5~A)
				   +------------------------- hour (B~F)


		date = F E D C B A 9 8 7 6 5 4 3 2 1 0
		        -----------   -----   -------
				     |          |        |
					 |          |        +--- day (0~4)
					 |          +------------ month (5~8)
					 +----------------------- year (9~F), from 1980
	*/

	int nIdx1;
	int nSec, nMin, nHur, nDay, nMot, nYer;
	int nLen;
	unsigned short nTime, nDate;
	unsigned char fileAttr;

	sysMemSet(pDEntry, 0, DOS_DIR_ENTRY_SIZE);


	/* field: file name and extension name */
	for (nIdx1 = 0; nIdx1 < 0x0B; ++nIdx1)
		pDEntry[nIdx1] = ' ';

	nLen = strlen(szFName);
	nLen = (nLen > 8 ? 8 : nLen);
	memcpy(pDEntry, szFName, nLen);

	if (!bCreateDir && szExtName && szExtName[0])
	{
		nLen = strlen(szExtName);
		nLen = (nLen > 3 ? 3 : nLen);
		memcpy(pDEntry+0x08, szExtName, nLen);
	}

	/* field: file attribute */
	fileAttr = 0;
	if (bCreateDir)
		fileAttr |= DOS_FA_SD;
	dosvEncodeBufRoutine_g(pDEntry+0x0B, 1, fileAttr);

	/* field: reserved */
	for (nIdx1 = 0; nIdx1 < 10; ++nIdx1)
		dosvEncodeBufRoutine_g(pDEntry+0x0C+nIdx1, 1, 0);

	/* field: time & date */
	sysCurrentTime(&nYer, &nMot, &nDay, &nHur, &nMin, &nSec);
	nTime = nSec/2 + (nMin << 5) +(nHur << 0x0B);
	nDate = nDay + (nMot << 5) + ((nYer-61) << 9);	/* TODO */
	dosvEncodeBufRoutine_g(pDEntry+0x16, 2, nTime);
	dosvEncodeBufRoutine_g(pDEntry+0x18, 2, nDate);
	
	/* field: first cluster */
	dosvEncodeBufRoutine_g(pDEntry+0x1A, 2, nFirstCluster);

	/* field: file size */
	dosvEncodeBufRoutine_g(pDEntry+0x1C, 4, 0);
}

static void dosMakeDirEntry32_ii(
				unsigned char *pDEntry,
				bool_T bCreateDir,
				char *szFName,
				char *szExtName,
				int nFirstCluster)
{
	/*
		directory entry content:

		start	field					size

		0x00	file name				8
		0x08	file extension			3
		0x0B	file attribute			1
		0x0C	reserved for Win-NT		1
		0x0D	time stamp				1
		0x0E	create time				2
		0x10	create date				2
		0x12	last access date		2
		0x14	1st cluster of file (hi) 2
		0x16	time of lastest change	2
		0x18	date of lastest change	2
		0x1A	1st cluster of file	(lo) 2
		0x1C	file size				4

				M             L
		attr =	S 6 5 4 3 2 1 S
				B             B
				| | | | | | | |
				| | | | | | | +-- 1 = read only (RO)
				| | | | | | +---- 1 = hidden file (HF)
				| | | | | +------ 1 = system file (SF)
				| | | | +-------- 1 = volume name (VN)
				| | | +---------- 1 = subdirectory (SD)
				| | +------------ 1 = archive bit (AB)
				| +-------------- (reserved)
				+---------------- (reserved)

		time = F E D C B A 9 8 7 6 5 4 3 2 1 0
			    -------   ---------   -------
				   |           |         |
				   |           |         +--- second/2 (0~4)
				   |           +------------- minute (5~A)
				   +------------------------- hour (B~F)


		date = F E D C B A 9 8 7 6 5 4 3 2 1 0
		        -----------   -----   -------
				     |          |        |
					 |          |        +--- day (0~4)
					 |          +------------ month (5~8)
					 +----------------------- year (9~F), from 1980
	*/

	int nIdx1;
	int nSec, nMin, nHur, nDay, nMot, nYer;
	int nLen;
	unsigned short nTime, nDate;
	unsigned char fileAttr;

	sysMemSet(pDEntry, 0, DOS_DIR_ENTRY_SIZE);


	/* field: file name and extension name */
	for (nIdx1 = 0; nIdx1 < 0x0B; ++nIdx1)
		pDEntry[nIdx1] = ' ';

	nLen = strlen(szFName);
	nLen = (nLen > 8 ? 8 : nLen);
	memcpy(pDEntry, szFName, nLen);

	if (!bCreateDir && szExtName && szExtName[0])
	{
		nLen = strlen(szExtName);
		nLen = (nLen > 3 ? 3 : nLen);
		memcpy(pDEntry+0x08, szExtName, nLen);
	}

	/* field: file attribute */
	fileAttr = 0;
	if (bCreateDir)
		fileAttr |= DOS_FA_SD;
	dosvEncodeBufRoutine_g(pDEntry+0x0B, 1, fileAttr);

	/* field: reserved for Windows NT (1 byte)*/
	dosvEncodeBufRoutine_g(pDEntry+0x0C, 1, 0);

	/* field: time stamp */
	dosvEncodeBufRoutine_g(pDEntry+0x0D, 1, 0);

	/* field: create time, create date, last access date */
	sysCurrentTime(&nYer, &nMot, &nDay, &nHur, &nMin, &nSec);
	nTime = nSec/2 + (nMin << 5) +(nHur << 0x0B);
	nDate = nDay + (nMot << 5) + (((nYer-1980) & 0x7F) << 9);	/* TODO */
	dosvEncodeBufRoutine_g(pDEntry+0x0E, 2, nTime);
	dosvEncodeBufRoutine_g(pDEntry+0x10, 2, nDate);
	dosvEncodeBufRoutine_g(pDEntry+0x12, 2, nDate);

	/* field: first cluster (hi) */
	dosvEncodeBufRoutine_g(pDEntry+0x14, 2, (nFirstCluster >> 16) & 0xFFFF);

	/* field: time & date */
	dosvEncodeBufRoutine_g(pDEntry+0x16, 2, nTime);
	dosvEncodeBufRoutine_g(pDEntry+0x18, 2, nDate);
	
	/* field: first cluster (lo) */
	dosvEncodeBufRoutine_g(pDEntry+0x1A, 2, nFirstCluster & 0xFFFF);

	/* field: file size */
	dosvEncodeBufRoutine_g(pDEntry+0x1C, 4, 0);
}

static void dosMakeDeletedDirEntry_ii(unsigned char *pDirEntry)
{
	sysMemSet(pDirEntry, 0, DOS_DIR_ENTRY_SIZE);
	pDirEntry[0] = (unsigned char) 0xE5;
}

static void dosSetDirEntryCluster16_ii(unsigned char *pDEntry, int nFirstCluster)
{
	dosvEncodeBufRoutine_g(pDEntry+0x1A, 2, nFirstCluster);
}

static void dosSetDirEntryCluster32_ii(unsigned char *pDEntry, int nFirstCluster)
{
	/* field: first cluster (hi) */
	dosvEncodeBufRoutine_g(pDEntry+0x14, 2, (nFirstCluster >> 16) & 0xFFFF);

	/* field: first cluster (lo) */
	dosvEncodeBufRoutine_g(pDEntry+0x1A, 2, nFirstCluster & 0xFFFF);
}


/* exposed APIs : */

static bool_T dosvFSInitted_g = FALSE;
void dosFSInitInMain(void)
{
	if (dosvFSInitted_g)
		return;

	dosGVarsInit_i();

	dosvFSInitted_g = TRUE;
}

char * dosGetMediaFS(dosBS_Id pBS)
{
	static char b[6];

	b[0] = 0;

	if (pBS && pBS->bValid)
	{
		switch (pBS->p1bs.fatSystem)
		{
		case DOS_FAT_BITS_12:
			strcpy(b, "FAT12");
			break;

		case DOS_FAT_BITS_16:
			strcpy(b, "FAT16");
			break;

		case DOS_FAT_BITS_32:
			strcpy(b, "FAT32");
			break;
		}
	}

	return b;
}

unsigned int dosGetMediaCapacity(dosBS_Id pBS)
{
	return ((pBS->io.getCapacity)());
}

unsigned int dosGetMediaDataStorage(dosBS_Id pBS)
{
	/* The media's storage means the capacity which can
		be used to store the file's content. That is to
		say that the size of the "DATA area".
	*/

	if (pBS)
		return (pBS->p1bs.numPhyEntriesPerFAT * pBS->p1bs.bytesPerCluster);
	else
		return (0);
}

unsigned int dosGetMediaFreeDataStorage(dosBS_Id pBS)
{
	if (pBS)
		return (pBS->p1bs.numFreeFATEntries * pBS->p1bs.bytesPerCluster);
	else
		return (0);
}

bool_T dosFileExist(dosBS_Id pBS, char *szName, bool_T bDirectory)
{
	unsigned char dE[DOS_DIR_ENTRY_SIZE];
	unsigned int nDummy;

	if (!pBS || !(pBS->bValid) || !dosIsValidPath(szName))
		return (FALSE);

	sysASCIIToUpper(szName);

	return (dosProbe_ii(pBS, bDirectory, szName, dE, &nDummy, &nDummy));
}

bool_T dosCreateFile(dosBS_Id pBS, char *szName, bool_T bDirectory)
{
	int nNameLen;
	int nFileStartIdx, nFirstPathEndIdx;
	bool_T bResult;
	char szMyFName[SYS_MAX_PATH_LEN], *pszMyFNameFract;
	unsigned char pDirEntry[DOS_DIR_ENTRY_SIZE];
	int nClusterNum;
	unsigned int nDummy;
	

	/*		  copy		   extract
		szName --> szMyFName --> szFN + szEN

		The purpose to copy szName to szMyFName is not to modify
		szName's content.
	*/

	/* 
	 * It's not needed to check whether or not the file exists
	 * or not before creating it because the functions called
	 * by this function to write data to the disk will check
	 * the job.
	 */


	/* examine the arguments */
	if (!pBS)
	{
		return (FALSE);
	}
	
	if (!szName || !szName[0] || !dosIsValidPath(szName))
	{
		return (FALSE);
	}


	/* prune the first and lastest '\' character(s) if there is any */
	nNameLen = strlen(szName);
	if (nNameLen == 1 && szName[0] == '\\')
	{
		return (FALSE);
	}
	if (szName[nNameLen-1] == '\\')
	{
		szName[nNameLen-1] = '\0';
	}
	if (*szName == '\\')
	{
		szName += 1;
	}


	/* copy the file name to another buffer */
	strcpy(szMyFName, szName);
	sysASCIIToUpper(szMyFName);

	if (!dosLocateFileNameIdx_ii(szMyFName, &nFileStartIdx)) /* xxx\xxx <- */
		return (FALSE);

	if (!dosLocateFirstPathNameIdx_ii(szMyFName, &nFirstPathEndIdx)) /* -> xxx\xxx */
		return (FALSE);

	/*
	 * (1) 
	 * szName = "abc\def\xyz"
	 *
	 *  ==> szMyFName = "abc"
	 *      pszMyFNameFract = "def\xyz"
	 *      szFN = "abc"
	 *      szEN = ""
	 *
	 * (2)
	 * szName = "uvwxyz.abc"
	 *
	 *  ==> szMyFName = "uvwxyz.abc"
	 *      pszMyFNameFract = ""
	 *      szFN = "uvwsyz"
	 *      szEN = "abc"
	 */
	szMyFName[nFirstPathEndIdx + 1] = (char) 0;
	pszMyFNameFract = szMyFName + nFirstPathEndIdx + 2;


	/* probe under the first level */
	if (nFileStartIdx == 0)
		bResult = dosProbe_ii(pBS, bDirectory, szMyFName, pDirEntry, &nDummy, &nDummy);
	else
		bResult = dosProbe_ii(pBS, TRUE, szMyFName, pDirEntry, &nDummy, &nDummy);


	/* need to create under the first level? */
	if (!bResult)
	{
		/* not found the 1st level; create the first level */

		if (pBS->p1bs.fatSystem == DOS_FAT_BITS_32)
		{
			bResult = dosCreateFileUnderDirFirstCluster_ii(pBS,
						!nFileStartIdx ? bDirectory : TRUE,
						pBS->p1bs.nRDStartCluster32, szMyFName);
		}
		else
		{
			char szFN[9], szEN[4];

			dosExtractExtension_ii(szMyFName, szFN, szEN);
			bResult = dosCreateFileUnderRD16_ii(pBS,
						!nFileStartIdx ? bDirectory : TRUE,
						szFN, szEN);
		}


		/* probe under the first level again */
		if (nFileStartIdx == 0)
			bResult = dosProbe_ii(pBS, bDirectory, szMyFName, pDirEntry, &nDummy, &nDummy);
		else
			bResult = dosProbe_ii(pBS, TRUE, szMyFName, pDirEntry, &nDummy, &nDummy);
	}


	if (nFileStartIdx && bResult)
	{
		if (pBS->p1bs.fatSystem == DOS_FAT_BITS_32)
			nClusterNum = dosGetFileFirstCluster32_ii(pDirEntry);
		else
			nClusterNum = dosGetFileFirstCluster16_ii(pDirEntry);

		bResult = dosCreateHierFile_ii(pBS,
						bDirectory, nClusterNum, pszMyFNameFract);
	}
	

	/* force to update the FAT */
	dosFATCacheFlush_i(pBS);

	dosFlushCache_i(pBS, &(pBS->DataCache));
	return (bResult);
}

static bool_T dosUnlinkFile_ii(dosBS_Id pBS, char *szName, bool_T bIsDir)
{
	bool_T bResult, bIsInRD16;
	int nDEBlockNum;
	int nDECluster;	
	int nDESector;
	int nDEOffset;
		/* either (nDECluster, nDEOffset) or (nDESector, nDEOffset) is used */

	int nFirstCluster;
	unsigned char dE[DOS_DIR_ENTRY_SIZE];


	sysASCIIToUpper(szName);

	bResult = dosProbe_ii(pBS, bIsDir, szName, dE,
							(unsigned int *) &nDEBlockNum,
							(unsigned int *) &nDEOffset);
	if (!bResult)
		return (FALSE);


	if (pBS->p1bs.fatSystem == DOS_FAT_BITS_32)
	{
		nFirstCluster = dosGetFileFirstCluster32_ii(dE);
		bIsInRD16 = FALSE;
	}
	else
	{
		nFirstCluster = dosGetFileFirstCluster16_ii(dE);
		bIsInRD16 = dosIsFileUnderRD16_ii(szName);
	}


	/* update FAT; recycle clusters */
	if (nFirstCluster)
		dosFATRecycleFileClusters_i(pBS, nFirstCluster);


	/* update directory entry */
	dosMakeDeletedDirEntry_ii(dE);
	if (bIsInRD16)
	{
		nDESector = nDEBlockNum; /* block num is in fact a sector num */
		dosWriteMediaSectorNByte_i(pBS, nDESector,
			nDEOffset, DOS_DIR_ENTRY_SIZE, dE);
	}
	else
	{
		nDECluster = nDEBlockNum; /* block num is in fact a cluster num */
		dosWriteDOSDAClusterNByte_Flush_i(pBS, &(pBS->DataCache),
			nDECluster,
			nDEOffset, DOS_DIR_ENTRY_SIZE, dE);
	}


	//flush FAT
	dosFATCacheFlush_i(pBS);

	//flush data
	dosFlushCache_i(pBS, &(pBS->DataCache));

	return (TRUE);
}

bool_T dosUnlinkFile(dosBS_Id pBS, char *szName)
{
	if (!pBS || !pBS->bValid || !dosIsValidPath(szName))
		return (FALSE);

	return (dosUnlinkFile_ii(pBS, szName, FALSE));
}

bool_T dosUnlinkDir(dosBS_Id pBS, char *szName)
{
	dosIter_Id iterId;
	dosIterInfo_S *pInfo;
	char *szTempName;
	char szIteredFileName[256];
	bool_T b1, b2;

	if (!pBS || !pBS->bValid || !dosIsValidPath(szName))
		return (FALSE);

	if (!dosFileExist(pBS, szName, TRUE))
		return (FALSE);

	iterId = dosStartToIterFile(pBS, szName, doscIterBoth);
	while (pInfo = dosIterOneFile(iterId))
	{
		if (strcmp(".", pInfo->szName) == 0 || strcmp("..", pInfo->szName) == 0)
			continue;

		strcpy(szIteredFileName, pInfo->szName);

		szTempName = dosBeautyFileName(szName, szIteredFileName);
		/*
		NOTE: szTempName stores the full name of the iterated file.
		*/

		b1 = dosFileExist(pBS, szTempName, TRUE);
		b2 = dosFileExist(pBS, szTempName, FALSE);

		if (b1 || b2)
		{
			/* the directory isn't empty */
			dosStopIterFile(iterId);
			return (FALSE);
		}
	}
	dosStopIterFile(iterId);

	return (dosUnlinkFile_ii(pBS, szName, TRUE));
}

dosFile_Id dosOpenFile(dosBS_Id pBS, char *szName)
{
	unsigned int nDEBlockNum, nDEOffset;
	unsigned char dirEntry[DOS_DIR_ENTRY_SIZE];
	dosFile_Id fId;
	int nFirstCluster;
	//int nRank;
	char szMyName[64];

	strcpy(szMyName, szName);
	sysASCIIToUpper(szMyName);

	if (!dosProbe_ii(pBS, FALSE, szMyName, dirEntry, &nDEBlockNum, &nDEOffset) ||
		!dosIsValidPath(szName))
		return (NULL);

#if 1
	fId = (dosFile_Id) sysMalloc(sizeof(dosFile_S));
	if (!fId)
		return (NULL);
#else	//added by ted for alignment buffer to speed-up SDC read/write performance
	void *temp;
	fId = (dosFile_Id) sysMalloc(sizeof(dosFile_S)+0x20);
	if (!fId)
		return (NULL);
	temp = (void *)fId;
	fId = (dosFile_Id)(((int)fId + 0x10) & 0xFFFFFFF0);
	fId->temp = temp;
#endif
	sysMemSet(fId, 0, sizeof(dosFile_S));
	

	if (pBS->p1bs.fatSystem != DOS_FAT_BITS_32)
		fId->bIsInRootDir = dosIsFileUnderRD16_ii(szMyName);
	else
		fId->bIsInRootDir = FALSE;


	if (fId->bIsInRootDir)
	{
		fId->nDESector = nDEBlockNum;
		fId->nDEOffset = nDEOffset;
	}
	else
	{
		fId->nDECluster = nDEBlockNum;
		fId->nDEOffset  = nDEOffset;
	}


	memcpy(fId->dirEntry, dirEntry, 32);

	fId->pBS = pBS;
	fId->nFileSize = dosGetFileSize_ii(dirEntry);
	fId->nFilePtrPos = 0;


	if (pBS->p1bs.fatSystem == DOS_FAT_BITS_32)
		nFirstCluster = dosGetFileFirstCluster32_ii(dirEntry);
	else
		nFirstCluster = dosGetFileFirstCluster16_ii(dirEntry);

#if 0
	/*
	 * 2006.06.16
	 * Don't allocate any cluster for empty files.
	 * If this file is "open for write", allocate the cluster
	 * at the write function.
	 */
	if (nFirstCluster == 0)
	{
		int nTerminalClusterId;

		/* This means that there is no cluster assigned to this file. */
		/* We prepare a cluster for it in our program */

		/* ask for a free cluster */
		nFirstCluster = dosFATGetFreeCluster_i(fId->pBS); 
		if (nFirstCluster == 0)
		{
			/* No free cluster is available! */
			sysFree(fId);
			return (NULL);
		}

		if (fId->pBS->p1bs.fatSystem == DOS_FAT_BITS_12)
			nTerminalClusterId = DOS_F12EC_LAST_CLUSTER_IN_FILE;
		else
			nTerminalClusterId = DOS_F16EC_LAST_CLUSTER_IN_FILE;

		/* update FAT table */
		dosFATUpdateClusterEntry_i(fId->pBS, 
					nFirstCluster, nTerminalClusterId);

		dosSetDirEntryCluster16_ii(fId->dirEntry, nFirstCluster);
	}
#endif

	fId->nBytesPerCluster = pBS->p1bs.bytesPerCluster;
	fId->nFileHeadCluster = nFirstCluster;
	fId->nFileNThCluster = nFirstCluster;
	fId->nClusterRank = 0;

	return (fId);
}

void dosCloseFile(dosFile_Id fId)
{
	if (!fId || !fId->pBS)
		return;


	dosFATCacheFlush_i(fId->pBS);


	/* correct the file size and update the directory entry */
	if (dosGetFileSize_ii(fId->dirEntry) != fId->nFileSize)
	{
		dosSetFileSize_ii(fId->dirEntry, fId->nFileSize);
		if (fId->bIsInRootDir)
		{
			dosWriteMediaSectorNByte_i(fId->pBS,
				fId->nDESector, fId->nDEOffset, DOS_DIR_ENTRY_SIZE, fId->dirEntry);
		}
		else
		{
			dosWriteDOSDAClusterNByte_i(fId->pBS, &(fId->pBS->DataCache),
				fId->nDECluster, fId->nDEOffset, DOS_DIR_ENTRY_SIZE, fId->dirEntry);
		}
	}

	/*
	 * TODO : Check whether the file size is 0 or not. If 0, "recycle" clusters
	 * 		  engaged by this file.
	 */

	dosFlushCache_i(fId->pBS, &(fId->pBS->DataCache));
	dosFlushCache_i(fId->pBS, &(fId->DataCache));

#if 1
	sysMemSet(fId, 0, sizeof(dosFile_S));
	sysFree(fId);
#else	//added by ted for alignment buffer to speed-up SDC read/write performance
	void *temp;
	temp = fId->temp;
	sysMemSet(temp, 0, sizeof(dosFile_S)+0x20);
	sysFree(temp);
#endif
}

int dosWriteFile(dosFile_Id fId, char *buffer, int nSize)
{
	int nTerminalClusterId;
	int nOffWithinCluster;
	int nWriteSizeThisRound;
	int nCurWriteSize;
	int nClusterFreeSize;


	if (!fId || !nSize)
		return (0);

	switch (fId->pBS->p1bs.fatSystem)
	{
	case DOS_FAT_BITS_12:
		nTerminalClusterId = DOS_F12EC_LAST_CLUSTER_IN_FILE;
		break;

	case DOS_FAT_BITS_16:
		nTerminalClusterId = DOS_F16EC_LAST_CLUSTER_IN_FILE;
		break;

	case DOS_FAT_BITS_32:
		nTerminalClusterId = DOS_F32EC_LAST_CLUSTER_IN_FILE;
		break;
	}


	/* empty file? */
	if (!fId->nFileSize && !fId->nFileHeadCluster)
	{
		/* This means that there is no cluster assigned to this file. */

		/* ask for a free cluster */
		fId->nFileHeadCluster = dosFATGetFreeCluster_i(fId->pBS); 
		if (!fId->nFileHeadCluster)
		{
			/* No free cluster is available! */
			//sysFree(fId); 2006/07/05
			return (0);
		}

		fId->nFileNThCluster = fId->nFileHeadCluster;
		fId->nClusterRank = 0;

		/* update FAT table */
		dosFATUpdateClusterEntry_i(fId->pBS, 
					fId->nFileHeadCluster, nTerminalClusterId);

		/* modify the directory entry */
		if (fId->pBS->p1bs.fatSystem == DOS_FAT_BITS_32)
			dosSetDirEntryCluster32_ii(fId->dirEntry, fId->nFileHeadCluster);
		else
			dosSetDirEntryCluster16_ii(fId->dirEntry, fId->nFileHeadCluster);
	}


	if (fId->nFileSize == fId->nFilePtrPos &&
		fId->nFileSize % fId->nBytesPerCluster == 0)
	{
		/* left the chance to sync Ptr in the following for loop ---+
		 *                                                          |
		 * for (...)                                                |
		 * {                                                        |
		 *    if (fId->nFilePtrPos < fId->nFileSize ...)            |
		 *    {                                                     |
		 *     ...                                                  |
		 *    }                                                     |
		 *    else                                                  |
		 *    {                                                     |
		 *     int nNewCluster;                                     |
		 *     ...     <--------------------------------------------+
		 *       (need to acquire extra cluster before "sync")
		 *    }
		 * }
		 */
	}
	else
	{
		dosSyncClusterByFilePtrPos_ii((dosFile_Id) fId);
		/* need to sync since the AP might seek the file pointer's position */
	}

	for (nCurWriteSize = 0; nCurWriteSize < nSize; )
	{
		/*
		 * In each ROUND of this for loop, data is written into the card
		 * only within one cluster. The program never updates data between
		 * 2 clusters in 1 ROUND.
		 */

		nOffWithinCluster = fId->nFilePtrPos % fId->nBytesPerCluster;

		sysDAssert(fId->nFilePtrPos <= fId->nFileSize);

		if (fId->nFilePtrPos < fId->nFileSize ||
			((fId->nFilePtrPos == fId->nFileSize) && nOffWithinCluster != 0))
		{
LLL_WRITE:

			/* calculate free size in this cluster */
			nClusterFreeSize = fId->nBytesPerCluster - nOffWithinCluster;

			/* calculate write size in this round */
			nWriteSizeThisRound = nSize - nCurWriteSize;
			nWriteSizeThisRound = nWriteSizeThisRound >= nClusterFreeSize ?
								  nClusterFreeSize : nWriteSizeThisRound;

			/* write data */
			dosWriteDOSDAClusterNByte_i(fId->pBS,
										&(fId->DataCache),
										fId->nFileNThCluster,
										nOffWithinCluster,
										nWriteSizeThisRound,
										(unsigned char *) buffer + nCurWriteSize);
			nCurWriteSize += nWriteSizeThisRound;

			/* update file pointer and file size */
			fId->nFilePtrPos += nWriteSizeThisRound;
			if (fId->nFilePtrPos > fId->nFileSize)
			{
				fId->nFileSize += ((((fId->nFilePtrPos-1) % fId->nBytesPerCluster)+1) -
								   (fId->nFileSize % fId->nBytesPerCluster));
			}

			/* sync. the cluster */
			if (fId->nFilePtrPos < fId->nFileSize &&
				fId->nFilePtrPos % fId->nBytesPerCluster == 0)
			{
				/* After this round, the file pointer position goes to the next cluster. */

				fId->nFileNThCluster = dosFATGetNextCluster_i(fId->pBS,
							fId->nFileNThCluster);
				fId->nClusterRank += 1;
			}
		}
		else
		{
			int nNewCluster;

			/* (fId->nFilePtrPos == fId->nFileSize) && nOffWithinCluster == 0 */

			//
			//  don't check this
			//
			//sysDAssert(dosFATIsNilFEC_i(fId->pBS, dosFATGetFEC_i(fId->pBS, fId->nFileNThCluster)));
			//
			//

			if (fId->nFileSize == 0)
			{
				goto LLL_WRITE;  /* In this function, a cluster has been
									allocated for this file in the very
									beginning of this function. */
			}


			nNewCluster = dosFATGetNextCluster_i(fId->pBS, fId->nFileNThCluster);
			if (dosFATIsNilFEC_i(fId->pBS, nNewCluster))
			{
				/* need to allocate an extra cluster for this file */
                
				nNewCluster = dosFATGetFreeCluster_i(fId->pBS);
                
				if (nNewCluster == 0)
					return (nCurWriteSize);
                
				/* update FAT table */
				dosFATUpdateClusterEntry_i(fId->pBS, 
							fId->nFileNThCluster, nNewCluster);
				dosFATUpdateClusterEntry_i(fId->pBS, 
							nNewCluster, nTerminalClusterId);
			}


			fId->nFileNThCluster = nNewCluster;
			fId->nClusterRank += 1;

			goto LLL_WRITE;
		}
	}

	return (nCurWriteSize);
}

int dosFileSize(dosBS_Id pBS, char *szName)
{
	unsigned char dE[DOS_DIR_ENTRY_SIZE];
	unsigned int nDummy;

	if (!(pBS->bValid) || !dosIsValidPath(szName))
		return (0);

	sysASCIIToUpper(szName);

	if (!dosProbe_ii(pBS, FALSE, szName, dE, &nDummy, &nDummy))
		return (0);

	return (dosGetFileSize_ii(dE));
}

bool_T dosFormatMedia(dosMediaIORoutine_S ioRoutine)
{
	/*
	 * TODO : Format the media card as FAT32. In this function,
	 *        media card is only formatted as FAT only.
	 */

	dosBS_S BS;
	unsigned char buffer[DOS_BYTES_PER_SEC];
	unsigned int nBytesEncoded;
	unsigned int nIdx1, nIdx2, nNum1;


	if (!ioRoutine.getCapacity || !ioRoutine.readNSector ||
		!ioRoutine.writeNSector || (ioRoutine.getCapacity)() < 16000000)
	{
		return (FALSE);
	}

	sysMemSet(&BS, 0, sizeof(dosBS_S));
	BS.io = ioRoutine;
	dosSetFmtInfo_i(&BS, (ioRoutine.getCapacity)());

	BS.DataCache.isValid = FALSE;
	BS.DataCache.isDirty = FALSE;
	BS.p1bs.FATCache.isValid = FALSE;
	BS.p1bs.FATCache.isDirty = FALSE;

    /* (1) write master boot sector, 512 bytes */

    sysMemSet(buffer, 0, DOS_BYTES_PER_SEC);

    nBytesEncoded = 0x1BE;

    dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 1, BS.mbs.p1BootId);
    nBytesEncoded += 1;

    dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 1, BS.mbs.p1StartHeadNum);
    nBytesEncoded += 1;

    dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 1, BS.mbs.p1StartSectorNum);
    nBytesEncoded += 1;

    dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 1, BS.mbs.p1StartCylinderNum);
    nBytesEncoded += 1;

    dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 1, BS.mbs.p1SystemId);
    nBytesEncoded += 1;

    dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 1, BS.mbs.p1EndHeadNum);
    nBytesEncoded += 1;

    dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 1, BS.mbs.p1EndSectorNum);
    nBytesEncoded += 1;

    dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 1, BS.mbs.p1EndCylinderNum);
    nBytesEncoded += 1;

    dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 4, BS.mbs.p1StartLogSectorNum);
    nBytesEncoded += 4;

    dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 4, BS.mbs.p1Size);
    nBytesEncoded += 4;


	//buffer[510] = BS.mbs.fixedData[0];
	//buffer[511] = BS.mbs.fixedData[1];
    sysMemCpy(buffer+510, BS.mbs.fixedData, 2);

    dosWriteMediaSectorNByte_i(&BS, 0, 0, DOS_BYTES_PER_SEC, buffer);


    /* (2) write partition-1 information */

    /* (2.1) write boot sector, 512 bytes */

    setBootSector((unsigned int *)buffer);

    nBytesEncoded = 0;

	//dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 3, BS.p1bs.jumpCmd);
    nBytesEncoded += 3;

	//dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 8, BS.p1bs.manufacture);
    nBytesEncoded += 8;

    dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 2, BS.p1bs.bytesPerSector);
    nBytesEncoded += 2;

    dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 1, BS.p1bs.sectorsPerCluster);
    nBytesEncoded += 1;

    dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 2, BS.p1bs.numReservedSectors);
    nBytesEncoded += 2; 

    dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 1, BS.p1bs.numFATs);
    nBytesEncoded += 1; 

    dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 2, BS.p1bs.numEntriesInRD);
    nBytesEncoded += 2; 

    dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 2, BS.p1bs.numSectorsInVolume);
    nBytesEncoded += 2; 

	//dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 1, BS.p1bs.mediaDescriptor);
    nBytesEncoded += 1; 

    dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 2, BS.p1bs.sectorsPerFAT);
    nBytesEncoded += 2;

	//dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 2, BS.p1bs.sectorsPerTrack);
    nBytesEncoded += 2;

	//dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 2, BS.p1bs.numRWHeads);
    nBytesEncoded += 2;

    dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 4, BS.p1bs.numHiddenSectors);
    nBytesEncoded += 4;

    dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 4, BS.p1bs.nNumPartSecs);
    nBytesEncoded += 4;

	//dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 1, BS.p1bs.nPhyDriveNum);
    nBytesEncoded += 1;

	//dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 1, BS.p1bs.reserved_1);
    nBytesEncoded += 1;

	//dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 1, BS.p1bs.extBootRecSignature);
    nBytesEncoded += 1;

	//dosvEncodeBufRoutine_g(buffer+nBytesEncoded, 4, BS.p1bs.volumeId);
    nBytesEncoded += 4;

    sysMemCpy(buffer+nBytesEncoded, BS.p1bs.volumeLabel, 11);
    nBytesEncoded += 11;

    sysMemCpy(buffer+nBytesEncoded, BS.p1bs.fileSystemType, 8);
    nBytesEncoded += 8;

    sysMemCpy(buffer+510, BS.p1bs.fixedData, 2);


    dosWriteMediaSectorNByte_i(&BS, BS.mbs.p1StartLogSectorNum,
            0, DOS_BYTES_PER_SEC, buffer);



    /* (2) write FAT(s) to media */

    sysMemSet(buffer, 0, DOS_BYTES_PER_SEC);
    for (nIdx1 = 0; nIdx1 < BS.p1bs.numFATs; ++nIdx1)
    {
        buffer[0] = 0xff;
        buffer[1] = 0xff;
        buffer[2] = 0xff;
        buffer[3] = 0xff;

        for (nIdx2 = 0; nIdx2 < BS.p1bs.sectorsPerFAT; ++nIdx2)
        {
            dosWriteMediaSectorBlank_i(&BS, BS.p1bs.numHiddenSectors + BS.p1bs.numReservedSectors +
                            nIdx1*BS.p1bs.sectorsPerFAT + nIdx2);

            if (nIdx2 == 0) {
                dosWriteMediaSectorNByte_i(&BS, BS.p1bs.numHiddenSectors + BS.p1bs.numReservedSectors +
                                nIdx1*BS.p1bs.sectorsPerFAT + nIdx2,    0, 4, buffer); 
            }
        }
    }


    /*
    nNum1 : number of clusters engaged by system 
    nNum2 : FAT entry content, either 12bit or 16bit
    */


    /* (3) clean root directory (RD) area */
    nNum1 = BS.p1bs.numEntriesInRD / (BS.p1bs.bytesPerSector/DOS_DIR_ENTRY_SIZE);

    for (nIdx1 = 0; nIdx1 < nNum1; ++nIdx1)
        dosWriteMediaSectorBlank_i(&BS, nIdx1 + BS.p1bs.nRDStartSector16);


    nNum1 = (BS.mbs.p1Size < 0x10000) ? BS.p1bs.numSectorsInVolume : BS.p1bs.nNumPartSecs;
    dosWriteMediaSectorBlank_i(&BS, nNum1);


	dosFlushCache_i(&BS, &(BS.DataCache));

    return (TRUE);
}

dosBS_Id dosInstallMedia(dosMediaIORoutine_S ioRoutine)
{
	dosBS_Id pBS;

	if (!ioRoutine.getCapacity || !ioRoutine.readNSector ||
		!ioRoutine.writeNSector)
	{
		return ((dosBS_Id) 0);
	}

	pBS = (dosBS_Id) sysMalloc(sizeof(dosBS_S));
	if (!pBS)
		return ((dosBS_Id) 0);
	sysMemSet(pBS, 0, sizeof(dosBS_S));

	pBS->io = ioRoutine;

	if (dosReadMediaBootSector_i(pBS) != doscMediaFormatted)
	{
		/* destroy the allocated memory */

		sysMemSet(pBS, 0, sizeof(dosBS_S));
		sysFree(pBS);
		pBS = (dosBS_Id) 0;

		return (pBS);
	}

	dosFATCountFreeEntries_i(pBS);

	return (pBS);
}

bool_T dosUninstallMedia(dosBS_Id pBS)
{
	if (pBS)
	{
		if (pBS->io.uninstall)
			(pBS->io.uninstall)();

		sysMemSet(pBS, 0, sizeof(dosBS_S));
		sysFree(pBS);
	}

	return (TRUE);
}

dosMediaIORoutine_S dosGetIORoutine(dosBS_Id pBS)
{
	dosMediaIORoutine_S io;

	if (pBS)
		return (pBS->io);
	else
	{
		sysMemSet(&io, 0, sizeof(dosMediaIORoutine_S));
		return (io);
	}
}

dosDirIter_S * dosOpenDirForIter_ii(dosBS_Id pBS, char *szName)
{
	unsigned char dirEntry[DOS_DIR_ENTRY_SIZE];
	dosDirIter_S *fDir;
	int nIdx;
	int nCurrentCluster, nFirstCluster;
	unsigned int nDummy;

	/* find the directory's 1st cluster */
	if (pBS->p1bs.fatSystem == DOS_FAT_BITS_32 && 
		(!szName || !szName[0] || !strcmp(szName, "\\")))
	{
		nFirstCluster = pBS->p1bs.nRDStartCluster32;
	}
	else
	{
		if (!dosProbe_ii(pBS, TRUE, szName, dirEntry, &nDummy, &nDummy))
			return (NULL);
    
		if (pBS->p1bs.fatSystem == DOS_FAT_BITS_32)
			nFirstCluster = dosGetFileFirstCluster32_ii(dirEntry);
		else
			nFirstCluster = dosGetFileFirstCluster16_ii(dirEntry);
	}


	fDir = (dosDirIter_S *) sysMalloc(sizeof(dosDirIter_S));
	if (!fDir)
		return (NULL);
	sysMemSet(fDir, 0, sizeof(dosDirIter_S));


	/*
	 * initialize the structure for dosReadFile()
	 */

	fDir->pBS = pBS;
	fDir->nBytesPerCluster = pBS->p1bs.bytesPerCluster;
	fDir->nFilePtrPos = 0;
	fDir->nFileHeadCluster = nFirstCluster;
	fDir->nFileNThCluster = nFirstCluster;
	fDir->nClusterRank = 0;
		/* If there are 10 clusters belonging to this directory, 
			cluster ranks arrange from 0 to 9. That is to say that
			rank 0 is the "first" one. */


	/* calcuate the "file size" of the directory */
	fDir->nFileSize = 0;
	for (nIdx = 0, nCurrentCluster = nFirstCluster; ; ++nIdx)
	{
		if (dosFATIsNilFEC_i(pBS, nCurrentCluster))
			break;
		else
			fDir->nFileSize += fDir->nBytesPerCluster;

		nCurrentCluster = dosFATGetNextCluster_i(pBS, nCurrentCluster);
	}

	return (fDir);
}

void dosCloseDirForIter_ii(dosDirIter_S *fId)
{
	sysFree(fId);
}

static void dosSyncClusterByFilePtrPos_ii(dosFile_Id fId)
{
	int nRank;
	int nIdx;

	/*
		access:
			fId->nFileNThCluster
			fId->nClusterRank
	*/

	nRank = fId->nFilePtrPos/fId->nBytesPerCluster;
	if (nRank == fId->nClusterRank)
	{
		/* perfect; nothing to do */
	}
	else if (nRank > fId->nClusterRank)
	{
		/* the AP seeks the file pointer forward or the dos
			changes to next cluster to read the file */
		for (nIdx = 0; nIdx < nRank - fId->nClusterRank; ++nIdx)
		{
			fId->nFileNThCluster = dosFATGetNextCluster_i(fId->pBS,
							fId->nFileNThCluster);
		}
		fId->nClusterRank = nRank;
	}
	else
	{
		/* the AP seek the file pointer back */

		fId->nFileNThCluster = fId->nFileHeadCluster;

		if (nRank == 0)
		{
			;
		}
		else
		{
			for (nIdx = 0; nIdx < nRank; ++nIdx)
			{
				fId->nFileNThCluster = dosFATGetNextCluster_i(fId->pBS,
							fId->nFileNThCluster);
			}
		}

		fId->nClusterRank = nRank;
	}
}

int dosReadFile(dosFile_Id fId, int nReadSize, char *buffer)
{
	int nBytesRead;	/* indicating how many bytes have been read already */
	int nCluster, nOffset;



	if (!fId || fId->nFileSize <= 0 || fId->nFileHeadCluster == 0)
	{
		/* NOTE : Window system doesn't necessary allocate any cluster
					for an empty file. That is to say that dosFS shall
					be able to open an empty file. While an empty file
					is opened, nothing could be read.
		*/

		return (0);
	}


	if (fId->nFilePtrPos < 0 || fId->nFilePtrPos >= fId->nFileSize)
		return (0);

	if (fId->nFilePtrPos + nReadSize > fId->nFileSize)
	{
		nReadSize = fId->nFileSize - fId->nFilePtrPos;
	}


	for (nBytesRead = 0; nBytesRead < nReadSize; )
	{
		dosSyncClusterByFilePtrPos_ii(fId);

		nCluster = fId->nFileNThCluster;
		nOffset = fId->nFilePtrPos%fId->nBytesPerCluster;

		if (nOffset != 0)
		{
			/* Can be totally read by only one read? */
			if (fId->nBytesPerCluster - nOffset >= nReadSize)
			{/*Yes!*/
				if (dosReadDOSDAClusterNByte_i(fId->pBS, &(fId->DataCache),
						nCluster, nOffset,
						nReadSize, (unsigned char *) (buffer+nBytesRead)))
				{
					fId->nFilePtrPos += nReadSize;
					return (nReadSize);	
				}
				else
				{
					return (nBytesRead);
				}
			}
			else
			{/*No!*/
				if (dosReadDOSDAClusterNByte_i(fId->pBS, &(fId->DataCache),
						nCluster, nOffset,
						fId->nBytesPerCluster-nOffset,
						(unsigned char *) (buffer+nBytesRead)))
				{
					nBytesRead += (fId->nBytesPerCluster-nOffset);
					fId->nFilePtrPos += (fId->nBytesPerCluster-nOffset);
				}
				else
				{
					return (nBytesRead);	
				}
			}
		}
		else
		{
			/* Shall read a whole cluster? */
			if (nReadSize - nBytesRead >= fId->nBytesPerCluster)
			{/*Yes?*/
				if (dosReadDOSDAClusterNByte_i(fId->pBS, &(fId->DataCache),
						nCluster, 0,
						fId->nBytesPerCluster,
						(unsigned char *) (buffer+nBytesRead)))
				{
					nBytesRead += fId->nBytesPerCluster;
					fId->nFilePtrPos += fId->nBytesPerCluster;
				}
				else
				{
					return (nBytesRead);	
				}
			}
			else
			{/*No!*/
				if (dosReadDOSDAClusterNByte_i(fId->pBS, &(fId->DataCache),
						nCluster, 0,
						nReadSize - nBytesRead,
						(unsigned char *) (buffer+nBytesRead)))
				{
					fId->nFilePtrPos += (nReadSize - nBytesRead);
					nBytesRead = nReadSize;
				}
				else
				{
					return (nBytesRead);	
				}
			}
		}
	}
	
	return (nBytesRead);
}

int dosFileSeek(dosFile_Id fId, int offset, int origin)
{
#define FAIL_TO_SEEK (-1)
	int nNewPosition;

	if (!fId)
		return (FAIL_TO_SEEK);

	switch (origin)
	{
	case DOS_SEEK_CUR:
		nNewPosition = fId->nFilePtrPos + offset;
		break;

	case DOS_SEEK_END:
		nNewPosition = fId->nFileSize - offset;
		break;

	case DOS_SEEK_SET:
		nNewPosition = offset;
		break;

	default:
		return (FAIL_TO_SEEK);
	}

	if (nNewPosition < 0 || nNewPosition >= fId->nFileSize)
		return (FAIL_TO_SEEK);

	fId->nFilePtrPos = nNewPosition;
	return (0); /* 0 means successful */
#undef FAIL_TO_SEEK
}

int dosFileTell(dosFile_Id fId)
{
#define ERROR_FILE_POSITION	(-1)

	if (!fId)
		return (ERROR_FILE_POSITION);

	return (fId->nFilePtrPos);
#undef  ERROR_FILE_POSITION
}

bool_T dosFileEOF(dosFile_Id fId)
{
	if (!fId)
		return (TRUE);

	return (fId->nFileSize <= fId->nFilePtrPos);
}

dosIter_Id dosStartToIterFile(dosBS_Id pBS, char *szDirName,
			dosIterType_E nIterType)
{
	char szMyName[SYS_MAX_PATH_LEN];
	dosIter_Id iterId;

	strcpy(szMyName, szDirName);
	sysASCIIToUpper(szMyName);

	if (!pBS || !pBS->bValid || !(dosIsValidPath(szMyName)))
		return (NULL);

	iterId = (dosIter_Id) sysMalloc(sizeof(dosIter_S));
	if (!iterId)
		return (NULL);
	sysMemSet(iterId, 0, sizeof(dosIter_S));

	iterId->pBS = pBS;

	if (pBS->p1bs.fatSystem == DOS_FAT_BITS_32 ||
		(szMyName[0] && strcmp(szMyName, "\\")))
	{
		/* iterate on any other directory (folder) */
		iterId->bIterRD16 = FALSE;

		iterId->pDir = dosOpenDirForIter_ii(pBS, szMyName);
		if (!iterId->pDir)
		{
			sysFree(iterId);
			return (NULL);
		}
	}
	else
	{
		/* iterate on the FAT12/16 RD */
		iterId->bIterRD16 = TRUE;
		iterId->pDir = NULL;
	}

	iterId->nIterType = nIterType;
	iterId->nNumRDEntriesPerCluster = pBS->p1bs.bytesPerCluster/DOS_DIR_ENTRY_SIZE;
	iterId->nNextRDEntryIdx = 0;

	return (iterId);
}

static dosIterInfo_S * dosIterOneFileOnSubDir_ii(dosIter_Id iterId)
{
	static dosIterInfo_S iterInfo;
	static char buffer[16];
	unsigned char dirEntry[DOS_DIR_ENTRY_SIZE];

	if (!iterId)
		return (NULL);

	sysMemSet(buffer, 0, 16);

	while (TRUE)
	{
		if ((iterId->nNextRDEntryIdx + 1)*DOS_DIR_ENTRY_SIZE > iterId->pDir->nFileSize)
		{
			/* no more data to be read */

			break;
		}

		dosReadFile((dosFile_Id) iterId->pDir, DOS_DIR_ENTRY_SIZE, (char *) dirEntry);

		if (dosIsAttrROHFSFVN_ii(dirEntry))
		{
			/* A directory entry with attributes read-only, hidden,
				system, and volume name is a special extension by
				VFAT. And it is used for storing long file names.
				Immediately ignore it! */

			continue;
		}

		if (dosIsAttrVN_ii(dirEntry))
		{
			/* volume name; discard */
			continue;
		}

		if (dirEntry[0] == '\0')	/* lastest directory entry */
			return (NULL);


		iterId->nNextRDEntryIdx += 1;

		if (dirEntry[0] == 0xE5)	/* erased */
		{
			continue;
		}

		if (iterId->nIterType == doscIterFileOnly)
		{
			if (dosIsNormalFile_ii(dirEntry))
			{
				dosGetFileName_ii(dirEntry, buffer);
				iterInfo.szName = buffer;
				iterInfo.isDir = FALSE;
				iterInfo.nSize = dosGetFileSize_ii(dirEntry);
				return (&iterInfo);
			}
		}
		else if (iterId->nIterType == doscIterBoth)
		{
			dosGetFileName_ii(dirEntry, buffer);
			iterInfo.szName = buffer;
			iterInfo.isDir = dosIsDir_ii(dirEntry);
			if (iterInfo.isDir)
				iterInfo.nSize = 0;
			else
				iterInfo.nSize = dosGetFileSize_ii(dirEntry);
			return (&iterInfo);
		}
		else /* doscIterDirOnly */
		{
			if (dosIsDir_ii(dirEntry))
			{
				dosGetFileName_ii(dirEntry, buffer);
				iterInfo.szName = buffer;
				iterInfo.isDir = TRUE;
				iterInfo.nSize = 0;
				return (&iterInfo);
			}
		}
	}

	return (NULL);
}

static dosIterInfo_S * dosIterOneFileOnRD16_ii(dosIter_Id iterId)
{
	/*
	 * For FAT12/16 only.
	 */

	static dosIterInfo_S iterInfo;
	static char buffer[16];
	unsigned char dirEntry[DOS_DIR_ENTRY_SIZE];
	int nRDEntriesPerSector;

	if (!iterId)
		return (NULL);

	nRDEntriesPerSector = DOS_BYTES_PER_SEC/DOS_DIR_ENTRY_SIZE;
	sysMemSet(buffer, 0, 16);

	while (TRUE)
	{
		if (iterId->nNextRDEntryIdx >= iterId->pBS->p1bs.numEntriesInRD)
		{
			/* all entries in root directory are searched */

			return (NULL);
		}

		dosReadMediaSectorNByte_i(
			iterId->pBS,
			iterId->pBS->p1bs.nRDStartSector16 +
				(iterId->nNextRDEntryIdx/nRDEntriesPerSector),
			(iterId->nNextRDEntryIdx % nRDEntriesPerSector)*DOS_DIR_ENTRY_SIZE,
			DOS_DIR_ENTRY_SIZE,
			dirEntry);

		++iterId->nNextRDEntryIdx;

		if (dosIsAttrROHFSFVN_ii(dirEntry))
		{
			/* A directory entry with attributes read-only, hidden,
				system, and volume name is a special extension by
				VFAT. And it is used for storing long file names.
				Immediately ignore it! */

			continue;
		}

		if (dosIsAttrVN_ii(dirEntry))
		{
			/* volume name; discard */
			continue;
		}

		if (dirEntry[0] == '\0')	/* no more "meaningful" directory entry */
			return (NULL);

		if (dirEntry[0] == 0xE5)	/* erased */
		{
			continue;
		}

		if (iterId->nIterType == doscIterFileOnly)
		{
			if (dosIsNormalFile_ii(dirEntry))
			{
				dosGetFileName_ii(dirEntry, buffer);
				iterInfo.szName = buffer;
				iterInfo.isDir = FALSE;
				iterInfo.nSize = dosGetFileSize_ii(dirEntry);
				return (&iterInfo);
			}
		}
		else if (iterId->nIterType == doscIterBoth)
		{
			dosGetFileName_ii(dirEntry, buffer);
			iterInfo.szName = buffer;
			iterInfo.isDir = dosIsDir_ii(dirEntry);
			if (iterInfo.isDir)
				iterInfo.nSize = 0;
			else
				iterInfo.nSize = dosGetFileSize_ii(dirEntry);
			return (&iterInfo);
		}
		else /* doscIterDirOnly */
		{
			if (dosIsDir_ii(dirEntry))
			{
				dosGetFileName_ii(dirEntry, buffer);
				iterInfo.szName = buffer;
				iterInfo.isDir = TRUE;
				iterInfo.nSize = 0;
				return (&iterInfo);
			}
		}
	}
}

dosIterInfo_S * dosIterOneFile(dosIter_Id iterId)
{
	if (iterId->bIterRD16)
		return (dosIterOneFileOnRD16_ii(iterId));
	else
		return (dosIterOneFileOnSubDir_ii(iterId));
}

void dosStopIterFile(dosIter_Id iterId)
{
	if (!iterId)
		return;

	if (!iterId->bIterRD16)
		dosCloseDirForIter_ii(iterId->pDir);

	sysFree(iterId);
}

