#ifndef NF_CONTROLLER
#define NF_CONTROLLER

#include "nf_types.h"

uint8_t nf_ctrl_initialize(nf_controller_t *pcontroller);

/******************************************************************************/
/** @name Special Function Registers. */
/******************************************************************************/
/** @{ */
/** Address of NAND flash memory space begin.
 * Slave interface of NANDFLASHCTRL occupies some memory space. 
 * In this space are mapped internal buffer (BUFFER) 
 * and a Special Function Registers (SFR). 
 */

#define NF_CONTROLLER_ADDRESS   0xFFF08000

/** Controller commands register */
#define NF_SFR_COMMAND          0x00
/** The main configurations register */
#define NF_SFR_CONTROL          0x04
/** The controller status register */
#define NF_SFR_STATUS           0x08
/** Interrupts mask register */
#define NF_SFR_INT_MASK         0x0C
/** Interrupts status register */
#define NF_SFR_INT_STATUS       0x10
/** The ECC module statuses register. */
#define NF_SFR_ECC_CTRL         0x14
/** The ECC offset in the spare area. */
#define NF_SFR_ECC_OFFSET       0x18
/** The most significant part of the address register 0. */
#define NF_SFR_ADDR0_H          0x24
/** The least significant part of the address register 0. */
#define NF_SFR_ADDR0_L          0x1C
/** The most significant part of the address register 1. */
#define NF_SFR_ADDR1_H          0x28
/** The least significant part of the address register 1. */
#define NF_SFR_ADDR1_L          0x20
/** The SPARE_SIZE register 
 * it stores the actual value of the NAND flash device 
 * spare area size */
#define NF_SFR_SPARE_SIZE       0x30
/** The hardware protect against the write/erase process control register. */
#define NF_SFR_PROTECT          0x38
/** Enables Look-up registers during NAND Flash memory addressing. */
#define NF_SFR_LOOKUP_EN        0x40
/** Look Up Table register 0. LUT is used to address remapping. */
#define NF_SFR_LOOKUP0          0x44
/** Look Up Table register 1. LUT is used to address remapping. */
#define NF_SFR_LOOKUP1          0x48
/** Look Up Table register 2. LUT is used to address remapping. */
#define NF_SFR_LOOKUP2          0x4C
/** Look Up Table register 3. LUT is used to address remapping. */
#define NF_SFR_LOOKUP3          0x50
/** Look Up Table register 4. LUT is used to address remapping. */
#define NF_SFR_LOOKUP4          0x54
/** Look Up Table register 5. LUT is used to address remapping. */
#define NF_SFR_LOOKUP5          0x58
/** Look Up Table register 6. LUT is used to address remapping. */
#define NF_SFR_LOOKUP6          0x5C
/** Look Up Table register 7. LUT is used to address remapping. */
#define NF_SFR_LOOKUP7          0x60
/** DMA module first transfer address offset. */
#define NF_SFR_DMA_ADDR         0x64
/** DMA module counters initial value. */
#define NF_SFR_DMA_CNT          0x68
/** DMA module control register. */
#define NF_SFR_DMA_CTRL         0x6C
/** The memory device control register. */
#define NF_SFR_MEM_CTRL         0x80
/** The custom page size value. */
#define NF_SFR_DATA_SIZE        0x84
/** The custom page size value. */
#define NF_SFR_READ_STATUS      0x88
/** The command sequence timings configuration SFR. */
#define NF_SFR_TIME_SEQ_0       0x8C
/** The command sequence timings configuration SFR. */
#define NF_SFR_TIME_SEQ_1       0xBC
/** The timings configurations register 0. */
#define NF_SFR_TIMINGS_ASYN     0x90
/** The timings configurations register 1. */
#define NF_SFR_TIMINGS_SYN      0x94
/** The FIFO module interface. */
#define NF_SFR_FIFO_DATA        0x98
/** The DQS signal delay select. */
#define NF_SFR_TIME_MODE        0x9C
/** The DMA module address offset. */
#define NF_SFR_DMA_ADDR_OFFSET  0xA0
/** The FIFO_INIT register 
 * it is a control register for the FIFO module. */
#define NF_SFR_FIFO_INIT        0xB0
/** it stores the set of the configuration for the two generic sequences 
 * available to mimic the almost any command available in the 
 * present or in the future NAND flash devices */
#define NF_SFR_GENERIC_SEQ_CTRL 0xB4

#define NF_SFR_FIFO_STATE       0xB8
/* The latest release move ECC_STAT out of ECC_CTRL */
#define NF_SFR_ECC_STAT			0xC4
/** @} */
/******************************************************************************/



/******************************************************************************/
/** @name Marcos to write read NANDFlash registers. */
/******************************************************************************/
#define IOWR_32NF(register_address, data) \
    (*((volatile uint32_t*)(register_address))) = data
#define IOWR_16NF(register_address, data) \
    (*((volatile uint16_t*)(register_address))) = data
#define IOWR_8NF(register_address, data) \
    (*((volatile uint8_t*)(register_address))) = data

#define IORD_32NF(register_address) \
    (*((volatile uint32_t*)(register_address)))
#define IORD_16NF(register_address) \
    (*((volatile uint16_t*)(register_address)))
#define IORD_8NF(register_address) \
    (*((volatile uint8_t*)(register_address)))
/******************************************************************************/


/******************************************************************************/
/** @name The CONTROL register masks. */
/******************************************************************************/
/** @{ */
/** In this mode controller sends only the single byte as the column address 
 * instead of the two bytes as it is done for the 
 * big block NAND flash devices */
#define NF_SFR_SMALL_BLOCK_ENABLE                (0x1uL << 21)
/** mask of address cycles 1*/
#define NF_SFR_CONTROL_ADDR_CYCLE1_MASK          (0x7uL << 18)
/** 5 address bytes sent to NAND Flash device */
#define NF_SFR_CONTROL_ADDR_CYCLE1_5             (0x5uL << 18)
/** 4 address bytes sent to NAND Flash device */
#define NF_SFR_CONTROL_ADDR_CYCLE1_4             (0x4uL << 18)
/** 3 address bytes sent to NAND Flash device */
#define NF_SFR_CONTROL_ADDR_CYCLE1_3             (0x3uL << 18)
/** 2 address bytes sent to NAND Flash device */
#define NF_SFR_CONTROL_ADDR_CYCLE1_2             (0x2uL << 18)
/** 1 address bytes sent to NAND Flash device */
#define NF_SFR_CONTROL_ADDR_CYCLE1_1             (0x1uL << 18)
/** 0 address bytes sent to NAND Flash device */
#define NF_SFR_CONTROL_ADDR_CYCLE1_0             (0x0uL << 18)

/** Enable address auto increment for address register 0 */
#define NF_SFR_CONTROL_ADDR0_AUTO_INCR_EN       (0x1uL << 17)
/** Enable address auto increment for address register 1 */
#define NF_SFR_CONTROL_ADDR1_AUTO_INCR_EN       (0x1uL << 16)

/** Controller work mode - synchronous */
#define NF_SFR_CONTROL_WORK_MODE_SYNCHRONOUS    (0x1uL << 15)
/** Protect mechanism enable */
#define NF_SFR_CONTROL_PROT_EN                  (0x1uL << 14)
/** Lookup enable flag */
#define NF_SFR_CONTROL_FLLOOKUP_EN              (0x1uL << 13)
/** NAND flash IO width - 8 bits*/
#define NF_SFR_CONTROL_IO_WIDTH_8               (0x0uL << 12)
/** NAND flash IO width - 16 bits*/
#define NF_SFR_CONTROL_IO_WIDTH_16              (0x1uL << 12)
/** Enable transfer custom data block instead of full data page*/
#define NF_SFR_CONTROL_CUSTOM_DATA_SIZE_EN      (0x1uL << 11)

#define NF_SFR_CONTROL_PAGE_SIZE_MASK           (0x7ul << 8)
/** The page size - 16384 bytes  */
#define NF_SFR_CONTROL_PAGE_SIZE_16384          (0x6uL << 8)
/** The page size - 8192 bytes  */
#define NF_SFR_CONTROL_PAGE_SIZE_8192           (0x5uL << 8)
/** The page size - 4096 bytes  */
#define NF_SFR_CONTROL_PAGE_SIZE_4096           (0x4uL << 8)
/** The page size - 2048 bytes  */
#define NF_SFR_CONTROL_PAGE_SIZE_2048           (0x3uL << 8)
/** The page size - 1024 bytes  */
#define NF_SFR_CONTROL_PAGE_SIZE_1024           (0x2uL << 8)
/** The page size - 512 bytes  */
#define NF_SFR_CONTROL_PAGE_SIZE_512            (0x1uL << 8)
/** The page size - 256 bytes  */
#define NF_SFR_CONTROL_PAGE_SIZE_256            (0x0uL << 8)

/** The Block Size - mask */
#define NF_SFR_CONTROL_BLOCK_SIZE_MASK          (0x3uL << 6)
/** The Block Size - 256 pages per block  */
#define NF_SFR_CONTROL_BLOCK_SIZE_256           (0x3uL << 6)
/** The Block Size - 128 pages per block  */
#define NF_SFR_CONTROL_BLOCK_SIZE_128           (0x2uL << 6)
/** The Block Size - 64 pages per block  */
#define NF_SFR_CONTROL_BLOCK_SIZE_64            (0x1uL << 6)
/** The Block Size - 32 pages per block  */
#define NF_SFR_CONTROL_BLOCK_SIZE_32            (0x0uL << 6)
/** Hardware ECC support enable  */
#define NF_SFR_CONTROL_ECC_EN                   (0x1uL << 5)
/** Enable global interrupts  */
#define NF_SFR_CONTROL_INT_EN                   (0x1uL << 4)
/** Enable spare area for the given command sequence */
#define NF_SFR_CONTROL_SPARE_EN                 (0x1uL << 3)

/** mask of address cycles 0*/
#define NF_SFR_CONTROL_ADDR_CYCLE0_MASK          (0x7uL << 0)
/** 5 address bytes sent to NAND Flash device */
#define NF_SFR_CONTROL_ADDR_CYCLE0_5             (0x5uL << 0)
/** 4 address bytes sent to NAND Flash device */
#define NF_SFR_CONTROL_ADDR_CYCLE0_4             (0x4uL << 0)
/** 3 address bytes sent to NAND Flash device */
#define NF_SFR_CONTROL_ADDR_CYCLE0_3             (0x3uL << 0)
/** 2 address bytes sent to NAND Flash device */
#define NF_SFR_CONTROL_ADDR_CYCLE0_2             (0x2uL << 0)
/** 1 address bytes sent to NAND Flash device */
#define NF_SFR_CONTROL_ADDR_CYCLE0_1             (0x1uL << 0)
/** 0 address bytes sent to NAND Flash device */
#define NF_SFR_CONTROL_ADDR_CYCLE0_0             (0x0uL << 0)
/** @} */
/******************************************************************************/

/******************************************************************************/
/** @name The STATUS register masks. */
/******************************************************************************/
/** @{ */
/** The mode busy synchronous bit */
#define NF_SFR_STATUS_CTRL_SYN_STAT             (0x1uL << 9)
/** The main controller status bit */
#define NF_SFR_STATUS_CTRL_STAT                 (0x1uL << 8)
/** Device x busy flag */
#define NF_SFR_STATUS_MEMX_RDY(x)               (0x1uL << x)
/** Device 7 busy flag */
#define NF_SFR_STATUS_MEM7_ST                   (0x1uL << 7)
/** Device 6 busy flag */
#define NF_SFR_STATUS_MEM6_ST                   (0x1uL << 6)
/** Device 5 busy flag */
#define NF_SFR_STATUS_MEM5_ST                   (0x1uL << 5)
/** Device 4 busy flag */
#define NF_SFR_STATUS_MEM4_ST                   (0x1uL << 4)
/** Device 3 busy flag */
#define NF_SFR_STATUS_MEM3_ST                   (0x1uL << 3)
/** Device 2 busy flag */
#define NF_SFR_STATUS_MEM2_ST                   (0x1uL << 2)
/** Device 1 busy flag */
#define NF_SFR_STATUS_MEM1_ST                   (0x1uL << 1)
/** Device 0 busy flag */
#define NF_SFR_STATUS_MEM0_ST                   (0x1uL << 0)
/** @} */
/******************************************************************************/


/******************************************************************************/
/** @name The INT_MASK register masks. */
/******************************************************************************/
/** @{ */
/** FIFO error interrupt mask */
#define NF_SFR_INT_MASK_FIFO_ERROR              (0x1uL << 12)
/** The memory device 7 is ready for the new command */
#define NF_SFR_INT_MASK_MEM7_RDY_INT_EN         (0x1uL << 11)
/** The memory device 6 is ready for the new command */
#define NF_SFR_INT_MASK_MEM6_RDY_INT_EN         (0x1uL << 10)
/** The memory device 5 is ready for the new command */
#define NF_SFR_INT_MASK_MEM5_RDY_INT_EN         (0x1uL << 9)
/** The memory device 4 is ready for the new command */
#define NF_SFR_INT_MASK_MEM4_RDY_INT_EN         (0x1uL << 8)
/** The memory device 3 is ready for the new command */
#define NF_SFR_INT_MASK_MEM3_RDY_INT_EN         (0x1uL << 7)
/** The memory device 2 is ready for the new command */
#define NF_SFR_INT_MASK_MEM2_RDY_INT_EN         (0x1uL << 6)
/** The memory device 1 is ready for the new command */
#define NF_SFR_INT_MASK_MEM1_RDY_INT_EN         (0x1uL << 5)
/** The memory device 0 is ready for the new command */
#define NF_SFR_INT_MASK_MEM0_RDY_INT_EN         (0x1uL << 4)
/** The ECC module detected that the error level sat by the 
 * ECC_CTRL.ERR_THRESHOLD was exceeded - enable interupt*/
#define NF_SFR_INT_MASK_ECC_TRSH_ERR_EN         (0x1uL << 3)
/** The ECC module detected uncorrectable errors number during 
 * the read operation - enable interupt */
#define NF_SFR_INT_MASK_ECC_FATAL_ERR_EN        (0x1uL << 2)
/** Command sequence ended - interrupt enable */
#define NF_SFR_INT_MASK_CMD_END_INT_EN          (0x1uL << 1)
/** Erase/Write protected area attempt - interrupt enable */
#define NF_SFR_INT_MASK_PROT_INT_EN             (0x1uL << 0)

/** @} */
/******************************************************************************/

/******************************************************************************/
/** @name The INT_STATUS register masks. */
/******************************************************************************/
/** @{ */
/**  */
/** Read to /Write from FIFO error */
#define NF_SFR_INT_STATUS_FIFO_ERROR                (0x1uL << 12)
/** The memory device x is ready for the new command */
#define NF_SFR_INT_STATUS_MEMX_RDY(x)           (0x1uL << (x + 4))
/** The ECC module detected. 
 * That the error level sat by the ECC_CTRL.ERR_THRESHOLD was exceeded */
#define NF_SFR_INT_STATUS_ECC_TRSH_ERR_FL       (0x1uL << 3)
/** The ECC module detected uncorrectable errors number 
 * during the read operation. */
#define NF_SFR_INT_STATUS_ECC_FATAL_ERR_FL      (0x1uL << 2)
/** Transfer sequence ended */
#define NF_SFR_INT_STATUS_CMD_END_INT_FL        (0x1uL << 1)
/** Erase/Write protected area attempt interrupt enable */
#define NF_SFR_INT_STATUS_PROT_INT_FL           (0x1uL << 0)
/** @} */
/******************************************************************************/

/******************************************************************************/
/** @name The ECC_CTRL register masks. */
/******************************************************************************/
/** @{ */
/** The acceptable errors level - mask */
#define NF_SFR_ECC_CTR_ERR_THRESHOLD_MASK       (0x1FuL << 9)
/** The ECC module correction mask */
#define NF_SFR_ECC_CTR_ECC_CAP_MASK             (0x07uL << 5)
/** The ECC module correction ability - 2 bits */
#define NF_SFR_ECC_CTR_ECC_CAP_2                (0x0uL << 5)
/** The ECC module correction ability - 4 bits */
#define NF_SFR_ECC_CTR_ECC_CAP_4                (0x1uL << 5)
/** The ECC module correction ability - 6 bits */
#define NF_SFR_ECC_CTR_ECC_CAP_6                (0x2uL << 5)
/** The ECC module correction ability - 8 bits */
#define NF_SFR_ECC_CTR_ECC_CAP_8                (0x3uL << 5)
/** The ECC module correction ability - 10 bits */
#define NF_SFR_ECC_CTR_ECC_CAP_10               (0x4uL << 5)
/** The ECC module correction ability - 12 bits */
#define NF_SFR_ECC_CTR_ECC_CAP_12               (0x5uL << 5)
/** The ECC module correction ability - 14 bits */
#define NF_SFR_ECC_CTR_ECC_CAP_14               (0x6uL << 5)
/** The ECC module correction ability - 16 bits */
#define NF_SFR_ECC_CTR_ECC_CAP_16               (0x7uL << 5)
/** The The number of errors */
#define NF_SFR_ECC_CTR_ECC_ERR_OVER             (0x1uL << 16)
/** The uncorrectable error flag.  */
#define NF_SFR_ECC_CTR_ECC_ERR_UNCORRECT        (0x1uL << 8)
/** The correctable error flag.  */
#define NF_SFR_ECC_CTR_ECC_ERR_CORRECT          (0x1uL << 0)
/** @} */
/******************************************************************************/

/******************************************************************************/
/** @name The DMA_CTRL register masks. */
/******************************************************************************/
/** @{ */
/** DMA_START to start DMA when command sequence will be send 
        to the NAND flash memory */
#define NF_SFR_DMA_CTRL_DMA_START               (0x1uL << 7)
/** DMA transfer direction. Write data from AHB to internal BUFFER */
#define NF_SFR_DMA_CTRL_DMA_DIR_READ            (0x1uL << 6)
/** DMA transfer direction. Read from internal BUFFER and write to AHB */
#define NF_SFR_DMA_CTRL_DMA_DIR_WRITE           (0x0uL << 6)
/** DMA the Scatter-Gather mode */
#define NF_SFR_DMA_CTRL_DMA_MODE_SG             (0x1uL << 5)
/** DMA the SFR mode */
#define NF_SFR_DMA_CTRL_DMA_MODE_SFR            (0x0uL << 5)

/** incremental burst (burst length is four) */
#define NF_SFR_DMA_CTRL_INC_BURST_LENGTH_4      (0x0uL << 2)
/** Stream burst (address const)  */
#define NF_SFR_DMA_CTRL_STREAM_BURST            (0x1uL << 2)
/** single transfer (address increment) */
#define NF_SFR_DMA_CTRL_SINGLE_TRANSFER         (0x2uL << 2)
/** burst of unspecified length (address increment) */
#define NF_SFR_DMA_CTRL_BURST                   (0x3uL << 2)
/** incremental burst (burst length eight) */
#define NF_SFR_DMA_CTRL_INC_BURST_LENGTH_8      (0x4uL << 2)
/** incremental burst (burst length sixteen) */
#define NF_SFR_DMA_CTRL_INC_BURST_LENGTH_16     (0x5uL << 2)


/** DMA error flag
 * The flag is set when transmission error occurs during the DMA transfer. 
 * Set when the logical "1" value on the "serror" line was set.  */
#define NF_SFR_DMA_ERR_FLAG                     (0x1uL << 1)
/** DMA ready flag. The flag is set transfer is completed  */
#define NF_SFR_DMA_READY_FLAG                   (0x1uL << 0)
/** @} */
/******************************************************************************/


/******************************************************************************/
/** @name The COMMAND register masks and macros. */
/******************************************************************************/
/** @{ */
/** macro prepares command*/
#define NF_SFR_COMMAND_CMD(CMD0, CMD1, CMD2)    (((uint32_t)CMD0 & 0xFF) << 8) \
                                                | (((uint32_t)CMD1 & 0xFF) << 16) \
                                                | (((uint32_t)CMD2 & 0xFF) << 24)
/** Address 0 register select */
#define NF_SFR_COMMAND_ADDR_SEL_0               (0x1uL << 7)
/** Address 1 register select */
#define NF_SFR_COMMAND_ADDR_SEL_1               (0x1uL << 7)
/** select the BIU module as input */
#define NF_SFR_COMMAND_INPUT_SEL_BIU            (0x0uL << 6)
/** select the DMA module as input */
#define NF_SFR_COMMAND_INPUT_SEL_DMA            (0x1uL << 6)
/** Command code macro */
#define NF_SFR_COMMAND_CMD_SEQ(CMD)             (CMD & 0x1F)
/** @} */
/******************************************************************************/

/******************************************************************************/
/** @name The generic command seqences masks.*/
/******************************************************************************/
/** @{ */
#define NF_SFR_GENERIC_SEQ_CTRL_COL_ADDR_EN         (0x1uL << 17)
#define NF_SFR_GENERIC_SEQ_CTRL_DATA_EN             (0x1uL << 16)
#define NF_SFR_GENERIC_SEQ_CTRL_SET_CMD3_CODE(CMD)  ((uint32_t)CMD << 8)
#define NF_SFR_GENERIC_SEQ_CTRL_ENABLE_DELAY_1      (0x2uL << 6)
#define NF_SFR_GENERIC_SEQ_CTRL_ENABLE_DELAY_0      (0x1uL << 6)
#define NF_SFR_GENERIC_SEQ_CTRL_DISABLE_BOTH_DELAYS (0x0uL << 6)
#define NF_SFR_GENERIC_SEQ_CTRL_CMD3_EN             (0x1uL << 5)
#define NF_SFR_GENERIC_SEQ_CTRL_CMD2_EN             (0x1uL << 4)
#define NF_SFR_GENERIC_SEQ_CTRL_ADDR1_EN            (0x1uL << 3)
#define NF_SFR_GENERIC_SEQ_CTRL_CMD1_EN             (0x1uL << 2)
#define NF_SFR_GENERIC_SEQ_CTRL_ADDR0_EN            (0x1uL << 1)
#define NF_SFR_GENERIC_SEQ_CTRL_CMD0_EN             (0x1uL << 0)
/** @} */
/******************************************************************************/

/******************************************************************************/
/** @name The Command seqences */
/******************************************************************************/
/** @{ */
#define SEQ_0               0x00
#define SEQ_1               0x21
#define SEQ_2               0x22
#define SEQ_3               0x03
#define SEQ_4               0x24
#define SEQ_5               0x25
#define SEQ_6               0x26
#define SEQ_7               0x27
#define SEQ_8               0x08
#define SEQ_9               0x29
#define SEQ_10              0x2A
#define SEQ_11              0x2B
#define SEQ_12              0x0C
#define SEQ_13              0x0D
#define SEQ_14              0x0E
#define SEQ_15              0x2F
#define SEQ_16              0x30
#define SEQ_17              0x11
#define SEQ_18              0x32
#define SEQ_19              0x13
/** @} */
/******************************************************************************/

/// X means unimportant
#define X                   0x00

/******************************************************************************/
/** @name The Command codes */
/******************************************************************************/
/** @{ */
// Reset Commands
#define NF_CMD_RESET                (NF_SFR_COMMAND_CMD(0xFF, X, X) | SEQ_0)
#define NF_CMD_SYNCH_RESET          (NF_SFR_COMMAND_CMD(0xFC, X, X) | SEQ_0)

// Identification Operations
#define NF_CMD_READ_ID              (NF_SFR_COMMAND_CMD(0x90, X, X) | SEQ_1)
#define NF_CMD_READ_PARAMETER_PAGE  (NF_SFR_COMMAND_CMD(0xEC, X, X) | SEQ_2)
#define NF_CMD_READ_UNIQUE_ID       (NF_SFR_COMMAND_CMD(0xED, X, X) | SEQ_2)

//Configuration Operations
#define NF_CMD_GET_FEATURES         (NF_SFR_COMMAND_CMD(0xEE, X, X) | SEQ_2)
#define NF_CMD_SET_FEATURES         (NF_SFR_COMMAND_CMD(0xEF, X, X) | SEQ_3)

// Status Operations
#define NF_CMD_READ_STATUS          (NF_SFR_COMMAND_CMD(0x70, X, X) | SEQ_4)
#define NF_CMD_SEL_LUN_WITH_STATUS  (NF_SFR_COMMAND_CMD(0x78, X, X) | SEQ_5)

// Column Address Operations
#define NF_CMD_CHANGE_READ_COLUMN   (NF_SFR_COMMAND_CMD(0x05, 0xE0, X) | SEQ_6)
#define NF_CMD_SEL_CACHE_REGISTER   (NF_SFR_COMMAND_CMD(0x06, 0xE0, X) | SEQ_7)
#define NF_CMD_CHANGE_WRITE_COLUMN  (NF_SFR_COMMAND_CMD(0x85, X, X) | SEQ_8)
#define NF_CMD_CHANGE_ROW_ADDRESS   (NF_SFR_COMMAND_CMD(0x85, X, X) | SEQ_12)

// Read Operations
#define NF_CMD_READ_PAGE            (NF_SFR_COMMAND_CMD(0x00, 0x30, X) | SEQ_10)
#define NF_CMD_READ_SPARE           (NF_SFR_COMMAND_CMD(0x50, 0x30, X) | SEQ_10) 	//2010/12/22 for small page
#define NF_CMD_READ_PAGE_1          (NF_SFR_COMMAND_CMD(0x00, 0x30, X) | SEQ_9)
#define NF_CMD_READ_PAGE_CACHE      (NF_SFR_COMMAND_CMD(0x31, X, X) | SEQ_11)
#define NF_CMD_READ_PAGE_CACHE_LAST (NF_SFR_COMMAND_CMD(0x3F, X, X) | SEQ_11)
#define NF_CMD_READ_MULTIPLANE      (NF_SFR_COMMAND_CMD(0x00, 0x32, X) | SEQ_9)
#define NF_CMD_READ_TWO_PLANE       (NF_SFR_COMMAND_CMD(0x00, 0x00, 0x30) \
                                     | SEQ_16)
// Program Operation
#define NF_CMD_PROGRAM_PAGE         (NF_SFR_COMMAND_CMD(0x80, 0x10, X) | SEQ_12)
#define NF_CMD_PROGRAM_PAGE1        (NF_SFR_COMMAND_CMD(0x80, X, X) | SEQ_13)
#define NF_CMD_PROGRAM_PAGE_CACHE   (NF_SFR_COMMAND_CMD(0x80, 0x15, X) | SEQ_12)
#define NF_CMD_PROGRAM_MULTIPLANE   (NF_SFR_COMMAND_CMD(0x80, 0x10, X) | SEQ_12)
#define NF_CMD_WRITE_PAGE           (NF_SFR_COMMAND_CMD(0x10, X, X) | SEQ_0)
#define NF_CMD_WRITE_PAGE_CACHE     (NF_SFR_COMMAND_CMD(0x15, X, X) | SEQ_0)
#define NF_CMD_WRITE_MULTIPLANE     (NF_SFR_COMMAND_CMD(0x11, X, X) | SEQ_0)

// Erase Operation
#define NF_CMD_ERASE_BLOCK          (NF_SFR_COMMAND_CMD(0x60, 0xD0, X) | SEQ_14)
#define NF_CMD_ERASE_BLOCK_SEQ10    (NF_SFR_COMMAND_CMD(0x60, 0xD0, X) | SEQ_10)    // OK in ACTi FPGA design for small page
#define NF_CMD_ERASE_MULTIPLANE     (NF_SFR_COMMAND_CMD(0x60, 0xD1, X) | SEQ_14)

// Copyback Operation
#define NF_CMD_COPYBACK_READ        (NF_SFR_COMMAND_CMD(0x00, 0x35, X) | SEQ_10)
#define NF_CMD_COPYBACK_PROGRAM     (NF_SFR_COMMAND_CMD(0x85, 0x10, X) | SEQ_9)
#define NF_CMD_COPYBACK_PROGRAM1    (NF_SFR_COMMAND_CMD(0x85, X, X) | SEQ_13)
#define NF_CMD_COPYBACK_MULTIPLANE  (NF_SFR_COMMAND_CMD(0x85, 0x11, X) | SEQ_12)

// Otp Operation
#define NF_CMD_PROGRAM_OTP          (NF_SFR_COMMAND_CMD(0xA0, 0x10, X) | SEQ_12)
#define NF_CMD_DATA_PROTECT_OTP     (NF_SFR_COMMAND_CMD(0xA5, 0x10, X) | SEQ_9)
#define NF_CMD_READ_PAGE_OTP        (NF_SFR_COMMAND_CMD(0xAF, 0x30, X) | SEQ_10)
/** @} */
/******************************************************************************/

/******************************************************************************/
/** @name Data transfer driection definitions 
 * @anchor directions */
/******************************************************************************/
/** @{ */
/** data read from memory device */
#define NF_TRANSFER_DIRECTION_READ      1
/** data write to memory device */
#define NF_TRANSFER_DIRECTION_WRITE     0
/** @} */
/******************************************************************************/

/******************************************************************************/
/** @name Data transfer mode definitions 
 * @anchor transfer_mode */
/******************************************************************************/
/** @{ */
/** data transfer manually in loop */
#define NF_TRANSFER_MODE_MANUAL                 0x1
/** data transfer in DMA SFR mode */
#define NF_TRANSFER_MODE_DMA_SFR                0x2
/** data transfer in DMA scather gather mode */
#define NF_TRANSFER_MODE_DMA_SCATHER_GATHER     0x3
/** @} */
/******************************************************************************/

/******************************************************************************/
/** @name Operation (command) type definitions */
/******************************************************************************/
/** @{ */
/** write operation */
#define NF_OPERATION_TYPE_WRITE         NF_TRANSFER_DIRECTION_WRITE
/** read operation */
#define NF_OPERATION_TYPE_READ          NF_TRANSFER_DIRECTION_READ
/** erase operation */
#define NF_OPERATION_TYPE_ERASE         0x3
/** @} */
/******************************************************************************/

/******************************************************************************/
/** @name Configuration options. Using by nf_ctrl_configure function 
 *  @anchor configuration
 * */
/******************************************************************************/
/** @{ */
/** set page size 
 * acceptable page size values are 256, 512, 1024, 2048, 4096, 8192, 16384*/
#define NF_CONFIGURE_SET_PAGE_SIZE          0x01
/** set block size size 
 * acceptable block size values are 32, 64, 128, 256 */
#define NF_CONFIGURE_SET_BLOCK_SIZE         0x02
/** set organization
 * acceptable organization values are 8, 16 */
#define NF_CONFIGURE_SET_ORGANIZATION       0x03
/** set controller transfer mode
 * acceptable transfer mode values are NF_TRANSFER_MODE_MANUAL, 
 * NF_TRANSFER_MODE_DMA_SFR, NF_TRANSFER_MODE_DMA_SCATHER_GATHER */
#define NF_CONFIGURE_SELECT_TRANSFER_MODE   0x04
/** set acceptable error level 
 * acceptable errors level values are 0 - 31 */
#define NF_CONFIGURE_SET_ECC_ERR_LEVEL      0x06
/** set ECC corection ability 
 * acceptable corection ability values are 
 * NF_SFR_ECC_CTR_ECC_CAP_2, NF_SFR_ECC_CTR_ECC_CAP_4
 * NF_SFR_ECC_CTR_ECC_CAP_6, NF_SFR_ECC_CTR_ECC_CAP_8, 
 * NF_SFR_ECC_CTR_ECC_CAP_12, NF_SFR_ECC_CTR_ECC_CAP_14
 * NF_SFR_ECC_CTR_ECC_CAP_16*/
#define NF_CONFIGURE_SET_ECC_ABILITY        0x07

/** set controller address 0 cycles count */
#define NF_CONFIGURE_SET_ADDR0_CYCLE_COUNT  0x08
/** set controller address 1 cycles count */
#define NF_CONFIGURE_SET_ADDR1_CYCLE_COUNT  0x09

/** set controller work mode to synchronous */
#define NF_CONFIGURE_SET_SYNCHRONOUS_MODE   0x0A
/** set controller work mode to asynchronous */
#define NF_CONFIGURE_SET_ASYNCHRONOUS_MODE  0x0B
/** set spare area size */
#define NF_CONFIGURE_SET_SPARE_AREA_SIZE    0x0C
/**  enable hardware ECC */
#define NF_CONFIGURE_SET_ENABLE_HW_ECC      0x0D
/**  disable hardware ECC */
#define NF_CONFIGURE_SET_DISABLE_HW_ECC     0x0E
/** set ECC code offset value
 *  from beginning of the page to the place where correction words 
 *  will be store. Offset should be bigger than page size   */
#define NF_CONFIGURE_SET_ECC_OFFSET         0x0F
/** set DMA burst type  
 * acceptable DMA burst type values are NF_SFR_DMA_CTRL_INC_BURST_LENGTH_4,
 * NF_SFR_DMA_CTRL_STREAM_BURST, NF_SFR_DMA_CTRL_SINGLE_TRANSFER, 
 * NF_SFR_DMA_CTRL_BURST, NF_SFR_DMA_CTRL_INC_BURST_LENGTH_8, 
 * NF_SFR_DMA_CTRL_INC_BURST_LENGTH_16 */
#define NF_CONFIGURE_SET_DMA_BURST_TYPE     0x10
/** @} */
/******************************************************************************/

/******************************************************************************/
/** @name macros helpful in command creation */
/******************************************************************************/
/** @{ */
#define NF_INIT_CMD_SIMPLE(command, code) {\
    command.command_code = code; \
    command.page_address_0 = 0; \
    command.page_offset_0 = 0; \
    command.page_address_1 = 0;\
    command.page_offset_1 = 0;\
    command.data_size = 0; \
}

#define NF_ERASE_CMD_WITH_ADDR(command, code, page, offset) {\
    command.command_code = code;\
    command.page_address_0 = page;\
    command.page_offset_0 = offset;\
    command.page_address_1 = 0;\
    command.page_offset_1 = 0;\
    command.data_size = 0;\
	command.ecc_mode = 0;\
}

#define NF_INIT_CMD_WITH_ADDR(command, code, page, offset) {\
    command.command_code = code;\
    command.page_address_0 = page;\
    command.page_offset_0 = offset;\
    command.page_address_1 = 0;\
    command.page_offset_1 = 0;\
    command.data_size = 0;\
	command.ecc_mode = 0;\
}

#define NF_INIT_CMD_WITH_ADDR_AND_DATA(command, code, page, offset, \
                                       buff, size, direction, ecc) {\
    command.command_code = code;\
    command.page_address_0 = page;\
    command.page_offset_0 = offset;\
    command.page_address_1 = 0;\
    command.page_offset_1 = 0;\
    command.buffer = buff;\
    command.data_size = size;\
    command.transfer_direction = direction;\
    command.ecc_mode = ecc;\
}

#define NF_INIT_CMD_WITH_2ADDR_AND_DATA(command, code, page0, offset0,  \
                                        page1, offset1,                 \
                                        buff, size, direction, ecc) {   \
    command.command_code = code;\
    command.page_address_0 = page0;\
    command.page_offset_0 = offset0;\
    command.page_address_1 = page1;\
    command.page_offset_1 = offset1;\
    command.buffer = buff;\
    command.data_size = size;\
    command.transfer_direction = direction;\
    command.ecc_mode = ecc;\
}
/** @} */
/******************************************************************************/

#define NF_IS_ECC_ENABLED(pcontroller) \
    ((IORD_32NF(pcontroller->reg_offset + NF_SFR_CONTROL) \
		& NF_SFR_CONTROL_ECC_EN) != 0)

#define NF_IS_WP_ENABLED(pcontroller) \
    ((IORD_32NF(pcontroller->reg_offset + NF_SFR_MEM_CTRL) \
		& (1uL << (pcontroller->ce + 8))) == 0)

/******************************************************************************/
/*!                                                        
 * @fn          uint8_t nf_ctrl_initialize(nf_controller_t *pcontroller)
 * @brief       Function intializes NAND Flash controller.
 * @param       pcontroller nand flash controller object
 * @return      Function returns 0 if everything is ok 
 *              otherwise returns error number   
*/
/******************************************************************************/
uint8_t nf_ctrl_initialize(nf_controller_t *pcontroller);

/******************************************************************************/
/*!                                                        
 * @fn      uint8_t nf_ctrl_execute_command(nf_controller_t *pcontroller, 
 *                                           nf_ctrl_command_t *command)
 * @brief       Function executes command on pcontroller.
 * @param       pcontroller nand flash controller object
 * @param       command command to execute
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
*/
/******************************************************************************/
uint8_t nf_ctrl_execute_command(nf_controller_t *pcontroller, 
                                nf_ctrl_command_t *command);

/******************************************************************************/
/*!                                                        
 * @fn      uint8_t nf_ctrl_configure(nf_controller_t *pcontroller, 
 *                                    uint8_t command, uint32_t argument);
 * @brief       Function configures pcontroller.
 * @param       pcontroller nand flash controller object
 * @param       command it defines what will be configure. All commands are
 *                  defined here @ref configuration
 * @param       argument additional argument
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
*/
/******************************************************************************/
uint8_t nf_ctrl_configure(nf_controller_t *pcontroller, uint8_t command, 
                          uint32_t argument);

/******************************************************************************/
/*!                                                        
 * @fn      uint8_t nf_ctrl_select_target(nf_controller_t *pcontroller, 
 *                                        uint8_t memory_number);
 * @brief       Function sets chip enable signal for given memory.
 * @param       pcontroller nand flash controller object
 * @param       memory_number number of memory which should be enable
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
*/
/******************************************************************************/
uint8_t nf_ctrl_select_target(nf_controller_t *pcontroller, 
                              uint8_t memory_number);

/******************************************************************************/
/*!                                                        
 * @fn      uint8_t nf_ctrl_set_protected_area(nf_controller_t *pcontroller, 
 *                                             uint16_t begin_protect_area, 
 *                                             uint16_t end_protect_area);
 * @brief       Function sets protected area which will be protect against 
 *                  the write/erase process
 * @param       pcontroller nand flash controller object
 * @param       begin_protect_area start block number of protect area
 * @param       end_protect_area end block number of protect area
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
*/
/******************************************************************************/
uint8_t nf_ctrl_set_protected_area(nf_controller_t *pcontroller, 
                                   uint16_t begin_protect_area, 
                                   uint16_t end_protect_area);

/******************************************************************************/
/*!                                                        
 * @fn      uint8_t nf_ctrl_get_protected_area(nf_controller_t *pcontroller, 
 *                                             uint16_t begin_protect_area, 
 *                                             uint16_t end_protect_area);
 * @brief       Function gets protected area 
 * @param       pcontroller nand flash controller object
 * @param       begin_protect_area start block number of protect area
 * @param       end_protect_area end block number of protect area
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
*/
/******************************************************************************/
uint8_t nf_ctrl_get_protected_area(nf_controller_t *pcontroller, 
                                   uint16_t *begin_protect_area, 
                                   uint16_t *end_protect_area);

/******************************************************************************/
/*!                                                        
 * @fn      uint8_t nf_ctrl_lookup_set(nf_controller_t *pcontroller, 
 *                                     uint16_t logical_address, 
 *                                     uint16_t physical_address);
 * @brief       Function sets one pair of addresses to lookup register 
 *                  and enables address remapping. The lookup registers build
 *                  a Look Up Table for readdressing blocks. It is helpful 
 *                  for Bad Block Management. 
 * @param       pcontroller nand flash controller object
 * @param       logical_address address of the block that will be replaced 
 *                  in the remapping process
 * @param       physical_address address of the block that will replace the 
 *                  bad one in the remapping process.
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
*/
/******************************************************************************/
uint8_t nf_ctrl_lookup_set(nf_controller_t *pcontroller, 
                           uint16_t logical_address, 
                           uint16_t physical_address);

/******************************************************************************/
/*!                                                        
 * @fn      uint8_t nf_ctrl_lookup_release(nf_controller_t *pcontroller, 
 *                                         uint16_t logical_address);
 * @brief   Function disables memory remapping for the logical_address 
 *                  parameter
 * @param       pcontroller nand flash controller object
 * @param       logical_address address of the block that is replaced 
 *                  in the remapping process 
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
*/
/******************************************************************************/
uint8_t nf_ctrl_lookup_release(nf_controller_t *pcontroller, 
                               uint16_t logical_address);

/******************************************************************************/
/*!                                                        
 * @fn      uint8_t nf_ctrl_lookup_release_all(nf_controller_t *pcontroller)
 * @brief   Function disables all remaping system
 * @param       pcontroller nand flash controller object
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
*/
/******************************************************************************/
uint8_t nf_ctrl_lookup_release_all(nf_controller_t *pcontroller);

/******************************************************************************/
/*!                                                        
 * @fn      uint8_t nf_ctrl_configure_timings(nf_controller_t *pcontroller,
 *                                              nf_timings_t *timings)
 * @brief   Function configure nand flash controller timings.
 * @param       pcontroller nand flash controller object
 * @param       timings timings settings to write
 * @return      Function returns 0 if everything is ok 
 *                  otherwise returns error number   
*/
/******************************************************************************/
uint8_t nf_ctrl_configure_timings(nf_controller_t *pcontroller,
                                  nf_timings_t *timings);

/******************************************************************************/
/*!                                                        
 * \fn      uint8_t nf_ctrl_set_write_protect(nf_controller_t *pcontroller, 
 *                                             uint8_t memory_number, 
 *                                             uint8_t enable)
 *
 * \brief   Function enables or disables write protect
 * \param   pcontroller nand flash controller object
 * \param   memory_number defines which memory shall be configured
 * \param   enable describes that write protect should be
 *              disabled or enabled
 *              0 - disable write protect
 *              1 - enable write protect
 * \return  Function returns 0 if everything is ok 
 *              otherwise returns error number          
 */
/******************************************************************************/
uint8_t nf_ctrl_set_write_protect(nf_controller_t *pcontroller, 
                                  uint8_t memory_number, uint8_t enable);

/******************************************************************************/
/*!                                                        
 * \fn      uint8_t nf_ctrl_disable_protected_area(nf_controller_t *pcontroller)
 *
 * \brief   Function disables protect area
 * \param   pcontroller nand flash controller object
 * \return  Function returns 0 if everything is ok 
 *              otherwise returns error number          
 */
/******************************************************************************/
uint8_t nf_ctrl_disable_protected_area(nf_controller_t *pcontroller);

/******************************************************************************/
/*!                                                        
 * \fn      uint8_t 
 *          nf_ctrl_execute_universal_command(nf_controller_t *pcontroller, 
 *                                            nf_ctrl_universal_command_t *command)
 *
 * \brief   Function executes commands which cannot be executed  by 
 *              nf_ctrl_execute_command function. With this function 
 *              user can mimic the almost every available command 
 *              supported by the NAND flash devices. 
 *              The cost of this universality is the additional effort required
 *              to trigger such command. 
 *              Sequence composition:
 *              CMD0 ADDR0 CMD3 ADDR1 CMD2 DEL0 DAT CMD1 DEL1
 * \param   pcontroller nand flash controller object
 * \param   command command to execute  
 * \return  Function returns 0 if everything is ok 
 *              otherwise returns error number          
 */
/******************************************************************************/
uint8_t nf_ctrl_execute_universal_command(nf_controller_t *pcontroller, 
                                          nf_ctrl_universal_command_t *command);

#endif
