//  ***************************************************************************
// 
//   Copyright (c) 2009  Evatronix SA 
//
//  ***************************************************************************
//
//   Please review the terms of the license agreement before using     
//   this file. If you are not an authorized user, please destroy this 
//   source code file and notify Evatronix SA immediately that you     
//   inadvertently received an unauthorized copy.                      
//
//  ***************************************************************************
/// @file           nf_controller.c
/// @brief          NAND Flash controller driver
/// @version        $Revision: 1.7 $
/// @author         Piotr Sroka
/// @date           $Date: 2010-05-14 11:03:42 $
//  ***************************************************************************
#include <asm/arch/nand/nf_controller.h>
#include <asm/arch/nand/nf_error.h>
#include <asm/arch/nand/nf_setup.h>
#include <asm/arch/nand/nf_utils.h>
#include <asm/arch/nand/nf_dma.h>
#include <common.h>
#include "hal.h"



static uint8_t ecc_size_io8[] = {4, 7, 10, 13, 17, 20, 23, 26};

/*****************************************************************************/
static void calculate_ecc_size(nf_controller_t *pcontroller)
{
    uint32_t ecc_size;

    // check how many bytes are used by ECC code
    // per 512 bytes, or 256 bytes if memory has page smaller than 512 bytes    
    if (pcontroller->page_size < 512)
        ecc_size = pcontroller->ecc_size_per_unit 
            * (pcontroller->page_size / 256);
    else
        ecc_size = pcontroller->ecc_size_per_unit 
            * (pcontroller->page_size / 512);

    if ((pcontroller->ecc_offset + ecc_size) 
        > pcontroller->page_size + pcontroller->spare_area_size){
        pcontroller->ecc_status = NF_ERR_ECC_OUT_OFF_PAGE;
    }
    else {
        pcontroller->ecc_status = NF_ERR_NO_ERRORS;
    }
}
/*****************************************************************************/

/*****************************************************************************/
static uint8_t set_organization(nf_controller_t *pcontroller, 
                                uint8_t organization)
{
    uint32_t tmp;

    switch (organization){
        // enable 8 bit data width
    case 8:
        tmp = IORD_32NF(pcontroller->reg_offset + NF_SFR_CONTROL);
        tmp = (tmp & ~NF_SFR_CONTROL_IO_WIDTH_16);
        IOWR_32NF(pcontroller->reg_offset + NF_SFR_CONTROL, tmp);
        break;
        // enable 16 bit data width
    case 16:
        tmp = IORD_32NF(pcontroller->reg_offset + NF_SFR_CONTROL);
        tmp = tmp | NF_SFR_CONTROL_IO_WIDTH_16;
        IOWR_32NF(pcontroller->reg_offset + NF_SFR_CONTROL, tmp);
        break;
    default:
        return NF_ERR_INVALID_PARAMETER;
    }

    // save new IO width in controller structure
    pcontroller->organization = organization;

    return NF_ERR_NO_ERRORS;
}
/*****************************************************************************/

/*****************************************************************************/
static uint8_t set_page_size(nf_controller_t *pcontroller, 
                             uint16_t page_size)
{  
    uint32_t tmp;
    uint32_t new_page_size;

//    if (page_size == pcontroller->page_size)
//        return NF_ERR_NO_ERRORS;

    switch(page_size){
    case 256:
        new_page_size = NF_SFR_CONTROL_PAGE_SIZE_256;
        break;
    case 512:
        new_page_size = NF_SFR_CONTROL_PAGE_SIZE_512;
        break;
    case 1024:
        new_page_size = NF_SFR_CONTROL_PAGE_SIZE_1024;
        break;
    case 2048:
        new_page_size = NF_SFR_CONTROL_PAGE_SIZE_2048;
        break;
    case 4096:
        new_page_size = NF_SFR_CONTROL_PAGE_SIZE_4096;
        break;
    case 8192:
        new_page_size = NF_SFR_CONTROL_PAGE_SIZE_8192;
        break;
    case 16384:
        new_page_size = NF_SFR_CONTROL_PAGE_SIZE_16384;
        break;
    default:
        return NF_ERR_INVALID_PARAMETER;
    }

    tmp =  IORD_32NF(pcontroller->reg_offset + NF_SFR_CONTROL);
    tmp = (tmp & ~NF_SFR_CONTROL_PAGE_SIZE_MASK) | new_page_size;

    if (page_size > 512){
        tmp &= ~NF_SFR_SMALL_BLOCK_ENABLE;
    }
    else {
        tmp |= NF_SFR_SMALL_BLOCK_ENABLE;
    }

    IOWR_32NF(pcontroller->reg_offset + NF_SFR_CONTROL, tmp);

    // save new page size in controller structure
    pcontroller->page_size = page_size;

    // ecc size depends on page size so we calculate it here
    calculate_ecc_size(pcontroller);

    return NF_ERR_NO_ERRORS;
}
/*****************************************************************************/

/*****************************************************************************/
static uint8_t set_block_size(nf_controller_t *pcontroller, 
                              uint16_t block_size)
{
    uint32_t new_block_size;
    uint32_t tmp;

    switch(block_size){
    case 32:
        new_block_size = NF_SFR_CONTROL_BLOCK_SIZE_32;
        break;
    case 64:
        new_block_size = NF_SFR_CONTROL_BLOCK_SIZE_64;
        break;
    case 128:
        new_block_size = NF_SFR_CONTROL_BLOCK_SIZE_128;
        break;
    case 256:
        new_block_size = NF_SFR_CONTROL_BLOCK_SIZE_256;
        break;
    default:
        return NF_ERR_INVALID_PARAMETER;        
    }

    // configure controller
    tmp =  IORD_32NF(pcontroller->reg_offset + NF_SFR_CONTROL);
    tmp = (tmp & ~NF_SFR_CONTROL_BLOCK_SIZE_MASK) | new_block_size;
    IOWR_32NF(pcontroller->reg_offset + NF_SFR_CONTROL, tmp);

    // save new block size in controller structure
    pcontroller->block_size = block_size;

    return NF_ERR_NO_ERRORS;
}
/*****************************************************************************/

/*****************************************************************************/
static uint8_t 
set_data_transfer_mode(nf_controller_t *pcontroller, uint8_t mode)
{
    uint8_t status;

    switch(mode){
    case NF_TRANSFER_MODE_MANUAL:
    case NF_TRANSFER_MODE_DMA_SFR:
        pcontroller->transfer_mode = mode;
        status = NF_ERR_NO_ERRORS;
        break;
    case NF_TRANSFER_MODE_DMA_SCATHER_GATHER:
#if USE_INDIRECT_BUFFER
        // scatter gather mode is not supported 
        // if USE_INDIRECT_BUFFER is enabled
        status = NF_ERR_DMA_SG_UNSUPPORTED_IB;
#else
        pcontroller->transfer_mode = mode;
        status = NF_ERR_NO_ERRORS;
#endif        
        break;
    default:
        status = NF_ERR_INVALID_PARAMETER;
    }
    return status;
}
/*****************************************************************************/

/*****************************************************************************/
static uint8_t set_ecc_ability(nf_controller_t *pcontroller, 
                               uint8_t error_level)
{
    uint32_t new_ecc_cap = 0;
    uint32_t ecc_ctrl;

#if NF_ECC_ALGORITHM != 216
    return NF_ERR_UNSUPPORTED_OPERATION;
#endif

    switch(error_level){
    case 2:
        new_ecc_cap |= NF_SFR_ECC_CTR_ECC_CAP_2;
        break;
    case 4:
        new_ecc_cap |= NF_SFR_ECC_CTR_ECC_CAP_4;
        break;
    case 6:
        new_ecc_cap |= NF_SFR_ECC_CTR_ECC_CAP_6;
        break;
    case 8:
        new_ecc_cap |= NF_SFR_ECC_CTR_ECC_CAP_8;
        break;
    case 10:
        new_ecc_cap |= NF_SFR_ECC_CTR_ECC_CAP_10;
        break;
    case 12:
        new_ecc_cap |= NF_SFR_ECC_CTR_ECC_CAP_12;
        break;
    case 14:
        new_ecc_cap |= NF_SFR_ECC_CTR_ECC_CAP_14;
        break;
    case 16:
        new_ecc_cap |= NF_SFR_ECC_CTR_ECC_CAP_16;
        break;
    default:
        return NF_ERR_INVALID_PARAMETER;
    }

    pcontroller->ecc_size_per_unit = ecc_size_io8[(error_level / 2) - 1];
    calculate_ecc_size(pcontroller);

    ecc_ctrl = IORD_32NF(pcontroller->reg_offset + NF_SFR_ECC_CTRL);
    ecc_ctrl &= ~NF_SFR_ECC_CTR_ECC_CAP_MASK;
    ecc_ctrl |= new_ecc_cap;
    IOWR_32NF(pcontroller->reg_offset + NF_SFR_ECC_CTRL, ecc_ctrl);

    return NF_ERR_NO_ERRORS;
}
/*****************************************************************************/

/*****************************************************************************/
static uint8_t set_ecc_err_level(nf_controller_t *pcontroller, 
                                 uint8_t error_level)
{
    uint32_t ecc_ctrl;

    if (error_level > 0x1F){
        return NF_ERR_INVALID_PARAMETER;
    } 

    ecc_ctrl = IORD_32NF(pcontroller->reg_offset + NF_SFR_ECC_CTRL);
    ecc_ctrl &= ~NF_SFR_ECC_CTR_ERR_THRESHOLD_MASK;
    ecc_ctrl |= ((uint8_t)error_level << 8);
    IOWR_32NF(pcontroller->reg_offset + NF_SFR_ECC_CTRL, ecc_ctrl);

    return NF_ERR_NO_ERRORS;
}
/*****************************************************************************/

/*****************************************************************************/
static uint8_t 
set_work_mode(nf_controller_t *pcontroller, uint8_t work_mode)
{
    uint32_t ctrl = IORD_32NF(pcontroller->reg_offset + NF_SFR_CONTROL);

    switch(work_mode){
    case NF_CONFIGURE_SET_SYNCHRONOUS_MODE:
        ctrl |= NF_SFR_CONTROL_WORK_MODE_SYNCHRONOUS 
            | NF_SFR_CONTROL_IO_WIDTH_16;
        break;
    case NF_CONFIGURE_SET_ASYNCHRONOUS_MODE:
        ctrl &= ~(NF_SFR_CONTROL_WORK_MODE_SYNCHRONOUS
                  | NF_SFR_CONTROL_IO_WIDTH_16);
        break;        
    default:
        return NF_ERR_INVALID_PARAMETER;
    }

    IOWR_32NF(pcontroller->reg_offset + NF_SFR_CONTROL, ctrl);

    return wait_for_value(pcontroller->reg_offset + NF_SFR_STATUS, 
                          NF_SFR_STATUS_CTRL_SYN_STAT, 20, 0);
}
/*****************************************************************************/

/*****************************************************************************/
static uint8_t
set_spare_area_size(nf_controller_t *pcontroller, uint8_t spare_size)
{
    pcontroller->spare_area_size = spare_size;

    // ecc size is placed in spare area so we calculate it here
    calculate_ecc_size(pcontroller);    

    return NF_ERR_NO_ERRORS;
}
/*****************************************************************************/

/*****************************************************************************/
static uint8_t
set_hardware_ecc(nf_controller_t *pcontroller, uint8_t ecc_state)
{
    // this function sets only controller flag
    // real ECC enabling/disabling is made by nf_ctrl_execute_command function
    if (ecc_state == NF_CONFIGURE_SET_ENABLE_HW_ECC){
        pcontroller->ecc_enabled = 1;
    }
    if (ecc_state == NF_CONFIGURE_SET_DISABLE_HW_ECC){
        pcontroller->ecc_enabled = 0;
    }
    return NF_ERR_NO_ERRORS;
}
/*****************************************************************************/

/*****************************************************************************/
static uint8_t
set_ecc_offset(nf_controller_t *pcontroller, uint16_t ecc_offset)
{  
    IOWR_32NF(pcontroller->reg_offset + NF_SFR_ECC_OFFSET, ecc_offset);

    pcontroller->ecc_offset = ecc_offset;

    calculate_ecc_size(pcontroller);    

    return NF_ERR_NO_ERRORS;    
}
/*****************************************************************************/

/*****************************************************************************/
static uint8_t
set_dma_burst_type(nf_controller_t *pcontroller, uint8_t burst_type)
{
    switch(burst_type){
    case NF_SFR_DMA_CTRL_INC_BURST_LENGTH_4:
    case NF_SFR_DMA_CTRL_STREAM_BURST:
    case NF_SFR_DMA_CTRL_SINGLE_TRANSFER:
    case NF_SFR_DMA_CTRL_BURST:
    case NF_SFR_DMA_CTRL_INC_BURST_LENGTH_8:
    case NF_SFR_DMA_CTRL_INC_BURST_LENGTH_16:
        pcontroller->dma_burst_type = burst_type;
        break;
    default:
        return  NF_ERR_INVALID_PARAMETER;
    }
    return NF_ERR_NO_ERRORS;    
}
/*****************************************************************************/

/*****************************************************************************/
static uint8_t 
pio_data_transfer(nf_controller_t *pcontroller, uint8_t *buffer, 
                  uint32_t buffer_size, uint8_t direction)
{
    uint32_t i;
    uint32_t *buf32 = (uint32_t *)buffer;

    // if buffer size is not aligned to 4 return error
    if (buffer_size & 0x3){
        return NF_ERR_INVALID_PARAMETER;
    }

    if (direction == NF_TRANSFER_DIRECTION_READ){
        while(!( IORD_32NF(pcontroller->reg_offset + NF_SFR_FIFO_STATE) & 1));
        for (i = 0; i < buffer_size / 4; i++){
            while(!( IORD_32NF(pcontroller->reg_offset + NF_SFR_FIFO_STATE) & 1));
            // read 4 bytes of data from fifo                
            buf32[i] = IORD_32NF(pcontroller->reg_offset + NF_SFR_FIFO_DATA);
        }
    }
    else if (direction == NF_TRANSFER_DIRECTION_WRITE) {
        while(!( IORD_32NF(pcontroller->reg_offset + NF_SFR_FIFO_STATE) & 1));          
        for (i = 0; i < buffer_size / 4; i++){
            //write 4 bytes to fifo
            IOWR_32NF(pcontroller->reg_offset + NF_SFR_FIFO_DATA, buf32[i]);
        }            
    }
    return NF_ERR_NO_ERRORS;
}
/*****************************************************************************/

/*****************************************************************************/
static uint8_t set_address_cycle_count(nf_controller_t *pcontroller, 
                                       uint8_t interface, uint8_t cycle_count)
{
    uint32_t ctrl = IORD_32NF(pcontroller->reg_offset + NF_SFR_CONTROL);

    switch(interface){
    case 0:
        ctrl &= ~NF_SFR_CONTROL_ADDR_CYCLE0_MASK;
        switch(cycle_count){
        case 0:
            ctrl |= NF_SFR_CONTROL_ADDR_CYCLE0_0;
            break;
        case 1:
            ctrl |= NF_SFR_CONTROL_ADDR_CYCLE0_1;
            break;
        case 2:
            ctrl |= NF_SFR_CONTROL_ADDR_CYCLE0_2;
            break;
        case 3:
            ctrl |= NF_SFR_CONTROL_ADDR_CYCLE0_3;
            break;
        case 4:
            ctrl |= NF_SFR_CONTROL_ADDR_CYCLE0_4;
            break;
        case 5:
            ctrl |= NF_SFR_CONTROL_ADDR_CYCLE0_5;
            break;
        default:
            return NF_ERR_INVALID_PARAMETER;
        }
        break;

    case 1:
        ctrl &= ~NF_SFR_CONTROL_ADDR_CYCLE1_MASK;
        switch(cycle_count){
        case 0:
            ctrl |= NF_SFR_CONTROL_ADDR_CYCLE1_0;
            break;
        case 1:
            ctrl |= NF_SFR_CONTROL_ADDR_CYCLE1_1;
            break;
        case 2:
            ctrl |= NF_SFR_CONTROL_ADDR_CYCLE1_2;
            break;
        case 3:
            ctrl |= NF_SFR_CONTROL_ADDR_CYCLE1_3;
            break;
        case 4:
            ctrl |= NF_SFR_CONTROL_ADDR_CYCLE1_4;
            break;
        case 5:
            ctrl |= NF_SFR_CONTROL_ADDR_CYCLE1_5;
            break;
        default:
            return NF_ERR_INVALID_PARAMETER;
        }
        break;        

    default:
        return NF_ERR_INVALID_PARAMETER;
    }

    IOWR_32NF(pcontroller->reg_offset + NF_SFR_CONTROL, ctrl);

    return NF_ERR_NO_ERRORS;
}
/*****************************************************************************/

/*****************************************************************************/
static uint8_t wait_until_controller_busy(nf_controller_t *pcontroller)
{
    return wait_for_value(pcontroller->reg_offset + NF_SFR_STATUS, 
                          NF_SFR_STATUS_CTRL_STAT, 20, 0);

}
/*****************************************************************************/

/*****************************************************************************/
uint8_t nf_ctrl_configure(nf_controller_t *pcontroller, uint8_t command, 
                          uint32_t argument)
{
    switch(command){
    case NF_CONFIGURE_SET_ADDR0_CYCLE_COUNT:
        return set_address_cycle_count(pcontroller, 0, (uint8_t)argument);
    case NF_CONFIGURE_SET_ADDR1_CYCLE_COUNT:
        return set_address_cycle_count(pcontroller, 1, (uint8_t)argument);
    case NF_CONFIGURE_SET_ENABLE_HW_ECC:
    case NF_CONFIGURE_SET_DISABLE_HW_ECC:
        return set_hardware_ecc(pcontroller, command);    
    case NF_CONFIGURE_SET_PAGE_SIZE:
        return set_page_size(pcontroller, (uint16_t)argument);
    case NF_CONFIGURE_SET_BLOCK_SIZE:
        return set_block_size(pcontroller, (uint16_t)argument);
    case NF_CONFIGURE_SET_ORGANIZATION:
        return set_organization(pcontroller, (uint8_t)argument);
    case NF_CONFIGURE_SELECT_TRANSFER_MODE:
        return set_data_transfer_mode(pcontroller, (uint8_t)argument);
    case NF_CONFIGURE_SET_ECC_ERR_LEVEL:
        return set_ecc_err_level(pcontroller, (uint8_t)argument);
    case NF_CONFIGURE_SET_ECC_ABILITY:
        return set_ecc_ability(pcontroller, (uint8_t)argument);
    case NF_CONFIGURE_SET_ECC_OFFSET:
        return set_ecc_offset(pcontroller, (uint16_t)argument);       
    case NF_CONFIGURE_SET_SYNCHRONOUS_MODE:
    case NF_CONFIGURE_SET_ASYNCHRONOUS_MODE:
        return set_work_mode(pcontroller, (uint8_t)command);
    case NF_CONFIGURE_SET_SPARE_AREA_SIZE:
        return set_spare_area_size(pcontroller, (uint8_t)argument);
    case NF_CONFIGURE_SET_DMA_BURST_TYPE:
        return set_dma_burst_type(pcontroller, (uint8_t)argument);
    default:
        return NF_ERR_INVALID_PARAMETER;
    }

    return NF_ERR_NO_ERRORS;
}
/*****************************************************************************/

/*****************************************************************************/
uint8_t nf_ctrl_select_target(nf_controller_t *pcontroller, 
                              uint8_t memory_number)
{
    uint32_t a;
    uint32_t old;

    if (memory_number > 7){
        return NF_ERR_INVALID_PARAMETER;
    }

    pcontroller->ce = memory_number;

    old = a = IORD_32NF(pcontroller->reg_offset + NF_SFR_MEM_CTRL);
    a = (a & 0xFFFFFFF8) | ((uint32_t)memory_number & 0x7);
    IOWR_32NF(pcontroller->reg_offset + NF_SFR_MEM_CTRL, a);

    return NF_ERR_NO_ERRORS;
}
/*****************************************************************************/

/*****************************************************************************/
uint8_t nf_ctrl_set_write_protect(nf_controller_t *pcontroller, 
                                  uint8_t memory_number,
                                  uint8_t enable)
{
    uint32_t a;

    if (memory_number > 7){
        return NF_ERR_INVALID_PARAMETER;
    }

    a = IORD_32NF(pcontroller->reg_offset + NF_SFR_MEM_CTRL);

    if (enable){
        a &= ~(1uL << (memory_number + 8));
    }
    else {
        a |= (1uL << (memory_number + 8));
    }

    IOWR_32NF(pcontroller->reg_offset + NF_SFR_MEM_CTRL, a);

    return NF_ERR_NO_ERRORS;
}
/*****************************************************************************/                              

/*****************************************************************************/
uint8_t nf_ctrl_initialize(nf_controller_t *pcontroller)
{
    // set maximum timings as default
//    IOWR_32NF(pcontroller->reg_offset + NF_SFR_TIME_SEQ_0, 0xFFFFFFFF);
//    IOWR_32NF(pcontroller->reg_offset + NF_SFR_TIME_SEQ_1, 0xFFFFFFFF);
//    IOWR_32NF(pcontroller->reg_offset + NF_SFR_TIMINGS_ASYN, 0xFFFFFFFF);
//    IOWR_32NF(pcontroller->reg_offset + NF_SFR_TIMINGS_SYN, 0xFFFFFFFF);    

    // reset fifo
    IOWR_32NF(pcontroller->reg_offset + NF_SFR_FIFO_INIT, 1);

	// Need to give default "page_size" and "spare_size"
	pcontroller->page_size = 4096;
	pcontroller->block_size = 128;

#ifdef DMA_TRANSFER_MODE	
	pcontroller->transfer_mode = NF_TRANSFER_MODE_DMA_SFR;
	pcontroller->dma_burst_type = NF_SFR_DMA_CTRL_BURST;	// unspecify
#else
	pcontroller->transfer_mode = NF_TRANSFER_MODE_MANUAL;	// pio mode
#endif	


    pcontroller->ecc_enabled = 0;

    // disbale write protect to all memories
    IOWR_32NF(pcontroller->reg_offset + NF_SFR_MEM_CTRL, 0xFF00);

#if NF_ECC_ALGORITHM == 1
    pcontroller->ecc_size_per_unit = 4;
#elif NF_ECC_ALGORITHM == 4
    pcontroller->ecc_size_per_unit = 7;
#elif NF_ECC_ALGORITHM == 8
    pcontroller->ecc_size_per_unit = 13;
#elif NF_ECC_ALGORITHM == 16
    pcontroller->ecc_size_per_unit = 26;
#elif NF_ECC_ALGORITHM == 216

    // set correction ability to 4 bits
    set_ecc_ability(pcontroller, 4);
#endif

    // enable write protect interrupt
//    IOWR_32NF(pcontroller->reg_offset + NF_SFR_INT_MASK, 
//              (NF_SFR_INT_MASK_PROT_INT_EN | NF_SFR_INT_MASK_FIFO_ERROR));

    set_address_cycle_count(pcontroller, 0, 5);
    set_address_cycle_count(pcontroller, 1, 5);
    set_organization(pcontroller, 8);

    pcontroller->ce = 0;	// default, let it be 0

    return NF_ERR_NO_ERRORS;
}
/*****************************************************************************/

/*****************************************************************************/
uint8_t nf_ctrl_execute_command(nf_controller_t *pcontroller, 
                                nf_ctrl_command_t *command)
{
    uint8_t status;
    uint32_t ctrl;
    uint32_t interrupt_status;

    if ((command->command_code != NF_CMD_RESET) 
        && (command->command_code != NF_CMD_SYNCH_RESET)){
				// wait until controller is busy
        status = wait_for_value(pcontroller->reg_offset + NF_SFR_STATUS, 
                                NF_SFR_STATUS_CTRL_STAT, 20, 0);
        if (status) {
			printf( "[%s]:line:[%d], status = %d\n", __FUNCTION__, __LINE__, status);
            return status;
		}
    }

    IOWR_32NF(pcontroller->reg_offset + NF_SFR_DATA_SIZE, command->data_size);

    // check if comand should transfer some data
    if (command->data_size){
        ctrl = IORD_32NF(pcontroller->reg_offset + NF_SFR_CONTROL);

        // check if data_size is divisible by 4 
        if ((command->data_size & 0x3)
            || ((command->page_offset_0 + command->data_size) 
                > (pcontroller->page_size + pcontroller->spare_area_size))){

			printf( " Invalid parameter check.....................\n");
            return NF_ERR_INVALID_PARAMETER;
        }

        // check if command should be executed with hardware ECC support
        if ((command->ecc_mode == WITH_ECC) && (pcontroller->ecc_enabled == 1)){
            if (command->page_offset_0 || command->page_offset_1 
                || (command->data_size > pcontroller->ecc_offset)
                || (command->data_size < pcontroller->page_size))
                return NF_ERR_INVALID_PARAMETER;

            // if ecc module is wrong configured then return error
            if (pcontroller->ecc_status){
                return pcontroller->ecc_status;
            }
            ctrl = ctrl | NF_SFR_CONTROL_ECC_EN;
            ctrl &= ~NF_SFR_CONTROL_CUSTOM_DATA_SIZE_EN;

            if (command->data_size > pcontroller->page_size){
                ctrl |= NF_SFR_CONTROL_SPARE_EN;
                IOWR_32NF(pcontroller->reg_offset + NF_SFR_SPARE_SIZE
                          , (command->data_size - pcontroller->page_size));
            } else {
                ctrl &= ~NF_SFR_CONTROL_SPARE_EN;
            }

            IOWR_32NF(pcontroller->reg_offset + NF_SFR_CONTROL, ctrl);
        }
        else {
            ctrl |= NF_SFR_CONTROL_CUSTOM_DATA_SIZE_EN;
            ctrl &= ~(NF_SFR_CONTROL_SPARE_EN | NF_SFR_CONTROL_ECC_EN);
            IOWR_32NF(pcontroller->reg_offset + NF_SFR_CONTROL, ctrl);

            IOWR_32NF(pcontroller->reg_offset + NF_SFR_DATA_SIZE, 
                      command->data_size);
        }

        if ((pcontroller->transfer_mode != NF_TRANSFER_MODE_MANUAL) 
            && (command->data_size > 4)){

            // select the DMA module as input of the FIFO 
            command->command_code |= NF_SFR_COMMAND_INPUT_SEL_DMA;

#if USE_INDIRECT_BUFFER
#   if COPY_DATA_TO_IB            
            if (command->transfer_direction == NF_TRANSFER_DIRECTION_WRITE){
                memcpy(pcontroller->dma_buffer, command->buffer, 
                       command->data_size);
            }
#   endif            
            // prepare dma data transfer with help indirect buffer
            status = nf_dma_prepare_transfer(pcontroller, 
                                             pcontroller->dma_buff_address, 
                                             command->data_size, 
                                             command->transfer_direction);
#else
            status = nf_dma_prepare_transfer(pcontroller, 
                                             (uint32_t)command->buffer, 
                                             command->data_size, 
                                             command->transfer_direction);
#endif            
            if (status)
                return status;

        }
    }

	if (command->ecc_mode != WITH_ECC) {
        ctrl = IORD_32NF(pcontroller->reg_offset + NF_SFR_CONTROL);
        ctrl |= NF_SFR_CONTROL_CUSTOM_DATA_SIZE_EN;
        ctrl &= ~(NF_SFR_CONTROL_SPARE_EN | NF_SFR_CONTROL_ECC_EN);
        IOWR_32NF(pcontroller->reg_offset + NF_SFR_CONTROL, ctrl);
	}

    IOWR_32NF(pcontroller->reg_offset + NF_SFR_DATA_SIZE, command->data_size);

	 if(pcontroller->nandid == NAND_ID_ST_128W3A)
	 {
		 if(command->command_code == NF_CMD_ERASE_BLOCK_SEQ10)
		 {
			 IOWR_32NF(pcontroller->reg_offset + NF_SFR_ADDR0_L, (command->page_address_0) | command->page_offset_0);
			 //ted    IOWR_32NF(pcontroller->reg_offset + NF_SFR_ADDR0_H, command->page_address_0 >> 0);
			 IOWR_32NF(pcontroller->reg_offset + NF_SFR_ADDR1_L, (command->page_address_1) | command->page_offset_1);
			 //ted    IOWR_32NF(pcontroller->reg_offset + NF_SFR_ADDR1_H, command->page_address_1 >> 0);
		 }
		 else
		 {
			 IOWR_32NF(pcontroller->reg_offset + NF_SFR_ADDR0_L, (command->page_address_0 << 8) | command->page_offset_0);
			 //ted    IOWR_32NF(pcontroller->reg_offset + NF_SFR_ADDR0_H, command->page_address_0 >> 0);
			 IOWR_32NF(pcontroller->reg_offset + NF_SFR_ADDR1_L, (command->page_address_1 << 8) | command->page_offset_1);
			 //ted    IOWR_32NF(pcontroller->reg_offset + NF_SFR_ADDR1_H, command->page_address_1 >> 0);
		 }
	 }
	else
	{
	    // address 0
	    IOWR_32NF(pcontroller->reg_offset + NF_SFR_ADDR0_L,
	              (command->page_address_0 << 16) | command->page_offset_0);
	    IOWR_32NF(pcontroller->reg_offset + NF_SFR_ADDR0_H,
	              command->page_address_0 >> 16);

	    // address 1
	    IOWR_32NF(pcontroller->reg_offset + NF_SFR_ADDR1_L,
	              (command->page_address_1 << 16) | command->page_offset_1);
	    IOWR_32NF(pcontroller->reg_offset + NF_SFR_ADDR1_H,
	              command->page_address_1 >> 16);
	}

    // Command
    IOWR_32NF(pcontroller->reg_offset + NF_SFR_COMMAND, command->command_code);

    // check if write protection error appears
    interrupt_status = IORD_32NF(pcontroller->reg_offset + NF_SFR_INT_STATUS);   
    if (interrupt_status & NF_SFR_INT_STATUS_PROT_INT_FL){
        // clear write protection error
        IOWR_32NF(pcontroller->reg_offset + NF_SFR_INT_STATUS, 0);
        // reset FIFO
        IOWR_32NF(pcontroller->reg_offset + NF_SFR_FIFO_INIT, 1);
		printf ( " Error in NF_ERR_PROTECTED .............\n");
        return NF_ERR_PROTECTED;
    }

    if (command->data_size){
        if ((pcontroller->transfer_mode != NF_TRANSFER_MODE_MANUAL) 
            && (command->data_size > 4)){ // data transmission in DMA mode

            if (command->transfer_direction == NF_TRANSFER_DIRECTION_READ){
                // wait until NAND flash memory device is busy
                status = wait_for_value(pcontroller->reg_offset + NF_SFR_STATUS, 
                                        NF_SFR_STATUS_MEMX_RDY(pcontroller->ce), 500, 1);

                if(status) {
					uint8_t loop = 10;
					uint8_t buf[20];
					uint8_t *p = buf;

					while(status && (loop > 0)) {
							// one more chance 
							p += sprintf(p, "+");
							status = wait_for_value(pcontroller->reg_offset + NF_SFR_STATUS, 
                                        NF_SFR_STATUS_MEMX_RDY(pcontroller->ce), 500, 1);
							loop--;
					}

					if (loop < 8) {
						p += sprintf(p, "\n");
//						printf("[%s]: %s", __FUNCTION__, buf);
					}
					if (status)
	                    return status;
				}

                // wait for finish DMA transfer and check status
                status = nf_dma_finish_transfer(pcontroller);
                if (status)
                    return status;                
#if USE_INDIRECT_BUFFER
#   if COPY_DATA_TO_IB                 
                // copy read data to buffer
                memcpy(command->buffer, pcontroller->dma_buffer, 
                       command->data_size);
#   endif
#endif            
            }
            else {
                // wait for finish DMA transfer and check status
                status = nf_dma_finish_transfer(pcontroller);
                if (status)
                    return status;
            }
        }
        else { // data transmission in manual mode
            if (command->transfer_direction == NF_TRANSFER_DIRECTION_READ){
                status = wait_until_controller_busy(pcontroller);
                if (status)
                    return status;  
            }

            status = pio_data_transfer(pcontroller, command->buffer, 
                                       command->data_size, 
                                       command->transfer_direction);
            if (status)
                return status;
        }
    }

    return NF_ERR_NO_ERRORS;
}
/*****************************************************************************/

/*****************************************************************************/
uint8_t nf_ctrl_execute_universal_command(nf_controller_t *pcontroller, 
                                          nf_ctrl_universal_command_t *command)
{
    uint32_t generic_sequence = 0;
    uint32_t generic_command_code = 0;
    nf_ctrl_command_t generic_command;

    if (command->flags & NF_CTRL_UNIV_CMD_CMD0_EN){
        generic_sequence |= NF_SFR_GENERIC_SEQ_CTRL_CMD0_EN; 
    }                            	
    if (command->flags & NF_CTRL_UNIV_CMD_CMD1_EN){
        generic_sequence |= NF_SFR_GENERIC_SEQ_CTRL_CMD1_EN; 
    }
    if (command->flags & NF_CTRL_UNIV_CMD_CMD2_EN){
        generic_sequence |= NF_SFR_GENERIC_SEQ_CTRL_CMD2_EN; 
    }
    if (command->flags & NF_CTRL_UNIV_CMD_CMD3_EN){
        generic_sequence |= NF_SFR_GENERIC_SEQ_CTRL_CMD3_EN;
        generic_sequence |= 
            NF_SFR_GENERIC_SEQ_CTRL_SET_CMD3_CODE(command->cmd3);
    }
    if (command->flags & NF_CTRL_UNIV_CMD_ADDR0_EN){
        generic_sequence |= NF_SFR_GENERIC_SEQ_CTRL_ADDR0_EN; 
    }
    if (command->flags & NF_CTRL_UNIV_CMD_ADDR1_EN){
        generic_sequence |= NF_SFR_GENERIC_SEQ_CTRL_ADDR1_EN; 
    }
    if (command->flags & NF_CTRL_UNIV_CMD_COL_ADDR_EN){
        generic_sequence |= NF_SFR_GENERIC_SEQ_CTRL_COL_ADDR_EN; 
    }


    switch(command->busy_delays){
    case NF_CTRL_UNIV_CMD_DELLAY_BOTH_DISABLED:
        generic_sequence |= NF_SFR_GENERIC_SEQ_CTRL_DISABLE_BOTH_DELAYS;
        break;
    case NF_CTRL_UNIV_CMD_DELLAY_0_ENABLED:
        generic_sequence |= NF_SFR_GENERIC_SEQ_CTRL_ENABLE_DELAY_0;
        break;
    case NF_CTRL_UNIV_CMD_DELLAY_1_ENABLED:
        generic_sequence |= NF_SFR_GENERIC_SEQ_CTRL_ENABLE_DELAY_1;
        break;
    default:
        return NF_ERR_INVALID_PARAMETER;
    }

    if (command->data_size){
        generic_sequence |= NF_SFR_GENERIC_SEQ_CTRL_DATA_EN;    
    }

    if (command->transfer_direction == NF_TRANSFER_DIRECTION_WRITE){
        generic_command_code = 
            (NF_SFR_COMMAND_CMD(command->cmd0, command->cmd1, command->cmd2) 
             | SEQ_19);
    }
    else {
        generic_command_code = 
            (NF_SFR_COMMAND_CMD(command->cmd0, command->cmd1, command->cmd2) 
             | SEQ_18);
    } 

    NF_INIT_CMD_WITH_2ADDR_AND_DATA(generic_command, generic_command_code, 
                                    command->page_address_0, 
                                    command->page_offset_0,  
                                    command->page_address_1, 
                                    command->page_offset_1,
                                    command->buffer, command->data_size, 
                                    command->transfer_direction, 
                                    command->ecc_mode);

    IOWR_32NF(pcontroller->reg_offset + NF_SFR_GENERIC_SEQ_CTRL, 
              generic_sequence); 
    return nf_ctrl_execute_command(pcontroller, &generic_command);
}
/*****************************************************************************/

/*****************************************************************************/
uint8_t nf_ctrl_disable_protected_area(nf_controller_t *pcontroller)
{
    uint32_t ctrl;

    // disable protect area mechanism
    ctrl = IORD_32NF(pcontroller->reg_offset + NF_SFR_CONTROL);
    ctrl &= ~NF_SFR_CONTROL_PROT_EN;
    IOWR_32NF(pcontroller->reg_offset + NF_SFR_CONTROL, ctrl); 

    return NF_ERR_NO_ERRORS;      
}
/*****************************************************************************/                                   

/******************************************************************************/
uint8_t nf_ctrl_set_protected_area(nf_controller_t *pcontroller, 
                                   uint16_t begin_protect_area, 
                                   uint16_t end_protect_area)
{  
    uint32_t protect_area = 
        ((uint32_t)begin_protect_area) | ((uint32_t)end_protect_area << 16);

    uint32_t ctrl;

    // set protect area
    IOWR_32NF(pcontroller->reg_offset + NF_SFR_PROTECT, protect_area);

    // enable protect area mechanism
    ctrl = IORD_32NF(pcontroller->reg_offset + NF_SFR_CONTROL);
    ctrl |= NF_SFR_CONTROL_PROT_EN;
    IOWR_32NF(pcontroller->reg_offset + NF_SFR_CONTROL, ctrl);

    return NF_ERR_NO_ERRORS;
}
/******************************************************************************/

/******************************************************************************/
uint8_t nf_ctrl_get_protected_area(nf_controller_t *pcontroller, 
                                   uint16_t *begin_protect_area, 
                                   uint16_t *end_protect_area)
{
    uint32_t protect_area;

    protect_area = IORD_32NF(pcontroller->reg_offset + NF_SFR_PROTECT); 
    *begin_protect_area = (uint16_t)(protect_area) & 0xFFFF;
    *end_protect_area = (uint16_t)(protect_area >> 16) & 0xFFFF;

    return NF_ERR_NO_ERRORS;
}
/******************************************************************************/

/******************************************************************************/
uint8_t nf_ctrl_lookup_set(nf_controller_t *pcontroller, 
                           uint16_t logical_address, 
                           uint16_t physical_address)
{
    uint8_t i = 0;
    uint32_t look_enable 
        = IORD_32NF(pcontroller->reg_offset + NF_SFR_LOOKUP_EN);

    uint32_t ctrl;
    uint32_t look_up_table[] = {
        NF_SFR_LOOKUP0, NF_SFR_LOOKUP1, NF_SFR_LOOKUP2, NF_SFR_LOOKUP3,
        NF_SFR_LOOKUP4, NF_SFR_LOOKUP5, NF_SFR_LOOKUP6, NF_SFR_LOOKUP7
    };

    ctrl = IORD_32NF(pcontroller->reg_offset + NF_SFR_CONTROL);
    ctrl |= NF_SFR_CONTROL_FLLOOKUP_EN;
    IOWR_32NF(pcontroller->reg_offset + NF_SFR_CONTROL, ctrl);

    if((logical_address & 0x8000) || (physical_address & 0x8000))
        return NF_ERR_INVALID_PARAMETER;

    for(i = 0; i < 8; i++)
        if ((look_enable & (1 << i)) == 0){      
            IOWR_32NF(pcontroller->reg_offset + look_up_table[i], 
                      ((uint32_t)physical_address << 16) | logical_address);

            look_enable |= (1 << i);
            IOWR_32NF(pcontroller->reg_offset + NF_SFR_LOOKUP_EN, look_enable);            
            return NF_ERR_NO_ERRORS;
        }

    return NF_ERR_NO_LOOKUP_TABLE_FULL;
}
/******************************************************************************/

/******************************************************************************/
uint8_t nf_ctrl_lookup_release(nf_controller_t *pcontroller, 
                               uint16_t logical_address)
{
    uint8_t i;
    uint32_t look_en = IORD_32NF(pcontroller->reg_offset + NF_SFR_LOOKUP_EN);   
    uint32_t lookup_table[] = {
        NF_SFR_LOOKUP0, NF_SFR_LOOKUP1, NF_SFR_LOOKUP2, NF_SFR_LOOKUP3,
        NF_SFR_LOOKUP4, NF_SFR_LOOKUP5, NF_SFR_LOOKUP6, NF_SFR_LOOKUP7
    };

    for(i = 0; i < 8; i++)
        if ((look_en & (1 << i)) && 
            ((IORD_32NF(pcontroller->reg_offset + lookup_table[i]) & 0xFFFF) 
             == logical_address)){

            IOWR_32NF(pcontroller->reg_offset + lookup_table[i], 0);

            look_en &= ~(1 << i);
            IOWR_32NF(pcontroller->reg_offset + NF_SFR_LOOKUP_EN, look_en);

            return NF_ERR_NO_ERRORS;
        }

    return NF_ERR_INVALID_ADDRESS;
}
/******************************************************************************/

/******************************************************************************/
uint8_t nf_ctrl_lookup_release_all(nf_controller_t *pcontroller)
{
    uint8_t i;
    uint32_t lookup_table[] = {
        NF_SFR_LOOKUP0, NF_SFR_LOOKUP1, NF_SFR_LOOKUP2, NF_SFR_LOOKUP3,
        NF_SFR_LOOKUP4, NF_SFR_LOOKUP5, NF_SFR_LOOKUP6, NF_SFR_LOOKUP7
    };

    uint32_t ctrl = IORD_32NF(pcontroller->reg_offset + NF_SFR_CONTROL);
    ctrl &= ~NF_SFR_CONTROL_FLLOOKUP_EN;
    IOWR_32NF(pcontroller->reg_offset + NF_SFR_CONTROL, ctrl);

    for(i = 0; i < 8; i++){
        IOWR_32NF(pcontroller->reg_offset + lookup_table[i], 0);
    }

    IOWR_32NF(pcontroller->reg_offset + NF_SFR_LOOKUP_EN, 0);

    return NF_ERR_NO_ERRORS;
}
/******************************************************************************/


/******************************************************************************/
uint8_t nf_ctrl_configure_timings(nf_controller_t *pcontroller,
                                  nf_timings_t *timings)
{
//    uint32_t thclk = 1000000000 / NF_SYTEM_CLK_HZ;
    uint32_t thclk = 1000000000 / halSysGetDeviceClockSrc(CONFIG_SYS_INPUT_CLOCK, HAL_MFP_NAND, NULL);
    uint8_t tccs, tadl, trhw, twhr, trwh, trwp, tcad, trr, twb;

    tccs = (uint8_t)(timings->tccs_ns / thclk);
    if ((thclk * tccs) < timings->tccs_ns)
        tccs++;

    tadl = (uint8_t)(timings->tadl_ns / thclk);
    if ((thclk * tadl) < timings->tadl_ns)
        tadl++;

    trhw = (uint8_t)(timings->trhw_ns / thclk);
    if ((thclk * trhw) < timings->trhw_ns)
        trhw++;

    twhr = (uint8_t)(timings->twhr_ns / thclk);
    if ((thclk * twhr) < timings->twhr_ns)
        twhr++;


    trwh = (uint8_t)(timings->trwh_ns / thclk);
    if ((thclk * trwh) < timings->trwh_ns)
        trwh++;

    trwp = (uint8_t)(timings->trwp_ns / thclk);
    if ((thclk * trwp) < timings->trwp_ns)
        trwp++;

    tcad = (uint8_t)(timings->tcad_ns / thclk);
    if ((thclk * tcad) < timings->tcad_ns)
        tcad++;

    trr = (uint8_t)(timings->trr_ns / thclk);
    if ((thclk * trr) < timings->trr_ns)
        trr++;

    twb = (uint8_t)(timings->twb_ns / thclk);
    if ((thclk * twb) < timings->twb_ns)
        twb++;


    IOWR_32NF(pcontroller->reg_offset + NF_SFR_TIME_SEQ_0,
              ((uint32_t)twhr << 24) | ((uint32_t)trhw << 16)
              | ((uint32_t)tadl << 8) | tccs);
    IOWR_32NF(pcontroller->reg_offset + NF_SFR_TIME_SEQ_1,
              ((uint32_t)trr << 9) | twb);
    IOWR_32NF(pcontroller->reg_offset + NF_SFR_TIMINGS_ASYN, (trwh << 4) | trwp);
    IOWR_32NF(pcontroller->reg_offset + NF_SFR_TIMINGS_SYN, tcad);

    return NF_ERR_NO_ERRORS;
}
/******************************************************************************/
