/*
 * (C) Copyright 2002
 * Sysgo Real-Time Solutions, GmbH <www.elinos.com>
 * Marius Groeger <mgroeger@sysgo.de>
 *
 * (C) Copyright 2002
 * Gary Jennejohn, DENX Software Engineering, <gj@denx.de>
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

/*
 * CPU specific code
 */
#include <config.h>
#include <common.h>
#include <command.h>
#include <asm/system.h>
#include <asm/sizes.h>
#include <asm/io.h>
#include <asm/arch/hardware.h>

DECLARE_GLOBAL_DATA_PTR;
/***************************************************************************************************
 *  Backwards compatible PAGE descriptor.(CP15 register c1 bit23 is set to 0.)
 *
 *          |----------------------------------------------------------------------
 *  Section |Section Base Address | NS | 0 | SBZ  | TEX | AP | P |DOMAIN|0|C|B|1|0|
 *          |----------------------------------------------------------------------
 *         31                    20 19  18  17~15  14~12 11~10 9   8~5   4 3 2 1 0 bits field
 ****************************************************************************************************/


int MFP_STS = -1;

#define __A1_writel		writel
#define __A1_readl		readl
#define A1_DDR_REMAP	__A1_writel(0x300, 0x9000008C)

#define CACHE_SETUP_CB		0x1e
#define CACHE_SETUP_RANGE_SIZE	SZ_64M

static void cp_delay (void)
{
	volatile int i;

	/* copro seems to need some delay between reading and writing */
	for (i = 0; i < 100; i++)
		nop();
}

static inline void A1_dram_bank_mmu_setup(int bank)
{
	u32 *page_table = (u32 *)gd->tlb_addr;
	bd_t *bd = gd->bd;
	int	i;

	//debug("%s: bank: %d\n", __func__, bank);

	/* Cacheable & bufferable */
	for (i = bd->bi_dram[bank].start >> 20;
	     i < (bd->bi_dram[bank].start + CACHE_SETUP_RANGE_SIZE) >> 20;
	     i++) {
		page_table[i] = i << 20 | (3 << 10) | CACHE_SETUP_CB;
	}

	/* Remap TEXT_BASE to 0x0, size 1M, Cacheable & bufferable */
	page_table[0] = TEXT_BASE | (3 << 10) | CACHE_SETUP_CB;
}

/*
 * Check Boot Jumper Status
 * 		read 0x9000002c
 *		0x200, NOR boot
 *		0, NAND boot
 *
 * retrun
 *		1 : boot from NAND
 *		0 : boot from NOR
 */
#define JUMPER_SET	0x2C
inline int A1_boot_from_nand(void)
{
	u32 reg;

	reg = readl(HAL_REG_SYS_BASE + JUMPER_SET);
	return (reg & (1 << 9)) ? 0 : 1;
}

/* to activate the MMU we need to set up virtual memory: use 1M areas */
static inline void A1_mmu_setup(void)
{
	u32 *page_table = (u32 *)gd->tlb_addr;
	int i;
	u32 reg;

	FTrace("enter");

	/* Set up an identity-mapping for all 4GB, rw for everyone */
	for (i = 0; i < 4096; i++)
		page_table[i] = i << 20 | (3 << 10) | 0x12;

	for (i = 0; i < CONFIG_NR_DRAM_BANKS; i++) {
		A1_dram_bank_mmu_setup(i);
	}

	/* Copy the page table address to cp15 */
	asm volatile("mcr p15, 0, %0, c2, c0, 0"
		     : : "r" (page_table) : "memory");
	/* Set the access control to all-supervisor */
	asm volatile("mcr p15, 0, %0, c3, c0, 0"
		     : : "r" (~0));
	/* and enable the mmu */
	reg = get_cr();	/* get control reg. */
	cp_delay();
	set_cr(reg | CR_M);
}

#define AHB_GATE_CTRL	0x24
#define nandc_hclk_gate	(1 << 3)
#define nandc_clkb_gate	(1 << 4)
#define lmc_hclk_gate	(1 << 13)
#define ENABLE_NANDC_CLK	(nandc_hclk_gate | nandc_clkb_gate)
#define ENABLE_LMC_CLK		lmc_hclk_gate

#ifdef CONFIG_ARCH_CPU_INIT
int arch_cpu_init (void)
{
	u32 val;

#ifdef CONFIG_USE_IRQ
	u32 add = TEXT_BASE;

	add = add - (128 + CONFIG_SYS_MALLOC_LEN + CONFIG_SYS_GBL_DATA_SIZE);

	IRQ_STACK_START = add;
	FIQ_STACK_START = (add - CONFIG_STACKSIZE_IRQ);
#endif

	/* Initailize TLB address */
	gd->tlb_addr = CONFIG_A1_TLB_ADDR;
	dram_init();
	vic_init();

	/* Check and turn on NAND clk gate */
	if (!A1_boot_from_nand()) {
		u32 reg = readl(HAL_REG_SYS_BASE + AHB_GATE_CTRL);
		reg |= ENABLE_NANDC_CLK;
		reg |= ENABLE_LMC_CLK;
		writel (reg, HAL_REG_SYS_BASE + AHB_GATE_CTRL);
	}

	/* CPU turbo mode register enable */
	val = __A1_readl(0x9910000C);
	__A1_writel (val | 0x80ul, 0x9910000C);

	/* Remap DDR_BASE to 0x0 */
	//A1_DDR_REMAP;

	/*
	 * Enable Cache.
	 * 1. Enable I-Cache
	 * 2. Fill Page table
	 * 3. Enable D-CACHE MMU.
	 */
	icache_enable();
	A1_mmu_setup();

	dcache_enable();
	return 0;
}
#endif /* CONFIG_ARCH_CPU_INIT */

unsigned int GET_APBCLOCK()
{
	return PLL_GetAPBClock(CONFIG_SYS_INPUT_CLOCK);
}

/*
 * Initializes on-chip MMC controllers.
 * to override, implement board_mmc_init()
 */
int cpu_mmc_init (bd_t * bis)
{
	FTrace("enter");
#ifdef CONFIG_ACTI_A1_MMC
	return mmc_legacy_init (bis);
#else
	return 0;
#endif
}

void flush_dcache_range(ulong start_addr, ulong stop)
{
	flush_cache(start_addr, stop);
}

void invalidate_dcache_range(ulong start_addr, ulong stop)
{
	flush_cache(start_addr, stop);
}
