/*
 * ALSA SoC I2S (McBSP) Audio Layer for UMVP I2S
 * Author:      Betta Lin, <Betta.Lin@acti.com>
 * Copyright:   (C) 2010 ACTI, Inc., 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#include <linux/init.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/delay.h>
#include <linux/io.h>
//#include <linux/clk.h>

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/initval.h>
#include <sound/soc.h>

#include <mach/hardware.h>

#include <asm/arch-umvp/a1_sys.h>

#include "umvp-pcm.h"
#include "umvp-i2s.h"


static struct umvp_pcm_dma_params umvp_i2s_pcm_out = {
	.name = "I2S PCM Stereo out",
};

static struct umvp_pcm_dma_params umvp_i2s_pcm_in = {
	.name = "I2S PCM Stereo in",
};

struct umvp_dai_dev {
	void __iomem			*base;
	struct clk			*clk;
	struct umvp_pcm_dma_params	*dma_params[2]; /* For SNDRV_PCM_STREAM_PLAYBACK/SNDRV_PCM_STREAM_CAPTURE */
};

static inline void umvp_write_reg(struct umvp_dai_dev *dev,
					   int reg, u32 val)
{
	__raw_writel(val, dev->base + reg);
}

static inline u32 umvp_read_reg(struct umvp_dai_dev *dev, int reg)
{
	return __raw_readl(dev->base + reg);
}

static void umvp_dai_start(struct snd_pcm_substream *substream)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct umvp_dai_dev *dev = rtd->dai->cpu_dai->private_data;
	struct snd_soc_device *socdev = rtd->socdev;
	struct snd_soc_platform *platform = socdev->card->platform;
	int ret;
	

	/* Start the sample generator and enable transmitter/receiver */

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		
#if 0		
		/* Stop the DMA to avoid data loss */
		if (platform->pcm_ops->trigger) {
			ret = platform->pcm_ops->trigger(substream,
				SNDRV_PCM_TRIGGER_STOP);
			if (ret < 0)
				printk(KERN_DEBUG "Playback DMA stop failed\n");
		}
#endif
		/* Enable the transmitter */
		umvp_write_reg(dev, DAI_INIT, umvp_read_reg(dev, DAI_INIT) | (1 << 2) );
		
#if 0
		/* Restart the DMA */
		if (platform->pcm_ops->trigger) {
			ret = platform->pcm_ops->trigger(substream,
				SNDRV_PCM_TRIGGER_START);
			if (ret < 0)
				printk(KERN_DEBUG "Playback DMA start failed\n");
		}
#endif

	} else {

		/* Enable the reciever */
		umvp_write_reg(dev, DAI_INIT, umvp_read_reg(dev, DAI_INIT) | (1 << 1) );
	}
	
	/* Restart the DMA */
	if (platform->pcm_ops->trigger) {
		ret = platform->pcm_ops->trigger(substream,
			SNDRV_PCM_TRIGGER_START);
		if (ret < 0)
			printk(KERN_DEBUG "Playback DMA start failed\n");
	}

}

static void umvp_dai_stop(struct snd_pcm_substream *substream)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct umvp_dai_dev *dev = rtd->dai->cpu_dai->private_data;
	
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) 
	/* Disable the transmitter */
		umvp_write_reg(dev, DAI_INIT, umvp_read_reg(dev, DAI_INIT) & ~0x04 );
	else
	{
		
		umvp_write_reg(dev, DAI_INIT, umvp_read_reg(dev, DAI_INIT) & ~0x02 );
		do {
			umvp_read_reg(dev, DAI_RX_DATA);
			udelay(2);
		} while ( umvp_read_reg(dev, DAI_RX_FIFO_CNTR) );
	}	
	
}

static int umvp_i2s_startup(struct snd_pcm_substream *substream,
			       struct snd_soc_dai *dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *cpu_dai = rtd->dai->cpu_dai;
	struct umvp_dai_dev *dev = rtd->dai->cpu_dai->private_data;
	
	cpu_dai->dma_data = dev->dma_params[substream->stream];
	
	return 0;
}

#define DEFAULT_BITPERSAMPLE	16

static int umvp_i2s_set_dai_fmt(struct snd_soc_dai *cpu_dai,
				   unsigned int fmt)
{
	struct umvp_dai_dev *dev = cpu_dai->private_data;
	unsigned int clk_ctl = umvp_read_reg(dev, DAI_CLOCK);

	switch (fmt & SND_SOC_DAIFMT_MASTER_MASK) {
	case SND_SOC_DAIFMT_CBS_CFS:

		/* Ted added */
		umvp_write_reg(dev, DAI_INIT, umvp_read_reg(dev, DAI_INIT) | 1 );
		mdelay(1);
		umvp_write_reg(dev, DAI_INIT, umvp_read_reg(dev, DAI_INIT) &0xFFFFFFFE );
		mdelay(1);

		/* cpu is master; codec clk & frame slave */
	
		//RX control register : set as master mode
		umvp_write_reg(dev, DAI_RX_CTRL, umvp_read_reg(dev, DAI_RX_CTRL) | (1 << 1));
		//TX control register : set as master mode
		umvp_write_reg(dev, DAI_TX_CTRL, (1<<3) | umvp_read_reg(dev, DAI_TX_CTRL) | (1<<5) );

		//clock control register : set as master mode
//ted	clk_ctl |= (1 << 12) | (1 << 11) | (1 << 10) | (1 << 9);
		clk_ctl |= (1 << 12) + (1 << 11) + (1 << 10) + (1 << 9) + (0 << 8);
		umvp_write_reg(dev, DAI_CLOCK, clk_ctl);
		break;
		
	case SND_SOC_DAIFMT_CBM_CFM: //betta

		/* Ted added */
		umvp_write_reg(dev, DAI_INIT, umvp_read_reg(dev, DAI_INIT) | 1 );
		mdelay(1);
		umvp_write_reg(dev, DAI_INIT, umvp_read_reg(dev, DAI_INIT) &0xFFFFFFFE );
		mdelay(1);


		/* cpu is slave; codec clk & frame master */
		
		//RX control register : set as slave mode
		umvp_write_reg(dev, DAI_RX_CTRL, umvp_read_reg(dev, DAI_RX_CTRL) & (~(1 << 1)));
		//TX control register : set as slave mode
		umvp_write_reg(dev, DAI_TX_CTRL,  (1<<3) | (umvp_read_reg(dev, DAI_TX_CTRL) & (~(1 << 5))) );

#if 0 //ted
		//clock control register : set as slave mode
		clk_ctl |= (1 << 10);
		clk_ctl &= ~((1 << 11) | (1 << 12) | (1 << 9)) ;
		umvp_write_reg(dev, DAI_CLOCK, clk_ctl);
		mdelay(1);

#else
		clk_ctl = (0 << 12) + (0 << 11) + (0 << 10) + (0 << 9) + (0 << 8);
		umvp_write_reg(dev, DAI_CLOCK, clk_ctl | (1 << 10)); //ted: should set bit 10 to 1. why???
		mdelay(1);
#endif
		umvp_write_reg(dev, DAI_CLOCK, clk_ctl);
		break;
	default:
		printk(KERN_ERR "%s:bad master\n", __func__);
		return -EINVAL;
	}

	switch (fmt & SND_SOC_DAIFMT_FORMAT_MASK)
	{
		/* I2S mode */
		case SND_SOC_DAIFMT_I2S:
			umvp_write_reg(dev, DAI_MODE, 4);
			break;
		/* Right Justified mode */
		case SND_SOC_DAIFMT_RIGHT_J:
		/* Left Justified mode */
		case SND_SOC_DAIFMT_LEFT_J:
	
		default:
			printk(KERN_ERR "%s:bad format\n", __func__);
			printk("fmt = %d\n", fmt);
			return -EINVAL;
	}
	
	
	return 0;
}

static int umvp_i2s_dai_set_dai_codec_clkdiv(struct snd_soc_dai *cpu_dai, int div_id, int div)
{
	struct umvp_dai_dev *dev = cpu_dai->private_data;

	if (div_id == UMVP_CODEC_CLK_DIV_ON)
	{
		umvp_write_reg(dev, DAI_CLOCKOUT_CTRL, 3);	//bit1 value always be 1
		umvp_write_reg(dev, DAI_CLOCKOUT_DIV, div-1);
	}
	if (div_id == UMVP_CODEC_CLK_DIV_OFF)
	{
		umvp_write_reg(dev, DAI_CLOCKOUT_CTRL, 0);	//bit1 value always be 1
		umvp_write_reg(dev, DAI_CLOCKOUT_DIV, 1);
	}
	else
		return -ENODEV;
	return 0;
}
					 


static int umvp_i2s_hw_params(struct snd_pcm_substream *substream,
				 struct snd_pcm_hw_params *params,
				 struct snd_soc_dai *dai)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct umvp_pcm_dma_params *dma_params = rtd->dai->cpu_dai->dma_data;
	struct umvp_dai_dev *dev = rtd->dai->cpu_dai->private_data;
	unsigned long rate = params_rate(params);
	unsigned int clk_ctl = umvp_read_reg(dev, DAI_CLOCK); 
	unsigned int divider, clk;


	switch (params_channels(params)) {
	case 1:
		/* TX Mono */
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK)
		{
			umvp_write_reg(dev, DAI_TX_CTRL, 0x01 | umvp_read_reg(dev, DAI_TX_CTRL) );
		}
		else/* RX Mono */
		{
			umvp_write_reg(dev, DAI_RX_CTRL, 0x10 | umvp_read_reg(dev, DAI_RX_CTRL));
		}
			
		break;
		
	case 2:
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) /* TX Stereo */
		{
			umvp_write_reg(dev, DAI_TX_CTRL, (~0x03) & umvp_read_reg(dev, DAI_TX_CTRL) );
		}
		else
		{
			umvp_write_reg(dev, DAI_RX_CTRL, (~0x10) & umvp_read_reg(dev, DAI_RX_CTRL));
		}
		break;
	default:
		printk(KERN_WARNING "umvp-i2s: unsupported channel numbers\n");
		return -EINVAL;
	}

	/*
	 * clock control register (used in I2S bus)
	 */
	halSysPLLIsMaster(1);

//	printk("===========I2S set clock======================\n");
	divider = PLL_GetFout(1, A1_SYS_CLOCK_SRC) / (12288 * 1000);
	clk = halSysSetDeviceClockDivider(A1_SYS_CLOCK_SRC, HAL_MFP_I2S, divider);
	umvp_write_reg(dev, DAI_CLOCK, clk_ctl | ((clk/(params_rate(params)*32*2) - 1) & 0xFF) );

	/* Determine xfer data type */
	switch (params_format(params)) {
#if 0
	case SNDRV_PCM_FORMAT_S8:
	
	printk("SNDRV_PCM_FORMAT_S8\n");
	case SNDRV_PCM_FORMAT_U8:
	printk("SNDRV_PCM_FORMAT_U8\n");
		umvp_write_reg(dev, DAI_WLEN, 8 - 1);
		break;
#endif
	case SNDRV_PCM_FORMAT_S16_LE:
	
	/* 2 bytes */
		dma_params->data_type = 2; 
	/*
	 * word length register : 
	 *    n-bit precision
	 */
		umvp_write_reg(dev, DAI_WLEN, 16 - 1);
	/*
	 * word position register :
	 *     16 - precision
	 */
		umvp_write_reg(dev, DAI_WPOS, 0);
		break;
#if 0
/* 4 bytes */
	case SNDRV_PCM_FORMAT_S32_LE:
		dma_params->data_type = 4;
		
		
		break;
#endif
	default:
		printk(KERN_WARNING "umvp-i2s: unsupported PCM format\n");
		return -EINVAL;
	}

	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
		/*
	 * TX FIFO threshold
	 */
		umvp_write_reg(dev, DAI_TX_FIFO_LTH, 63);

	} else {
		/*
	 * RX FIFO threshold
	 */
	umvp_write_reg(dev, DAI_RX_FIFO_GTH, 1);

	}
	return 0;
}

static int umvp_i2s_trigger(struct snd_pcm_substream *substream, int cmd,
			       struct snd_soc_dai *dai)
{
	int ret = 0;

	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
	case SNDRV_PCM_TRIGGER_RESUME:
	case SNDRV_PCM_TRIGGER_PAUSE_RELEASE:

		umvp_dai_start(substream);
		break;
	case SNDRV_PCM_TRIGGER_STOP:
	case SNDRV_PCM_TRIGGER_SUSPEND:
	case SNDRV_PCM_TRIGGER_PAUSE_PUSH:
		umvp_dai_stop(substream);
		break;
	default:
		ret = -EINVAL;
	}

	return ret;
}

static int umvp_i2s_probe(struct platform_device *pdev,
			     struct snd_soc_dai *dai)
{
	struct snd_soc_device *socdev = platform_get_drvdata(pdev);
	struct snd_soc_card *card = socdev->card;
	struct snd_soc_dai *cpu_dai = card->dai_link[pdev->id].cpu_dai;
	struct umvp_dai_dev *dev;
	struct resource *mem, *ioarea;
	//struct umvp_snd_platform_data *pdata;
	int ret;

#if 1
	mem = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!mem) {
		dev_err(&pdev->dev, "no mem resource?\n");
		return -ENODEV;
	}

	ioarea = request_mem_region(mem->start, (mem->end - mem->start) + 1,
				    pdev->name);
	if (!ioarea) {
		dev_err(&pdev->dev, "McBSP region already claimed\n");
		return -EBUSY;
	}
#endif
	dev = kzalloc(sizeof(struct umvp_dai_dev), GFP_KERNEL);
	if (!dev) {
		ret = -ENOMEM;
		goto err_release_region;
	}

	cpu_dai->private_data = dev;
	dev->dma_params[SNDRV_PCM_STREAM_PLAYBACK] = &umvp_i2s_pcm_out;
	dev->dma_params[SNDRV_PCM_STREAM_CAPTURE] = &umvp_i2s_pcm_in;

#if 0
	dev->clk = clk_get(&pdev->dev, "McBSPCLK");
	if (IS_ERR(dev->clk)) {
		ret = -ENODEV;
		goto err_free_mem;
	}
	clk_enable(dev->clk);


	dev->base = (void __iomem *)IO_ADDRESS(mem->start);
#else /* remap the region */
	
	//dev->base = ioremap_nocache(mem->start, mem->end - mem->start + 1);
	dev->base = (void __iomem *)IO_ADDRESS(mem->start);
	if (!dev->base) {
		ret = -ENXIO;
		dev_err(&pdev->dev, "Unable to ioremap DAi register IO.\n");
		goto err_free_mem;
	}

#endif
	//pdata = pdev->dev.platform_data;
	
	/*
	 * software reset
	 */
	umvp_write_reg(dev, DAI_INIT, 1);
	
	mdelay(1);
	umvp_write_reg(dev, DAI_INIT, 0);
	mdelay(1);

	return 0;

err_free_mem:
	kfree(dev);
err_release_region:
	
	release_mem_region(mem->start, (mem->end - mem->start) + 1);

	return ret;
}

static void umvp_i2s_remove(struct platform_device *pdev,
			       struct snd_soc_dai *dai)
{
	struct snd_soc_device *socdev = platform_get_drvdata(pdev);
	struct snd_soc_card *card = socdev->card;
	struct snd_soc_dai *cpu_dai = card->dai_link[pdev->id].cpu_dai;
	struct umvp_dai_dev *dev = cpu_dai->private_data;
	struct resource *mem;

#if 0
	clk_disable(dev->clk);
	clk_put(dev->clk);
	dev->clk = NULL;
#endif

	kfree(dev);

	//iounmap((void *)dev->base);
	mem = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	release_mem_region(mem->start, (mem->end - mem->start) + 1);

}

#define DAVINCI_I2S_RATES	SNDRV_PCM_RATE_8000_96000

struct snd_soc_dai umvp_i2s_dai = {
	.name = "umvp-i2s",
	.id = 0,
	.probe = umvp_i2s_probe,
	.remove = umvp_i2s_remove,
	.playback = {
		.channels_min = 1,
		.channels_max = 2,
		.rates = DAVINCI_I2S_RATES,
		.formats = SNDRV_PCM_FMTBIT_S16_LE,},
	.capture = {
		.channels_min = 1,
		.channels_max = 2,
		.rates = DAVINCI_I2S_RATES,
		.formats = SNDRV_PCM_FMTBIT_S16_LE,},
	.ops = {
		.startup = umvp_i2s_startup, /* soc_pcm_open call it. */
		.trigger = umvp_i2s_trigger,
		.hw_params = umvp_i2s_hw_params,
		.set_fmt = umvp_i2s_set_dai_fmt,
		.set_clkdiv = umvp_i2s_dai_set_dai_codec_clkdiv,
	},
};
EXPORT_SYMBOL_GPL(umvp_i2s_dai);

static int __init umvp_i2s_init(void)
{
	return snd_soc_register_dai(&umvp_i2s_dai);
}
module_init(umvp_i2s_init);

static void __exit umvp_i2s_exit(void)
{
	snd_soc_unregister_dai(&umvp_i2s_dai);
}
module_exit(umvp_i2s_exit);

MODULE_AUTHOR("Vladimir Barinov");
MODULE_DESCRIPTION("TI DAVINCI I2S (McBSP) SoC Interface");
MODULE_LICENSE("GPL");
