//================================================================================
// FILE NAME:      umvp_dai.c
//--------------------------------------------------------------------------------
// Reason-ID       Reason
//--------------------------------------------------------------------------------
// R:2013114-00:   Create
//================================================================================
/*
 * Please set your tab-stop value as 4.
 */
//#include <linux/config.h>
#include <linux/module.h>
#include <linux/init.h>

#include <linux/kernel.h>       /* printk() */
#include <linux/slab.h>         /* kmalloc() */
#include <linux/fs.h>           /* everything... */
#include <linux/errno.h>        /* error codes */
#include <linux/types.h>        /* size_t */
#include <linux/delay.h>        /* mdeley() msleep() */
#include <linux/interrupt.h>    /*tasklet  */
#include <linux/string.h>
#include <linux/ioctl.h>        /* for the _IOW etc */
#include <linux/wait.h>         /* wait queue head */

#include <linux/sound.h>
#include <asm/uaccess.h>        /* copy from user */
#include <asm/atomic.h>
#include <asm/io.h>
#include <mach/hardware.h>      /* IO_ADDRESS */
#include <mach/platform2500.h>
#include <asm/arch-umvp/dma.h>
#include <asm/arch-umvp/a1_sys.h>

#include "umvp_audio_codec.h"
#include "umvp_wm8940.h"

/* -----------------------------------------------------------------------
    define debug macro
 ----------------------------------------------------------------------- */
static u_int debug_ctrl = 1;
#define MSG_PFX         "[umvp_au]"
#define dprintk(num, format, args...) \
    do { \
        if (debug_ctrl >= num) \
            printk(MSG_PFX format, ##args); \
    } while (0)

/* -----------------------------------------------------------------------*/
#define A1AUDIO_VERSION       "V1.00.04"

/* Use 'S' as magic number */
#define A1AUDIO_IOC_MAGIC  's'
#define A1AUDIO_INONOFF         _IOW (A1AUDIO_IOC_MAGIC,  1, int)
#define A1AUDIO_OUTONOFF        _IOW (A1AUDIO_IOC_MAGIC,  2, int)
#define A1AUDIO_INVOLUME        _IOW (A1AUDIO_IOC_MAGIC,  3, int)
#define A1AUDIO_OUTVOLUME       _IOW (A1AUDIO_IOC_MAGIC,  4, int)
#define A1AUDIO_INPGAVOL        _IOW (A1AUDIO_IOC_MAGIC,  5, int)
#define A1AUDIO_MIC_TYPE        _IOW (A1AUDIO_IOC_MAGIC,  6, int)

/* ... more to come */
#define A1AUDIO_DAIRESET        _IO(A1AUDIO_IOC_MAGIC, 10) /* debugging tool */
#define A1AUDIO_IOC_MAXNR  10
/* -----------------------------------------------------------------------*/

/* number of read (also write) buffers can't exceed this number */
#define GUC_AUDIO_MAX_BUF_NUM   128
//#define DEFAULT_FRAG_NUM        64
//#define DEFAULT_FRAG_SIZE       1024

  #define DEFAULT_FRAG_NUM        64
  #define DEFAULT_FRAG_SIZE       1000

/* The frequency of the oscillator mounted on the GUC audio daughtor board.*/
  #define GUC_AUDIO_OSC 24576000

/* DAI control registers */
#define DAI_BASE        IO_ADDRESS(UMVP_DAI_BASE)
#define DAI_MODE                (DAI_BASE + 0x0)
#define DAI_RX_CTRL             (DAI_BASE + 0x4)
#define DAI_TX_CTRL             (DAI_BASE + 0x8)
#define DAI_WLEN                (DAI_BASE + 0xC)
#define DAI_WPOS                (DAI_BASE + 0x10)
#define DAI_SLOT                (DAI_BASE + 0x14)
#define DAI_TX_FIFO_LTH         (DAI_BASE + 0x18)
#define DAI_RX_FIFO_GTH         (DAI_BASE + 0x1C)
#define DAI_CLOCK               (DAI_BASE + 0x20)
#define DAI_INIT                (DAI_BASE + 0x24)
#define DAI_TX_FIFO_FLAG        (DAI_BASE + 0x28)
#define DAI_TX_LEFT_DATA        (DAI_BASE + 0x2C)
#define DAI_TX_RIGHT_DATA       (DAI_BASE + 0x30)
#define DAI_RX_FIFO_FLAG        (DAI_BASE + 0x34)
#define DAI_RX_DATA             (DAI_BASE + 0x38)
#define DAI_TX_FIFO_CNTR        (DAI_BASE + 0x3C)
#define DAI_RX_FIFO_CNTR        (DAI_BASE + 0x40)
#define DAI_CLOCKOUT_CTRL       (DAI_BASE + 0x4C)
#define DAI_CLOCKOUT_DIV        (DAI_BASE + 0x50)

#define DAI_TX_INT_EN           (DAI_BASE + 0x44)
#define DAI_RX_INT_EN           (DAI_BASE + 0x48)

#define DAI_TX_LEFT_DATA_PHY    (UMVP_DAI_BASE + 0x2C)
#define DAI_RX_DATA_PHY         (UMVP_DAI_BASE + 0x38)

#define UMVP_READ_REG(r)      (*((volatile unsigned int *) (r)))
#define UMVP_WRITE_REG(r,v)   (*((volatile unsigned int *) (r)) = ((unsigned int)   (v)))

/* private div controll ID */
#define A1_CODEC_CLK_DIV_ON       0
#define A1_CODEC_CLK_DIV_OFF      1

/* -----------------------------------------------------------------------*/
//#define DBG
#ifdef DBG
#define KDUMP(x...) printk(x)
#else
#define KDUMP(x...) do { } while (0)
#endif

#ifdef DBG
static void umvp_dump(void)
{
    KDUMP("DAI_MODE = 0x%04X\n",         __raw_readl(DAI_MODE) );
    KDUMP("DAI_RX_CTRL = 0x%04X\n",      __raw_readl(DAI_RX_CTRL) );
    KDUMP("DAI_TX_CTRL = 0x%04X\n",      __raw_readl(DAI_TX_CTRL) );
    KDUMP("DAI_WLEN = 0x%04X\n",         __raw_readl(DAI_WLEN) );
    KDUMP("DAI_WPOS = 0x%04X\n",         __raw_readl(DAI_WPOS) );
    KDUMP("DAI_SLOT = 0x%04X\n",         __raw_readl(DAI_SLOT) );
    KDUMP("DAI_TX_FIFO_LTH = 0x%04X\n",  __raw_readl(DAI_TX_FIFO_LTH) );
    KDUMP("DAI_RX_FIFO_GTH = 0x%04X\n",  __raw_readl(DAI_RX_FIFO_GTH) );
    KDUMP("DAI_CLOCK = 0x%04X\n",        __raw_readl(DAI_CLOCK) );
    KDUMP("DAI_INIT  = 0x%04X\n",        __raw_readl(DAI_INIT) );
    KDUMP("DAI_TX_FIFO_FLAG = 0x%04X\n", __raw_readl(DAI_TX_FIFO_FLAG) );
    KDUMP("DAI_TX_LEFT_DATA = 0x%04X\n", __raw_readl(DAI_TX_LEFT_DATA) );
    KDUMP("DAI_TX_RIGHT_DATA = 0x%04X\n",__raw_readl(DAI_TX_RIGHT_DATA) );
    KDUMP("DAI_RX_FIFO_FLAG = 0x%04X\n", __raw_readl(DAI_RX_FIFO_FLAG) );
    KDUMP("DAI_RX_DATA = 0x%04X\n",      __raw_readl(DAI_RX_DATA) );
    KDUMP("DAI_TX_FIFO_CNTR = 0x%04X\n", __raw_readl(DAI_TX_FIFO_CNTR) );
    KDUMP("DAI_RX_FIFO_CNTR = 0x%04X\n", __raw_readl(DAI_RX_FIFO_CNTR) );
    KDUMP("DAI_CLOCKOUT_CTRL = 0x%04X\n",__raw_readl(DAI_CLOCKOUT_CTRL) );
    KDUMP("DAI_CLOCKOUT_DIV = 0x%04X\n", __raw_readl(DAI_CLOCKOUT_DIV) );

    KDUMP("DAI_TX_INT_EN = 0x%04X\n",__raw_readl(DAI_TX_INT_EN) );
    KDUMP("DAI_RX_INT_EN = 0x%04X\n", __raw_readl(DAI_RX_INT_EN) );

}
#endif

enum {
    TX_STATE_STOP = 0,
    TX_STATE_INIT,
    TX_STATE_RUN
};

enum {
    RX_STATE_STOP = 0,
    RX_STATE_INIT,
    RX_STATE_RUN,
};

struct frag_buf {
    unsigned int *buf_a;        /* address value from allocator */
    unsigned int  buf_p;        /* physical address value */
            void *buf_v;        /* virtual io address of the physical address "ioremap-ped" */
             int  nSize;        /* size of this fragment buffer (in byte) */
             int  nValid;       /* size of valid data in this fragment buffer (in byte) */
             int  nOpStart;     /* index */
  struct frag_buf *next;
//struct semaphore sem;
};
struct frag_buf txdma,    rxdma;
struct frag_buf *wr_head, *wr_tail;
struct frag_buf *rd_head, *rd_tail;

static void au_i2s_reset(void);

static void au_i2s_buffer_reset(void)
{
   wr_head = NULL; wr_tail = NULL;
   rd_head = NULL; rd_tail  = NULL;
}

static struct frag_buf * alloc_one_fragment(int size, int locking)
{
    struct frag_buf *temp;

    temp = kmalloc(sizeof(struct frag_buf), GFP_KERNEL);
    if (!temp)
        return (NULL);

    temp->buf_a = (unsigned int *) kmalloc(size, GFP_KERNEL);
    if (!temp->buf_a) {
        kfree(temp);
        return (NULL);
    }

    temp->buf_p    = virt_to_phys(temp->buf_a);
    temp->buf_v    = ioremap(temp->buf_p, size);
    temp->nSize    = size;
    temp->nValid   = 0;
    temp->nOpStart = 0;

#if 0
    if (locking)
        init_MUTEX_LOCKED(&temp->sem);
    else
        init_MUTEX(&temp->sem);
#endif

    temp->next = NULL;

    return (temp);
}

static void free_one_fragment(struct frag_buf *p)
{
    //up(&p->sem);
    iounmap(p->buf_v);
    kfree(p->buf_a);
    kfree(p);
}

static void free_fragments(void)
{
    struct frag_buf *p, *q, *h;

    dprintk(1,KERN_INFO "%s:\n", __func__);

    /* note that read/write buffers are circular lists */
    p = wr_head;
    h = p;
    while(p) {
        q = p->next;
        free_one_fragment(p);
        p = q;
        if (p == h) break;
    }

    p = rd_head;
    h = p;
    while(p) {
        q = p->next;
        free_one_fragment(p);
        p = q;
        if (p == h) break;
    }

    au_i2s_buffer_reset();
}

static int alloc_fragments(int nr, int size)
{
    int nIdx;
    struct frag_buf *t, *temp;

    dprintk(1,KERN_INFO "%s: nr(%d) size(%d)\n", __func__, nr, size);

    /***************************
     * allocate for write buffer
     ***************************/
    wr_head = alloc_one_fragment(size, 0); //init the semaphore without locking
    if (!wr_head) return (-1);

    wr_head->next = wr_head;
    wr_tail = wr_head;
    t = wr_tail;

    for (nIdx = 1; nIdx < nr; ++nIdx) {
        temp = alloc_one_fragment(size, 0);
        if (!temp) {
            free_fragments();
            return (-1);
        }

        /* append to the circular link list */
        temp->next = t->next;
        t->next = temp;
        t = temp;
    }
    /*
    dprintk(4,KERN_INFO "write buffer: wr_head(%p) wr_head->next(%p)\n", wr_head, wr_head->next);
    dprintk(4,KERN_INFO "write buffer: wr_tail(%p) wr_tail->next(%p)\n", wr_tail, wr_tail->next);
    */

    /**************************
     * allocate for read buffer
     **************************/
    rd_head = alloc_one_fragment(size, 1); //init the semaphore with locking
    if (!rd_head) {
        free_fragments();
        return (-1);
    }

    rd_head->next = rd_head;
    rd_tail = rd_head;
    t = rd_tail;

    for (nIdx = 1; nIdx < nr; ++nIdx) {
        temp = alloc_one_fragment(size, 1);
        if (!temp) {
            free_fragments();
            return (-1);
        }
        /* append to the circular link list */
        temp->next = t->next;
        t->next = temp;
        t = temp;
    }

    /*
    dprintk(4,KERN_INFO "read buffer: rd_head(%p) rd_head->next(%p)\n", rd_head, rd_head->next);
    dprintk(4,KERN_INFO "read buffer: rd_tail(%p) rd_tail->next(%p)\n", rd_tail, rd_tail->next);
    */
    return (0);
}
/*
 * global variables in this file
 */
struct umvp_dmap_params dmap_tx, dmap_rx;

typedef struct {
 struct semaphore sem;
      hw_params_t param;
       spinlock_t lock;
wait_queue_head_t rx_q, tx_q;       /* read and write queues */
              int tx_started,rx_started;
              int dma_ch_tx, dma_ch_rx;
              int dsp_id;
              int mixer_id;
}au_i2s_t;

static au_i2s_t au_dai;
static au_i2s_t *pi2s = &au_dai;
static void au_dai_enqueue_dma(int dir);

static void start_dai_tx(void)
{
    register unsigned int nRegVal;

    nRegVal = UMVP_READ_REG(DAI_INIT);
    nRegVal |= (1 << 2);
    UMVP_WRITE_REG(DAI_INIT, nRegVal);
    mdelay(1);
}

static void stop_dai_tx(void)
{
    register unsigned int nRegVal;

    nRegVal = UMVP_READ_REG(DAI_INIT) & ~0x04;
    UMVP_WRITE_REG(DAI_INIT, nRegVal);
}

static void start_dai_rx(void)
{
    register unsigned int nRegVal;

    nRegVal = UMVP_READ_REG(DAI_INIT);
    nRegVal |= (1 << 1);
    UMVP_WRITE_REG(DAI_INIT, nRegVal);
    mdelay(1);
}

static void stop_dai_rx(void)
{
    register unsigned int nRegVal;

    nRegVal = UMVP_READ_REG(DAI_INIT)  & ~0x02;
    UMVP_WRITE_REG(DAI_INIT, nRegVal);

    /* wait until receiver FIFO counter = 0  */
    do {
        UMVP_READ_REG(DAI_RX_DATA);
        udelay(2);
    } while ( UMVP_READ_REG(DAI_RX_FIFO_CNTR) );


}

#ifdef DBG
#define LP_PERIOD 20
static int __ioctl_loopback_test1(void)
{
    /* this test doesn't utilize the read/write buffers */
    int bLeftChannel;
    unsigned int nNumSamples;
    unsigned int nRXFIFOFlag;
    unsigned int nSampleSound = 0;
    unsigned int nClkCtlRegVal;

    dprintk(1,KERN_INFO "\n >>> loopback test for %d seconds <<< \n", LP_PERIOD);

    /* computing the samping rate and determine the samping number */
    nClkCtlRegVal  = UMVP_READ_REG(DAI_CLOCK);
    nClkCtlRegVal &= 0xFF;
    nNumSamples    = GUC_AUDIO_OSC/((nClkCtlRegVal+1)*32*2);
    nNumSamples   *= LP_PERIOD; /* test for a while */

    /* start the DAI RX and TX */
    UMVP_WRITE_REG(DAI_INIT, 1);
    mdelay(1);
    UMVP_WRITE_REG(DAI_INIT, 6);

    bLeftChannel = 1;
    while (nNumSamples) {
        nRXFIFOFlag = UMVP_READ_REG(DAI_RX_FIFO_FLAG);
        if (!(nRXFIFOFlag & 1)) {
            /* RX FIFO isn't empty */
            if (pi2s->param.channels == A1_SOC_CHANNEL_STEREO) {
                if (bLeftChannel) {
                    nSampleSound = (UMVP_READ_REG(DAI_RX_DATA)) << 16;
                    bLeftChannel = 0;
                } else {
                    nSampleSound += (UMVP_READ_REG(DAI_RX_DATA));
                    UMVP_WRITE_REG(DAI_TX_LEFT_DATA, nSampleSound);
                    bLeftChannel = 1;
                    --nNumSamples;
                }
            }else{
                nSampleSound = (UMVP_READ_REG(DAI_RX_DATA)) & 0xFFFF;
                UMVP_WRITE_REG(DAI_TX_LEFT_DATA, nSampleSound);
                --nNumSamples;
            }
        }
    }
    /* stop the DAI RX and TX */
    UMVP_WRITE_REG(DAI_INIT, 0);
    dprintk(1,KERN_INFO " <<<<< loopback test end >>>>>> \n");
    return (0);
}
#endif

static loff_t
umvp_dai_llseek(struct file *filp, loff_t nOffset, int nOrigin)
{
    return (0);
}

static ssize_t
umvp_dai_read(struct file *filp, char *pBuf, size_t nCount, loff_t *pPos)
{
    ssize_t nTotalReadSize;
    ssize_t nReadThisRound;
    ssize_t nRemainInBuf;

#if 0
    static unsigned long t1=0, t2=0, cnt=0;
    if (pi2s->rx_started == RX_STATE_RUN) {
        cnt++;
        t2 = jiffies;
        if ((t2-t1) >= 100){
            printk("_t2-t1 = %lu, cnt = %lu\n", t2 - t1, cnt);
            t1 = t2;
            cnt = 0;
        }
    }
 #endif

    if (pi2s->rx_started == RX_STATE_STOP) return 0;

    #if 0
        if (down_interruptible(&pi2s->sem))
            return -ERESTARTSYS;
    #endif

    if (pi2s->rx_started == RX_STATE_INIT) {
        rd_head = rd_tail;
        au_dai_enqueue_dma(A1_SOC_CAPTURE);
        start_dai_rx();
        umvp_dmap_start(pi2s->dma_ch_rx, &dmap_rx);
        pi2s->rx_started = RX_STATE_RUN;
        //return 0;
    }

    nTotalReadSize = 0;
    while (nTotalReadSize < nCount)
    {
        /* nothing to read */
        while (rd_tail == rd_head) {
           //up(&pi2s->sem);
           if (filp->f_flags & O_NONBLOCK)
               return -EAGAIN;
           if (wait_event_interruptible(pi2s->rx_q, (rd_tail != rd_head)))
               return -ERESTARTSYS;    /* signal: tell the fs layer to handle it */

           #if 0
               /* otherwise loop, but first reacquire the lock */
               if (down_interruptible(&pi2s->sem))
                   return -ERESTARTSYS;
           #endif
        }

        /* ok, data is there, return something */
        nReadThisRound = nCount - nTotalReadSize;
        nRemainInBuf   = rd_head->nValid - rd_head->nOpStart;
        if (nReadThisRound >= nRemainInBuf) {
            nReadThisRound  = nRemainInBuf;
            if (nReadThisRound > 0) { //R:20131126
                copy_to_user(pBuf+nTotalReadSize,
                            ((char *) rd_head->buf_v) + rd_head->nOpStart,
                             nReadThisRound);
                rd_head->nOpStart += nReadThisRound;
            }else{
                rd_head->nOpStart = 0;
                rd_head->nValid   = 0;
            }
            rd_head = rd_head->next;

        } else {
            copy_to_user(pBuf+nTotalReadSize,
                        ((char *) rd_head->buf_v) + rd_head->nOpStart,
                         nReadThisRound);
            rd_head->nOpStart += nReadThisRound;
            /* There are still valid data in this fragment! */
        }
        nTotalReadSize += nReadThisRound;
    }
//  up(&pi2s->sem);

    return (nTotalReadSize);
}

static ssize_t
umvp_dai_write(struct file *filp, const char *pBuf, size_t nCount, loff_t *pPos)
{
    ssize_t nTotalWriteSize, nWriteThisRound;

    if (pi2s->tx_started == TX_STATE_STOP) return 0;

    #if 0
        if (down_interruptible(&pi2s->sem))
            return -ERESTARTSYS;
    #endif

    nTotalWriteSize = 0;
    while (nTotalWriteSize < nCount)
    {
        /* Make sure there's space to write */
        if (pi2s->tx_started == TX_STATE_RUN) {
            while ((wr_tail == wr_head)&&(wr_head->nValid != 0)) {
                //up(&pi2s->sem);
                if (filp->f_flags & O_NONBLOCK)
                    return -EAGAIN;

                dprintk(4,KERN_INFO"\"%s\" writing: going to sleep\n",current->comm);
                if (wait_event_interruptible(pi2s->tx_q, (wr_tail != wr_head)))
                    return -ERESTARTSYS;    /* signal: tell the fs layer to handle it */

                #if 0
                    /* otherwise loop, but first reacquire the lock */
                    if (down_interruptible(&pi2s->sem))
                        return -ERESTARTSYS;
                #endif
            }
        }

        nWriteThisRound = nCount - nTotalWriteSize;
        if (nWriteThisRound >= wr_tail->nSize)
            nWriteThisRound  = wr_tail->nSize;

        copy_from_user(((char *) wr_tail->buf_v), pBuf+nTotalWriteSize, nWriteThisRound);
        wr_tail->nValid   = nWriteThisRound;
        wr_tail->nOpStart = 0;

        if (pi2s->tx_started == TX_STATE_INIT) {
            wr_head = wr_tail;
            au_dai_enqueue_dma(A1_SOC_PLAYBACK);
            start_dai_tx();
            umvp_dmap_start(pi2s->dma_ch_tx, &dmap_tx);
            pi2s->tx_started = TX_STATE_RUN;
        }

        nTotalWriteSize += nWriteThisRound;
        /* advance one */
        wr_tail = wr_tail->next;
    }

  //up(&pi2s->sem);
    return (nTotalWriteSize);
}

static int au_codec_set_sensitivity(int HiLo)
{
    enum {_LO=0, _HI};
    enum {_PASSIVE=0, _ACTIVE};
    int  ret;

    if (au_codec.chip_id == SSM2602) {
        if (HiLo)
            ret = au_ssm2602_mic_sensitivity(_HI);
        else
            ret = au_ssm2602_mic_sensitivity(_LO);

    } else {
        if (HiLo)
            ret = au_wm8940_pga_boost(_ACTIVE);
        else
            ret = au_wm8940_pga_boost(_PASSIVE);
    }

    return ret;
}

static int umvp_dai_ioctl(struct inode *inode, struct file *filp,
                          unsigned int cmd, unsigned long arg)
{
    int val, ret = 0, err = 0;
    /*
     * extract the type and number bitfields, and don't decode
     * wrong cmds: return ENOTTY (inappropriate ioctl) before access_ok()
     */
    if (_IOC_TYPE(cmd) != A1AUDIO_IOC_MAGIC) return -ENOTTY;
    if (_IOC_NR(cmd)    > A1AUDIO_IOC_MAXNR) return -ENOTTY;

    if (_IOC_DIR(cmd) & _IOC_READ)
        err = !access_ok(VERIFY_WRITE, (void *) arg, _IOC_SIZE(cmd));
    else if (_IOC_DIR(cmd) & _IOC_WRITE)
        err = !access_ok(VERIFY_READ,  (void *) arg, _IOC_SIZE(cmd));
    if (err)
        return -EFAULT;

    switch (cmd) {
    case A1AUDIO_INONOFF:
         ret = __get_user(val, (int __user *) arg);
         dprintk(1,KERN_INFO "A1AUDIO_INONOFF: val(%d)\n", val);
         if (val) {
            pi2s->rx_started = RX_STATE_INIT;

         }else {
            umvp_dmap_stop(pi2s->dma_ch_rx);
            pi2s->rx_started = RX_STATE_STOP;
         }
         break;

    case A1AUDIO_OUTONOFF:
         ret = __get_user(val, (int __user *) arg);
         dprintk(3,KERN_INFO "A1AUDIO_OUTONOFF: val(%d)\n", val);
         if (val) {
            pi2s->tx_started = TX_STATE_INIT;

         } else {
            umvp_dmap_stop(pi2s->dma_ch_tx);
            pi2s->tx_started = TX_STATE_STOP;
         }
         break;

    case A1AUDIO_MIC_TYPE:
    case A1AUDIO_INVOLUME:
         ret = __get_user(val, (int __user *) arg);
         dprintk(1,KERN_INFO "A1AUDIO_INVOLUME: val(%d)\n", val);
         ret = au_codec_set_sensitivity(val);
         break;

    case A1AUDIO_INPGAVOL:
         ret = __get_user(val, (int __user *) arg);
         if ((val <0)||(val > 100)) return -EINVAL;

         if (au_codec.chip_id == WM8940) {
             int volume;
             volume = (val * 63) / 100;
             dprintk(1,KERN_INFO "A1AUDIO_INPGAVOL: val(%d) volume(%d).\n", val, volume);
             ret = au_wm8940_involume(volume);
         }
         break;

    case A1AUDIO_OUTVOLUME:
    {
         int volume;
        /* -----------------------------
         * 000 0000 to 010 1111 = mute
         * 011 0000 = -73db
         * ....
         * 111 1001 = 0 db (default)
         * ...1 db steps up to
         * 111 1111 = +6db
         *-----------------------------
           web value    codec value
               0           mute
            1 ~ 69       -73db ~ -1db
            70 ~ 100     0db ~ +6db
         *------------------------------
         */
         ret = __get_user(volume, (int __user *) arg);

         if (au_codec.chip_id == SSM2602)
         {
             if (volume <= 0) {
                val = 0;
             }else if ((volume > 0)&&(volume < 70)) {
                val = 0x79 - (70 - volume);
             }else if ((volume >= 70)&&(volume <= 100)) {
                val = 0x79 + (((volume - 70)*6)/(100 - 70)); /* 121 = 0x79 is 0db */
             }else {
                return -EINVAL;
             }
             dprintk(3,KERN_INFO "A1AUDIO_OUTVOLUME: volume(%d) mapping to val(%d)\n", volume, val);
             ret = au_ssm2602_outvolume(val);
         }else {
             ret = au_wm8940_outvolume(volume);
         }
         break;
    }
    case A1AUDIO_DAIRESET:
         au_i2s_reset();
         break;
    default:  /* redundant, as cmd was checked against MAXNR */
        return -ENOTTY;
    }

    dprintk(5,KERN_INFO "%s: return value(%d)\n", __func__, ret);
    return ret;
}

static int umvp_mixer_ioctl(struct inode *inode, struct file *filp,
                            unsigned int cmd, unsigned long arg)
{
    return (0);
}

static void au_dai_enqueue_dma(int dir)
{
    if (dir == A1_SOC_PLAYBACK) {
        dmap_tx.transf_type     = memory_to_peripheral;
        dmap_tx.peripheral_type = umvp_dai_tx;

        if (pi2s->param.channels == A1_SOC_CHANNEL_STEREO)
            dmap_tx.transf_width = transf_width_word;
        else
            dmap_tx.transf_width = transf_width_halfword;
        dmap_tx.src_addr_inc = 1;   /* src is memory; memory address is incremental */
        dmap_tx.dst_addr_inc = 0;   /* dst is DAI-TX; memory address is not incremental */
        dmap_tx.transf_bytes = wr_head->nValid;      /* period_size; */
        dmap_tx.src_addr_phy = wr_head->buf_p;
        dmap_tx.dst_addr_phy = DAI_TX_LEFT_DATA_PHY; /* UMVP_DAI_BASE + DAI_TX_LEFT_DATA */

    } else {
        /* start DMA to fill one fragment */
        dmap_rx.transf_type = peripheral_to_memory;
        dmap_rx.peripheral_type = umvp_dai_rx;
        dmap_rx.transf_width = transf_width_halfword;
        dmap_rx.src_addr_inc = 0;   /* src is DAI-RX; memory address is not incremental */
        dmap_rx.dst_addr_inc = 1;   /* dst is memory; memory address is incremental */
        dmap_rx.transf_bytes = rd_tail->nSize;
        dmap_rx.src_addr_phy = DAI_RX_DATA_PHY;     /* UMVP_DAI_BASE + DAI_RX_DATA; */
        dmap_rx.dst_addr_phy = rd_tail->buf_p;
    }
}
static void write_task(unsigned long arg)
{
    //printk("w\n");
    if (pi2s->tx_started == TX_STATE_RUN) {
        wr_head->nValid = 0;
        wake_up_interruptible(&pi2s->tx_q);
        wr_head = wr_head->next;
        if (!wr_head->nValid) {
            /* no valid data for TX */
            register unsigned int nRegVal;
            do {
                /* wait until FIFO empty */
                nRegVal = UMVP_READ_REG(DAI_TX_FIFO_FLAG);
            } while ((nRegVal & 1) != 1);
            stop_dai_tx();
            pi2s->tx_started = TX_STATE_INIT;
            dprintk(4,KERN_INFO "tx");
         } else {
            spin_lock(&pi2s->lock);
            au_dai_enqueue_dma(A1_SOC_PLAYBACK);
            umvp_dmap_start(pi2s->dma_ch_tx, &dmap_tx);
            spin_unlock(&pi2s->lock);
        }
    }

}

DECLARE_TASKLET(tasklet_write, write_task, 0);
static void isr_for_tx(unsigned long data)
{
    tasklet_hi_schedule(&tasklet_write);
}

static void read_task(unsigned long arg)
{
    if (pi2s->rx_started == RX_STATE_RUN) {
        /* DMA fill one fragment data and allow for "read" */
        rd_tail->nValid   = rd_tail->nSize;
        rd_tail->nOpStart = 0;
        /* awake dai read */
        rd_tail = rd_tail->next;
        wake_up_interruptible(&pi2s->rx_q);
        /* start DMA to fill one fragment */
        spin_lock(&pi2s->lock);
        au_dai_enqueue_dma(A1_SOC_CAPTURE);
        umvp_dmap_start(pi2s->dma_ch_rx, &dmap_rx);
        spin_unlock(&pi2s->lock);
    }

}

DECLARE_TASKLET(tasklet_read, read_task, 0);
static void isr_for_rx(unsigned long data)
{
    tasklet_hi_schedule(&tasklet_read);
}

static void au_i2s_reset(void)
{
    /* software reset */
    UMVP_WRITE_REG( DAI_INIT, UMVP_READ_REG(DAI_INIT) | 1 );
    mdelay(1);
    UMVP_WRITE_REG(DAI_INIT, UMVP_READ_REG(DAI_INIT) & (~1));
    mdelay(1);
}

static int au_i2s_set_clkdiv(int div_id, int div)
{
    if (div_id == A1_CODEC_CLK_DIV_ON) {
        UMVP_WRITE_REG(DAI_CLOCKOUT_CTRL, 3);   //bit1 value always be 1
        UMVP_WRITE_REG(DAI_CLOCKOUT_DIV, div-1);

    }else if (div_id == A1_CODEC_CLK_DIV_OFF) {
        UMVP_WRITE_REG(DAI_CLOCKOUT_CTRL, 0);   //bit1 value always be 1
        UMVP_WRITE_REG(DAI_CLOCKOUT_DIV, 1);

    }else  return -ENODEV;

    return 0;
}

static int au_i2s_set_fmt(unsigned int fmt)
{
    unsigned int clk_ctl = UMVP_READ_REG(DAI_CLOCK);

    switch (fmt) {
    case A1_SOC_DAIFMT_CBS_CFS:
//  case A1_SOC_DAIFMT_CBS_CFM:  //???
        /* cpu is master; codec clk & frame slave */
        /* RX control register : set as master mode */
        UMVP_WRITE_REG(DAI_RX_CTRL, UMVP_READ_REG(DAI_RX_CTRL) | (1 << 1));
        /* TX control register : set as master mode */
        UMVP_WRITE_REG(DAI_TX_CTRL, (1<<3) | UMVP_READ_REG(DAI_TX_CTRL) | (1<<5) );

        /* clock control register : set as master mode */
        clk_ctl |= (1 << 12) + (1 << 11) + (1 << 10) + (1 << 9) + (0 << 8);
        UMVP_WRITE_REG(DAI_CLOCK, clk_ctl);
        break;

    case A1_SOC_DAIFMT_CBM_CFM:
        /* cpu is slave; codec clk & frame master */
        /*RX control register : set as slave mode */
        UMVP_WRITE_REG(DAI_RX_CTRL, UMVP_READ_REG(DAI_RX_CTRL) & (~(1 << 1)));
        /* TX control register : set as slave mode */
        UMVP_WRITE_REG(DAI_TX_CTRL,  (1<<3) | (UMVP_READ_REG(DAI_TX_CTRL) & (~(1 << 5))) );

        clk_ctl = (0 << 12) + (0 << 11) + (0 << 10) + (0 << 9) + (0 << 8);
        UMVP_WRITE_REG(DAI_CLOCK, clk_ctl | (1 << 10)); //ted: should set bit 10 to 1. why???
        mdelay(1);
        UMVP_WRITE_REG(DAI_CLOCK, clk_ctl);
        break;

    default:
        dprintk(0,KERN_INFO "%s:bad master\n", __func__);
        return -EINVAL;
    }

    return 0;
}

static int au_i2s_set_channels(unsigned int ch)
{
    switch (ch) {
    case A1_SOC_CHANNEL_MONO:
         //dprintk(1,KERN_INFO "%s: set to MONO \n", __func__);
         /* TX Mono */
         UMVP_WRITE_REG(DAI_TX_CTRL, 0x01 | UMVP_READ_REG(DAI_TX_CTRL) );
         /* RX Mono */
         UMVP_WRITE_REG(DAI_RX_CTRL, 0x10 | UMVP_READ_REG(DAI_RX_CTRL));
         break;
    case A1_SOC_CHANNEL_STEREO:
         /* TX Stereo */
         UMVP_WRITE_REG(DAI_TX_CTRL, (~0x03) & UMVP_READ_REG(DAI_TX_CTRL) );
         /* RX Stereo */
         UMVP_WRITE_REG(DAI_RX_CTRL, (~0x10) & UMVP_READ_REG(DAI_RX_CTRL));
         break;
    default:
         dprintk(0,KERN_INFO "%s: unsupported channel numbers\n", __func__);
         return -EINVAL;
    }
    return 0;
}

static int au_i2s_set_clk(unsigned int samp_rate)
{
    unsigned int clk_ctl = UMVP_READ_REG(DAI_CLOCK);
    unsigned int divider, clk;

    /* clock control register (used in I2S bus) */
    halSysPLLIsMaster(1);
    divider = PLL_GetFout(1, A1_SYS_CLOCK_SRC) / (12288 * 1000);
    clk     = halSysSetDeviceClockDivider(A1_SYS_CLOCK_SRC, HAL_MFP_I2S, divider);
    UMVP_WRITE_REG( DAI_CLOCK, clk_ctl | ((clk/(samp_rate*32*2) - 1) & 0xFF) );

    return 0;
}


static int au_i2s_set_word_length(int wd_len)
{
    /* Determine xfer data type */
    switch (wd_len) {
    case A1_SOC_FORMAT_S16_LE:
         /* word length register : n-bit precision */
         UMVP_WRITE_REG(DAI_WLEN, 16 - 1);
         /*word position register :16 - precision */
         UMVP_WRITE_REG(DAI_WPOS, 0);
         break;
    case A1_SOC_FORMAT_S24_LE:
         /* word length register : n-bit precision */
         UMVP_WRITE_REG(DAI_WLEN, 24 - 1);
         /*word position register :16 - precision */
         UMVP_WRITE_REG(DAI_WPOS, 0);
         break;
    default:
        dprintk(0,KERN_INFO "%s: unsupported PCM format\n", __func__);
        return -EINVAL;
    }

    return 0;
}

/*------------------------------------------------------------------- */
 int au_i2s_init(void)
{
    hw_params_t *param = &pi2s->param;

    dprintk(1,KERN_INFO "a1 audio version %s\n", A1AUDIO_VERSION);
    memset(pi2s, 0, sizeof(au_i2s_t));

    param->channels = A1_SOC_CHANNEL_MONO;
    param->rate     = A1_SOC_RATE_DEFAULT;
    param->format   = A1_SOC_FORMAT_S16_LE;

    au_i2s_buffer_reset();
    au_i2s_reset();
    au_i2s_set_channels(param->channels);
    /* word length register : n-bit precision */
    au_i2s_set_word_length(param->format);

    /* set I2S mode (the protocol between DAI and codec) */
    UMVP_WRITE_REG(DAI_MODE, 0x04);
    /* TX FIFO threshold */
    UMVP_WRITE_REG(DAI_TX_FIFO_LTH, 63);
    /* RX FIFO threshold */
    UMVP_WRITE_REG(DAI_RX_FIFO_GTH, 1);

    /* cpu is slave; codec clk & frame master */
    au_i2s_set_fmt(A1_SOC_DAIFMT_CBM_CFM);

    au_i2s_set_clkdiv(A1_CODEC_CLK_DIV_OFF, 0);
    au_i2s_set_clk(param->rate);

    stop_dai_tx();
    stop_dai_rx();

    /* creat audio fragment buffer */
    if (alloc_fragments(DEFAULT_FRAG_NUM, DEFAULT_FRAG_SIZE) < 0) {
        dprintk(0,KERN_INFO "Audio alloc fragment Error !!!\n");
        return -ENOMEM;
    }

    pi2s->dma_ch_tx = umvp_dmap_request_some_channel(1, isr_for_tx,0);
    if (pi2s->dma_ch_tx < 0)
        dprintk(0,KERN_INFO "fail to request a DMA channel for TX\n");

    pi2s->dma_ch_rx = umvp_dmap_request_some_channel(0, isr_for_rx,0);
    if (pi2s->dma_ch_rx < 0)
        dprintk(0,KERN_INFO "fail to request a DMA channel for RX\n");

    dprintk(1,KERN_INFO "dma_ch_rx = %d.\n", pi2s->dma_ch_rx);
    dprintk(1,KERN_INFO "dma_ch_tx = %d.\n", pi2s->dma_ch_tx);

    /* initial member */
    init_MUTEX(&pi2s->sem);
    spin_lock_init(&pi2s->lock);
    init_waitqueue_head(&pi2s->rx_q);
    init_waitqueue_head(&pi2s->tx_q);

    return 0;
}

int  au_i2s_release(void)
{
    dprintk(1,KERN_INFO "%s\n", __func__);

    if ((wr_head)||(rd_head))
        free_fragments();

    umvp_dmap_release_channel(pi2s->dma_ch_tx);
    umvp_dmap_release_channel(pi2s->dma_ch_rx);

    tasklet_kill(&tasklet_write);
    tasklet_kill(&tasklet_read);

    pi2s->rx_started = RX_STATE_STOP;
    pi2s->tx_started = TX_STATE_STOP;

    return 0;
}

static atomic_t umvp_dai_available = ATOMIC_INIT(1);

static int umvp_dai_open(struct inode *inode, struct file *filp)
{
    /* not allow more than one to open this device */
    if (!atomic_dec_and_test (&umvp_dai_available)) {
        atomic_inc(&umvp_dai_available);
        /* already open */
        return -EBUSY;
    }

    dprintk(1,KERN_INFO "%s\n", __func__);

    if (au_codec.chip_id == SSM2602) {
        /* disable DAC mute */
        au_ssm2602_dac_mute(0);
        au_ssm2602_start();
    } else {
        au_wm8940_dac_mute(0);
        au_wm8940_start();
    }

    mdelay(10);
    /* start dai */
    pi2s->rx_started = RX_STATE_INIT;
    pi2s->tx_started = TX_STATE_INIT;

#if 0
    __ioctl_loopback_test1();
#endif

    return (0);
}

static int umvp_dai_release(struct inode *inode, struct file *filp)
{
    dprintk(1,KERN_INFO "%s \n", __func__);

    stop_dai_tx();
    stop_dai_rx();
    umvp_dmap_stop(pi2s->dma_ch_tx);
    umvp_dmap_stop(pi2s->dma_ch_rx);

    if (au_codec.chip_id == SSM2602) {
        au_ssm2602_stop();
    } else {
        au_wm8940_stop();
    }

    pi2s->rx_started = RX_STATE_STOP;
    pi2s->tx_started = TX_STATE_STOP;

    /* allow another application to open this device */
    atomic_inc(&umvp_dai_available); /* release the device */

    return (0);
}

struct file_operations umvp_dsp_fops =
{
    llseek:     umvp_dai_llseek,
    read:       umvp_dai_read,
    write:      umvp_dai_write,
    ioctl:      umvp_dai_ioctl,
    open:       umvp_dai_open,
    release:    umvp_dai_release,
};

struct file_operations umvp_mixer_fops =
{
    .ioctl  = umvp_mixer_ioctl
};

int __init umvp_dai_init(void)
{
    pi2s->dsp_id   = register_sound_dsp(&umvp_dsp_fops, -1);
    pi2s->mixer_id = register_sound_mixer(&umvp_mixer_fops, -1);

    return (0);
}

void __exit umvp_dai_exit(void)
{
    unregister_sound_dsp(pi2s->dsp_id);
    unregister_sound_mixer(pi2s->mixer_id);
}

module_init(umvp_dai_init);
module_exit(umvp_dai_exit);

