#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/ioport.h>
#include <asm/io.h>
#include <linux/delay.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/partitions.h>
#include <linux/dma-mapping.h>
#include "acti/nf_types.h"
#include "acti/nf_controller.h"
#include "acti/nf_memory.h"
#include "acti/nf_error.h"
#include "acti/nf_utils.h"

#define	DRIVER_NAME	"acti_nand"
//#define DEFAULT_PARTS
static int debug_mtd = 0;

/* MTD structure for NAND controller */
static struct mtd_info *a1_mtd;

#define NAND_HW_CTRL_BASE	0x90500000
#define NAND_HW_CTRL_SIZE	0x400
#define NAND_IRQ			6
#define NAND_CTRL_BUF_SIZE	8192

/** The base address for the register map.
 * This address is mapped to the base address of the register map. This
 * address is mapped during the initialisation of the host controller.
 */
u8 *map_adr = NULL;
nf_controller_t *pcontroller = NULL;
nf_memory_t *nfmem = NULL;

static struct nand_ecclayout A1_OOB_16 = {
	.eccbytes = 4,
	.eccpos = {0, 1, 2, 3},
	.oobfree = {
		{.offset = 6,
		 .length = 10} },
};

static struct nand_ecclayout A1_OOB_64 = {
	.eccbytes = 28,
	.eccpos = {         32, 33, 34, 35, 36, 37, 38, 39,
				40, 41, 42, 43, 44, 45, 46, 47, 48, 49,
				50, 51, 52, 53, 54, 55, 56, 57, 58, 59},
	.oobfree = {
		{.offset = 1,
		 .length = 31} },
};

static struct nand_ecclayout A1_OOB_218 = {
	.eccbytes = 56,
	.eccpos = {      160, 161, 162, 163, 164, 165, 166, 167, 168, 169,
				170, 171, 172, 173, 174, 175, 176, 177, 178, 179,
				180, 181, 182, 183, 184, 185, 186, 187, 188, 189,
				190, 191, 192, 193, 194, 195, 196, 197, 198, 199,
				200, 201, 202, 203, 204, 205, 206, 207, 208, 209,
				210, 211, 212, 213, 214, 215	},
	.oobfree = {
		{.offset = 1,
		 .length = 159} },
};

static uint8_t scan_ff_pattern[] = { 0xff, };

static struct nand_bbt_descr A1_small_bbt = {
	.options = NAND_BBT_SCAN2NDPAGE,
	.offs = 5,
	.len = 1,
	.pattern = scan_ff_pattern
};

static struct nand_bbt_descr A1_large_bbt = {
	.options = NAND_BBT_SCAN2NDPAGE,
	.offs = 0,
	.len = 1,
	.pattern = scan_ff_pattern
};

/* Generic flash bbt decriptors
*/
static uint8_t bbt_pattern[] = {'A', 'C', 'T', 'i' };
static uint8_t mirror_pattern[] = {'i', 'T', 'C', 'A' };
static struct nand_bbt_descr small_bbt_main_descr = {
	.options = NAND_BBT_LASTBLOCK | NAND_BBT_CREATE | NAND_BBT_WRITE
		| NAND_BBT_2BIT | NAND_BBT_VERSION | NAND_BBT_PERCHIP,
	.offs =	6,
	.len = 4,
	.veroffs = 10,
	.maxblocks = 4,
	.pattern = bbt_pattern
};

static struct nand_bbt_descr small_bbt_mirror_descr = {
	.options = NAND_BBT_LASTBLOCK | NAND_BBT_CREATE | NAND_BBT_WRITE
		| NAND_BBT_2BIT | NAND_BBT_VERSION | NAND_BBT_PERCHIP,
	.offs =	6,
	.len = 4,
	.veroffs = 10,
	.maxblocks = 4,
	.pattern = mirror_pattern
};

static struct nand_bbt_descr bbt_main_descr = {
	.options = NAND_BBT_LASTBLOCK | NAND_BBT_CREATE | NAND_BBT_WRITE
		| NAND_BBT_2BIT | NAND_BBT_VERSION | NAND_BBT_PERCHIP,
	.offs =	1,
	.len = 4,
	.veroffs = 5,
	.maxblocks = 4,
	.pattern = bbt_pattern
};

static struct nand_bbt_descr bbt_mirror_descr = {
	.options = NAND_BBT_LASTBLOCK | NAND_BBT_CREATE | NAND_BBT_WRITE
		| NAND_BBT_2BIT | NAND_BBT_VERSION | NAND_BBT_PERCHIP,
	.offs =	1,
	.len = 4,
	.veroffs = 5,
	.maxblocks = 4,
	.pattern = mirror_pattern
};


#ifdef CONFIG_MTD_CMDLINE_PARTS
static const char *part_probes[] = { "cmdlinepart", NULL };
#endif

#ifdef DEFAULT_PARTS
/*
 * Define static partitions for flash device
 */
static const struct mtd_partition partitions_info_small[] = {
	{
	 .name = "u-boot",
	 .offset = 0,
	 .size = 0x60000,	//128K * 3
	},
	{
	 .name = "u-boot-env",
	 .offset = MTDPART_OFS_APPEND,
	 .size = 0x20000,	//128K
	},
	{
	 .name = "kernel",
	 .offset = MTDPART_OFS_APPEND,
	 .size = 0x400000,	// 4M
	},
	{
	 .name = "fs",
	 .offset = MTDPART_OFS_APPEND,
	 .size = 0x1000000,	// 16M
	},
};

static const struct mtd_partition partitions_info_large[] = {
	{
	 .name = "u-boot",
	 .offset = 0,
	 .size = 0x60000,	//128K * 3
	},
	{
	 .name = "u-boot-env",
	 .offset = MTDPART_OFS_APPEND,
	 .size = 0x20000,	//128K
	},
	{
	 .name = "kernel",
	 .offset = MTDPART_OFS_APPEND,
	 .size = 0x400000,	// 4M
	},
	{
	 .name = "fs",
	 .offset = MTDPART_OFS_APPEND,
	 .size = 0x1000000,	// 16M
	},
	{
	 .name = "kernel-2",
	 .offset = MTDPART_OFS_APPEND,
	 .size = 0x400000,	// 4M
	},
	{
	 .name = "fs-2",
	 .offset = MTDPART_OFS_APPEND,
//	 .size = MTDPART_SIZ_FULL,
	 .size = 0x1000000,	// 16M		
	},
};
#endif

static void a1_hwcontrol(struct mtd_info *mtd, int cmd, unsigned int bitmask)
{
}

static uint8_t a1_nand_read_byte(struct mtd_info *mtd)
{
//	struct nand_chip *chip = mtd->priv;
	uint8_t data8;

	if(pcontroller->last_cmd == NAND_CMD_READID) {
		data8 = (pcontroller->dma_buffer[pcontroller->pio_bufptr++]);
		return data8;
	}
	// FIXME, hack
	else if(pcontroller->last_cmd == NAND_CMD_READOOB) {
		data8 = (pcontroller->dma_buffer[pcontroller->pio_bufptr++]);
//		printk(KERN_INFO ">>>>   read_oob return %x\n", data8);
		return data8;
	}
	else if(pcontroller->last_cmd == NAND_CMD_STATUS) {
		return NAND_STATUS_WP;
	}
	else {
		printk (KERN_INFO "%s, last_cmd----[0x%x], return 0\n",
			__FUNCTION__, pcontroller->last_cmd);
		return 0;
	}
}

static int a1_device_ready(struct mtd_info *mtd)
{
	return 1;
}

/**
 * nand_command - [DEFAULT] Send command to NAND device
 * @mtd:	MTD device structure
 * @command:	the command to be sent
 * @column:	the column address for this command, -1 if none
 * @page_addr:	the page address for this command, -1 if none
 *
 * Send command to NAND device. This function is used for small page
 * devices (256/512 Bytes per page)
 */

/* Used by the upper layer to write command to NAND Flash for
 * different operations to be carried out on NAND Flash */
static void a1_nand_command(struct mtd_info *mtd, unsigned command,
				int column, int page_addr)
{
	int status;
	unsigned int tmpbuf;
	pcontroller->last_cmd = command;

	switch (command) {

		case NAND_CMD_READID:		
			pcontroller->ecc_enabled = 0;
			nf_mem_read_id(nfmem, pcontroller->dma_buffer);
			/* myhack, record last command and reset bufer index to the beginning */
			pcontroller->pio_bufptr = 0;
			return;

		case NAND_CMD_CACHEDPROG:	
			printk( "	...[NAND_CMD_CACHEDPROG]	FIXME!!	\n"); 		
			return;

		case NAND_CMD_PAGEPROG:		
			printk( "	...[NAND_CMD_PAGEPROG]		FIXME!!	\n"); 		
			return;


		case NAND_CMD_SEQIN:		
			/* we don't need to handle it */
			// just save the page, column address
			pcontroller->column = column;
			pcontroller->page_addr = page_addr;
			return;

		case NAND_CMD_ERASE1:		
			/*
			 * Basically, upper layer driver will divide a erase into two phases.
			 * step-1: send command "NAND_CMD_ERASE1", and "page address"
			 * step-2: send command "NAND_CMD_ERASE2", only
			 * soluton: we just need skip "NAND_CMD_ERASE2" command
			 */
			pcontroller->ecc_enabled = 0;
			nf_mem_block_erase (nfmem, page_addr >> nfmem->pages_per_block_shift);
			break;
		case NAND_CMD_ERASE2:		
			/* we don't need to handle it */
		case NAND_CMD_RNDIN:		
		case NAND_CMD_STATUS:		
		case NAND_CMD_DEPLETE1:		
			/*
			 * read error status commands require only a short delay
			 */
		case NAND_CMD_STATUS_ERROR:
		case NAND_CMD_STATUS_ERROR1:
		case NAND_CMD_STATUS_ERROR2:
		case NAND_CMD_STATUS_ERROR3:
			return;
			
		case NAND_CMD_READOOB:
			pcontroller->pio_bufptr = 0;
			pcontroller->ecc_enabled = 0;
			if (nfmem->nandid == NAND_ID_ST_128W3A)
			{
				status = nf_mem_page_spare(nfmem, page_addr, 0, pcontroller->dma_buffer, mtd->oobsize);
				if (status != 0) {
					printk ( "%s: status = 0x%x\n", __FUNCTION__, status);
				}
				nf_mem_page_read(nfmem, 0, 0, &tmpbuf, 0);
			}
			else
				nf_mem_page_read(nfmem, page_addr, mtd->writesize, pcontroller->dma_buffer, mtd->oobsize);
			return;
	
		case NAND_CMD_RESET:
			nf_mem_reset(nfmem);
			return;
	
		case NAND_CMD_RNDOUT:
			return;
	
		case NAND_CMD_READ0:
			// we don't need to hook READ command here, upper will invoke chip->ecc.read_page()
			// printk( "	...[NAND_CMD_READ0]\n");
			pcontroller->column = column;
			pcontroller->page_addr = page_addr;
			return;

			/* This applies to read commands */
		default:
			/*
			 * If we don't have access to the busy pin, we apply the given
			 * command delay
			 */
			printk( "	...[default catch!!!!]\n");
	}
}

static int a1_nand_wait(struct mtd_info *mtd, struct nand_chip *chip)
{
		return 0;
}

static void a1_nand_select(struct mtd_info *mtd, int chipnr)
{
	if (chipnr > 0) {
		nf_ctrl_select_target(pcontroller, chipnr);
	}
}

static int a1_alloc_host(void)
{

	if (map_adr)
			return -EBUSY;

	/* Lock memory region. */
	if (!request_mem_region((unsigned long)NAND_HW_CTRL_BASE, NAND_HW_CTRL_SIZE, "acti_nand")) {
			printk(KERN_INFO "%s: request_mem_region failed!!\n", __FUNCTION__);
			return -ENOMEM;
	}

	map_adr = (u8 *) ioremap_nocache((unsigned long)NAND_HW_CTRL_BASE,
					 (size_t)NAND_HW_CTRL_SIZE);
	if (!map_adr) {
		printk(KERN_INFO "%s: ioremap_nocache failed!!\n", __FUNCTION__);
		release_mem_region((unsigned long)NAND_HW_CTRL_BASE, NAND_HW_CTRL_SIZE) ;
		return -ENOMEM;
	}

	printk(KERN_INFO "map_adr >>>  = %lx\n", map_adr);

	return 0;
}
/**
 * nand_write_page - [REPLACEABLE] write one page
 * @mtd:	MTD device structure
 * @chip:	NAND chip descriptor
 * @buf:	the data to write
 * @page:	page number to write
 * @cached:	cached programming
 * @raw:	use _raw version of write_page
 */
int a1_nand_write_page(struct mtd_info *mtd, struct nand_chip *chip,
			   const uint8_t *buf, int page, int cached, int raw)
{
	// save page to host, mtd don't send "page" info to    chip->ecc.write_page_XXX
	pcontroller->page_addr = page;

	if (unlikely(raw))
		chip->ecc.write_page_raw(mtd, chip, buf);
	else
		chip->ecc.write_page(mtd, chip, buf);
	return 0;
}

/**
 * a1_hwecc_read_page- [REPLACABLE] hardware ecc based page read function
 * @mtd:	mtd info structure
 * @chip:	nand chip info structure
 * @buf:	buffer to store read data
 * @page:	page number to read
 * Not for syndrome calculating ecc controllers which need a special oob layout
 */
int a1_hwecc_read_page(struct mtd_info *mtd,
		struct nand_chip *chip, uint8_t *buf)
{
	int status = 0;
	uint32_t page_number = pcontroller->page_addr;
	int workaround = 0;

	if( nf_mem_page_empty(nfmem, page_number)){
//		printk(KERN_INFO " >>> Trying to reading a empty page, turn off ecc\n");
		pcontroller->ecc_enabled = 0;
		workaround = 1;
	}
	else {
		pcontroller->ecc_enabled = 1;
	}

	status = nf_mem_page_read(nfmem, page_number, 0, buf, mtd->writesize);

	if (status == 0)
		return 0;
	else
	{
		printk(KERN_INFO" FIXME:  line %d, at <<%s>>, page = %#x, ecc_enabled = %d, workaroud = %d\n", __LINE__, __FUNCTION__, page_number, pcontroller->ecc_enabled, workaround);
		return -EIO;
	}
}

void a1_hwecc_write_page(struct mtd_info *mtd,
					      struct nand_chip *chip,
					      const uint8_t *buf)
{
	int status = 0;
	uint32_t page_number = pcontroller->page_addr;
	uint32_t	tmp_buf = pcontroller->dma_buffer + pcontroller->page_size;
	uint8_t *p = (uint8_t *)tmp_buf;
	uint32_t *eccpos = chip->ecc.layout->eccpos;
	int i;

	pcontroller->ecc_enabled = 1;
	status = nf_mem_page_program(nfmem, page_number, 0, buf, mtd->writesize);
	if (status != 0) {
		printk ( "%s: status = 0x%x\n", __FUNCTION__, status);
	}
	pcontroller->ecc_enabled = 0;

	// read spare
	if (nfmem->nandid == NAND_ID_ST_128W3A)
	{
		status = nf_mem_page_spare(nfmem, page_number, 0, tmp_buf, mtd->oobsize);
		if (status != 0) {
			printk ( "%s: status = 0x%x\n", __FUNCTION__, status);
		}
	}
	else
		nf_mem_page_read(nfmem, page_number, nfmem->page_size, tmp_buf, mtd->oobsize);

	for (i = 0; i < chip->ecc.total; i++)
		chip->oob_poi[eccpos[i]] = p[eccpos[i]];

	if (nfmem->nandid == NAND_ID_ST_128W3A) {
		status = nf_mem_page_program(nfmem, page_number, 0, chip->oob_poi, mtd->oobsize);
		// restore address to normal, issue a dummy read
		nf_mem_page_read(nfmem, 0, 0, tmp_buf, 0);

	}
	else {
		status = nf_mem_page_program(nfmem, page_number, nfmem->page_size, chip->oob_poi, mtd->oobsize);
	}

	if (status != 0) {
		printk ( "%s: status = 0x%x\n", __FUNCTION__, status);
	}
}

/**
 * a1_nand_write_oob - [REPLACABLE] the most common OOB data write function
 * @mtd:	mtd info structure
 * @chip:	nand chip info structure
 * @page:	page number to write
 */
static int a1_nand_write_oob(struct mtd_info *mtd, struct nand_chip *chip,
			      int page)
{
	int status = 0;
	const uint8_t *buf = chip->oob_poi;
	int length = mtd->oobsize;
	unsigned int tmpbuf;

	pcontroller->ecc_enabled = 0;
	if (nfmem->nandid == NAND_ID_ST_128W3A)
	{
		status = nf_mem_page_spare(nfmem, page, 0, &tmpbuf, 0);
		if (status != 0) {
			printk ( "%s: status = 0x%x\n", __FUNCTION__, status);
		}
		status = nf_mem_page_program(nfmem, page, 0, (uint8_t *)buf, length);
		if (status != 0) {
			printk ( "%s: status = 0x%x\n", __FUNCTION__, status);
		}
		// restore address to normal, issue a dummy read
		nf_mem_page_read(nfmem, 0, 0, &tmpbuf, 0);
	}
	else
		status = nf_mem_page_program(nfmem, page, nfmem->page_size, (uint8_t *)buf, length);

	return (status == 0) ? 0 : -EIO ;
}
/**
 * a1_nand_read_oob - [REPLACABLE] the most common OOB data read function
 * @mtd:	mtd info structure
 * @chip:	nand chip info structure
 * @page:	page number to read
 * @sndcmd:	flag whether to issue read command or not
 */
static int a1_nand_read_oob(struct mtd_info *mtd, struct nand_chip *chip,
			     int page, int sndcmd)
{
	int status = 0;
	const uint8_t *buf = chip->oob_poi;
	int length = mtd->oobsize;
	unsigned int tmpbuf;

	pcontroller->ecc_enabled = 0;
	if (nfmem->nandid == NAND_ID_ST_128W3A)
	{
		status = nf_mem_page_spare(nfmem, page, 0, (uint8_t *)buf, length);
		if (status != 0) {
			printk ( "%s: status = 0x%x\n", __FUNCTION__, status);
		}
		// restore address to normal, issue a dummy read
		nf_mem_page_read(nfmem, 0, 0, &tmpbuf, 0);
	}
	else
		status = nf_mem_page_read(nfmem, page, nfmem->page_size, (uint8_t *)buf, length);

	if (status != 0) {
		printk ( "%s: status = 0x%x\n", __FUNCTION__, status);
	}
	return 1;
}

void init_NF_MEM(char id_buf[])
{
	struct nand_chip	*chip;
	chip = (struct nand_chip *)a1_mtd->priv;

	pcontroller->nandid = *((unsigned long *)id_buf);
	nfmem->manufacturer_id = id_buf[0];
	nfmem->nandid = *((unsigned long *)id_buf);

	// 1G, K9K8G08U0B
	printk(" %s: %d	a1_mtd->writesize = 0x%x\n", __FUNCTION__, __LINE__, a1_mtd->writesize);
	nfmem->page_size = a1_mtd->writesize;

	printk(" %s: %d	a1_mtd->erasesize = 0x%x\n", __FUNCTION__, __LINE__, a1_mtd->erasesize);
	nfmem->pages_per_block = (a1_mtd->erasesize / a1_mtd->writesize);
	printk(" %s: %d	nfmem->pages_per_block = 0x%x\n", __FUNCTION__, __LINE__, nfmem->pages_per_block);

	printk(" %s: %d	a1_mtd->oobsize = 0x%x\n", __FUNCTION__, __LINE__, a1_mtd->oobsize);
	nfmem->spare_area_size = a1_mtd->oobsize;
	nfmem->blocks_count = ((unsigned long)(chip->chipsize >> 3) / a1_mtd->erasesize) << 3;
	printk(" %s: %d	nfmem->blocks_count = 0x%x\n", __FUNCTION__, __LINE__, nfmem->blocks_count);

	nfmem->cache_prog_supp = 0;
	nfmem->copy_back_support = 0;
	nfmem->page_shift = chip->page_shift;
	nfmem->pages_per_block_shift = get_fsb(nfmem->pages_per_block);
	nfmem->current_work_mode = MEMORY_ASYNCHRONOUS_MODE;

#if 0  //R:20121025 change nand addr cycle from 5 to 4
	nfmem->addr_cyc_cnt = (nfmem->page_size <= 512) ? 3 : 5;
#else
	nfmem->addr_cyc_cnt = (nfmem->page_size <= 512) ? 3 : 4;
#endif
	printk(" %s: %d	nfmem->addr_cyc_cnt = 0x%x\n", __FUNCTION__, __LINE__, nfmem->addr_cyc_cnt);
	
	/* If flash chip timing need to be adjusted.
	 * Fill nfmem->timings.* here
	 */

	// updated by Ted for samsung K9F1G08
#if 1
    /* RE# or WE# high hold time */
    nfmem->timings.trwh_ns = 10+5;
    /* RE# or WE# pulse width */
    nfmem->timings.trwp_ns = 15+5;
    /* WE# high to RE# low time */
    nfmem->timings.twhr_ns = 60+5;
    /* RE# high to WE# low time */
    nfmem->timings.trhw_ns = 100+5;
    /* ALE to data start time */
    //nfmem->timings.tadl_ns = 100;
    nfmem->timings.tadl_ns = 100+5;
    /* Change column setup */
    //nfmem->timings.tccs_ns = 200;
    nfmem->timings.tccs_ns = 100+0;

    /* Busy time for interface change.*/
    #if 0
	    nfmem->timings.twb_ns = 100 + 0;
    #else
	    nfmem->timings.twb_ns = 100 - 10;      //R:20121025 100 is max
    #endif

    /* Read high to Read low. */
    nfmem->timings.trr_ns = 20+5;
    //printk("[%s]: timings.twb_ns(%d)\n", __FUNCTION__, nfmem->timings.twb_ns); 
#endif
}


/* Initialize nand memory */
int scan_and_init_NF_MEM ()
{
	char id_buf[8];

	// Reset First, that is necessary for Large Page device
	nf_mem_reset(nfmem);

	// Read ID
	if (nf_mem_read_id(nfmem, id_buf) != NF_ERR_NO_ERRORS) {
		printk("    : Failed to detect nand chip\n");
		return -1;
	}

	printk("  NAND Chip Signature: %x-%x-%x-%x\n", id_buf[0], id_buf[1], id_buf[2], id_buf[3]);
	if (id_buf[0] == 0 && id_buf[1] == 0 && id_buf[2] == 0 && id_buf[3] == 0)
		return -1;

	init_NF_MEM (id_buf);
	return 0;
}

/**
 * nand_write_page_raw - [Intern] raw page write function
 * @mtd:	mtd info structure
 * @chip:	nand chip info structure
 * @buf:	data buffer
 *
 * Not for syndrome calculating ecc controllers, which use a special oob layout
 */
void a1_nand_write_page_raw(struct mtd_info *mtd, struct nand_chip *chip,
			      uint8_t *buf)
{
	uint32_t page_number = pcontroller->page_addr;
	unsigned int tmpbuf;

//	printf ("[%s] chip->ops.mode = [%d], chip->ops.oobbuf = 0x%x\n", __FUNCTION__,
//					chip->ops.mode, chip->ops.oobbuf);

	// raw read page first without ecc
	pcontroller->ecc_enabled = 0;
	nf_mem_page_program(nfmem, page_number, 0, buf, mtd->writesize);
	if (nfmem->nandid == NAND_ID_ST_128W3A)
	{
		nf_mem_page_spare(nfmem, page_number, 0, &tmpbuf, 0);
		nf_mem_page_program(nfmem, page_number, 0, chip->oob_poi, mtd->oobsize);
	}
	else
		nf_mem_page_program(nfmem, page_number, nfmem->page_size, chip->oob_poi, mtd->oobsize);
}


/**
 * nand_read_page_raw - [Intern] read raw page data without ecc
 * @mtd:	mtd info structure
 * @chip:	nand chip info structure
 * @buf:	buffer to store read data
 * @page:	page number to read

	chip->read_buf(mtd, buf, mtd->writesize);
	chip->read_buf(mtd, chip->oob_poi, mtd->oobsize);
 *
 * Not for syndrome calculating ecc controllers, which use a special oob layout
 */
int a1_nand_read_page_raw(struct mtd_info *mtd, struct nand_chip *chip,
			      uint8_t *buf)
{
	int status = 0;
	uint32_t page_number = pcontroller->page_addr;

	unsigned int tmpbuf;

	// raw read page first without ecc
	if (pcontroller->last_cmd == NAND_CMD_READ0) {
		pcontroller->ecc_enabled = 0;
		nf_mem_page_read(nfmem, page_number, 0, buf, mtd->writesize);

		if (nfmem->nandid == NAND_ID_ST_128W3A)
		{
			status = nf_mem_page_spare(nfmem, page_number, 0, chip->oob_poi, mtd->oobsize);
			if (status != 0) {
				printk ( "%s: status = 0x%x\n", __FUNCTION__, status);
			}
			// restore address to normal, issue a dummy read
			nf_mem_page_read(nfmem, 0, 0, &tmpbuf, 0);
		}

		else
			nf_mem_page_read(nfmem, page_number, nfmem->page_size, chip->oob_poi, mtd->oobsize);

		return 0;
	}
	else {
		printk("Help!! ......[%s:%d]BUG!!\n", __FUNCTION__, __LINE__);
		return -EIO;
	}
}

static int a1_nand_verify(struct mtd_info *mtd,
		const uint8_t *buf, int len)
{
	return 0;
}

/*
 * Register nand_chip's callbacks.
 */
void nand_chip_init_fn(struct nand_chip *chip)
{
	chip->options &= ~NAND_BUSWIDTH_16;	// we support 8 bits only
	chip->cmdfunc	 = a1_nand_command;
	chip->cmd_ctrl	 = a1_hwcontrol;
	chip->read_byte  = a1_nand_read_byte;
	chip->dev_ready  = a1_device_ready;
	chip->waitfunc	 = a1_nand_wait;
	chip->select_chip = a1_nand_select;
/** function to write a page according to the ecc generator requirements **/
	chip->write_page = a1_nand_write_page;	
//	chip->block_bad = a1_nand_block_bad;
	chip->verify_buf = a1_nand_verify;

	//FIXME: following ecc related function
	chip->ecc.mode   = NAND_ECC_HW;
	chip->ecc.read_page = a1_hwecc_read_page;
	chip->ecc.write_page = a1_hwecc_write_page;
	chip->ecc.write_oob = a1_nand_write_oob;
	chip->ecc.read_oob = a1_nand_read_oob;
	chip->ecc.read_page_raw = a1_nand_read_page_raw;
	chip->ecc.write_page_raw = a1_nand_write_page_raw;
}

void nand_chip_init_ecc(struct nand_chip *chip)
{
	// Specify ecc_layout
	chip->ecc.size = nfmem->page_size;		

	if (nfmem->page_size <= 512) {
		chip->ecc.layout = &A1_OOB_16;

		chip->badblock_pattern = &A1_small_bbt;
		chip->bbt_td = &small_bbt_main_descr;
		chip->bbt_md = &small_bbt_mirror_descr;
		chip->ecc.bytes = 4;	// 2-bit correction
		chip->options   |= (NAND_USE_FLASH_BBT | NAND_NO_READRDY |NAND_NO_AUTOINCR );
		return;
	}
	else if (nfmem->page_size == 2048)
		chip->ecc.layout = &A1_OOB_64;

	else if (nfmem->page_size == 4096)
		chip->ecc.layout = &A1_OOB_218;

	chip->badblock_pattern = &A1_large_bbt;
	chip->bbt_td = &bbt_main_descr;
	chip->bbt_md = &bbt_mirror_descr;
	chip->ecc.bytes = 7;	// 4-bit correction

	/* The NAND_SKIP_BBTSCAN option is necessary for 'overridesize' */
	/* and 'badblocks' parameters to work */
	chip->options   |= (NAND_USE_FLASH_BBT | NAND_NO_READRDY |NAND_NO_AUTOINCR );
}

static int __init acti_nand_init(void)
{
	struct nand_chip *chip;
	const char *part_type = 0;
	struct mtd_partition *mtd_parts = 0;
	int mtd_parts_nb = 0;
	int retval = -ENOMEM;

	printk(KERN_INFO "%s driver initializing !!\n", DRIVER_NAME);

	// Process controller resources reservation
	retval = a1_alloc_host();
	if (retval != 0) {
		printk(KERN_INFO "Error in allocating host resources\n");
		return retval;
	}

	// Do controller instance, (nf_controller + nf_memory)
	pcontroller = kzalloc(sizeof(struct nf_controller) + sizeof(struct nf_memory), GFP_KERNEL);
	if (!pcontroller) {
		printk(KERN_INFO "Error in allocating pcontroller\n");
		goto error;
	}


	// Link hw-resources to instance
	pcontroller->reg_offset = (uint32_t)map_adr;

	// Do flash memory instance
	nfmem = (nf_memory_t *)(pcontroller + 1);
	nfmem->pcontroller = pcontroller;
	nfmem->ce = 0;


	// Allocate DMA memory for controller
	pcontroller->dma_buffer = dma_alloc_coherent(NULL, NAND_CTRL_BUF_SIZE, &(pcontroller->dma_buff_address), GFP_KERNEL);
	if (!pcontroller->dma_buffer) {
			printk(KERN_INFO "Failed to allocate dma buffer\n");
			goto error;
	}


	/* Allocate and initialize mtd_info, nand_chip and nandsim structures */
	a1_mtd = kzalloc(sizeof(struct mtd_info) + sizeof(struct nand_chip), GFP_KERNEL);

	if (!a1_mtd) {
		printk("unable to allocate core structures.\n");
		return -ENOMEM;
	}

	chip= (struct nand_chip *)(a1_mtd + 1);
    a1_mtd->priv = (void *)chip;

	// initialize controller
	nf_ctrl_initialize(pcontroller);

    //Register nand_chip's callbacks.
    nand_chip_init_fn(chip);

    if (nand_scan_ident(a1_mtd, 1)) {
		printk("unable to detected nand chip\n");
		goto err_exit;
	}

	/* use nf_mem_read_id library to get basic nand chip information first */
	retval = scan_and_init_NF_MEM();
	if (retval != 0) {
		printk("unable to detected nand chip\n");
		goto err_exit;
	}
	
	nand_chip_init_ecc(chip);
	
	nf_mem_select_target(nfmem);
	
	// apply timing settings
	nf_ctrl_configure_timings(pcontroller, &(nfmem->timings));
	
	a1_mtd->owner = THIS_MODULE;
	a1_mtd->name = "acti_nand";

	/* Scan to find existence of the device */
	if (nand_scan (a1_mtd, 1)) {
		printk("No NAND device\n");
		retval = -ENXIO;
		goto err_exit;
	}
	
	printk (KERN_INFO " Compiliation Time: %s [%s]\n", __TIME__, __DATE__);
	printk (KERN_INFO " HZ = %d\n", HZ);

#ifdef CONFIG_MTD_CMDLINE_PARTS
	mtd_parts_nb = parse_mtd_partitions(a1_mtd, part_probes, &mtd_parts, 0);
	if (mtd_parts_nb > 0)
		part_type = "command line";
	else
		mtd_parts_nb = 0;
#endif
#ifdef DEFAULT_PARTS
	if (nfmem->page_size < 0x800) {
		mtd_parts =  partitions_info_small;
		mtd_parts_nb = ARRAY_SIZE(partitions_info_small);
	}
	else {
		mtd_parts =  partitions_info_large;
		mtd_parts_nb = ARRAY_SIZE(partitions_info_large);
	}
	part_type = "static";
#endif

	/* Register the partitions */
	if (mtd_parts_nb) {
		printk(KERN_NOTICE "Using %s partition definition\n", part_type);
		add_mtd_partitions(a1_mtd, mtd_parts, mtd_parts_nb);
	}
	else {
		printk(KERN_INFO "Registering \"%s\" as whole device\n", a1_mtd->name);
		add_mtd_device(a1_mtd);
	}
	//add_mtd_partitions(a1_mtd, partitions_info_large, ARRAY_SIZE(partitions_info_large));

    return 0;

err_exit:
	nand_release(a1_mtd);

error:
	if (map_adr) {
		iounmap(map_adr);
		release_mem_region((unsigned long)NAND_HW_CTRL_BASE, NAND_HW_CTRL_SIZE) ;
		map_adr = 0;	
	}               	

	if (pcontroller) {
		if(pcontroller->dma_buffer) {
			dma_free_coherent(NULL, NAND_CTRL_BUF_SIZE, pcontroller->dma_buffer, pcontroller->dma_buff_address);
			pcontroller->dma_buffer = 0;
		}
		kfree(pcontroller);
	}

	kfree(a1_mtd);
	return  retval;

}

static void __exit acti_nand_exit(void)
{
	printk(KERN_INFO "%s driver uninitialized !!\n", DRIVER_NAME);

	if (map_adr) {
		iounmap(map_adr);
		release_mem_region((unsigned long)NAND_HW_CTRL_BASE, NAND_HW_CTRL_SIZE) ;
		map_adr = 0;
	}


	if (pcontroller) {
		if(pcontroller->dma_buffer) {
			dma_free_coherent(NULL, NAND_CTRL_BUF_SIZE, pcontroller->dma_buffer, pcontroller->dma_buff_address);
			pcontroller->dma_buffer = 0;
		}
		kfree(pcontroller);
	}

	if (a1_mtd) {
		del_mtd_partitions(a1_mtd);
		del_mtd_device(a1_mtd);
		kfree(a1_mtd);
	}

}

module_init(acti_nand_init);
module_exit(acti_nand_exit);

MODULE_AUTHOR("ACTi Corporation");
MODULE_DESCRIPTION("ACTi NAND MTD driver");
MODULE_LICENSE("GPL");
