/*
 * MTD driver for the MX29LV640BT Flash Memory (non-CFI) on UMVP2500.
 *
 * Author: Rudolph.lu <rudolph.lu@globalunichip.com>
 * 
 *  Copyright (C) Global Unichip Corp.
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include <mach/hardware.h>

#define MAX_FLASH_SECT		(128)	/* max number of sectors on one chip */
#define MAIN_FLASH_SECT		(127)	/* Num of sub sectors on one chip */
#define SUB_FLASH_SECT		(8)	/* Num of sub sectors on one chip */
#define MAIN_SECT_SIZE		(0x10000)	/* 64 KB */
#define SUB_SECT_SIZE		(0x2000)	/* 8 KB */
#define NUM_FLASH			(1)
#define MANUFACTURE_ID		(0xc2)
#define DEVICE_ID			(0x7e)

//Flash command data.
#define CMD_AA				0xAA
#define CMD_90				0x90
#define CMD_55				0x55
#define CMD_80				0x80
#define CMD_10				0x10
#define CMD_30				0x30
#define CMD_A0				0xA0
#define CMD_PROG_RESET		0xf0
#define ErasedWord			0xFFFFFFFF

//Flash command address.
#define MEM_FLASH_ADDR1			(*(volatile u8 *)(UMVP2500_P2V(UMVP2500_FLASHBASE) + 0xAAA ))
#define MEM_FLASH_ADDR2			(*(volatile u8 *)(UMVP2500_P2V(UMVP2500_FLASHBASE) + 0x555 ))
#define MEM_FLASH_ADDR3			(*(volatile u8 *)(UMVP2500_P2V(UMVP2500_FLASHBASE) + 0x000 ))
#define MEM_FLASH_ADDR4			(*(volatile u8 *)(UMVP2500_P2V(UMVP2500_FLASHBASE) + 0x002 ))

#define GET_FLASH_4B(offset)	(*(volatile u32 *)(UMVP2500_P2V(UMVP2500_FLASHBASE) + (offset)))
#define GET_FLASH_2B(offset)	(*(volatile u16 *)(UMVP2500_P2V(UMVP2500_FLASHBASE) + (offset)))
#define GET_FLASH_1B(offset)	(*(volatile u8 	*)(UMVP2500_P2V(UMVP2500_FLASHBASE) + (offset)))

#define UBOOT_LEN			(0x200000)
#define KERNEL_LEN			(0x400000)
#define INITRD_LEN			(0x400000)
#define FS_LEN				(UMVP2500_FLASHSIZE-UBOOT_LEN-KERNEL_LEN-INITRD_LEN)
#define KERNEL_START		(UBOOT_LEN)
#define INITRD_START		(KERNEL_START+KERNEL_LEN)
#define FS_START			(INITRD_START+INITRD_LEN)

static struct mtd_info mtd;

static struct mtd_erase_region_info erase_regions[] = {
	/* parameter blocks */
	{
	 .offset = 0,
	 .erasesize = (SUB_SECT_SIZE * NUM_FLASH),
	 .numblocks = SUB_FLASH_SECT,
	 },
	/* main blocks */
	{
	 .offset = (SUB_FLASH_SECT * NUM_FLASH * SUB_SECT_SIZE),
	 .erasesize = (MAIN_SECT_SIZE * NUM_FLASH),
	 .numblocks = MAIN_FLASH_SECT,
	 }
};

static struct mtd_partition umvp2500_partitions[] = {
	/* u-boot */
	{
	 .name = "u-boot",
	 .offset = 0,
	 .size = UBOOT_LEN,
	 },
	/* kernel */
	{
	 .name = "kernel",
	 .offset = KERNEL_START,
	 .size = KERNEL_LEN,
	 }
	/* initial ramdisk */
//	{
//	 .name = "initrd file system",
//	 .offset = INITRD_START,
//	 .size = INITRD_LEN,
//	 },
//	/* Filesystem  */
//	{
//	 .name = "file system",
//	 .offset = FS_START,
//	 .size = FS_LEN,
//	 }
};

/*----------------------------------------------------------
 *  Sector erase.
 *	BOB: ok
 */

static void
sector_erase(u32 base, u32 erase_size)
{
	u32 i = 0, j = 0;

	// Write Data 0xAAAAAAAA in address in 0xAAA.
	MEM_FLASH_ADDR1 = CMD_AA;
	// Write Data 0x55555555 in address in 0x555.
	MEM_FLASH_ADDR2 = CMD_55;
	// Write Data 0x80808080 in address in 0xAAA.
	MEM_FLASH_ADDR1 = CMD_80;
	// Write Data 0xAAAAAAAA in address in 0xAAA.
	MEM_FLASH_ADDR1 = CMD_AA;
	// Write Data 0x55555555 in address in 0x555.
	MEM_FLASH_ADDR2 = CMD_55;
	// Write Data 0x30303030 in erase sector address.
	GET_FLASH_1B(base) = CMD_30;

	//Check erased
	for (i = base; i < (base + erase_size);) {
		do {
			__asm__ __volatile__("":::"memory");
			j = GET_FLASH_4B(base);
		} while (j != ErasedWord);
		i += 4;
	}

}

/*-----------------------------------------------------------
 * Erase flash sector.
 * mtd_info:
 * instr: Store information for  
 */
static int
flash_erase(struct mtd_info *mtd, struct erase_info *instr)
{
	u32 addr, len, i = 0;
	u32 erase_size = 0;

	//printk (KERN_WARNING "Erase(addr = 0x%.8x, len = 0x%x)\n", (u32)instr->addr, (u32)instr->len);
	/* sanity checks */
	if (instr->addr + instr->len > mtd->size)
		return (-EINVAL);

	addr = (u32) instr->addr;
	len = (u32) instr->len;

	while (1) {
		if (i >= len)
			break;

		if (addr < erase_regions[1].offset) 
			erase_size = SUB_SECT_SIZE * NUM_FLASH;
		 else 
			erase_size = MAIN_SECT_SIZE * NUM_FLASH;

		sector_erase(addr, erase_size);

		i += erase_size;
		addr += erase_size;
	}

	instr->state = MTD_ERASE_DONE;
	mtd_erase_callback(instr);

	return 0;
}

/*------------------------------------------
 * Get data from flash.
 * mtd: MTD flash info.
 * from:Base address for data load. 
 * len: Length of data want to load.
 * retlen:Modified length.
 * buf: data buffer.
 */
static int
flash_read(struct mtd_info *mtd, loff_t from, size_t len, size_t * retlen,
	   u_char * buf)
{
	u_char *t_buf = buf;
	u32 t_len = len, t_from = (u32) from;
	u32 i = 0;

	/* sanity checks */
	if (!len)
		return (0);
	if (from + len > mtd->size)
		return (-EINVAL);

	printk (KERN_WARNING "flash_read from:0x%x, len:0x%x\n",(u32)from,(u32)len);

	if ((((u32) t_buf & ALIGN4) != 0) || ((t_from & ALIGN4) != 0)) {
		//Handle unalign case
		for (; t_len != 0; t_len--) {
			*(t_buf++) = GET_FLASH_1B(t_from);
			t_from++;
		}
		*retlen = len;
		goto end;
	} else {
		for (;;) {
			if ((i + 4) > len) {
				for (; i < len; i++) {
					*(t_buf++) = GET_FLASH_1B(t_from);
					t_from++;
				}
				*retlen = i;
				goto end;
			}

			*((u32 *) t_buf) = GET_FLASH_4B(t_from);
			t_from += 4;
			i += 4;
			t_buf += 4;
		}
	}

 end:
	return 0;
}

/*-----------------------------------
 * Program Flash
 * add:Base address.
 * data:Modified data.
 */
void
flash_byte(ulong add, u8 data)
{
	u32 dd = 0;

	// 4 Cycles
	MEM_FLASH_ADDR1 = CMD_AA;
	MEM_FLASH_ADDR2 = CMD_55;
	MEM_FLASH_ADDR1 = CMD_A0;
	GET_FLASH_1B(add) = data;

	for (;;) {
		dd = GET_FLASH_1B(add);
		//Verify OK!
		if (dd == data)
			break;
	}
}

/*-----------------------------------------------------------------------
 * Store data to flash.
 * mtd: MTD flash info.
 * to:  Base address for data store. 
 * len: Length of data want to store.
 * retlen:Modified length.
 * buf: data buffer.
 */
static int
flash_write(struct mtd_info *mtd, loff_t to, size_t len, size_t * retlen,
	    const u_char * buf)
{
	const u_char *t_buf = buf;
	u_char tsrc;
	u32 i = 0, t_data = 0, t_to = (u32) to;
	u32 cnt = len, k = 0, y = 0;

	/* sanity checks */
	if (!len)
		return (0);
	if (to + len > mtd->size)
		return (-EINVAL);

	printk (KERN_WARNING "flash_write: To:0x%x Len:0x%x\n",(u32)to,(u32)len);      

	//Check every 4 bytes from un-aligned data address.
	t_to &= ~(ALIGN4);
	if ((to - t_to) != 0) {
		//Get writting distance from align add.         
		i = 4 - (to - t_to);

		t_data = GET_FLASH_4B(t_to);

		for (; i < 4; i++) {
			//Get be modified data. 
			k = t_data & (BYTEMASK << (i * 8));
			k >>= (i * 8);

			t_data &= ~(BYTEMASK << (i * 8));
			tsrc = *(t_buf++);
			tsrc &= BYTEMASK;

			//Check erase data.     
			y = tsrc | k;
			if (y > k) {
				printk(KERN_WARNING
				       "Can not write data to un-erase sector! :y:%x k:%x\n",
				       y, k);
				return (-EINVAL);
			}

			t_data |= (tsrc << (i * 8));
			i++;
			cnt--;
			if (cnt == 0)
				break;
		}
		flash_word(t_to, t_data);
	}

	i = 0;
	for (; cnt != 0; cnt--) {
		if (i == 0)
			t_data = GET_FLASH_4B(t_to);
		//Get be modified data. 
		k = t_data & (BYTEMASK << (i * 8));
		k >>= (i * 8);

		t_data &= ~(BYTEMASK << (i * 8));
		tsrc = *(t_buf++);
		tsrc &= BYTEMASK;

		//Check erase data.     
		y = tsrc | k;
		if (y > k) {
			printk(KERN_WARNING
			       "Can not write data to un-erase sector! :y:%x k:%x\n",
			       y, k);
			return (-EINVAL);
		}

		t_data |= (tsrc << (i * 8));
		i++;
		if (i == 4) {
			flash_word(t_to, t_data);
			t_to += 4;
			i = 0;
			*retlen += 4;
		}
	}

	if (i != 0)
		flash_word(t_to, t_data);

	*retlen = len;

	return 0;
}

/*-------------------------------------------------------------------
 * Probe for 8MB*4 flash memory on a UMVP2500.
 *
 * Returns 1 if we found 8MB*4bit flash memory on UMVP2500, 0 otherwise.
 */
static int
flash_probe(void)
{
	u32 rlt = 1;
	u8 tmp = 0;

	// Manufacture ID
	// Write Data 0xAAAAAAAA in address in 0xAAA.
	MEM_FLASH_ADDR1 = CMD_AA;
	// Write Data 0x55555555 in address in 0x555.
	MEM_FLASH_ADDR2 = CMD_55;
	// Write Data 0x90909090 in address in 0xAAA.
	MEM_FLASH_ADDR1 = CMD_90;

	tmp = MEM_FLASH_ADDR3;
	if (tmp != MANUFACTURE_ID) {
		printk(KERN_WARNING "MANUFACTURE_ID not match\n");
		rlt = 0;
	}
	//Device ID
	// Write Data 0xAAAAAAAA in address in 0xAAA.
	MEM_FLASH_ADDR1 = CMD_AA;
	// Write Data 0x55555555 in address in 0x555.
	MEM_FLASH_ADDR2 = CMD_55;
	// Write Data 0x90909090 in address in 0xAAA.
	MEM_FLASH_ADDR1 = CMD_90;

	tmp = MEM_FLASH_ADDR4;
	if (tmp != DEVICE_ID) {
		printk(KERN_WARNING "Device not match\n");
		rlt = 0;
	}

	//Reset Flash
	MEM_FLASH_ADDR1 = CMD_PROG_RESET;

	return rlt;
}

static int __init
umvp_flash_init(void)
{
	int result;

	memset(&mtd, 0, sizeof (mtd));

	printk(KERN_WARNING
	       " Probing for MX29LV640BT/BB flash on UMVP2500...\n");

	if (!flash_probe()) {
		printk(KERN_WARNING
		       "Found no UMVP2500 compatible flash device\n");
		return (-ENXIO);
	}

	mtd.name = "UMVP2500";
	mtd.type = MTD_NORFLASH;
	mtd.writesize = 1;
	mtd.flags = MTD_CAP_NORFLASH;
	mtd.size = UMVP2500_FLASHSIZE;
	mtd.erasesize = MAIN_SECT_SIZE * NUM_FLASH;
	mtd.numeraseregions = ARRAY_SIZE(erase_regions);
	mtd.eraseregions = erase_regions;
	mtd.erase = flash_erase;
	mtd.read = flash_read;
	mtd.write = flash_write;
	mtd.owner = THIS_MODULE;

	result =
	    add_mtd_partitions(&mtd, umvp2500_partitions,
			       ARRAY_SIZE(umvp2500_partitions));

	return result;
}

static void __exit
umvp_flash_exit(void)
{

	del_mtd_partitions(&mtd);
}

module_init(umvp_flash_init);
module_exit(umvp_flash_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Rudolph Lu");
MODULE_DESCRIPTION("MTD driver for MX29LV640BT UMVP2500 board");
