/*
 *  linux/arch/arm/mach-umvp/umvp3000_irq.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/init.h>
#include <linux/stddef.h>
#include <linux/list.h>
#include <linux/timer.h>

#include <mach/hardware.h>
#include <mach/platform2500.h>
#include <asm/irq.h>
#include <asm/io.h>

#include <asm/mach/irq.h>

#ifdef DBG
#define	KDUMP(x...)	do{printk(x);} while(0)
#else
#define	KDUMP(x...)	do { } while (0)
#endif

#define VIC_BASE_VA             IO_ADDRESS(UMVP_VIC_BASE)

#define UMVP_READ_REG(r)        (*((volatile unsigned int *) (r)))
#define UMVP_WRITE_REG(r,v)     (*((volatile unsigned int *) (r)) = ((unsigned int)   (v)))

static void vic_reset(void)
{
    UMVP_WRITE_REG(VIC_BASE_VA + UMVP_VIC_INT_SEL_OFFSET,      0x00000000);
    UMVP_WRITE_REG(VIC_BASE_VA + UMVP_VIC_ENABLE_SET_OFFSET,   0x00000000);
    UMVP_WRITE_REG(VIC_BASE_VA + UMVP_VIC_ENABLE_CLEAR_OFFSET, 0xFFFFFFFF);
    UMVP_WRITE_REG(VIC_BASE_VA + UMVP_VIC_EDGE_CLEAR_OFFSET,   0xFFFFFFFF);
}

void vic_set_intr_trigger(int irqnr, vic_intr_trig_mode_E mode)
{
    unsigned int nRegValue;
    unsigned int nVICVA;
    unsigned int nMask = 0;

    if (irqnr < 32)
    {
        nMask = (1 << irqnr);
		nVICVA = VIC_BASE_VA;		/* switch to VIC1 */
    }

    nMask = (1 << irqnr);

    /* edge or level trigger */
    nRegValue = UMVP_READ_REG(VIC_BASE_VA + UMVP_VIC_SENSE_OFFSET);
    if (mode == vicc_edge_activeFalling || mode == vicc_edge_activeRaising ||
        mode == vicc_edge_activeBoth)
    {
        /* edge trigger : corresponding bit shall be reset as 0 */
        nRegValue &= (~nMask);
    }
    else
    {
        /* level trigger : corresponding bit shall be set as 1 */
        nRegValue |= nMask;
    }
    UMVP_WRITE_REG(VIC_BASE_VA + UMVP_VIC_SENSE_OFFSET, nRegValue);



    /* active high or active low */
    nRegValue = UMVP_READ_REG(VIC_BASE_VA + UMVP_VIC_EVENT_OFFSET);
    if (mode == vicc_edge_activeFalling || mode == vicc_level_activeLow)
    {
        /* active low : corresponding bit shall be reset as 0 */
        nRegValue &= (~nMask);
    }
    else if (mode == vicc_edge_activeRaising || mode == vicc_level_activeHigh)
    {
        /* active high : corresponding bit shall be set as 1 */
        nRegValue |= nMask;
    }
    UMVP_WRITE_REG(VIC_BASE_VA + UMVP_VIC_EVENT_OFFSET, nRegValue);



    /* both edge? */
    nRegValue = UMVP_READ_REG(VIC_BASE_VA + UMVP_VIC_BOTH_EDGE_OFFSET);
    if (mode == vicc_edge_activeBoth)
    {
        /* both edge : corresponding bit shall be set as 1 */
        nRegValue |= nMask;
    }
    else
    {
        /* single edge : corresponding bit shall be reset as 0 */
        nRegValue &= (~nMask);
    }
    UMVP_WRITE_REG(VIC_BASE_VA + UMVP_VIC_BOTH_EDGE_OFFSET, nRegValue);
}

static void vic_enable_intr(int irq)
{
	register unsigned int regVal;

	regVal = UMVP_READ_REG(VIC_BASE_VA + UMVP_VIC_ENABLE_SET_OFFSET);
	regVal |= (1 << irq);
	UMVP_WRITE_REG(VIC_BASE_VA + UMVP_VIC_ENABLE_SET_OFFSET, regVal);
}

static void vic_disable_intr(int irq)
{
	register unsigned int regVal;

	regVal = UMVP_READ_REG(VIC_BASE_VA + UMVP_VIC_ENABLE_CLEAR_OFFSET);
	regVal |= (1 << irq);
	UMVP_WRITE_REG(VIC_BASE_VA + UMVP_VIC_ENABLE_CLEAR_OFFSET, regVal);
}

static void vic_select_fiq_intr(int irq)
{
	register unsigned int regVal;

	regVal = UMVP_READ_REG(VIC_BASE_VA + UMVP_VIC_INT_SEL_OFFSET);
	regVal |= (1 << irq);

	KDUMP("vic_select_fiq_intr(): write %08x to VIC\n", regVal);

	UMVP_WRITE_REG(VIC_BASE_VA + UMVP_VIC_INT_SEL_OFFSET, regVal);
}

static void vic_select_irq_intr(int irq)
{
	register unsigned int regVal;

	regVal = UMVP_READ_REG(VIC_BASE_VA + UMVP_VIC_INT_SEL_OFFSET);

	regVal &= ~(1 << irq);
	KDUMP("vic_select_irq_intr(): write %08x to VIC\n", regVal);
	UMVP_WRITE_REG(VIC_BASE_VA + UMVP_VIC_INT_SEL_OFFSET, regVal);
}

void vic_clear_intr(int irq)
{
	if (irq < 32)
	{
		UMVP_WRITE_REG(VIC_BASE_VA + UMVP_VIC_EDGE_CLEAR_OFFSET, (1 << irq));
	}
}
///////////////////////////////////////////
///////////////////////////////////////////
///////////////////////////////////////////

static void sc_mask_irq(unsigned int irq)
{
	vic_disable_intr(irq);
}

static void sc_unmask_irq(unsigned int irq)
{
	vic_enable_intr(irq);
}

static struct irq_chip sc_chip = {
	.ack	= sc_mask_irq,
	.mask	= sc_mask_irq,
	.unmask = sc_unmask_irq,
};

void __init umvp2500_init_irq(void)
{
	unsigned int i;

	vic_reset();

//	for (i = 0; i < NR_IRQS; i++)
	for (i = 0; i < NR_A1_IRQS; i++)
	{
		if ((1 << i) & UMVP_VIC_VALID_INTMASK)
		{
			set_irq_chip(i, &sc_chip);
			set_irq_handler(i, handle_level_irq);
			set_irq_flags(i, IRQF_VALID | IRQF_PROBE);
		}
	}

	vic_set_intr_trigger (IRQ_TIMERINT0, vicc_edge_activeRaising);
	vic_set_intr_trigger (INT_DMAP, 	vicc_level_activeHigh);
	vic_set_intr_trigger (INT_SDC, 		vicc_level_activeHigh);
	vic_set_intr_trigger (INT_USB, 		vicc_level_activeHigh);
	vic_set_intr_trigger (INT_ETHERNET, vicc_level_activeHigh);
	vic_set_intr_trigger (INT_HX8290, 	vicc_level_activeHigh);
	vic_set_intr_trigger (INT_HX9170, 	vicc_level_activeHigh);
	vic_set_intr_trigger (INT_ISP_C1, 	vicc_level_activeHigh);
	vic_set_intr_trigger (INT_VCAP,		vicc_level_activeHigh);
	vic_set_intr_trigger (INT_GPIO0, 	vicc_level_activeHigh);
	vic_set_intr_trigger (INT_GPIO1, 	vicc_level_activeHigh);
	vic_set_intr_trigger (INT_GPIO2, 	vicc_level_activeHigh);
	vic_set_intr_trigger (INT_GPIO3, 	vicc_level_activeHigh);
	vic_set_intr_trigger (INT_GPIO4, 	vicc_level_activeHigh);

	vic_set_intr_trigger (INT_UARTINT4,	vicc_level_activeHigh); 

	//FIQ
	i = FIQ_START + INT_VCAP;
	set_irq_chip(i, &sc_chip);
	set_irq_handler(i, handle_level_irq);
	set_irq_flags(i, IRQF_VALID | IRQF_PROBE);
}

