/*********************************************************************************
 *	 (c) Copyright 2010, ACTi Corporation, Inc. ALL RIGHTS RESERVED
 *	 
 *	 All software are Copyright 2010 by ACTi Corporation. ALL RIGHTS RESERVED.
 *	 Redistribution and use in source and binary forms, with or without modification,
 *	 are strictly prohibited.
 *	 
 *	 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS ``AS IS'' AND ANY EXPRESS
 *	 OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 *	 OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *	 IN NO EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY DIRECT, INDIRECT,
 *	 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *	 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *	 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 *	 AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *	 OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *	 OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **********************************************************************************/
#include <hardware.h>
#include <board.h>
#include <nand.h>
#include <io.h>
#include <type.h>

#define __a1_nand_writel(off, v)	writel(v, (A1_NAND_BASE + off))
#define __a1_nand_readl(off)		readl(A1_NAND_BASE + off)
#define __A1_NAND_DMA_BUFFER_ADDR	(A1_MEMORY_BASE + 0x01000000)
#define nop() __asm__ __volatile__("mov\tr0,r0\t@ nop\n\t");

static void cp_delay (void)
{
	volatile int i;

	/* copro seems to need some delay between reading and writing */
	for (i = 0; i < 1; i++)
		nop();
}


uint8_t wait_for_value(uint32_t reg_addr, uint32_t mask, uint32_t timeout_us, 
                       uint8_t is_set)
{
    if (is_set == 0){
        // wait until bit/bits will clear
		while (((__a1_nand_readl(reg_addr) & mask) != 0) && (timeout_us > 0)) {
			cp_delay();
			timeout_us --;
		}
	}
	else {
        // wait until bit/bits will set
		while (((__a1_nand_readl(reg_addr) & mask) == 0) && (timeout_us > 0)) {
			cp_delay();
			timeout_us --;
		}
	}
	return (timeout_us == 0) ? NF_ERR_TIMEOUT : NF_ERR_NO_ERRORS;
}


uint8_t wait_until_target_busy()
{
    uint8_t status;
	uint8_t loop = 15;

    // wait for transfer sequence ended by NAND flash controller
    status = wait_for_value(NF_SFR_STATUS, 
                            NF_SFR_STATUS_CTRL_STAT, 100, 0);

    while(status && (loop > 0)) {
			// one more chance 
   			status = wait_for_value(NF_SFR_STATUS, 
                            NF_SFR_STATUS_CTRL_STAT, 100, 0);
			loop--;
	}

	if (status) {
        return status;
	}

    // wait until NAND flash memory device is busy
    status = wait_for_value(NF_SFR_STATUS, 
                          0x0, 500, 1);
    return status;
}

int nand_block_is_bad (int block)
{
	uint32_t page_addr = (block * CONFIG_SYS_NAND_PAGE_COUNT);
	uint32_t status;
	uint8_t *dest = (uint8_t *)(__A1_NAND_DMA_BUFFER_ADDR);

	wait_until_target_busy();

	__a1_nand_writel( NF_SFR_CONTROL, 0x00140b45);							// 0x04

	/* READ OOB, offset = page size */
	__a1_nand_writel( NF_SFR_ADDR0_L, 
				((page_addr & 0xffff)<< 16) | CONFIG_SYS_NAND_PAGE_SIZE);	// 0x1c

	__a1_nand_writel( NF_SFR_ADDR0_H, (page_addr >> 16));					// 0x24
	__a1_nand_writel( NF_SFR_DATA_SIZE, CONFIG_SYS_NAND_SPARE_SIZE);	// 0x84
	__a1_nand_writel( NF_SFR_DMA_ADDR, dest);								// 0x64
	__a1_nand_writel( NF_SFR_DMA_CNT, CONFIG_SYS_NAND_SPARE_SIZE);		// 0x68

	// DMA start, DMA read, unspecified length
	__a1_nand_writel( NF_SFR_DMA_CTRL, 
				(DMA_START|DMA_READ|DMA_CTRL_BURST));						// 0x6c

	__a1_nand_writel( NF_SFR_COMMAND, 0x0030006a);							// 0x0

    status = wait_for_value(NF_SFR_STATUS, 
                            NF_SFR_STATUS_MEMX_RDY(0), 100, 1);

    status = wait_for_value(NF_SFR_DMA_CTRL, 
                            NF_SFR_DMA_READY_FLAG, 500, 1);

	// clear interrupt
	__a1_nand_writel( NF_SFR_INT_STATUS, 0x0);						// 0x6c

	return (*dest == 0xff) ? 0 : 1;
}

// just implement READ_PAGE in this function

int nand_read_page(int block, int page, u8 *dst)
{
	uint32_t page_addr; 
	uint32_t status;

	page_addr = (block * CONFIG_SYS_NAND_PAGE_COUNT) + page;
	wait_until_target_busy();

	__a1_nand_writel( NF_SFR_CONTROL, 0x00140365);						// 0x04
	__a1_nand_writel( NF_SFR_ADDR0_L, (page_addr & 0xffff)<< 16);		// 0x1c
	__a1_nand_writel( NF_SFR_ADDR0_H, (page_addr >> 16));				// 0x24
	__a1_nand_writel( NF_SFR_DMA_ADDR, dst);							// 0x64
	__a1_nand_writel( NF_SFR_DMA_CNT, CONFIG_SYS_NAND_PAGE_SIZE);		// 0x68

	// DMA start, DMA read, unspecified length
	__a1_nand_writel( NF_SFR_DMA_CTRL, 
				(DMA_START|DMA_READ|DMA_CTRL_BURST));					// 0x6c
	__a1_nand_writel( NF_SFR_COMMAND, 0x0030006a);						// 0x0

    status = wait_for_value( NF_SFR_STATUS, 
                            NF_SFR_STATUS_MEMX_RDY(0), 100, 1);

    status = wait_for_value( NF_SFR_DMA_CTRL, 
                            NF_SFR_DMA_READY_FLAG, 500, 1);

	// clear interrupt
	__a1_nand_writel( NF_SFR_INT_STATUS, 0x0);							// 0x6c

	return 0;
}

int nand_load (unsigned int offs, unsigned int uboot_size, u8 *dst)
{
	unsigned int block, block2read;
	unsigned int page;

	/*
	 * offs has to be aligned to a page address!
	 */
	block = offs / CONFIG_SYS_NAND_BLOCK_SIZE;
	block2read = (offs + uboot_size - 1) / CONFIG_SYS_NAND_BLOCK_SIZE;
	page = (offs % CONFIG_SYS_NAND_BLOCK_SIZE) / CONFIG_SYS_NAND_PAGE_SIZE;

	while (block <= block2read) {
		if (!nand_block_is_bad (block)) {
			/*
			 * Skip bad blocks
			 */
			while (page < CONFIG_SYS_NAND_PAGE_COUNT) {
				nand_read_page (block, page, dst);
				dst += CONFIG_SYS_NAND_PAGE_SIZE;
				page++;
			}

			page = 0;
		} else {
			block2read++;
		}

		block++;
	}

	return 0;
}

int board_nand_init()
{
	// ecc ctrl
	__a1_nand_writel( NF_SFR_ECC_CTRL, 0x420);

	// ecc offset, 0x820
	__a1_nand_writel( NF_SFR_ECC_OFFSET, (CONFIG_SYS_NAND_PAGE_SIZE + 0x20));

	// timing
	__a1_nand_writel( NF_SFR_CONTROL, 0x00140365);
	__a1_nand_writel( NF_SFR_TIME_SEQ_0, 0x06020300);
	__a1_nand_writel( NF_SFR_TIME_SEQ_1, 0x00000408);
	__a1_nand_writel( NF_SFR_TIMINGS_ASYN, 0x000000CC);
	__a1_nand_writel( NF_SFR_MEM_CTRL, 0xFF00);
}

/*
 * The main entry for NAND booting. It's necessary that SDRAM is already
 * configured and available since this code loads the main U-Boot image
 * from NAND into SDRAM and starts it from there.
 */
void nand_boot(void)
{
	int ret;
	__attribute__((noreturn)) void (*uboot)(void);

	/*
	 * Init board specific nand support
	 */
	board_nand_init();

	/*
	 * Load U-Boot image from NAND into RAM
	 */
	ret = nand_load(CONFIG_SYS_NAND_U_BOOT_OFFS, CONFIG_SYS_NAND_U_BOOT_SIZE,
			(u8 *)CONFIG_SYS_NAND_U_BOOT_DST);

	/*
	 * Jump to U-Boot image
	 */
	uboot = (void *)CONFIG_SYS_NAND_U_BOOT_START;
	(*uboot)();
}
