#include <hardware.h>
#include <ddr.h>

/*
 * These macros provide a convenient way to write 8, 16 and 32 bit data
 * to any address.
 * Registers r4 and r5 are used, any data in these registers are
 * overwritten by the macros.
 * The macros are valid for any ARM architecture, they do not implement
 * any memory barriers so caution is recommended when using these when the
 * caches are enabled or on a multi-core system.
 */
.macro	write32, addr, data
	ldr		r4, =\addr
	ldr		r5, =\data
	str		r5, [r4]
.endm

.macro	write16, addr, data
	ldr		r4, =\addr
	ldrh	r5, =\data
	strh	r5, [r4]
.endm

.macro	write8, addr, data
	ldr		r4, =\addr
	ldrb	r5, =\data
	strb	r5, [r4]
.endm

/*
 * This macro generates a loop that can be used for delays in the code.
 * Register r4 is used, any data in this register is overwritten by the
 * macro.
 * The macro is valid for any ARM architeture. The actual time spent in the
 * loop will vary from CPU to CPU though.
 */

.macro	wait_timer, time
	ldr		r4, =\time
1:
	nop
	subs	r4, r4, #1
	bcs		1b
.endm

.macro	wait_bic, addr, value
	ldr		r4, =\addr
2:
	ldr 	r5, [r4]
	cmp 	r5, #\value
	beq 	2b
.endm


/**************************************************************************
 * DDR Controller initialization
 *
 **************************************************************************/
.global __a1_setup_ddr
__a1_setup_ddr:
	write32		0x90d00098, 0x000F000F

	/* RTT, DS registers are moved to system register */
	/* Change from 0xF503FF to 0x02503FF, stress test on 4 EVBs */
	write32		0x99000080, 0x02503FF

	/* Set DRAM Configuration Register (DCR)*/
	write32 	0x90d00004, 0x10241c8

	/* Set I/O Configuration Register (IOCR)*/
	write32		0x90d00008, 0xc000002b

	/* Set DRAM Refresh Register (DRR)*/
	write32		0x90d00010, 0x861b136

	/* Set Timing Parameter Register 0 (TPR0)*/
	write32		0x90d00014, 0x30b36692

	/* Set Timing Parameter Register 1 (TPR1)*/
	write32 	0x90d00018, 0x0010ac

	/* Set Timing Parameter Register 2 (TPR2)*/
	write32 	0x90d0001c, 0x01a0c8

	/* Set Timing Parameter Register 3 (TPR3)*/
	write32 	0x90d000b8, 0x002a29

	/* Set DRAM Mode Register (MR)*/
	write32 	0x90d001f0, 0x000a52

	/* Set DRAM Extended Mode Register 1 (EMR1)*/
	write32 	0x90d001f4, 0x000044
///////////////////////////////////////////////////////////////////
	/* Start DRAM Initialization */
	//write32 	MCTL_CCR, MCTL_CCR_INIT_VALUE
	write32		0x90d00000, 0x80000004
	wait_bic 	SYSTEM_REG_DDR2_INIT_STS, DDR2_INIT_STS_BUSY

	/* Start DRAM Data Training Process*/
	write32		0x90d00000, 0x40000004
	//write32 	MCTL_CCR, MCTL_CCR_TRAIN_VALUE
	wait_bic 	SYSTEM_REG_DDR2_INIT_STS, DDR2_INIT_STS_BUSY

	///////////////////////////////////////////////////////////
	/* Doing 2nd times */
	///////////////////////////////////////////////////////////

	/* Set Timing Parameter Register 0 (TPR0)*/
	write32 	0x90d00014, 0xb0b36692

	MOV	 		pc, lr

/*************************************************************************
 *
 * Setup Serial Output console
 *
 ***********************************************************************/
.global __a1_console_init
.align 4
__a1_console_init:
   /* Setup LCR - 8bit, 1stop, No Parity, No break, Clear DLAB */                                                                                                                         
    LDR    r2, =A1_UART0_BASE
    ORR    r2, r2, #0xc
    MOV    r3, #0x83
    STR    r3, [r2]
				    
    /* Setup Divisor Latch LSB - assuming 57600 bps */

    LDR    r2,=A1_UART0_BASE
    // MOV    r3, #0x7a 	// for 57600
    MOV    r3, #0x3D 		// for 115200
    STR    r3, [r2]                                                                                                                                                                        
    /* Setup Divisor Latch MSB */ 

    LDR    r2,=A1_UART0_BASE
    ORR    r2, r2, #0x4
    MOV    r3, #0x0
    STR    r3, [r2]
	                                                                                                                                                                                           
    /* Enable RBR and THR for TX, RX by disabling DLAB in LCR */                                                                                                                           
    LDR    r2,=A1_UART0_BASE
    ORR    r2, r2, #0xc
    MOV    r3, #0x3
    STR    r3, [r2]
                                                                                                                                                                                           
    /* Enable RX and TX FIFO, set Trigger level to 56 */                                                                                                                                   
    LDR    r2,=A1_UART0_BASE
    ORR    r2, r2, #0x8
    MOV    r3, #0xb1
    STR    r3, [r2]

    LDR    r2,=A1_UART0_BASE
    ORR    r2, r2, #0x4
    MOV    r3, #0x85
    STR    r3, [r2]

	/* back to my caller */
	MOV    pc, lr	

/********************************************************************
 *  WriteSerial                                                     *
 *                                                                  *
 *  DESCRIPTION:                                                    *
 *      Output string to serial port.                               *
 *                                                                  *
 *  PARAMETERS:                                                     *
 *      r0 - ptr to null-terminated string to output.               *
 *                                                                  *
 *  RETURNS:                                                        *
 *      Nothing.                                                    *
 *                                                                  *
 *  REGISTER USE:                                                   *
 *      r0 - character ptr                                          *
 *      r1 - working register                                       *
 *      r2 - character value                                        *
 ********************************************************************/
.global WriteSerial
WriteSerial:

    /*************************************
     * Check for NULL end of string and  *
     * advance ptr to next character.    *
     *************************************/
1:
    LDRB	r2, [r0], #1
    CMP		r2, #0
    BEQ		0f

    /*****************************************
     * Check the UART to see if it is empty. *
     *****************************************/
2:  LDR 	r1, =A1_UART0_BASE
    ORR 	r1, r1, #0x14
    LDRB 	r1, [r1]
    ANDS 	r1, r1, #0x20
    BEQ 	2b

    /****************************
     * Write the character out. *
     ****************************/
    LDR		r1, =A1_UART0_BASE
    STRB	r2, [r1]

    /****************************
     * Go on to next character. *
     ****************************/
    B		1b
0:
	MOV    	pc, lr	

/********************************************************************
 *  __a1_boot_from_nand                                             *
 *                                                                  *
 *  DESCRIPTION:                                                    *
 *      Check system boot from NAND or NOR.                         *
 *                                                                  *
 *                                                                  *
 *  RETURNS:                                                        *
 *      R0 = 0	NOR-boot                                            *
 *      R0 = 1	NAND-boot                                           *
 *                                                                  *
 ********************************************************************/

.global	__a1_boot_from_nand
__a1_boot_from_nand:
	MOV		r0, #1
	LDR		r1, =0x9900002C
	LDR		r2, =0x200
	LDR		r1, [r1]
	CMP		r1, r2
	MOVEQ	r0,	#0
	MOV    	pc, lr	
